/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.HashSet;
import java.math.BigInteger;

import org.apache.log4j.Logger;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.ByteString;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.MessageLite;
import com.mapr.baseutils.BitSetBytesHelperUtils;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cli.common.ListCommand;
import com.mapr.cli.common.PluggableAlarmUtil;
import com.mapr.cli.common.NodeField;
import com.mapr.cli.common.NodesCommonUtils;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import com.mapr.cliframework.base.inputparams.IntegerInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.cliframework.util.FieldInfo;
import com.mapr.cliframework.util.FilterUtil;
import com.mapr.fs.Rpc;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.*;
import com.mapr.fs.proto.Common.PluggableAlarm;
import com.mapr.fs.proto.Security.ServerKeyType;
import com.mapr.fs.cldb.util.Util;
import com.mapr.fs.cli.proto.CLIProto.Filter;
import com.mapr.fs.cli.proto.CLIProto.Limiter;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Common.AlarmId;
import com.mapr.fs.proto.Common.AlarmMsg;
import com.mapr.security.MaprSecurityException;

public class ServerCommands extends ListCommand implements CLIInterface {

  private static final Logger LOG = Logger.getLogger(ServerCommands.class);

  private Map<String, List<String>> runningServices = null;
  private Map<String, List<String>> configuredServices = null;

  // Note: this is also referenced in DiskCommands.java, Heatmap.java
  public static final int NUM_NODES_PER_RPC = 25;

  public static final String FILTER_PARAM_NAME = "filter";
  public static final String COLUMNS_PARAM_NAME = "columns";
  public static final String ALARMEDNODES_PARAM_NAME = "alarmednodes";
  public static final String NFSNODES_PARAM_NAME = "nfsnodes";

  public static final String SORT_PARAM_NAME = "sort";
  public static final String SORT_DIRECTION_PARAM_NAME = "dir";
  public static final String OUTPUT_PARAM_NAME = "output";
  public static final String START_PARAM_NAME = "start";
  public static final String LIMIT_PARAM_NAME = "limit";
  public static final String TOPO_PARENT_PARAM_NAME = "path";

  public static final String SP_CANREMOVE_PARAM_NAME = "canremovesp";
  public static final String SP_CANREMOVE_SPID_PARAM_NAME = "spid";
  public static final String FILESERVER_ID_PARAM_NAME = "serverids";
  public static final String HOSTID_PARAM_NAME = "hostids";
  public static final String FILESERVER_HOSTPORT_PARAM_NAME = "hostports";
  public static final String FILESERVER_NODES_PARAM_NAME = "nodes";
  public static final String FILESERVER_MARKMAINTENANCE_TIMEOUT_MINUTES = "timeoutminutes";
  public static final String FILESERVER_TOPOLOGY_PARAM_NAME = "topology";
  public static final String BLOCK_OUT_MOVES_PARAM_NAME = "blockMovesOut";
  public static final String BLOCK_IN_MOVES_PARAM_NAME = "blockMovesIn";
  public static final String LOCAL_CONTAINERS_PARAM_NAME = "localcontainers";
  public static final String NUM_SPS_PERINSTANCE_PARAM = "numSpsPerInstance";
  public static final String NUM_INSTANCES_ON_A_NODE_PARAM = "numInstances";
  public static final String MEMORY_PERINSTANCE_PARAM = "memoryPerInstance";

  public static final String MULTI_ARG_SEP = ",";
  
  private static final String CLDB_PARAM_NAME = "cldb";

  private static int MAX_NODEFIELDINFO = 1;
  
  private static Map<NodeState, String> nodeStateDesc = new ImmutableMap.Builder<NodeState, String>()
       .put(NodeState.CRITICAL_NO_HEARTBEAT, "No heartbeat from node for more than 5 minutes")
       .put(NodeState.CRITICAL_NO_DISKS, "All MapR-Fs disks on the node are dead or offlined")
       .put(NodeState.CRITICAL_FILESERVER_REPLICATE, "FileServer process in replicate state")
       .put(NodeState.CRITICAL_FILESERVER_DEAD, "FileServer process is dead/inactive")
       .put(NodeState.CRITICAL_NFS_DEAD, "NFS process is dead")
       .put(NodeState.CRITICAL_OPT_MAPR_FULL, "MapR Install directory " + MapRCliUtil.getMapRInstallDir() + " is full")
       .put(NodeState.CRITICAL_HIGH_MFS_MEMORY, "FileServer process on the node has high memory usage")
       .put(NodeState.MAINTENANCE_FILESERVER, "FileServer process in maintenance mode")
       .put(NodeState.DRAINING_MISSING_HEARTBEAT, "No heartbeat from node for more than 60 seconds")
       .put(NodeState.DRAINING_SERVICES_DOWN, "One or more services is down")
       .put(NodeState.DRAINING_ALARMS_RAISED, "One or more alarms raised")
       .put(NodeState.HEALTHY, "Healthy")
       .build();

  private static Map<String, BaseInputParameter> baseParams =
     new ImmutableMap.Builder<String, BaseInputParameter>()
        .put(MapRCliUtil.CLUSTER_NAME_PARAM,
  	      new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
  	           "cluster name",
  	           CLIBaseClass.NOT_REQUIRED,
  	           null))
  	    .build();  	

  // FIXME re-indent column exceed 80
  public static Map<NodeField, FieldInfo> fieldTable; // Gets set by pluggable alarms
  public static ImmutableMap.Builder<NodeField, FieldInfo> fieldTableBuilder = new ImmutableMap.Builder<NodeField, FieldInfo>()
      .put(new NodeField(NodeInfo.Id),
          new FieldInfo(NodeInfo.Id.getNumber(), "id", "id", Long.class))
      .put(new NodeField(NodeInfo.Ip),
          new FieldInfo(NodeInfo.Ip.getNumber(), "ip", "ip", String.class))
      .put(
          new NodeField(NodeInfo.Hostname),
          new FieldInfo(NodeInfo.Hostname.getNumber(), "hn", "hostname",
              String.class))
      .put(
          new NodeField(NodeInfo.RackPath),
          new FieldInfo(NodeInfo.RackPath.getNumber(), "rp", "racktopo",
              String.class))
      .put(
          new NodeField(NodeInfo.SwitchPath),
          new FieldInfo(NodeInfo.SwitchPath.getNumber(), "sp", "switchtopo",
              String.class))
      .put(
          new NodeField(NodeInfo.Status),
          new FieldInfo(NodeInfo.Status.getNumber(), "h", "health",
              Integer.class))
      .put(
          new NodeField(NodeInfo.StatusDesc),
          new FieldInfo(NodeInfo.StatusDesc.getNumber(), "hd", "healthDesc",
              String.class))        
      .put(
          new NodeField(NodeInfo.Services),
          new FieldInfo(NodeInfo.Services.getNumber(), "svc", "service",
              String.class))
      .put(
          new NodeField(NodeInfo.ConfiguredServices),
          new FieldInfo(NodeInfo.ConfiguredServices.getNumber(), "csvc", "configuredservice",
              String.class))
      .put(
          new NodeField(NodeInfo.FSHB),
          new FieldInfo(NodeInfo.FSHB.getNumber(), "fhb", "fs-heartbeat",
              Integer.class))
      .put(
          new NodeField(NodeInfo.JTHB),
          new FieldInfo(NodeInfo.JTHB.getNumber(), "jhb", "jt-heartbeat",
              Integer.class))
      .put(
          new NodeField(NodeInfo.DiskTotal),
          new FieldInfo(NodeInfo.DiskTotal.getNumber(), "dst", "dtotal",
              Long.class))
      .put(
          new NodeField(NodeInfo.DiskUsed),
          new FieldInfo(NodeInfo.DiskUsed.getNumber(), "dsu", "dused",
              Long.class))
      .put(
          new NodeField(NodeInfo.DiskAvail),
          new FieldInfo(NodeInfo.DiskAvail.getNumber(), "dsa", "davail",
              Long.class))
      .put(new NodeField(NodeInfo.Rpc),
          new FieldInfo(NodeInfo.Rpc.getNumber(), "rpc", "rpcs", Long.class))
      .put(new NodeField(NodeInfo.RpcIn),
          new FieldInfo(NodeInfo.RpcIn.getNumber(), "rpi", "rpcin", Long.class))
      .put(
          new NodeField(NodeInfo.RpcOut),
          new FieldInfo(NodeInfo.RpcOut.getNumber(), "rpo", "rpcout",
              Long.class))
      .put(
          new NodeField(NodeInfo.DiskCount),
          new FieldInfo(NodeInfo.DiskCount.getNumber(), "dsc", "disks",
              Integer.class))
      .put(
          new NodeField(NodeInfo.MapRDiskCount),
          new FieldInfo(NodeInfo.MapRDiskCount.getNumber(), "nmd",
              "MapRfs disks", Integer.class))
      .put(
          new NodeField(NodeInfo.DiskReadOps),
          new FieldInfo(NodeInfo.DiskReadOps.getNumber(), "dro", "dreads",
              Long.class))
      .put(
          new NodeField(NodeInfo.DiskReadKbytes),
          new FieldInfo(NodeInfo.DiskReadKbytes.getNumber(), "drk", "dreadK",
              Long.class))
      .put(
          new NodeField(NodeInfo.DiskWriteOps),
          new FieldInfo(NodeInfo.DiskWriteOps.getNumber(), "dwo", "dwrites",
              Long.class))
      .put(
          new NodeField(NodeInfo.DiskWriteKbytes),
          new FieldInfo(NodeInfo.DiskWriteKbytes.getNumber(), "dwk", "dwriteK",
              Long.class))
      .put(
          new NodeField(NodeInfo.CpuCount),
          new FieldInfo(NodeInfo.CpuCount.getNumber(), "cpc", "cpus",
              Integer.class))
      .put(
          new NodeField(NodeInfo.CpuUtil),
          new FieldInfo(NodeInfo.CpuUtil.getNumber(), "cpu", "utilization",
              Long.class))
      .put(
          new NodeField(NodeInfo.MemTotal),
          new FieldInfo(NodeInfo.MemTotal.getNumber(), "mt", "mtotal",
              Integer.class))
      .put(
          new NodeField(NodeInfo.MemUsed),
          new FieldInfo(NodeInfo.MemUsed.getNumber(), "mu", "mused",
              Integer.class))
      .put(
          new NodeField(NodeInfo.TTMapSlotsTotal),
          new FieldInfo(NodeInfo.TTMapSlotsTotal.getNumber(), "tms",
              "ttmapSlots", Integer.class))
      .put(
          new NodeField(NodeInfo.TTMapSlotsOccupied),
          new FieldInfo(NodeInfo.TTMapSlotsOccupied.getNumber(), "tmu",
              "ttmapUsed", Integer.class))
      .put(
          new NodeField(NodeInfo.TTReduceSlotsTotal),
          new FieldInfo(NodeInfo.TTReduceSlotsTotal.getNumber(), "trs",
              "ttReduceSlots", Integer.class))
      .put(
          new NodeField(NodeInfo.TTReduceSlotsOccupied),
          new FieldInfo(NodeInfo.TTReduceSlotsOccupied.getNumber(), "tru",
              "ttReduceUsed", Integer.class))
      .put(
          new NodeField(NodeInfo.FailedDisks),
          new FieldInfo(NodeInfo.FailedDisks.getNumber(), "nfd", "faileddisks",
              Integer.class))
      .put(
          new NodeField(NodeInfo.NumInstances),
          new FieldInfo(NodeInfo.NumInstances.getNumber(), "ni", "numInstances",
              String.class))
      .put(
          new NodeField(NodeInfo.NumSpsPerInstance),
          new FieldInfo(NodeInfo.NumSpsPerInstance.getNumber(), "nsp", "spsPerInstance",
              String.class))

      // Alarms
      .put(
          new NodeField(NodeInfo.LogLevelAlarm),
          new FieldInfo(NodeInfo.LogLevelAlarm.getNumber(), "lla",
              "LogLevelAlarm", Integer.class))
      .put(
          new NodeField(NodeInfo.ServiceCLDBDownAlarm),
          new FieldInfo(NodeInfo.ServiceCLDBDownAlarm.getNumber(), "sca",
              "ServiceCLDBDownNotRunningAlarm", Integer.class))
      .put(
          new NodeField(NodeInfo.ServiceFileserverDownAlarm),
          new FieldInfo(NodeInfo.ServiceFileserverDownAlarm.getNumber(),
              "sfsa", "ServiceFileserverDownNotRunningAlarm", Integer.class))
      .put(
          new NodeField(NodeInfo.ServiceJTDownAlarm),
          new FieldInfo(NodeInfo.ServiceJTDownAlarm.getNumber(), "sja",
              "ServiceJTDownNotRunningAlarm", Integer.class))
      .put(
          new NodeField(NodeInfo.ServiceTTDownAlarm),
          new FieldInfo(NodeInfo.ServiceTTDownAlarm.getNumber(), "sta",
              "ServiceTTDownNotRunningAlarm", Integer.class))
      .put(
          new NodeField(NodeInfo.ServiceHBMasterDownAlarm),
          new FieldInfo(NodeInfo.ServiceHBMasterDownAlarm.getNumber(), "shma",
              "ServiceHBMasterDownNotRunningAlarm", Integer.class))
      .put(
          new NodeField(NodeInfo.ServiceHBRegionDownAlarm),
          new FieldInfo(NodeInfo.ServiceHBRegionDownAlarm.getNumber(), "shra",
              "ServiceHBRegionDownNotRunningAlarm", Integer.class))
      .put(
          new NodeField(NodeInfo.ServiceNFSDownAlarm),
          new FieldInfo(NodeInfo.ServiceNFSDownAlarm.getNumber(), "sna",
              "ServiceNFSDownNotRunningAlarm", Integer.class))
      .put(
          new NodeField(NodeInfo.ServiceWebserverDownAlarm),
          new FieldInfo(NodeInfo.ServiceWebserverDownAlarm.getNumber(), "swa",
              "ServiceWebserverDownNotRunningAlarm", Integer.class))
      .put(
          new NodeField(NodeInfo.ServiceHoststatsDownAlarm),
          new FieldInfo(NodeInfo.ServiceHoststatsDownAlarm.getNumber(), "sha",
              "ServiceHoststatsDownNotRunningAlarm", Integer.class))
       .put(
          new NodeField(NodeInfo.ServiceHueDownAlarm),
          new FieldInfo(NodeInfo.ServiceHueDownAlarm.getNumber(), "shuea",
              "ServiceHueDownNotRunningAlarm", Integer.class))
       .put(
          new NodeField(NodeInfo.ServiceHttpfsDownAlarm),
          new FieldInfo(NodeInfo.ServiceHttpfsDownAlarm.getNumber(), "shfsa",
              "ServiceHttpfsDownNotRunningAlarm", Integer.class))
       .put(
          new NodeField(NodeInfo.ServiceBeeswaxDownAlarm),
          new FieldInfo(NodeInfo.ServiceBeeswaxDownAlarm.getNumber(), "sbwa",
              "ServiceBeeswaxDownNotRunningAlarm", Integer.class))
       .put(
          new NodeField(NodeInfo.ServiceHiveMetaDownAlarm),
          new FieldInfo(NodeInfo.ServiceHiveMetaDownAlarm.getNumber(), "shsma",
              "ServiceHiveDownNotRunningAlarm", Integer.class))
      .put(
          new NodeField(NodeInfo.ServiceHs2DownAlarm),
          new FieldInfo(NodeInfo.ServiceHs2DownAlarm.getNumber(), "shsa",
              "ServiceHs2DownNotRunningAlarm", Integer.class))
       .put(
          new NodeField(NodeInfo.ServiceOozieDownAlarm),
          new FieldInfo(NodeInfo.ServiceOozieDownAlarm.getNumber(), "sooza",
              "ServiceOozieDownNotRunningAlarm", Integer.class))
      .put(
          new NodeField(NodeInfo.DiskFailureAlarm),
          new FieldInfo(NodeInfo.DiskFailureAlarm.getNumber(), "fda",
              "DiskFailureAlarm", Integer.class))
      .put(
          new NodeField(NodeInfo.VersionMismatchAlarm),
          new FieldInfo(NodeInfo.VersionMismatchAlarm.getNumber(), "vma",
              "VersionMismatchAlarm", Integer.class))
      .put(
          new NodeField(NodeInfo.TimeSkewAlarm),
          new FieldInfo(NodeInfo.TimeSkewAlarm.getNumber(), "tsa",
              "TimeSkewAlarm", Integer.class))
      .put(
          new NodeField(NodeInfo.HbProcessingSlow),
          new FieldInfo(NodeInfo.HbProcessingSlow.getNumber(), "hps",
              "HbProcessingSlow", Integer.class))
      .put(
          new NodeField(NodeInfo.NodeRootPartitionFull),
          new FieldInfo(NodeInfo.NodeRootPartitionFull.getNumber(), "rpf",
              "RootPartitionFullAlarm", Integer.class))
      .put(
          new NodeField(NodeInfo.NodeOptMapRFull),
          new FieldInfo(NodeInfo.NodeOptMapRFull.getNumber(), "hmf",
              "HomeMapRFullAlarm", Integer.class))
      .put(
          new NodeField(NodeInfo.NodeCorePresent),
          new FieldInfo(NodeInfo.NodeCorePresent.getNumber(), "ncp",
              "CorePresentAlarm", Integer.class))

      .put(
          new NodeField(NodeInfo.NodeHighMfsMemory),
          new FieldInfo(NodeInfo.NodeHighMfsMemory.getNumber(), "nhmm",
              "HighMfsMemoryAlarm", Integer.class))

      .put(
          new NodeField(NodeInfo.NodePamMisconfigured),
          new FieldInfo(NodeInfo.NodePamMisconfigured.getNumber(), "pma",
              "PamMisconfiguredAlarm", Integer.class))

      .put(
          new NodeField(NodeInfo.NodeTTLocaldirFull),
          new FieldInfo(NodeInfo.NodeTTLocaldirFull.getNumber(), "tla",
              "TTLocaldirFullAlarm", Integer.class))

      .put(
          new NodeField(NodeInfo.NodeNoHeartbeat),
          new FieldInfo(NodeInfo.NodeNoHeartbeat.getNumber(), "nha",
              "NodeNoHeartbeatAlarm", Integer.class))

      .put(
          new NodeField(NodeInfo.NodeMaprUserMismatch),
          new FieldInfo(NodeInfo.NodeMaprUserMismatch.getNumber(), "nma",
              "NodeMaprUserMismatchAlarm", Integer.class))

      .put(
          new NodeField(NodeInfo.NodeDuplicateHostId),
          new FieldInfo(NodeInfo.NodeDuplicateHostId.getNumber(), "ndh",
              "NodeDuplicateHostIdAlarm", Integer.class))
              
     .put(
         new NodeField(NodeInfo.NodeMetricsWriteProblemAlarm),
         new FieldInfo(NodeInfo.NodeMetricsWriteProblemAlarm.getNumber(), "nmw",
             "NodeMetricsWriteProblemAlarm", Integer.class))

     .put(
         new NodeField(NodeInfo.NodeTooManyContainersAlarm),
         new FieldInfo(NodeInfo.NodeTooManyContainersAlarm.getNumber(), "nmc",
             "NodeTooManyContainersAlarm", Integer.class))

     .put(
         new NodeField(NodeInfo.NodeM7ConfigMismatchAlarm),
         new FieldInfo(NodeInfo.NodeM7ConfigMismatchAlarm.getNumber(), "nca",
             "NodeM7ConfigMismatchAlarm", Integer.class))

     .put(
         new NodeField(NodeInfo.IncorrectTopologyAlarm),
         new FieldInfo(NodeInfo.IncorrectTopologyAlarm.getNumber(), "ita",
             "IncorrectTopologyAlarm", Integer.class))

      .put(
          new NodeField(NodeInfo.VirtualIp),
          new FieldInfo(NodeInfo.VirtualIp.getNumber(), "vip", "vip",
              String.class))
      .put(
          new NodeField(NodeInfo.VirtualIpEnd),
          new FieldInfo(NodeInfo.VirtualIpEnd.getNumber(), "vipe", "vipe",
              String.class))
      .put(new NodeField(NodeInfo.NetMask),
          new FieldInfo(NodeInfo.NetMask.getNumber(), "nm", "nm", String.class))
      .put(
          new NodeField(NodeInfo.MacAddress),
          new FieldInfo(NodeInfo.MacAddress.getNumber(), "mac", "mac",
              String.class))
      .put(new NodeField(NodeInfo.Gateway),
          new FieldInfo(NodeInfo.Gateway.getNumber(), "gw", "gw", String.class))

      .put(
          new NodeField(NodeInfo.BytesReceived),
          new FieldInfo(NodeInfo.BytesReceived.getNumber(), "br",
              "bytesReceived", Long.class))
     .put(
         new NodeField(NodeInfo.BytesSent),
         new FieldInfo(NodeInfo.BytesSent.getNumber(), "bs", "bytesSent",
             Long.class))
     .put(
         new NodeField(NodeInfo.NumResyncSlots),
         new FieldInfo(NodeInfo.NumResyncSlots.getNumber(), "nrs", "numResyncSlots",
             Integer.class))
     .put(
         new NodeField(NodeInfo.CpuUptime),
         new FieldInfo(NodeInfo.CpuUptime.getNumber(), "cpt", "uptime",
             Long.class))
     .put(
         new NodeField(NodeInfo.BlockMovesOut),
         new FieldInfo(NodeInfo.BlockMovesOut.getNumber(), "bmo", "blockMovesOut",
             Boolean.class))
     .put(
         new NodeField(NodeInfo.BlockMovesIn),
         new FieldInfo(NodeInfo.BlockMovesIn.getNumber(), "bmi", "blockMovesIn",
             Boolean.class))
      .put(
          new NodeField(NodeInfo.DbPuts10s),
          new FieldInfo(NodeInfo.DbPuts10s.getNumber(),
              "p10s", "numPutsInLastTenSeconds",
              Long.class))

      .put(
          new NodeField(NodeInfo.DbPuts1m),
          new FieldInfo(NodeInfo.DbPuts1m.getNumber(),
              "p1m", "numPutsInLastMinute",
              Long.class))

      .put(
          new NodeField(NodeInfo.DbPuts5m),
          new FieldInfo(NodeInfo.DbPuts5m.getNumber(),
              "p5m", "numPutsInLastFiveMinutes",
              Long.class))

      .put(
          new NodeField(NodeInfo.DbPuts15m),
          new FieldInfo(NodeInfo.DbPuts15m.getNumber(),
              "p15m", "numPutsInLastFifteenMinutes",
              Long.class))

      .put(
          new NodeField(NodeInfo.DbGets10s),
          new FieldInfo(NodeInfo.DbGets10s.getNumber(),
              "g10s", "numGetsInLastTenSeconds",
              Long.class))

      .put(
          new NodeField(NodeInfo.DbGets1m),
          new FieldInfo(NodeInfo.DbGets1m.getNumber(),
              "g1m", "numGetsInLastMinute",
              Long.class))

      .put(
          new NodeField(NodeInfo.DbGets5m),
          new FieldInfo(NodeInfo.DbGets5m.getNumber(),
              "g5m", "numGetsInLastFiveMinutes",
              Long.class))

      .put(
          new NodeField(NodeInfo.DbGets15m),
          new FieldInfo(NodeInfo.DbGets15m.getNumber(),
              "g15m", "numGetsInLastFifteenMinutes",
              Long.class))

      .put(
          new NodeField(NodeInfo.DbScans10s),
          new FieldInfo(NodeInfo.DbScans10s.getNumber(),
              "s10s", "numScansInLastTenSeconds",
              Long.class))

      .put(
          new NodeField(NodeInfo.DbScans1m),
          new FieldInfo(NodeInfo.DbScans1m.getNumber(),
              "s1m", "numScansInLastMinute",
              Long.class))

      .put(
          new NodeField(NodeInfo.DbScans5m),
          new FieldInfo(NodeInfo.DbScans5m.getNumber(),
              "s5m", "numScansInLastFiveMinutes",
              Long.class))

      .put(
          new NodeField(NodeInfo.DbScans15m),
          new FieldInfo(NodeInfo.DbScans15m.getNumber(),
              "s15m", "numScansInLastFifteenMinutes",
              Long.class));


  static {
    NodeInfo [] values = NodeInfo.values();
    int max = 0;
    for ( NodeInfo value : values ) {
      if ( value.getNumber() > max )
        max = value.getNumber();
    }
    MAX_NODEFIELDINFO = max;
  }

  public static Map<String, BaseInputParameter> serverListParams = 
  	new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(baseParams)
      .put(
          ServerCommands.SORT_PARAM_NAME,
          new TextInputParameter(ServerCommands.SORT_PARAM_NAME, "none",
              CLIBaseClass.NOT_REQUIRED, "hostname").setInvisible(true))
      .put(
          ServerCommands.SORT_DIRECTION_PARAM_NAME,
          new TextInputParameter(ServerCommands.SORT_DIRECTION_PARAM_NAME,
              "none", CLIBaseClass.NOT_REQUIRED, "ASC").setInvisible(true))
      .put(
          ServerCommands.OUTPUT_PARAM_NAME,
          new TextInputParameter(ServerCommands.OUTPUT_PARAM_NAME, "verbose",
              CLIBaseClass.NOT_REQUIRED, "verbose"))
      .put(
          ServerCommands.START_PARAM_NAME,
          new IntegerInputParameter(ServerCommands.START_PARAM_NAME, "start",
              CLIBaseClass.NOT_REQUIRED, 0))
      .put(
          ServerCommands.LIMIT_PARAM_NAME,
          new IntegerInputParameter(ServerCommands.LIMIT_PARAM_NAME, "limit",
              CLIBaseClass.NOT_REQUIRED, Integer.MAX_VALUE))
      .put(
          ServerCommands.FILTER_PARAM_NAME,
          new TextInputParameter(ServerCommands.FILTER_PARAM_NAME, "none",
              CLIBaseClass.NOT_REQUIRED, "none"))
      .put(
          ServerCommands.COLUMNS_PARAM_NAME,
          new TextInputParameter(ServerCommands.COLUMNS_PARAM_NAME, "none",
              CLIBaseClass.NOT_REQUIRED, "all"))
      .put(
          ServerCommands.ALARMEDNODES_PARAM_NAME,
          new BooleanInputParameter(ServerCommands.ALARMEDNODES_PARAM_NAME,
              "alarmednodes", CLIBaseClass.NOT_REQUIRED, 0))
      .put(
          ServerCommands.NFSNODES_PARAM_NAME,
          new BooleanInputParameter(ServerCommands.NFSNODES_PARAM_NAME,
              "nfsnodes", CLIBaseClass.NOT_REQUIRED, 0))
      .put(
          NodesCommonUtils.ZK_CONNECTSTRING,
          new TextInputParameter(NodesCommonUtils.ZK_CONNECTSTRING,
              "ZooKeeper Connect String: 'host:port,host:port,host:port,...'",
              CLIBaseClass.NOT_REQUIRED, null))
      .build();

  public static Map<String, BaseInputParameter> topoParams = 
  	new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(baseParams)
      // .put(ServerCommands.START_PARAM_NAME,
      // new IntegerInputParameter(ServerCommands.START_PARAM_NAME,
      // "start",
      // CLIBaseClass.NOT_REQUIRED,
      // 0))
      // .put(ServerCommands.LIMIT_PARAM_NAME,
      // new IntegerInputParameter(ServerCommands.LIMIT_PARAM_NAME,
      // "limit",
      // CLIBaseClass.NOT_REQUIRED,
      // 0))
      .put(
          ServerCommands.TOPO_PARENT_PARAM_NAME,
          new TextInputParameter(ServerCommands.TOPO_PARENT_PARAM_NAME, "path",
              CLIBaseClass.NOT_REQUIRED, null))
      .build();

  public static Map<String, BaseInputParameter> cldbMasterParams= 
  	new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(baseParams).build();

  public static Map<String, BaseInputParameter> canRemoveSpParams =
  	new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(baseParams)
      .put(SP_CANREMOVE_SPID_PARAM_NAME,
          new TextInputParameter(SP_CANREMOVE_SPID_PARAM_NAME,
              "spid", CLIBaseClass.REQUIRED, null))
      .build();

  public static Map<String, BaseInputParameter> moveParams = 
  	new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(baseParams)
      .put(
          ServerCommands.FILESERVER_ID_PARAM_NAME,
          new TextInputParameter(ServerCommands.FILESERVER_ID_PARAM_NAME,
              "serverids", CLIBaseClass.REQUIRED, null))
       .put(
          ServerCommands.FILESERVER_TOPOLOGY_PARAM_NAME,
          new TextInputParameter(ServerCommands.FILESERVER_TOPOLOGY_PARAM_NAME,
              "topology", CLIBaseClass.REQUIRED, null))
       .build();

  public static Map<String, BaseInputParameter> modifyParams =
  	new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(baseParams)
      .put(
          ServerCommands.FILESERVER_NODES_PARAM_NAME,
          new TextInputParameter(ServerCommands.FILESERVER_NODES_PARAM_NAME,
              "nodes", CLIBaseClass.REQUIRED, null))
      .put(
          ServerCommands.BLOCK_OUT_MOVES_PARAM_NAME,
          new BooleanInputParameter(ServerCommands.BLOCK_OUT_MOVES_PARAM_NAME,
              "true/false", CLIBaseClass.NOT_REQUIRED, null).setInvisible(true))
      .put(
          ServerCommands.BLOCK_IN_MOVES_PARAM_NAME,
          new BooleanInputParameter(ServerCommands.BLOCK_IN_MOVES_PARAM_NAME,
              "true/false", CLIBaseClass.NOT_REQUIRED, null).setInvisible(true))
      .put(
          ServerCommands.NUM_SPS_PERINSTANCE_PARAM,
          new IntegerInputParameter(ServerCommands.NUM_SPS_PERINSTANCE_PARAM,
              "<number of SPs per mfs instance>", CLIBaseClass.NOT_REQUIRED, null))
      .put(
          ServerCommands.NUM_INSTANCES_ON_A_NODE_PARAM,
          new IntegerInputParameter(ServerCommands.NUM_INSTANCES_ON_A_NODE_PARAM,
              "<number of mfs instances on a node>", CLIBaseClass.NOT_REQUIRED, null).setInvisible(true))
      .put(
          ServerCommands.MEMORY_PERINSTANCE_PARAM,
          new IntegerInputParameter(ServerCommands.MEMORY_PERINSTANCE_PARAM,
              "<memory (in MB) for each mfs instance>", CLIBaseClass.NOT_REQUIRED, null).setInvisible(true))
      .build();

  public static Map<String, BaseInputParameter> allowIntoClusterParams =
  	new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(baseParams)
      .put(
          ServerCommands.HOSTID_PARAM_NAME,
          new TextInputParameter(ServerCommands.HOSTID_PARAM_NAME,
              "hostids", CLIBaseClass.REQUIRED, null))
       .build();

  public static Map<String, BaseInputParameter> markMaintenanceParams = 
  	new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(baseParams)
      .put(
          ServerCommands.FILESERVER_ID_PARAM_NAME,
          new TextInputParameter(ServerCommands.FILESERVER_ID_PARAM_NAME,
              "serverids", CLIBaseClass.NOT_REQUIRED, null))
      .put(
          ServerCommands.FILESERVER_NODES_PARAM_NAME,
          new TextInputParameter(ServerCommands.FILESERVER_NODES_PARAM_NAME,
              "nodes", CLIBaseClass.NOT_REQUIRED, null))
      .put(
          ServerCommands.FILESERVER_MARKMAINTENANCE_TIMEOUT_MINUTES,
          new IntegerInputParameter(
              ServerCommands.FILESERVER_MARKMAINTENANCE_TIMEOUT_MINUTES,
              "timeoutminutes", CLIBaseClass.REQUIRED, null))
       .build();

  public static Map<String, BaseInputParameter> failoverNodeParams = 
  	new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(baseParams)
      .put(
          ServerCommands.FILESERVER_NODES_PARAM_NAME,
          new TextInputParameter(ServerCommands.FILESERVER_NODES_PARAM_NAME,
              "nodes", CLIBaseClass.NOT_REQUIRED, null))
      .put(
          ServerCommands.LOCAL_CONTAINERS_PARAM_NAME,
          new BooleanInputParameter(ServerCommands.LOCAL_CONTAINERS_PARAM_NAME,
              "localcontainers", CLIBaseClass.NOT_REQUIRED, 0))
       .build();

  static final CLICommand serverMarkMaintenanceCommand = new CLICommand(
      "maintenance",
      "usage: node maintenance (-serverids serverid[,servierid2] | "
          + " -nodes node1[,node2] -timeoutminutes [0|minutes] ",
      ServerCommands.class, ExecutionTypeEnum.NATIVE, markMaintenanceParams,
      null);

  static final CLICommand serverFailoverCommand = new CLICommand(
      "failover",
      "usage: node failover -nodes node1[,node2] -localcontainers=[true|false]",
      ServerCommands.class, ExecutionTypeEnum.NATIVE, failoverNodeParams, null)
    .setUsageInVisible(true);

  static final CLICommand serverMoveCommand = new CLICommand("move",
      "usage: node move -serverids serverid[,serverid2,serverid3]"
          + " -topology newtoplogy", ServerCommands.class,
      ExecutionTypeEnum.NATIVE, moveParams, null)
      .setShortUsage("node move -serverids serverid[,serverid2,serverid3]"
          + " -topology topology -cluster clustername");

  static final CLICommand serverModifyCommand = new CLICommand("modify",
      "usage: node modify -nodes hostname[,hostname]"
          + " [-blockMovesOut true/false] [-blockMovesIn true/false]"
          + " [-" + NUM_SPS_PERINSTANCE_PARAM + "<count>]"
          + " [-" + MEMORY_PERINSTANCE_PARAM + "<amount of memory>]",
      ServerCommands.class,
      ExecutionTypeEnum.NATIVE, modifyParams, null)
      .setShortUsage("node modify -nodes hostname[,hostname2,hostname3]"
          + " -blockMovesOut true/false -blockMovesIn true/false"
          + " [-" + NUM_SPS_PERINSTANCE_PARAM + "<count>]"
          + " [-" + MEMORY_PERINSTANCE_PARAM + "<amount of memory>]");

  static final CLICommand allowIntoClusterCommand = new CLICommand("allow-into-cluster",
      "usage: node allow-into-cluster -hostids hostid[,hostid2,hostid3]",
      ServerCommands.class,
      ExecutionTypeEnum.NATIVE, allowIntoClusterParams, null)
      .setShortUsage("node allow-into-cluster -hostids hostid[,hostid2,hostid3]"
          + " -cluster clustername");

  static final CLICommand canRemoveSpCommand = new CLICommand(SP_CANREMOVE_PARAM_NAME,
      "usage: node -canremovesp <spid>", ServerCommands.class, ExecutionTypeEnum.NATIVE,
      canRemoveSpParams, null).setShortUsage("node -canremovesp <spid>");

  static final CLICommand cldbMasterCommand = new CLICommand("cldbmaster",
      "usage: node cldbmaster", ServerCommands.class, ExecutionTypeEnum.NATIVE,
      cldbMasterParams, null).setShortUsage("node cldbmaster");

  static final CLICommand serverListCommand = new CLICommand("list",
      // "usage: node list -sort fieldName -output terse|verbose -h cldbIp -p cldbPort",
      "usage: node list -output terse|verbose -h cldbIp -p cldbPort",
      ServerCommands.class, ExecutionTypeEnum.NATIVE, serverListParams, null)
  // .setShortUsage("node list -sort fieldName -h cldbIp (a.b.c.d) -p cldbPort");
      .setShortUsage("node list -h cldbIp (a.b.c.d) -p cldbPort");

  static final CLICommand topoCommand = new CLICommand("topo",
      "usage: node topo -h cldbIp (a.b.c.d) -p cldbPort", ServerCommands.class,
      ExecutionTypeEnum.NATIVE, topoParams, null)
      .setShortUsage("usage: node topo -h cldbIp (a.b.c.d)  -p cldbPort");

  private static final CLICommand listCldbsCommand =
     new CLICommand (
         "listcldbs",
         "usage: node listcldbs -cluster <clustername> | -cldb <cldb hostname/ip>",
         ServerCommands.class,
         CLICommand.ExecutionTypeEnum.NATIVE,
         new ImmutableMap.Builder<String, BaseInputParameter>()
           .putAll(baseParams)
           .put(
             ServerCommands.CLDB_PARAM_NAME,
             new TextInputParameter(ServerCommands.CLDB_PARAM_NAME,
              "<cldb hostname|ip:port>", 
              CLIBaseClass.NOT_REQUIRED, 
              null))      
  	       .build(),
  	     null)
	    .setShortUsage("node listcldbs -cluster <clustername> | -cldb <hostname|ip:port>");

  private static final CLICommand listZksCommand =
     new CLICommand (
         "listzookeepers",
         "usage: node listzookeepers -cluster <clustername>",
         ServerCommands.class,
         CLICommand.ExecutionTypeEnum.NATIVE,
         new ImmutableMap.Builder<String, BaseInputParameter>()
           .putAll(baseParams)
           .put(
             ServerCommands.CLDB_PARAM_NAME,
             new TextInputParameter(ServerCommands.CLDB_PARAM_NAME,
              "<cldb hostname|ip:port>", 
              CLIBaseClass.NOT_REQUIRED, 
              null))
           .build(),
         null)
    .setShortUsage("node listzookeepers -cluster <clustername> | -cldb <hostname|ip:port>");  

  private static final CLICommand listCldbsAndZksCommand =
    new CLICommand (
        "listcldbzks",
        "usage: node listcldbzks -cluster <clustername> | -cldb <hostname|ip:port>",
        ServerCommands.class,
        CLICommand.ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>()
          .putAll(baseParams)
          .put(
            ServerCommands.CLDB_PARAM_NAME,
            new TextInputParameter(ServerCommands.CLDB_PARAM_NAME,
             "<cldb hostname|ip:port>", 
             CLIBaseClass.NOT_REQUIRED, 
             null))      
 	       .build(),
 	     null)
	    .setShortUsage("node listcldbzks -cluster <clustername> | -cldb <hostname|ip:port>")
	    .setUsageInVisible(true);

  public static final CLICommand serverCommands = new CLICommand("node",
      "node [list|move|modify|allow-into-cluster|services|" +
      "topo|remove|heatmap|listcldbs|listzookeepers|" +
      "cldbmaster|canremovesp|maintenance|metrics|listcldbzks]",
      CLIUsageOnlyCommand.class, ExecutionTypeEnum.NATIVE, new CLICommand[] {
          serverListCommand, serverMoveCommand, serverModifyCommand,
          allowIntoClusterCommand, cldbMasterCommand, canRemoveSpCommand,
          serverMarkMaintenanceCommand, topoCommand,
          serverFailoverCommand,
          NodeRemoveCommand.nodeRemove,
          NodeMetricsCommand.nodeMetricsCmd,
          NodeMetricsCommand.showMemoryCmd,
          NodeServicesManagementCommand.nodeServices,
          Heatmap.heatmapCmds,
          listCldbsCommand,
          listZksCommand,
          listCldbsAndZksCommand})
      .setShortUsage("node [list|move|modify|allow-into-cluster|services|" +
          "topo|remove|heatmap|listcldbs|listzookeepers|" +
          "cldbmaster|canremovesp|maintenance|metrics|show|listcldbzks]");

  public ServerCommands(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }

  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    if (LOG.isDebugEnabled()) {
      LOG.debug("serverCommands start");
    }
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);

    try {
      int port = Rpc.initialize(0, 0, null); // Client
      if (port < 0)
        throw new IOException("Error in RPC init");
    } catch (Exception e) {
      /**
      * <MAPR_ERROR>
      * Message:Exception in Rpc.initialize <exception>
      * Function:ServerCommands.executeRealCommand()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      System.out.println("Exception in Rpc.initialize " + e);
    }
    // First get all pluggable alarms and add all them to the immutable map
    fieldTable = PluggableAlarmUtil.appendNodeMap(getUserCredentials(), fieldTableBuilder);
    MAX_NODEFIELDINFO  = PluggableAlarmUtil.getMaxNumNodes(getUserCredentials(), MAX_NODEFIELDINFO);

    String cmd = cliCommand.getCommandName();
    if (cmd.equalsIgnoreCase("list")) {
      if (getParamBooleanValue(ALARMEDNODES_PARAM_NAME, 0)
          && getParamBooleanValue(NFSNODES_PARAM_NAME, 0)) {
        /**
        * <MAPR_ERROR>
        * Message:Cannot use both alarmednodes and nfsnodes together
        * Function:ServerCommands.executeRealCommand()
        * Meaning:You cannot specify both the {{alarmednodes}} parameter and the {{nfsnodes}} parameter.
        * Resolution:Check the command syntax and try again.
        * </MAPR_ERROR>
        */
        out.addError(new OutputError(Errno.EINVAL,
            "Cannot use both alarmednodes and nfsnodes together"));
        return output;
      }
      // getServicesInfo just once per command. It is a big overhead to do it
      // multiple times
      getServicesInfo();
      list(out);
    } else if (cmd.equalsIgnoreCase("topo")) {
      ClusterTopologyRequest.Builder req = ClusterTopologyRequest.newBuilder()
          .setCreds(getUserCredentials());

      if (isParamPresent(ServerCommands.TOPO_PARENT_PARAM_NAME)) {
        req.setPath(getParamTextValue(ServerCommands.TOPO_PARENT_PARAM_NAME, 0));
      }

      byte[] replyData;
      try {
        if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
          replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
              getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0),
              Common.MapRProgramId.CldbProgramId.getNumber(),
              CLDBProto.CLDBProg.ClusterTopologyProc.getNumber(), req.build(),
              ClusterTopologyResponse.class);
        } else {
          replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
              Common.MapRProgramId.CldbProgramId.getNumber(),
              CLDBProto.CLDBProg.ClusterTopologyProc.getNumber(), req.build(),
              ClusterTopologyResponse.class);
        }
        if ( replyData == null ) {
        	out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
        	return output;
        }
        ClusterTopologyResponse resp = ClusterTopologyResponse
            .parseFrom(replyData);

        for (String n : resp.getNodeList()) {
          /* create an empty ouutput node and add key values to it */
          OutputNode topoInfo = new OutputNode();
          topoInfo.addChild(new OutputNode("path", n));
          out.addNode(topoInfo);
        }
      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
      } catch (Exception e) {
        // log error
        /**
        * <MAPR_ERROR>
        * Message:InvalidProtocolBufferException  Exception <error>
        * Function:ServerCommands.executeRealCommand()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        throw new CLIProcessingException("InvalidProtocolBufferException "
            + " Exception", e);
      }
    } else if (cmd.equalsIgnoreCase("move")) {
      try {
        return fileServerMove();
      } catch (Exception e) {
        /**
        * <MAPR_ERROR>
        * Message:Send request Exception <error>
        * Function:ServerCommands.executeRealCommand()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        throw new CLIProcessingException("Send request Exception", e);
      }
    } else if (cmd.equalsIgnoreCase("modify")) {
      try {
        return fileServerModify();
      } catch (Exception e) {
        /**
        * <MAPR_ERROR>
        * Message:Send request Exception <error>
        * Function:ServerCommands.executeRealCommand()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        throw new CLIProcessingException("Send request Exception", e);
      }
    } else if (cmd.equalsIgnoreCase("allow-into-cluster")) {
      try {
        return fileServerAllowIntoCluster();
      } catch (Exception e) {
        /**
        * <MAPR_ERROR>
        * Message:Send request Exception <error>
        * Function:ServerCommands.executeRealCommand()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        throw new CLIProcessingException("Send request Exception", e);
      }
    } else if (cmd.equalsIgnoreCase("maintenance")) {
      try {
        return fileServerMaintenance();
      } catch (Exception e) {
        /**
        * <MAPR_ERROR>
        * Message:Send request Exception <error>
        * Function:ServerCommands.executeRealCommand()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        throw new CLIProcessingException("Send request Exception", e);
      }
    } else if (cmd.equalsIgnoreCase("failover")) {
      try {
        return failoverFileServer();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      }
    } else if (cmd.equalsIgnoreCase(SP_CANREMOVE_PARAM_NAME)) {
      try {
        if (isParamPresent(SP_CANREMOVE_SPID_PARAM_NAME)) {
          String spid = getParamTextValue(SP_CANREMOVE_SPID_PARAM_NAME, 0);
          return canRemoveSP(spid);
        } else {
          LOG.error("Invalid request: SP id missing with canremovesp.");
          out.addError(new OutputError(Errno.EINVAL, "Invalid request: Storagepool id missing with canremovesp"));
          return output;
        }
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      }
    } else if (cmd.equalsIgnoreCase("cldbmaster")) {
      try {
        return cldbMaster();
      } catch (Exception e) {
        /**
        * <MAPR_ERROR>
        * Message:Send request Exception <error>
        * Function:ServerCommands.executeRealCommand()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        throw new CLIProcessingException("Send request Exception", e);
      }
    } else if (cliCommand.getCommandName().equalsIgnoreCase("listcldbs")) {
    	listCldbs(out);
    } else if (cliCommand.getCommandName().equalsIgnoreCase("listzookeepers")) {
      listZks(out);
    } else if (cliCommand.getCommandName().equalsIgnoreCase("listcldbzks")) {
      listCldbZks(out);
    }

    if (LOG.isDebugEnabled()) {
      LOG.debug("serverCommands end");
    }

    return output;
  }

  private CommandOutput canRemoveSP(String spid) throws CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);

    canRemoveSpRequest req = canRemoveSpRequest.newBuilder()
                                               .setCreds(getUserCredentials())
                                               .setSpid(spid)
                                               .build();
    byte[] replyData = null;
    try {
      if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
            getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0),
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg.CanRemoveSPProc.getNumber(), req,
            canRemoveSpResponse.class);
      } else {
        replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg.CanRemoveSPProc.getNumber(), req,
            canRemoveSpResponse.class);
      }
    } catch (Exception e) {
      throw new CLIProcessingException(e);
    }

    if (replyData == null) {
  	  out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
  	  return output;
    }
    try {
      canRemoveSpResponse resp = canRemoveSpResponse.parseFrom(replyData);
      if (resp == null) {
        out.addError(new OutputError(Errno.EOPFAILED, "Could not contact any CLDB for Master"));
        return output;
      }

      if (resp.getStatus() != 0) {
        LOG.error("node canRemoveSP: failed with status: " + resp.getStatus());
        out.addError(new OutputError(resp.getStatus(), "Can not remove SP: " + spid + ". " + resp.getErrMsg())
                          .setPropagateErrorSupport(true)
                          .setField(spid));
        return output;
      }
    } catch (Exception e) {
      LOG.error("node canRemoveSP: failed with Exception " + e);
      out.addError(new OutputError(Errno.EOPFAILED,
          "node canRemoveSP: failed with Exception " + e));
      return output;
    }

    return output;
  }

  private CommandOutput cldbMaster() throws CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);

    isCLDBMasterRequest req = isCLDBMasterRequest.newBuilder().build();
    byte[] replyData = null;

    try {
      if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
            getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0),
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg.IsCLDBMasterProc.getNumber(), req,
            isCLDBMasterResponse.class);
      } else {
        replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg.IsCLDBMasterProc.getNumber(), req,
            isCLDBMasterResponse.class);
      }
    } catch (Exception e) {
      throw new CLIProcessingException(e);
    }

    if (replyData == null) {
  	  out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
  	  return output;
    }
    try {
      isCLDBMasterResponse resp = isCLDBMasterResponse.parseFrom(replyData);
      if (resp == null) {
        out.addError(new OutputError(Errno.EOPFAILED, "Could not contact any CLDB for Master"));
        return output;
      }

      if (resp.getStatus() != 0) {
        /**
        * <MAPR_ERROR>
        * Message:node cldbmaster: failed with status: <status>
        * Function:ServerCommands.cldbMaster()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        LOG.error("node cldbmaster: failed with status: " + resp.getStatus());
        out.addError(new OutputError(resp.getStatus(),
            "node cldbmaster failed with status:" + resp.getStatus()));
        return output;
      }
      String master = resp.getCurrentMaster().equals("") ? "No Master" : resp
          .getCurrentMaster();
      out.addNode(new OutputNode("cldbmaster", master));
    } catch (Exception e) {
      /**
      * <MAPR_ERROR>
      * Message:node cldbmaster: failed with status: <status>
      * Function:ServerCommands.cldbMaster()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      LOG.error("node cldbmaster: failed with Exception " + e);
      out.addError(new OutputError(Errno.EOPFAILED,
          "node cldbmaster: failed with Exception " + e));
      return output;
    }

    return output;
  }

  private CommandOutput fileServerMove() throws CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);
    List<String> serverIds = new ArrayList<String>();
    String fileServerIds = getParamTextValue(
        ServerCommands.FILESERVER_ID_PARAM_NAME, 0);

    if (fileServerIds.contains(ServerCommands.MULTI_ARG_SEP)) {
      serverIds.addAll(Arrays.asList(fileServerIds
          .split(ServerCommands.MULTI_ARG_SEP)));
    } else {
      serverIds.add(fileServerIds);
    }

    for (String serverIdString : serverIds) {
      long serverId = -1;
      try {
        serverId = new BigInteger(serverIdString).longValue();
      } catch (NumberFormatException e) {
        /**
        * <MAPR_ERROR>
        * Message:node move: Invalid serverId specified
        * Function:ServerCommands.cldbMaster()
        * Meaning:The specified server ID was invalid.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        LOG.error("node move: Invalid serverId specified");
        out.addError(new OutputError(Errno.EINVAL, "Invalid serverid "
            + serverIdString + " specified"));
        continue;
      }

      String topology = getParamTextValue(
          ServerCommands.FILESERVER_TOPOLOGY_PARAM_NAME, 0);

      FileServerMoveRequest req = FileServerMoveRequest.newBuilder()
          .setServerId(serverId).setTopology(topology).setCreds(
              getUserCredentials()).build();
      byte[] replyData = null;
      try {
        if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
          replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
              getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0),
              Common.MapRProgramId.CldbProgramId.getNumber(),
              CLDBProto.CLDBProg.FileServerMoveProc.getNumber(), req,
              FileServerMoveResponse.class);
        } else {
          replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
              Common.MapRProgramId.CldbProgramId.getNumber(),
              CLDBProto.CLDBProg.FileServerMoveProc.getNumber(), req,
              FileServerMoveResponse.class);
        }
      } catch (Exception e) {
        throw new CLIProcessingException(e);
      }

      if ( replyData == null ) {
    	  out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
    	  continue;
      }
      try {
        FileServerMoveResponse resp = FileServerMoveResponse
            .parseFrom(replyData);

        if (resp.getStatus() != 0) {
          /**
          * <MAPR_ERROR>
          * Message:node move: failed with status: <status> for fileServer <server ID>
          * Function:ServerCommands.fileServerMove()
          * Meaning:An error occurred.
          * Resolution:Contact technical support.
          * </MAPR_ERROR>
          */
          String errMsg = Errno.toString(resp.getStatus());
          if (resp.hasErrMsg()) errMsg = resp.getErrMsg();
          String msg = "node move for fileServer " 
                          + serverId
                          + " failed with "
                          + errMsg;
          LOG.error(msg);
          out.addError(new OutputError(resp.getStatus(), msg));
          continue;
        }
      } catch (Exception e) {
        /**
        * <MAPR_ERROR>
        * Message:node move: failed with InvalidProtocolBufferException <exception> for server <server ID>
        * Function:ServerCommands.fileServerMove()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        LOG.error("node move: failed with InvalidProtocolBufferException " + e
            + " for server " + serverIdString);
        out.addError(new OutputError(Errno.EOPFAILED,
            "node move: failed with InvalidProtocolBufferException " + e
                + " for server " + serverIdString));
        continue;
      }
    }
    return output;
  }

  private CommandOutput fileServerModify() throws CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);

    List<String> hostnames = new ArrayList<String>();
    String hosts = getParamTextValue(ServerCommands.FILESERVER_NODES_PARAM_NAME, 0);
    if (hosts.contains(ServerCommands.MULTI_ARG_SEP)) {
      hostnames.addAll(Arrays.asList(
          hosts.split(ServerCommands.MULTI_ARG_SEP)));
    } else {
      hostnames.add(hosts);
    }

    if (!isParamPresent(BLOCK_IN_MOVES_PARAM_NAME) &&
        !isParamPresent(BLOCK_OUT_MOVES_PARAM_NAME) &&
        !isParamPresent(NUM_SPS_PERINSTANCE_PARAM) &&
        !isParamPresent(NUM_INSTANCES_ON_A_NODE_PARAM) &&
        !isParamPresent(MEMORY_PERINSTANCE_PARAM)) {
      String errMsg = "node modify: invalid arguments. One of "
        + " -" + BLOCK_OUT_MOVES_PARAM_NAME
        + " -" + BLOCK_IN_MOVES_PARAM_NAME
        + " -" + NUM_SPS_PERINSTANCE_PARAM
        + " -" + NUM_INSTANCES_ON_A_NODE_PARAM
        + " -" + MEMORY_PERINSTANCE_PARAM
        + " needs to be specified";
      LOG.error(errMsg);
      out.addError(new OutputError(Errno.EINVAL, errMsg));
      return output;
    }
    FileServerModifyRequest.Builder builder = FileServerModifyRequest.newBuilder();
    builder.setCreds(getUserCredentials());

    if (isParamPresent(BLOCK_OUT_MOVES_PARAM_NAME)) {
      builder.setBlockMovesOut(getParamBooleanValue(BLOCK_OUT_MOVES_PARAM_NAME, 0));
    }
    if (isParamPresent(BLOCK_IN_MOVES_PARAM_NAME)) {
      builder.setBlockMovesIn(getParamBooleanValue(BLOCK_IN_MOVES_PARAM_NAME, 0));
    }

    MfsInstancesInfo.Builder mfsInstancesInfoBuilder = null;
    if (isParamPresent(NUM_SPS_PERINSTANCE_PARAM)) {
      int numSpsPerInstance = getParamIntValue(NUM_SPS_PERINSTANCE_PARAM, 0);
      if (numSpsPerInstance < 0) {
        String errMsg = "Invalid Value for " + NUM_SPS_PERINSTANCE_PARAM + 
            ". Must be greater than or equal to Zero";
        LOG.error(errMsg);
        out.addError(new OutputError(Errno.EINVAL, errMsg));
        return output;
      }
      if (mfsInstancesInfoBuilder == null) {
        mfsInstancesInfoBuilder = MfsInstancesInfo.newBuilder();
      }
      mfsInstancesInfoBuilder.setNumSpsPerInstance(numSpsPerInstance);
    }

    if (isParamPresent(MEMORY_PERINSTANCE_PARAM)) {
      int memoryPerInstance = getParamIntValue(MEMORY_PERINSTANCE_PARAM, 0);
      if (memoryPerInstance <= 0) {
        String errMsg = "Invalid Value for " + MEMORY_PERINSTANCE_PARAM + 
            ". Must be greater than Zero";
        LOG.error(errMsg);
        out.addError(new OutputError(Errno.EINVAL, errMsg));
        return output;
      }
      if (mfsInstancesInfoBuilder == null) {
        mfsInstancesInfoBuilder = MfsInstancesInfo.newBuilder();
      }
      mfsInstancesInfoBuilder.setMemoryPerInstance(memoryPerInstance);
    }
    
    if (isParamPresent(NUM_INSTANCES_ON_A_NODE_PARAM)) {
      int numInstances = getParamIntValue(NUM_INSTANCES_ON_A_NODE_PARAM, 0);
      if (numInstances < 0) {
        String errMsg = "Invalid Value for " + NUM_INSTANCES_ON_A_NODE_PARAM + 
            ". Must be greater than or equal to Zero";
        LOG.error(errMsg);
        out.addError(new OutputError(Errno.EINVAL, errMsg));
        return output;
      }
      if (mfsInstancesInfoBuilder == null) {
        mfsInstancesInfoBuilder = MfsInstancesInfo.newBuilder();
      }
      mfsInstancesInfoBuilder.setNumInstances(numInstances);
    }
    if (mfsInstancesInfoBuilder != null) {
      builder.setMfsInstancesInfo(mfsInstancesInfoBuilder.build());
    }

    FileServerModifyRequest commonReq = builder.build();
    for (String host : hostnames) {
      FileServerModifyRequest req = FileServerModifyRequest
          .newBuilder(commonReq)
          .setHostname(host)
          .build(); 
      byte[] replyData = null;
      try {
        if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
          replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
              getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0),
              Common.MapRProgramId.CldbProgramId.getNumber(),
              CLDBProto.CLDBProg.FileServerModifyProc.getNumber(),
              req, FileServerModifyResponse.class);
        } else {
          replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
              Common.MapRProgramId.CldbProgramId.getNumber(),
              CLDBProto.CLDBProg.FileServerModifyProc.getNumber(),
              req, FileServerModifyResponse.class);
        }
      } catch (Exception e) {
        throw new CLIProcessingException(e);
      }
      if (replyData == null) {
        out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
        continue;
      }
      try {
        FileServerModifyResponse resp = FileServerModifyResponse
            .parseFrom(replyData);
        if (resp.getStatus() != 0) {
          String errMsg;
          if (resp.hasErrMsg()) {
            errMsg = resp.getErrMsg();
          }
          else {
            errMsg = Errno.toString(resp.getStatus());
          }
          String msg = "node modify for " + host + " failed with " + errMsg;
          LOG.error(msg);
          out.addError(new OutputError(resp.getStatus(), msg));
          continue;
        }
      } catch (Exception e) {
        String msg = "node modify for " + host + " failed due to exception " + e.toString();
        LOG.error(msg);
        out.addError(new OutputError(Errno.EOPFAILED, msg));
        continue;
      }
    }
    return output;
  }

  private CommandOutput fileServerAllowIntoCluster() throws CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);
    List<String> serverIds = new ArrayList<String>();
    String fileServerIds = getParamTextValue(
        ServerCommands.HOSTID_PARAM_NAME, 0);

    if (fileServerIds.contains(ServerCommands.MULTI_ARG_SEP)) {
      serverIds.addAll(Arrays.asList(fileServerIds
          .split(ServerCommands.MULTI_ARG_SEP)));
    } else {
      serverIds.add(fileServerIds);
    }

    for (String serverIdString : serverIds) {
      long serverId = -1;
      try {
        serverId = new BigInteger(serverIdString).longValue();
      } catch (NumberFormatException e) {
        /**
        * <MAPR_ERROR>
        * Message:node allow-into-cluster: Invalid hostid specified
        * Function:ServerCommands.fileServerAllowIntoCluster()
        * Meaning:The specified server ID was invalid.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        LOG.error("node allow-into-cluster: Invalid hostid specified");
        out.addError(new OutputError(Errno.EINVAL, "Invalid hostid "
            + serverIdString + " specified"));
        continue;
      }

      FileServerAllowIntoClusterRequest req = FileServerAllowIntoClusterRequest.newBuilder()
                                     .setServerId(serverId)
                                     .setCreds(getUserCredentials())
                                     .build();
      byte[] replyData = null;
      try {
        if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
          replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
              getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0),
              Common.MapRProgramId.CldbProgramId.getNumber(),
              CLDBProto.CLDBProg.FileServerAllowIntoClusterProc.getNumber(), req,
              FileServerAllowIntoClusterResponse.class);
        } else {
          replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
              Common.MapRProgramId.CldbProgramId.getNumber(),
              CLDBProto.CLDBProg.FileServerAllowIntoClusterProc.getNumber(), req,
              FileServerAllowIntoClusterResponse.class);
        }
      } catch (Exception e) {
        throw new CLIProcessingException(e);
      }

      if ( replyData == null ) {
    	  out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
    	  continue;
      }

      try {
        FileServerAllowIntoClusterResponse resp = FileServerAllowIntoClusterResponse.parseFrom(replyData);

        if (resp.getStatus() != 0) {
          /**
          * <MAPR_ERROR>
          * Message:node allow-into-cluster: failed with status: <status> for fileServer <server ID>
          * Function:ServerCommands.fileServerAllowIntoCluster()
          * Meaning:An error occurred.
          * Resolution:Contact technical support.
          * </MAPR_ERROR>
          */
          String errMsg = Errno.toString(resp.getStatus());
          if (resp.hasErrMsg()) errMsg = resp.getErrMsg();
          String msg = "node allow-into-cluster for hostid "
                          + serverId
                          + " failed with "
                          + errMsg;
          LOG.error(msg);
          out.addError(new OutputError(resp.getStatus(), msg));
          continue;
        }
      } catch (Exception e) {
        /**
        * <MAPR_ERROR>
        * Message:node allow-into-cluster: failed with InvalidProtocolBufferException <exception> for server <server ID>
        * Function:ServerCommands.fileServerAllowIntoCluster()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        String msg = "node allow-into-cluster: failed with InvalidProtocolBufferException " 
                        + e
                        + " for hostid " 
                        + serverIdString;
        LOG.error(msg);
        out.addError(new OutputError(Errno.EOPFAILED, msg));
        continue;
      }
    }
    return output;
  }

  private CommandOutput fileServerMaintenance() throws CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);

    if (isParamPresent(ServerCommands.FILESERVER_ID_PARAM_NAME)
        && isParamPresent(ServerCommands.FILESERVER_NODES_PARAM_NAME)) {
      /**
      * <MAPR_ERROR>
      * Message:Please specify either <parameter> or <parameter> not both
      * Function:ServerCommands.fileServerMaintenance()
      * Meaning:Two mutually exclusive parameters were specified.
      * Resolution:Check the command syntax and try again.
      * </MAPR_ERROR>
      */
      out.addError(new OutputError(Errno.EINVAL, "Please specify either "
          + ServerCommands.FILESERVER_NODES_PARAM_NAME + " or "
          + ServerCommands.FILESERVER_ID_PARAM_NAME + " not both"));
      return output;
    }

    if (!isParamPresent(ServerCommands.FILESERVER_ID_PARAM_NAME)
        && !isParamPresent(ServerCommands.FILESERVER_NODES_PARAM_NAME)) {
      /**
      * <MAPR_ERROR>
      * Message:Please specify either <parameter> or <parameter> not both
      * Function:ServerCommands.fileServerMaintenance()
      * Meaning:One of the indicated parameters is required.
      * Resolution:Check the command syntax and try again.
      * </MAPR_ERROR>
      */
      out.addError(new OutputError(Errno.EINVAL, "Please specify either "
          + ServerCommands.FILESERVER_NODES_PARAM_NAME + " or "
          + ServerCommands.FILESERVER_ID_PARAM_NAME
          + " atleast one option is required"));
      return output;
    }

    boolean serverIds = isParamPresent(ServerCommands.FILESERVER_ID_PARAM_NAME);
    String serversString = "";
    List<String> servers = new ArrayList<String>();

    if (serverIds) {
      serversString = getParamTextValue(
          ServerCommands.FILESERVER_ID_PARAM_NAME, 0);
    } else {
      serversString = getParamTextValue(
          ServerCommands.FILESERVER_NODES_PARAM_NAME, 0);
    }

    if (serversString.contains(ServerCommands.MULTI_ARG_SEP)) {
      servers.addAll(Arrays.asList(serversString
          .split(ServerCommands.MULTI_ARG_SEP)));
    } else {
      servers.add(serversString);
    }

    for (String server : servers) {
      int timeOutMinutes = getParamIntValue(
          ServerCommands.FILESERVER_MARKMAINTENANCE_TIMEOUT_MINUTES, 0);

      FileServerMarkMaintenanceRequest.Builder req = FileServerMarkMaintenanceRequest
          .newBuilder()
          .setMaintenanceTimeOutMinutes(timeOutMinutes)
          .setCreds(getUserCredentials());

      if (serverIds) {
        long serverId = -1;
        try {
          serverId = new BigInteger(server).longValue();
        } catch (NumberFormatException e) {
          /**
          * <MAPR_ERROR>
          * Message:node maintenance: Invalid serverId <server ID> specified for server 
          * Function:ServerCommands.fileServerMaintenance()
          * Meaning:The {{serverID}} specified was invalid.
          * Resolution:Check the command syntax and try again.
          * </MAPR_ERROR>
          */
          LOG.error("node maintenance: Invalid serverId"
              + " specified for server " + server);
          out.addError(new OutputError(Errno.EINVAL, "Invalid serverid "
              + server + " specified"));
          continue;
        }
        req.setServerId(serverId);
      } else {
		List<String> ips = NodesCommonUtils.convertHostToIp(Collections.singletonList(server));
		if ( ips.isEmpty() ) {
	         /**
	          * <MAPR_ERROR>
	          * Message:Could not resolve IP for <server>
	          * Function:ServerCommands.fileServerMaintenance()
	          * Meaning:An error occurred.
	          * Resolution:Contact technical support.
	          * </MAPR_ERROR>
	          */
		  out.addError(new OutputError(Errno.EINVAL, "Can not get valid IP address out of provided name: " + server));
		  continue;
	    }
		int hostIP = Util.ipToInt(ips.get(0));
        req.setHostIP(hostIP);
      }

      byte[] replyData = null;
      try {
        if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
          replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
              getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0),
              Common.MapRProgramId.CldbProgramId.getNumber(),
              CLDBProto.CLDBProg.FileServerMarkMaintenanceProc.getNumber(),
              req.build(), FileServerMarkMaintenanceResponse.class);
        } else {
          replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
              Common.MapRProgramId.CldbProgramId.getNumber(),
              CLDBProto.CLDBProg.FileServerMarkMaintenanceProc.getNumber(),
              req.build(), FileServerMarkMaintenanceResponse.class);
        }
      } catch (Exception e) {
        throw new CLIProcessingException(e);
      }

      if ( replyData == null ) {
    	  out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
    	  continue;
      }
      try {
        FileServerMarkMaintenanceResponse resp = FileServerMarkMaintenanceResponse
            .parseFrom(replyData);

        if (resp.getStatus() != 0) {
          /**
          * <MAPR_ERROR>
          * Message:node maintenance: failed with status: <status> for fileServer <server>
          * Function:ServerCommands.fileServerMaintenance()
          * Meaning:An error occurred.
          * Resolution:Contact technical support.
          * </MAPR_ERROR>
          */
          String errMsg = Errno.toString(resp.getStatus());
          if (resp.hasErrMsg()) errMsg = resp.getErrMsg();
          String msg = "node maintenance for fileServer " 
                          + server
                          + " failed with "
                          + errMsg;
          LOG.error(msg);
          out.addError(new OutputError(resp.getStatus(), msg));
          continue;
        }
      } catch (Exception e) {
        /**
        * <MAPR_ERROR>
        * Message:node maintenance: failed with InvalidProtocolBufferException <exception> for server <server>
        * Function:ServerCommands.fileServerMaintenance()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        LOG
            .error("node maintenance: failed with InvalidProtocolBufferException "
                + e + " for server " + server);
        out.addError(new OutputError(Errno.EOPFAILED,
            "node maintenance: failed with InvalidProtocolBufferException " + e
                + " for server " + server));
        continue;
      }
    }
    return output;
  }

  private CommandOutput failoverFileServer() throws CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);

    if (!isParamPresent(ServerCommands.FILESERVER_NODES_PARAM_NAME)) {
      out.addError(new OutputError(Errno.EINVAL, "Please specify one or more "
          + "nodes on which the masters and vips need to be failed over"));
      return output;
    }

    String serversString = "";
    List<String> servers = new ArrayList<String>();

    serversString = getParamTextValue(ServerCommands.FILESERVER_NODES_PARAM_NAME, 0);

    if (serversString.contains(ServerCommands.MULTI_ARG_SEP)) {
      servers.addAll(Arrays.asList(serversString
          .split(ServerCommands.MULTI_ARG_SEP)));
    } else {
      servers.add(serversString);
    }
    boolean failoverLocalContainers = getParamBooleanValue(ServerCommands.LOCAL_CONTAINERS_PARAM_NAME, 0);

    for (String server : servers) {
      FileServerFailoverRequest.Builder req =
        FileServerFailoverRequest.newBuilder();
      req.setCreds(getUserCredentials());
      req.setHostname(server);
      req.setFailoverLocalContainers(failoverLocalContainers);

      byte[] replyData = null;
      try {
        if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
          replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
              getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0),
              Common.MapRProgramId.CldbProgramId.getNumber(),
              CLDBProto.CLDBProg.FileServerFailoverProc.getNumber(),
              req.build(), FileServerFailoverResponse.class);
        } else {
          replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
              Common.MapRProgramId.CldbProgramId.getNumber(),
              CLDBProto.CLDBProg.FileServerFailoverProc.getNumber(),
              req.build(), FileServerFailoverResponse.class);
        }
      } catch (Exception e) {
        throw new CLIProcessingException(e);
      }

      if ( replyData == null ) {
    	  out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
    	  continue;
      }
      try {
        FileServerFailoverResponse resp = FileServerFailoverResponse.parseFrom(replyData);

        if (resp.getStatus() != 0) {
          String errMsg = Errno.toString(resp.getStatus());
          if (resp.hasErrMsg()) errMsg = resp.getErrMsg();
          String msg = "node failover for fileServer " 
                          + server
                          + " failed with "
                          + errMsg;
          LOG.error(msg);
          out.addError(new OutputError(resp.getStatus(), msg));
          continue;
        }
      } catch (Exception e) {
        LOG.error("node failover: failed with InvalidProtocolBufferException "
                + e + " for server " + server);
        out.addError(new OutputError(Errno.EOPFAILED,
            "node failover: failed with InvalidProtocolBufferException " + e
                + " for server " + server));
        continue;
      }
    }
    return output;
  }

  private FileServerListRequest.Builder getFileServerListRequestBuilder()
      throws CLIProcessingException {
    List<Filter> filters = getFilters(fieldTable, FILTER_PARAM_NAME);
    BitSet columns = getColumns();
    ByteString bString = ByteString.copyFrom(BitSetBytesHelperUtils.toByteArray(columns));
    // for backward compatibility with CLDB
    // we will still set setColumns (even it is not really correct ones 
    // and everything on columnsAdd bytes array
    long columnsOld = BitSetBytesHelperUtils.convert(columns);
    boolean alarmedNodes = getParamBooleanValue(ALARMEDNODES_PARAM_NAME, 0);
    boolean nfsNodes = getParamBooleanValue(NFSNODES_PARAM_NAME, 0);

    Limiter limiter = getNextLimiter(getParamIntValue(START_PARAM_NAME, 0), 0,
        getParamIntValue(START_PARAM_NAME, 0), getParamIntValue(
            LIMIT_PARAM_NAME, 0), NUM_NODES_PER_RPC);
    return FileServerListRequest.newBuilder().setCreds(getUserCredentials())
        .addAllFilter(filters).setColumns(columnsOld)
        .setAlarmednodes(alarmedNodes).setNfsnodes(nfsNodes)
        .setLimiter(limiter)
        .setColumnsAdd(bString);
  }

  @Override
  public FileServerListRequest buildNextRequest(MessageLite prevReq,
      MessageLite prevResp) throws CLIProcessingException {
    FileServerListRequest.Builder newReqBuilder = null;
    if (prevReq != null) {
      newReqBuilder = FileServerListRequest
          .newBuilder((FileServerListRequest) prevReq);
    } else {
      newReqBuilder = getFileServerListRequestBuilder();
    }

    if (prevResp != null) {
      int prevStart = newReqBuilder.getLimiter().getStart();
      int prevCount = ((FileServerListResponse) prevResp).getInfoCount();
      int origStart = getParamIntValue(START_PARAM_NAME, 0);
      int origLimit = getParamIntValue(LIMIT_PARAM_NAME, 0);
      newReqBuilder.setLimiter(getNextLimiter(prevStart, prevCount, origStart,
          origLimit, NUM_NODES_PER_RPC));
    }

    return newReqBuilder.build();
  }

  @Override
  public boolean hasMore(MessageLite prevReq, MessageLite prevResp)
      throws CLIProcessingException {
    return hasMore(getParamIntValue(START_PARAM_NAME, 0), getParamIntValue(
        LIMIT_PARAM_NAME, 0), ((FileServerListRequest) prevReq).getLimiter()
        .getStart(), ((FileServerListResponse) prevResp).getInfoCount());
  }

  @Override
  public void processResponse(OutputHierarchy out, MessageLite resp)
      throws CLIProcessingException {
    BitSet columns = getColumns();
    boolean terse = getParamTextValue(OUTPUT_PARAM_NAME, 0).equals("terse");

    FileServerListResponse response = (FileServerListResponse) resp;
    for (FileServerInfo fs : response.getInfoList()) {
        out.addNode(formatFileServerInfo(fs, terse, columns));
    }

    if (response.hasTotal()) {
      out.setTotal(response.getTotal());
    }
  }
  @Override
  public FileServerListResponse sendRequest(MessageLite request)
      throws CLIProcessingException {
    FileServerListRequest req = (FileServerListRequest) request;
    byte[] replyData = null;
    try {
      if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
            getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0),
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg.FileServerListProc.getNumber(), req,
            FileServerListResponse.class);
      } else {
        replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg.FileServerListProc.getNumber(), req,
            FileServerListResponse.class);
      }
    } catch (Exception e) {
      throw new CLIProcessingException(e);
    }

    if (replyData != null) {
      return getFileServerListResponse(replyData);
    } else {
      /**
      * <MAPR_ERROR>
      * Message:RPC Request to list Nodes failed. No data returned
      * Function:ServerCommands.sendRequest()
      * Meaning:A communication error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      LOG.error("RPC Request to list Nodes failed. No data returned");
      return null;
    }
  }

  private FileServerListResponse getFileServerListResponse(byte[] replyData)
      throws CLIProcessingException {
    try {
      return FileServerListResponse.parseFrom(replyData);
    } catch (InvalidProtocolBufferException ipbe) {
      /**
      * <MAPR_ERROR>
      * Message:Exception while parsing the RPC response data into FileServerListResponse proto object.
      * Function:ServerCommands.sendRequest()
      * Meaning:An exception occurred while parsing the response.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      throw new CLIProcessingException("Exception while parsing the RPC "
          + "response data into FileServerListResponse proto object.", ipbe);
    }
  }

  private void getServicesInfo() throws CLIProcessingException {
	String zkConnectString =  null;
	  
	if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
		zkConnectString = CLDBRpcCommonUtils.getInstance().getZkConnect(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0));
	} else {
    	zkConnectString = CLDBRpcCommonUtils.getInstance().getZkConnect();
    }
	
    if (zkConnectString == null || zkConnectString.trim().isEmpty()) {
      if (isParamPresent(NodesCommonUtils.ZK_CONNECTSTRING))
        zkConnectString = getParamTextValue(NodesCommonUtils.ZK_CONNECTSTRING,
            0);
    }
    if (zkConnectString == null || zkConnectString.trim().isEmpty()) {
      LOG.error("zkConnectString is null/empty. Cannot proceed further.");
      return;
    }
    try {
      if (LOG.isDebugEnabled()) {
        LOG.debug("getServicesInfo start");
      }
      runningServices = NodesCommonUtils
          .findServicesRunningHierarchy(zkConnectString);
      configuredServices = NodesCommonUtils
          .findServicesConfiguredHierarchy(zkConnectString);
    } catch (Throwable t) {
      /**
      * <MAPR_ERROR>
      * Message:Error while trying to fetch services info from ZK. Check ZK connection
      * Function:ServerCommands.getServicesInfo()
      * Meaning:An error occurred when trying to retrieve information from ZooKeeper.
      * Resolution:Make sure that ZooKeeper has a quorum and is reachable over the network, 
      * and that you have run configure.sh with valid ZooKeeper hostnames or IP addresses.
      * </MAPR_ERROR>
      */
      LOG.error("Error while trying to fetch services info from ZK. Check ZK connection", t);
    }
    if (LOG.isDebugEnabled()) {
      LOG.debug("getServicesInfo end");
    }
  }

  private BitSet getColumns() throws CLIProcessingException {
    BitSet columns = new BitSet();
    // set all bits to 1 for whole filedMap
    columns.set(0, MAX_NODEFIELDINFO+1);
    //long columns = 0xffffffff;

    String columnsString = getParamTextValue(COLUMNS_PARAM_NAME, 0);
    if (columnsString != null && (!columnsString.equals("all"))) {
      if (!columnsString.contains("ip")) {
        // needed to get services status info
        columnsString = columnsString.concat(",ip");
      }
      if ( !columnsString.contains("hn") && !columnsString.contains("hostname")) {
          // needed to get services status info
          columnsString = columnsString.concat(",hn");  
      }

      columns = FilterUtil.getColumns(fieldTable, columnsString.trim());
    }
    return columns;
  }
  
  public static int formatNodeState(NodeState s) {
    String str = s.toString();
    if (str.startsWith("CRITICAL"))
      return 4;
    if (str.startsWith("MAINTENANCE")) {
      return 3;
    }
    if (str.startsWith("DRAINING"))
      return 2;
    if (str.startsWith("HEALTHY"))
      return 0;
    return 5;
  }
  
  private OutputNode formatFileServerInfo(FileServerInfo fs, boolean terse, BitSet columns) {
    FileServerHeartbeatStats hb = fs.getHbStats();
    OutputNode serverInfo = new OutputNode();
    String runningServicesString = null;
    String configuredServicesString = null;
    List<PluggableAlarm> pluggableAlarms = null;

    try {
      pluggableAlarms = PluggableAlarmUtil.getNodeAlarms(getUserCredentials(), 0, 50);
    } catch (CLIProcessingException e) {
      LOG.error("Error getting alarms. Alarm message might be incomplete: " + e.getMessage());
    }


    if (columns.get(NodeInfo.Id.getNumber()) ) {
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.Id)).getName(
          terse), Long.toString(fs.getFileServerId())));
    }
    if (columns.get(NodeInfo.Ip.getNumber()) ) {
      Set<String> ipSet = new HashSet<String>(fs.getAddressList().size());
      for (Common.IPAddress ip : fs.getAddressList()) {
        String ipA = Util.intToIp(ip.getHost());
        if (!ipSet.add(ipA)) {
          /* each vip is reported with same ip addresses,
           * skip displaying same details more than once.*/
          continue;
        }
        serverInfo.addChild(new IpAddressNode(fieldTable.get(new NodeField(NodeInfo.Ip)).getName(
            terse), Util.intToIp(ip.getHost())));
        if (runningServices != null) {
          List<String> services = runningServices.get(ipA);
          if (services == null) {
            services = new ArrayList<String>();
          }
          if (fs.hasLoopbackNfsRunning() && fs.getLoopbackNfsRunning()) {
            services.add("loopbacknfs");
          }
          if (fs.hasFuseClientRunning() && fs.getFuseClientRunning()) {
            String svcName = formatFuseClientInfo(fs);
            if (svcName != null) {
              services.add(svcName);
            }
          }
          if (fs.getNfsInstanceInfoList().size() > 0) {
            services.addAll(formatNFSInstanceInfo(fs.getNfsInstanceInfoList(), true /* getRunningOnly */));
          }

          if (!services.isEmpty()) {
            StringBuilder sb = new StringBuilder();
            for (String service : services) {
              sb.append(service);
              sb.append(",");
            }
            runningServicesString = sb.deleteCharAt(sb.length() - 1).toString();
          }
        }
        if (configuredServices != null) {
          List<String> services = configuredServices.get(ipA);
          if (services == null) {
            services = new ArrayList<String>();
          }
          if (fs.getLoopbackNfsConfigured()) {
            services.add("loopbacknfs");
          }
          if (fs.hasPosixClientInfo()) {
            String svcName = formatFuseClientInfo(fs);
            if (svcName != null) {
              services.add(svcName);
            }
          }
          if (fs.getNfsInstanceInfoList().size() > 0) {
            services.addAll(formatNFSInstanceInfo(fs.getNfsInstanceInfoList(), false /* getRunningOnly */));
          }
          if (!services.isEmpty()) {
            StringBuilder sb = new StringBuilder();
            for (String service : services) {
              sb.append(service);
              sb.append(",");
            }
            configuredServicesString = sb.deleteCharAt(sb.length() - 1).toString();
          }
        }
      }
    }
    if (columns.get(NodeInfo.Hostname.getNumber()) ) {
    	String hostName = MapRCliUtil.getHostname(fs);
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.Hostname))
          .getName(terse), hostName));
      if (runningServices != null) {
        List<String> services = runningServices.get(hostName);
        if (services == null) {
          services = new ArrayList<String>();
        }
        if (fs.hasLoopbackNfsRunning() && fs.getLoopbackNfsRunning()) {
          services.add("loopbacknfs");
        }
        if (fs.hasFuseClientRunning() && fs.getFuseClientRunning()) {
          String svcName = formatFuseClientInfo(fs);
          if (svcName != null) {
            services.add(svcName);
          }
        }
        if (fs.getNfsInstanceInfoList().size() > 0) {
          services.addAll(formatNFSInstanceInfo(fs.getNfsInstanceInfoList(), true /* getRunningOnly */));
        }
        if (!services.isEmpty()) {
          StringBuilder sb = new StringBuilder();
          for (String service : services) {
            sb.append(service);
            sb.append(",");
          }
          runningServicesString = sb.deleteCharAt(sb.length() - 1).toString();
        }
      }
      if (configuredServices != null) {
        List<String> services = configuredServices.get(hostName);
        if (services == null) {
          services = new ArrayList<String>();
        }
        if (fs.getLoopbackNfsConfigured()) {
          services.add("loopbacknfs");
        }
        if (fs.hasPosixClientInfo()) {
          String svcName = formatFuseClientInfo(fs);
          if (svcName != null) {
            services.add(svcName);
          }
        }
        if (fs.getNfsInstanceInfoList().size() > 0) {
          services.addAll(formatNFSInstanceInfo(fs.getNfsInstanceInfoList(), false /* getRunningOnly */));
        }
        if (!services.isEmpty()) {
          StringBuilder sb = new StringBuilder();
          for (String service : services) {
            sb.append(service);
            sb.append(",");
          }
          configuredServicesString = sb.deleteCharAt(sb.length() - 1).toString();
        }
      }
    }
    if (columns.get(NodeInfo.RackPath.getNumber())) {
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.RackPath))
          .getName(terse), fs.getNetworkLocation()));
    }
    /*
     * if ((columns & (1L << NodeInfo.SwitchPath.getNumber())) != 0) { // switch
     * DUMMY serverInfo.addChild(new
     * OutputNode(fieldTable.get(new NodeField(NodeInfo.SwitchPath)) .getName(terse),
     * "/default-switch/" + Long.toString(fs.getFileServerId()))); }
     */
    if (columns.get(NodeInfo.Status.getNumber())) {
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.Status))
          .getName(terse), formatNodeState(fs.getNodeState())));
    }
    if (columns.get(NodeInfo.StatusDesc.getNumber())) {
      String desc = nodeStateDesc.get(fs.getNodeState());
      if (desc == null || desc.trim().isEmpty())
       desc = "Unknown reason";
      
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.StatusDesc))
          .getName(terse), desc));
    }    
    if (columns.get(NodeInfo.Services.getNumber())) {
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.Services))
          .getName(terse), (runningServicesString == null ? "" : runningServicesString))); // service
    }
    if (columns.get(NodeInfo.ConfiguredServices.getNumber())) {
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.ConfiguredServices))
          .getName(terse), (configuredServicesString == null ? "" : configuredServicesString))); // service
    }
    if (columns.get(NodeInfo.FSHB.getNumber())) {
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.FSHB)).getName(
          terse), fs.getLastHeartbeatSec())); // fs-hb
    }
    if (columns.get(NodeInfo.JTHB.getNumber())) {
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.JTHB)).getName(
          terse), 2)); // jt-hb DUMMY
    }
    if (columns.get(NodeInfo.DiskTotal.getNumber())) {
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.DiskTotal))
          .getName(terse), hb.getServerCapacitySizeMB() / 1024));
    }
    if (columns.get(NodeInfo.DiskUsed.getNumber())) {
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.DiskUsed))
          .getName(terse), hb.getServerUsedSizeMB() / 1024));
    }
    if (columns.get(NodeInfo.DiskAvail.getNumber())) {
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.DiskAvail))
          .getName(terse), hb.getServerAvailableSizeMB() / 1024));
    }
    if (columns.get(NodeInfo.Rpc.getNumber())) {
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.Rpc)).getName(
          terse), hb.getRpcCount()));
    }
    if (columns.get(NodeInfo.RpcIn.getNumber())) {
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.RpcIn))
          .getName(terse), hb.getRpcInBytes()));
    }
    if (columns.get(NodeInfo.RpcOut.getNumber())) {
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.RpcOut))
          .getName(terse), hb.getRpcOutBytes()));
    }
    if (columns.get(NodeInfo.DiskCount.getNumber())) {
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.DiskCount))
          .getName(terse), hb.getDiskCount()));
    }
    if (columns.get(NodeInfo.MapRDiskCount.getNumber())) {
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.MapRDiskCount))
          .getName(terse), hb.getMaprdiskCount()));
    }
    if (columns.get(NodeInfo.FailedDisks.getNumber())) {
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.FailedDisks))
          .getName(terse), hb.getFaileddisks()));
    }
    if (columns.get(NodeInfo.DiskReadOps.getNumber())) {
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.DiskReadOps))
          .getName(terse), hb.getDiskReadOps()));
    }
    if (columns.get(NodeInfo.DiskReadKbytes.getNumber())) {
      serverInfo
          .addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.DiskReadKbytes))
              .getName(terse), hb.getDiskReadKBytes()));
    }
    if (columns.get(NodeInfo.DiskWriteOps.getNumber())) {
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.DiskWriteOps))
          .getName(terse), hb.getDiskWriteOps()));
    }
    if (columns.get(NodeInfo.DiskWriteKbytes.getNumber())) {
      serverInfo.addChild(new OutputNode(fieldTable.get(
          new NodeField(NodeInfo.DiskWriteKbytes)).getName(terse), hb.getDiskWriteKBytes()));
    }
    if (columns.get(NodeInfo.CpuCount.getNumber())) {
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.CpuCount))
          .getName(terse), hb.getCpuCount()));
    }
    if (columns.get(NodeInfo.CpuUtil.getNumber())) {
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.CpuUtil))
          .getName(terse), 100 - hb.getCpuIdle()));
    }
    if (columns.get(NodeInfo.CpuUptime.getNumber())) {
      Object uptime = terse ? Long.valueOf(hb.getCpuUptime()) : new Date(hb
          .getCpuUptime()).toString();
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.CpuUptime))
          .getName(terse), uptime));
    }
    if (columns.get(NodeInfo.MemTotal.getNumber())) {
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.MemTotal))
          .getName(terse), hb.getMemoryTotalMB()));
    }
    if (columns.get(NodeInfo.MemUsed.getNumber())) {
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.MemUsed))
          .getName(terse), hb.getMemoryUsedMB()));
    }
    if (columns.get(NodeInfo.TTMapSlotsTotal.getNumber())) {
      serverInfo.addChild(new OutputNode(fieldTable.get(
          new NodeField(NodeInfo.TTMapSlotsTotal)).getName(terse), hb.getTtMapSlots())); // DUMMY
    }
    if (columns.get(NodeInfo.TTMapSlotsOccupied.getNumber())) {
      serverInfo.addChild(new OutputNode(fieldTable.get(
          new NodeField(NodeInfo.TTMapSlotsOccupied)).getName(terse), hb.getTtMapUsed())); // DUMMY
    }
    if (columns.get(NodeInfo.TTReduceSlotsTotal.getNumber())) {
      serverInfo.addChild(new OutputNode(fieldTable.get(
          new NodeField(NodeInfo.TTReduceSlotsTotal)).getName(terse), hb.getTtReduceSlots())); // DUMMY
    }
    if (columns.get(NodeInfo.TTReduceSlotsOccupied.getNumber())) {
      serverInfo
          .addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.TTReduceSlotsOccupied)).getName(terse), hb
              .getTtReduceUsed())); // DUMMY
    }
    if (columns.get(NodeInfo.BytesReceived.getNumber())) {
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.BytesReceived))
          .getName(terse), hb.getNetworkBytesRecd()));
    }
    if (columns.get(NodeInfo.BytesSent.getNumber())) {
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.BytesSent))
          .getName(terse), hb.getNetworkBytesXmit()));
    }
    if (hb.hasNumResyncSlots() &&
        columns.get(NodeInfo.NumResyncSlots.getNumber())) {
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.NumResyncSlots))
          .getName(terse), hb.getNumResyncSlots()));
    }
    if (columns.get(NodeInfo.BlockMovesOut.getNumber())) {
      boolean blockMovesOut = false;
      if (fs.hasBlockMovesOut()) blockMovesOut = fs.getBlockMovesOut();
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.BlockMovesOut))
          .getName(terse), blockMovesOut));
    }
    if (columns.get(NodeInfo.BlockMovesIn.getNumber())) {
      boolean blockMovesIn = false;
      if (fs.hasBlockMovesIn()) blockMovesIn = fs.getBlockMovesIn();
      serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.BlockMovesIn))
          .getName(terse), blockMovesIn));
    }
    if (columns.get(NodeInfo.NumInstances.getNumber()) ||
        columns.get(NodeInfo.NumSpsPerInstance.getNumber())) {
      MfsInstancesInfo tmp = fs.getMfsInstancesInfo();

      if (columns.get(NodeInfo.NumInstances.getNumber())) {
        serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.NumInstances))
            .getName(terse), (tmp.hasNumInstances()
                              ? String.valueOf(tmp.getNumInstances())
                              : "N/A")));
      }
      if (columns.get(NodeInfo.NumSpsPerInstance.getNumber())) {
        serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.NumSpsPerInstance))
            .getName(terse), (tmp.hasNumSpsPerInstance() 
                              ? String.valueOf(tmp.getNumSpsPerInstance())
                              : "N/A")));
      }
    }
    if (fs.hasDbStats()) {

      if (columns.get(NodeInfo.DbPuts10s.getNumber())) {
        serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.DbPuts10s))
            .getName(terse), getDbStatValue(NodeInfo.DbPuts10s, fs.getDbStats())));
      }
      if (columns.get(NodeInfo.DbPuts1m.getNumber())) {
        serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.DbPuts1m))
            .getName(terse), getDbStatValue(NodeInfo.DbPuts1m, fs.getDbStats())));
      }
      if (columns.get(NodeInfo.DbPuts5m.getNumber())) {
        serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.DbPuts5m))
            .getName(terse), getDbStatValue(NodeInfo.DbPuts5m, fs.getDbStats())));
      }
      if (columns.get(NodeInfo.DbPuts15m.getNumber())) {
        serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.DbPuts15m))
            .getName(terse), getDbStatValue(NodeInfo.DbPuts15m, fs.getDbStats())));
      }
      if (columns.get(NodeInfo.DbGets10s.getNumber())) {
        serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.DbGets10s))
            .getName(terse), getDbStatValue(NodeInfo.DbGets10s, fs.getDbStats())));
      }
      if (columns.get(NodeInfo.DbGets1m.getNumber())) {
        serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.DbGets1m))
            .getName(terse), getDbStatValue(NodeInfo.DbGets1m, fs.getDbStats())));
      }
      if (columns.get(NodeInfo.DbGets5m.getNumber())) {
        serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.DbGets5m))
            .getName(terse), getDbStatValue(NodeInfo.DbGets5m, fs.getDbStats())));
      }
      if (columns.get(NodeInfo.DbGets15m.getNumber())) {
        serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.DbGets15m))
            .getName(terse), getDbStatValue(NodeInfo.DbGets15m, fs.getDbStats())));
      }
      if (columns.get(NodeInfo.DbScans10s.getNumber())) {
        serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.DbScans10s))
            .getName(terse), getDbStatValue(NodeInfo.DbScans10s, fs.getDbStats())));
      }
      if (columns.get(NodeInfo.DbScans1m.getNumber())) {
        serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.DbScans1m))
            .getName(terse), getDbStatValue(NodeInfo.DbScans1m, fs.getDbStats())));
      }
      if (columns.get(NodeInfo.DbScans5m.getNumber())) {
        serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.DbScans5m))
            .getName(terse), getDbStatValue(NodeInfo.DbScans5m, fs.getDbStats())));
      }
      if (columns.get(NodeInfo.DbScans15m.getNumber())) {
        serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.DbScans15m))
            .getName(terse), getDbStatValue(NodeInfo.DbScans15m, fs.getDbStats())));
      }
    }

    // Alarms Information
    for (AlarmMsg fsAlarm : fs.getFsAlarmsList()) {
      if (fsAlarm.getAlarmId().compareTo(AlarmId.NODE_ALARM_DEBUG_LOGGING) == 0) {
        if (columns.get(NodeInfo.LogLevelAlarm.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.LogLevelAlarm)).getName(terse),
              fsAlarm.getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
      } else if (fsAlarm.getAlarmId().compareTo(
          AlarmId.NODE_ALARM_SERVICE_CLDB_DOWN) == 0) {
        if (columns.get(NodeInfo.ServiceCLDBDownAlarm.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.ServiceCLDBDownAlarm)).getName(terse), fsAlarm
              .getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
      } else if (fsAlarm.getAlarmId().compareTo(
          AlarmId.NODE_ALARM_SERVICE_FILESERVER_DOWN) == 0) {
        if (columns.get(NodeInfo.ServiceFileserverDownAlarm.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.ServiceFileserverDownAlarm)).getName(terse), fsAlarm
              .getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
      } else if (fsAlarm.getAlarmId().compareTo(
          AlarmId.NODE_ALARM_SERVICE_JT_DOWN) == 0) {
        if (columns.get(NodeInfo.ServiceJTDownAlarm.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.ServiceJTDownAlarm)).getName(terse), fsAlarm
              .getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
      } else if (fsAlarm.getAlarmId().compareTo(
          AlarmId.NODE_ALARM_SERVICE_TT_DOWN) == 0) {
        if (columns.get(NodeInfo.ServiceTTDownAlarm.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.ServiceTTDownAlarm)).getName(terse), fsAlarm
              .getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
      } else if (fsAlarm.getAlarmId().compareTo(
          AlarmId.NODE_ALARM_SERVICE_HBMASTER_DOWN) == 0) {
        if (columns.get(NodeInfo.ServiceHBMasterDownAlarm.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.ServiceHBMasterDownAlarm)).getName(terse), fsAlarm
              .getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
      } else if (fsAlarm.getAlarmId().compareTo(
          AlarmId.NODE_ALARM_SERVICE_HBREGION_DOWN) == 0) {
        if (columns.get(NodeInfo.ServiceHBRegionDownAlarm.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.ServiceHBRegionDownAlarm)).getName(terse), fsAlarm
              .getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
      } else if (fsAlarm.getAlarmId().compareTo(
          AlarmId.NODE_ALARM_SERVICE_NFS_DOWN) == 0) {
        if (columns.get(NodeInfo.ServiceNFSDownAlarm.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.ServiceNFSDownAlarm)).getName(terse), fsAlarm
              .getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
      } else if (fsAlarm.getAlarmId().compareTo(
          AlarmId.NODE_ALARM_SERVICE_WEBSERVER_DOWN) == 0) {
        if (columns.get(NodeInfo.ServiceWebserverDownAlarm.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.ServiceWebserverDownAlarm)).getName(terse), fsAlarm
              .getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
      } else if (fsAlarm.getAlarmId().compareTo(
          AlarmId.NODE_ALARM_SERVICE_HOSTSTATS_DOWN) == 0) {
        if (columns.get(NodeInfo.ServiceHoststatsDownAlarm.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.ServiceHoststatsDownAlarm)).getName(terse), fsAlarm
              .getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
      } else if (fsAlarm.getAlarmId()
          .compareTo(AlarmId.NODE_ALARM_DISK_FAILURE) == 0) {
        if (columns.get(NodeInfo.DiskFailureAlarm.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.DiskFailureAlarm)).getName(terse), fsAlarm
              .getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
      } else if (fsAlarm.getAlarmId().compareTo(
          AlarmId.NODE_ALARM_VERSION_MISMATCH) == 0) {
        if (columns.get(NodeInfo.VersionMismatchAlarm.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.VersionMismatchAlarm)).getName(terse), fsAlarm
              .getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
      } else if (fsAlarm.getAlarmId().compareTo(AlarmId.NODE_ALARM_TIME_SKEW) == 0) {
        if (columns.get(NodeInfo.TimeSkewAlarm.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.TimeSkewAlarm)).getName(terse),
              fsAlarm.getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
      } else if (fsAlarm.getAlarmId().compareTo(AlarmId.NODE_ALARM_HB_PROCESSING_SLOW) == 0) {
        if (columns.get(NodeInfo.HbProcessingSlow.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.HbProcessingSlow)).getName(terse),
              fsAlarm.getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
      } else if (fsAlarm.getAlarmId().compareTo(
          AlarmId.NODE_ALARM_ROOT_PARTITION_FULL) == 0) {
        if (columns.get(NodeInfo.NodeRootPartitionFull.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.NodeRootPartitionFull)).getName(terse), fsAlarm
              .getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
      } else if (fsAlarm.getAlarmId().compareTo(
          AlarmId.NODE_ALARM_OPT_MAPR_FULL) == 0) {
        if (columns.get(NodeInfo.NodeOptMapRFull.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.NodeOptMapRFull)).getName(terse),
              fsAlarm.getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
      } else if (fsAlarm.getAlarmId()
          .compareTo(AlarmId.NODE_ALARM_CORE_PRESENT) == 0) {
        if (columns.get(NodeInfo.NodeCorePresent.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.NodeCorePresent)).getName(terse),
              fsAlarm.getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
      } else if (fsAlarm.getAlarmId()
          .compareTo(AlarmId.NODE_ALARM_HIGH_MFS_MEMORY) == 0) {
        if (columns.get(NodeInfo.NodeHighMfsMemory.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.NodeHighMfsMemory)).getName(terse),
              fsAlarm.getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
      } else if (fsAlarm.getAlarmId()
          .compareTo(AlarmId.NODE_ALARM_PAM_MISCONFIGURED) == 0) {
        if (columns.get(NodeInfo.NodePamMisconfigured.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.NodePamMisconfigured)).getName(terse),
              fsAlarm.getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
      } else if (fsAlarm.getAlarmId()
          .compareTo(AlarmId.NODE_ALARM_TT_LOCALDIR_FULL) == 0) {
        if (columns.get(NodeInfo.NodeTTLocaldirFull.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.NodeTTLocaldirFull)).getName(terse),
              fsAlarm.getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
      } else if (fsAlarm.getAlarmId()
          .compareTo(AlarmId.NODE_ALARM_NO_HEARTBEAT) == 0) {
        if (columns.get(NodeInfo.NodeNoHeartbeat.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.NodeNoHeartbeat)).getName(terse),
              fsAlarm.getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
      } else if (fsAlarm.getAlarmId()
          .compareTo(AlarmId.NODE_ALARM_MAPRUSER_MISMATCH) == 0) {
        if (columns.get(NodeInfo.NodeMaprUserMismatch.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
            new NodeField(NodeInfo.NodeMaprUserMismatch)).getName(terse),
            fsAlarm.getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
      } else if (fsAlarm.getAlarmId()
          .compareTo(AlarmId.NODE_ALARM_DUPLICATE_HOSTID) == 0) {
        if (columns.get(NodeInfo.NodeDuplicateHostId.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.NodeDuplicateHostId)).getName(terse),
              fsAlarm.getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
      } else if (fsAlarm.getAlarmId()
          .compareTo(AlarmId.NODE_ALARM_METRICS_WRITE_PROBLEM) == 0) {
        if (columns.get(NodeInfo.NodeMetricsWriteProblemAlarm.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.NodeMetricsWriteProblemAlarm)).getName(terse),
              fsAlarm.getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
      } else if (fsAlarm.getAlarmId()
          .compareTo(AlarmId.NODE_ALARM_TOO_MANY_CONTAINERS) == 0) {
        if (columns.get(NodeInfo.NodeTooManyContainersAlarm.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.NodeTooManyContainersAlarm)).getName(terse),
              fsAlarm.getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
      } else if (fsAlarm.getAlarmId()
          .compareTo(AlarmId.NODE_ALARM_INCORRECT_TOPOLOGY_ALARM) == 0) {
        if (columns.get(NodeInfo.IncorrectTopologyAlarm.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.IncorrectTopologyAlarm)).getName(terse),
              fsAlarm.getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
    } else if (fsAlarm.getAlarmId()
          .compareTo(AlarmId.NODE_ALARM_SERVICE_HUE_DOWN) == 0) {
        if (columns.get(NodeInfo.ServiceHueDownAlarm.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.ServiceHueDownAlarm)).getName(terse),
              fsAlarm.getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
    } else if (fsAlarm.getAlarmId()
          .compareTo(AlarmId.NODE_ALARM_SERVICE_HTTPFS_DOWN) == 0) {
        if (columns.get(NodeInfo.ServiceHttpfsDownAlarm.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.ServiceHttpfsDownAlarm)).getName(terse),
              fsAlarm.getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
    } else if (fsAlarm.getAlarmId()
          .compareTo(AlarmId.NODE_ALARM_SERVICE_BEESWAX_DOWN) == 0) {
        if (columns.get(NodeInfo.ServiceBeeswaxDownAlarm.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.ServiceBeeswaxDownAlarm)).getName(terse),
              fsAlarm.getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
    } else if (fsAlarm.getAlarmId()
          .compareTo(AlarmId.NODE_ALARM_SERVICE_HIVEMETA_DOWN) == 0) {
        if (columns.get(NodeInfo.ServiceHiveMetaDownAlarm.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.ServiceHiveMetaDownAlarm)).getName(terse),
              fsAlarm.getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
    } else if (fsAlarm.getAlarmId()
          .compareTo(AlarmId.NODE_ALARM_SERVICE_HS2_DOWN) == 0) {
        if (columns.get(NodeInfo.ServiceHs2DownAlarm.getNumber())) {
          serverInfo.addChild(new OutputNode(fieldTable.get(
              new NodeField(NodeInfo.ServiceHs2DownAlarm)).getName(terse),
              fsAlarm.getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
        }
    } else if (fsAlarm.getAlarmId()
        .compareTo(AlarmId.NODE_ALARM_SERVICE_OOZIE_DOWN) == 0) {
      if (columns.get(NodeInfo.ServiceOozieDownAlarm.getNumber())) {
        serverInfo.addChild(new OutputNode(fieldTable.get(
            new NodeField(NodeInfo.ServiceOozieDownAlarm)).getName(terse),
            fsAlarm.getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
      }
    } else {
        // Pluggable alarm
        for (PluggableAlarm pluggableAlarm : pluggableAlarms) {
          if (fsAlarm.getAlarmName().equals(pluggableAlarm.getName())) {
            serverInfo.addChild(new OutputNode(terse ? pluggableAlarm.getTerse() : pluggableAlarm.getDisplayName(),
                fsAlarm.getAlarmState() ? fsAlarm.getAlarmTimeStamp() : 0));
            break;
          }
        }
    }
  }

    // Virtual IPs
    if (columns.get(NodeInfo.VirtualIp.getNumber())) {
      for (Common.IPAddress ip : fs.getAddressList()) {
        if (!ip.hasVirtualIP()) {
          continue;
        }

        String ipA = Util.longToIp(ip.getVirtualIP());
        serverInfo.addChild(new OutputNode(fieldTable.get(new NodeField(NodeInfo.VirtualIp))
            .getName(terse), ipA));
      }
    }
    return serverInfo;
  }

  List<String> formatNFSInstanceInfo(List<NFSInstanceInfo> nfsiInfos, boolean getRunningOnly) {
    List<String> names = new ArrayList<String>();
    for (NFSInstanceInfo nfsi : nfsiInfos) {
      if (getRunningOnly) {
        if (nfsi.hasLoopbackNfsRunning() && nfsi.getLoopbackNfsRunning()) {
          names.add("loopbacknfs");
        }
        if (nfsi.hasFuseClientRunning() && nfsi.getFuseClientRunning()) {
          names.add(posixclientTypeToString(nfsi.getPosixClientType()));
        }
      } else {
        if (nfsi.hasLoopbackNfsConfigured() && nfsi.getLoopbackNfsConfigured()) {
          names.add("loopbacknfs");
        }
        if (nfsi.hasPosixClientType()) {
          names.add(posixclientTypeToString(nfsi.getPosixClientType()));
        }
      }
    }
    return names;
  }

  String formatFuseClientInfo(FileServerInfo fs) {
    if (!fs.hasPosixClientInfo()) {
      return null;
    }
    PosixClientInfo pcInfo = fs.getPosixClientInfo();
    return posixclientTypeToString(pcInfo.getClientType());
  }

  String posixclientTypeToString (PosixClientType type) {
    String svcName = "posixclient";
    switch (type) {
      case BASIC:
        svcName += "basic";
        break;
      case GOLD:
        svcName += "gold";
        break;
      case PLATINUM:
        svcName += "platinum";
        break;
      default:
        svcName = null;
        break;
    }
    return svcName;
  }
  
  private ClusterInfoResponse doClusterInfo(OutputHierarchy out, 
  		                                      boolean needCldbList) {  	
  	try {
  		ClusterInfoRequest req = ClusterInfoRequest.newBuilder()
                                               .setCreds(getUserCredentials())
                                               .setColumns(0xffffffff)
                                               .setNeedCldbList(needCldbList)
                                               .build();
  		byte[] data;
  		
  		if (isParamPresent(CLDB_PARAM_NAME)) {
  			int cldbPort = MapRCliUtil.CLDB_DEFAULT_PORT;
  			
  			String cldbHostStr = getParamTextValue(CLDB_PARAM_NAME, 0);
  		  if (cldbHostStr == null || cldbHostStr.isEmpty())
  		   return null;
  		  
  		  // Parse port #
  		  if (cldbHostStr.contains(":")) {
  		  	String [] arr = cldbHostStr.split(":");
  		  	cldbHostStr = arr[0];
  		  	cldbPort = Integer.valueOf(arr[1]);
  		  	if (cldbPort <= 0) 
  		  		cldbPort = MapRCliUtil.CLDB_DEFAULT_PORT;
  		  }
  		   
  		  List<String> ips = NodesCommonUtils.convertHostToIp(Collections.singletonList(cldbHostStr));
		    if ( ips.isEmpty() ) {
		      out.addError(new OutputError(Errno.EINVAL, 
		      		"Can not get valid IP address out of provided name: " + cldbHostStr));
		    	return null;
		    }
		    cldbHostStr = ips.get(0);
		     
   			int cldbHost = Util.ipToInt(cldbHostStr);
		    try {
		      int port = Rpc.initialize(0, 0, null); // Client
		      if (port < 0) {
		  	    out.addError(new OutputError(Errno.ERPCFAILED, "Failed to initialize Rpc"));
		  	    return null;
		      }
		    } catch (Exception e) {
		       /**
		         * <MAPR_ERROR>
		         *  Failed to Initialze RPC
		         * </MAPR_ERROR>
		         */
		       out.addError(new OutputError(Errno.ERPCFAILED, "Failed to initialize Rpc"));
		       return null;
		    }
		    
                    // TODO: Need to figure out which cluster we are
                    // talking to. Given the ip, we need to map it back
                    // to cluster name and use that cluster's ticket to
                    // talk to this CLDB.
		    long binding = Rpc.createBindingFor(cldbHost, cldbPort, null, ServerKeyType.ServerKey.getNumber());

		    data = Rpc.sendRequest(binding,
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg.ClusterInfoProc.getNumber(),
            req);
  		} else if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
  	    data = CLDBRpcCommonUtils.getInstance().sendRequest (
  	               getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
  	               Common.MapRProgramId.CldbProgramId.getNumber(),
  	               CLDBProto.CLDBProg.ClusterInfoProc.getNumber(),
  	               req,
  	               ClusterInfoResponse.class);
  	  } else {
  	    data = CLDBRpcCommonUtils.getInstance().sendRequest(
  	               Common.MapRProgramId.CldbProgramId.getNumber(),
  	               CLDBProto.CLDBProg.ClusterInfoProc.getNumber(),
  	               req,
  	               ClusterInfoResponse.class);
  	  }
  	
  	  if (data == null) {
  	    out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
  	    return null;
  	  }
  	
  	  // success
  	  ClusterInfoResponse resp = ClusterInfoResponse.parseFrom(data);
  	  
  	  int status = resp.getStatus();
    	if (status != 0) {
    	  out.addError(new OutputError(status, Errno.toString(status)));
    	  return null;
    	}
    	
    	return resp;
  	} catch (Exception e) {
  	  out.addError (new CommandOutput.OutputHierarchy.OutputError (
  	                Errno.EOPFAILED, 
  	                "doClusterInfo: " + Errno.toString(Errno.EOPFAILED)));
  	  LOG.error("Exception during doClusterInfo", e);
  	  return null;
  	}  	
  }

  private Long getDbStatValue(NodeInfo stat, DbNodeStats dbNodeStats) {
    int statDuration;
    List<Integer> durations = dbNodeStats.getDurationsList();
    List<Long> statValues;
    switch(stat) {
      case DbPuts10s:
        statDuration = 10;
        statValues = dbNodeStats.getPutsCountList();
        break;
      case DbPuts1m:
        statDuration = 60;
        statValues = dbNodeStats.getPutsCountList();
        break;
      case DbPuts5m:
        statDuration = 300;
        statValues = dbNodeStats.getPutsCountList();
        break;
      case DbPuts15m:
        statDuration = 900;
        statValues = dbNodeStats.getPutsCountList();
        break;

      case DbGets10s:
        statDuration = 10;
        statValues = dbNodeStats.getGetsCountList();
        break;
      case DbGets1m:
        statDuration = 60;
        statValues = dbNodeStats.getGetsCountList();
        break;
      case DbGets5m:
        statDuration = 300;
        statValues = dbNodeStats.getGetsCountList();
        break;
      case DbGets15m:
        statDuration = 900;
        statValues = dbNodeStats.getGetsCountList();
        break;

      case DbScans10s:
        statDuration = 10;
        statValues = dbNodeStats.getScansCountList();
        break;
      case DbScans1m:
        statDuration = 60;
        statValues = dbNodeStats.getScansCountList();
        break;
      case DbScans5m:
        statDuration = 300;
        statValues = dbNodeStats.getScansCountList();
        break;
      case DbScans15m:
        statDuration = 900;
        statValues = dbNodeStats.getScansCountList();
        break;
      default:
        return 0L;
    }

    int index = durations.indexOf(statDuration);
    if (index == -1 || durations.size() != statValues.size()) {
      return 0L;
    } else {
      return statValues.get(index);
    }
  }
  
  private void listCldbs (OutputHierarchy out) throws CLIProcessingException
  {    
    ClusterInfoResponse resp = doClusterInfo(out, true);
    if (resp != null) {
    	String cldblist = resp.getCldbList();
    	if (cldblist != null && !cldblist.isEmpty())
        out.addNode(new OutputNode("CLDBs", cldblist));
    }
  }
    
  private void listZks (OutputHierarchy out) throws CLIProcessingException
  {  
    ClusterInfoResponse resp = doClusterInfo(out, false);
    if (resp != null) {
    	String zklist = resp.getZkConnectString();
    	if (zklist != null && !zklist.isEmpty())
        out.addNode(new OutputNode("Zookeepers", zklist));
    }
  }
  
  private void listCldbZks (OutputHierarchy out) throws CLIProcessingException
  {

    ClusterInfoResponse resp = doClusterInfo(out, true);
    if (resp != null) {
    	String cldblist = resp.getCldbList();
      OutputNode node = new OutputNode();
    	if (cldblist != null && !cldblist.isEmpty())
        node.addChild(new OutputNode("CLDBs", cldblist));
    	
    	String zklist = resp.getZkConnectString();
    	if (zklist != null && !zklist.isEmpty())
        node.addChild(new OutputNode("Zookeepers", zklist));

      out.addNode(node);
    }
  }

  @Override
  public String getCommandUsage() {
    return "node list";
  }
  
  static class IpAddressNode extends OutputNode {
    public IpAddressNode(String name, Object value) {
      super(name, value);
    }
    
    public String peersString() {
      StringBuffer sb = new StringBuffer();
      boolean addComma = false;
      for(OutputNode ipaddr : super.getPeers()) {
        if (addComma) {
          sb.append(",");
        }
        sb.append(ipaddr.getValue().toString());
        addComma = true;
      }
      
      return sb.toString();
    }
  }
}
