/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;


import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.Arrays;
import java.util.List;

import org.apache.commons.lang.ArrayUtils;
import org.apache.log4j.Logger;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.InvalidProtocolBufferException;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.TextCommandOutput;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.IntegerInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.GetMapRUserTicketReq;
import com.mapr.fs.cldb.proto.CLDBProto.GetMapRUserTicketResp;
import com.mapr.fs.cldb.proto.CLDBProto.GetClusterTicketReq;
import com.mapr.fs.cldb.proto.CLDBProto.GetClusterTicketResp;
import com.mapr.fs.cldb.proto.CLDBProto.MirrorStartRequest;
import com.mapr.fs.cldb.proto.CLDBProto.MirrorStartResponse;
import com.mapr.fs.cldb.proto.CLDBProto.MirrorType;
import com.mapr.fs.cldb.proto.CLDBProto.VolumeLookupResponse;
import com.mapr.fs.cldb.proto.CLDBProto.VolumeProperties;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Security.Key;
import com.mapr.fs.proto.Security.ServerKeyType;
import com.mapr.fs.proto.Security.TicketAndKey;
import com.mapr.login.MapRLogin;
import com.mapr.login.MapRLoginException;
import com.mapr.login.client.MapRLoginClient;
import com.mapr.login.client.MapRLoginHttpsClient;
import com.mapr.login.common.GenTicketTypeRequest.TicketType;
import com.mapr.security.MutableInt;
import com.mapr.security.Security;
import com.mapr.security.MaprSecurityException;


public class SecurityCommands extends CLIBaseClass implements CLIInterface {


  public SecurityCommands(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }

  private static final Logger LOG = Logger.getLogger(SecurityCommands.class);

  public static final String OUTKEYFILE_PARAM_NAME = "keyfile";
  public static final String INKEYFILE_PARAM_NAME = "inkeyfile";
	public static final String TICKETFILE_PARAM_NAME = "ticketfile";
	public static final String CLUSTER_PARAM_NAME = "cluster";
  public static final String UID_PARAM_NAME = "mapruid";
  public static final String GID_PARAM_NAME = "maprgid";
  public static final String USERNAME_PARAM_NAME = "maprusername";
  public static final String MAPR_CLUSTER_USER_NAME = "clusterusername";
  public static final String MAPR_SERVER_TICKETFILE_PARAM_NAME = "inmaprserverticketfile";
  static final String defaultMapRServerTicketFileLocation = "/opt/mapr/conf/maprserverticket";
  static final String defaultMapRUserTicketFileLocation = "/opt/mapr/conf/mapruserticket";
  static final String defaultMapRClusterTicketFileLocation = "/opt/mapr/conf/maprclusterticket";
  public static final String TICKET_TYPE_PARAM_NAME = "type";
  public static final String OUT_PARAM_NAME = "out";
  public static final String USER_PARAM_NAME = "user";
  
	public static final String securityGenKeyUsage = 
			"security genkey -keyfile keyfilepath";

	public static String securityGenTicketUsage = "security genticket "
	    + "-cluster clustername "
      + "-maprusername maprusername "
      + "-mapruid mapruid "
      + "-maprgid maprgid "
	    + "-inkeyfile inkeyfilepath"
	    + "-ticketfile ticketfilepath";

	public static String securityGenerateTicketUsage = "security generateticket "
      + "-cluster clustername "
      + "-out out "
      + "-user user"
      + "-type tickettype";

	 public static String securityPrintKeyUsage = "security printkey"
       + "-keyfile keyfilepath";

	 public static String securityGetMapRUserTicketUsage = "security getmapruserticket"
	      + "-inmaprserverticket serverticketpath -ticketfile ticketfilepath";

	 public static String securityGetMapRClusterTicketUsage = "security getmaprclusterticket"
	      + "-clusterusername username -inmaprserverticket serverticketpath -ticketfile ticketfilepath";

   public static String securityGenMfsUtilTicketUsage = "security genmfsutilticket"
              + "-inmaprserverticket serverticketpath -ticketfile ticketfilepath";

	 
   public static String securityPrintTicketUsage = "security printticket"
       + "-ticketfile ticketfilepath";
   
   static final CLICommand securityGenKeyCommand =
       new CLICommand(
           "genkey", 
           "usage : " + securityGenKeyUsage,
           SecurityCommands.class, ExecutionTypeEnum.NATIVE,
           new ImmutableMap.Builder<String, BaseInputParameter>()
           .put(
               SecurityCommands.OUTKEYFILE_PARAM_NAME,
               new TextInputParameter(
                   SecurityCommands.OUTKEYFILE_PARAM_NAME, "keyfile",
                   CLIBaseClass.REQUIRED, null))
                   .build(), 
                   null)
   .setShortUsage(securityGenKeyUsage);

   static final CLICommand securityGenTicketCommand =
       new CLICommand(
           "genticket", 
           "usage : " + securityGenTicketUsage,
           SecurityCommands.class, ExecutionTypeEnum.NATIVE,
           new ImmutableMap.Builder<String, BaseInputParameter>()
           .put(
               SecurityCommands.INKEYFILE_PARAM_NAME,
               new TextInputParameter(
                   SecurityCommands.INKEYFILE_PARAM_NAME, "inkeyfile",
                   CLIBaseClass.REQUIRED, null))
           .put(
               SecurityCommands.TICKETFILE_PARAM_NAME,
               new TextInputParameter(
                   SecurityCommands.TICKETFILE_PARAM_NAME, "ticketfile",
                   CLIBaseClass.REQUIRED, null))
            .put(
               SecurityCommands.CLUSTER_PARAM_NAME,
               new TextInputParameter(
                   SecurityCommands.CLUSTER_PARAM_NAME, "cluster",
                   CLIBaseClass.REQUIRED, null))
            .put(
               SecurityCommands.USERNAME_PARAM_NAME,
               new TextInputParameter(
                   SecurityCommands.USERNAME_PARAM_NAME, "maprusername",
                   CLIBaseClass.REQUIRED, null))
           .put(
               SecurityCommands.UID_PARAM_NAME,
               new IntegerInputParameter(
                   SecurityCommands.UID_PARAM_NAME, "mapruid",
                   CLIBaseClass.REQUIRED, null))
           .put(
               SecurityCommands.GID_PARAM_NAME,
               new IntegerInputParameter(
                   SecurityCommands.GID_PARAM_NAME, "maprgid",
                   CLIBaseClass.REQUIRED, null))
                   .build(), 
                   null)
   .setShortUsage(securityGenTicketUsage);


   static final CLICommand securityGetMapRUserTicketCommand =
       new CLICommand(
           "getmapruserticket", 
           "usage : " + securityGetMapRUserTicketUsage,
           SecurityCommands.class, ExecutionTypeEnum.NATIVE,
           new ImmutableMap.Builder<String, BaseInputParameter>()
           .put(
               SecurityCommands.MAPR_SERVER_TICKETFILE_PARAM_NAME,
               new TextInputParameter(
                   SecurityCommands.MAPR_SERVER_TICKETFILE_PARAM_NAME, 
                   "inmaprserverticketfile",
                   CLIBaseClass.NOT_REQUIRED, 
                   defaultMapRServerTicketFileLocation))
           .put(
               SecurityCommands.TICKETFILE_PARAM_NAME,
               new TextInputParameter(
                   SecurityCommands.TICKETFILE_PARAM_NAME, "ticketfile",
                   CLIBaseClass.NOT_REQUIRED, defaultMapRUserTicketFileLocation))
                   .build(), 
                   null)
   .setShortUsage(securityGetMapRUserTicketUsage);

   static final CLICommand securityGetMapRClusterTicketCommand =
       new CLICommand(
           "getmaprclusterticket", 
           "usage : " + securityGetMapRClusterTicketUsage,
           SecurityCommands.class, ExecutionTypeEnum.NATIVE,
           new ImmutableMap.Builder<String, BaseInputParameter>()
           .put(
               SecurityCommands.MAPR_CLUSTER_USER_NAME,
               new TextInputParameter(
                   SecurityCommands.MAPR_CLUSTER_USER_NAME,
                   "clusterusername",
                   CLIBaseClass.REQUIRED, null))
           .put(
               SecurityCommands.MAPR_SERVER_TICKETFILE_PARAM_NAME,
               new TextInputParameter(
                   SecurityCommands.MAPR_SERVER_TICKETFILE_PARAM_NAME, 
                   "inmaprserverticketfile",
                   CLIBaseClass.NOT_REQUIRED, 
                   defaultMapRServerTicketFileLocation))
           .put(
               SecurityCommands.TICKETFILE_PARAM_NAME,
               new TextInputParameter(
                   SecurityCommands.TICKETFILE_PARAM_NAME, "ticketfile",
                   CLIBaseClass.NOT_REQUIRED, 
                   defaultMapRClusterTicketFileLocation))
                   .build(), 
                   null)
   .setShortUsage(securityGetMapRClusterTicketUsage);

   static final CLICommand securityGenMfsUtilTicketCommand =
       new CLICommand(
         "genmfsutilticket", 
         "usage : " + securityGenMfsUtilTicketUsage,
           SecurityCommands.class, ExecutionTypeEnum.NATIVE,
           new ImmutableMap.Builder<String, BaseInputParameter>()
           .put(
               SecurityCommands.MAPR_SERVER_TICKETFILE_PARAM_NAME,
               new TextInputParameter(
                   SecurityCommands.MAPR_SERVER_TICKETFILE_PARAM_NAME, 
                   "inmaprserverticketfile",
                   CLIBaseClass.NOT_REQUIRED, 
                   defaultMapRServerTicketFileLocation))
           .put(
               SecurityCommands.TICKETFILE_PARAM_NAME,
               new TextInputParameter(
                   SecurityCommands.TICKETFILE_PARAM_NAME, "ticketfile",
                   CLIBaseClass.NOT_REQUIRED, defaultMapRUserTicketFileLocation))
                   .build(), 
                   null)
   .setShortUsage(securityGenMfsUtilTicketUsage)
   .setUsageInVisible(true);

   static final CLICommand securityPrintKeyCommand =
       new CLICommand(
           "printkey", 
           "usage : " + securityPrintKeyUsage,
           SecurityCommands.class, ExecutionTypeEnum.NATIVE,
           new ImmutableMap.Builder<String, BaseInputParameter>()
           .put(
               SecurityCommands.INKEYFILE_PARAM_NAME,
               new TextInputParameter(
                   SecurityCommands.INKEYFILE_PARAM_NAME, "keyfile",
                   CLIBaseClass.REQUIRED, null))
                   .build(), 
                   null)
   .setShortUsage(securityPrintKeyUsage);

   static final CLICommand securityPrintTicketCommand =
       new CLICommand(
           "printticket", 
           "usage : " + securityPrintTicketUsage,
           SecurityCommands.class, ExecutionTypeEnum.NATIVE,
           new ImmutableMap.Builder<String, BaseInputParameter>()
           .put(
               SecurityCommands.TICKETFILE_PARAM_NAME,
               new TextInputParameter(
                   SecurityCommands.TICKETFILE_PARAM_NAME, "ticketfile",
                   CLIBaseClass.REQUIRED, null))
                   .build(), 
                   null)
   .setShortUsage(securityPrintTicketUsage);

   static final CLICommand securityGenerateTicketCommand =
       new CLICommand(
           "generateticket",
           "usage : " + securityGenerateTicketUsage,
           SecurityCommands.class, ExecutionTypeEnum.NATIVE,
           new ImmutableMap.Builder<String, BaseInputParameter>()
           .put(
               SecurityCommands.CLUSTER_PARAM_NAME,
               new TextInputParameter(
                   SecurityCommands.CLUSTER_PARAM_NAME, "cluster",
                   CLIBaseClass.REQUIRED, null))
           .put(
               SecurityCommands.OUT_PARAM_NAME,
               new TextInputParameter(
                   SecurityCommands.OUT_PARAM_NAME, "out",
                   CLIBaseClass.REQUIRED, null))
            .put(
               SecurityCommands.USER_PARAM_NAME,
               new TextInputParameter(
                   SecurityCommands.USER_PARAM_NAME, "user",
                   CLIBaseClass.REQUIRED, null))
           .put(
               SecurityCommands.TICKET_TYPE_PARAM_NAME,
               new TextInputParameter(
                   SecurityCommands.TICKET_TYPE_PARAM_NAME, "type",
                   CLIBaseClass.REQUIRED, null))
                   .build(),
                   null)
   .setShortUsage(securityGenerateTicketUsage);

   public static CLICommand securityCommands = 
       new CLICommand("security", "security",
       CLIUsageOnlyCommand.class, 
       ExecutionTypeEnum.NATIVE, 
       new CLICommand [] {         /* array of subcommands */
         securityGenKeyCommand,
         securityGenTicketCommand,
         securityGenerateTicketCommand,
         securityGetMapRUserTicketCommand,
         securityGetMapRClusterTicketCommand,
         securityGenMfsUtilTicketCommand,
//         securityPrintKeyCommand,
//         securityPrintTicketCommand
        }
   ).setShortUsage("security [genkey|genticket|getmapruserticket]");

  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    
    if (cliCommand.getCommandName().equalsIgnoreCase("genkey")) {
      try {         
        return generateKey();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      } 
    } else if (cliCommand.getCommandName().equalsIgnoreCase("genticket")) {
      try {
        return generateTicket();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      }
    } else if (cliCommand.getCommandName().equalsIgnoreCase("getmapruserticket")) {
      try {
        return getMapRUserTicket();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      }
    } else if (cliCommand.getCommandName().equalsIgnoreCase("getmaprclusterticket")) {
      try {
        return getMapRClusterTicket();
      } catch (Exception e) {
        throw new CLIProcessingException("Send Request Exception", e);
      }
    } else if (cliCommand.getCommandName().equalsIgnoreCase("genmfsutilticket")) {
      try {
        return genMfsUtilTicket();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      }
    } else if (cliCommand.getCommandName().equalsIgnoreCase("printkey")) {
      try {
        return printKey();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      }
    } else if (cliCommand.getCommandName().equalsIgnoreCase("printticket")) {
      try {
        return printTicket();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      }
    } else if (cliCommand.getCommandName().equalsIgnoreCase("generateticket")) {
      try {
        return generateUserTicket();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      }
    }
    return null;
  }
   private CommandOutput generateKey() throws CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);

    String keyFileName= getParamTextValue(OUTKEYFILE_PARAM_NAME, 0);
    if ((keyFileName == null) || keyFileName.isEmpty()) {                
      out.addError(new OutputError(Errno.EINVAL, "Invalid keyfile path")
      .setField(OUTKEYFILE_PARAM_NAME)
      .setFieldValue(keyFileName));
      return output;
    }
    
    File keyFile = new File(keyFileName);
    if (keyFile.exists()) {
      out.addError(new OutputError(Errno.EEXIST, 
          "keyfile " + keyFileName + " already exists"));
      return output;
    }
    
    // Create the key file 
    try {
      if (keyFile.createNewFile() == false) {
        out.addError(new OutputError(Errno.EEXIST, 
            "keyfile " + keyFileName + " already exists"));
        return output;
      }
      
      // TODO : Put JNI method to create the file with perm 600
      // Following is indirect mechanism in java to set the perm
      // to 600
      keyFile.setExecutable(false, false);
      keyFile.setWritable(false, false);
      keyFile.setReadable(false, false);
      keyFile.setWritable(true, true);
      keyFile.setReadable(true, true);
    } catch (IOException e) {
      LOG.error("Failed to create file " + keyFileName, e);
      out.addError(new OutputError(Errno.E_NOT_ENOUGH_PRIVILEGES, 
          "Failed to create file " + keyFileName));
      return output;
    }
    
    MutableInt err = new MutableInt();
    Key key = Security.GenerateRandomKey();
    byte [] encodedKey = 
        Security.EncodeDataForWritingToKeyFile(key.toByteArray(), err);
    if (encodedKey == null) {
      out.addError(new OutputError(err.GetValue(), 
          "failed to generate key"));
      return output;
    }
    
    try {
      FileOutputStream fout;
      fout = new FileOutputStream(keyFile);
      fout.write(encodedKey);
      fout.write(new String("\n").getBytes("UTF-8"));
      fout.close();
    } catch (FileNotFoundException e) {
      LOG.error("Failed to open file " + keyFileName, e);
      out.addError(new OutputError(Errno.E_NOT_ENOUGH_PRIVILEGES, 
          "Failed to open file " + keyFileName));
      return output;
    } catch (IOException e) {
      LOG.error("Failed to write to file " + keyFileName, e);
      out.addError(new OutputError(Errno.E_NOT_ENOUGH_PRIVILEGES, 
          "Failed to write to file " + keyFileName));
      return output;
    }
    return output;
  }

  private CommandOutput generateUserTicket() throws CLIProcessingException {
     CommandOutput output = new CommandOutput();
     OutputHierarchy out = new OutputHierarchy();
     output.setOutput(out);
     //Remove first parameter in CLI command
     String[] args = (String[])ArrayUtils.remove(getInput().getRawInput(), 0);
     try {
       MapRLogin.execute(args);
     } catch (IOException e) {
       LOG.error("Failed to create ticket", e);
       out.addError(new OutputError(Errno.E_NOT_ENOUGH_PRIVILEGES,
           "Failed to create ticket with error "+e.getMessage()));
       return output;
     } catch (Exception e) {
       LOG.error("Failed to create ticket", e);
       out.addError(new OutputError(Errno.EINVALMISC,
           "Failed to create ticket with error "+e.getMessage()));
       return output;
     }
     return output;
  }

  private CommandOutput generateTicket() throws CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);

    String keyFileName= getParamTextValue(INKEYFILE_PARAM_NAME, 0);
    if ((keyFileName == null) || keyFileName.isEmpty()) {                
      out.addError(new OutputError(Errno.EINVAL, "Invalid keyfile path")
      .setField(INKEYFILE_PARAM_NAME)
      .setFieldValue(keyFileName));
      return output;
    }
    
    File keyFile = new File(keyFileName);
    if (keyFile.exists() == false) {
      out.addError(new OutputError(Errno.ENOENT, 
          "keyfile " + keyFileName + " doesn't exist"));
      return output;
    }
    
    if (keyFile.canRead() == false) {
      out.addError(new OutputError(Errno.EPERM, 
          "keyfile " + keyFileName + " is not readable"));
      return output;
    }
    
    byte [] fileData = new byte [4096];
    int len;
    try {
      FileInputStream fin;
      fin = new FileInputStream(keyFile);
      len = fin.read(fileData);
      if ((fileData[len - 1] == '\n') || 
          (fileData[len - 1] == 0)) {
        len = len - 1;
      }
      fin.close();
    } catch (FileNotFoundException e) {
      LOG.error("Failed to open file " + keyFileName, e);
      out.addError(new OutputError(Errno.E_NOT_ENOUGH_PRIVILEGES, 
          "Failed to open file " + keyFileName));
      return output;
    } catch (IOException e) {
      LOG.error("Failed to read file " + keyFileName, e);
      out.addError(new OutputError(Errno.E_NOT_ENOUGH_PRIVILEGES, 
          "Failed to read file " + keyFileName));
      return output;
    }
    
    byte [] encodedKey = new byte[len];
    System.arraycopy(fileData, 0, encodedKey, 0, len);
    MutableInt err = new MutableInt();
    byte [] decodedKey = Security.DecodeDataFromKeyFile(encodedKey, err);
    if (decodedKey == null) {
      out.addError(new OutputError(err.GetValue(), "Failed to parse keyfile"));
      return output;
    }
    
    Key key;
    try {
      key = Key.parseFrom(decodedKey);
    } catch (InvalidProtocolBufferException e) {
      LOG.error("Failed to parse keyfile " + keyFileName, e);
      out.addError(new OutputError(Errno.EBADMSG, 
          "Failed to parse keyfile " + keyFileName));
      return output;
    }

    // Generate the ticketandkey now
    String clusterName = getParamTextValue(CLUSTER_PARAM_NAME, 0);
    if ((clusterName == null) || clusterName.isEmpty()) {                
      out.addError(new OutputError(Errno.EINVAL, "Invalid cluster name")
      .setField(CLUSTER_PARAM_NAME)
      .setFieldValue(clusterName));
      return output;
    }

    // get uid, gid
    int uid = getParamIntValue(UID_PARAM_NAME, 0);
    int gid[] = new int[1];
    gid[0] = getParamIntValue(GID_PARAM_NAME, 0);
    long expiryTime = Security.MAX_EXPIRY_TIME;

    String maprUserName = getParamTextValue(USERNAME_PARAM_NAME, 0);
    if ((maprUserName == null) || maprUserName.isEmpty()) {                
      out.addError(new OutputError(Errno.EINVAL, "Invalid maprusername")
      .setField(USERNAME_PARAM_NAME)
      .setFieldValue(maprUserName));
      return output;
    }

    TicketAndKey ticketAndKey = 
        Security.GenerateTicketAndKeyUsingServerKey(
            ServerKeyType.CldbKey, 
            key, maprUserName, uid, gid, expiryTime,
            0/*non-renewable ticket*/, false /*isExternal*/, err);
    
    if (ticketAndKey == null) {
      out.addError(new OutputError(err.GetValue(), "Failed to generate ticketfile"));
      return output;
    }
    
    byte [] encodedticket = 
        Security.EncodeDataForWritingToKeyFile(ticketAndKey.toByteArray(), err);
    
    if (encodedticket == null) {
      out.addError(new OutputError(err.GetValue(), 
          "failed to encode ticketfile"));
      return output;
    }

    String ticketFileName= getParamTextValue(TICKETFILE_PARAM_NAME, 0);
    if ((ticketFileName == null) || ticketFileName.isEmpty()) {                
      out.addError(new OutputError(Errno.EINVAL, "Invalid ticketfile path")
      .setField(TICKETFILE_PARAM_NAME)
      .setFieldValue(ticketFileName));
      return output;
    }
    
    File ticketFile = new File(ticketFileName);
    if (ticketFile.exists()) {
      out.addError(new OutputError(Errno.EEXIST, 
          "ticketfile " + ticketFileName + " already exists"));
      return output;
    }

    // Create the ticket file 
    try {
      if (ticketFile.createNewFile() == false) {
        out.addError(new OutputError(Errno.EEXIST, 
            "ticketfile " + ticketFileName + " already exists"));
        return output;
      }
      
      // TODO : Put JNI method to create the file with perm 600
      // Following is indirect mechanism in java to set the perm
      // to 600
      ticketFile.setExecutable(false, false);
      ticketFile.setWritable(false, false);
      ticketFile.setReadable(false, false);
      ticketFile.setWritable(true, true);
      ticketFile.setReadable(true, true);
    } catch (IOException e) {
      LOG.error("Failed to create file " + ticketFileName, e);
      out.addError(new OutputError(Errno.E_NOT_ENOUGH_PRIVILEGES, 
          "Failed to create file " + ticketFileName));
      return output;
    }

    try {
      FileOutputStream fout;
      fout = new FileOutputStream(ticketFile);
      fout.write(clusterName.getBytes("UTF-8"));
      fout.write(new String(" ").getBytes("UTF-8"));
      fout.write(encodedticket);
      fout.write(new String("\n").getBytes("UTF-8"));
      fout.close();
    } catch (FileNotFoundException e) {
      LOG.error("Failed to open file " + ticketFileName, e);
      out.addError(new OutputError(Errno.E_NOT_ENOUGH_PRIVILEGES, 
          "Failed to open file " + ticketFileName));
      return output;
    } catch (IOException e) {
      LOG.error("Failed to write to file " + ticketFileName, e);
      out.addError(new OutputError(Errno.E_NOT_ENOUGH_PRIVILEGES, 
          "Failed to write to file " + ticketFileName));
      return output;
    }
    return output;
  }

  private CommandOutput getMapRUserTicket() throws CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);

    String maprServerTicketFileName = 
        getParamTextValue(MAPR_SERVER_TICKETFILE_PARAM_NAME, 0);
    if ((maprServerTicketFileName == null) || 
         maprServerTicketFileName.isEmpty()) {                
      out.addError(new OutputError(Errno.EINVAL, "Invalid maprserverticketfile path")
      .setField(MAPR_SERVER_TICKETFILE_PARAM_NAME)
      .setFieldValue(maprServerTicketFileName));
      return output;
    }

    String maprUserTicketFileName = 
        getParamTextValue(TICKETFILE_PARAM_NAME, 0);
    if ((maprUserTicketFileName == null) || 
         maprUserTicketFileName.isEmpty()) {                
      out.addError(new OutputError(Errno.EINVAL, "Invalid mapruserticketfile path")
      .setField(TICKETFILE_PARAM_NAME)
      .setFieldValue(maprUserTicketFileName));
      return output;
    }
    
    // if the ticket file already exists and valid then
    // no need to create it again
    long lastModifiedmaprUserTicket = 0;
    long lastModifiedmaprServerTicket = 0;
    int err = 0;
    File maprUserTicketFile = new File(maprUserTicketFileName);
    if ( maprUserTicketFile.canRead() ) {
      lastModifiedmaprUserTicket = maprUserTicketFile.lastModified();
    }
    File maprServerTicketFile = new File(maprServerTicketFileName);
    if ( maprServerTicketFile.canRead() ) {
      lastModifiedmaprServerTicket = maprServerTicketFile.lastModified();
    }
    if ( lastModifiedmaprUserTicket < lastModifiedmaprServerTicket ) {
      LOG.info("mapruserticket does not exist or it is older then maprserverticket. Regenerating it");
    } else {
      err = Security.SetTicketAndKeyFile(maprUserTicketFileName);
      if (err == 0) {
        out.addNode(new OutputNode("Not creating a new ticket file as valid ticketfile \""
            + maprUserTicketFileName + "\" already exists"));
        return output;
      }
    }

    // Get the cluster name from mapr-clusters.conf file 
    String clusterName = CLDBRpcCommonUtils.getInstance().getCurrentClusterName();
    err = Security.SetTicketAndKeyFile(maprServerTicketFileName);
    if (err != 0) {
      out.addError(new OutputError(err, 
                  "Failed to load the maprserverticketfile " 
                  + maprServerTicketFileName));
      return output;            
    }
    
    // Send RPC to CLDB to get the mapruserticket
    GetMapRUserTicketReq.Builder req = GetMapRUserTicketReq.newBuilder();
    GetMapRUserTicketResp resp;
    byte[] data = null;

    req.setCreds(getUserCredentials());

    try {
      data = CLDBRpcCommonUtils.getInstance().sendRequest(clusterName,
          Common.MapRProgramId.CldbProgramId.getNumber(),
          CLDBProto.CLDBProg.GetMapRUserTicket.getNumber(),
          req.build(),
          GetMapRUserTicketResp.class, ServerKeyType.CldbKey);
      
      if (data == null) {
        /**
        * <MAPR_ERROR>
        * Message:Exception while processing RPC
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        throw new CLIProcessingException("Exception while processing RPC");
      }
      resp = GetMapRUserTicketResp.parseFrom(data);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      /**
      * <MAPR_ERROR>
      * Message:Exception while sending RPC to cluster
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      throw new CLIProcessingException("Exception while sending RPC to cluster"
          + clusterName, e);
    }

    if (resp.getStatus() != 0) {
      /**
      * <MAPR_ERROR>
      * Message:Mirror start RPC to CLDB for volume <volume>@<cluster> failed with status <status>
      * Function:VolumeMirrorCommands.startMirror()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      LOG.error("Rpc to CLDB failed with status " + resp.getStatus());
      out.addError(new OutputError(resp.getStatus(), 
          "Failed to send RPC to CLDB for cluster" + clusterName));
      return output;
    }
    
    // Got the ticketandkey for the mapruserticket from cldb write
    // to the file now
    File tmpUserTicketFile = null;
    File userTicketFile = new File(maprUserTicketFileName);
    try {

      // TODO : Put JNI method to create the file with perm 600
      // Following is indirect mechanism in java to set the perm
      // to 600
      String parentDir = userTicketFile.getParent();
      if (parentDir == null) {
        parentDir = ".";
      }  
      tmpUserTicketFile = File.createTempFile(".maprticket", null /*suffix*/, new File(parentDir));
      tmpUserTicketFile.deleteOnExit();
      tmpUserTicketFile.setExecutable(false, false);
      tmpUserTicketFile.setWritable(false, false);
      tmpUserTicketFile.setReadable(false, false);
      tmpUserTicketFile.setWritable(true, true);
      tmpUserTicketFile.setReadable(true, true);
    } catch (IOException e) {
      LOG.error("Failed to create file " + tmpUserTicketFile, e);
      out.addError(new OutputError(Errno.E_NOT_ENOUGH_PRIVILEGES, 
          "Failed to create file " + tmpUserTicketFile));
      return output;
    }

    MutableInt mutablerr = new MutableInt();
    byte [] encodedticket = 
        Security.EncodeDataForWritingToKeyFile(
            resp.getMaprUserTicketAndKey().toByteArray(), mutablerr);
    
    if (encodedticket == null) {
      out.addError(new OutputError(mutablerr.GetValue(), 
          "failed to encode ticket"));
      return output;
    }

    // Write to the file now
    try {
      FileOutputStream fout;
      fout = new FileOutputStream(tmpUserTicketFile);
      fout.write(clusterName.getBytes("UTF-8"));
      fout.write(new String(" ").getBytes("UTF-8"));
      fout.write(encodedticket);
      fout.write(new String("\n").getBytes("UTF-8"));
      fout.close();
    } catch (FileNotFoundException e) {
      LOG.error("Failed to open file " + tmpUserTicketFile, e);
      out.addError(new OutputError(Errno.E_NOT_ENOUGH_PRIVILEGES, 
          "Failed to open file " + tmpUserTicketFile));
      return output;
    } catch (IOException e) {
      LOG.error("Failed to write to file " + tmpUserTicketFile, e);
      out.addError(new OutputError(Errno.E_NOT_ENOUGH_PRIVILEGES, 
          "Failed to write to file " + tmpUserTicketFile));
      return output;
    }
    
    // Rename the file to the mapr user ticket file
    // Write to the file now
    if (!tmpUserTicketFile.renameTo(userTicketFile)) {
      LOG.error("Failed to rename to file " + userTicketFile);
      out.addError(new OutputError(Errno.E_NOT_ENOUGH_PRIVILEGES, 
          "Failed to write to file " + userTicketFile));
      return output;
    }
    out.addNode(new OutputNode("Successfully wrote the mapruserticket to "
                                + maprUserTicketFileName));
    return output;
  }

  private CommandOutput getMapRClusterTicket() throws CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);
  
    String maprClusterUserName = 
        getParamTextValue(MAPR_CLUSTER_USER_NAME, 0);
    if ((maprClusterUserName == null) || maprClusterUserName.isEmpty()) {
      out.addError(new OutputError(Errno.EINVAL, "Invalid cluster user name")
        .setField(MAPR_CLUSTER_USER_NAME)
        .setFieldValue(maprClusterUserName));
      return output;
    }

    String maprServerTicketFileName = 
        getParamTextValue(MAPR_SERVER_TICKETFILE_PARAM_NAME, 0);
    if ((maprServerTicketFileName == null) || 
         maprServerTicketFileName.isEmpty()) {                
      out.addError(new OutputError(Errno.EINVAL, "Invalid maprserverticketfile path")
      .setField(MAPR_SERVER_TICKETFILE_PARAM_NAME)
      .setFieldValue(maprServerTicketFileName));
      return output;
    }

    String maprClusterTicketFileName = 
        getParamTextValue(TICKETFILE_PARAM_NAME, 0);
    if ((maprClusterTicketFileName == null) || 
         maprClusterTicketFileName.isEmpty()) {                
      out.addError(new OutputError(Errno.EINVAL, "Invalid maprclusterticketfile path")
      .setField(TICKETFILE_PARAM_NAME)
      .setFieldValue(maprClusterTicketFileName));
      return output;
    }
    
    int err = 0;
    // Get the cluster name from mapr-clusters.conf file 
    String clusterName = CLDBRpcCommonUtils.getInstance().getCurrentClusterName();
    err = Security.SetTicketAndKeyFile(maprServerTicketFileName);
    if (err != 0) {
      out.addError(new OutputError(err, 
                  "Failed to load the maprserverticketfile " 
                  + maprServerTicketFileName));
      return output;            
    }
    
    // Send RPC to CLDB to get the mapruserticket
    GetClusterTicketReq.Builder req = GetClusterTicketReq.newBuilder();
    GetClusterTicketResp resp;
    byte[] data = null;

    req.setCreds(getUserCredentials());
    req.setClusterUserName(maprClusterUserName);

    try {

      data = CLDBRpcCommonUtils.getInstance().sendRequest(clusterName,
          Common.MapRProgramId.CldbProgramId.getNumber(),
          CLDBProto.CLDBProg.GetClusterTicketProc.getNumber(),
          req.build(),
          GetClusterTicketResp.class, ServerKeyType.CldbKey);
      
      if (data == null) {
        /**
        * <MAPR_ERROR>
        * Message:Exception while processing RPC
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        throw new CLIProcessingException("Exception while processing RPC");
      }
      resp = GetClusterTicketResp.parseFrom(data);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      /**
      * <MAPR_ERROR>
      * Message:Exception while sending RPC to cluster
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      throw new CLIProcessingException("Exception while sending RPC to cluster"
          + clusterName, e);
    }

    if (resp.getStatus() != 0) {

      LOG.error("Rpc to CLDB failed with status " + resp.getStatus());
      if (resp.getStatus() == Errno.ENOENT) {
        out.addError(new OutputError(Errno.EINVAL, "Invalid cluster user name")
          .setField(MAPR_CLUSTER_USER_NAME)
          .setFieldValue(maprClusterUserName));
        return output;
      }

      if (resp.getStatus() == Errno.EPERM) {
        out.addError(new OutputError(Errno.EPERM, "No permissions to generate cluster ticket")
          .setField(MAPR_CLUSTER_USER_NAME)
          .setFieldValue(maprClusterUserName));
        return output;
      }

      /**
      * <MAPR_ERROR>
      * Message:Mirror start RPC to CLDB for volume <volume>@<cluster> failed with status <status>
      * Function:SecurityCommands.getMapRClusterTicket()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      out.addError(new OutputError(resp.getStatus(), 
          "Failed to send RPC to CLDB for cluster: " + clusterName));
      return output;
    }
    
    // Got the ticketandkey for the maprclusterticket from cldb write
    // to the file now
    File tmpClusterTicketFile = null;
    File clusterTicketFile = new File(maprClusterTicketFileName);
    try {

      // TODO : Put JNI method to create the file with perm 600
      // Following is indirect mechanism in java to set the perm
      // to 600
      String parentDir = clusterTicketFile.getParent();
      if (parentDir == null) {
        parentDir = ".";
      }  
      tmpClusterTicketFile = File.createTempFile(".maprticket", null /*suffix*/, new File(parentDir));
      tmpClusterTicketFile.deleteOnExit();
      tmpClusterTicketFile.setExecutable(false, false);
      tmpClusterTicketFile.setWritable(false, false);
      tmpClusterTicketFile.setReadable(false, false);
      tmpClusterTicketFile.setWritable(true, true);
      tmpClusterTicketFile.setReadable(true, true);
    } catch (IOException e) {
      LOG.error("Failed to create file " + tmpClusterTicketFile, e);
      out.addError(new OutputError(Errno.E_NOT_ENOUGH_PRIVILEGES, 
          "Failed to create file " + tmpClusterTicketFile));
      return output;
    }

    MutableInt mutablerr = new MutableInt();
    byte [] encodedticket = 
        Security.EncodeDataForWritingToKeyFile(
            resp.getClusterTicketAndKey().toByteArray(), mutablerr);
    
    if (encodedticket == null) {
      out.addError(new OutputError(mutablerr.GetValue(), 
          "failed to encode ticket"));
      return output;
    }

    // Write to the file now
    try {
      FileOutputStream fout;
      fout = new FileOutputStream(tmpClusterTicketFile);
      fout.write(clusterName.getBytes("UTF-8"));
      fout.write(new String(" ").getBytes("UTF-8"));
      fout.write(encodedticket);
      fout.write(new String("\n").getBytes("UTF-8"));
      fout.close();
    } catch (FileNotFoundException e) {
      LOG.error("Failed to open file " + tmpClusterTicketFile, e);
      out.addError(new OutputError(Errno.E_NOT_ENOUGH_PRIVILEGES, 
          "Failed to open file " + tmpClusterTicketFile));
      return output;
    } catch (IOException e) {
      LOG.error("Failed to write to file " + tmpClusterTicketFile, e);
      out.addError(new OutputError(Errno.E_NOT_ENOUGH_PRIVILEGES, 
          "Failed to write to file " + tmpClusterTicketFile));
      return output;
    }
    
    // Rename the file to the mapr cluster ticket file
    // Write to the file now
    if (!tmpClusterTicketFile.renameTo(clusterTicketFile)) {
      LOG.error("Failed to rename to file " + clusterTicketFile);
      out.addError(new OutputError(Errno.E_NOT_ENOUGH_PRIVILEGES, 
          "Failed to write to file " + clusterTicketFile));
      return output;
    }
    out.addNode(new OutputNode("Successfully wrote the mapruserticket to "
                                + maprClusterTicketFileName));
    return output;
  }

  private CommandOutput genMfsUtilTicket() throws CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);

    String maprServerTicketFileName = 
        getParamTextValue(MAPR_SERVER_TICKETFILE_PARAM_NAME, 0);
    if ((maprServerTicketFileName == null) || 
         maprServerTicketFileName.isEmpty()) {                
      out.addError(new OutputError(Errno.EINVAL, "Invalid maprserverticketfile path")
      .setField(MAPR_SERVER_TICKETFILE_PARAM_NAME)
      .setFieldValue(maprServerTicketFileName));
      return output;
    }

    String maprUserTicketFileName = 
        getParamTextValue(TICKETFILE_PARAM_NAME, 0);
    if ((maprUserTicketFileName == null) || 
         maprUserTicketFileName.isEmpty()) {                
      out.addError(new OutputError(Errno.EINVAL, "Invalid mapruserticketfile path")
      .setField(TICKETFILE_PARAM_NAME)
      .setFieldValue(maprUserTicketFileName));
      return output;
    }
    
    // Get the cluster name from mapr-clusters.conf file 
    String clusterName = CLDBRpcCommonUtils.getInstance().getCurrentClusterName();
    int error = Security.SetTicketAndKeyFile(maprServerTicketFileName);
    if (error != 0) {
      out.addError(new OutputError(error, 
                  "Failed to load the maprserverticketfile " 
                  + maprServerTicketFileName));
      return output;            
    }

    MutableInt err = new MutableInt();
    TicketAndKey mfsUtilsTicket = Security.GetTicketAndKeyForCluster(
                                    ServerKeyType.CldbKey,
                                    clusterName, err);
    if (mfsUtilsTicket == null) {
        out.addError(new OutputError(err.GetValue(),
                     "Failed to generate mfs utils ticket and key"));
        return output;
    }

    String username = mfsUtilsTicket.getUserCreds().getUserName();
    int uid = mfsUtilsTicket.getUserCreds().getUid();
    List<Integer> gidsList = mfsUtilsTicket.getUserCreds().getGidsList();
    int numGids = gidsList.size();
    int[] gids = new int[numGids];
    for(int index = 0; index < numGids; index++) {
      gids[index] = gidsList.get(index);
    }
    long expiryTime = Security.MAX_EXPIRY_TIME;
    TicketAndKey ticketAndKey =
        Security.GenerateTicketAndKeyUsingServerKey(
            ServerKeyType.MfsUtilKey,
            mfsUtilsTicket.getUserKey(),
            username, uid, gids, expiryTime,
            0/*non-renewable ticket*/, false /*isExternal*/, err);

    if (ticketAndKey == null) {
      out.addError(new OutputError(err.GetValue(), "Failed to generate ticketfile"));
      return output;
    }

    byte [] encodedticket =
        Security.EncodeDataForWritingToKeyFile(ticketAndKey.toByteArray(), err);

    if (encodedticket == null) {
      out.addError(new OutputError(err.GetValue(),
          "failed to encode ticketfile"));
      return output;
    }

    String ticketFileName= getParamTextValue(TICKETFILE_PARAM_NAME, 0);
    if ((ticketFileName == null) || ticketFileName.isEmpty()) {
      out.addError(new OutputError(Errno.EINVAL, "Invalid ticketfile path")
      .setField(TICKETFILE_PARAM_NAME)
      .setFieldValue(ticketFileName));
      return output;
    }

    File ticketFile = new File(ticketFileName);
    if (ticketFile.exists()) {
      out.addError(new OutputError(Errno.EEXIST,
          "ticketfile " + ticketFileName + " already exists"));
      return output;
    }

    // Create the ticket file 
    try {
      if (ticketFile.createNewFile() == false) {
        out.addError(new OutputError(Errno.EEXIST,
            "ticketfile " + ticketFileName + " already exists"));
        return output;
      }

      // TODO : Put JNI method to create the file with perm 600
      // Following is indirect mechanism in java to set the perm
      // to 600
      ticketFile.setExecutable(false, false);
      ticketFile.setWritable(false, false);
      ticketFile.setReadable(false, false);
      ticketFile.setWritable(true, true);
      ticketFile.setReadable(true, true);
    } catch (IOException e) {
      LOG.error("Failed to create file " + ticketFileName, e);
      out.addError(new OutputError(Errno.E_NOT_ENOUGH_PRIVILEGES,
          "Failed to create file " + ticketFileName));
      return output;
    }

    try {
      FileOutputStream fout;
      fout = new FileOutputStream(ticketFile);
      fout.write(clusterName.getBytes("UTF-8"));
      fout.write(new String(" ").getBytes("UTF-8"));
      fout.write(encodedticket);
      fout.write(new String("\n").getBytes("UTF-8"));
      fout.close();
    } catch (FileNotFoundException e) {
      LOG.error("Failed to open file " + ticketFileName, e);
      out.addError(new OutputError(Errno.E_NOT_ENOUGH_PRIVILEGES,
          "Failed to open file " + ticketFileName));
      return output;
    } catch (IOException e) {
      LOG.error("Failed to write to file " + ticketFileName, e);
      out.addError(new OutputError(Errno.E_NOT_ENOUGH_PRIVILEGES,
          "Failed to write to file " + ticketFileName));
      return output;
    }
    return output;
                     
    
  }

  private CommandOutput printTicket() {
    return null;
  }

  private CommandOutput printKey() {
    // TODO Auto-generated method stub
    return null;
  }
}
