package com.mapr.cli;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.InvalidProtocolBufferException;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandLineOutput;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.TextCommandOutput;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.NoValueInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.fs.cldb.proto.CLDBProto.CLDBProg;
import com.mapr.fs.cldb.proto.CLDBProto.ResourceUsageUnit;
import com.mapr.fs.cldb.proto.CLDBProto.ResourceUsageLimit;
import com.mapr.fs.cldb.proto.CLDBProto.RLimitDiskUsage;
import com.mapr.fs.cldb.proto.CLDBProto.RLimitRequestType;
import com.mapr.fs.cldb.proto.CLDBProto.RLimitResourceType;
import com.mapr.fs.cldb.proto.CLDBProto.RLimitRequest;
import com.mapr.fs.cldb.proto.CLDBProto.RLimitResponse;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Security.CredentialsMsg;
import com.mapr.security.MaprSecurityException;

public class RlimitCommands extends CLIBaseClass implements CLIInterface {
  static final String RESOURCE_PARAM_NAME = "resource";
  static final String VALUE_PARAM_NAME = "value";
  private static final Logger LOG = Logger.getLogger(RlimitCommands.class);
  
  static final Map<String, BaseInputParameter> params = new ImmutableMap.Builder<String, BaseInputParameter>()
  .put(RlimitCommands.RESOURCE_PARAM_NAME,
      new TextInputParameter(RlimitCommands.RESOURCE_PARAM_NAME,
          "resource name [disk]",
          CLIBaseClass.REQUIRED, 
          null))
  .put(MapRCliUtil.CLUSTER_NAME_PARAM,
      new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
          "cluster name",
          CLIBaseClass.NOT_REQUIRED,
          null))
  .build();
  
  static final CLICommand getRlimitCommand = new CLICommand(
      "get",
      "obtain limit for a resource",
      RlimitCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(RlimitCommands.params)
        .build(),
      null)
  .setShortUsage("get -resource disk");
  
  static final CLICommand setRlimitCommand = new CLICommand(
      "set",
      "set limit for a resource",
      RlimitCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(RlimitCommands.params)
        .put(RlimitCommands.VALUE_PARAM_NAME,
            new TextInputParameter(RlimitCommands.VALUE_PARAM_NAME,
                "limit value for the resource",
                CLIBaseClass.REQUIRED,
                null))
        .build(),
        null)
  .setShortUsage("set -resource disk -value <limit>");
  
  static final CLICommand[] rlimitSubCommands = {
    getRlimitCommand,
    setRlimitCommand,
  };
  
  static final String usageStr = "rlimit [set|get] -resource <resource>";
  public static final CLICommand rlimitCommands =
    new CLICommand(
        "rlimit",
        "usage: " + usageStr,
        CLIUsageOnlyCommand.class,
        ExecutionTypeEnum.NATIVE,
        rlimitSubCommands)
    .setShortUsage(usageStr);

  public RlimitCommands(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }

  String unitToString(ResourceUsageUnit unit) {
	  switch (unit) {
	  case B:
		  return "B";
	  case KB:
		  return "KB";
	  case MB:
		  return "MB";
	  case GB:
		  return "GB";
	  case TB:
		  return "TB";
	  case PB:
		  return "PB";
	  }
	  return "";
  }
  
  CommandOutput getRlimit() throws CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();

    String resourceType = getParamTextValue(RESOURCE_PARAM_NAME, 0);
    if (!resourceType.equalsIgnoreCase("disk")) {
    	out.addError(
    			new OutputError(Errno.EOPFAILED, 
    					"rlimit get failed for resource type " + resourceType
    					+ ". Valid resources are disk.")
    			.setField(RESOURCE_PARAM_NAME));
    	output.setOutput(out);
    	return output;
    }
    
	RLimitRequest.Builder req = RLimitRequest.newBuilder()
											 .setCreds(getUserCredentials())
											 .setRequestType(RLimitRequestType.GET_REQUEST)
											 .setResource(RLimitResourceType.DISK_RESOURCE);
	RLimitResponse resp = null;
    try {
    	byte[] data = null;
    	if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
    		data =  CLDBRpcCommonUtils.getInstance().sendRequest(
    					getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0), 
    					Common.MapRProgramId.CldbProgramId.getNumber(),
    					CLDBProg.RLimitGetRequestProc.getNumber(),
    					req.build(), RLimitResponse.class);
    	} else {
    		data =  CLDBRpcCommonUtils.getInstance().sendRequest(
    					Common.MapRProgramId.CldbProgramId.getNumber(),
    					CLDBProg.RLimitGetRequestProc.getNumber(),
    					req.build(), RLimitResponse.class);
    	}

    	if (data == null) {
    		out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
    		output.setOutput(out);
    		return output;
    	}

    	// parse the response
    	resp = RLimitResponse.parseFrom(data);
    	int status = resp.getStatus();
    	if (status != 0) {
    		String errMsg = (resp.hasErrMsg() ? resp.getErrMsg() : Errno.toString(status));

    		out.addError(new OutputError(Errno.EOPFAILED, 
                    			"Obtaining rlimit for resource " + resourceType + 
                    			" failed with error - " + errMsg));
    		output.setOutput(out);
    		return output;          
    	}
    } catch (InvalidProtocolBufferException e) {
        throw new CLIProcessingException(
            "InvalidProtocolBufferException " + "Exception", e);
    } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
        out.addError(new OutputError(Errno.EOPFAILED, 
        	"Error sending the RPC to obtain rlimit"));
        output.setOutput(out);
        return output;    	
    }

    RLimitDiskUsage diskUsage = resp.getDiskUsage();
    String unitStr = unitToString(diskUsage.getLimit().getUnit());
    OutputNode node = new OutputNode();
    node.addChild(new OutputNode("limit", diskUsage.getLimit().getValue() + unitStr));
    node.addChild(new OutputNode("clusterSize", diskUsage.getClusterSpaceAvailableMB() + "MB"));
    node.addChild(new OutputNode("currentUsage", diskUsage.getClusterSpaceProvisionedMB() + "MB"));
    out.addNode(node);
    output.setOutput(out);
    return output;
  }
  
  ResourceUsageLimit getDiskSpace(String value) throws Exception {
	  ResourceUsageLimit.Builder result = ResourceUsageLimit.newBuilder()
	  														.setUnit(ResourceUsageUnit.B);
	  
	  // allowed numbers are of the form
	  // <number>[MGTPK][B]
	  if (value.endsWith("B") ||
		  value.endsWith("b")) {
		  // ends with byte
		  value = value.substring(0, value.length() - 1);
	  }
	  
	  if (value.endsWith("K") ||
		  value.endsWith("k")) {
		  result.setUnit(ResourceUsageUnit.KB);
		  value = value.substring(0, value.length() - 1);
	  } else if (value.endsWith("M") ||
			     value.endsWith("m")) {
		  result.setUnit(ResourceUsageUnit.MB);
		  value = value.substring(0, value.length() - 1);
	  } else if (value.endsWith("G") ||
			     value.endsWith("g")) {
		  result.setUnit(ResourceUsageUnit.GB);
		  value = value.substring(0, value.length() - 1);
	  } else if (value.endsWith("T") ||
			  	 value.endsWith("t")) {
		  result.setUnit(ResourceUsageUnit.TB);
		  value = value.substring(0, value.length() - 1);
	  } else if (value.endsWith("P") ||
			     value.endsWith("p")) {
		  result.setUnit(ResourceUsageUnit.PB);
		  value = value.substring(0, value.length() - 1);
	  }
	  
	  long num = Long.valueOf(value);
	  result.setValue(num);
	  return result.build();
  }

  CommandOutput setRlimit() throws CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();

    String resourceType = getParamTextValue(RESOURCE_PARAM_NAME, 0);
    if (!resourceType.equalsIgnoreCase("disk")) {
    	out.addError(
    			new OutputError(Errno.EOPFAILED, 
    					"rlimit set failed for resource type " + resourceType
    					+ ". Valid resources are disk.")
    			.setField(RESOURCE_PARAM_NAME));
    	output.setOutput(out);
    	return output;
    }
    
    String valueType = getParamTextValue(VALUE_PARAM_NAME, 0);
    ResourceUsageLimit usageLimit;
    try {
    	usageLimit = getDiskSpace(valueType);
    } catch (Exception e) {
    	// invalid number
    	out.addError(
    			new OutputError(Errno.EOPFAILED,
    					"Invalid number for resource disk " + valueType
    					+ ". Valid numbers are <number>[KMGTP][B].")
    				.setField(VALUE_PARAM_NAME));
    	output.setOutput(out);
    	return output;
    }
    
    try {
    	RLimitRequest.Builder req = RLimitRequest.newBuilder()
    											 .setCreds(getUserCredentials())
    											 .setRequestType(RLimitRequestType.SET_REQUEST)
    											 .setResource(RLimitResourceType.DISK_RESOURCE)
    											 .setLimit(usageLimit);
    	byte[] data = null;
    	if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
    		data =  CLDBRpcCommonUtils.getInstance().sendRequest(
    					getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0), 
    					Common.MapRProgramId.CldbProgramId.getNumber(),
    					CLDBProg.RLimitSetRequestProc.getNumber(),
    					req.build(), RLimitResponse.class);
    	} else {
    		data =  CLDBRpcCommonUtils.getInstance().sendRequest(
    					Common.MapRProgramId.CldbProgramId.getNumber(),
    					CLDBProg.RLimitSetRequestProc.getNumber(),
    					req.build(), RLimitResponse.class);
    	}

    	if (data == null) {
    		out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
    		output.setOutput(out);
    		return output;
    	}

    	// parse the response
    	RLimitResponse resp = RLimitResponse.parseFrom(data);
    	int status = resp.getStatus();
    	if (status != 0) {
    		String errMsg = (resp.hasErrMsg() ? resp.getErrMsg() : Errno.toString(status));

    		out.addError(new OutputError(Errno.EOPFAILED, 
                    			"Setting rlimit for resource " + resourceType + 
                    			" failed with error - " + errMsg));
    		output.setOutput(out);
    		return output;          
    	}
    } catch (InvalidProtocolBufferException e) {
        throw new CLIProcessingException(
            "InvalidProtocolBufferException " + "Exception", e);
    } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
        out.addError(new OutputError(Errno.EOPFAILED, 
        	"Error sending the RPC to set rlimit"));
        output.setOutput(out);
        return output;    	
    }

    output.setOutput(out);
    return output;
  }

  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    if (!super.validateInput()) {
      CommandOutput output = new CommandOutput();
      OutputHierarchy out = new OutputHierarchy();
      output.setOutput(out);
      return output;
    }

    String command = cliCommand.getCommandName();
    if (command.equalsIgnoreCase("get")) {
      return getRlimit();
    }
    
    if (command.equalsIgnoreCase("set")) {
      return setRlimit();
    }
    
    return new TextCommandOutput(("rlimit command failed: unknown command " + 
        command + " received.").getBytes());
  }
  
  @Override
  public String getCommandUsage() {
    return usageStr;
  }
}
