/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.log4j.Logger;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.ZooDefs.Ids;
import org.apache.zookeeper.ZooDefs.Perms;
import org.apache.zookeeper.data.Id;
import org.apache.zookeeper.ZooKeeper;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.json.JSONTokener;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.InvalidProtocolBufferException;
import com.mapr.baseutils.acls.SecurityCommandHelper;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.baseutils.zookeeper.ZKUtils;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.ServiceManagingOperation;
import com.mapr.cli.common.AuthManager;
import com.mapr.cli.common.NodesCommonUtils;
import com.mapr.cli.common.ServicesEnum;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.EnumInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.fs.cldb.proto.Accesscontrol.ClientAuthorizationRequest;
import com.mapr.fs.cldb.proto.Accesscontrol.ClientAuthorizationResponse;
import com.mapr.fs.cldb.proto.CLDBProto.CLDBProg;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Security.AccessControlList;
import com.mapr.fs.proto.Security.CredentialsMsg;
import com.mapr.security.Security;
import com.mapr.security.JNISecurity;

public class NodeServicesManagementCommand extends CLIBaseClass implements
		CLIInterface {

	private static final Logger LOG = Logger.getLogger(NodeServicesManagementCommand.class);
	public static final String FILTER_PARAM = "filter";
	public static final String NODES_PARAM = "nodes";
	public static final String SERVICE_NAME_PARAM = "name";
  public static final String SERVICE_MULTI_PARAM = "multi";
	public static final String SERVICE_ACTION_PARAM = "action";
	public static final String ZK_CONNECTSTRING = "zkconnect";
	public static final int TIMEOUT_SERVER = 30 * 1000;
	public static final int MAX_CHARS_COUNT = 60;

	public static final String NODES_SERVICES_MNGMNT_TOP_PATH = "/nodes";
	public static final String NODES_SERVICES_MNGMNT_SERVICE_PATH = "/services";
	
  public static List<org.apache.zookeeper.data.ACL> CREATOR_ALL_ACL_PLUS = new ArrayList<org.apache.zookeeper.data.ACL>();
  public static List<org.apache.zookeeper.data.ACL> CREATOR_ALL_ACL_PLUS_DIR = new ArrayList<org.apache.zookeeper.data.ACL>();
  
  static {
  CREATOR_ALL_ACL_PLUS.add(new org.apache.zookeeper.data.ACL(Perms.ADMIN | Perms.CREATE | Perms.WRITE | Perms.DELETE, Ids.AUTH_IDS));
  // TODO Actually only "mapr" and current user should be able to do DELETE, but I may not be able to get "mapr" user here 
  // there is no much harm is done if any authenticated user can remove znodes for "start/stop/restart"
  CREATOR_ALL_ACL_PLUS.add(new org.apache.zookeeper.data.ACL(Perms.READ, new Id("sasl", "anyone")));
  // Directory under which final znode will be created - anybody can do anything as long as this user authenticated
  CREATOR_ALL_ACL_PLUS_DIR.add(new org.apache.zookeeper.data.ACL(Perms.ALL, new Id("sasl", "anyone")));
  }
  
  public static List<org.apache.zookeeper.data.ACL> ZK_ACLS = Ids.OPEN_ACL_UNSAFE;
  public static List<org.apache.zookeeper.data.ACL> ZK_ACLS_DIR = Ids.OPEN_ACL_UNSAFE;
	
	public static final CLICommand nodeServices = new CLICommand(
			"services", "service management command ", NodeServicesManagementCommand.class, ExecutionTypeEnum.NATIVE, 
							new ImmutableMap.Builder<String,BaseInputParameter>()
							.put(MapRCliUtil.CLUSTER_NAME_PARAM,
									new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
											"cluster name",
											CLIBaseClass.NOT_REQUIRED,
											null))
							.put(FILTER_PARAM, new TextInputParameter(FILTER_PARAM, "node names filter. Please put it in quotes\"\"", CLIBaseClass.NOT_REQUIRED, null))
							.put(ZK_CONNECTSTRING, new TextInputParameter(ZK_CONNECTSTRING, "ZooKeeper Connect String: 'host:port,host:port,host:port,...'", CLIBaseClass.NOT_REQUIRED, null))
							.put(NODES_PARAM, new TextInputParameter(NODES_PARAM, "node names space separated", CLIBaseClass.NOT_REQUIRED, null))
							.put(ServicesEnum.cldb.name(), new TextInputParameter(ServicesEnum.cldb.name(), "managing cldb service: " + Arrays.asList(ServiceManagingOperation.values()), CLIBaseClass.NOT_REQUIRED, null))
							.put(ServicesEnum.fileserver.name(), new TextInputParameter(ServicesEnum.fileserver.name(), "managing fileserver service: " + Arrays.asList(ServiceManagingOperation.values()), CLIBaseClass.NOT_REQUIRED, null))
							.put(ServicesEnum.hbmaster.name(), new TextInputParameter(ServicesEnum.hbmaster.name(), "managing hbmaster service: " + Arrays.asList(ServiceManagingOperation.values()), CLIBaseClass.NOT_REQUIRED, null))
							.put(ServicesEnum.hbregionserver.name(), new TextInputParameter(ServicesEnum.hbregionserver.name(), "managing hbregionserver service: " + Arrays.asList(ServiceManagingOperation.values()), CLIBaseClass.NOT_REQUIRED, null))
							.put(ServicesEnum.jobtracker.name(), new TextInputParameter(ServicesEnum.jobtracker.name(), "managing jobtracker service: " + Arrays.asList(ServiceManagingOperation.values()), CLIBaseClass.NOT_REQUIRED, null))
							.put(ServicesEnum.nfs.name(), new TextInputParameter(ServicesEnum.nfs.name(), "managing nfs service: " + Arrays.asList(ServiceManagingOperation.values()), CLIBaseClass.NOT_REQUIRED, null))
							.put(ServicesEnum.tasktracker.name(), new TextInputParameter(ServicesEnum.tasktracker.name(), "managing tasktracker service: " + Arrays.asList(ServiceManagingOperation.values()), CLIBaseClass.NOT_REQUIRED, null))
							.put(ServicesEnum.webserver.name(), new TextInputParameter(ServicesEnum.webserver.name(), "managing webserver service: " + Arrays.asList(ServiceManagingOperation.values()), CLIBaseClass.NOT_REQUIRED, null))
							.put(SERVICE_NAME_PARAM, new TextInputParameter(SERVICE_NAME_PARAM, "service name to perform action on", CLIBaseClass.NOT_REQUIRED, null))
							.put(SERVICE_ACTION_PARAM, new EnumInputParameter(SERVICE_ACTION_PARAM, "service action. One of: " + Arrays.asList(ServiceManagingOperation.values()), CLIBaseClass.NOT_REQUIRED, ServiceManagingOperation.class, null))
              .put(SERVICE_MULTI_PARAM, new TextInputParameter(SERVICE_MULTI_PARAM, "service json to parse to start/stop multiple services", CLIBaseClass.NOT_REQUIRED, null).setInvisible(true))
							.build(),null);

  private AuthManager authManager;

	public NodeServicesManagementCommand(ProcessedInput input,
			CLICommand cliCommand) {
		super(input, cliCommand);
		authManager = AuthManager.getInstance();
	}

	@Override
	public CommandOutput executeRealCommand() throws CLIProcessingException {
	  OutputHierarchy oh = new OutputHierarchy();
	  CommandOutput co = new CommandOutput(oh);
    co.setOutput(oh);

	  String zkConnectString = null;
	  String clusterName = null;

	  CredentialsMsg creds = getUserCredentials();
	  
	  boolean isRoot = (creds.hasUid() && creds.getUid() == 0);
	  if (!isRoot) {
	    boolean isAuthorized = authManager.canPerformClusterActions(
	        SecurityCommandHelper.CLUSTER_START_STOP_SERVICES_MASK, creds, oh);
	    if (isAuthorized == false) {
	      return co;
	    }
	  }
	    
	  if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
	    clusterName = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0);
	    zkConnectString = CLDBRpcCommonUtils.getInstance().getZkConnect(clusterName);
	  } else {
	    zkConnectString = CLDBRpcCommonUtils.getInstance().getZkConnect();
	    clusterName = CLDBRpcCommonUtils.getInstance().getCurrentClusterName();
	  }
	  
    if ( JNISecurity.IsSecurityEnabled(clusterName)) {
      ZK_ACLS = CREATOR_ALL_ACL_PLUS;
      ZK_ACLS_DIR = CREATOR_ALL_ACL_PLUS_DIR;
    }

	  if ( (zkConnectString == null || zkConnectString.trim().isEmpty()) && isParamPresent(NodesCommonUtils.ZK_CONNECTSTRING) ) {
	    zkConnectString = getParamTextValue(NodesCommonUtils.ZK_CONNECTSTRING,0);
	  }
	  if ( LOG.isInfoEnabled()) {
	    LOG.info("ZKConnectString: " + zkConnectString);
	  }

	  if ( zkConnectString == null ) {
	    /**
	     * <MAPR_ERROR>
	     * Message:ZK Connect string is null. Check if ZK Connect string was provided correctly
	     * Function:NodeServicesManagementCommand.executeRealCommand()
	     * Meaning:The ZooKeeper connect string was not specified.
	     * Resolution:Check the command syntax and try again.
	     * </MAPR_ERROR>
	     */
	    String errorMsg = "Unable to obtain the ZooKeeper connection string from the CLDB. Make sure that the CLDB is running and accessible.";
	    if ( isParamPresent(ServicesEnum.cldb.name()) ) {
	      errorMsg = errorMsg + "\n If you are trying to start CLDB please specify valid Zookeeper connection string using " + NodesCommonUtils.ZK_CONNECTSTRING + " parameter";
	    }
	    oh.addError(new OutputError(Errno.EZKCANTCONNECT, errorMsg));
	    return co;    		
	  }
			
		List<String> nodes = NodesCommonUtils.findNodeIps(zkConnectString);
			
		List<String> nodeHostNames = new ArrayList<String>();
		if ( isParamPresent(NODES_PARAM)) {
			nodeHostNames = input.getParameterByName(NODES_PARAM).getParamValues();
		} else if ( isParamPresent(FILTER_PARAM) ) {
			String filter = getParamTextValue(FILTER_PARAM, 0);
			try {
				nodeHostNames = NodesCommonUtils.findFilteredNodeIps(filter, zkConnectString);
			} catch(CLIProcessingException e) {
				/**
				* <MAPR_ERROR>
				* Message:Error while trying to get nodes list. Can not do an action on service
				* Function:NodeServicesManagementCommand.executeRealCommand()
				* Meaning:An error occurred.
				* Resolution:Contact technical support.
				* </MAPR_ERROR>
				*/

				oh.addError(new OutputError(Errno.EOPFAILED, "Error while trying to get nodes list. Can not do an action on service"));
				return co;
			}
		} else {
			/**
			* <MAPR_ERROR>
			* Message:Invalid parameters provided: neither \"filter\", nor \"nodes\" was specified
			* Function:NodeServicesManagementCommand.executeRealCommand()
			* Meaning:You must specify either the {{filter}} parameter or the {{nodes}} parameter.
			* Resolution:Check the command syntax and try again.
			* </MAPR_ERROR>
			*/

			oh.addError(new OutputError(Errno.EMISSING, "Invalid parameters provided: neither \"filter\", nor \"nodes\" was specified"));
			return co;
		}

		if ( nodeHostNames == null || nodeHostNames.isEmpty() ) {
			/**
			* <MAPR_ERROR>
			* Message:No nodes match input parameters
			* Function:NodeServicesManagementCommand.executeRealCommand()
			* Meaning:No nodes matching the input parameters were found.
			* Resolution:Check the command syntax and the node names, and try again.
			* </MAPR_ERROR>
			*/

			oh.addError(new OutputError(Errno.ENOMATCH, "No nodes matched input parameters"));
			return co;
		}
		
		List<String> nodeNames = NodesCommonUtils.convertHostToIpIncludingLocal(nodeHostNames);
		nodeNames.addAll(NodesCommonUtils.convertIpToHost(nodeHostNames));
		nodeNames.addAll(nodeHostNames);
		
		Set<String> nodeNamesSet = new HashSet<String>(nodeNames);
		if ( LOG.isDebugEnabled() ) {
			LOG.debug("NodeNamesSet: " + nodeNamesSet);
		}
		nodeNamesSet.removeAll(nodes);
		nodeNames.removeAll(nodeNamesSet);
		Set<String> nodeNamesUnique = new HashSet<String>(nodeNames);
		
		if ( nodeNamesUnique.isEmpty() ) {
			/**
			* <MAPR_ERROR>
			* Message:No nodes match input parameters
			* Function:NodeServicesManagementCommand.executeRealCommand()
			* Meaning:No nodes matching the input parameters were found.
			* Resolution:Check the command syntax and the node names, and try again.
			* </MAPR_ERROR>
			*/
			StringBuilder outputErrorParams = new StringBuilder(nodeHostNames.toString());
			if ( outputErrorParams.length() > MAX_CHARS_COUNT ) {
				outputErrorParams.delete(MAX_CHARS_COUNT + 1, outputErrorParams.length());
				outputErrorParams.append("...");
			}
			oh.addError(new OutputError(Errno.ENOMATCH, "Input for " + NODES_PARAM + ": " +  outputErrorParams.toString() + " does not match the IP address or hostname of any " +
					"cluster nodes.  Please specify a node in the same format shown in the " +
					"output of the \"maprcli node list\" command"));
			LOG.error("Input for " + NODES_PARAM + ": " +  outputErrorParams.toString() + " does not match the IP address or hostname of any " +
					"cluster nodes.  Please specify a node in the same format shown in the " +
					"output of the \"maprcli node list\" command");
			return co;
		}
		
		if ( nodeNamesUnique.size() < nodeHostNames.size() ) {
			LOG.error("Not all input nodes were identified. Command will be executed only for: " + nodeNamesUnique);
			oh.addError(new OutputError(Errno.ENOMATCH, "Not all input nodes were identified. Command will be executed only for: " + nodeNamesUnique + 
					". Please specify nodes in the same format shown in the " +
					"output of the \"maprcli node list\" command"));
		}
		// would be nice to check if nodeHostNames.size() != nodeNames.size(), but as this response does not return
		// anything it will look like completely error response and not as partial
		
		// clean set of nodes to stop
		int retries = 0;
		while ( retries++ < ZKUtils.ZK_RETRIALS) {
			try {
				ZooKeeper s_zk = NodesCommonUtils.connect ( zkConnectString );
				Map<String, List<String>> confServices = NodesCommonUtils.findServicesConfiguredHierarchy(zkConnectString);
			  for ( String node : nodeNamesUnique ) {
			  	if ( LOG.isDebugEnabled() ) {
			  		LOG.debug("Node: " + node);
			  	}
				  List<String> confServicesPerNode = confServices.get(node);
				  if ( confServicesPerNode == null || confServicesPerNode.isEmpty() ) {
					LOG.error("Node: " + node + " does not have any service configured");
					oh.addError(new OutputError(Errno.EINVALMISC, "Node: " + node + " does not have any service configured"));
					// continue with next node
					continue;
				  }

				try {
					s_zk.create(NODES_SERVICES_MNGMNT_TOP_PATH, new byte[0], Ids.OPEN_ACL_UNSAFE, CreateMode.PERSISTENT);
				} catch (KeeperException.NodeExistsException e1) {
					// expected one
				} 
				try {
					s_zk.create(NODES_SERVICES_MNGMNT_TOP_PATH + "/" + node, new byte[0], Ids.OPEN_ACL_UNSAFE, CreateMode.PERSISTENT);
				} catch (KeeperException.NodeExistsException e1) {
					// expected one
				} 
				try {
					s_zk.create(NODES_SERVICES_MNGMNT_TOP_PATH + "/" + node + NODES_SERVICES_MNGMNT_SERVICE_PATH, new byte[0], Ids.OPEN_ACL_UNSAFE, CreateMode.PERSISTENT);
				} catch (KeeperException.NodeExistsException e1) {
					// expected one
				} 
				for ( ServicesEnum service : ServicesEnum.values()) {
					if ( isParamPresent(service.name()) ) {
						String serviceOperation = getParamTextValue(service.name(), 0);
						try {
							ServiceManagingOperation.valueOf(serviceOperation);
						} catch (IllegalArgumentException e) {
							/**
							* <MAPR_ERROR>
							* Message:Service operation type for service: <service name> is not valid: <service operation>
							* Function:NodeServicesManagementCommand.executeRealCommand()
							* Meaning:An error occurred.
							* Resolution:Contact technical support.
							* </MAPR_ERROR>
							*/

							LOG.error("Service operation type for service: " + service.name() + " is not valid: " + serviceOperation);
							oh.addError(new OutputError(Errno.ECLDBINFOINVALID, "Service operation type for service: " + service.name() + " is not valid: " + serviceOperation).setField(service.name()).setFieldValue(serviceOperation));
							// continue with next service
							continue;
						}
						if ( !confServicesPerNode.contains(service.name()) ) {
							LOG.error("Service: " + service.name() + " is not configured on node: " + node);
							oh.addError(new OutputError(Errno.EINVALMISC, "Service: " + service.name() + " is not configured on node: " + node));
							// continue with next service
							continue;
						}
						helperAction(s_zk, node, service.name(), serviceOperation, oh);
					}
				}
				if (isParamPresent(SERVICE_NAME_PARAM)) {
					String serviceName = getParamTextValue(SERVICE_NAME_PARAM, 0);
					if (!isParamPresent(SERVICE_ACTION_PARAM)) {
						/**
						* <MAPR_ERROR>
						* Message: <service action> parameter has to be present when <service name> is present
						* Function:NodeServicesManagementCommand.executeRealCommand()
						* Meaning:You must specify an action to perform on the specified service.
						* Resolution:Check the command syntax and try again.
						* </MAPR_ERROR>
						*/

						LOG.error(SERVICE_ACTION_PARAM + " parameter has to be present when " + SERVICE_NAME_PARAM + " is present");
						oh.addError(new OutputError(Errno.EMISSING, SERVICE_ACTION_PARAM + " parameter has to be present when " + SERVICE_NAME_PARAM + " is present"));
						return co;
					}

					if ( getParamObjectValue(SERVICE_ACTION_PARAM, 0) instanceof ServiceManagingOperation) {
						ServiceManagingOperation action = (ServiceManagingOperation) getParamObjectValue(SERVICE_ACTION_PARAM, 0);
						if ( !confServicesPerNode.contains(serviceName) ) {
							LOG.error("Service: " + serviceName + " is not configured on node: " + node);
							oh.addError(new OutputError(Errno.EINVALMISC, "Service: " + serviceName + " is not configured on node: " + node));
							// continue with next service
							continue;
						}
						helperAction(s_zk, node, serviceName, action.name(), oh);
					} else {
						/**
						* <MAPR_ERROR>
						* Message:action on the service is not of correct Enum type. You can use only one of following: <service operations>
						* Function:NodeServicesManagementCommand.executeRealCommand()
						* Meaning:An incorrect service action was specified.
						* Resolution:Check the command syntax and try again.
						* </MAPR_ERROR>
						*/

						LOG.error("action on the service is not of correct Enum type. You can use only one of following: " + Arrays.asList(ServiceManagingOperation.values()));
						oh.addError(new OutputError(Errno.EINVAL, "action on the service is not of correct Enum type. You can use only one of following: " + Arrays.asList(ServiceManagingOperation.values())));
						return co;
					}
				}
        if (isParamPresent(SERVICE_MULTI_PARAM)) {
          String multiParam = getParamTextValue(SERVICE_MULTI_PARAM, 0);
          JSONTokener token = new JSONTokener(multiParam);
          try {
            JSONArray jarray = new JSONArray(token);
            for (int i = 0; i < jarray.length(); i++) {
              JSONObject obj = jarray.getJSONObject(i);
              String serviceName = obj.getString("name");
              if (!obj.has("action")) {
                /**
                 * <MAPR_ERROR>
                 * Message: <service action> parameter has to be present when <service name> is present
                 * Function:NodeServicesManagementCommand.executeRealCommand()
                 * Meaning:You must specify an action to perform on the specified service.
                 * Resolution:Check the command syntax and try again.
                 * </MAPR_ERROR>
                 */

                LOG.error("action parameter has to be present when name is present");
                oh.addError(new OutputError(Errno.EMISSING, "action parameter has to be present when name is present"));
                return co;
              }

              String strAction = obj.getString("action");
              ServiceManagingOperation action = null;

              for (ServiceManagingOperation s : ServiceManagingOperation.values()) {
                if (s.name().equals(strAction)) {
                  action = s;
                  break;
                }
              }
              if (action != null) {
                if ( !confServicesPerNode.contains(serviceName) ) {
                  LOG.error("Service: " + serviceName + " is not configured on node: " + node);
                  oh.addError(new OutputError(Errno.EINVALMISC, "Service: " + serviceName + " is not configured on node: " + node));
                  // continue with next service
                  continue;
                }
                helperAction(s_zk, node, serviceName, action.name(), oh);
              } else {
                /**
                 * <MAPR_ERROR>
                 * Message:action on the service is not of correct Enum type. You can use only one of following: <service operations>
                 * Function:NodeServicesManagementCommand.executeRealCommand()
                 * Meaning:An incorrect service action was specified.
                 * Resolution:Check the command syntax and try again.
                 * </MAPR_ERROR>
                 */

                LOG.error("action on the service is not of correct Enum type. You can use only one of following: " + Arrays.asList(ServiceManagingOperation.values()));
                oh.addError(new OutputError(Errno.EINVAL, "action on the service is not of correct Enum type. You can use only one of following: " + Arrays.asList(ServiceManagingOperation.values())));
                return co;
              }
            }
          } catch (JSONException e) {
            LOG.error("Error parsing JSON string: " + multiParam);
            oh.addError(new OutputError(Errno.EINVAL, "Error parsing JSON string"));
          }
        }
			  }
			break;
		} catch (KeeperException e) {
			/**
			* <MAPR_ERROR>
			* Message:KeeperException while trying to create ZK struct for service management
			* Function:NodeServicesManagementCommand.executeRealCommand()
			* Meaning:An error occurred.
			* Resolution:Contact technical support.
			* </MAPR_ERROR>
			*/

			LOG.error("KeeperException while trying to create ZK struct for service management", e);
			oh.addError(new OutputError(Errno.EZKCANTCONNECT, "KeeperException while trying to create ZK struct for service management" + e.getLocalizedMessage()));
			return co;
		} catch (InterruptedException e) {
			/**
			* <MAPR_ERROR>
			* Message:InterruptedException while trying to create ZK struct for service management
			* Function:NodeServicesManagementCommand.executeRealCommand()
			* Meaning:An error occurred.
			* Resolution:Contact technical support.
			* </MAPR_ERROR>
			*/

			LOG.error("InterruptedException while trying to create ZK struct for service management", e);
			oh.addError(new OutputError(Errno.EZKCANTCONNECT, "InterruptedException while trying to create ZK struct for service management" + e.getLocalizedMessage()));
			return co;
		}
	}
		return co;
	}
	
	private void helperAction(ZooKeeper s_zk, String node, String service, String serviceOperation, OutputHierarchy oh) throws KeeperException, InterruptedException {
		try {
			s_zk.create(NODES_SERVICES_MNGMNT_TOP_PATH + "/" + node + NODES_SERVICES_MNGMNT_SERVICE_PATH + "/" + service, new byte[0], ZK_ACLS_DIR, CreateMode.PERSISTENT);
		} catch (KeeperException.NodeExistsException e1) {
			// expected one
		}
		try {
			// output time when node was created
			byte data[] = String.valueOf(System.currentTimeMillis()).getBytes();
			String createdNode = s_zk.create(NODES_SERVICES_MNGMNT_TOP_PATH + "/" + node + NODES_SERVICES_MNGMNT_SERVICE_PATH + "/" + service + "/" + serviceOperation + "_", data, ZK_ACLS, CreateMode.PERSISTENT_SEQUENTIAL);
			if ( LOG.isInfoEnabled() )
			  LOG.info("Created znode: " +  createdNode + " to " + serviceOperation + " service " + service);
		} catch (KeeperException.NodeExistsException e1) {
			// too bad - most likely somebody already issued a command to control service, but either it was not yet executed, or node was not removed for some reason
			LOG.warn("Please wait. The previous " + serviceOperation + " command for " + service + " is executing.");
			oh.addError(new OutputError(Errno.EOPFAILED, "Please wait. The previous " + serviceOperation + " command for " + service + " is executing.").setField(service).setFieldValue(serviceOperation));
			// continue with next service
		}
	}
}
