/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.math.BigInteger;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.common.collect.ImmutableMap;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.baseutils.Errno;
import com.mapr.cli.common.NodesCommonUtils;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.FileServerRemoveRequest;
import com.mapr.fs.cldb.proto.CLDBProto.FileServerRemoveResponse;
import com.mapr.fs.cldb.proto.CLDBProto.Service;
import com.mapr.fs.proto.Common;

public class NodeRemoveCommand extends CLIBaseClass implements CLIInterface {

	public static final String FILTER_PARAM = "filter";
	public static final String NODES_PARAM = "nodes";
	public static final String HOSTIDS_PARAM = "hostids";
	public static final String ZK_CONNECTSTRING = "zkconnect";
  public static final String SERVICE_PARAM = "service";
  public static final String KEEP_DISKS_PARAM = "keepDisks";

  public static final String SERVICE_NFSSERVER_ARG = "nfsserver";
  public static final String SERVICE_FILESERVER_ARG = "fileserver";
	
	public static final int TIMEOUT_SERVER = 30 * 1000;
	public static final String SERVER_PATH = "/servers";
	public static long TIMEOUT = 3600000L;
	public static final Log LOG = LogFactory.getLog(NodeRemoveCommand.class);
	
	private String zkConnectString;
	
	public static final CLICommand nodeRemove = new CLICommand(
			"remove", "remove node from service ", NodeRemoveCommand.class, ExecutionTypeEnum.NATIVE, 
							new ImmutableMap.Builder<String,BaseInputParameter>()
							.put(FILTER_PARAM, new TextInputParameter(FILTER_PARAM, "node names filter. Please put it in quotes\"\"", CLIBaseClass.NOT_REQUIRED, null))
							.put(ZK_CONNECTSTRING, new TextInputParameter(ZK_CONNECTSTRING, "ZooKeeper Connect String: 'host:port,host:port,host:port,...'", CLIBaseClass.NOT_REQUIRED, null))
							.put(NODES_PARAM, new TextInputParameter(NODES_PARAM, "space-separated list of node names", CLIBaseClass.NOT_REQUIRED, null))
							.put(HOSTIDS_PARAM, new TextInputParameter(HOSTIDS_PARAM, "space-separated list of hostids", CLIBaseClass.NOT_REQUIRED, null))
							.put(SERVICE_PARAM, new TextInputParameter(SERVICE_PARAM, "Service to be removed. Either fileserver or nfsserver.", CLIBaseClass.NOT_REQUIRED, null))
							.put(KEEP_DISKS_PARAM, 
                   new BooleanInputParameter(KEEP_DISKS_PARAM, 
                                             "Retain disks (default: false)", 
                                             CLIBaseClass.NOT_REQUIRED, 
                                             null).setInvisible(true))
							.build(),	null);

	
	public NodeRemoveCommand(ProcessedInput input, CLICommand cliCommand) {
		super(input, cliCommand);
	}

	@Override
	public CommandOutput executeRealCommand() throws CLIProcessingException {
		OutputHierarchy oh = new OutputHierarchy();
		CommandOutput co = new CommandOutput(oh);
		
		   	zkConnectString = CLDBRpcCommonUtils.getInstance().getZkConnect();
	    	if ( (zkConnectString == null || zkConnectString.trim().isEmpty()) && isParamPresent(NodesCommonUtils.ZK_CONNECTSTRING)) {
	    		zkConnectString = getParamTextValue(NodesCommonUtils.ZK_CONNECTSTRING,0);
	    	}
	    	if ( zkConnectString == null ) {
	    		/**
	    		* <MAPR_ERROR>
	    		* Message:ZK Connect string is null. Check if ZK Connect string was provided correctly
	    		* Function:NodeRemoveCommand.executeRealCommand()
	    		* Meaning:The specified ZooKeeper connect string was NULL.
	    		* Resolution:Check the command syntax and try again.
	    		* </MAPR_ERROR>
	    		*/
	    		oh.addError(new OutputError(Errno.EZKCANTCONNECT, "ZK Connect string is null. Check if ZK Connect string was provided correctly"));
				return co;
	    	}

    LOG.info("NodeRemove: zkConnectString = " + zkConnectString);

		List<String> nodeHostNames = new ArrayList<String>();
		List<Long> nodeHostIds = new ArrayList<Long>();
		if ( isParamPresent(NODES_PARAM)) {
			List<String> nodeNames = input.getParameterByName(NODES_PARAM).getParamValues();
			for (String node : nodeNames) {
				// split the nodes string based on space as
				// UI sends it as a single space separated string.
				// For CLI, this code path ends up copying "nodeNames" list
				// into "nodeHostNames" list.
				Collections.addAll(nodeHostNames, node.split(" "));
			}   
    } else if ( isParamPresent(HOSTIDS_PARAM)) {
      List<String> hostIdsList = input.getParameterByName(HOSTIDS_PARAM).getParamValues();
      for (String hostIds : hostIdsList) {
        // split the hostids string based on space as
        // UI sends it as a single space separated string.
        // For CLI, this code path ends up copying "hostIds" list
        // into "nodeHostIds" list.
        for (String hostId : hostIds.split(" ")) {
          try {
            long value = new BigInteger(hostId.trim()).longValue();
            nodeHostIds.add(value);
          } catch (Exception e) {
						/**
						* <MAPR_ERROR>
						* Message:Error while trying to parse hostids. Cannot do an action on service
						* Function:NodeRemoveCommand.executeRealCommand()
						* Meaning:An error occurred.
						* Resolution:Contact technical support.
						* </MAPR_ERROR>
						*/
						oh.addError(new OutputError(Errno.EOPFAILED,
                         "Error while trying to parse hostid "
                         +  hostId.trim()));
						return co;
          }
        }
      }
		} else if ( isParamPresent(FILTER_PARAM) ) {
			String filter = getParamTextValue(FILTER_PARAM, 0);
			try {
				nodeHostNames = NodesCommonUtils.findFilteredNodeIps(filter, zkConnectString);
			} catch(CLIProcessingException e) {
				/**
				* <MAPR_ERROR>
				* Message:Error while trying to get nodes list. Cannot do an action on service
				* Function:NodeRemoveCommand.executeRealCommand()
				* Meaning:An error occurred.
				* Resolution:Contact technical support.
				* </MAPR_ERROR>
				*/
				oh.addError(new OutputError(Errno.EOPFAILED, "Error while trying to get nodes list. Cannot do an action on service"));
				return co;
			}
		} else {
			/**
			* <MAPR_ERROR>
			* Message:Invalid parameters provided: neither \"filter\", nor \"nodes\" was specified
			* Function:NodeRemoveCommand.executeRealCommand()
			* Meaning:You must specify either the {{filter}} parameter or the {{nodes}} parameter.
			* Resolution:Check the command syntax and try again.
			* </MAPR_ERROR>
			*/
			oh.addError(new OutputError(Errno.EMISSING, "Invalid parameters provided: neither \"filter\", nor \"nodes\" was specified"));
			return co;
		}

		if ( (nodeHostNames == null || nodeHostNames.isEmpty()) &&
         (nodeHostIds == null || nodeHostIds.isEmpty()) ) {
			/**
			* <MAPR_ERROR>
			* Message:No nodes match input parameters
			* Function:NodeRemoveCommand.executeRealCommand()
			* Meaning:No nodes matching the input parameters were found.
			* Resolution:Check the command syntax and the node names, and try again.
			* </MAPR_ERROR>
			*/
			oh.addError(new OutputError(Errno.ENOMATCH, "No nodes match input parameters"));
			return co;
		}
		
    if ( isParamPresent(SERVICE_PARAM) ) {
      String service = getParamTextValue(SERVICE_PARAM, 0);
      if (!service.equalsIgnoreCase(SERVICE_FILESERVER_ARG) &&
          !service.equalsIgnoreCase(SERVICE_NFSSERVER_ARG)) {
        oh.addError(new OutputError(Errno.ENOMATCH, "Unknown service"));
        return co;
      }
    }

		/*List<String> nodeNames = NodesCommonUtils.convertIpToHost(nodeHostNames);
		if (nodeNames != null && !nodeNames.isEmpty()) {
		  processNodes(nodeNames, oh);
		}*/
		processNodes(nodeHostNames, nodeHostIds, oh);
		return co;
	}
	
	private void processNodes(List<String> nodeHostNames,
      List<Long> nodeHostIds,
	    OutputHierarchy oh) throws CLIProcessingException {
	  
    String cluster = null;
    if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
      cluster = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0);
      if (!CLDBRpcCommonUtils.getInstance().isValidClusterName(cluster)) {
        oh.addError(new OutputError(Errno.EUCLUSTER, "Invalid cluster: " + cluster));
        return;
      }
    }
  
    Service serviceId = Service.SERVICE_ALL;
    if ( isParamPresent(SERVICE_PARAM) ) {
      String service = getParamTextValue(SERVICE_PARAM, 0);
      if (service.equalsIgnoreCase(SERVICE_FILESERVER_ARG)) {
        serviceId = Service.SERVICE_FILESERVER;
      } else if (service.equalsIgnoreCase(SERVICE_NFSSERVER_ARG)) {
        serviceId = Service.SERVICE_NFSSERVER;
      } 
    }

    boolean keepDisks = false;
    if ( isParamPresent(KEEP_DISKS_PARAM) ) {
      keepDisks = getParamBooleanValue(KEEP_DISKS_PARAM, 0);
    }

	  // First remove fileserver from CLDB
    if (nodeHostNames != null) {
	    for (int i = 0 ; i < nodeHostNames.size(); i++) {
	      String nodeName = nodeHostNames.get(i).trim();
        if (nodeName == null || nodeName.isEmpty())
          continue;
     
        FileServerRemoveRequest req = FileServerRemoveRequest.newBuilder()
                                      .setHostname(nodeName)
                                      .setCreds(getUserCredentials())
                                      .setService(serviceId)
                                      .setKeepDisks(keepDisks)
                                      .build();
        
        LOG.info("NodeRemove: Attempting to remove node: " + nodeName);
     
	      // Remove FileServer
        byte[] replyData = null;
        try {
          if (cluster != null) {
            replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
                cluster, Common.MapRProgramId.CldbProgramId.getNumber(),
                CLDBProto.CLDBProg.FileServerRemoveProc.getNumber(), 
                req, FileServerRemoveResponse.class);
          } else {
            replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
                Common.MapRProgramId.CldbProgramId.getNumber(),
                CLDBProto.CLDBProg.FileServerRemoveProc.getNumber(),
                req, FileServerRemoveResponse.class);
          }
        } catch (Exception e) {
          throw new CLIProcessingException(e);
        }
   
        if ( replyData == null ) {
      	  oh.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
      	  continue;
        }
     
        try {
          FileServerRemoveResponse resp = FileServerRemoveResponse
                                                .parseFrom(replyData);
          if (resp.getStatus() == 0) {
            LOG.info("NodeRemove: Removed node: " + nodeName);
            continue;
          }
     
          String msg = "node remove failed for node " + nodeName + ", Error: ";
          switch(resp.getStatus()) {
            case Errno.ENOENT:
              msg += "No such hostname, need valid hostname";
              break;
     
            case Errno.EBUSY:
              msg += Errno.toString(resp.getStatus()) + 
                      "\n\n1. Stop services on the node: " + nodeName +
                      "\n2. Wait for 5 minutes until NODE_ALARM_NO_HEARTBEAT is raised for node: " + nodeName + 
                      "\n3. Run this operation again";
              break;
     
            default:
              msg += Errno.toString(resp.getStatus());
              break;
          }
     
          if (resp.hasErrMsg())
            msg += "; " + resp.getErrMsg();
          oh.addError(new OutputError(resp.getStatus(), msg));
          LOG.error(msg);
     
        } catch (Exception e) {
          LOG.error("node remove: failed with Exception " +
              e + " for server " + nodeName);
          oh.addError(new OutputError(Errno.EOPFAILED, 
              "node remove: failed with Exception " + 
              e + " for server " + nodeName));
          continue;
        } 
	    }
    }
    if (nodeHostIds != null) {
      for (Long hostId : nodeHostIds) {
        FileServerRemoveRequest req = FileServerRemoveRequest.newBuilder()
                                      .setFileServerId(hostId)
                                      .setService(serviceId)
                                      .setCreds(getUserCredentials())
                                      .setKeepDisks(keepDisks)
                                      .build();
        
        LOG.info("NodeRemove: Attempting to remove hostid: " + hostId);
     
	      // Remove FileServer
        byte[] replyData = null;
        try {
          if (cluster != null) {
            replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
                cluster, Common.MapRProgramId.CldbProgramId.getNumber(),
                CLDBProto.CLDBProg.FileServerRemoveProc.getNumber(), 
                req, FileServerRemoveResponse.class);
          } else {
            replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
                Common.MapRProgramId.CldbProgramId.getNumber(),
                CLDBProto.CLDBProg.FileServerRemoveProc.getNumber(),
                req, FileServerRemoveResponse.class);
          }
        } catch (Exception e) {
          throw new CLIProcessingException(e);
        }
   
        if ( replyData == null ) {
      	  oh.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
      	  continue;
        }
     
        try {
          FileServerRemoveResponse resp = FileServerRemoveResponse
                                                .parseFrom(replyData);
          if (resp.getStatus() == 0) {
            LOG.info("NodeRemove: Removed hostid: " + hostId);
            continue;
          }
     
          String msg = "node remove failed for hostid: " + hostId + ", Error: ";
          switch(resp.getStatus()) {
            case Errno.ENOENT:
              msg += "No such hostid, need valid hostid";
              break;
     
            case Errno.EBUSY:
              msg += Errno.toString(resp.getStatus()) + 
                      "\n\n1. Stop services on the node with ID: " + hostId + 
                      "\n2. Wait for 5 minutes until NODE_ALARM_NO_HEARTBEAT is raised for node: " + hostId + 
                      "\n3. Run this operation again";
              break;
     
            default:
              msg += Errno.toString(resp.getStatus());
              break;
          }
     
          if (resp.hasErrMsg())
            msg += "; " + resp.getErrMsg();
          oh.addError(new OutputError(resp.getStatus(), msg));
          LOG.error(msg);
     
        } catch (Exception e) {
          LOG.error("node remove: failed with Exception " +
              e + " for hostid " + hostId);
          oh.addError(new OutputError(Errno.EOPFAILED, 
              "node remove: failed with Exception " + 
              e + " for hostid " + hostId));
          continue;
        } 
	    }
    }
	  return;
	}	
}
