/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.log4j.Logger;
import org.apache.zookeeper.ZooKeeper;

import com.google.common.collect.ImmutableMap;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.baseutils.Errno;
import com.mapr.cli.common.NodesCommonUtils;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;

@Deprecated
public class NodePathSwitchCommand extends CLIBaseClass implements CLIInterface {

	private static final Logger LOG = Logger.getLogger(NodePathSwitchCommand.class);
	
	public static final String FILTER_PARAM = "filter";
	public static final String NODES_PARAM = "nodes";
	public static final String WHICH_PARAM = "which";
	public static final String PATH_PARAM = "path";
	public static final String ZK_CONNECTSTRING = "zkconnect";
	public static final int TIMEOUT_SERVER = 30 * 1000;

	public static final String NODES_SERVICES_MNGMNT_TOP_PATH = "/nodes";
	public static final String NODES_SERVICES_MNGMNT_SERVICE_PATH = "/services";
	

	public static final CLICommand nodePathCommand = new CLICommand(
			"path", "path change command ", NodePathSwitchCommand.class, ExecutionTypeEnum.NATIVE, 
							new ImmutableMap.Builder<String,BaseInputParameter>()
							.put(FILTER_PARAM, new TextInputParameter(FILTER_PARAM, "node names filter. Please put it in quotes\"\"", CLIBaseClass.NOT_REQUIRED, null))
							.put(ZK_CONNECTSTRING, new TextInputParameter(ZK_CONNECTSTRING, "ZooKeeper Connect String: 'host:port,host:port,host:port,...'", CLIBaseClass.NOT_REQUIRED, "localhost:5181"))
							.put(NODES_PARAM, new TextInputParameter(NODES_PARAM, "node names space separated", CLIBaseClass.NOT_REQUIRED, null))
							.put(WHICH_PARAM, new TextInputParameter(WHICH_PARAM, "which path to change: switch, rack or both", CLIBaseClass.NOT_REQUIRED, PathTypeEnum.RACK.getExtName()))
							.put(PATH_PARAM, new TextInputParameter(PATH_PARAM, "path to change", CLIBaseClass.REQUIRED, null))
							.put(MapRCliUtil.CLUSTER_NAME_PARAM,
									new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
											"cluster name",
											CLIBaseClass.NOT_REQUIRED,null))
							.build(),null);

	public NodePathSwitchCommand(ProcessedInput input, CLICommand cliCommand) {
		super(input, cliCommand);
	}

	@Override
	public CommandOutput executeRealCommand() throws CLIProcessingException {
		OutputHierarchy oh = new OutputHierarchy();
		CommandOutput co = new CommandOutput(oh);
		
		   	String zkConnectString = CLDBRpcCommonUtils.getInstance().getZkConnect();
	    	if ( zkConnectString == null ) {
	    		getParamTextValue(NodesCommonUtils.ZK_CONNECTSTRING,0);
	    	}

			List<String> nodes = NodesCommonUtils.findNodeIps(zkConnectString);
	
			List<String> nodeHostNames = new ArrayList<String>();
			if ( isParamPresent(NODES_PARAM)) {
				nodeHostNames = input.getParameterByName(NODES_PARAM).getParamValues();
			} else if ( isParamPresent(FILTER_PARAM) ) {
				String filter = getParamTextValue(FILTER_PARAM, 0);
				try {
					nodeHostNames = NodesCommonUtils.findFilteredNodeIps(filter, zkConnectString);
				} catch(CLIProcessingException e) {
					/**
					* <MAPR_ERROR>
					* Message:Error while trying to get nodes list. Cannot do an action on service
					* Function:NodePathSwitchCommand.executeRealCommand()
					* Meaning:An error occurred.
					* Resolution:Contact technical support.
					* </MAPR_ERROR>
					*/

					oh.addError(new OutputError(Errno.EOPFAILED, "Error while trying to get nodes list. Cannot do an action on service"));
					return co;
				}
			} else {
				/**
				* <MAPR_ERROR>
				* Message:Invalid parameters provided: neither \"filter\", nor \"nodes\" were specified
				* Function:NodePathSwitchCommand.executeRealCommand()
				* Meaning:You must specify either the {{filter}} parameter or the {{node}} parameter.
				* Resolution:Check the command syntax and try again.
				* </MAPR_ERROR>
				*/
				oh.addError(new OutputError(Errno.EMISSING, "Invalid parameters provided: neither \"filter\", nor \"nodes\" were specified"));
				return co;
			}
	
			if ( nodeHostNames == null || nodeHostNames.isEmpty() ) {
				/**
				* <MAPR_ERROR>
				* Message:No nodes match input parameters
				* Function:NodePathSwitchCommand.executeRealCommand()
				* Meaning:No nodes matching the input parameters were found.
				* Resolution:Check the command syntax and the node names, and try again.
				* </MAPR_ERROR>
				*/
				oh.addError(new OutputError(Errno.ENOMATCH, "No nodes match input parameters"));
				return co;
			}
			
			List<String> nodeNames = NodesCommonUtils.convertHostToIp(nodeHostNames);
			
			Set<String> nodeNamesSet = new HashSet<String>(nodeNames);
			nodeNamesSet.removeAll(nodes);
			nodeNames.removeAll(nodeNamesSet);
	
			// TODO add functionality to change path
			// your node names to use: nodeNames
			
		return co;
	}

	/**
	 * Enum to handle change options
	 * Had to introduce getExtName() method - as "switch" is reserved word
	 * @author yufeldman
	 *
	 */
	public static enum PathTypeEnum {
		SWITCH {
			public String getExtName() {
				return "switch";
			}
		},
		RACK {
			public String getExtName() {
				return "rack";
			}
		},
		BOTH {
			public String getExtName() {
				return "both";
			}
		};
		
		public String getExtName() {
			return null;
		}
	}
}
