/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import java.io.EOFException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.text.ParseException;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.Path;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.InvalidProtocolBufferException;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.baseutils.Errno;
import com.mapr.cli.DiskCommands.DiskEntryField;
import com.mapr.cli.common.NodesCommonUtils;
import com.mapr.cli.common.RemoteCommandExecutor;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import com.mapr.cliframework.base.inputparams.IntegerInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.fs.MapRFileSystem;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.FileServerRemoveRequest;
import com.mapr.fs.cldb.proto.CLDBProto.FileServerRemoveResponse;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.clustermetrics.ClusterMetricsProto.*;
import com.mapr.fs.proto.clustermetrics.ClusterMetricsProto.NodeMetric.*;
import com.mapr.util.MapRFSUtil;
import com.mapr.fs.proto.clustermetrics.ClusterMetricsProto.CommandId;

public class NodeMetricsCommand extends CLIBaseClass implements CLIInterface {

  private static final Log LOG = LogFactory.getLog(NodeMetricsCommand.class);
  public static final String NODES_PARAM = "nodes";
  public static final String START_TIME_PARAM = "start";
  public static final String END_TIME_PARAM = "end";
  public static final String INTERVAL_PARAM = "interval";
  public static final String EVENTS_PARAM = "events";
  public static final String COLUMNS_PARAM = "columns";
  public static final String NODE_PARAM = "node";
  public static final String OUTPUT_PARAM_NAME = "output";
  public static final String PROCESSES_NUMBER_PARAM_NAME = "numberofprocesses";
  
  public static final int MIN_INTERVAL = 10; // 10 seconds


  static String[][] processListFieldNames = {
   {"mem","memory"},
   {"pid","processid"},
   {"cpu","cpu"},
   {"euser","user"},
   {"sz","realmemoryusage"},
   {"rss","virtualmemoryusage"},
   {"pri","priority"},
   {"s","processstatuscode"},
   {"cmd","command"}
  };

  public static final CLICommand nodeMetricsCmd = new CLICommand(
      "metrics",
      "display node metrics ",
      NodeMetricsCommand.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String,BaseInputParameter>()
        .put(NODES_PARAM,
            new TextInputParameter(NODES_PARAM,
                "space-separated list of node names",
                CLIBaseClass.REQUIRED,
                null))
        .put(START_TIME_PARAM,
            new TextInputParameter(START_TIME_PARAM,
                "start_time (UTC timestamp in millisecond or a UTC date in MM/DD/YY format)",
                CLIBaseClass.REQUIRED,
                null))
        .put(END_TIME_PARAM,
            new TextInputParameter(END_TIME_PARAM,
                "end_time (UTC timestamp in millisecond or a UTC date in MM/DD/YY format)",
                CLIBaseClass.REQUIRED,
                null))
        .put(INTERVAL_PARAM,
            new IntegerInputParameter(INTERVAL_PARAM,
                "interval (in seconds. Minimum value is 10 seconds)",
                CLIBaseClass.NOT_REQUIRED,
                null))
        .put(EVENTS_PARAM,
              new BooleanInputParameter(EVENTS_PARAM,
                  "print node events only",
                  CLIBaseClass.NOT_REQUIRED,
                  false))
        .put(COLUMNS_PARAM,
              new TextInputParameter(COLUMNS_PARAM,
                  "columns",
                  CLIBaseClass.NOT_REQUIRED,
                  null))               
        .put(MapRCliUtil.CLUSTER_NAME_PARAM,
              new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
                  "cluster name",
                  CLIBaseClass.NOT_REQUIRED,
                  null))
        .build(),
      null)
  .setShortUsage("metrics -nodes <nodes> -start <start time> -end <end time> [-interval <interval seconds> -events <1|0> -cluster <clustername>]");

  public static final CLICommand showMemoryCmd = new CLICommand(
      "show",
      "display services memory",
      NodeMetricsCommand.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String,BaseInputParameter>()
        .put(NODE_PARAM,
           new TextInputParameter(NODE_PARAM,
             "node name",
              CLIBaseClass.REQUIRED,
              null))
        .put(PROCESSES_NUMBER_PARAM_NAME,
           new TextInputParameter(PROCESSES_NUMBER_PARAM_NAME,
             "number of processes ",
              CLIBaseClass.NOT_REQUIRED,
              "10"))
        .put(DiskCommands.OUTPUT_PARAM_NAME,
           new TextInputParameter(OUTPUT_PARAM_NAME,
             "<terse|verbose>",
             CLIBaseClass.NOT_REQUIRED,
             "verbose"))
        .build(),
      null)
  .setShortUsage("show -node <node>");

  private class MetricsDate {
    public int month;
    public int date;
    public int year;
    public int hour;

    MetricsDate(int m, int d, int y, int h) {
      month = m;
      date = d;
      year = y;
      hour = h;
    }

    MetricsDate(Date d) {
      month = d.getMonth() + 1;
      date = d.getDate();
      year = 1900 + d.getYear();
      hour = d.getHours();
    }
  }

  public NodeMetricsCommand(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }

  private static String prefixedInt(int n) {
    return ((n < 10) ? "0":"") + n;
  }
  
  private boolean canPrintMetricColumn (String [] columnsArr, String column) {
  	// If no columns were specified in command line, print all columns
  	if (columnsArr == null || columnsArr.length == 0)
  		return true;
  	
  	for (String c: columnsArr) {
  		if (c.equalsIgnoreCase(column))
  			return true;
  	}
  	
  	return false;
  }

  private void printNodeMetrics (String node, NodeMetric nm, long timestamp, boolean events,
                                 OutputHierarchy out, String [] columnsArr) {

    if (!nm.hasMetricAttrs())
      return;

    if (!events) {
      OutputNode metricsNode = new OutputNode();
      // Always print Timestamps and hostname
      metricsNode.addChild(new OutputNode("NODE", node));
      metricsNode.addChild(new OutputNode("TIMESTAMPSTR", new Date(timestamp)));      
      metricsNode.addChild(new OutputNode("TIMESTAMP", timestamp));

      // Add CPU Metrics
      if (nm.getCpuMetricsList() != null && nm.getCpuMetricsCount() > 0 && 
      		canPrintMetricColumn(columnsArr, "CPUS")) {
      	OutputNode cpuNodes = new OutputNode("CPUS");

        for (CpuMetrics cm: nm.getCpuMetricsList()) {
          OutputNode cpuNode = new OutputNode(cm.getCpuName());

          if (cm.hasCpuIdle())
            cpuNode.addChild(new OutputNode("CPUIDLE", cm.getCpuIdle()));
          if (cm.hasCpuIowait())
            cpuNode.addChild(new OutputNode("CPUIOWAIT", cm.getCpuIowait()));
          if (cm.hasCpuTotal())
            cpuNode.addChild(new OutputNode("CPUTOTAL", cm.getCpuTotal()));

          cpuNodes.addChild(cpuNode);
        }
        if (cpuNodes.getChildren() != null && !cpuNodes.getChildren().isEmpty())
          metricsNode.addChild(cpuNodes);
      }

      // Add Disk Metrics
      if (nm.getDiskMetricsList() != null && nm.getDiskMetricsCount() > 0 && 
      		canPrintMetricColumn(columnsArr, "DISKS")) {
        OutputNode diskNodes = new OutputNode("DISKS");

        for (DiskMetrics dm: nm.getDiskMetricsList()) {
          OutputNode diskNode = new OutputNode(dm.getDiskName());

          if (dm.hasReadOps())
            diskNode.addChild(new OutputNode("READOPS", dm.getReadOps()));
          if (dm.hasReadKBytes())
            diskNode.addChild(new OutputNode("READKB", dm.getReadKBytes()));
          if (dm.hasWriteOps())
            diskNode.addChild(new OutputNode("WRITEOPS", dm.getWriteOps()));
          if (dm.hasWriteKBytes())
            diskNode.addChild(new OutputNode("WRITEKB", dm.getWriteKBytes()));

          diskNodes.addChild(diskNode);
        }
        if (diskNodes.getChildren() != null && !diskNodes.getChildren().isEmpty())
          metricsNode.addChild(diskNodes);
      }

      // Add Network Metrics
      if (nm.getNetMetricsList() != null && nm.getNetMetricsCount() > 0 && 
      		canPrintMetricColumn(columnsArr, "NETWORK")) {
        OutputNode networkNodes = new OutputNode("NETWORK");

        for (NetworkMetrics ntm: nm.getNetMetricsList()) {
          OutputNode networkNode = new OutputNode(ntm.getIfaceName());

          if (ntm.hasBytesIn())
            networkNode.addChild(new OutputNode("BYTESIN", ntm.getBytesIn()));
          if (ntm.hasBytesOut())
            networkNode.addChild(new OutputNode("BYTESOUT", ntm.getBytesOut()));
          if (ntm.hasPktsIn())
            networkNode.addChild(new OutputNode("PKTSIN", ntm.getPktsIn()));
          if (ntm.hasPktsOut())
            networkNode.addChild(new OutputNode("PKTSOUT", ntm.getPktsOut()));

          networkNodes.addChild(networkNode);
        }
        if (networkNodes.getChildren() != null && !networkNodes.getChildren().isEmpty())
          metricsNode.addChild(networkNodes);
      }

      // Add Final Metrics
      if (nm.hasFinalMetric()) {
        FinalMetric fm = nm.getFinalMetric();
        if (fm != null && fm.hasHostid() && 
        		canPrintMetricColumn(columnsArr, "HOSTID"))
          metricsNode.addChild(new OutputNode("HOSTID", fm.getHostid()));
      }

      // Add Misc Metrics
      if (nm.hasCpuUptime() && canPrintMetricColumn(columnsArr, "CPUTIME"))
        metricsNode.addChild(new OutputNode("CPUUPTIME", nm.getCpuUptime()));
      if (nm.hasMemoryUsed() && canPrintMetricColumn(columnsArr, "MEMORYUSED"))
        metricsNode.addChild(new OutputNode("MEMORYUSED", nm.getMemoryUsed()));
      if (nm.hasCpuIdle() && canPrintMetricColumn(columnsArr, "CPUIDLE"))
        metricsNode.addChild(new OutputNode("CPUIDLE", nm.getCpuIdle()));
      if (nm.hasReadOps() && canPrintMetricColumn(columnsArr, "READOPS"))
        metricsNode.addChild(new OutputNode("READOPS", nm.getReadOps()));
      if (nm.hasReadKBytes() && canPrintMetricColumn(columnsArr, "READKBYTES"))
        metricsNode.addChild(new OutputNode("READKBYTES", nm.getReadKBytes()));
      if (nm.hasWriteOps() && canPrintMetricColumn(columnsArr, "WRITEOPS"))
        metricsNode.addChild(new OutputNode("WRITEOPS", nm.getWriteOps()));
      if (nm.hasWriteKBytes() && canPrintMetricColumn(columnsArr, "WRITEKBYTES"))
        metricsNode.addChild(new OutputNode("WRITEKBYTES", nm.getWriteKBytes()));
      if (nm.hasBytesIn() && canPrintMetricColumn(columnsArr, "BYTESIN"))
        metricsNode.addChild(new OutputNode("BYTESIN", nm.getBytesIn()));
      if (nm.hasBytesOut() && canPrintMetricColumn(columnsArr, "BYTESOUT"))
        metricsNode.addChild(new OutputNode("BYTESOUT", nm.getBytesOut()));
      if (nm.hasServerUsedSizeMB() && canPrintMetricColumn(columnsArr, "SERVUSEDSIZEMB"))
        metricsNode.addChild(new OutputNode("SERVUSEDSIZEMB", nm.getServerUsedSizeMB()));
      if (nm.hasServerAvailableSizeMB() && canPrintMetricColumn(columnsArr, "SERVAVAILSIZEMB"))
        metricsNode.addChild(new OutputNode("SERVAVAILSIZEMB", nm.getServerAvailableSizeMB()));
      if (nm.hasRpcCount() && canPrintMetricColumn(columnsArr, "RPCCOUNT"))
        metricsNode.addChild(new OutputNode("RPCCOUNT", nm.getRpcCount()));
      if (nm.hasRpcInBytes() && canPrintMetricColumn(columnsArr, "RPCINBYTES"))
        metricsNode.addChild(new OutputNode("RPCINBYTES", nm.getRpcInBytes()));
      if (nm.hasRpcOutBytes() && canPrintMetricColumn(columnsArr, "RPCOUTBYTES"))
        metricsNode.addChild(new OutputNode("RPCOUTBYTES", nm.getRpcOutBytes()));
      if (nm.hasCpuNice() && canPrintMetricColumn(columnsArr, "CPUNICE"))
        metricsNode.addChild(new OutputNode("CPUNICE", nm.getCpuNice()));
      if (nm.hasCpuSystem() && canPrintMetricColumn(columnsArr, "CPUSYSTEM"))
        metricsNode.addChild(new OutputNode("CPUSYSTEM", nm.getCpuSystem()));
      if (nm.hasCpuUser() && canPrintMetricColumn(columnsArr, "CPUUSER"))
        metricsNode.addChild(new OutputNode("CPUUSER", nm.getCpuUser()));
      if (nm.hasMemoryCached() && canPrintMetricColumn(columnsArr, "MEMORYCACHED"))
        metricsNode.addChild(new OutputNode("MEMORYCACHED", nm.getMemoryCached()));
      if (nm.hasMemoryShared() && canPrintMetricColumn(columnsArr, "MEMORYSHARED"))
        metricsNode.addChild(new OutputNode("MEMORYSHARED", nm.getMemoryShared()));
      if (nm.hasMemoryBuffers() && canPrintMetricColumn(columnsArr, "MEMORYBUFFERS"))
        metricsNode.addChild(new OutputNode("MEMORYBUFFERS", nm.getMemoryBuffers()));
      if (nm.hasSwapFree() && canPrintMetricColumn(columnsArr, "SWAPFREE"))
        metricsNode.addChild(new OutputNode("SWAPFREE", nm.getSwapFree()));
      if (nm.hasPktsIn() && canPrintMetricColumn(columnsArr, "PKTSIN"))
        metricsNode.addChild(new OutputNode("PKTSIN", nm.getPktsIn()));
      if (nm.hasPktsOut() && canPrintMetricColumn(columnsArr, "PKTSOUT"))
        metricsNode.addChild(new OutputNode("PKTSOUT", nm.getPktsOut()));
      if (nm.hasLoadOnePerc() && canPrintMetricColumn(columnsArr, "LOAD1PERCENT"))
        metricsNode.addChild(new OutputNode("LOAD1PERCENT", nm.getLoadOnePerc()));
      if (nm.hasLoadFivePerc() && canPrintMetricColumn(columnsArr, "LOAD5PERCENT"))
        metricsNode.addChild(new OutputNode("LOAD5PERCENT", nm.getLoadFivePerc()));
      if (nm.hasLoadFifteenPerc() && canPrintMetricColumn(columnsArr, "LOAD15PERCENT"))
        metricsNode.addChild(new OutputNode("LOAD15PERCENT", nm.getLoadFifteenPerc()));
      if (nm.hasProcRun() && canPrintMetricColumn(columnsArr, "PROCRUN"))
        metricsNode.addChild(new OutputNode("PROCRUN", nm.getProcRun()));
      if (nm.hasTtmapused() && canPrintMetricColumn(columnsArr, "TTMAPUSED"))
        metricsNode.addChild(new OutputNode("TTMAPUSED", nm.getTtmapused()));
      if (nm.hasTtreduceused() && canPrintMetricColumn(columnsArr, "TTREDUCEUSED"))
        metricsNode.addChild(new OutputNode("TTREDUCEUSED", nm.getTtreduceused()));

      out.addNode(metricsNode);

    } else if (nm.hasEventMetric()) {
      OutputNode eventsNode = new OutputNode();
      eventsNode.addChild(new OutputNode("NODE", node));
      eventsNode.addChild(new OutputNode("TIMESTAMP", timestamp));

      NodeMetric.EventMetric nEventMetric = nm.getEventMetric();

      if (nEventMetric.hasDiskCount() && canPrintMetricColumn(columnsArr, "DISCKCOUNT"))
        eventsNode.addChild(new OutputNode("DISKCOUNT", nEventMetric.getDiskCount()));
      if (nEventMetric.hasCpuCount() && canPrintMetricColumn(columnsArr, "CPUCOUNT"))
        eventsNode.addChild(new OutputNode("CPUCOUNT", nEventMetric.getCpuCount()));
      if (nEventMetric.hasTotalMB() && canPrintMetricColumn(columnsArr, "TOTALMEMORY"))
        eventsNode.addChild(new OutputNode("TOTALMEMORY", nEventMetric.getTotalMB()));
      if (nEventMetric.hasRootFull() && canPrintMetricColumn(columnsArr, "ROOTFULLSTATUS"))
        eventsNode.addChild(new OutputNode("ROOTFULLSTATUS", nEventMetric.getRootFull()));
      if (nEventMetric.hasOptMaprFull() && canPrintMetricColumn(columnsArr, "MAPRFULLSTATUS"))
        eventsNode.addChild(new OutputNode("MAPRFULLSTATUS", nEventMetric.getOptMaprFull()));
      if (nEventMetric.hasCorePresent() && canPrintMetricColumn(columnsArr, "COREPRESENTSTATUS"))
        eventsNode.addChild(new OutputNode("COREPRESENTSTATUS", nEventMetric.getCorePresent()));
      if (nEventMetric.hasNicCount() && canPrintMetricColumn(columnsArr, "TOTALNICS"))
        eventsNode.addChild(new OutputNode("TOTALNICS", nEventMetric.getNicCount()));
      if (nEventMetric.hasFaileddisks() && canPrintMetricColumn(columnsArr, "FAILEDDISKS"))
        eventsNode.addChild(new OutputNode("FAILEDDISKS", nEventMetric.getFaileddisks()));
      if (nEventMetric.hasMaprdiskCount() && canPrintMetricColumn(columnsArr, "MAPRDISKCOUNT"))
        eventsNode.addChild(new OutputNode("MAPRDISKCOUNT", nEventMetric.getMaprdiskCount()));
      if (nEventMetric.hasServerCapacitySizeMB() && canPrintMetricColumn(columnsArr, "SERVCAPACITYSIZEMB"))
        eventsNode.addChild(new OutputNode("SERVCAPACITYSIZEMB", nEventMetric.getServerCapacitySizeMB()));
      if (nEventMetric.hasSwapTotal() && canPrintMetricColumn(columnsArr, "SWAPTOTAL"))
        eventsNode.addChild(new OutputNode("SWAPTOTAL", nEventMetric.getSwapTotal()));
      if (nEventMetric.hasTtmapslots() && canPrintMetricColumn(columnsArr, "TTMAPSLOTS"))
        eventsNode.addChild(new OutputNode("TTMAPSLOTS", nEventMetric.getTtmapslots()));
      if (nEventMetric.hasTtreduceslots() && canPrintMetricColumn(columnsArr, "TTREDUCESLOTS"))
        eventsNode.addChild(new OutputNode("TTREDUCESLOTS", nEventMetric.getTtreduceslots()));

      if (nEventMetric.getConfServiceList() != null && nEventMetric.getConfServiceCount() > 0) {
        for (Pair p: nEventMetric.getConfServiceList()) {
        	if (canPrintMetricColumn(columnsArr, p.getName()))
            eventsNode.addChild(new OutputNode(p.getName(), p.getValue()));
        }
      }

      if (nEventMetric.getRunningServiceList() != null && nEventMetric.getRunningServiceCount() > 0) {
        for (Pair p: nEventMetric.getRunningServiceList()) {
        	if (canPrintMetricColumn(columnsArr, p.getName()))
            eventsNode.addChild(new OutputNode(p.getName(), p.getValue()));
        }
      }

      if (nEventMetric.getServiceFailedList() != null && nEventMetric.getServiceFailedCount() > 0) {
        for (Pair p: nEventMetric.getServiceFailedList()) {
        	if (canPrintMetricColumn(columnsArr, p.getName()))
            eventsNode.addChild(new OutputNode(p.getName(), p.getValue()));
        }
      }

      if (nEventMetric.getServiceStoppedList() != null && nEventMetric.getServiceStoppedCount() > 0) {
        for (Pair p: nEventMetric.getServiceStoppedList()) {
        	if (canPrintMetricColumn(columnsArr, p.getName()))
            eventsNode.addChild(new OutputNode(p.getName(), p.getValue()));
        }
      }

      if (nEventMetric.getAlarmsList() != null && nEventMetric.getAlarmsCount() > 0) {
        for (Pair p: nEventMetric.getAlarmsList()) {
        	if (canPrintMetricColumn(columnsArr, p.getName()))
            eventsNode.addChild(new OutputNode(p.getName(), p.getValue()));
        }
      }

      out.addNode(eventsNode);
    }
  }

  private long parseNodeMetricsFromProto (String node, byte [] buf, int interval,
                                         boolean events, OutputHierarchy out,
                                         String [] columnsArr, long prevTimestamp)
        throws InvalidProtocolBufferException {

    CombineMessageRequest.Builder requestBuilder
             = CombineMessageRequest.newBuilder().mergeFrom(buf);    

    for (TimedMetric tm : requestBuilder.getTimedMetricList()) {
    	if (interval > MIN_INTERVAL && 
    			tm.getTimestamp() < prevTimestamp + 1000 * interval) {
    		continue;
    	}
    	prevTimestamp = tm.getTimestamp();
      for (Metric m: tm.getMetricList()) {
        if (m.hasNodeMetric()) {        	
          printNodeMetrics(node, m.getNodeMetric(), tm.getTimestamp(), events, 
          		             out, columnsArr);
        }
      }
    }
    return prevTimestamp;
  }

  private long readNodeMetricsFromFile(String node, 
                                       MapRFileSystem fs, String metricsFile,
                                       int startHour, int endHour, int interval,
                                       boolean events, OutputHierarchy out,
                                       String [] columnsArr, long prevTimestamp) {
    FSDataInputStream fsdisM = null;

    try {
      if (!fs.isFile(new Path(metricsFile))) {
        return prevTimestamp;
      }

      fsdisM = fs.open(new Path(metricsFile));
      // read file length and compare with file length - if do not match - there is an issue somewhere
      int fileLength = fsdisM.readInt();

       while (true) {
         int nextRecordbytes = fsdisM.readInt();
         byte [] buf = new byte[nextRecordbytes];
         int readBytesCount = fsdisM.read(buf, 0, nextRecordbytes);
         if ( readBytesCount < 0 ) {
            if ( fileLength == nextRecordbytes ) {
              LOG.info("Reached end of file sucessfully. Lengths match: " + fileLength);
            } else {
              // might be that file is still being written into
              LOG.warn("Warning while reading data file. " + 
                       "File might be still being written into. " + 
                       "Length does not match. First length: " + fileLength + 
                       ", Last Length: " + nextRecordbytes);
            }
            if ( fsdisM != null ) {
              fsdisM.close();
            }
            return prevTimestamp;
         }
         prevTimestamp = parseNodeMetricsFromProto(node, buf, interval, events, out, 
                                                   columnsArr, prevTimestamp);
        }
    }  catch (EOFException e) {
      try {
        if ( fsdisM != null) {
          fsdisM.close();
        }
      } catch (Exception e1) {
        out.addError(new OutputError(Errno.EOPFAILED, 
                     "Exception while reading metrics file: " + metricsFile + 
                     " for node: " + node + ", " + e.getMessage()));
      }
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, 
                   "Exception while reading metrics file: " + metricsFile + 
                   " for node: " + node + ", " + e.getMessage()));
    }

    return prevTimestamp;
  }

  private void readNodeMetricsFromMapRFs(String node, MetricsDate start, MetricsDate end,
                                         int interval, boolean events,
                                         OutputHierarchy out, String [] columnsArr) {

    /* Set the metrics Dir */
    String metricsDir = "/var/mapr/local/" + node + "/metrics/";
    try {
      if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        String cluster = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0);
        if (cluster != null && !cluster.isEmpty()) {
          metricsDir = "/mapr/" + cluster + metricsDir;
        }
      }
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, 
                        "Could not fetch cluster name: " + e.getMessage()));
      return;
    }

    /* Open connection to MapR-FS */
    MapRFileSystem fs = MapRFSUtil.getMapRFileSystem();
    if (fs == null) {
      out.addError(new OutputError(Errno.EOPFAILED, "Could not connect to MapR-FS"));
      return;
    }

    try {
      if (!fs.isDirectory(new Path(metricsDir))) {
         out.addError(new OutputError(Errno.ENOENT, 
                      "Could not find metrics directory " + metricsDir + 
                      " for node: " + node));
         return;
      }
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, 
                   "Exception while checking metrics directory " + metricsDir + 
                   " for node: " + node + ", " + e.getMessage()));
      return;
    }

    long prevTimestamp = 0;

    for (int y = start.year; y <= end.year; y++) {
      for (int m = 1; m <= 12; m++) {
        if (y == start.year && m < start.month)
          continue;
        if (y == end.year && m > end.month)
          break;
        for (int d = 1; d <= 31; d++) {
          if (y == start.year && m == start.month && d < start.date)
            continue;
          if (y == end.year && m == end.month && d > end.date)
            break;

          int startHour = 0;
          int endHour = 24;

          if (y == start.year && m == start.month && d == start.date)
            startHour = start.hour;
          else if (y == end.year && m == end.month && d == end.date && end.hour != 0)
            endHour = end.hour;

          for ( int hour = startHour; hour < endHour; hour++ ) {
            String metricsFile = metricsDir + "clustermetrics." +
                                 y + "-" + prefixedInt(m) + "-" + prefixedInt(d) + 
                                 "." + prefixedInt(hour) +
                                 ":00:00";          
            prevTimestamp = readNodeMetricsFromFile(node, fs, metricsFile, 
                                                    startHour, endHour,
                                                    interval, events, 
                                                    out, columnsArr, 
                                                    prevTimestamp);
          }
        }
      }
    }
  }

  private Date getValidDate (OutputHierarchy out, String param, String keyword) {
  	
  	try {
  		String value = getParamTextValue(param, 0);
  		if (value == null) {
	  	  out.addError(new OutputError(Errno.EINVAL, "Invalid " + keyword + " time"));
	      return null;
	  	}
  		
  		Date dt = null;
  		try {
        dt = new Date(Long.parseLong(value)); // try if value is a timestamp
      } catch (NumberFormatException nfe) {
      	try {
          DateFormat dateInstance = DateFormat.getDateInstance(DateFormat.SHORT);
          dateInstance.setLenient(false);
          dt = dateInstance.parse(value); // try if value is a date string (mm/dd/yyyy)
      	} catch (ParseException pe) {
      		out.addError(new OutputError(Errno.EINVAL, "Invalid " + keyword + " time"));
      		return null;
      	}
      }
      
      if (dt == null)
      	return null;
      
      // Date should be > Unix Epoch
      if (!dt.after(DateFormat.getDateInstance(DateFormat.SHORT).parse("1/1/70"))) {
        out.addError(new OutputError(Errno.EINVAL, 
        		keyword + " time cannot be before Unix epoch (1/1/70)"));
        return null;
      }
      
      return dt;
      
  	} catch (Exception e) {
  		out.addError(new OutputError(Errno.EOPFAILED,
          "Exception while parsing " + keyword + " date " + e.getMessage()));
  	} 
  	return null;
  }

  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    OutputHierarchy oh = new OutputHierarchy();
    CommandOutput co = new CommandOutput(oh);

    String cmd = cliCommand.getCommandName();
    if (cmd.equalsIgnoreCase("show")) {
      try {
        String nodeName = getParamTextValue(NODE_PARAM, 0);
        String numProcesses = getParamTextValue(PROCESSES_NUMBER_PARAM_NAME, 0);
        getServicesMemory(nodeName,numProcesses,oh);
      } catch (Exception e) {
        oh.addError(new OutputError(Errno.EOPFAILED, e.getLocalizedMessage()));
        return co;
      }
    } else if (cmd.equalsIgnoreCase("metrics")) {
      List<String> nodeNames = input.getParameterByName(NODES_PARAM).getParamValues();
      if ( nodeNames == null || nodeNames.isEmpty() ) {
        /**
         * <MAPR_ERROR>
         * Message:Invalid list of nodes
         * Function:NodeMetricsCommand.executeRealCommand()
         * Meaning:No nodes matching the input parameters were found.
         * Resolution:Check the command syntax and the node names, and try again.
         * </MAPR_ERROR>
         */
        oh.addError(new OutputError(Errno.EINVAL, "Invalid list of nodes"));
        return co;
      }
      Date startDate = getValidDate(oh, START_TIME_PARAM, "Start");
      if (startDate == null)
        return co;
      Date endDate = getValidDate(oh, END_TIME_PARAM, "End");
      if (endDate == null)
        return co;

      // startDate should be <= endDate
      if (endDate.compareTo(startDate) < 0) {
        oh.addError(new OutputError(Errno.EINVAL,
            "End time should be greater than Start time"));
        return co;
      }

      boolean events = getParamBooleanValue(EVENTS_PARAM, 0);
      int interval = -1;

      if (isParamPresent(INTERVAL_PARAM)) {
        int i = getParamIntValue(INTERVAL_PARAM, 0);
        if (i < MIN_INTERVAL) {
          oh.addError(new OutputError(Errno.EINVAL, 
              "Interval should be atleast 10 seconds"));
          return co;
        }
        interval = i;
      }

      String [] columnsArr = null;

      if (isParamPresent(COLUMNS_PARAM)) {
        String columns = getParamTextValue(COLUMNS_PARAM, 0);
        if (columns != null)
          columns = columns.trim();
          columnsArr = columns.split(",");    	
      }

      if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        String cluster = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0);
        if (!CLDBRpcCommonUtils.getInstance().isValidClusterName(cluster)) {
          oh.addError(new OutputError(Errno.EUCLUSTER, "Invalid cluster: " + cluster));
          return co;
        }
      }

      for (String node: nodeNames) {
        readNodeMetricsFromMapRFs(node, new MetricsDate(startDate), 
            new MetricsDate(endDate), interval, 
            events, oh, columnsArr);
      }
    }
    return co;
  }

  public void getServicesMemory(String host, String numProcesses,OutputHierarchy oh) throws CLIProcessingException {
    ExecuteCommandRequest.Builder requestBuilder = ExecuteCommandRequest.newBuilder();
    requestBuilder.setCommandId(CommandId.PROCESS_MEMORY_LIST);
    requestBuilder.setArgs(numProcesses);
    requestBuilder.setCreds(getUserCredentials());

    try {
      List<String> response = RemoteCommandExecutor.execute(host, requestBuilder.build());
      for(String s: response) {
        boolean verbose = true;
        if (isParamPresent(OUTPUT_PARAM_NAME) &&
            getParamTextValue(OUTPUT_PARAM_NAME, 0).equalsIgnoreCase("terse")) {
          verbose = false;
        }
        s = s.trim();
        if (s.isEmpty())
          continue;
        OutputNode node = new OutputNode();
        StringTokenizer tokenizer = new StringTokenizer(s);
        String[] outputValues = new String[tokenizer.countTokens()];
        int count=0;
        while(tokenizer.hasMoreTokens()) {
          outputValues[count++] = tokenizer.nextToken();
        }
        for(int i=0;i<processListFieldNames.length;i++) {
          String columnName = verbose ? processListFieldNames[i][1] : processListFieldNames[i][0].toString();
          node.addChild(new OutputNode(columnName, outputValues[i]));
        }

        oh.addNode(node);
      }
    } catch(CLIProcessingException e) {
      throw new CLIProcessingException("Error trying to reach host \"" + host +"\". Check if hostname is valid and up");
    }
  }
}
