/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.data.Stat;

import com.google.common.collect.ImmutableMap;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.zookeeper.ZKUtils;
import com.mapr.cli.common.ServicesEnum;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;

import com.mapr.fs.proto.Common.ServiceData;
import com.google.protobuf.InvalidProtocolBufferException;

public class NodeInstallCommand extends CLIBaseClass implements CLIInterface {

	private static final Logger LOG = Logger.getLogger(NodeInstallCommand.class);
	
	public static final String NODE_PARAM = "node";
	public static final String ROLES_PARAM = "roles";
	public static final String DISKS_PARAM = "disks";
	public static final String TOPO_PARAM = "topology";
	
	private static final String CLDB_SERVICE_MASTER_PATH = "/services/cldb/" + ZKUtils.SERVICE_MASTER_NODE;
	
	private ZooKeeper s_zk;
	
	public static final CLICommand nodeRemove = new CLICommand(
			"install", "install Mapr software on a node ", NodeRemoveCommand.class, ExecutionTypeEnum.NATIVE, 
							new ImmutableMap.Builder<String,BaseInputParameter>()
							.put(NODE_PARAM, new TextInputParameter(NODE_PARAM, "node names ", CLIBaseClass.REQUIRED, null))
							.put(NodeRemoveCommand.ZK_CONNECTSTRING, new TextInputParameter(NodeRemoveCommand.ZK_CONNECTSTRING, "ZooKeeper Connect String: 'host:port,host:port,host:port,...'", CLIBaseClass.NOT_REQUIRED, "localhost:5181"))
							.put(ROLES_PARAM, new TextInputParameter(ROLES_PARAM, "services to install.", CLIBaseClass.REQUIRED, null))
							.put(DISKS_PARAM, new TextInputParameter(DISKS_PARAM, "disks to format. Has no effect at present.usus all", CLIBaseClass.NOT_REQUIRED, "all"))
							.put(TOPO_PARAM, new TextInputParameter(TOPO_PARAM, "topology. Has no effect at present. Uses default one", CLIBaseClass.NOT_REQUIRED, null))
							.build(),null);

	public NodeInstallCommand(ProcessedInput input, CLICommand cliCommand) {
		super(input, cliCommand);
	}

	@Override
	public CommandOutput executeRealCommand() throws CLIProcessingException {
		OutputHierarchy oh = new OutputHierarchy();
		CommandOutput co = new CommandOutput(oh);

		String nodeIp = getParamTextValue(NODE_PARAM, 0);
		List<String> roles = input.getParameterByName(NODE_PARAM).getParamValues();
		
		int cldbPort = -1;
		String cldbHost = null;
		
		try {
			connect ( getParamTextValue(NodeRemoveCommand.ZK_CONNECTSTRING, 0) );
		} catch (IOException e) {
			/**
			* <MAPR_ERROR>
			* Message:Cannot connect to ZooKeeper with error: <error>
			* Function:NodeInstallCommand.executeRealCommand()
			* Meaning:A communication error occurred.
			* Resolution:Make sure the ZooKeeper is running, has a quorum, and is reachable over the network.
			* </MAPR_ERROR>
			*/
			oh.addError(new OutputError(Errno.EZKCANTCONNECT, "Cannot connect to ZooKeeper with error:" + e.getLocalizedMessage()));
			return co;
		}

		List<String> nodes = new ArrayList<String>();
		try {
			Stat stats = s_zk.exists(CLDB_SERVICE_MASTER_PATH, false );
			if ( stats != null ) {
				// cldb is known get IP and port
				byte[] cldbBytes = s_zk.getData(CLDB_SERVICE_MASTER_PATH, null, stats);
				ServiceData cldbInfo = ServiceData.parseFrom(cldbBytes);
				if (!cldbInfo.hasHost() || !cldbInfo.hasPort()) {
					/**
					* <MAPR_ERROR>
					* Message:CLDB Master Info is invalid. Cannot Install node: <IP address>
					* Function:NodeInstallCommand.executeRealCommand()
					* Meaning:A serious error occurred.
					* Resolution:Contact technical support.
					* </MAPR_ERROR>
					*/
					oh.addError(new OutputError(Errno.ECLDBINFOINVALID, "CLDB Master Info is invalid. Cannot Install node: " + nodeIp));
					return co;
				}
				cldbHost = cldbInfo.getHost();
				cldbPort = cldbInfo.getPort();
			}
			
		} catch (KeeperException e) {
			// TODO retry for lost connection - although unlikely to happen
			/**
			* <MAPR_ERROR>
			* Message:Cannot get data <CLDB service master path> with error: <message>
			* Function:NodeInstallCommand.executeRealCommand()
			* Meaning:A serious error occurred.
			* Resolution:Contact technical support.
			* </MAPR_ERROR>
			*/
			oh.addError(new OutputError(Errno.EZKCANTCONNECT, "Cannot get data " + CLDB_SERVICE_MASTER_PATH + "with error: "+ e.getLocalizedMessage()));
			return co;
		} catch (InterruptedException e) {
			/**
			* <MAPR_ERROR>
			* Message:Cannot get data <CLDB service master path> with error: <message>
			* Function:NodeInstallCommand.executeRealCommand()
			* Meaning:A serious error occurred.
			* Resolution:Contact technical support.
			* </MAPR_ERROR>
			*/
			oh.addError(new OutputError(Errno.EZKCANTCONNECT,"Cannot get data " + CLDB_SERVICE_MASTER_PATH + "with error: "+ e.getLocalizedMessage()));
			return co;
		} catch (InvalidProtocolBufferException e) {
			/**
			* <MAPR_ERROR>
			* Message:Cannot get data <CLDB service master path> with error: <message>
			* Function:NodeInstallCommand.executeRealCommand()
			* Meaning:A serious error occurred.
			* Resolution:Contact technical support.
			* </MAPR_ERROR>
			*/
			oh.addError(new OutputError(Errno.EZKCANTCONNECT,"Cannot get data " + CLDB_SERVICE_MASTER_PATH + "with error: "+ e.getLocalizedMessage()));
			return co;
		}

		if ( cldbPort != -1 && cldbHost != null ) {
			// cldb credentials are found
			try {
				// TODO actually go to CLDB machine and run script from there
				String retString = new String(executeSimpleSHHCommand(NodeRemoveCommand.TIMEOUT, "ssh root@" + cldbHost + " clustersetup.sh"));
				if ( LOG.isInfoEnabled() ) {
					LOG.info(retString);
				}
				// start warden there 
				retString = new String(executeSimpleSHHCommand(NodeRemoveCommand.TIMEOUT, "ssh root@" + nodeIp + " /etc/init.d/warden start"));
				if ( LOG.isInfoEnabled() ) {
					LOG.info(retString);
				}

			} catch( CLIProcessingException e) {
				/**
				* <MAPR_ERROR>
				* Message:Cannot stop services on : <IP address> command failed with error:  <error>
				* Function:NodeInstallCommand.executeRealCommand()
				* Meaning:A serious error occurred.
				* Resolution:Contact technical support.
				* </MAPR_ERROR>
				*/
				LOG.error("Cannot stop services on : " + nodeIp + " command failed with error: " + e.getLocalizedMessage());
				oh.addError(new OutputError(Errno.EOPFAILED,"Can not stop services on : " + nodeIp + " command failed with error: " + e.getLocalizedMessage()).setField(NODE_PARAM).setFieldValue(nodeIp));
				return co;
			}
			//oh.addNode(new OutputNode("status", "OK"));
			return co;
		}
		//TODO change this heuristics to smth. more useful later
		if ( roles.contains(ServicesEnum.cldb.name())) {
			// it is may be a control node
			try {
				// TODO actually go to CLDB machine and run script from there (as nodeIp is cldb machine)
				String retString = new String(executeSimpleSHHCommand(NodeRemoveCommand.TIMEOUT, "ssh root@" + nodeIp + " clustersetup.sh"));
				if ( LOG.isInfoEnabled() ) {
					LOG.info(retString);
				}

				// start warden there to get CLDB for later install
				retString = new String(executeSimpleSHHCommand(NodeRemoveCommand.TIMEOUT, "ssh root@" + nodeIp + " /etc/init.d/warden start"));
				if ( LOG.isInfoEnabled() ) {
					LOG.info(retString);
				}

			} catch( CLIProcessingException e) {
				/**
				* <MAPR_ERROR>
				* Message:Cannot stop services on : <IP address> command failed with error: <error>
				* Function:NodeInstallCommand.executeRealCommand()
				* Meaning:A serious error occurred.
				* Resolution:Contact technical support.
				* </MAPR_ERROR>
				*/
				LOG.error("Cannot stop services on : " + nodeIp + " command failed with error: " + e.getLocalizedMessage());
				oh.addError(new OutputError(Errno.EOPFAILED,"Cannot stop services on : " + nodeIp + " command failed with error: " + e.getLocalizedMessage()).setField(NODE_PARAM).setFieldValue(nodeIp));
				return co;
			}
			//oh.addNode(new OutputNode("status", "OK"));
			return co;
			
		}
		
		// nothing to look for here: no cldb, no cldb role
		/**
		* <MAPR_ERROR>
		* Message:Cannot install: <IP address> as CLDB is not installed/running anywhere
		* Function:NodeInstallCommand.executeRealCommand()
		* Meaning:A serious error occurred.
		* Resolution:Contact technical support.
		* </MAPR_ERROR>
		*/
		oh.addError(new OutputError(Errno.EOPFAILED,"Cannot install: " + nodeIp + " as CLDB is not installed/running anywhere").setField(NODE_PARAM).setFieldValue(nodeIp));
		return co;
	}
	
	private void connect(String zkConnectString) throws IOException {
	    try {
    		s_zk = new ZooKeeper(zkConnectString, NodeRemoveCommand.TIMEOUT_SERVER, null);
		} catch (IOException e) {
	        /**
	        * <MAPR_ERROR>
	        * Message:Could not connect to ZooKeeper server: <ZooKeeper connect string>
	        * Function:NodeInstallCommand.executeRealCommand()
	        * Meaning:An error occurred.
	        * Resolution:Contact technical support.
	        * </MAPR_ERROR>
	        */
	        LOG.fatal("Could not connect to ZooKeeper server: " + zkConnectString, e);
	        throw e;
		}
	}


}
