/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.cli;

import com.google.common.collect.ImmutableMap;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cli.common.NodesCommonUtils;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.cliframework.base.inputparams.IntegerInputParameter;
import com.mapr.security.MaprSecurityException;
import com.mapr.security.Security;
import com.mapr.security.MutableInt;
import com.mapr.security.JNISecurity.MutableErr;

import com.mapr.cliframework.base.CommandOutput.*;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.*;
import com.mapr.fs.Rpc;
import com.mapr.fs.cldb.util.Util;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Security.ServerKeyType;
import com.mapr.fs.proto.Security.TicketAndKey;
import mapr.fs.Nfsmon.*;

import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * to suport nfs mgmt
 */
public class NFSMgmtCommands extends CLIBaseClass
{
  private static final String NFSHostIp = "nfshost";
  private static final String NFSPort = "nfsport";
  private static final String IS_USER_MODE = "isusermode";

  public static Map<String, BaseInputParameter> baseParams =
    new ImmutableMap.Builder<String, BaseInputParameter>()
    .put(NFSHostIp,
        new TextInputParameter(
          NFSHostIp, "ip/hostname", CLIBaseClass.NOT_REQUIRED, "127.0.0.1"))
      .put(NFSPort,
        new IntegerInputParameter(
            NFSPort, "port", CLIBaseClass.NOT_REQUIRED, 9998))
      .put(IS_USER_MODE,
        new TextInputParameter(
            IS_USER_MODE, "isusermode", CLIBaseClass.NOT_REQUIRED, null))
     .build();

  private static final CLICommand refreshExportsCmd =
    new CLICommand (
        "refreshexports",
        "usage: nfsmgmt refreshexports -nfshost host -nfsport port",
        NFSMgmtCommands.class,
        CLICommand.ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>().
          putAll(baseParams)
        .build(),
        null)
    .setShortUsage("nfsmgmt refreshexports -nfshost host -nfsport port");

  private static final CLICommand refreshGidCahceCmd =
    new CLICommand (
        "refreshgidcache",
        "usage: nfsmgmt refreshgidcache -nfshost host -nfsport port",
        NFSMgmtCommands.class,
        CLICommand.ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>().
          putAll(baseParams)
        .build(),
        null)
    .setShortUsage("nfsmgmt refreshgidcache -nfshost host -nfsport port");

  // main command
  public static final CLICommand nfsMgmtCmds =
    new CLICommand (
         "nfsmgmt", "nfsmgmt [refreshexports|refreshgidcache]",
         CLIUsageOnlyCommand.class,
         CLICommand.ExecutionTypeEnum.NATIVE,
         new CLICommand[] {refreshExportsCmd, refreshGidCahceCmd}
       )
      .setShortUsage("nfsmgmt [refreshexports|refreshgidcache]");

  String clusterName = null;

  public NFSMgmtCommands (ProcessedInput input, CLICommand cliCommand)
  {
    super(input, cliCommand);
  }

  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException
  {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);

    clusterName = CLDBRpcCommonUtils.getInstance().getCurrentClusterName();

    try {
      int port = Rpc.initialize(0, 0, null); // Client
      if (port < 0)
        throw new CLIProcessingException("Error in RPC init");
    } catch (Exception e) {
      System.out.println("Exception in Rpc.initialize " + e);
      throw new CLIProcessingException ("Exception in Rpc.initialize ", e);
    }

    int port = getParamIntValue(NFSPort, 0);
    String hostName = getParamTextValue(NFSHostIp, 0);
    if ( !( hostName.equalsIgnoreCase("localhost") || hostName.equalsIgnoreCase("127.0.0.1")) ) {
 		List<String> ips = NodesCommonUtils.convertHostToIp(Collections.singletonList(hostName));
		if ( ips.isEmpty() ) {
		  out.addError(new OutputError(Errno.EINVAL, "Can not get valid IP address out of provided name: " + hostName));
		  return output;
	    }
		hostName = ips.get(0);
    }
    int hostip = Util.ipToInt(hostName);

    // If NFS is running in user mode, create a ticket using the user key.
    if (isParamPresent(IS_USER_MODE) &&
        getParamTextValue(IS_USER_MODE, 0).equalsIgnoreCase("true")) {
      MutableInt err = CreateNFSUserTicket(clusterName);

      if (err.GetValue() != 0) {
        out.addError(new OutputError(Errno.INTERROR,
                                     "Cannot create the ticket for nfs in user mode"));
        return output;
      }
    }


//    int hostip = Util.ipToInt(getParamTextValue(NFSHostIp, 0));
    long binding = Rpc.createBindingFor(hostip, port, clusterName, ServerKeyType.ServerKey.getNumber());

    if (cliCommand.getCommandName().equalsIgnoreCase("refreshexports")) {
      refreshExports(out, binding);
    } else if (cliCommand.getCommandName().equalsIgnoreCase("refreshgidcache")) {
      refreshGidCache(out, binding);
    }

    return output;
  }

  private MutableInt CreateNFSUserTicket(String clusterName) {
    MutableInt err = new MutableInt();
    TicketAndKey ticketAndKey =
      Security.GetTicketAndKeyForCluster(ServerKeyType.ServerKey,
                                         clusterName,
                                         err);
    if (err.GetValue() != 0)
      return err;
  
    // Update the security layer to use the user key as server key.
    Security.SetKey(ServerKeyType.ServerKey,
                    ticketAndKey.getUserKey());

    // some random gid. this will not be used anywhere.
    String userName = ticketAndKey.getUserCreds().getUserName();
    int[] gids = new int[1];
    gids[0] = userName.length();
    
    ticketAndKey =
      Security.GenerateTicketAndKey(ServerKeyType.ServerKey,
                                    userName,
                                    userName.length(),
                                    gids,
                                    Security.MAX_EXPIRY_TIME,
                                    0 /*non-renewal*/,
                                    true,
                                    false /*canUserImpersonate*/,
                                    err);
    if (err.GetValue() != 0)
      return err;

    // Set the new ticket as the server ticket.
    Security.SetTicketAndKey(ServerKeyType.ServerKey,
                             clusterName,
                             ticketAndKey);
    return err;
  }
	  

  private void refreshExports (OutputHierarchy out, long binding)
          throws CLIProcessingException
  {
    NFSMgmtResponse resp = null;
    NFSMgmtRequest.Builder b = NFSMgmtRequest.newBuilder();
    b.setRefreshExports(true);

    byte[] data;
    try {
      data = Rpc.sendRequest (
              binding,
              Common.MapRProgramId.NFSServerMgmtProgramId.getNumber(),
              NFSMgmtProg.NFSMgmtProc.getNumber(),
              b.build());

      resp = NFSMgmtResponse.parseFrom(data);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }

    if (resp == null) {
      out.addError(new OutputError(Errno.EOPFAILED, "No response from NFS server"));
      return;
    }

    int status = resp.getStatus();
    if (status != 0) {
      String msg = resp.hasErrMsg() ? resp.getErrMsg(): Errno.toString(status);
      out.addError(new OutputError(status, msg));
      return;
    }

    if (resp.hasErrMsg()) {
      // just a warning msg
      out.addNode(new OutputNode(resp.getErrMsg()));
    }
  }

  private void refreshGidCache (OutputHierarchy out, long binding)
          throws CLIProcessingException
  {
    NFSMgmtResponse resp = null;
    NFSMgmtRequest.Builder b = NFSMgmtRequest.newBuilder();
    b.setRefreshGidCache(true);

    byte[] data;
    try {
      data = Rpc.sendRequest (
              binding,
              Common.MapRProgramId.NFSServerMgmtProgramId.getNumber(),
              NFSMgmtProg.NFSMgmtProc.getNumber(),
              b.build());

      resp = NFSMgmtResponse.parseFrom(data);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }

    if (resp == null) {
      out.addError(new OutputError(Errno.EOPFAILED, "No response from NFS server"));
      return;
    }

    int status = resp.getStatus();
    if (status != 0) {
      String msg = resp.hasErrMsg() ? resp.getErrMsg(): Errno.toString(status);
      out.addError(new OutputError(status, msg));
      return;
    }

    if (resp.hasErrMsg()) {
      // just a warning msg
      out.addNode(new OutputNode(resp.getErrMsg()));
    }
  }
}
