/**
 * MapR CLI commands for MCS
 * 
 * Save: Saves the content to MapR FS. If key parameter is present then the key is looked up in the file and its contents are updated else the file is overwritten
 * 
 * Read: Reads the content from MapR FS. If key parameter is present then the only the contents pertaining to the key are returned else the contents of the whole file is returned.
 * 
 * 
 */
package com.mapr.cli;

import java.io.EOFException;
import java.io.IOException;

import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.Path;
import org.apache.log4j.Logger;
import org.json.JSONException;
import org.json.JSONObject;

import com.google.common.collect.ImmutableMap;
import com.mapr.baseutils.Errno;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.JSONObjectInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.fs.MapRFileSystem;
import com.mapr.util.MapRFSUtil;

/**
 * @author Naveen Tirupattur
 * 
 */
public class McsCommands extends CLIBaseClass implements CLIInterface {

	private static final Logger LOG = Logger.getLogger(McsCommands.class);
	public static final String FILE_NAME = "file";
	public static final String VALUE = "value";
	public static final String KEY = "key";

	public McsCommands(ProcessedInput input, CLICommand cliCommand) {
		super(input, cliCommand);
	}

	public static final CLICommand saveToFileCommand = new CLICommand("save",
			"save -file filename -value filecontent -key keyvalue",
			McsCommands.class, ExecutionTypeEnum.NATIVE,
			new ImmutableMap.Builder<String, BaseInputParameter>()
					.put(FILE_NAME,
							new TextInputParameter(FILE_NAME, "file",
									CLIBaseClass.REQUIRED, null))
					.put(VALUE,
							new JSONObjectInputParameter(VALUE, "content",
									CLIBaseClass.REQUIRED))
					.put(KEY,
							new TextInputParameter(KEY, "key",
									CLIBaseClass.NOT_REQUIRED, null)).build(),
			null).setShortUsage("Save to file").setUsageInVisible(true);

	public static final CLICommand readFromFileCommand = new CLICommand("read",
			"read -file filename -key keyvalue", McsCommands.class,
			ExecutionTypeEnum.NATIVE,
			new ImmutableMap.Builder<String, BaseInputParameter>()
					.put(FILE_NAME,
							new TextInputParameter(FILE_NAME, "file",
									CLIBaseClass.REQUIRED, null))
					.put(KEY,
							new TextInputParameter(KEY, "key",
									CLIBaseClass.NOT_REQUIRED, null)).build(),
			null).setShortUsage("read from file").setUsageInVisible(true);

	@Override
	public CommandOutput executeRealCommand() throws CLIProcessingException {
		OutputHierarchy ch = new OutputHierarchy();
		CommandOutput co = new CommandOutput(ch);

		// Get the file name
		String fileName = input.getParameterByName(FILE_NAME).getParamValues()
				.get(0);

		if (fileName == null || fileName.length() == 0) {

			LOG.error("Invalid parameters. File name is either null or empty.");
			ch.addError(new CommandOutput.OutputHierarchy.OutputError(
					Errno.EINVAL,
					"Invalid parameters.File name is either null or empty"));
		}

		try {

			if (cliCommand.getCommandName().equalsIgnoreCase(
					saveToFileCommand.getCommandName())) {
				save(ch, fileName);

			} else if (cliCommand.getCommandName().equalsIgnoreCase(
					readFromFileCommand.getCommandName())) {
				read(ch, fileName);

			}
		} catch (IllegalArgumentException e) {
			/**
			 * <MAPR_ERROR> Message:Provided Input is not compliant with enums:
			 * <JSON string> Function:SaveToFileCommand.executeRealCommand()
			 * Meaning:The input to the command was specified incorrectly.
			 * Resolution:Check the command syntax and try again. </MAPR_ERROR>
			 */
			LOG.error("Provided Input is not compliant with enums: " + fileName);
			ch.addError(new OutputError(Errno.EINVALMISC,
					"Provided Input is not compliant with enums: " + fileName));
		} catch (IOException e) {
			LOG.error("I/O Exception while executing the command", e);
			ch.addError(new OutputError(Errno.ERPCFAILED,
					"Request to execute the command failed. No data returned"));
		} catch (Exception e) {
			/**
			 * <MAPR_ERROR> Message:RPC Request to create Save user session
			 * failed. No data returned
			 * Function:McsCommands.executeRealCommand() Meaning:An error
			 * occurred. Resolution:Contact technical support. </MAPR_ERROR>
			 */
			LOG.error("Request to execute the command failed. No data returned");
			ch.addError(new OutputError(Errno.ERPCFAILED,
					"Request to execute the command failed. No data returned"));
		}
		return co;
	}

	private void save(OutputHierarchy ch, String fileName)
			throws CLIProcessingException, IOException, JSONException {

		if (LOG.isDebugEnabled()) {
			LOG.debug("Saving to file for user: " + getUserLoginId());
		}

		// Get the file content
		String jsonString = input.getParameterByName(VALUE).getParamValues()
				.get(0);

		// Check if content is empty
		if (jsonString == null || jsonString.length() == 0) {
			LOG.error("Required parameters are either null or empty.");
			ch.addError(new OutputError(Errno.EINVALMISC,
					"Required parameters are either null or empty."));
		}

		String key = null;
		if (input.getParameterByName(KEY) != null) {
			key = input.getParameterByName(KEY).getParamValues().get(0);
		}

		// Open the file and write the content
		FSDataOutputStream fsDos = null;
		String filePath = "/var/mapr/mcsmetadata/" + fileName;
		MapRFileSystem fs = MapRFSUtil.getMapRFileSystem();
		Path p = new Path(filePath);
		JSONObject jsonObj = null;

		// If key is not present then just overwrite the file
		if (key == null || key.trim().length() == 0) {
			fsDos = fs.create(p, true);
			fsDos.writeBytes(jsonString);
		} else {

			jsonObj = readFromFile(filePath);
			// If jsonobj is null, create a new object
			if (jsonObj == null) {

				jsonObj = new JSONObject();
			}

			// update the json object
			jsonObj.put(key, new JSONObject(jsonString));

			// update the file with new json string
			fsDos = fs.create(p, true);
			fsDos.writeBytes(jsonObj.toString());
		}

		fsDos.close();
		fs.close();

	}

	private void read(OutputHierarchy ch, String fileName)
			throws CLIProcessingException, IOException, JSONException {
		if (LOG.isDebugEnabled()) {
			LOG.debug("Reading the file: " + fileName);
		}

		// Get the file path
		String filePath = "/var/mapr/mcsmetadata/" + fileName;
		JSONObject jsonObj = readFromFile(filePath);
		if (jsonObj == null) {
			ch.addError(new OutputError(Errno.EOPFAILED,
					"Invalid file. Please provide the correct path"));
		}
		String key = null;
		if (input.getParameterByName(KEY) != null) {
			key = input.getParameterByName(KEY).getParamValues().get(0);
		}
		OutputNode node = new OutputNode();
		// If key is null return the contents of the whole file
		if (key == null || key.trim().length() == 0) {
			node.addChild(new OutputNode("", jsonObj));

		} else {
			Object obj = null;
			try {
				obj = jsonObj.get(key);
			} catch (JSONException je) {
				LOG.info("Exception while parsing the json object: "
						+ je.getMessage());

			}
			// Return the contents of the key
			if (obj != null)
				node.addChild(new OutputNode("", obj));

		}
		ch.addNode(node);
	}

	private JSONObject readFromFile(String filePath) throws IOException,
			JSONException {
		FSDataInputStream fsDis = null;

		// Get the file system object
		MapRFileSystem fs = MapRFSUtil.getMapRFileSystem();

		Path p = new Path(filePath);

		// If the path doesn't exist then return null;
		if (!fs.exists(p))
			return null;

		// Open the file
		try {
			fsDis = fs.open(p);
			// read file
			long fileLength = fsDis.getFileLength();
			byte[] buf = new byte[(int) fileLength];
			int bytesRead = fsDis.read(buf, 0, (int) fileLength);
			String content = new String(buf);
			return new JSONObject(content);
		} catch (EOFException eofEx) {
			LOG.error("EOF reached", eofEx);
			if (fsDis != null) {
				try {
					fsDis.close();
				} catch (IOException ie) {
					LOG.error("Exception while close the input stream", ie);
				}
			}
		}
		return null;

	}

	public static final CLICommand mcsCommands = new CLICommand("mcs", "mcs",
			CLIUsageOnlyCommand.class, ExecutionTypeEnum.NATIVE,
			new CLICommand[] { saveToFileCommand, readFromFileCommand })
			.setShortUsage("mcs [save|read]").setUsageInVisible(true);

}
