/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.mapr.security.JNISecurity;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.mapred.JobClient;
import org.apache.hadoop.yarn.client.api.YarnClient;
import org.apache.hadoop.yarn.conf.HAUtil;
import org.apache.hadoop.yarn.conf.YarnConfiguration;
import org.apache.hadoop.yarn.proto.YarnServerResourceManagerServiceProtos;
import org.apache.log4j.Logger;

import com.google.protobuf.MessageLite;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cli.common.NodesCommonUtils;
import com.mapr.cli.common.ServicesEnum;
import com.mapr.fs.MapRFileSystem;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.CLDBConfigParams.CLDBConfigParam;
import com.mapr.fs.cldb.proto.CLDBProto.CLDBConfigRequest;
import com.mapr.fs.cldb.proto.CLDBProto.CLDBConfigResponse;
import com.mapr.fs.cldb.proto.CLDBProto.FileServerInfo;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Common.FidMsg;
import com.mapr.fs.proto.Security.CredentialsMsg;
import com.mapr.util.LDAPUtil;
import com.mapr.security.MaprSecurityException;

import com.mapr.fs.proto.Common.ServiceData;
import com.mapr.fs.proto.Common.ExtendedInfo;
import com.google.protobuf.InvalidProtocolBufferException;

public class MapRCliUtil {

  private static ExecutorService es = Executors.newFixedThreadPool(10);
  public static final int JTTimeout = 6000; // 6000 ms
  public static final String ParamEntityQuerySource = "mapr.entityquerysource";
  public static final String ParamDomainName = "mapr.domainname";
  public static final int    CLDB_DEFAULT_PORT = 7222;
  public static final String CLUSTER_NAME_PARAM = "cluster";
  private static String systemhostname = getHostname();
  private static final Logger LOG = Logger.getLogger(MapRCliUtil.class);

  private static YarnClient yarnClient = null;
  private static final Object yarnClientLock = new Object();

  public static String fetchEmail(Properties prop, String user, boolean type) {
    String querysource = prop.getProperty(ParamEntityQuerySource);
    if (querysource != null && querysource.equalsIgnoreCase("ldap")) {
     String ldapEmail;
     try {
       ldapEmail = LDAPUtil.getEmailAddress(prop, user, type);
     } catch (IOException e) {
       return null;
     }
      if (ldapEmail != null && !ldapEmail.isEmpty()) {
        return ldapEmail;
      }
    } else {
      String domainname = prop.getProperty(ParamDomainName);
      if (domainname != null && !domainname.isEmpty()) {
        return (user + "@" + domainname);
      }
    }
    return null;
  }

  public static int fetchCLDBParams(String cluster,
                                    Properties CLDBProperties,
                                    CredentialsMsg creds)
     throws CLIProcessingException {

    try {
        CLDBConfigRequest req =
            CLDBConfigRequest.newBuilder().setLoad(true).setCreds(creds).build();
        byte[] replyData;

        if (cluster != null && !cluster.isEmpty()) {
          replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
                          cluster,
                          Common.MapRProgramId.CldbProgramId.getNumber(),
                          CLDBProto.CLDBProg.CLDBConfigProc.getNumber(), req,
                          CLDBConfigResponse.class);
        } else {
         replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
                         Common.MapRProgramId.CldbProgramId.getNumber(),
                         CLDBProto.CLDBProg.CLDBConfigProc.getNumber(), req,
                         CLDBConfigResponse.class);
        }

        if (replyData == null) {
          return -1;
        }

        CLDBConfigResponse resp = CLDBConfigResponse.parseFrom(replyData);
        int status = resp.getStatus();
        if (status != 0) return status;

        for (CLDBConfigParam param : resp.getParams().getParamsList()) {
           CLDBProperties.setProperty(param.getKeys(), param.getValues());
        }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
         /**
         * <MAPR_ERROR>
         * Message:Exception: <message>
         * Function:MapRCliUtil.fetchCLDBParams()
         * Meaning:A serious error occurred.
         * Resolution:Contact technical support.
         * </MAPR_ERROR>
         */
         LOG.error("Exception: " + e.getLocalizedMessage());
         return -1;
    }
    return 0;
  }

  public static long quotaStringToMB (String quota) {
    long val = -1;
    int mf = 1;

    quota = quota.toUpperCase();

    if (quota.endsWith("M")) {
      quota = quota.substring(0, quota.length() - 1);
    } else if (quota.endsWith("MB")) {
      quota = quota.substring(0, quota.length() - 2);
    } else if (quota.endsWith("G")) {
      quota = quota.substring(0, quota.length() - 1);
      mf = 1024;
    } else if (quota.endsWith("GB")) {
      quota = quota.substring(0, quota.length() - 2);
      mf = 1024;
    } else if (quota.endsWith("T")) {
      quota = quota.substring(0, quota.length() - 1);
      mf = 1024 * 1024;
    } else if (quota.endsWith("TB")) {
      quota = quota.substring(0, quota.length() - 2);
      mf = 1024 * 1024;
    } else if (quota.endsWith("P")) {
      quota = quota.substring(0, quota.length() - 1);
      mf = 1024 * 1024 * 1024;
    } else if (quota.endsWith("PB")) {
      quota = quota.substring(0, quota.length() - 2);
      mf = 1024 * 1024 * 1024;
    }

    if (quota.matches("[0-9]+"))
      val = mf * Long.valueOf(quota);
    else if (quota.matches("[0-9]*\\.[0-9]*") && !quota.equals("."))
      val = (long)((double) mf * Double.valueOf(quota));
    return val;
  }

  public static int ipToInt(String addr) {
    if (addr.equals("localhost"))
      addr = "127.0.0.1";
    String[] addrArray = addr.split("\\.");
    long num = 0;
    for (int i=0;i<addrArray.length;i++) {
        int power = 3-i;
        num += ((Integer.parseInt(addrArray[i])%256 * Math.pow(256,power)));
    }
    return (int)num;
  }

  public static boolean validateEmail(String email) {
    Pattern p = Pattern.compile(".+@.+\\.[a-z]+");
    for (String testEmail : email.split(" ")) {
      Matcher m = p.matcher(email);
      if (!m.matches()) {
        return false;
      }
    }
    return true;
  }

  public static String getHostname(FileServerInfo fs) {
    try {
      Common.IPAddress ip = fs.getAddressList().get(0);
      if (ip != null) {
        return ip.getHostname();
      }
    } catch (Exception e) {
      return null;
    }
    return null;
  }

  public static String getHostname() {
    String hostnameFile = getMapRInstallDir() + "/hostname";

    try {
      FileReader fr = new FileReader(hostnameFile);
      if (fr == null)
        return null;

      BufferedReader b = new BufferedReader(fr);
      String strLine = b.readLine();
      if (strLine != null && !strLine.isEmpty()) {
        strLine = strLine.trim();
        if (!strLine.isEmpty())
          return strLine;
      }
    } catch (Exception e) {
      return null;
    }
    return null;
  }

  public static boolean isLocalAddr(String host) {

    if (host.equalsIgnoreCase("127.0.0.1"))
      return true;

    if (systemhostname != null && host.equalsIgnoreCase(systemhostname))
      return true;

    try {
      String localhostname = InetAddress.getLocalHost().getHostName();
      if (host.equalsIgnoreCase(localhostname))
        return true;

      InetAddress [] localaddrs = InetAddress.getAllByName(localhostname);
      for (InetAddress addr: InetAddress.getAllByName(host)) {
        for (InetAddress localaddr : localaddrs) {
          if (addr.equals(localaddr))
            return true;
        }
      }
    } catch (Exception e) {
      return false;
    }
    return false;
  }

  public static String getMapRInstallDir() {
    String maprInstallDir = System.getProperty("mapr.home.dir");
    if (maprInstallDir == null) {
      maprInstallDir = System.getenv("MAPR_HOME");
      if (maprInstallDir == null) {
        maprInstallDir = "/opt/mapr";
      }
    }
    return maprInstallDir;
  }

  public static MapRFileSystem getMapRFileSystem() throws CLIProcessingException {
    Configuration c = new Configuration();
    c.set("fs.defaultFS", "maprfs:///");
    c.set("fs.mapr.disable.namecache", "true");
    MapRFileSystem fs;

    try {
      fs = new MapRFileSystem();
      URI uri = new URI(c.get("fs.defaultFS"));
      fs.initialize(uri, c);
    } catch (Exception e) {
      /**
      * <MAPR_ERROR>
      * Message:Failed to initialize MapRFileSystem <error>
      * Function:MapRCliUtil.getMapRFileSystem()
      * Meaning:A serious error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      throw new CLIProcessingException("Failed to initialize MapRFileSystem " + e);
    }
    return fs;
  }

  public static boolean parentDirExists(MapRFileSystem fs, String clusterName, String path)
      throws CLIProcessingException {
    if (path.equalsIgnoreCase("/"))
      return true; // Let CLDB fail mount of this.
    if (!path.startsWith("/"))
      return false; // Absolute path has to be specified
    String parent = new File(path).getParent();
    if (parent == null)
      return false;

    boolean exists = false;
    try {
      if (clusterName != null)
        parent = "/mapr/" + clusterName + parent;
      exists = fs.exists(new Path(parent));
    } catch (Exception e) {
    }
    return exists;
  }

  public static int getParentCid(MapRFileSystem fs, String clusterName, String path)
      throws CLIProcessingException {
    if (!path.startsWith("/")) {
      LOG.error ("getParentCid path has to be absolute path. Specified: " + path);
      return 0; // Absolute path has to be specified
    }
    String parent = new File(path).getParent();
    if (parent == null) {
      LOG.error ("getParentCid parent could not be found. Path: " + path);
      return 0;
    }

    if (clusterName != null)
      parent = "/mapr/" + clusterName + parent;
    return fs.getCidFromPath(new Path(parent));
  }

  public static boolean createParentDir(MapRFileSystem fs, String clusterName, String path)
      throws CLIProcessingException {
		if (!path.startsWith("/"))
		  return false; // Absolute path has to be specified
		String parent = new File(path).getParent();
		if (parent == null)
		  return false;

		boolean ret = false;
		try {
		  if (clusterName != null)
		    parent = "/mapr/" + clusterName + parent;
		  ret = fs.mkdirs(new Path(parent));
		} catch (Exception e) {
		}
		return ret;
  }

  public static byte[] sendRpc(MessageLite request, int procId, String cluster, Class<? extends MessageLite> responseClass) throws Exception {
    byte[] reply = null;
    CLDBRpcCommonUtils instance = CLDBRpcCommonUtils.getInstance();
    if (cluster != null) {
      reply = instance.sendRequest(cluster, Common.MapRProgramId.CldbProgramId.getNumber(), procId, request, responseClass);
    } else {
      reply = instance.sendRequest(Common.MapRProgramId.CldbProgramId.getNumber(), procId, request, responseClass);
    }
    return reply;
  }

  public static JobClient getJobClient(final String zkConnectString) {
    if (zkConnectString == null || zkConnectString.trim().isEmpty())
      return null;

    Callable<JobClient> jtStatisCallable = new Callable<JobClient>() {
      @Override
      public JobClient call() throws Exception {

        InetSocketAddress addr = null;
        ServiceData hostInfo =
          NodesCommonUtils.getServiceMasterData(zkConnectString, ServicesEnum.jobtracker.name());

        if ( hostInfo != null ) {
        	if (hostInfo.hasIsRunning() && hostInfo.getIsRunning()) {
        		if (hostInfo.hasHost() && hostInfo.hasPort()) {
        			// construct RM Address:
        			try {
        				addr = new InetSocketAddress(hostInfo.getHost(),
        						hostInfo.getPort());
        			} catch (IllegalArgumentException ex) {
        				LOG.error("JT Info is not valid: " + hostInfo.getHost() + hostInfo.getPort());
        			}
        		}
        	}
        }

        
        
        if ( LOG.isDebugEnabled() ) {
          LOG.debug("getJobClient jt found");
        }
        if (addr == null) {
          return null;
        }

        Configuration conf = new Configuration();
        conf.setInt("ipc.client.connect.max.retries", 2); // 2 retries

        JobClient jc = new JobClient(addr, conf);
        jc.setConf(conf);
        return jc;
      }
    };

    List<Callable<JobClient>> callableList = new ArrayList<Callable<JobClient>>();
    callableList.add(jtStatisCallable);

    try {
      JobClient jc = es.invokeAny(callableList, JTTimeout, TimeUnit.MILLISECONDS);
      return jc;
    } catch (InterruptedException e) {
      LOG.error("InterruptedException during JT Status thread execution");
      return null;
    } catch (ExecutionException e) {
      LOG.error("ExecutionException during JT Status thread execution", e);
      if ( e.getCause() != null && e.getCause() instanceof OutOfMemoryError) {
        LOG.fatal("OutOfMemory Error. Application needs to be restarted", e);
        System.exit(1);
      }
      return null;
    } catch (TimeoutException e) {
      LOG.error("TimeoutException during JT Status thread execution", e);
      return null;
    }
  }

  public static <T> T asyncInvoke(Callable<T> callable, String execContext) {
    return (T) asyncInvoke(callable, JTTimeout, execContext);
  }

  public static <T> T asyncInvoke(Callable<T> callable, int timeout, String execContext) {
    List<Callable<T>> callableList = new ArrayList<Callable<T>>();
    callableList.add((callable));

    try {
      return es.invokeAny(callableList, timeout, TimeUnit.MILLISECONDS);
    } catch (InterruptedException e) {
      LOG.error("InterruptedException caught invoking Callable for " + execContext);
      return null;
    } catch (ExecutionException e) {
      if ( e.getCause() != null && e.getCause() instanceof OutOfMemoryError) {
        LOG.fatal("OutOfMemory Error. Application needs to be restarted", e);
        System.exit(1);
      }
      LOG.error("Caught Execution Exception during " + execContext, e);
      return null;
    } catch (TimeoutException e) {
      LOG.error("TimeoutException caught during " + execContext, e);
      return null;
    }
  }

  public static YarnClient getYarnClient(final String zkConnectString) {
    if (zkConnectString == null || zkConnectString.trim().isEmpty())
      return null;

    Callable<YarnClient> rmStatIsCallable = new Callable<YarnClient>() {
      @Override
      public YarnClient call() throws Exception {
        
        // Find out if there is at least one RM process up. Warden will have a master node for it.
        boolean rmHasMaster = false;
        ServiceData hostInfo =
        		NodesCommonUtils.getServiceMasterData(zkConnectString, "resourcemanager");
        if (hostInfo != null &&
            hostInfo.hasIsRunning() && hostInfo.getIsRunning()) {
            rmHasMaster = true;
        }
        if (LOG.isDebugEnabled() && rmHasMaster) {
          LOG.debug("getYarnClient RM found");
        }

        synchronized (yarnClientLock) {
          if (rmHasMaster == false) { // Could not find master node for RM.
            if(LOG.isDebugEnabled())
              LOG.debug("No RM Master found. Returning yarnClient as null");
            return yarnClient = null;
          }

          // Return the cached yarn client if there is one created already and is still valid i.e.
          // Warden thinks RM has a master node.
          // Internally yarnClient will fail over to a new RM if it has not already.
          if (yarnClient != null) {
            if(LOG.isDebugEnabled())
              LOG.debug("Returning CACHED yarnClient");
            return yarnClient;
          }
        }

        // Create a new Yarn Config object.
        YarnConfiguration conf = new YarnConfiguration();

        // Now connect to the RM and initialize Client.
        YarnClient yc = YarnClient.createYarnClient();
        if(yc == null)
          return null;

        yc.init(conf);
        yc.start();

        synchronized (yarnClientLock) {
          if(LOG.isDebugEnabled())
            LOG.debug("Returning NEW yarnClient");
          return yarnClient = yc;
        }
      }
    };

    List<Callable<YarnClient>> callableList = new ArrayList<Callable<YarnClient>>();
    callableList.add(rmStatIsCallable);

    try {
      YarnClient yc = es.invokeAny(callableList, JTTimeout, TimeUnit.MILLISECONDS);
      return yc;
    } catch (InterruptedException e) {
      LOG.error("InterruptedException during RM Status thread execution");
      return null;
    } catch (ExecutionException e) {
      LOG.error("ExecutionException during RM Status thread execution", e);
      if ( e.getCause() != null && e.getCause() instanceof OutOfMemoryError) {
        LOG.fatal("OutOfMemory Error. Application needs to be restarted", e);
        System.exit(1);
      }
      return null;
    } catch (TimeoutException e) {
      LOG.error("TimeoutException during RM Status thread execution", e);
      return null;
    }

  }

  public static String getRealRMWebAddress(String zkConnectString, ServiceData masterServiceData) {
    if (zkConnectString == null || zkConnectString.trim().isEmpty())
      return null;

    YarnConfiguration conf = new YarnConfiguration();
    if (HAUtil.isCustomRMHAEnabled(conf)) {
      // Custom RM HA enabled. Interpret according to data from MapR's Warden/ZK based RM failover.
      boolean isSecureCluster = JNISecurity.IsSecurityEnabled(CLDBRpcCommonUtils.getInstance().getCurrentClusterName());
      String portKeyName = isSecureCluster ? "WEBAPP_HTTPS_PORT" : "WEBAPP_PORT";
      String port = null;
      for(ExtendedInfo extInfo : masterServiceData.getExtinfoList()) {
        if(portKeyName.equals(extInfo.getKey()))
          port = extInfo.getValue();
      }
      if(port != null) {
        return masterServiceData.getHost() + ':' + port;
      }
      else {
        return masterServiceData.getHost();
      }
    }
    else if( HAUtil.isHAEnabled(conf)) {
      // If Open Source RM HA is enabled, lookup the current RM Master ID and return its corresponding web address.
      String rmMasterId = NodesCommonUtils.getCurrentRMMasterID(zkConnectString);
      if(rmMasterId != null) {
        boolean isSecureCluster = JNISecurity.IsSecurityEnabled(CLDBRpcCommonUtils.getInstance().getCurrentClusterName());
        String baseYarnConf =  isSecureCluster ?
                YarnConfiguration.RM_WEBAPP_HTTPS_ADDRESS : YarnConfiguration.RM_WEBAPP_ADDRESS;
        String defaultAddress = isSecureCluster ?
                YarnConfiguration.DEFAULT_RM_WEBAPP_HTTPS_ADDRESS : YarnConfiguration.DEFAULT_RM_WEBAPP_ADDRESS;
        return conf.get(baseYarnConf + "." + rmMasterId, defaultAddress);
      } else {
        return null;
      }
    }
    else {
      // HA is not enabled, get this straight from Yarn config.
      return conf.get(YarnConfiguration.RM_WEBAPP_ADDRESS,
              YarnConfiguration.DEFAULT_RM_WEBAPP_ADDRESS);
    }
  }

  public static final String MAPR_PATH_PREFIX = "/mapr/";
  public static final int MAPR_PATH_PREFIX_LENGTH = MAPR_PATH_PREFIX.length();

  public static String extractClusterNameFromFullyQualifiedPath(String fullyQualifiedPath) {
    if (fullyQualifiedPath.startsWith(MAPR_PATH_PREFIX)) {
      String remainingPath = fullyQualifiedPath.substring(MAPR_PATH_PREFIX_LENGTH);
      int index = remainingPath.indexOf("/");
      if (index != -1 && index != remainingPath.length() - 1) {
        return remainingPath.substring(0, index);
      }
    }
    return null;
  }
  public static String getFidAsString(FidMsg fid) {
    StringBuilder builder = new StringBuilder();
    builder.append(fid.getCid());
    builder.append(".");
    builder.append(fid.getCinum());
    builder.append(".");
    builder.append(fid.getUniq());
    return builder.toString();
  }


}
