package com.mapr.cli;

import com.google.common.collect.ImmutableMap;
import com.mapr.baseutils.BaseUtilsHelper;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.security.Security;
import com.mapr.security.MutableInt;
import com.mapr.security.JNISecurity.MutableErr;
import com.mapr.cli.common.NodesCommonUtils;
import com.mapr.cli.common.ServicesEnum;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.IntegerInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.fs.Rpc;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.ChangeLogLevelRequest;
import com.mapr.fs.cldb.proto.CLDBProto.ChangeLogLevelResponse;
import com.mapr.fs.cldb.util.Util;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Gtrace;
import com.mapr.fs.proto.Security.ServerKeyType;
import com.mapr.fs.proto.Security.TicketAndKey;
import com.mapr.fs.proto.Security.Key;
import com.mapr.login.client.MapRLoginClient;
import com.mapr.login.client.MapRLoginHttpsClient;
import com.mapr.security.MaprHttpURL;

import org.apache.hadoop.http.HttpServer;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;

import com.mapr.security.MaprSecurityException;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URL;
import java.net.URLConnection;
import java.util.Collections;
import java.util.List;
import java.util.regex.Pattern;

public class LogLevelChangesCommand extends CLIBaseClass implements
		CLIInterface {
	
	private static final Logger LOG = Logger.getLogger(LogLevelChangesCommand.class);
	
	public static final String CLASS_NAME = "classname";
	public static final String LOG_LEVEL_NAME = "loglevel";
	public static final String NODE_IP = "node";
	public static final String PORT = "port";
	public static final String IS_USER_MODE = "isusermode";

	public static final CLICommand jtChangeLogLevelCommand = new CLICommand( 
			 ServicesEnum.jobtracker.name(), 
			    "change log level for a class in JT", 
			    LogLevelChangesCommand.class, ExecutionTypeEnum.NATIVE, 
			      new ImmutableMap.Builder<String, BaseInputParameter>() 
			        .put(CLASS_NAME, 
			          new TextInputParameter( 
			        		  CLASS_NAME, 
			            "name of the Java class for which log level should be set", CLIBaseClass.REQUIRED, null))
			        .put(LOG_LEVEL_NAME, 
			          new TextInputParameter( 
			        		  LOG_LEVEL_NAME, 
			            "Log Level: FATAL|ERROR|WARN|INFO|DEBUG|TRACE", CLIBaseClass.REQUIRED, null))
   			        .put(NODE_IP, 
			          new TextInputParameter( 
			        		  NODE_IP, 
			            "address of the node", CLIBaseClass.REQUIRED, null))
			          .put(PORT, 
                new IntegerInputParameter( 
                    PORT, 
                  "JT webserver port", CLIBaseClass.NOT_REQUIRED, null))
			         .build(), null) 
			    .setShortUsage("Setting log level for a particular class in a particluar service on a particular node");

	 public static final CLICommand ttChangeLogLevelCommand = new CLICommand( 
			 ServicesEnum.tasktracker.name(), 
			    "change log level for a class in TT", 
			    LogLevelChangesCommand.class, ExecutionTypeEnum.NATIVE, 
			      new ImmutableMap.Builder<String, BaseInputParameter>() 
			        .put(CLASS_NAME, 
			          new TextInputParameter( 
			        		  CLASS_NAME, 
			            "name of the Java class for which log level should be set", CLIBaseClass.REQUIRED, null))
			        .put(LOG_LEVEL_NAME, 
			          new TextInputParameter( 
			        		  LOG_LEVEL_NAME, 
			            "Log Level: FATAL|ERROR|WARN|INFO|DEBUG|TRACE", CLIBaseClass.REQUIRED, null))
   			        .put(NODE_IP, 
			          new TextInputParameter( 
			        		  NODE_IP, 
			            "address of the node", CLIBaseClass.REQUIRED, null))
			          .put(PORT, 
                new IntegerInputParameter( 
                    PORT, 
                  "TT webserver port", CLIBaseClass.NOT_REQUIRED, null))
                .build(), null) 
			    .setShortUsage("Setting log level for a particular class in a particluar service on a particular node");

	 public static final CLICommand hbMasterChangeLogLevelCommand = new CLICommand( 
			 ServicesEnum.hbmaster.name(), 
			    "change log level for a class in HBase master", 
			    LogLevelChangesCommand.class, ExecutionTypeEnum.NATIVE, 
			      new ImmutableMap.Builder<String, BaseInputParameter>() 
			        .put(CLASS_NAME, 
			          new TextInputParameter( 
			        		  CLASS_NAME, 
			            "name of the Java class for which log level should be set", CLIBaseClass.REQUIRED, null))
			        .put(LOG_LEVEL_NAME, 
			          new TextInputParameter( 
			        		  LOG_LEVEL_NAME, 
			            "Log Level: FATAL|ERROR|WARN|INFO|DEBUG|TRACE", CLIBaseClass.REQUIRED, null))
   			        .put(NODE_IP, 
			          new TextInputParameter( 
			        		  NODE_IP, 
			            "address of the node", CLIBaseClass.REQUIRED, null))
			          .put(PORT, 
                new IntegerInputParameter( 
                    PORT, 
                  "HBase Master webserver port", CLIBaseClass.NOT_REQUIRED, null))
			         .build(), null) 
			    .setShortUsage("Setting log level for a particular class in a particluar service on a particular node");

	 public static final CLICommand hbRegionChangeLogLevelCommand = new CLICommand( 
			 ServicesEnum.hbregionserver.name(), 
			    "change log level for a class in HbaseRegionServer", 
			    LogLevelChangesCommand.class, ExecutionTypeEnum.NATIVE, 
			      new ImmutableMap.Builder<String, BaseInputParameter>() 
			        .put(CLASS_NAME, 
			          new TextInputParameter( 
			        		  CLASS_NAME, 
			            "name of the Java class for which log level should be set", CLIBaseClass.REQUIRED, null))
			        .put(LOG_LEVEL_NAME, 
			          new TextInputParameter( 
			        		  LOG_LEVEL_NAME, 
			            "Log Level: FATAL|ERROR|WARN|INFO|DEBUG|TRACE", CLIBaseClass.REQUIRED, null))
   			        .put(NODE_IP, 
			          new TextInputParameter( 
			        		  NODE_IP, 
			            "address of the node", CLIBaseClass.REQUIRED, null))
			          .put(PORT, 
                new IntegerInputParameter( 
                   PORT, 
                  "HBase RegionServer webserver port", CLIBaseClass.NOT_REQUIRED, null))
			         .build(), null) 
			    .setShortUsage("Setting log level for a particular class in a particluar service on a particular node");

	 public static final CLICommand nfsChangeLogLevelCommand = new CLICommand( 
			 ServicesEnum.nfs.name(), 
			    "change log level for a class in NFS Server", 
			    LogLevelChangesCommand.class, ExecutionTypeEnum.NATIVE, 
			      new ImmutableMap.Builder<String, BaseInputParameter>() 
			        .put(CLASS_NAME, 
			          new TextInputParameter( 
			        		  CLASS_NAME, 
			            "name of the module for which log level should be set. It should be one of listed under \"maprcli trace info\"", CLIBaseClass.REQUIRED, null))
			        .put(LOG_LEVEL_NAME, 
			          new TextInputParameter( 
			        		  LOG_LEVEL_NAME, 
			            "Log Level: FATAL|ERROR|WARN|INFO|DEBUG|TRACE", CLIBaseClass.REQUIRED, null))
   			        .put(NODE_IP, 
			          new TextInputParameter( 
			        		  NODE_IP, 
			            "address of the node", CLIBaseClass.REQUIRED, null))
			          .put(PORT, 
                new IntegerInputParameter( 
                    PORT, 
                  "nfs port", CLIBaseClass.NOT_REQUIRED, null))
                .put(IS_USER_MODE,
                new TextInputParameter( 
                    IS_USER_MODE, 
                  "is user mode", CLIBaseClass.NOT_REQUIRED, null))
			         .build(), null) 
			    .setShortUsage("Setting log level for a particular class in a particluar service on a particular node");

	 public static final CLICommand cldbChangeLogLevelCommand = new CLICommand( 
			 ServicesEnum.cldb.name(), 
			    "change log level for a class in CLDB", 
			    LogLevelChangesCommand.class, ExecutionTypeEnum.NATIVE, 
			      new ImmutableMap.Builder<String, BaseInputParameter>() 
			        .put(CLASS_NAME, 
			          new TextInputParameter( 
			        		  CLASS_NAME, 
			            "name of the Java class for which log level should be set", CLIBaseClass.REQUIRED, null))
			        .put(LOG_LEVEL_NAME, 
			          new TextInputParameter( 
			        		  LOG_LEVEL_NAME, 
			            "Log Level: FATAL|ERROR|WARN|INFO|DEBUG|TRACE", CLIBaseClass.REQUIRED, null))
   			        .put(NODE_IP, 
			          new TextInputParameter( 
			        		  NODE_IP, 
			            "address of the node", CLIBaseClass.REQUIRED, null))
			          .put(PORT, 
                new IntegerInputParameter( 
                    PORT, 
                  "CLDB port", CLIBaseClass.NOT_REQUIRED, null))
			         .build(), null) 
			    .setShortUsage("Setting log level for a particular class in a particluar service on a particular node");

	 public static final CLICommand mfsChangeLogLevelCommand = new CLICommand( 
			 ServicesEnum.fileserver.name(), 
			    "change log level for a class in MFS Server", 
			    LogLevelChangesCommand.class, ExecutionTypeEnum.NATIVE, 
			      new ImmutableMap.Builder<String, BaseInputParameter>() 
			        .put(CLASS_NAME, 
			          new TextInputParameter( 
			        		  CLASS_NAME, 
			            "name of the module for which log level should be set. It should be one of listed under \"maprcli trace info\"", CLIBaseClass.REQUIRED, null))
			        .put(LOG_LEVEL_NAME, 
			          new TextInputParameter( 
			        		  LOG_LEVEL_NAME, 
			            "Log Level: FATAL|ERROR|WARN|INFO|DEBUG|TRACE", CLIBaseClass.REQUIRED, null))
   			        .put(NODE_IP, 
			          new TextInputParameter( 
			        		  NODE_IP, 
			            "address of the node", CLIBaseClass.REQUIRED, null))
			          .put(PORT, 
                new IntegerInputParameter( 
                    PORT, 
                  "mfs port", CLIBaseClass.NOT_REQUIRED, null))
			         .build(), null) 
			    .setShortUsage("Setting log level for a particular class in a particluar service on a particular node");

        String clusterName = null;

	public LogLevelChangesCommand(ProcessedInput input, CLICommand cliCommand) {
		super(input, cliCommand);
	}

	@Override
	public boolean validateInput() throws IllegalArgumentException {
		if ( !super.validateInput()) {
			return false;
		}
		try {
			String logLevel = getParamTextValue(LOG_LEVEL_NAME,0);
			Level level = Level.toLevel(logLevel);
			if ( level == Level.DEBUG && !logLevel.equalsIgnoreCase("DEBUG")) {
				LOG.error("Invalid log level: " + logLevel);
				output.getOutput().addError(new OutputError(Errno.EINVAL, "Invalid log level").setField(LOG_LEVEL_NAME));
				return false;
			}
		} catch (CLIProcessingException e) {
			LOG.error(e);
			output.getOutput().addError(new OutputError(Errno.EINVAL, "Exception while processing log level param").setField(LOG_LEVEL_NAME));
			return false;
		}
		return true;
	}
	
	@Override
	public CommandOutput executeRealCommand() throws CLIProcessingException {
		OutputHierarchy ch = new OutputHierarchy();
		CommandOutput co = new CommandOutput(ch);

		

                clusterName = CLDBRpcCommonUtils.getInstance().getCurrentClusterName();

		try {
			//ensure authenticated
			MapRLoginClient client = new MapRLoginHttpsClient();
			client.authenticateIfNeeded(clusterName);
		} catch (Exception e) {
			LOG.error("Exception unable to authenticate ", e);
			ch.addError(new OutputError(Errno.ERPCFAILED, "authentication failed"));
			return co;
                }


		for ( ServicesEnum service : ServicesEnum.values()) {
			if ( cliCommand.getCommandName().equalsIgnoreCase(service.name())) {
				switch (service) {
				case jobtracker:
				case tasktracker:
				case hbmaster:
				case hbregionserver:
				  String port = service.getPort();
				  if ( isParamPresent(PORT) ) {
				    port = Integer.toString(getParamIntValue(PORT, 0));
				  }
					StringBuilder sb = new StringBuilder(BaseUtilsHelper.getUrlScheme());
					sb.append(getParamTextValue(NODE_IP, 0));
					sb.append(":");
					sb.append(port);
					sb.append("/logLevel?log=");
					sb.append(getParamTextValue(CLASS_NAME,0));
					sb.append("&level=");
					sb.append(getParamTextValue(LOG_LEVEL_NAME,0));
					process(sb.toString(), service, ch);
					break;
				case cldb:
					changeCLDBloglevel(ch);
					break;
				case nfs:
				case fileserver:
					setFSLogLevel(service, ch);
					break;
				default:
					ch.addError(new OutputError(Errno.ENOSYS, "Settting log level for service: " + service.name() + " is not implemented yet"));
					break;	
				}
			}
		}
		return co;
	}

	/**
	 * CLDB log level change
	 * @param out
	 * @throws CLIProcessingException
	 */
	  private void changeCLDBloglevel(OutputHierarchy out) throws CLIProcessingException {        	    
		     String logLevel = getParamTextValue(LOG_LEVEL_NAME, 0);
		     String className = getParamTextValue(CLASS_NAME, 0);
		     
		     int cldbPort = 7222;
         if ( isParamPresent(PORT) ) {
           cldbPort = getParamIntValue(PORT, 0);
         }
		     String cldbHostString = getParamTextValue(NODE_IP, 0);
		     if (!( cldbHostString.equalsIgnoreCase("localhost") || cldbHostString.equalsIgnoreCase("127.0.0.1")) ) {
			     List<String> ips = NodesCommonUtils.convertHostToIp(Collections.singletonList(cldbHostString));
			     if ( ips.isEmpty() ) {
			    	 out.addError(new OutputError(Errno.EINVAL, "Can not get valid IP address out of provided name: " + cldbHostString));
			    	 return;
			     }
			     cldbHostString = ips.get(0);
		     }
		     int cldbHost = Util.ipToInt(cldbHostString);
		   
		     try {
		       int port = Rpc.initialize(0, 0, null); // Client
		       if (port < 0)
		         throw new IOException("Failed to initalize RPC");
		     } catch (Exception e) {
		        /**
		         * <MAPR_ERROR>
		         *  Failed to Initialze RPC
		         * </MAPR_ERROR>
		         */
		         LOG.error("Exception in Rpc.initialize " + e);
		     }
		     long binding = Rpc.createBindingFor(cldbHost, cldbPort, clusterName, ServerKeyType.ServerKey.getNumber());

		       
		     ChangeLogLevelRequest req = ChangeLogLevelRequest.newBuilder()
		                                   .setCreds(getUserCredentials())
		                                   .setLogLevel(logLevel)
		                                   .setClassName(className)
		                                   .build();
		     byte[] replyData;
		     try {
		       replyData = Rpc.sendRequest(binding,
	                     Common.MapRProgramId.CldbProgramId.getNumber(),
	                     CLDBProto.CLDBProg.ChangeLogLevelProc.getNumber(),
	                     req);
		       if (replyData == null) {
		         /**
		           * <MAPR_ERROR/>
		           */
		         LOG.error("Got null reply from RPC");
		         out.addError(new OutputError(Errno.ERPCFAILED, "dump changeloglevel rpc failed"));
		         return;
		       }
		       ChangeLogLevelResponse resp = ChangeLogLevelResponse
		                                           .parseFrom(replyData);
		       if (resp.getStatus() != 0) {
		    	   int status = resp.getStatus();
		    	   out.addError(new OutputError(status, "Was not able to set loglevel for CLDB Node: " + cldbHostString + " with status: " + status));
		       }
		       
         } catch (MaprSecurityException e) {
           throw new CLIProcessingException(
             "MaprSecurityException " + "Exception", e);
		     } catch (Exception e) {
		       /**
		        * <MAPR_ERROR>
		        *  Exception processing dump command
		        * </MAPR_ERROR>
		        */
		       LOG.error("Exception processing dump command " + e.getLocalizedMessage());
		       out.addError(new OutputError(Errno.ERPCFAILED, "dump changeloglevel failed"));
		     }
		    
		    return;
	  }

	  /**
	   * Helper method to set log level for C++ side of the code
	   * @param service
	   * @param out
	   * @throws CLIProcessingException
	   */
	  private void setFSLogLevel(ServicesEnum service, OutputHierarchy out) throws CLIProcessingException {
	    try {
	        int port = Rpc.initialize(0, 0, null); // Client
	        if (port < 0) {
	        	LOG.error("Error in RPC init");
	        	out.addError(new OutputError(Errno.ERPCFAILED, "Error in RPC init"));
	        	return;
	        }
	      } catch (Exception e) {
	        LOG.error("Exception in Rpc.initialize ", e);
	        out.addError(new OutputError(Errno.ERPCFAILED, "Error in RPC init"));
	        return;
	      }

      try {
	      int port = Integer.valueOf(service.getPort());
        if ( isParamPresent(PORT) ) {
          port = getParamIntValue(PORT, 0);
        }

	      String hostName = getParamTextValue(NODE_IP, 0);
	      if ( !( hostName.equalsIgnoreCase("localhost") || hostName.equalsIgnoreCase("127.0.0.1")) ) {
			  List<String> ips = NodesCommonUtils.convertHostToIp(Collections.singletonList(hostName));
			  if ( ips.isEmpty() ) {
			  	 out.addError(new OutputError(Errno.EINVAL, "Can not get valid IP address out of provided name: " + hostName));
			   	 return;
			  }
			  hostName = ips.get(0);
	      }
	      int hostip = Util.ipToInt(hostName);

        MutableInt err;
        // If NFS is running in user mode, create a ticket using the user key.
        if (isParamPresent(IS_USER_MODE) &&
            getParamTextValue(IS_USER_MODE, 0).equalsIgnoreCase("true")) {
          err = CreateNFSUserTicket(clusterName);
          
          if (err.GetValue() != 0) {
            out.addError(new OutputError(Errno.INTERROR,
                                         "Cannot create the ticket for nfs in user mode"));
            return;
          }
        }

	      long binding = Rpc.createBindingFor(hostip, port, clusterName, ServerKeyType.ServerKey.getNumber());
	      Gtrace.GTraceRequest.Builder req = Gtrace.GTraceRequest.newBuilder();
	      req.setReqType(Gtrace.GTraceRequestType.setLevel);
	      req.setModule(getParamTextValue(CLASS_NAME, 0));
	      req.setLevel(getParamTextValue(LOG_LEVEL_NAME, 0));
	      byte[] replyData;
	        replyData = Rpc.sendRequest(binding,
	            Common.MapRProgramId.GTraceProgramId.getNumber(),
	            Gtrace.GTraceProg.GTraceProc.getNumber(),
	            req.build());
	        if ( replyData == null ) {
	        	LOG.error("null reponse while trying to set log level for: " + getParamTextValue(NODE_IP, 0) + ":" + port);
	        	out.addError(new OutputError(Errno.INTERROR, "null reponse while trying to set log level for: " + getParamTextValue(NODE_IP, 0) + ":" + port));
	        }
	        Gtrace.GTraceResponse resp = Gtrace.GTraceResponse.parseFrom(replyData);

	        if (resp.getStatus() == Errno.EINVAL) {
	            out.addError(new OutputError(Errno.EINVAL,"Operation failed :(EINVAL) Invalid parameters"));
	          } 
	          if (resp.getStatus() == Errno.ENOMEM) {
	            out.addError(new OutputError(Errno.ENOMEM, "Operation failed :(ENOMEM) could not allocate memory"));
	          } 
      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
      } catch (Exception e) {
    	  out.addError(new OutputError(Errno.INTERROR, "Exception during setLogLevel processing"));
      }
	  }

    private MutableInt CreateNFSUserTicket(String clusterName) {
      MutableInt err = new MutableInt();
      TicketAndKey ticketAndKey =
        Security.GetTicketAndKeyForCluster(ServerKeyType.ServerKey,
                                           clusterName,
                                           err);
      if (err.GetValue() != 0)
        return err;
    
      // Update the security layer to use the user key as server key.
      Security.SetKey(ServerKeyType.ServerKey,
                      ticketAndKey.getUserKey());

      // some random gid. this will not be used anywhere.
      String userName = ticketAndKey.getUserCreds().getUserName();
      int[] gids = new int[1];
      gids[0] = userName.length();
      
      ticketAndKey =
        Security.GenerateTicketAndKey(ServerKeyType.ServerKey,
                                      userName,
                                      userName.length(),
                                      gids,
                                      Security.MAX_EXPIRY_TIME,
                                      0 /*non-renewal*/,
                                      true,
                                      false /*canUserImpersonate*/,
                                      err);
      if (err.GetValue() != 0)
        return err;

      // Set the new ticket as the server ticket.
      Security.SetTicketAndKey(ServerKeyType.ServerKey,
                               clusterName,
                               ticketAndKey);
      return err;
    }
	  
	  static final String MARKER = "<!-- OUTPUT -->";
	  static final Pattern TAG = Pattern.compile("<[^>]*>");

	  /**
	   * Helper method to process "hadoop" based setloglevel: JT, TT, hbase
	   * @param urlstring
	   * @param service
	   * @param out
	   * @throws CLIProcessingException
	   */
	  private void process(String urlstring, ServicesEnum service, OutputHierarchy out) throws CLIProcessingException {
		    try {
		      String logLevel = getParamTextValue(LOG_LEVEL_NAME,0);
		      URL url = new URL(urlstring);
		      LOG.info("Connecting to " + url);

          URLConnection connection = MaprHttpURL.openConnection(url);
		      BufferedReader in = new BufferedReader(new InputStreamReader(
		          connection.getInputStream()));
		      for(String line; (line = in.readLine()) != null; )
		        if (line.startsWith(MARKER)) {
		          String result = TAG.matcher(line).replaceAll("");
		          LOG.info(result);
		          if ( result.startsWith("Effective level: ")) {
		        	  String finalLogLevel = result.replace("Effective level: ", "");
		        	  if ( !logLevel.equalsIgnoreCase(finalLogLevel.trim())) {
		        		  out.addError(new OutputError(Errno.INTERROR, "Log Level for : " + service + " Was not set to: " + logLevel));
		        	  }
		        	 
		          }
		        }
		      in.close();
		    } catch (Exception e) {
		      LOG.error("Exception while trying to set loglevel for: " + service, e);
		      out.addError(new OutputError(Errno.INTERROR, "Exception while trying to set loglevel for: " + service));
		    }
		  }

	  
}
