/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.TextFormat;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.CommandOutput.*;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.*;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import com.mapr.cliframework.base.inputparams.IntegerInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.license.LicenseUtil;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.License.*;
import com.mapr.fs.proto.Security.CredentialsMsg;
import org.apache.log4j.Logger;
import com.mapr.security.MaprSecurityException;

import java.io.ByteArrayInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509CRL;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * License support
 */
public class LicenseCommands extends CLIBaseClass
{
  private static final SimpleDateFormat dateFormat =
          new SimpleDateFormat("MMM d, yyyy");

  private static final Logger LOG = Logger.getLogger(LicenseCommands.class);

  private static final String License = "license";
  private static final String IsFile = "is_file";
  private static final String CRL  = "crl";
  private static final String LicenseId = "license_id";

  public static Map<String, BaseInputParameter> baseParams =
    new ImmutableMap.Builder<String, BaseInputParameter>()
    .put(MapRCliUtil.CLUSTER_NAME_PARAM,
      new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
          "cluster name",
          CLIBaseClass.NOT_REQUIRED,
          null))
         .build();

  private static final CLICommand listCommand =
    new CLICommand (
        "list",
        "usage: license list -cluster clustername",
        LicenseCommands.class,
        CLICommand.ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>().
          putAll(baseParams)
        .build(),
        null)
    .setShortUsage("license list -cluster clustername");

  private static final CLICommand appsCommand =
    new CLICommand (
        "apps",
        "usage: license apps -cluster clustername",
        LicenseCommands.class,
        CLICommand.ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>().
          putAll(baseParams)
        .build(),
        null)
    .setShortUsage("license apps -cluster clustername");

  private static final CLICommand showIdCommand =
    new CLICommand (
        "showid",
        "usage: license showid -cluster clustername",
        LicenseCommands.class,
        CLICommand.ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>().
          putAll(baseParams)
            .put("showNodes",
                new BooleanInputParameter("showNodes",
                    "show the total licensed nodes nodes available",
                    CLIBaseClass.NOT_REQUIRED,
                    false))
        .build(),
        null)
    .setShortUsage("license showid -cluster clustername [-showNodes true|false]");

  // -file treats the licensestring as a filename
  private static final CLICommand addCommand =
    new CLICommand (
        "add",
        "usage: license add -is_file [true|false] " +
                "-license long_license_string -cluster clustername",
        LicenseCommands.class,
        CLICommand.ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>().
          putAll(baseParams)
        .put(License, new TextInputParameter(
                License, "long_license_string", CLIBaseClass.REQUIRED, null))
        .put (IsFile, new BooleanInputParameter(
                IsFile, "true|false", CLIBaseClass.NOT_REQUIRED, false))
        .build(),
        null)
    .setShortUsage("license add -is_file [true|false] " +
            "-license long_license_string -cluster clustername");

  private static final CLICommand removeCommand =
      new CLICommand (
          "remove",
          "usage: license remove -license_id id -cluster clustername",
          LicenseCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>().
            putAll(baseParams)
          .put(LicenseId, new TextInputParameter(
                  LicenseId, "id",
                  CLIBaseClass.REQUIRED, null))
          .build(),
          null)
      .setShortUsage("license remove -license_id id -cluster clustername");

  private static final CLICommand addCrlCommand =
    new CLICommand (
        "addcrl",
        "usage: license addcrl -is_file [true|false] -crl crlstring " +
                "-cluster clustername",
        LicenseCommands.class,
        CLICommand.ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>().
          putAll(baseParams)
        .put(CRL, new TextInputParameter(CRL, "crlstring",
                CLIBaseClass.REQUIRED, null))
        .put(IsFile, new BooleanInputParameter(
                IsFile, "true|false", CLIBaseClass.NOT_REQUIRED, false))
        .build(),
        null)
    .setShortUsage("license addcrl -is_file [true|false] -crl crlstring " +
            "-cluster clustername");

  private static final CLICommand listCRLCommand =
    new CLICommand (
        "listcrl",
        "usage: license listcrl -cluster clustername",
        LicenseCommands.class,
        CLICommand.ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>().
          putAll(baseParams)
        .build(),
        null)
    .setShortUsage("license listcrl -cluster clustername");


  // main command
  public static final CLICommand licenseCommands =
    new CLICommand (
         "license", "license [add|remove|list|showid|addcrl|listcrl]",
         CLIUsageOnlyCommand.class,
         CLICommand.ExecutionTypeEnum.NATIVE,
         new CLICommand[] {
           addCommand, removeCommand, listCommand, showIdCommand,
              appsCommand, addCrlCommand, listCRLCommand
         }
       )
    .setShortUsage("license [add|remove|list|apps|showid|addcrl|listcrl]");

  public LicenseCommands (ProcessedInput input, CLICommand cliCommand)
  {
    super(input, cliCommand);
  }

  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException
  {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);

    if (!super.validateInput()) {
      return output;
    }


    if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
      String cluster = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0);
      if (!CLDBRpcCommonUtils.getInstance().isValidClusterName(cluster)) {
        out.addError(new OutputError(Errno.EUCLUSTER, "Invalid cluster: " + cluster));
        return output;
      }
    }

    if (cliCommand.getCommandName().equalsIgnoreCase("add")) {
      addLicense(out);
    } else if (cliCommand.getCommandName().equalsIgnoreCase("list")) {
      listLicenses(out);
    } else if (cliCommand.getCommandName().equalsIgnoreCase("remove")) {
      removeLicense(out);
    } else if (cliCommand.getCommandName().equalsIgnoreCase("addcrl")) {
      addCRL (out);
    } else if (cliCommand.getCommandName().equalsIgnoreCase("showid")) {
      showClusterID(out);
    } else if (cliCommand.getCommandName().equalsIgnoreCase("apps")) {
      showCapabilities (out);
    } else if (cliCommand.getCommandName().equalsIgnoreCase("listcrl")) {
      listCRLs (out);
    }

    return output;
  }
  
  public static LicenseIdResponse fetchClusterID (String cluster, CredentialsMsg creds) throws CLIProcessingException {
    byte[] data;
    LicenseIdResponse resp = null;
    LicenseIdRequest req = LicenseIdRequest.newBuilder()
                                           .setCreds(getLicenseCredentials(creds))
                                           .build();

    try {
      if (cluster != null && !cluster.isEmpty()) {
        data = CLDBRpcCommonUtils.getInstance().sendRequest (
                    cluster,
                    Common.MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.GetLicenseIdProc.getNumber(),
                    req,
                    LicenseIdResponse.class
              );
      } else {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(
                    Common.MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.GetLicenseIdProc.getNumber(),
                    req,
                    LicenseIdResponse.class
          );
      }

      if (data == null) {
        return null;
      }

      // success
      resp = LicenseIdResponse.parseFrom(data);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      LOG.error("Exception during fetch ClusterID", e);
      return null;
    }
    return resp;
  }

  private void showClusterID (OutputHierarchy out)
      throws CLIProcessingException
  {
    String cluster = (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) ? 
                         getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0) : 
                           null;
    LicenseIdResponse licId = fetchClusterID(cluster, getUserCredentials());
    
    if (licId == null) {
      out.addError (new CommandOutput.OutputHierarchy.OutputError (
          Errno.EOPFAILED, "Show ClusterID: " +
          Errno.toString(Errno.EOPFAILED))
        );
      return;
    }

    int status = licId.getStatus();
    if (status != 0) {
      String msg = licId.hasMessage() ? licId.getMessage() : Errno.toString(status);
      out.addError(new CommandOutput.OutputHierarchy.OutputError(status, msg));
      return;
    }

    OutputNode id = new OutputNode();
    id.addChild(new OutputNode("id", licId.getClusterid()));
    if (getParamBooleanValue("showNodes", 0)) {
      id.addChild(new OutputNode("currentLicensedNodes", licId.getNodesUsed()));
      id.addChild(new OutputNode("maxLicensedNodes", licId.getNodesTotal()));


    }
    out.addNode (id);
  }

  private String getLicenseString (boolean isFile, String lic)
        throws CLIProcessingException
  {
    if (! isFile) {
      return lic;
    }

    StringBuilder sb = new StringBuilder();
    FileReader reader = null;

    try {
      reader = new FileReader(lic);
      char[] data = new char[1024];
      int len;
      while ((len = reader.read (data)) != -1) {
        sb.append(data, 0, len);
      }
    } catch (FileNotFoundException fnfe) {
      throw new CLIProcessingException(fnfe.getMessage(), fnfe);
    } catch (IOException ioe) {
      throw new CLIProcessingException(ioe.getMessage(), ioe);
    } finally {
      try { if (reader != null) reader.close(); } catch (IOException ioe) { }
    }

    return sb.toString();
  }

  static LicenseCredentialsMsg getLicenseCredentials(CredentialsMsg creds) {
    LicenseCredentialsMsg.Builder b = LicenseCredentialsMsg.newBuilder();
    b.setUid(creds.getUid());
    b.addAllGids(creds.getGidsList());
    return b.build();
  }

  private void addLicense (OutputHierarchy out)
      throws CLIProcessingException
  {
    String lic = getParamTextValue(License, 0);
    boolean file = getParamBooleanValue(IsFile, 0);

    AddLicenseResponse resp;
    byte[] data;

    try {
      String licenseStr = getLicenseString(file, lic);
      AddLicenseRequest.Builder b = AddLicenseRequest.newBuilder();
      b.setLicense(licenseStr);
      b.setCreds(getLicenseCredentials(getUserCredentials()));

      if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        data = CLDBRpcCommonUtils.getInstance().sendRequest (
                    getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
                    Common.MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.AddLicenseProc.getNumber(),
                    b.build(),
                    AddLicenseResponse.class
              );
      } else {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(
                    Common.MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.AddLicenseProc.getNumber(),
                    b.build(),
                    AddLicenseResponse.class
          );
      }

      if (data == null) {
    	  out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
    	  return;
      }

      // success
      resp = AddLicenseResponse.parseFrom(data);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      out.addError (new CommandOutput.OutputHierarchy.OutputError (
            Errno.EOPFAILED, "AddLicense: " +
            e.getMessage())
          );
      LOG.error("Exception during AddLicense", e);
      return;
    }

    int status = resp.getStatus();
    if (status != 0) {
      String msg = resp.hasMessage() ? resp.getMessage(): Errno.toString(status);
      out.addError(new OutputError(status, msg));
    }
  }

  private void listLicenses (OutputHierarchy out)
      throws CLIProcessingException
  {
    ShowLicenseRequest.Builder b = ShowLicenseRequest.newBuilder();
    b.setAll(true);
    b.setCreds(getLicenseCredentials(getUserCredentials()));

    ShowLicenseResponse resp;
    byte[] data;

    try {
      if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        data = CLDBRpcCommonUtils.getInstance().sendRequest (
                    getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
                    Common.MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.ShowLicenseProc.getNumber(),
                    b.build(),
                    ShowLicenseResponse.class
              );
      } else {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(
                    Common.MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.ShowLicenseProc.getNumber(),
                    b.build(),
                    ShowLicenseResponse.class
          );
      }

      if (data == null) {
    	  out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
    	  return;
      }

      // success
      resp = ShowLicenseResponse.parseFrom(data);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      out.addError (new CommandOutput.OutputHierarchy.OutputError (
            Errno.EOPFAILED, "ShowLicense: " +
            e.getMessage())
          );
      LOG.error("Exception during AddLicense", e);
      return;
    }

    int status = resp.getStatus();
    if (status != 0) {
      String msg = resp.hasMessage() ? resp.getMessage() : Errno.toString(status);
      out.addError(new OutputError(status, msg));
      return;
    }

    // the license expiry time is in seconds.
    long time = System.currentTimeMillis();
    int now = (int) (time/1000);
    
    for (LicenseInfo lInfo : resp.getLicenseInfoList()) {      
      OutputNode row = new OutputNode();
      out.addNode(row);

      OutputNode id = new OutputNode("id", lInfo.getHash());
      row.addChild(id);

      String desc = lInfo.getDescription();
      OutputNode d = new OutputNode("description", desc);
      row.addChild(d);

      if (lInfo.hasExpirationdate()) {
        long exp = lInfo.getExpirationdate();
        Date date = new Date (exp*1000); // its in seconds
        String eFmt = dateFormat.format(date);
        OutputNode dData = new OutputNode("expiry", eFmt);
        row.addChild(dData);
      }

      if (lInfo.hasIssuedate()) {
        long iss = lInfo.getIssuedate();
        Date date = new Date (iss*1000); // its in seconds
        String eFmt = dateFormat.format(date);
        OutputNode dData = new OutputNode("issue", eFmt);
        row.addChild(dData);
      }

      if (lInfo.hasModules()) {
        OutputNode dData = new OutputNode("modules", lInfo.getModules());
        row.addChild(dData);
      }

      String nfsClientNodes = getMaxNfsClientNodes(lInfo, Feature.NFS_CLIENT, Feature.NFS_CLIENT_BASE,
          Feature.POSIX_CLIENT, Feature.POSIX_CLIENT_BASE);
      String goldNfsClientNodes = getMaxNfsClientNodes(lInfo, Feature.POSIX_CLIENT_GOLD);
      String platNfsClientNodes = getMaxNfsClientNodes(lInfo, Feature.POSIX_CLIENT_PLATINUM);
      if (!nfsClientNodes.equals("")) {
        OutputNode nfsClientNodesCount = new OutputNode("posixnodes", nfsClientNodes);
        row.addChild(nfsClientNodesCount);
      } else if (!goldNfsClientNodes.equals("")) {
        OutputNode nfsClientNodesCount = new OutputNode("goldposixnodes", goldNfsClientNodes);
        row.addChild(nfsClientNodesCount);
      } else if (!platNfsClientNodes.equals("")) {
        OutputNode nfsClientNodesCount = new OutputNode("platinumposixnodes", platNfsClientNodes);
        row.addChild(nfsClientNodesCount);
      } else {
        String maxNodes = getMaxNodes(lInfo);
        OutputNode m = new OutputNode("maxnodes", maxNodes);
        row.addChild(m);

      }

      boolean additionalFeature = lInfo.getLicType().getNumber() == LicenseType.AdditionalFeatures.getNumber() ||
            lInfo.getLicType().getNumber() == LicenseType.AdditionalFeaturesBase.getNumber();
      row.addChild(new OutputNode("isAdditionalFeature", additionalFeature));

      boolean deletable =
            lInfo.getLicType().getNumber() != LicenseType.Base.getNumber() &&
            lInfo.getLicType().getNumber() != LicenseType.AdditionalFeaturesBase.getNumber();
      OutputNode del = new OutputNode("deletable", deletable);
      row.addChild(del);

      boolean grace = lInfo.getExpirationdate() < now;
      OutputNode g = new OutputNode("grace", grace);
      row.addChild(g);

      String l = TextFormat.printToString(lInfo);
      OutputNode lic = new OutputNode("license", l);
      row.addChild(lic);
    }
  }

  private final String getMaxNfsClientNodes (LicenseInfo lInfo, Feature ... feature) {
    String m = "";
    List<Capability> l = lInfo.getCapabilitiesList();
    for (Capability c: l) {
      for (Feature f : feature) {
        if (c.getFeature().getNumber() == f.getNumber()) {
          m = c.getFeatureData().getMaxNfsClientNodes();
          break;
        }
      }
    }
    return m;
  }

  private final String getMaxNfsClientNodes (LicenseInfo lInfo) {
    String m = "";
    List<Capability> l = lInfo.getCapabilitiesList();
    for (Capability c: l) {
      if (c.getFeature().getNumber() == Feature.NFS_CLIENT_BASE.getNumber() ||
          c.getFeature().getNumber() == Feature.NFS_CLIENT.getNumber()) {
        m = c.getFeatureData().getMaxNfsClientNodes();
        break;
      }
    }

    return m;
  }

  private final String getMaxNodes (LicenseInfo lInfo)
  {
    String m = "unlimited";
    List<Capability> l = lInfo.getCapabilitiesList();
    for (Capability c: l) {
      if (c.getFeature().getNumber() == Feature.MAXNODES.getNumber()) {
        m = c.getFeatureData().getMaxNodes();
        break;
      }
    }

    return m;
  }

  private void addCRL (OutputHierarchy out)
      throws CLIProcessingException
  {
    String crl = getParamTextValue(CRL, 0);
    boolean file = getParamBooleanValue(IsFile, 0);

    AddCRLResponse resp;
    byte[] data;

    try {
      String crlStr = getLicenseString(file, crl);
      AddCRLRequest.Builder b = AddCRLRequest.newBuilder();
      b.setCrl(crlStr);
      b.setCreds(getLicenseCredentials(getUserCredentials()));

      String hash = LicenseUtil.getHash(crlStr.getBytes("UTF-8"));
      b.setHash(hash);

      if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        data = CLDBRpcCommonUtils.getInstance().sendRequest (
                    getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
                    Common.MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.AddCRLProc.getNumber(),
                    b.build(),
                    AddCRLResponse.class
              );
      } else {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(
                    Common.MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.AddCRLProc.getNumber(),
                    b.build(),
                    AddCRLResponse.class
          );
      }

      if (data == null) {
     	out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
     	return;
      }

      // success
      resp = AddCRLResponse.parseFrom(data);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      out.addError (new CommandOutput.OutputHierarchy.OutputError (
            Errno.EOPFAILED, "AddCRL: " +
            e.getMessage())
          );
      LOG.error("Exception during AddCRL", e);
      return;
    }

    int status = resp.getStatus();
    if (status != 0) {
      String msg = resp.hasMessage() ? resp.getMessage(): Errno.toString(status);
      out.addError(new OutputError(status, msg));
    }
  }

  private void listCRLs (OutputHierarchy out)
      throws CLIProcessingException
  {
    ShowCRLRequest.Builder b = ShowCRLRequest.newBuilder();
    b.setAll(true);
    b.setCreds(getLicenseCredentials(getUserCredentials()));

    ShowCRLResponse resp;
    byte[] data;

    try {
      if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        data = CLDBRpcCommonUtils.getInstance().sendRequest (
                    getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
                    Common.MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.ShowCRLProc.getNumber(),
                    b.build(),
                    ShowCRLResponse.class
              );
      } else {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(
                    Common.MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.ShowCRLProc.getNumber(),
                    b.build(),
                    ShowCRLResponse.class
          );
      }

      if (data == null) {
    	  out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
    	  return;
      }

      // success
      resp = ShowCRLResponse.parseFrom(data);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      out.addError (new CommandOutput.OutputHierarchy.OutputError (
            Errno.EOPFAILED, "ListCRL: " +
            e.getMessage())
          );
      LOG.error("Exception during listcrl", e);
      return;
    }

    int status = resp.getStatus();
    if (status != 0) {
      String msg = resp.hasMessage() ? resp.getMessage() : Errno.toString(status);
      out.addError(new OutputError(status, msg));
      return;
    }

    dumpCRL(out, resp);
  }

  private String getCRLData (String s)
  {
    String data = null;
    try {
      ByteArrayInputStream bis = new ByteArrayInputStream(s.getBytes("UTF-8"));
      CertificateFactory cf = CertificateFactory.getInstance("X.509");
      X509CRL c = (X509CRL) cf.generateCRL(bis);
      data = c.toString();
    } catch (Exception e) {
      // ignore
      data = "could not open CRL, skipping";
    }

    return data;
  }



  private void dumpCRL(OutputHierarchy out, ShowCRLResponse resp)
  {
    List<String> crls = resp.getCrlsList();
    for (int i=0; i < crls.size(); ++i) {
      OutputNode row = new OutputNode();
      out.addNode(row);

      OutputNode idx = new OutputNode("index", i);
      row.addChild(idx);

      String data = getCRLData(crls.get(i));
      OutputNode d = new OutputNode("data", data);
      row.addChild(d);
    }
  }

  private void removeLicense (OutputHierarchy out)
      throws CLIProcessingException
  {
    String hash = getParamTextValue(LicenseId, 0);
    RemoveLicenseRequest.Builder b = RemoveLicenseRequest.newBuilder();
    b.setHash(hash);
    b.setCreds(getLicenseCredentials(getUserCredentials()));

    RemoveLicenseResponse resp;
    byte[] data;

    try {
      if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        data = CLDBRpcCommonUtils.getInstance().sendRequest (
                    getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
                    Common.MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.RemoveLicenseProc.getNumber(),
                    b.build(),
                    RemoveLicenseResponse.class
              );
      } else {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(
                    Common.MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.RemoveLicenseProc.getNumber(),
                    b.build(),
                    RemoveLicenseResponse.class
          );
      }

      if (data == null) {
    	  out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
    	  return;
       }

      // success
      resp = RemoveLicenseResponse.parseFrom(data);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      out.addError (new CommandOutput.OutputHierarchy.OutputError (
            Errno.EOPFAILED, "RemoveLicense: " +
            e.getMessage())
          );
      LOG.error("Exception during RemoveLicense", e);
      return;
    }

    int status = resp.getStatus();
    if (status != 0) {
      String msg = resp.hasMessage() ? resp.getMessage(): Errno.toString(status);
      out.addError(new OutputError(status, msg));
    }
  }

  // similar to showLicense but only gets the system capabilities
  private void showCapabilities (OutputHierarchy out)
      throws CLIProcessingException
  {
    ShowLicenseRequest.Builder b = ShowLicenseRequest.newBuilder();
    b.setAll(false); // gets the capabilities.
    b.setCreds(getLicenseCredentials(getUserCredentials()));

    ShowLicenseResponse resp;
    byte[] data;

    try {
      if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        data = CLDBRpcCommonUtils.getInstance().sendRequest (
                    getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
                    Common.MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.ShowLicenseProc.getNumber(),
                    b.build(),
                    ShowLicenseResponse.class
              );
      } else {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(
                    Common.MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.ShowLicenseProc.getNumber(),
                    b.build(),
                    ShowLicenseResponse.class
          );
      }

      if (data == null) {
    	  out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
    	  return;
      }

      // success
      resp = ShowLicenseResponse.parseFrom(data);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      out.addError (new CommandOutput.OutputHierarchy.OutputError (
            Errno.EOPFAILED, "ShowLicense: " +
            e.getMessage())
          );
      LOG.error("Exception during AddLicense", e);
      return;
    }

    int status = resp.getStatus();
    if (status != 0) {
      String msg = resp.hasMessage() ? resp.getMessage() : Errno.toString(status);
      out.addError(new OutputError(status, msg));
      return;
    }

    // the license expiry time is in seconds.
    long time = System.currentTimeMillis();
    int now = (int) (time/1000);
    boolean hasJM = false;
    boolean spoofJM = true;
    
    for (LicenseInfo lInfo : resp.getLicenseInfoList()) {      
      List<Capability> cList = lInfo.getCapabilitiesList();
      boolean grace = lInfo.getExpirationdate() < now;
      
      for (Capability c: cList) {
        OutputNode row = new OutputNode();
        out.addNode(row);

        OutputNode g = new OutputNode("grace", grace);
        row.addChild(g);

        String cap = c.getFeature().toString();
        OutputNode oc = new OutputNode("capability", cap);
        row.addChild(oc);

        if (c.hasFeatureData()) {
          String fd;
          if (c.getFeatureData().hasMaxNfsNodes()) {
            fd = c.getFeatureData().getMaxNfsNodes();
          } else if (c.getFeatureData().hasMaxNodes()) {
            fd = c.getFeatureData().getMaxNodes();
          } else {
            fd = "";
          }

          OutputNode fData = new OutputNode("featuredata", fd);
          row.addChild(fData);
        }
         
        // Hack: For Job Management (new features in 2.0)
        if (c.hasName() && (lInfo.getVersion().compareToIgnoreCase("5.0") < 0) && 
            (lInfo.getLicType().getNumber() > LicenseType.Registered.getNumber())) {
          if (c.getName().equalsIgnoreCase("SNAPSHOTS") || 
              c.getName().equalsIgnoreCase("MIRRORING"))
            hasJM = true;
          else if (c.getName().equalsIgnoreCase("JM_CHARTS") || 
            c.getName().equalsIgnoreCase("JM_HISTOGRAMS"))
            spoofJM = false;
        }
      } // for each Capability
    } // for each LicenseInfo
    
    if (spoofJM && hasJM) {
      OutputNode row1 = new OutputNode();
      row1.addChild(new OutputNode("grace", false));
      row1.addChild(new OutputNode("capability", "JM_CHARTS"));
   
      OutputNode row2 = new OutputNode();
      row2.addChild(new OutputNode("grace", false));
      row2.addChild(new OutputNode("capability", "JM_HISTOGRAMS"));
      
      out.addNode(row1);
      out.addNode(row2);
    }
  } // END showCapabilities
}
