/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.security.PrivilegedExceptionAction;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import com.mapr.cliframework.base.inputparams.NoValueInputParameter;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.ipc.RemoteException;
import org.apache.hadoop.mapred.JobClient;
import org.apache.hadoop.mapred.JobPriority;
import org.apache.hadoop.mapred.RunningJob;
import org.apache.hadoop.mapred.TaskAttemptID;
import org.apache.hadoop.mapred.JobID;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.yarn.api.records.ApplicationId;
import org.apache.hadoop.yarn.client.api.YarnClient;
import org.apache.hadoop.yarn.util.ConverterUtils;
import org.apache.log4j.Logger;

import com.google.common.collect.ImmutableMap;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cli.common.NodesCommonUtils;
import com.mapr.cli.common.ServicesEnum;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;

import com.mapr.fs.proto.Common.ServiceData;
import com.google.protobuf.InvalidProtocolBufferException;

import com.mapr.fs.proto.Common.ServiceData;
import com.google.protobuf.InvalidProtocolBufferException;

public class JobCommands extends CLIBaseClass implements CLIInterface {
  private static final Logger LOG = Logger.getLogger(JobCommands.class);
  private static ExecutorService es = Executors.newFixedThreadPool(10);
  static final String JOBID_PARAM = "jobid";
  private static final String TAID_PARAM = "taskattemptid";
  private static final String JOB_PRIORITY_PARAM = "priority";
  static final String JOB_LOGVIEWROOT_PARAM = "todir";
  static final String JOB_CONF_PARAM = "jobconf";
  static final String MR2_PARAM = "mr2";

  public JobCommands(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }

  private static Map<String, BaseInputParameter> baseParams =
     new ImmutableMap.Builder<String, BaseInputParameter>()
        .put(MapRCliUtil.CLUSTER_NAME_PARAM,
  	      new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
  	           "cluster name",
  	           CLIBaseClass.NOT_REQUIRED,
  	           null))
  	    .build();  	

  static final CLICommand killJobCmd = new CLICommand(
      "kill",
      "kill a running mapreduce job",
      JobCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
          .putAll(baseParams)
          .put(JOBID_PARAM,
              new TextInputParameter(JOBID_PARAM,
                  "job id",
                  CLIBaseClass.REQUIRED,
                  null))
          .put(MR2_PARAM, new NoValueInputParameter(MR2_PARAM,
                  "Specify this if the job is running on Yarn/MR2",
                  CLIBaseClass.NOT_REQUIRED, false))
          .build(),
      null);

  static final CLICommand modifyJobCmd = new CLICommand(
      "changepriority",
      "change priority of a running mapreduce job",
      JobCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
          .putAll(baseParams)
          .put(JOBID_PARAM,
              new TextInputParameter(JOBID_PARAM,
                  "job id",
                  CLIBaseClass.REQUIRED,
                  null))
          .put(JOB_PRIORITY_PARAM,
              new TextInputParameter(JOB_PRIORITY_PARAM,
                  "priority NORMAL|LOW|VERY_LOW|HIGH|VERY_HIGH",
                  CLIBaseClass.REQUIRED,
                  null))
          .build(),
      null);

  static final CLICommand setupLogLinksJobCmd = new CLICommand(
      "linklogs",
      "Creates symbolic links to all task log locations "
       + "of jobs matching <jobPattern>",
      SetupJobLogLinks.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
          .putAll(baseParams)
          .put(JOBID_PARAM,
              new TextInputParameter(JOBID_PARAM,
                  "job id",
                  CLIBaseClass.REQUIRED,
                  null))
          .put(JOB_LOGVIEWROOT_PARAM,
              new TextInputParameter(JOB_LOGVIEWROOT_PARAM,
                  "job log view root directory",
                  CLIBaseClass.REQUIRED,
                  null))
          .put(JOB_CONF_PARAM,
              new TextInputParameter(JOB_CONF_PARAM,
                  "path to job configuration (job.xml)",
                  CLIBaseClass.NOT_REQUIRED,
                  null))
          .build(),
      null);

  static final CLICommand killTaCmd = new CLICommand(
      "killattempt",
      "kill a mapreduce task attempt",
      JobCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
          .putAll(baseParams)
          .put(TAID_PARAM,
              new TextInputParameter(TAID_PARAM,
                  "task-attempt id",
                  CLIBaseClass.REQUIRED,
                  null))
          .build(),
      null);
  
  static final CLICommand failTaCmd = new CLICommand(
      "failattempt",
      "fail a mapreduce task attempt",
      JobCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
          .putAll(baseParams)
          .put(TAID_PARAM,
              new TextInputParameter(TAID_PARAM,
                  "task-attempt id",
                  CLIBaseClass.REQUIRED,
                  null))
          .build(),
      null);
  
  static final CLICommand statusJobCmd = new CLICommand(
	      "status",
	      "get the status of a running mapreduce job",
	      JobCommands.class,
	      ExecutionTypeEnum.NATIVE,
	      new ImmutableMap.Builder<String, BaseInputParameter>()
	          .putAll(baseParams)
	          .put(JOBID_PARAM,
	              new TextInputParameter(JOBID_PARAM,
	                  "job id",
	                  CLIBaseClass.REQUIRED,
	                  null))
	          .build(),
	      null);

  /* Define sub command */
  public static final CLICommand taskCmds = new CLICommand(
      "task",
      "",
      CLIUsageOnlyCommand.class,
      ExecutionTypeEnum.NATIVE,
        new CLICommand[] { /* array of subcommands */
          killTaCmd,
          failTaCmd
        }
      ).setShortUsage("task [killattempt|failattempt]");

  /* Define sub command */
  public static final CLICommand jobCmds = new CLICommand(
      "job",
      "",
      CLIUsageOnlyCommand.class,
      ExecutionTypeEnum.NATIVE,
        new CLICommand[] { /* array of subcommands */
          killJobCmd,
          modifyJobCmd,
          setupLogLinksJobCmd,
          statusJobCmd
        }
      ).setShortUsage("job [kill|changepriority|linklogs|status]");


  private JobClient getJobClient(OutputHierarchy out) {
    final String zkConnectString;
    if ((zkConnectString = getZkConnectString()) == null) {
      out.addError(new OutputError(Errno.EOPFAILED,
              "Failed to get a valid Zookeeper Connect string"));
      return null;
    }

      try {
      JobClient jc = getJobClient(zkConnectString);
      if (jc == null) {
        out.addError(new OutputError(Errno.EOPFAILED,
            "Failed to connect to jobtracker"));
        return null;
      }

      if (jc.getConf() == null) {
        out.addError(new OutputError(Errno.EOPFAILED,
            "Failed to get Jobclient configuration"));
        return null;
      }

      return jc;
    } catch (RemoteException re) {
      out.addError(new OutputError(Errno.EOPFAILED, getCauseFromExceptionMessage(re)));
      return null;
    }
  }

  private String getZkConnectString() {
    final String zkConnectString;
    try {
      if (isParamPresent(NodesCommonUtils.ZK_CONNECTSTRING)) {
        zkConnectString = getParamTextValue(NodesCommonUtils.ZK_CONNECTSTRING, 0);
      } else if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        String cluster = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0);
        zkConnectString = CLDBRpcCommonUtils.getInstance().getZkConnect(cluster);
      } else {
        zkConnectString = CLDBRpcCommonUtils.getInstance().getZkConnect();
      }
    } catch (CLIProcessingException e) {
      return null;
    }
    return zkConnectString;
  }

    private RunningJob getRunningJob(OutputHierarchy out, final String jobid) {
    final JobClient jc = getJobClient(out);
    if (jc == null) {
      return null;
    }

    try {
      RunningJob rj = getProxyUser().doAs(new PrivilegedExceptionAction<RunningJob>() {

        @Override
        public RunningJob run() throws Exception {
          RunningJob rj = jc.getJob(JobID.forName(jobid));
          return rj;
        }

      });
      //RunningJob rj = jc.getJob(JobID.forName(jobid));
      return rj;
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED,
                                   "Failed to get Job information for "  + jobid +
                                   ", Error: " + getCauseFromExceptionMessage(e)));
    }

    return null;
  }

  /* This is a function that does the real work. */
  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    final OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);


    if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
      String cluster = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0);
      if (!CLDBRpcCommonUtils.getInstance().isValidClusterName(cluster)) {
        out.addError(new OutputError(Errno.EUCLUSTER, "Invalid cluster: " + cluster));
        return output;
      }
    }
    
    String cmd = cliCommand.getCommandName();

    if (cmd.equalsIgnoreCase("kill")) {
      String jobid = getParamTextValue(JOBID_PARAM, 0);
      if ((jobid == null) || jobid.trim().isEmpty()) {
        out.addError(new OutputError(Errno.EINVAL, "Invalid Job Id"));
        return output;
      }

      if(isParamPresent(MR2_PARAM)) {
        final String appIdStr;
        if(jobid.startsWith(ConverterUtils.APPLICATION_PREFIX)) {
          appIdStr = jobid;
        } else {
          appIdStr = jobid.replaceFirst("job", ConverterUtils.APPLICATION_PREFIX);
        }
        try {
          final ApplicationId appid = ConverterUtils.toApplicationId(appIdStr);
          getProxyUser().doAs(new PrivilegedExceptionAction<Void>() {
            @Override
            public Void run() throws Exception {
              final String zkConnectString = getZkConnectString();
              if(zkConnectString != null) {
                final YarnClient yc = MapRCliUtil.getYarnClient(zkConnectString);
                yc.killApplication(appid);
              } else {
                out.addError(new OutputError(Errno.EOPFAILED,
                        "Failed to get a valid Zookeeper Connect string"));
              }
              return null;
            }
          });
        } catch (Exception e) {
          out.addError(new OutputError(Errno.EOPFAILED, "Failed to kill Application with app id: " + appIdStr
                  + ", Error: " + e.getMessage()));
        }
      } else {
        final RunningJob rj = getRunningJob(out, jobid);
        if (rj != null) {
          try {
            getProxyUser().doAs(new PrivilegedExceptionAction<Void>() {

              @Override
              public Void run() throws Exception {
                rj.killJob();
                return null;
              }

            });

            //rj.killJob();
          } catch (Exception e) {
            out.addError(new OutputError(Errno.EOPFAILED,
                    "Failed to get kill job: " + jobid +
                            ", Error: " + getCauseFromExceptionMessage(e)));
          }
        }
      }

    } else if (cmd.equalsIgnoreCase("changepriority")) {
      String jobid = getParamTextValue(JOBID_PARAM, 0);
      if ((jobid == null) || jobid.trim().isEmpty()) {
        out.addError(new OutputError(Errno.EINVAL, "Invalid Job Id"));
        return output;
      }

      final String priority = getParamTextValue(JOB_PRIORITY_PARAM, 0);
      if (priority == null || priority.trim().isEmpty() ||
        JobPriority.valueOf(priority) == null) {
        out.addError(new OutputError(Errno.EINVAL, "Invalid priority"));
        return output;
      }

      final RunningJob rj = getRunningJob(out, jobid);
      if (rj != null) {
        try {
          getProxyUser().doAs(new PrivilegedExceptionAction<Void>() {

            @Override
            public Void run() throws Exception {
              rj.setJobPriority(priority);
              return null;
            }

          });
          //rj.setJobPriority(priority);
        } catch (Exception e) {
          out.addError(new OutputError(Errno.EOPFAILED,
                                       "Failed to change priority for job: "  + jobid +
                                       ", Error: " + getCauseFromExceptionMessage(e)));
        }
      }

    } else if (cmd.equalsIgnoreCase("killattempt")) {
      final String taid = getParamTextValue(TAID_PARAM, 0);
      if ((taid == null) || taid.trim().isEmpty()) {
        out.addError(new OutputError(Errno.EINVAL, "Invalid Task Attempt Id"));
        return output;
      }

      final JobClient jc = getJobClient(out);
      if (jc != null) {
        try {
          getProxyUser().doAs(new PrivilegedExceptionAction<Void>() {

            @Override
            public Void run() throws Exception {
              TaskAttemptID taidObj = TaskAttemptID.forName(taid);
              RunningJob rj = jc.getJob(taidObj.getJobID());
              if (rj != null)
                rj.killTask(taidObj, false);
              return null;
            }
          });
        } catch (Exception e) {
          out.addError(new OutputError(Errno.EOPFAILED,
                                       "Failed to kill task: " + taid +
                                       ", Error: " + getCauseFromExceptionMessage(e)));
        }
      }

    } else if (cmd.equalsIgnoreCase("failattempt")) {
      String jobid = getParamTextValue(JOBID_PARAM, 0);
      if ((jobid == null) || jobid.trim().isEmpty()) {
        out.addError(new OutputError(Errno.EINVAL, "Invalid Job Id"));
        return output;
      }

      final String taid = getParamTextValue(TAID_PARAM, 0);
      if ((taid == null) || taid.trim().isEmpty()) {
        out.addError(new OutputError(Errno.EINVAL, "Invalid Task Attempt Id"));
        return output;
      }
      final JobClient jc = getJobClient(out);
      if (jc != null) {
        try {
          getProxyUser().doAs(new PrivilegedExceptionAction<Void>() {

            @Override
            public Void run() throws Exception {
              TaskAttemptID taidObj = TaskAttemptID.forName(taid);
              RunningJob rj = jc.getJob(taidObj.getJobID());
              if (rj != null)
                rj.killTask(taidObj, true);
              return null;
            }
          });
        } catch (Exception e) {
          out.addError(new OutputError(Errno.EOPFAILED,
                                       "Failed to fail task: " + taid +
                                       ", Error: " + getCauseFromExceptionMessage(e)));
        }
      }
    } else if (cmd.equalsIgnoreCase("status")) {
        String jobid = getParamTextValue(JOBID_PARAM, 0);
        if ((jobid == null) || jobid.trim().isEmpty()) {
          out.addError(new OutputError(Errno.EINVAL, "Invalid Job Id"));
          return output;
        }

        final RunningJob rj = getRunningJob(out, jobid);
        try {
			if (rj != null && rj.getJobStatus()!=null) {
				
				OutputNode node = new OutputNode();
				node.addChild(new OutputNode("Map Progress", rj.getJobStatus().mapProgress()));
				node.addChild(new OutputNode("Reduce Progress", rj.getJobStatus().reduceProgress()));
				out.addNode(node);

			}
		} catch (IOException e) {
			// TODO Auto-generated catch block
			LOG.error("Error while getting the job status for job: "+jobid);
			throw new CLIProcessingException("Error while getting the job status for job: "+jobid);
		}

      }
    return output;
  }

  public JobClient getJobClient(final String zkConnectString) throws RemoteException {
    if (zkConnectString == null || zkConnectString.trim().isEmpty())
      return null;

    Callable<JobClient> jtStatisCallable = new Callable<JobClient>() {
      @Override
      public JobClient call() throws Exception {
    	  
        InetSocketAddress addr = null;
        ServiceData hostInfo =
        		NodesCommonUtils.getServiceMasterData(zkConnectString, ServicesEnum.jobtracker.name());
        
        if ( hostInfo != null ) {
        	if (hostInfo.hasIsRunning() && hostInfo.getIsRunning()) {
        		if (hostInfo.hasHost() && hostInfo.hasPort()) {
        			// construct JT Address:
        			try {
        				addr = new InetSocketAddress(hostInfo.getHost(),
        						hostInfo.getPort());
        			} catch (IllegalArgumentException ex) {
        				LOG.error("JT Info is not valid: " + hostInfo.getHost() + hostInfo.getPort());
        			}
        		}
        	}
        }

        
        if ( LOG.isDebugEnabled() ) {
          LOG.debug("getJobClient jt found");
        }
        if (addr == null) {
          return null;
        }
        final InetSocketAddress jtAddr = addr;
        final Configuration conf = new Configuration();
        conf.setInt("ipc.client.connect.max.retries", 2); // 2 retries

        return getProxyUser().doAs(new PrivilegedExceptionAction<JobClient>() {

          @Override
          public JobClient run() throws Exception {
            JobClient jc = new JobClient(jtAddr, conf);
            jc.setConf(conf);
            return jc;
          }

        });

      }
    };

    List<Callable<JobClient>> callableList = new ArrayList<Callable<JobClient>>();
    callableList.add(jtStatisCallable);

    try {
      JobClient jc = es.invokeAny(callableList, MapRCliUtil.JTTimeout, TimeUnit.MILLISECONDS);
      return jc;
    } catch (InterruptedException e) {
      /**
       * <MAPR_ERROR>
       * Message:InterruptedException during JT Status thread execution
       * Function:MapRCliUtil.getJobClient()
       * Meaning:An error occurred.
       * Resolution:Contact technical support.
       * </MAPR_ERROR>
       */
      LOG.error("InterruptedException during JT Status thread execution");
      return null;
    } catch (ExecutionException e) {
       /**
         * <MAPR_ERROR>
         * Message:ExecutionException during JT Status thread execution
         * Function:MapRCliUtil.getJobClient()
         * Meaning:An error occurred.
         * Resolution:Contact technical support.
         * </MAPR_ERROR>
         */
      if ( e.getCause() != null && e.getCause() instanceof OutOfMemoryError) {
        LOG.fatal("OutOfMemory Error. Application needs to be restarted", e);
        System.exit(1);
      }
      LOG.error("ExecutionException during JT Status thread execution", e);
      // throw RemoteException to the caller so that the exact failure at JT end
      // can be reported back to CLI output / UI.
      if (e.getCause() instanceof RemoteException) {
        throw (RemoteException) e.getCause();
      }
      return null;
    } catch (TimeoutException e) {
       /**
         * <MAPR_ERROR>
         * Message:TimeoutException during JT Status thread execution
         * Function:MapRCliUtil.getJobClient()
         * Meaning:An error occurred.
         * Resolution:Contact technical support.
         * </MAPR_ERROR>
         */
      LOG.error("TimeoutException during JT Status thread execution", e);
      return null;
    }
  }

  /**
   * If the logged in user is same as the process' user, returns the logged in user (CLI case).
   * If the logged in user is different from the process' user, creates and returns the proxy user (Webserver case).
   *
   * @return
   * @throws IOException
   */
  private UserGroupInformation getProxyUser() throws IOException {
    UserGroupInformation currUser = UserGroupInformation.getLoginUser();
    if (currUser.getUserName().equals(getUserLoginId())) {
      return currUser;
    }
    return UserGroupInformation.createProxyUser(getUserLoginId(), currUser);
  }

  /**
   * If the message itself contains a stack trace, this method parses the
   * first line of the stack trace and extracts the message text, if any.
   *
   * The basic case simply returns e.getMessage().
   *
   * @param e the exception
   * @return
   */
  private String getCauseFromExceptionMessage(Exception e) {
    String error = e.getMessage();
    if(error == null) return "";

    if (error.contains("\n")) {
      error = error.split("\n")[0];
    }

    if (error.contains(":")) {
      error = error.split(":")[1];
    }
    LOG.error(e.getMessage());
    return error;
  }

}
