/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import com.mapr.cli.common.PluggableAlarmUtil;
import org.apache.log4j.Logger;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.ByteString;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.MessageLite;
import com.mapr.baseutils.BitSetBytesHelperUtils;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cli.common.ListCommand;
import com.mapr.cli.common.NodeField;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.cliframework.util.FilterUtil;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.FileServerInfo;
import com.mapr.fs.cldb.proto.CLDBProto.FileServerListRequest;
import com.mapr.fs.cldb.proto.CLDBProto.FileServerListResponse;
import com.mapr.fs.cldb.proto.CLDBProto.NodeInfo;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Common.AlarmId;
import com.mapr.fs.proto.Common.AlarmMsg;
import com.mapr.fs.proto.Common.PluggableAlarm;


public class Heatmap extends ListCommand implements CLIInterface {

  public static final String VIEW_PARAM_NAME   = "view";
  public static final String FILTER_PARAM_NAME   = "filter";

  private static final int MAX_LIMIT = Integer.MAX_VALUE;

  static enum Views {
    VIEW_STATUS("status"),
    VIEW_CPU("cpu"),
    VIEW_MEMORY("memory"),
    VIEW_DISKSPACE("diskspace"),
    VIEW_DEBUG_LOGGING("NODE_ALARM_DEBUG_LOGGING"),
    VIEW_SERVICE_CLDB_DOWN("NODE_ALARM_SERVICE_CLDB_DOWN"),
    VIEW_SERVICE_FILESERVER_DOWN("NODE_ALARM_SERVICE_FILESERVER_DOWN"),
    VIEW_SERVICE_JT_DOWN("NODE_ALARM_SERVICE_JT_DOWN"),
    VIEW_SERVICE_TT_DOWN("NODE_ALARM_SERVICE_TT_DOWN"),
    VIEW_SERVICE_HBMASTER_DOWN("NODE_ALARM_SERVICE_HBMASTER_DOWN"),
    VIEW_SERVICE_HBREGION_DOWN("NODE_ALARM_SERVICE_HBREGION_DOWN"),
    VIEW_SERVICE_NFS_DOWN("NODE_ALARM_SERVICE_NFS_DOWN"),
    VIEW_SERVICE_WEBSERVER_DOWN("NODE_ALARM_SERVICE_WEBSERVER_DOWN"),
    VIEW_SERVICE_HOSTSTATS_DOWN("NODE_ALARM_SERVICE_HOSTSTATS_DOWN"),
    VIEW_DISK_FAILURE("NODE_ALARM_DISK_FAILURE"),
    VIEW_VERSION_MISMATCH("NODE_ALARM_VERSION_MISMATCH"),
    VIEW_TIME_SKEW("NODE_ALARM_TIME_SKEW"),
    VIEW_HB_PROCESSING_SLOW("NODE_ALARM_HB_PROCESSING_SLOW"),
    VIEW_ROOT_PARTITION_FULL("NODE_ALARM_ROOT_PARTITION_FULL"),
    VIEW_OPT_MAPR_FULL("NODE_ALARM_OPT_MAPR_FULL"),
    VIEW_CORE_PRESENT("NODE_ALARM_CORE_PRESENT"),
    VIEW_HIGH_MFS_MEMORY("NODE_ALARM_HIGH_MFS_MEMORY"),
    VIEW_PAM_MISCONFIGURED("NODE_ALARM_PAM_MISCONFIGURED"),
    VIEW_TT_LOCALDIR_FULL("NODE_ALARM_TT_LOCALDIR_FULL"),
    VIEW_NO_HEARTBEAT("NODE_ALARM_NO_HEARTBEAT"),
    VIEW_UID_MISMATCH("NODE_ALARM_MAPRUSER_MISMATCH"),
    VIEW_DUPLICATE_HOSTID("NODE_ALARM_DUPLICATE_HOSTID"),
    VIEW_METRICS_WRITE_PROBLEM("NODE_ALARM_METRICS_WRITE_PROBLEM"),
    VIEW_TOO_MANY_CONTAINERS("NODE_ALARM_TOO_MANY_CONTAINERS"),
    VIEW_SERVICE_HUE_DOWN("NODE_ALARM_SERVICE_HUE_DOWN"),
    VIEW_SERVICE_HTTPFS_DOWN("NODE_ALARM_SERVICE_HTTPFS_DOWN"),
    VIEW_SERVICE_BEESWAX_DOWN("NODE_ALARM_SERVICE_BEESWAX_DOWN"),
    VIEW_SERVICE_HIVEMETA_DOWN("NODE_ALARM_SERVICE_HIVEMETA_DOWN"),
    VIEW_SERVICE_HS2_DOWN("NODE_ALARM_SERVICE_HS2_DOWN"),
    VIEW_SERVICE_OOZIE_DOWN("NODE_ALARM_SERVICE_OOZIE_DOWN"),
    VIEW_PLUGGABLE_ALARM("PLUGGABLE_ALARM");

    String view = null;
    Views(String view) {
      this.view = view;
    }

    String getView() {
      return this.view;
    }
  };

 private static final Logger LOG = Logger.getLogger(Heatmap.class);

  /* Define sub command */
  public static final CLICommand heatmapCmds = new CLICommand(
      "heatmap",
      "",
      Heatmap.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .put(Heatmap.VIEW_PARAM_NAME,
          new TextInputParameter(Heatmap.VIEW_PARAM_NAME,
              "view [status|cpu|memory|diskspace|" +
                    "NODE_ALARM_SERVICE_CLDB_DOWN|" +
                    "NODE_ALARM_SERVICE_FILESERVER_DOWN|" +
                    "NODE_ALARM_SERVICE_JT_DOWN|" +
                    "NODE_ALARM_SERVICE_TT_DOWN|" +
                    "NODE_ALARM_SERVICE_HBMASTER_DOWN|" +
                    "NODE_ALARM_SERVICE_HBREGION_DOWN|" +
                    "NODE_ALARM_SERVICE_NFS_DOWN|" +
                    "NODE_ALARM_SERVICE_WEBSERVER_DOWN|" +
                    "NODE_ALARM_SERVICE_HOSTSTATS_DOWN|" +
                    "NODE_ALARM_DEBUG_LOGGING|" +
                    "NODE_ALARM_DISK_FAILURE|" +
                    "NODE_ALARM_VERSION_MISMATCH|" +
                    "NODE_ALARM_TIME_SKEW|" +
                    "NODE_ALARM_ROOT_PARTITION_FULL|" +
                    "NODE_ALARM_OPT_MAPR_FULL|" +
                    "NODE_ALARM_CORE_PRESENT|" +
                    "NODE_ALARM_HIGH_MFS_MEMORY|" +
                    "NODE_ALARM_PAM_MISCONFIGURED|" +
                    "NODE_ALARM_TT_LOCALDIR_FULL|" +
                    "NODE_ALARM_NO_HEARTBEAT|" +
                    "NODE_ALARM_MAPRUSER_MISMATCH|" +
                    "NODE_ALARM_DUPLICATE_HOSTID|" +
                    "NODE_ALARM_METRICS_WRITE_PROBLEM|" +
                    "NODE_ALARM_TOO_MANY_CONTAINERS|" +
                    "NODE_ALARM_SERVICE_HUE_DOWN|" +
                    "NODE_ALARM_SERVICE_HTTPFS_DOWN|" +
                    "NODE_ALARM_SERVICE_BEESWAX_DOWN|" +
                    "NODE_ALARM_SERVICE_HIVEMETA_DOWN|" +
                    "NODE_ALARM_SERVICE_HS2_DOWN|" +
                    "NODE_ALARM_SERVICE_OOZIE_DOWN" +
                  "]",
              CLIBaseClass.NOT_REQUIRED,
              "status"))
      .put(Heatmap.FILTER_PARAM_NAME,
          new TextInputParameter(Heatmap.FILTER_PARAM_NAME,
              "none",
              CLIBaseClass.NOT_REQUIRED,
              "none"))
      .put(MapRCliUtil.CLUSTER_NAME_PARAM,
          new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
          "cluster name",
          CLIBaseClass.NOT_REQUIRED,
          null))
         .build(),
      null
  ).setShortUsage("heatmap -view [status|cpu|memory|diskspace|" +
  		"NODE_ALARM_SERVICE_CLDB_DOWN|" +
      "NODE_ALARM_SERVICE_FILESERVER_DOWN|" +
      "NODE_ALARM_SERVICE_JT_DOWN|" +
      "NODE_ALARM_SERVICE_TT_DOWN|" +
      "NODE_ALARM_SERVICE_HBMASTER_DOWN|" +
      "NODE_ALARM_SERVICE_HBREGION_DOWN|" +
      "NODE_ALARM_SERVICE_NFS_DOWN|" +
      "NODE_ALARM_SERVICE_WEBSERVER_DOWN|" +
      "NODE_ALARM_SERVICE_HOSTSTATS_DOWN|" +
      "NODE_ALARM_DEBUG_LOGGING|" +
      "NODE_ALARM_DISK_FAILURE|" +
      "NODE_ALARM_VERSION_MISMATCH|" +
      "NODE_ALARM_TIME_SKEW|" +
      "NODE_ALARM_ROOT_PARTITION_FULL|" +
      "NODE_ALARM_OPT_MAPR_FULL|" +
      "NODE_ALARM_CORE_PRESENT|" +
      "NODE_ALARM_HIGH_MFS_MEMORY|" +
      "NODE_ALARM_PAM_MISCONFIGURED|" +
      "NODE_ALARM_TT_LOCALDIR_FULL|" +
      "NODE_ALARM_NO_HEARTBEAT|" +
      "NODE_ALARM_MAPRUSER_MISMATCH|" +
      "NODE_ALARM_DUPLICATE_HOSTID|" +
      "NODE_ALARM_METRICS_WRITE_PROBLEM|" +
      "NODE_ALARM_TOO_MANY_CONTAINERS|" +
      "NODE_ALARM_SERVICE_HUE_DOWN|" +
      "NODE_ALARM_SERVICE_HTTPFS_DOWN|" +
      "NODE_ALARM_SERVICE_BEESWAX_DOWN|" +
      "NODE_ALARM_SERVICE_HIVEMETA_DOWN|" +
      "NODE_ALARM_SERVICE_HS2_DOWN|" +
      "NODE_ALARM_SERVICE_OOZIE_DOWN" +
      "]"
  );

  public Heatmap(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }

  /* This is a function that does the real work. */
  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException  {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);
    OutputNode topNode = new OutputNode();
    out.addNode(topNode);
    ServerCommands.fieldTable = PluggableAlarmUtil.appendNodeMap(getUserCredentials(), ServerCommands.fieldTableBuilder);
    list(out);
    return output;
  }

  @Override
  public FileServerListRequest buildNextRequest(MessageLite prevReq, MessageLite prevResp) throws CLIProcessingException {
    FileServerListRequest.Builder newReqBuilder = null;
    if (prevReq != null) {
      newReqBuilder = FileServerListRequest.newBuilder((FileServerListRequest) prevReq);
    } else {
      newReqBuilder = getFileServerListRequestBuilder();
    }

    if (prevResp != null) {
      int prevStart = newReqBuilder.getLimiter().getStart();
      int prevCount = ((FileServerListResponse) prevResp).getInfoCount();
      newReqBuilder.setLimiter(getNextLimiter(prevStart, prevCount,
          0, MAX_LIMIT, ServerCommands.NUM_NODES_PER_RPC));
    }

    return newReqBuilder.build();
  }

  @Override
  public boolean hasMore(MessageLite prevReq, MessageLite prevResp) throws CLIProcessingException {
    return hasMore(0, MAX_LIMIT,
        ((FileServerListRequest) prevReq).getLimiter().getStart(),
        ((FileServerListResponse) prevResp).getInfoCount());
  }

  @Override
  public void processResponse(OutputHierarchy out, MessageLite response) throws CLIProcessingException {
    FileServerListResponse resp = (FileServerListResponse) response;
    if (resp.getStatus() != 0) {
      /**
      * <MAPR_ERROR>
      * Message:RPC to view Heatmap failed
      * Function:Heatmap.processResponse()
      * Meaning:A communication error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      LOG.error("RPC to view Heatmap failed");
      out.addError(new OutputError(resp.getStatus(), Errno.toString(resp.getStatus())));
      return;
    }
    Map<String, List<FileServerInfo>> fileServerMap = new TreeMap<String, List<FileServerInfo>>();

    for (FileServerInfo fs : resp.getInfoList()) {
      if (!fs.hasNetworkLocation()) {
        continue;
      }
      String rp = fs.getNetworkLocation();
      if (rp.isEmpty()) {
        continue;
      }
      rp = rp.substring(0, rp.lastIndexOf("/"));

      if (!fileServerMap.containsKey(rp)) {
        fileServerMap.put(rp, new ArrayList<FileServerInfo>());
      }
      fileServerMap.get(rp).add(fs);
    }

    OutputNode topNode = out.getOutputNodes().get(0);
    if (!fileServerMap.isEmpty()) {
      for (Map.Entry<String, List<FileServerInfo>> fsInfoEntry : fileServerMap.entrySet()) {
        OutputNode rackNode = getChildNode(topNode, fsInfoEntry.getKey());
        if (rackNode == null) {
          rackNode = new OutputNode(fsInfoEntry.getKey());
          topNode.addChild(rackNode);
        } // else, re-use the rack object to add nodes to it

        for (FileServerInfo fs : fsInfoEntry.getValue()) {
          int status = ServerCommands.formatNodeState(fs.getNodeState());
          boolean hbStats = fs.hasHbStats();
          switch (getView()) {
          case VIEW_STATUS:
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), status));
            break;

          case VIEW_CPU:
            int cpuUtil;
            if (!hbStats) {
              cpuUtil = -1;
            } else {
              cpuUtil = 100 - (int) (fs.getHbStats().getCpuIdle());
            }
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), cpuUtil));
            break;

          case VIEW_MEMORY:
            double memUtil;
            if (!hbStats) {
              memUtil = -1.0;
            } else {
              memUtil = (100.0 * fs.getHbStats().getMemoryUsedMB())
                  / fs.getHbStats().getMemoryTotalMB();
            }
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), (int) memUtil));
            break;

          case VIEW_DISKSPACE:
            double diskUtil;
            if (!hbStats) {
              diskUtil = -1.0;
            } else {
              diskUtil = (100.0 * fs.getHbStats().getServerUsedSizeMB())
                  / fs.getHbStats().getServerCapacitySizeMB();
            }

            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), (int) diskUtil));
            break;

          case VIEW_DEBUG_LOGGING:
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_DEBUG_LOGGING)));
            break;

          case VIEW_SERVICE_CLDB_DOWN:
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_SERVICE_CLDB_DOWN)));
            break;

          case VIEW_SERVICE_FILESERVER_DOWN:
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_SERVICE_FILESERVER_DOWN)));
            break;

          case VIEW_SERVICE_JT_DOWN:
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_SERVICE_JT_DOWN)));
            break;

          case VIEW_SERVICE_TT_DOWN:
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_SERVICE_TT_DOWN)));
            break;

          case VIEW_SERVICE_HBMASTER_DOWN:
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_SERVICE_HBMASTER_DOWN)));
            break;

          case VIEW_SERVICE_HBREGION_DOWN:
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_SERVICE_HBREGION_DOWN)));
            break;

          case VIEW_SERVICE_NFS_DOWN:
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_SERVICE_NFS_DOWN)));
            break;

          case VIEW_SERVICE_WEBSERVER_DOWN:
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_SERVICE_WEBSERVER_DOWN)));
            break;

          case VIEW_SERVICE_HOSTSTATS_DOWN:
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_SERVICE_HOSTSTATS_DOWN)));
            break;

          case VIEW_DISK_FAILURE:
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), fs.getHbStats()
                .getFaileddisks()));
            break;

          case VIEW_VERSION_MISMATCH:
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_VERSION_MISMATCH)));
            break;

          case VIEW_TIME_SKEW:
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_TIME_SKEW)));
            break;

          case VIEW_HB_PROCESSING_SLOW:
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_HB_PROCESSING_SLOW)));
            break;


          case VIEW_ROOT_PARTITION_FULL:
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_ROOT_PARTITION_FULL)));
            break;

          case VIEW_OPT_MAPR_FULL:
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_OPT_MAPR_FULL)));
            break;

          case VIEW_CORE_PRESENT:
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_CORE_PRESENT)));
            break;

          case VIEW_HIGH_MFS_MEMORY:
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_HIGH_MFS_MEMORY)));
            break;

          case VIEW_PAM_MISCONFIGURED:
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_PAM_MISCONFIGURED)));
            break;

          case VIEW_TT_LOCALDIR_FULL:
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_TT_LOCALDIR_FULL)));
            break;

          case VIEW_NO_HEARTBEAT:
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_NO_HEARTBEAT)));
            break;

          case VIEW_UID_MISMATCH:
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_MAPRUSER_MISMATCH)));
            break;

          case VIEW_DUPLICATE_HOSTID:
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_DUPLICATE_HOSTID)));
            break;
          case VIEW_METRICS_WRITE_PROBLEM:
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_METRICS_WRITE_PROBLEM)));
            break;
          case VIEW_TOO_MANY_CONTAINERS:
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_TOO_MANY_CONTAINERS)));
            break;
          case VIEW_SERVICE_HUE_DOWN:
             rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_SERVICE_HUE_DOWN)));
            break;
          case VIEW_SERVICE_HTTPFS_DOWN:
             rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_SERVICE_HTTPFS_DOWN)));
            break;
          case VIEW_SERVICE_BEESWAX_DOWN:
             rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_SERVICE_BEESWAX_DOWN)));
            break;
          case VIEW_SERVICE_HIVEMETA_DOWN:
             rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_SERVICE_HIVEMETA_DOWN)));
            break;
          case VIEW_SERVICE_HS2_DOWN:
             rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_SERVICE_HS2_DOWN)));
            break;
          case VIEW_SERVICE_OOZIE_DOWN:
             rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                AlarmId.NODE_ALARM_SERVICE_OOZIE_DOWN)));
            break;
          case VIEW_PLUGGABLE_ALARM:
            // Check has already been done in getView
            PluggableAlarm pluggableAlarm = PluggableAlarmUtil.getAlarmByName(getUserCredentials(),
                getParamTextValue(VIEW_PARAM_NAME, 0));
            rackNode.addChild(new OutputNode(MapRCliUtil.getHostname(fs), getAlarmState(fs,
                pluggableAlarm.getName())));
            break;
          default:
            break;
          }
        }
      }
    }
  }

  @Override
  public FileServerListResponse sendRequest(MessageLite request) throws CLIProcessingException {
    FileServerListRequest req = (FileServerListRequest) request;
    byte[] replyData = null;
    try {
      if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        replyData =  CLDBRpcCommonUtils.getInstance().sendRequest(
                        getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
                        Common.MapRProgramId.CldbProgramId.getNumber(),
                        CLDBProto.CLDBProg.FileServerListProc.getNumber(),
                        req, FileServerListResponse.class);
      } else {
        replyData =  CLDBRpcCommonUtils.getInstance().sendRequest(
                        Common.MapRProgramId.CldbProgramId.getNumber(),
                        CLDBProto.CLDBProg.FileServerListProc.getNumber(),
                        req, FileServerListResponse.class);
      }
    } catch (Exception e) {
      throw new CLIProcessingException(e);
    }

    if (replyData != null) {
      return getFileServerListResponse(replyData);
    } else {
      /**
      * <MAPR_ERROR>
      * Message:RPC Request to list Nodes failed. No data returned
      * Function:Heatmap.sendRequest()
      * Meaning:A communication error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      LOG.error("RPC Request to list Nodes failed. No data returned");
      return null;
    }
  }

  private OutputNode getChildNode(OutputNode parentNode, String child) {
    for (OutputNode node : parentNode.getChildren()) {
      if (node.getName().equals(child)) {
        return node;
      }
    }
    return null;
  }

  private FileServerListResponse getFileServerListResponse(byte[] replyData)
      throws CLIProcessingException {
    try {
      return FileServerListResponse.parseFrom(replyData);
    } catch (InvalidProtocolBufferException ipbe) {
      /**
      * <MAPR_ERROR>
      * Message:Exception while parsing the RPC response data into FileServerListResponse proto object.
      * Function:Heatmap.getFileServerListResponse()
      * Meaning:An exception occurred while parsing the response.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      throw new CLIProcessingException("Exception while parsing the RPC " +
          "response data into FileServerListResponse proto object.", ipbe);
    }
  }

  private FileServerListRequest.Builder getFileServerListRequestBuilder() throws CLIProcessingException {
    BitSet columns = getColumns();
    ByteString bString = ByteString.copyFrom(BitSetBytesHelperUtils.toByteArray(columns));
    // for backward compatibility with CLDB
    // we will still set setColumns (even it is not really correct ones
    // and everything on columnsAdd bytes array
    long columnsOld = BitSetBytesHelperUtils.convert(columns);

    return FileServerListRequest.newBuilder()
               .setCreds(getUserCredentials())
               .addAllFilter(getFilters(ServerCommands.fieldTable, FILTER_PARAM_NAME))
               .setColumns(columnsOld)
               .setColumnsAdd(bString)
               .setLimiter(getNextLimiter(0, 0, 0, MAX_LIMIT, ServerCommands.NUM_NODES_PER_RPC));
  }

  private Views getView() throws CLIProcessingException {
    String name = getParamTextValue(VIEW_PARAM_NAME, 0);
    for (Views view : Views.values()) {
      if (name.equalsIgnoreCase(view.getView())) {
        return view;
      }
    }
    if (PluggableAlarmUtil.getAlarmByName(getUserCredentials(), name) != null) {
      return Views.VIEW_PLUGGABLE_ALARM;
    }

    /**
    * <MAPR_ERROR>
    * Message:No view found with name: <name>
    * Function:Heatmap.getView()
    * Meaning:The specified view was not found.
    * Resolution:Check the command syntax and try again.
    * </MAPR_ERROR>
    */
    throw new CLIProcessingException("No view found with name: " + name);
  }

  private BitSet getColumns() throws CLIProcessingException {
    String viewField = getParamTextValue(VIEW_PARAM_NAME, 0);
    String nodeInfoField = null;
    if (viewField.equalsIgnoreCase("status")) {
      nodeInfoField = "";
    } else if (viewField.equalsIgnoreCase("cpu")) {
      nodeInfoField = ServerCommands.fieldTable
                                    .get(new NodeField(NodeInfo.CpuUtil))
                                    .getShortName();
    } else if (viewField.equalsIgnoreCase("memory")) {
      nodeInfoField = ServerCommands.fieldTable
                                    .get(new NodeField(NodeInfo.MemUsed))
                                    .getShortName() +
                      "," +
                      ServerCommands.fieldTable
                                    .get(new NodeField(NodeInfo.MemTotal))
                                    .getShortName()              ;
    } else if (viewField.equalsIgnoreCase("diskspace")) {
      nodeInfoField = ServerCommands.fieldTable
                                    .get(new NodeField(NodeInfo.DiskUsed))
                                    .getShortName() +
                      "," +
                      ServerCommands.fieldTable
                                    .get(new NodeField(NodeInfo.DiskTotal))
                                    .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_DEBUG_LOGGING")) {
      nodeInfoField = ServerCommands.fieldTable
                                    .get(new NodeField(NodeInfo.LogLevelAlarm))
                                    .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_SERVICE_CLDB_DOWN")) {
      nodeInfoField = ServerCommands.fieldTable
                                    .get(new NodeField(NodeInfo.ServiceCLDBDownAlarm))
                                    .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_SERVICE_FILESERVER_DOWN")) {
      nodeInfoField = ServerCommands.fieldTable
                                    .get(new NodeField(NodeInfo.ServiceFileserverDownAlarm))
                                    .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_SERVICE_JT_DOWN")) {
      nodeInfoField = ServerCommands.fieldTable
                                    .get(new NodeField(NodeInfo.ServiceJTDownAlarm))
                                    .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_SERVICE_TT_DOWN")) {
      nodeInfoField = ServerCommands.fieldTable
                                    .get(new NodeField(NodeInfo.ServiceTTDownAlarm))
                                    .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_SERVICE_HBMASTER_DOWN")) {
      nodeInfoField = ServerCommands.fieldTable
                                    .get(new NodeField(NodeInfo.ServiceHBMasterDownAlarm))
                                    .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_SERVICE_HBREGION_DOWN")) {
      nodeInfoField = ServerCommands.fieldTable
                                    .get(new NodeField(NodeInfo.ServiceHBRegionDownAlarm))
                                    .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_SERVICE_NFS_DOWN")) {
      nodeInfoField = ServerCommands.fieldTable
                                    .get(new NodeField(NodeInfo.ServiceNFSDownAlarm))
                                    .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_SERVICE_WEBSERVER_DOWN")) {
      nodeInfoField = ServerCommands.fieldTable
                                    .get(new NodeField(NodeInfo.ServiceWebserverDownAlarm))
                                    .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_SERVICE_HOSTSTATS_DOWN")) {
      nodeInfoField = ServerCommands.fieldTable
                                    .get(new NodeField(NodeInfo.ServiceHoststatsDownAlarm))
                                    .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_DISK_FAILURE")) {
      nodeInfoField = ServerCommands.fieldTable
                                    .get(new NodeField(NodeInfo.FailedDisks))
                                    .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_VERSION_MISMATCH")) {
      nodeInfoField = ServerCommands.fieldTable
                                    .get(new NodeField(NodeInfo.VersionMismatchAlarm))
                                    .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_TIME_SKEW")) {
      nodeInfoField = ServerCommands.fieldTable
                                    .get(new NodeField(NodeInfo.TimeSkewAlarm))
                                    .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_HB_PROCESSING_SLOW")) {
      nodeInfoField = ServerCommands.fieldTable
                                    .get(new NodeField(NodeInfo.HbProcessingSlow))
                                    .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_ROOT_PARTITION_FULL")) {
      nodeInfoField = ServerCommands.fieldTable
                                    .get(new NodeField(NodeInfo.NodeRootPartitionFull))
                                    .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_OPT_MAPR_FULL")) {
      nodeInfoField = ServerCommands.fieldTable
                                    .get(new NodeField(NodeInfo.NodeOptMapRFull))
                                    .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_CORE_PRESENT")) {
      nodeInfoField = ServerCommands.fieldTable
                                    .get(new NodeField(NodeInfo.NodeCorePresent))
                                    .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_HIGH_MFS_MEMORY")) {
      nodeInfoField = ServerCommands.fieldTable
                                    .get(new NodeField(NodeInfo.NodeHighMfsMemory))
                                    .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_PAM_MISCONFIGURED")) {
      nodeInfoField = ServerCommands.fieldTable
                                    .get(new NodeField(NodeInfo.NodePamMisconfigured))
                                    .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_TT_LOCALDIR_FULL")) {
      nodeInfoField = ServerCommands.fieldTable
                                    .get(new NodeField(NodeInfo.NodeTTLocaldirFull))
                                    .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_NO_HEARTBEAT")) {
      nodeInfoField = ServerCommands.fieldTable
                                    .get(new NodeField(NodeInfo.NodeNoHeartbeat))
                                    .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_MAPRUSER_MISMATCH")) {
      nodeInfoField = ServerCommands.fieldTable
          .get(new NodeField(NodeInfo.NodeMaprUserMismatch))
          .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_DUPLICATE_HOSTID")) {
      nodeInfoField = ServerCommands.fieldTable
          .get(new NodeField(NodeInfo.NodeDuplicateHostId))
          .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_METRICS_WRITE_PROBLEM")) {
      nodeInfoField = ServerCommands.fieldTable
      .get(new NodeField(NodeInfo.NodeMetricsWriteProblemAlarm))
      .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_TOO_MANY_CONTAINERS")) {
      nodeInfoField = ServerCommands.fieldTable
          .get(new NodeField(NodeInfo.NodeTooManyContainersAlarm))
          .getShortName();
     } else if (viewField.equalsIgnoreCase("NODE_ALARM_SERVICE_HUE_DOWN")) {
      nodeInfoField = ServerCommands.fieldTable
          .get(new NodeField(NodeInfo.ServiceHueDownAlarm))
          .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_SERVICE_HTTPFS_DOWN")) {
      nodeInfoField = ServerCommands.fieldTable
          .get(new NodeField(NodeInfo.ServiceHttpfsDownAlarm))
          .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_SERVICE_BEESWAX_DOWN")) {
      nodeInfoField = ServerCommands.fieldTable
          .get(new NodeField(NodeInfo.ServiceBeeswaxDownAlarm))
          .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_SERVICE_HIVEMETA_DOWN")) {
      nodeInfoField = ServerCommands.fieldTable
          .get(new NodeField(NodeInfo.ServiceHiveMetaDownAlarm))
          .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_SERVICE_HS2_DOWN")) {
      nodeInfoField = ServerCommands.fieldTable
          .get(new NodeField(NodeInfo.ServiceHs2DownAlarm))
          .getShortName();
    } else if (viewField.equalsIgnoreCase("NODE_ALARM_SERVICE_OOZIE_DOWN")) {
      nodeInfoField = ServerCommands.fieldTable
          .get(new NodeField(NodeInfo.ServiceOozieDownAlarm))
          .getShortName();
    } else {
      PluggableAlarm pluggableAlarm = PluggableAlarmUtil.getAlarmByName(getUserCredentials(), viewField);
      if (pluggableAlarm != null) {
        nodeInfoField = pluggableAlarm.getTerse();
      } else {
        throw new CLIProcessingException("Invalid view: " + viewField);
      }
    }

    String columnsString = ServerCommands.fieldTable.get(new NodeField(NodeInfo.Hostname)).getShortName() +
                           "," +
                           ServerCommands.fieldTable.get(new NodeField(NodeInfo.RackPath)).getShortName() +
                           "," +
                           ServerCommands.fieldTable.get(new NodeField(NodeInfo.Status)).getShortName() +
                           "," +
                           nodeInfoField;

    return FilterUtil.getColumns(ServerCommands.fieldTable, columnsString.trim());
  }

  /**
   * Returns the state of the alarm specified by AlarmId.
   * Calls getAlarmState with the AlarmId name
   *
   * @param fs
   *          The {@link FileServerInfo} object containing the {@link AlarmMsg}
   *          objects.
   * @param id
   *          {@link AlarmId} of the alarm.
   * @return 1 if alarm is raised, 0 if clear
   */
  private int getAlarmState(FileServerInfo fs, AlarmId id) {
    return getAlarmState(fs, id.name());
  }

  /**
   * Returns the state of the alarm specified by the Alarm Name.
   *
   * @param fs
   *          The {@link FileServerInfo} object containing the {@link AlarmMsg}
   *          objects.
   * @param alarmName
   *          {@link PluggableAlarm} the alarm name to look up. Either genetrated from {@link AlarmId} or {@link PluggableAlarm}
   * @return 1 if alarm is raised, 0 if clear
   */
  // TODO(smarella): Instead of returning 0 when alarm not found, throw an exception
  private int getAlarmState(FileServerInfo fs, String alarmName) {
    List<AlarmMsg> alarms = fs.getFsAlarmsList();
    if (alarms.size() > 0) {
      for (AlarmMsg alarm : alarms) {
        if (alarm.getAlarmName().equalsIgnoreCase(alarmName)) {
          return alarm.getAlarmState() ? 1 : 0;
        }
      }
      /**
      * <MAPR_ERROR>
      * Message:Could not find the alarm. AlarmId: <alarm>. FileServerId : <FileServer ID>
      * Function:Heatmap.getAlarmState()
      * Meaning:The specified alarm could not be found.
      * Resolution:Check the alarm name and try again.
      * </MAPR_ERROR>
      */
      LOG.error("Could not find the alarm. AlarmId: " + alarmName + ". FileServerId : "
          + fs.getFileServerId());
      return 0;
    }
    /**
    * <MAPR_ERROR>
    * Message:No alarms found. FileServerId : <FileServer ID>
    * Function:Heatmap.getAlarmState()
    * Meaning:No alarms were found for the specified FileServer.
    * Resolution:Contact technical support.
    * </MAPR_ERROR>
    */
    LOG.error("No alarms found. FileServerId : " + fs.getFileServerId());
    return 0;
  }
}
