/* Copyright (c) 2014 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Formatter;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import java.util.regex.Matcher;
import java.math.BigInteger;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.ByteString;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cli.common.NodesCommonUtils;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.TextCommandOutput;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import com.mapr.cliframework.base.inputparams.IntegerInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.cliframework.base.inputparams.LongInputParameter;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.fs.Rpc;
import com.mapr.fs.AceHelper;
import com.mapr.fs.cldb.conf.CLDBConfiguration;
import com.mapr.fs.cldb.proto.CLDBProto.*;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.util.Util;
import com.mapr.fs.proto.Security.ServerKeyType;
import com.mapr.fs.proto.Common.FidMsg;
import com.mapr.fs.proto.Common.FileType;
import com.mapr.fs.proto.Common.FileSubType;
import com.mapr.fs.proto.Common.FSKeyType;
import com.mapr.fs.proto.Common.Server;
import com.mapr.fs.proto.Common.MapRProgramId;
import com.mapr.fs.proto.Common.FileCompressionType;
import com.mapr.fs.proto.Common.IPAddress;
import com.mapr.fs.proto.Common.StoragePoolInfo;
import com.mapr.fs.proto.Common.AlarmMsg;
import com.mapr.fs.proto.Common.VolumeAces;
import com.mapr.fs.proto.Common.VolumeAceEntry;
import com.mapr.fs.proto.Dbserver;
import com.mapr.fs.proto.Common.FileACE;
import com.mapr.fs.proto.Common.FileACEs;
import com.mapr.fs.proto.Common.FSAccessType;
import com.mapr.fs.proto.Fileserver.DiskXAttr;
import com.mapr.fs.proto.Fileserver.InlineXAttrs;
import com.mapr.fs.proto.Fileserver.DeleteAsyncEntry;
import com.mapr.fs.proto.Fileserver.KvMsg;
import com.mapr.fs.proto.Fileserver.KvList;
import com.mapr.fs.proto.Fileserver.KvStoreKey;
import com.mapr.fs.proto.Fileserver.GetattrRequest;
import com.mapr.fs.proto.Fileserver.GetattrResponse;
import com.mapr.fs.proto.Fileserver.ScanFileClustersRequest;
import com.mapr.fs.proto.Fileserver.ScanFileClustersResponse ;
import com.mapr.fs.proto.Fileserver.TabletRangeCheckRequest;
import com.mapr.fs.proto.Fileserver.TabletRangeCheckResponse;
import com.mapr.fs.proto.Fileserver.TabletRangeCheckResponse;
import com.mapr.fs.proto.Fileserver.KvStoreMultiOpDBRequest;
import com.mapr.fs.proto.Fileserver.KvStoreMultiOpDBResponse;
import com.mapr.fs.proto.Fileserver.FSProg;
import com.mapr.fs.proto.Fileserver.XAttrValue;
import com.mapr.fs.proto.Dbserver.DBInternalDefaults;
import com.mapr.fs.proto.Dbserver.SplitDesc;
import com.mapr.fs.proto.Dbserver.TabletMapEntry;
import com.mapr.fs.proto.Dbserver.SegmentMapEntry;
import com.mapr.fs.proto.Dbserver.SpillMapEntry;
import com.mapr.fs.proto.Dbserver.TimeRange;
import com.mapr.fs.proto.Dbserver.SpillKeyEntry;
import com.mapr.fs.proto.Dbserver.PartitionMapEntry;
import com.mapr.fs.proto.Dbserver.PartitionMapEntry.BucketDesc;
import com.mapr.fs.proto.Dbserver.TabletCraftRequest;
import com.mapr.fs.proto.Dbserver.TabletCraftResponse;
import com.mapr.fs.proto.Dbserver.TestScanRequest;
import com.mapr.fs.proto.Dbserver.TestScanResponse;
import com.mapr.fs.proto.Dbserver.RawSpillScanRequest;
import com.mapr.fs.proto.Dbserver.RawSpillScanResponse;
import com.mapr.fs.proto.Dbserver.KeyMapEntry;
import com.mapr.fs.proto.Dbserver.KeyMapCookie;
import com.mapr.fs.proto.Dbserver.SchemaFamily;
import com.mapr.fs.proto.Dbserver.TableAttr;
import com.mapr.fs.proto.Dbserver.TableReplicaDesc;
import com.mapr.fs.proto.Dbserver.Qualifier;
import com.mapr.fs.proto.Dbserver.DBProg;
import com.mapr.fs.proto.Dbserver.TabletStatRequest;
import com.mapr.fs.proto.Dbserver.TabletStatResponse;
import com.mapr.fs.proto.Dbserver.GetPartitionSplitsRequest;
import com.mapr.fs.proto.Dbserver.GetPartitionSplitsResponse;
import com.mapr.fs.proto.Dbserver.SpaceUsage;
import com.mapr.fs.proto.Dbserver.TabletDesc;
import com.mapr.fs.proto.Dbserver.TimeRange;
import com.mapr.fs.proto.Dbserver.MergeDesc;
import com.mapr.fs.proto.Dbserver.CidVNEntry;
import com.mapr.fs.proto.Dbserver.ReplBucketDesc;
import com.mapr.fs.proto.Dbreplicator.DBReplicatorProg;
import com.mapr.fs.proto.Dbreplicator.GetHostNamesRequest;
import com.mapr.fs.proto.Dbreplicator.GetHostNamesResponse;
import com.mapr.fs.proto.Marlincommon.MarlinTableAttr;

import com.mapr.fs.proto.Common.FidMsg;
import com.mapr.fs.MapRFileSystem;
import com.mapr.fs.MapRFileStatus;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.BinaryString;
import org.apache.log4j.Logger;
import com.mapr.security.MaprSecurityException;

import com.mapr.fs.cldb.table.TableUtils;
import com.mapr.cli.table.TabletStats;

public class FidCommands extends CLIBaseClass implements CLIInterface {
  private static final String FID_COMMAND_FID_PARAM_NAME = "fid";
  private static final String FID_COMMAND_DIR_PARAM_NAME = "dirraw";
  private static final String FID_COMMAND_KVTYPE_PARAM_NAME = "kvtype";
  private static final String FID_COMMAND_HOST_PARAM_NAME = "host";
  private static final String FID_COMMAND_PORT_PARAM_NAME = "port";
  private static final String FID_COMMAND_TABLE_PARAM_NAME = "path";
  private static final String FID_COMMAND_CID_PARAM_NAME = "cid";
  private static final String FID_COMMAND_FTYPE_PARAM_NAME = "ftype";
  private static final String FID_COMMAND_DUMPFULLKEYS_PARAM_NAME = "dumpfullkeys";
  private static final String FID_COMMAND_KEYIDXVERSION_PARAM_NAME = "keyidxversion";

  private static final String FID_COMMAND_FID_PARAM_DESC = "fid";
  private static final String FID_COMMAND_DIR_PARAM_DESC = "scan directory inode as kvstore";
  private static final String FID_COMMAND_KVTYPE_PARAM_DESC =
    "cldb kvtype: cinfo|csize|cmap|fsprop|spprop|vprop|sinfo";
  private static final String FID_COMMAND_HOST_PARAM_DESC = "IP/hostname";
  private static final String FID_COMMAND_PORT_PARAM_DESC = "port";
  private static final String FID_COMMAND_TABLE_PARAM_DESC = "table path";
  private static final String FID_COMMAND_CID_PARAM_DESC = "container id";
  private static final String FID_COMMAND_FTYPE_PARAM_DESC =
    "table|tabletmap|tablet|segmap|spillmap";
  private static final String FID_COMMAND_DUMPFULLKEYS_PARAM_DESC = "dump full keys <true|false>";
  private static final String FID_COMMAND_KEYIDXVERSION_PARAM_DESC = "key index version";


  public static final String MULTI_ARG_SEP = ",";
  public static final String COLUMN_SEP = ":";
  private static final Logger LOG = Logger.getLogger(DumpCommands.class);
  private static Pattern printableStringName =
                Pattern.compile("\\p{Print}+");
  // Usage
  static String statUsage = "stat -fid fidx";
  static String dumpUsage = "dump -fid fid";

  private static final int DefaultGatewayPort = 7660;

  public static enum KVFormatType {
    TABLE,
    TABLET_MAP,
    SCHEMA_INFO,
    TABLET,
    SEGMENT_MAP,
    SPILL_MAP,
    DEFER_MAP,
    GENERIC_KV,
    DIR,
    XATTR,
    CLDB_CONTAINER_INFO,
    CLDB_CONTAINER_SIZE_INFO,
    CLDB_CID_MAP,
    CLDB_FS_PROP,
    CLDB_SP_PROP,
    CLDB_VOLUME_NAME,
    CLDB_VOLUME_PROP,
    CLDB_VOLUME_ACES,
    CLDB_VOLUME_QUOTA,
    CLDB_SNAPSHOT_INFO,
    CLDB_SP_MAP,
    MFS_SNAP_MAP,
    INVALID;
  }

  public class NameEnt {
    Long cid;
    Long cinum;
    Long uniq;
    int ftype;
    String name;
    int nmlen;
  }

  public static Map<String, BaseInputParameter> baseParams =
    new ImmutableMap.Builder<String, BaseInputParameter>()
    .put(MapRCliUtil.CLUSTER_NAME_PARAM, new TextInputParameter(
        MapRCliUtil.CLUSTER_NAME_PARAM, "cluster_name",
        CLIBaseClass.NOT_REQUIRED, null))
    .build();

  // Cmds
  static final CLICommand dump = new CLICommand(
      "dump",
      "Usage : " + dumpUsage,
      FidCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(FidCommands.baseParams)
        .put(FidCommands.FID_COMMAND_FID_PARAM_NAME,
             new TextInputParameter(
                 FidCommands.FID_COMMAND_FID_PARAM_NAME,
                 FidCommands.FID_COMMAND_FID_PARAM_DESC,
                 CLIBaseClass.REQUIRED,
                 null))
        .put(FidCommands.FID_COMMAND_KVTYPE_PARAM_NAME,
             new TextInputParameter(
                 FidCommands.FID_COMMAND_KVTYPE_PARAM_NAME,
                 FidCommands.FID_COMMAND_KVTYPE_PARAM_DESC,
                 CLIBaseClass.NOT_REQUIRED,
                 null))
        .put(FidCommands.FID_COMMAND_DIR_PARAM_NAME,
             new BooleanInputParameter(
                 FidCommands.FID_COMMAND_DIR_PARAM_NAME,
                 FidCommands.FID_COMMAND_DIR_PARAM_DESC,
                 CLIBaseClass.NOT_REQUIRED,
                 false))
        .build(),
       null
    ).setShortUsage(dumpUsage);

  static final CLICommand stat = new CLICommand(
      "stat",
      "usage : " + statUsage,
      FidCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(FidCommands.baseParams)
        .put(FidCommands.FID_COMMAND_FID_PARAM_NAME,
             new TextInputParameter(
                 FidCommands.FID_COMMAND_FID_PARAM_NAME,
                 FidCommands.FID_COMMAND_FID_PARAM_DESC,
                 CLIBaseClass.REQUIRED,
                 null))
        .build(),
       null
    ).setShortUsage(statUsage);

  public FidCommands(ProcessedInput input, CLICommand cliCommand)
      throws CLIProcessingException {
    super(input, cliCommand);
  }

  String clusterName = null;

  void init() throws CLIProcessingException {
    try {
      int port = Rpc.initialize(0, 0, null); // Client
      if (port < 0)
        throw new IOException("Failed to initalize RPC");
    } catch (Exception e) {
        /**
        * <MAPR_ERROR>
        * Message:Exception in Rpc.initialize <error>
        * Function:FidCommands.init()
        * Meaning:An error occurred: failed to Initialze RPC.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        LOG.error("Exception in Rpc.initialize " + e);
    }
  }

  public long getLong(byte [] b) {
    long number = 0;
    int len = b.length;

    while (len > 0) {
      number = number << 8;
      number = number + (b[len - 1] & 0xff) ;
      len--;
    }

    return number;
  }

  public NameEnt getNameEnt(ByteString bs, int offset) {
    NameEnt ent = new NameEnt();

    ent.cid = getLong(bs.substring(offset, offset + 4).toByteArray());
    ent.cinum = getLong(bs.substring(offset + 4, offset + 8).toByteArray());
    ent.uniq = getLong(bs.substring(offset + 8, offset + 12).toByteArray());
    ent.nmlen = (int)getLong(bs.substring(offset + 12, offset + 14).toByteArray());
    ent.name = new String(bs.substring(offset + 16, offset + 15 + ent.nmlen).toByteArray());

    byte b = bs.byteAt(offset + 15);
    ent.ftype = b & 0x1F;

    return ent;
  }

  public CommandOutput executeRealCommand() throws CLIProcessingException {
    init();
    if (!super.validateInput()) {
      return output;
    }

    if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
      clusterName = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0);
    } else {
      clusterName = CLDBRpcCommonUtils.getInstance().getCurrentClusterName();
    }

    try {
      if (cliCommand.getCommandName().equalsIgnoreCase("dump")) {
        return dump();
      } else if (cliCommand.getCommandName().equalsIgnoreCase("stat")) {
        return stat();
      } else {
        return new TextCommandOutput("Fid unknown sub-command".getBytes());
      }
    } catch (Exception e) {
      throw new CLIProcessingException("Send request Exception", e);
    }
  }

  // Convert string to fidmsg.
  private FidMsg stringToFid(String s) {
    String[] temp = s.split("\\.");

    if (temp.length != 3)
      return null;

    int cid = Integer.parseInt(temp[0]);
    int cinum = Integer.parseInt(temp[1]);
    int uniq = Integer.parseInt(temp[2]);
    return FidMsg.newBuilder()
                 .setCid(cid)
                 .setCinum(cinum)
                 .setUniq(uniq)
                 .build();
  }

  private CommandOutput getDumpFromFid(FidMsg fid, boolean dirraw,
    KVFormatType kvtype, OutputHierarchy out) throws CLIProcessingException {

    String cluster = null;
    if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
      cluster = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0);
    }

    KVFormatType ftype = KVFormatType.INVALID;
    if (isParamPresent(FID_COMMAND_FTYPE_PARAM_NAME)) {
      String ftypestr = getParamTextValue(FID_COMMAND_FTYPE_PARAM_NAME, 0);
      if (ftypestr.equalsIgnoreCase("table")) {
        ftype = KVFormatType.TABLE;
      } else if (ftypestr.equalsIgnoreCase("tabletmap")) {
        ftype = KVFormatType.TABLET_MAP;
      } else if (ftypestr.equalsIgnoreCase("tablet")) {
        ftype = KVFormatType.TABLET;
      } else if (ftypestr.equalsIgnoreCase("segmap")) {
        ftype = KVFormatType.SEGMENT_MAP;
      } else if (ftypestr.equalsIgnoreCase("spillmap")) {
        ftype = KVFormatType.SPILL_MAP;
      } else if (ftypestr.equalsIgnoreCase("dir")) {
        ftype = KVFormatType.DIR;
      } else if (ftypestr.equalsIgnoreCase("defermap")) {
        ftype = KVFormatType.DEFER_MAP;
      }
      if (ftype == KVFormatType.INVALID) {
        out.addError(new OutputError(Errno.EINVAL,"Invalid ftype " + ftypestr));
        output.setOutput(out);
        return output;
      }
    } else {
      // lookup container in cldb
      long dbinding = getBindingForContainer(fid.getCid());
      GetattrResponse resp;

      if (dbinding == -1) {
        out.addError(new OutputError(Errno.ERPCFAILED,
                                     "container lookup failed"));
        return output;
      }

      // Send GetAttr to mfs
      GetattrRequest req = GetattrRequest.newBuilder()
                                         .setNode(fid)
                                         .setCreds(getUserCredentials())
                                         .build();
      try {
        byte[] replyData;
        replyData = Rpc.sendRequest(dbinding,
                                  MapRProgramId.FileServerProgramId.getNumber(),
                                  FSProg.GetattrProc.getNumber(),
                                  req);
        if (replyData == null) {
          out.addError(new OutputError(Errno.ERPCFAILED, "getattr rpc failed"));
          return output;
        }

        resp = GetattrResponse.parseFrom(replyData);
        if (resp.getStatus() != 0) {
          out.addError(new OutputError(resp.getStatus(),
                       "GetAttr failed, Error : " +
                       Errno.toString(resp.getStatus())));
          return output;
        }
      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);
      } catch (Exception e) {
        LOG.error("Exception processing stat command");
        out.addError(new OutputError(Errno.ERPCFAILED, "mfs rpc failed"));
        return output;
      }

      FileType itype = resp.getAttr().getType();
      FileSubType subtype = resp.getAttr().getSubtype();
      if (itype == FileType.FTDirectory) {
        ftype = KVFormatType.DIR;
      } else if (itype == FileType.FTRegular) {
        out.addError(new OutputError(Errno.EINVAL,
                                     "Cannot dump files of itype " + itype +
                                     " subtype " + subtype));
      } else if (itype == FileType.FTKvstore) {
        if (subtype == FileSubType.FSTKvTable) {
          ftype = KVFormatType.TABLE;
        } else if (subtype == FileSubType.FSTKvTabletMap) {
          ftype = KVFormatType.TABLET_MAP;
        } else if (subtype == FileSubType.FSTKvSchema) {
          ftype = KVFormatType.SCHEMA_INFO;
        } else if (subtype == FileSubType.FSTKvTablet) {
          ftype = KVFormatType.TABLET;
        } else if (subtype == FileSubType.FSTKvSegMap) {
          ftype = KVFormatType.SEGMENT_MAP;
        } else if (subtype == FileSubType.FSTKvSpillMap) {
          ftype = KVFormatType.SPILL_MAP;
        } else if (subtype == FileSubType.FSTKvXattr) {
          ftype = KVFormatType.XATTR;
        } else if (fid.getCinum() == 22) {
          ftype = KVFormatType.DEFER_MAP;
        } else if (kvtype != null) {
          ftype = kvtype;
        } else {
          ftype = KVFormatType.GENERIC_KV;
        }
      } else {
        out.addError(new OutputError(Errno.EINVAL, "Cannot dump files of itype "
                                     + itype));
        return output;
      }
    }

    KvStoreKey kstart = null;
    KvStoreKey kend = null;
    int maxkeys = 0;
    int idxoffset = 0;
    int idxsize = 0;

    if (ftype == KVFormatType.DIR && !dirraw) {
      dumpDir(fid, out);
    } else {
      dumpKeyValues(fid, ftype, kstart, kend, maxkeys, out);
    }
    return output;

  }

  private CommandOutput dump() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    String fidstr = getParamTextValue(FID_COMMAND_FID_PARAM_NAME, 0);
    FidMsg fid = stringToFid(fidstr);
    if (fid == null) {
      out.addError(new OutputError(Errno.EINVAL, "Invalid fid " + fidstr));
      output.setOutput(out);
      return output;
    }

    boolean dirraw = false;
    String kvtypeStr = null;
    KVFormatType kvtype = null;
    if (isParamPresent(FID_COMMAND_DIR_PARAM_NAME)) {
      dirraw = getParamBooleanValue(FID_COMMAND_DIR_PARAM_NAME, 0);
      kvtype = KVFormatType.GENERIC_KV;
    }

    if (isParamPresent(FID_COMMAND_KVTYPE_PARAM_NAME)) {
      kvtypeStr = getParamTextValue(FID_COMMAND_KVTYPE_PARAM_NAME, 0);
      if (kvtypeStr.equals("cinfo")) {
        kvtype = KVFormatType.CLDB_CONTAINER_INFO;
      } else if (kvtypeStr.equals("csize")) {
        kvtype = KVFormatType.CLDB_CONTAINER_SIZE_INFO;
      } else if (kvtypeStr.equals("cmap")) {
        kvtype = KVFormatType.CLDB_CID_MAP;
      } else if (kvtypeStr.equals("fsprop")) {
        kvtype = KVFormatType.CLDB_FS_PROP;
      } else if (kvtypeStr.equals("spprop")) {
        kvtype = KVFormatType.CLDB_SP_PROP;
      } else if (kvtypeStr.equals("sinfo")) {
        kvtype = KVFormatType.CLDB_SNAPSHOT_INFO;
      } else if (kvtypeStr.equals("vname")) {
        kvtype = KVFormatType.CLDB_VOLUME_NAME;
      } else if (kvtypeStr.equals("vprop")) {
        kvtype = KVFormatType.CLDB_VOLUME_PROP;
      } else if (kvtypeStr.equals("vaces")) {
        kvtype = KVFormatType.CLDB_VOLUME_ACES;
      } else if (kvtypeStr.equals("vquota")) {
        kvtype = KVFormatType.CLDB_VOLUME_QUOTA;
      } else if (kvtypeStr.equals("smap")) {
        kvtype = KVFormatType.MFS_SNAP_MAP;
      } else if (kvtypeStr.equals("spmap")) {
        kvtype = KVFormatType.CLDB_SP_MAP;
      }
    }

    getDumpFromFid(fid, dirraw, kvtype, out);

    return output;
  }

  private CommandOutput stat()  throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    String fidstr = getParamTextValue(FID_COMMAND_FID_PARAM_NAME, 0);
    FidMsg fid = stringToFid(fidstr);
    if (fid == null) {
      out.addError(new OutputError(Errno.EINVAL, "Invalid fid " + fidstr));
      output.setOutput(out);
      return output;
    }

    // lookup container in cldb
    long dbinding = getBindingForContainer(fid.getCid());
    if (dbinding == -1) {
      out.addError(new OutputError(Errno.ERPCFAILED,"container lookup failed"));
      return output;
    }

    // Send GetAttr to mfs
    GetattrRequest req = GetattrRequest.newBuilder()
                                       .setNode(fid)
                                       .setCreds(getUserCredentials())
                                       .build();
    try {
      byte[] replyData;
      replyData = Rpc.sendRequest(dbinding,
                                  MapRProgramId.FileServerProgramId.getNumber(),
                                  FSProg.GetattrProc.getNumber(),
                                  req);
      if (replyData == null) {
        LOG.error("Got null reply from RPC");
        out.addError(new OutputError(Errno.ERPCFAILED, "getattr rpc failed"));
        return output;
      }

      GetattrResponse resp = GetattrResponse.parseFrom(replyData);
      OutputNode dout = new OutputNode();
      if (resp.getStatus() == 0) {
        dout.addChild(new OutputNode("type",
                                     resp.getAttr().getType().toString()));

        String subtype = printableSubType(resp.getAttr().getType(),
                                          resp.getAttr().getSubtype());
        dout.addChild(new OutputNode("subtype", subtype));
        dout.addChild(new OutputNode("parent",
                                     printableFid(resp.getAttr().getParent())));
        dout.addChild(new OutputNode("size", resp.getAttr().getSize()));
        dout.addChild(new OutputNode("nblocks", resp.getAttr().getNblocks()));
        String cs = "off";
        if (resp.getAttr().getCanCompress())
          cs = getCompressionType(FileCompressionType.valueOf(resp.getAttr().getCompressorType()));

        dout.addChild(new OutputNode("compression", cs));
        dout.addChild(new OutputNode("deleteFlags",
                                     resp.getAttr().getDeleteFlags()));
        dout.addChild(new OutputNode("atime",
                                     resp.getAttr().getAtime().getSec()));
        dout.addChild(new OutputNode("mtime",
                                     resp.getAttr().getMtime().getSec()));
        int mode = resp.getAttr().getMode();
        dout.addChild(new OutputNode("mode",
                                     Integer.toOctalString(mode)));
        dout.addChild(new OutputNode("uid", resp.getAttr().getUid()));
        dout.addChild(new OutputNode("gid", resp.getAttr().getGid()));
        dout.addChild(new OutputNode("nlink", resp.getAttr().getNlink()));
        dout.addChild(new OutputNode("xattrInum",
                                     resp.getAttr().getXattrInum()));
        if (resp.getAttr().hasVersion())
          dout.addChild(new OutputNode("version",
                                       resp.getAttr().getVersion()));
        dout.addChild(new OutputNode("networkencryption",
                                     resp.getAttr().getWireSecurityEnabled()));
        if (resp.getAttr().hasXattrs()) {
          dout.addChild(new OutputNode("xattrs",
                                           dumpInlineXAttrs(resp.getAttr().getXattrs())));
        }

        if (resp.getAttr().hasNBtreeLevels()) {
            dout.addChild(new OutputNode("nlevels", resp.getAttr().getNBtreeLevels()));
        }

        out.addNode(dout);
      } else {
        out.addError(new OutputError(resp.getStatus(),
                     "GetAttr failed, Error : " +
                     Errno.toString(resp.getStatus())));
      }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      LOG.error("Exception processing stat command");
      out.addError(new OutputError(Errno.ERPCFAILED, "db rpc failed"));
    }
    output.setOutput(out);
    return output;
  }

  private CommandOutput getHostNames()  throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    String ipStr = getParamTextValue(FID_COMMAND_HOST_PARAM_NAME, 0);
    IPAddress gatewayIp = buildIPFromString(ipStr);
    if (gatewayIp == null) {
      out.addError(new OutputError(Errno.EINVAL, "Invalid gateway host " + ipStr));
      output.setOutput(out);
      return output;
    }

    int gatewayPort = getParamIntValue(FID_COMMAND_PORT_PARAM_NAME, 0);
    long dbinding = getBindingForGateway(gatewayIp.getHost(), gatewayPort);
    if (dbinding == -1) {
      out.addError(new OutputError(Errno.ERPCFAILED,"Gateway connect failed"));
      return output;
    }

    // Send GetAttr to mfs
    GetHostNamesRequest req = GetHostNamesRequest.newBuilder()
                                       .setCreds(getUserCredentials())
                                       .build();
    try {
      byte[] replyData;
      replyData = Rpc.sendRequest(dbinding,
                                  MapRProgramId.DBReplicatorServerProgramId.getNumber(),
                                  DBReplicatorProg.GetHostNamesProc.getNumber(),
                                  req);
      if (replyData == null) {
        LOG.error("Got null reply from RPC");
        out.addError(new OutputError(Errno.ERPCFAILED, "gethostnames rpc failed"));
        return output;
      }

      GetHostNamesResponse resp = GetHostNamesResponse.parseFrom(replyData);
      OutputNode dout = new OutputNode();
      if (resp.getStatus() == 0) {
        for (String name : resp.getNamesList()) {
          dout.addChild(new OutputNode("HostNames", name));
        }
        out.addNode(dout);
      } else {
        out.addError(new OutputError(resp.getStatus(),
                     "GetHostNames failed, Error : " +
                     Errno.toString(resp.getStatus())));
      }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      LOG.error("Exception processing getHostNames command");
      out.addError(new OutputError(Errno.ERPCFAILED, "db rpc failed"));
    }
    output.setOutput(out);
    return output;
  }

  private long getBindingForContainer(int cid) throws CLIProcessingException {
    // lookup container in cldb
    int dbHost, dbPort;
    byte[] replyData;
    ContainerLookupRequest creq = ContainerLookupRequest.newBuilder()
                                    .addContainerId(cid)
                                    .setCreds(getUserCredentials())
                                    .build();
    try {
      replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
                    MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.ContainerLookupProc.getNumber(),
                    creq,
                    ContainerLookupResponse.class);

      if (replyData == null) {
        // most likely can not connect to CLDB
        LOG.error("Couldn't connect to the CLDB service");
        return -1;
      }

      ContainerLookupResponse resp;
      resp = ContainerLookupResponse.parseFrom(replyData);
      if (resp.getStatus() == 0) {
        Server server = resp.getContainers(0).getMServer();
        dbHost = server.getIps(0).getHost();
        dbPort = server.getIps(0).getPort();
      } else {
        LOG.error("Container lookup failed : Error " + Errno.toString(resp.getStatus()));
        return -1;
      }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      LOG.error("Container lookup failed");
      return -1;
    }

    return Rpc.createBindingFor(dbHost, dbPort, clusterName, ServerKeyType.ServerKey.getNumber());
  }

  // constructs a "pretty printed binary string" to its raw format.
  private ByteString binaryKeyToByteString(String raws) {
    byte[] bArr = BinaryString.toBytesBinary(raws);
    return ByteString.copyFrom(bArr);
  }

  private ByteString prepareVarKey(String raws) {
    return binaryKeyToByteString(raws);
  }

  private OutputNode formatDirEntry(MapRFileStatus d) {
    String childFid = d.getCid() + "." + d.getCinum() + "." + d.getUniq();
    String s = d.getPath().toString();
    int l = s.lastIndexOf('/');

    OutputNode dout = new OutputNode();
    dout.addChild(new OutputNode("name", s.substring(l+1)));
    dout.addChild(new OutputNode("fid", childFid));
    dout.addChild(new OutputNode("isDir", d.isDir() ? "true" : "false"));
    return dout;
  }

  private OutputNode formatKeyMapEntry(KeyMapEntry ent) {
    OutputNode dout = new OutputNode();
    ByteString bs = null;

    dout.addChild(new OutputNode("shared", ent.getSharedLen()));
    dout.addChild(new OutputNode("nonshared", ent.getNonsharedLen()));
    dout.addChild(new OutputNode("value", ent.getValueLen()));

    if (ent.hasBlkOff()) {
      dout.addChild(new OutputNode("childblockoffset", ent.getBlkOff()));
    }
    if (ent.hasBlkSize()) {
      dout.addChild(new OutputNode("childblocksize", ent.getBlkSize()));
    }

    if (!ent.hasBlkOff() && !ent.hasBlkSize()) {
      try {
        SpillKeyEntry spillVal;

        if (ent.hasKvalue())
          spillVal = ent.getKvalue();
        else
          spillVal = SpillKeyEntry.parseFrom(ent.getValue());

        if (spillVal.hasInlineValue()) {
          dout.addChild(new OutputNode("inlineValue",
                        printableKey(spillVal.getInlineValue().toByteArray())));
          dout.addChild(new OutputNode("inlineValueSz",
                        spillVal.getInlineValue().size()));
        }

        int cnt = spillVal.getValuesCount();
        for (int index = 0; index < cnt; ++index) {
          OutputNode valueout = new OutputNode("familyData");
          boolean addFamily = false;

          SpillKeyEntry.FamilyValue fv = spillVal.getValues(index);
          if (fv.hasId()) {
            valueout.addChild(new OutputNode("id", fv.getId()));
            addFamily = true;
          }
          if (fv.hasOffset()) {
            valueout.addChild(new OutputNode("offset", fv.getOffset()));
            addFamily = true;
          }
          if (fv.hasLength()) {
            valueout.addChild(new OutputNode("length", fv.getLength()));
            addFamily = true;
          }
          if (fv.hasInlined()) {
            valueout.addChild(new OutputNode("inlined", fv.getInlined()));
            addFamily = true;
          }

          if (addFamily )
            dout.addChild(valueout);
        }
      } catch (Throwable e) {
        dout.addChild(new OutputNode("error", "PARSE ERROR"));
      }
    }

    dout.addChild(new OutputNode("key",
                  printableKey(ent.getKey().toByteArray())));
    dout.addChild(new OutputNode("value",
                  printableKey(ent.getValue().toByteArray())));

    return dout;
  }

  private static final Map<FSAccessType, String> fsAccessTypeMap =
      new ImmutableMap.Builder<FSAccessType, String>()
      .put(FSAccessType.AceRead, new String("readfile"))
      .put(FSAccessType.AceWrite, new String("writefile"))
      .put(FSAccessType.AceExecute, new String("executefile"))
      .put(FSAccessType.AceReadDir, new String("readdir"))
      .put(FSAccessType.AceAddChild, new String("addchild"))
      .put(FSAccessType.AceDeleteChild, new String("deletechild"))
      .put(FSAccessType.AceLookupDir, new String("lookupdir"))
      .build();

  private OutputNode dumpFileACEs(ByteString xval) {
    OutputNode aceOut = new OutputNode("ace");
    try {
      FileACEs face = FileACEs.parseFrom(xval.toByteArray());
      if (face.hasNoInherit()) {
        aceOut.addChild(new OutputNode("noInherit", face.getNoInherit()));
      }
      if (face.hasPreserveModeBits()) {
        aceOut.addChild(new OutputNode("preserveModeBits", face.getPreserveModeBits()));
      }
      for (FileACE a : face.getAcesList()) {
        aceOut.addChild(new OutputNode(fsAccessTypeMap.get(a.getAccessType()), AceHelper.toInfix(a.getBoolExp().toStringUtf8())));
      }
    } catch (InvalidProtocolBufferException e) {
      LOG.error("Exception processing dumpFileACEs InvalidProtocolBufferException");
    } catch (IOException e) {
      LOG.error("Exception processing dumpFileACEs IOException");
    }
    return aceOut;
  }

  private OutputNode dumpInlineXAttrs(ByteString xattrs) {
    OutputNode xout = new OutputNode("xattrs");
    try {
      InlineXAttrs ix = InlineXAttrs.parseFrom(xattrs.toByteArray());
      for (DiskXAttr dx : ix.getXattrsList()) {
        xout.addChild(new OutputNode(dx.getName(), dx.getXval().getValue().toStringUtf8()));
      }
      if (ix.getXattrsCount() == 1) {
        ByteString xval = ix.getXattrs(0).getXval().getValue();
        if (ix.getXattrs(0).getName().charAt(0) == 'A') {
          OutputNode aceOut = dumpFileACEs(xval);
          xout.addChild(aceOut);
        }
      }
    } catch (InvalidProtocolBufferException e) {
      LOG.error("Exception processing dumpInlineXAttrs InvalidProtocolBufferException");
    }
    return xout;
  }

  private void dumpIPAddress(String val, IPAddress ip, OutputNode out)
  {
    OutputNode ipOut = new OutputNode(val);
    out.addChild(ipOut);
    if (ip.hasHost()) {
      ipOut.addChild(new OutputNode("ip",
                                    Util.intToIp(ip.getHost()) + ":" + ip.getPort()));
      ipOut.addChild(new OutputNode("host", ip.getHost()));
    }
    if (ip.hasPort())
      ipOut.addChild(new OutputNode("port", ip.getPort()));
    if (ip.hasSyncReplica())
      ipOut.addChild(new OutputNode("syncReplica", ip.getSyncReplica()));
    if (ip.hasEpoch())
      ipOut.addChild(new OutputNode("epoc", ip.getEpoch()));
    if (ip.hasHostname())
      ipOut.addChild(new OutputNode("hostname", ip.getHostname()));
    if (ip.hasState())
      ipOut.addChild(new OutputNode("state", ip.getState()));
    if (ip.hasResync())
      ipOut.addChild(new OutputNode("resync", ip.getResync()));
    if (ip.hasVirtualIP())
      ipOut.addChild(new OutputNode("virtualip", ip.getVirtualIP()));
  }

  private void dumpServer(String val, Server s, OutputNode out)
  {
    OutputNode sout = new OutputNode(val);
    out.addChild(sout);
    if (s.hasServerId())
      sout.addChild(new OutputNode("serverId", s.getServerId()));
    if (s.hasSyncReplica())
      sout.addChild(new OutputNode("syncReplica", s.getSyncReplica()));
    if (s.hasEpoch())
      sout.addChild(new OutputNode("epoch", s.getEpoch()));
    if (s.hasState())
      sout.addChild(new OutputNode("state", s.getState()));
    if (s.hasResync())
      sout.addChild(new OutputNode("resync", s.getResync()));
    if (s.hasHostname())
      sout.addChild(new OutputNode("hostname", s.getHostname()));
    if (s.hasFixedByFsck())
      sout.addChild(new OutputNode("fixedByFsck", s.getFixedByFsck()));
    if (s.hasChosenSp())
      sout.addChild(new OutputNode("chosensp", s.getChosenSp()));
    if (s.hasTopology())
      sout.addChild(new OutputNode("topology", s.getTopology()));
    if (s.hasDeleteAfterRepl())
      sout.addChild(new OutputNode("deleteafterrepl", s.getDeleteAfterRepl()));
    if (s.hasPliId())
      sout.addChild(new OutputNode("pliid", s.getPliId()));

    if (s.hasSpInfo()) {
      StoragePoolInfo sp = s.getSpInfo();
      OutputNode si = new OutputNode("spInfo: ");
      sout.addChild(si);
      if (sp.hasSpId())
        si.addChild(new OutputNode("spId", sp.getSpId()));
      if (sp.hasClusterUuid()) {
        si.addChild(new OutputNode("clusterUuid H", sp.getClusterUuid().getId640()));
        si.addChild(new OutputNode("clusterUuid L", sp.getClusterUuid().getId641()));
      }
      if (sp.hasCapacitySizeMB())
        si.addChild(new OutputNode("capacitysizemb",
                                    sp.getCapacitySizeMB()));
      if (sp.hasUsedSizeMB())
        si.addChild(new OutputNode("usedsizemb", sp.getUsedSizeMB()));
      if (sp.hasAvailableSizeMB())
        si.addChild(new OutputNode("availablesizemb",
                                    sp.getAvailableSizeMB()));
      if (sp.hasOnlineCount())
        si.addChild(new OutputNode("onlineCount", sp.getOnlineCount()));
    }

    if (s.getSecondaryPortsCount() != 0) {
      int k = 0;
      for (int port :  s.getSecondaryPortsList()) {
        sout.addChild(new OutputNode("secPort" + k, port));
        ++k;
      }
    }

    if (s.getIpsCount() != 0) {
      int k = 0;
      for (IPAddress ip : s.getIpsList()) {
        dumpIPAddress("ip" + k, ip, sout);
        ++k;
      }
    }
  }

  private void dumpServers(String stypeStr, List<Server> slist, OutputNode out) {
    OutputNode ssOut = new OutputNode(stypeStr);
    out.addChild(ssOut);
    int i = 0;
    for(Server s : slist) {
      dumpServer("Server" + i, s, ssOut);
    }
  }

  private void dumpContainerInfo(ContainerInfo ci, OutputNode out)
  {
    if (ci.hasVolumeId())
      out.addChild(new OutputNode("volid", ci.getVolumeId()));
    if (ci.hasContainerId())
      out.addChild(new OutputNode("cid ", ci.getContainerId()));
    if (ci.hasVolumeId())
      out.addChild(new OutputNode("volId ", ci.getVolumeId()));
    if (ci.hasSnapshotId())
      out.addChild(new OutputNode("snapId ", ci.getSnapshotId()));
    if (ci.hasLatestEpoch())
      out.addChild(new OutputNode("latestEpoch ", ci.getLatestEpoch()));
    if (ci.hasRwContainerId())
      out.addChild(new OutputNode("rwCid ", ci.getRwContainerId()));
    if (ci.hasRwVolumeId())
      out.addChild(new OutputNode("rwVolId ", ci.getRwVolumeId()));
    if (ci.hasNameContainer())
      out.addChild(new OutputNode("isNameContainer ", ci.getNameContainer()));
    if (ci.hasMirrorContainer())
      out.addChild(new OutputNode("mirrorCid ", ci.getMirrorContainer()));
    if (ci.hasType())
      out.addChild(new OutputNode("ReplType ", ci.getType()));
    if (ci.hasLogicalSizeMB())
      out.addChild(new OutputNode("logicalSzMB ", ci.getLogicalSizeMB()));
    if (ci.hasChainSeqNumber())
      out.addChild(new OutputNode("chainSeqNumber ", ci.getChainSeqNumber()));
    if (ci.hasOwnedSizeMB())
      out.addChild(new OutputNode("ownedSzMB ", ci.getOwnedSizeMB()));
    if (ci.hasSharedSizeMB())
      out.addChild(new OutputNode("sharedSzMB ", ci.getSharedSizeMB()));
    if (ci.hasFixedByFsck())
      out.addChild(new OutputNode("isFixedByFsck ", ci.getFixedByFsck()));
    if (ci.hasMtime())
      out.addChild(new OutputNode("mtime ", ci.getMtime()));
    if (ci.hasChainSizeMB())
      out.addChild(new OutputNode("chainSzMB ", ci.getChainSizeMB()));
    if (ci.hasHasForcedMaster())
      out.addChild(new OutputNode("hasForcedMaster ", ci.getHasForcedMaster()));
    if (ci.hasNumInumUsed())
      out.addChild(new OutputNode("numinumused ", ci.getNumInumUsed()));
    if (ci.hasSalt())
      out.addChild(new OutputNode("salt ", ci.getSalt()));
    if (ci.hasCreatorContainerId())
      out.addChild(new OutputNode("creatorCid ", ci.getCreatorContainerId()));
    if (ci.hasCreatorVolumeUuid()) {
      out.addChild(new OutputNode("creatorVolumeId H",
                                  ci.getCreatorVolumeUuid().getId640()));
      out.addChild(new OutputNode("creatorVolumeId L",
                                  ci.getCreatorVolumeUuid().getId641()));
    }
    if (ci.hasUseActualCreatorId())
      out.addChild(new OutputNode("useActualCreatorId ",
                                  ci.getUseActualCreatorId()));
    if (ci.hasSnapshotsOwnedSize())
      out.addChild(new OutputNode("snapOSize ", ci.getSnapshotsOwnedSize()));
    if (ci.hasMServer())
      dumpServer("Master", ci.getMServer(), out);

    if (ci.getAServersCount() != 0)
      dumpServers("active: ", ci.getAServersList(), out);
    if (ci.getIServersCount() != 0)
      dumpServers("inactive: ", ci.getIServersList(), out);
    if (ci.getUServersCount() != 0)
      dumpServers("unused", ci.getUServersList(), out);
    if (ci.getUnreachableServersCount() != 0)
      dumpServers("unreachable", ci.getUnreachableServersList(), out);
  }

  private void dumpContainerInfo(ByteString val, OutputNode out) {
    try {
      ContainerInfo ci = ContainerInfo.parseFrom(val);
      dumpContainerInfo(ci, out);
    } catch (Throwable e) {
      out.addChild(new OutputNode("fid", "UNKNOWN"));
    }
  }

  private void dumpContainerSizeInfo(ByteString val, OutputNode out) {
    try {
      ContainerSizeInfo csi = ContainerSizeInfo.parseFrom(val);
      if (csi.hasContainerId()) {
        out.addChild(new OutputNode("containerId", csi.getContainerId()));
      }
      if (csi.hasVolumeId()) {
        out.addChild(new OutputNode("volumeId", csi.getVolumeId()));
      }
      if (csi.hasSnapshotId()) {
        out.addChild(new OutputNode("snapshotID", csi.getSnapshotId()));
      }
      if (csi.hasLogicalSizeMB()) {
        out.addChild(new OutputNode("logcialSizeMB", csi.getLogicalSizeMB()));
      }
      if (csi.hasOwnedSizeMB()) {
        out.addChild(new OutputNode("ownedSizeMB", csi.getOwnedSizeMB()));
      }
      if (csi.hasSharedSizeMB()) {
        out.addChild(new OutputNode("sharedSizeMB", csi.getSharedSizeMB()));
      }
      if (csi.hasMtime()) {
        out.addChild(new OutputNode("mtime", csi.getMtime()));
      }
      if (csi.hasChainSizeMB()) {
        out.addChild(new OutputNode("chainSizeMB", csi.getChainSizeMB()));
      }
      if (csi.hasNumInumUsed()) {
        out.addChild(new OutputNode("numInodeSize", csi.getNumInumUsed()));
      }
      if (csi.hasSnapshotsOwnedSize()) {
        out.addChild(new OutputNode("snapOwned", csi.getSnapshotsOwnedSize()));
      }
    } catch (Throwable e) {
      out.addChild(new OutputNode("cid", "UNKNOWN"));
    }
  }

  private void dumpSnapshotInfo(SnapshotInfo si, OutputNode out)
  {
    if (si.hasSnapshotId())
      out.addChild(new OutputNode("SnapshotId", si.getSnapshotId()));
    if (si.hasRwVolumeId())
      out.addChild(new OutputNode("RwVolid", si.getRwVolumeId()));
    if (si.hasCostCentreId())
      out.addChild(new OutputNode("CostCentreId", si.getCostCentreId()));
    if (si.hasCreateTime())
      out.addChild(new OutputNode("CreateTime", si.getCreateTime()));
    if (si.hasDeleteTime())
      out.addChild(new OutputNode("DeleteTime", si.getDeleteTime()));
    if (si.hasSnapshotName())
      out.addChild(new OutputNode("SnapshotName", si.getSnapshotName()));
    if (si.hasRootContainerId())
      out.addChild(new OutputNode("RootContainerId", si.getRootContainerId()));
    if (si.hasRwVolumeName())
      out.addChild(new OutputNode("RwVolumeName", si.getRwVolumeName()));
    if (si.hasIsMirrorSnapshot())
      out.addChild(new OutputNode("IsMirrorSnapshot", si.getIsMirrorSnapshot()));
    if (si.hasSnapshotInProgress())
      out.addChild(new OutputNode("SnapshotInProgress", si.getSnapshotInProgress()));
    if (si.hasSnapshotOwnedSizeMB())
      out.addChild(new OutputNode("SnapshotOwnedSizeMB", si.getSnapshotOwnedSizeMB()));
    if (si.hasSnapshotSharedSizeMB())
      out.addChild(new OutputNode("SnapshotSharedSizeMB", si.getSnapshotSharedSizeMB()));
    if (si.hasMountDir())
      out.addChild(new OutputNode("MountDir", si.getMountDir()));
    if (si.hasOwnerId())
      out.addChild(new OutputNode("OwnerId", si.getOwnerId()));
    if (si.hasDataSrcSnapCreateTimeMillis())
      out.addChild(new OutputNode("DataSrcSnapCreateTimeMillis",
                                  si.getDataSrcSnapCreateTimeMillis()));
    if (si.hasVerifier())
      out.addChild(new OutputNode("Verifier", si.getVerifier()));
    if (si.hasSnapshotCumulativeReclaimSizeMB())
      out.addChild(new OutputNode("SnapshotCumulativeReclaimSizeMB",
                                  si.getSnapshotCumulativeReclaimSizeMB()));
    if (si.hasDeleteInProg())
      out.addChild(new OutputNode("DeleteInProg", si.getDeleteInProg()));
  }

  private void dumpSnapshotInfo(ByteString val, OutputNode out)
  {
    try {
      SnapshotInfo si = SnapshotInfo.parseFrom(val);
      dumpSnapshotInfo(si, out);
    } catch (Throwable e) {
      out.addChild(new OutputNode("snapId", "UNKNOWN"));
    }
  }

  private void dumpVolumeProperties(ByteString val, OutputNode out)
  {
    try {
      VolumeProperties vp = VolumeProperties.parseFrom(val);
      if (vp.hasVolumeId())
        out.addChild(new OutputNode("VolumeId", vp.getVolumeId()));
      if (vp.hasVolumeName())
        out.addChild(new OutputNode("VolumeName", vp.getVolumeName()));
      if (vp.hasRootContainerId())
        out.addChild(new OutputNode("RootContainerId", vp.getRootContainerId()));
      if (vp.hasMountDir())
        out.addChild(new OutputNode("MountDir", vp.getMountDir()));
      if (vp.hasVolumeQuotaSizeMB())
        out.addChild(new OutputNode("VolumeQuotaSizeMB", vp.getVolumeQuotaSizeMB()));
      if (vp.hasReadOnly())
        out.addChild(new OutputNode("ReadOnly", vp.getReadOnly()));
      if (vp.hasMounted())
        out.addChild(new OutputNode("Mounted", vp.getMounted()));
      if (vp.hasLocalVolume())
        out.addChild(new OutputNode("LocalVolume", vp.getLocalVolume()));
      if (vp.hasOwnerId())
        out.addChild(new OutputNode("OwnerId", vp.getOwnerId()));
      if (vp.hasIsMirrorVol())
        out.addChild(new OutputNode("IsMirrorVol", vp.getIsMirrorVol()));
      if (vp.hasSchedulingPolicyId())
        out.addChild(new OutputNode("SchedulingPolicyId", vp.getSchedulingPolicyId()));
      if (vp.hasVolumeQuotaAdvisorySizeMB())
        out.addChild(new OutputNode("VolumeQuotaAdvisorySizeMB",
                                    vp.getVolumeQuotaAdvisorySizeMB()));
      if (vp.hasSchedulingPolicyName())
        out.addChild(new OutputNode("SchedulingPolicyName",
                                    vp.getSchedulingPolicyName()));
      if (vp.hasNumContainers())
        out.addChild(new OutputNode("NumContainers", vp.getNumContainers()));
      if (vp.hasNumSnapshots())
        out.addChild(new OutputNode("NumSnapshots", vp.getNumSnapshots()));
      if (vp.hasParentVolumeId())
        out.addChild(new OutputNode("ParentVolumeId", vp.getParentVolumeId()));
      if (vp.hasRootDirPerms())
        out.addChild(new OutputNode("RootDirPerms", vp.getRootDirPerms()));
      if (vp.hasReReplicationTimeOutSec())
        out.addChild(new OutputNode("ReReplicationTimeOutSec",
                                    vp.getReReplicationTimeOutSec()));
      if (vp.hasInGfsck())
        out.addChild(new OutputNode("InGfsck", vp.getInGfsck()));
      if (vp.hasShuffleVolume())
        out.addChild(new OutputNode("ShuffleVolume", vp.getShuffleVolume()));
      if (vp.hasVolumeUUID())
        out.addChild(new OutputNode("VolumeUUID", vp.getVolumeUUID()));
      if (vp.hasDeleteInProg())
        out.addChild(new OutputNode("DeleteInProg", vp.getDeleteInProg()));
      if (vp.hasNeedsGfsck())
        out.addChild(new OutputNode("NeedsGfsck", vp.getNeedsGfsck()));
      if (vp.hasMaxInodesAlarmThreshold())
        out.addChild(new OutputNode("MaxInodesAlarmThreshold",
                                    vp.getMaxInodesAlarmThreshold()));
      if (vp.hasMaxSizeSeenSoFar())
        out.addChild(new OutputNode("MaxSizeSeenSoFar", vp.getMaxSizeSeenSoFar()));
      if (vp.hasNewAclFormat())
        out.addChild(new OutputNode("NewAclFormat", vp.getNewAclFormat()));
      if (vp.hasVolumetype())
        out.addChild(new OutputNode("Volumetype", vp.getVolumetype()));
      if (vp.hasCreatorContainerId())
        out.addChild(new OutputNode("CreatorContainerId",
                                    vp.getCreatorContainerId()));
      if (vp.hasLimitVolumeSpread())
        out.addChild(new OutputNode("LimitVolumeSpread", vp.getLimitVolumeSpread()));
      if (vp.hasMirrorSchedulingPolicyId())
        out.addChild(new OutputNode("MirrorSchedulingPolicyId",
                                    vp.getMirrorSchedulingPolicyId()));
      if (vp.hasMirrorThrottle())
        out.addChild(new OutputNode("MirrorThrottle", vp.getMirrorThrottle()));
      if (vp.hasNumNamespaceReplicas())
        out.addChild(new OutputNode("NumNamespaceReplicas",
                                    vp.getNumNamespaceReplicas()));
    } catch (Throwable e) {
      out.addChild(new OutputNode("volId", "UNKNOWN"));
    }
  }

  private void dumpVolumeQuota(ByteString val, OutputNode out)
  {
    try {
      VolumeQuotaInfo vq = VolumeQuotaInfo.parseFrom(val);
      if (vq.hasCostCenterId())
        out.addChild(new OutputNode("CostCenterId", vq.getCostCenterId()));
      if (vq.hasVolumeQuotaSizeMB())
        out.addChild(new OutputNode("VolumeQuotaSizeMB", vq.getVolumeQuotaSizeMB()));
      if (vq.hasVolumeUsedSizeMB())
        out.addChild(new OutputNode("VolumeUsedSizeMB", vq.getVolumeUsedSizeMB()));
      if (vq.hasVolumeAdvisoryQuotaSizeMB())
        out.addChild(new OutputNode("VolumeAdvisoryQuotaSizeMB", vq.getVolumeAdvisoryQuotaSizeMB()));
      if (vq.hasVolumeLogicalUsedSizeMB())
        out.addChild(new OutputNode("VolumeLogicalUsedSizeMB", vq.getVolumeLogicalUsedSizeMB()));
      if (vq.hasVolumeOwnedSizeMB())
        out.addChild(new OutputNode("VolumeOwnedSizeMB", vq.getVolumeOwnedSizeMB()));
      if (vq.hasVolumeSharedSizeMB())
        out.addChild(new OutputNode("VolumeSharedSizeMB", vq.getVolumeSharedSizeMB()));
    } catch (Throwable e) {
      out.addChild(new OutputNode("volId", "UNKNOWN"));
    }
  }

  private void dumpVolumeAces(ByteString val, OutputNode out)
  {
    try {
      VolumeAces va = VolumeAces.parseFrom(val);

      if (va.getAcesCount() != 0) {
        int k = 0;
        for (VolumeAceEntry vae : va.getAcesList()) {
          OutputNode aeOut = new OutputNode("ace" + k);
          out.addChild(aeOut);
          if (vae.hasExpr())
            aeOut.addChild(new OutputNode("expr", vae.getExpr()));
          if (vae.hasAccessType())
            aeOut.addChild(new OutputNode("accessType", vae.getAccessType()));
          ++k;
        }
      }
    } catch (Throwable e) {
      out.addChild(new OutputNode("volId", "UNKNOWN"));
    }
  }

  private void dumpVolumeTableId(ByteString val, OutputNode out)
  {
    try {
      VolumeTableId vp = VolumeTableId.parseFrom(val);
      if (vp.hasVolumeId())
        out.addChild(new OutputNode("VolumeId", vp.getVolumeId()));
    } catch (Throwable e) {
      out.addChild(new OutputNode("volId", "UNKNOWN"));
    }
  }

  private void dumpSPMap(byte[] bArr, OutputNode out)
  {
    try {
      Long[] parts = new Long[3];
      TableUtils.getSPContainerMapKeyParts(bArr, parts);
      out.addChild(new OutputNode("spIdx", parts[0].intValue()));
      out.addChild(new OutputNode("volId", parts[1].intValue()));
      out.addChild(new OutputNode("cid", parts[2].intValue()));
    } catch (Throwable e) {
      out.addChild(new OutputNode("spIdx:volId:cid", "UNKNOWN"));
    }
  }

  private void dumpUint32(ByteString val, OutputNode out) 
  {
    BigInteger bi  = new BigInteger(val.toByteArray());
    out.addChild(new OutputNode("u32", Integer.reverseBytes(bi.intValue())));
  }

  private void dumpMfsNodeConfiguration(MfsNodeConfiguration nc, OutputNode out)
  {
    OutputNode ncOut = new OutputNode("MfsNodeConfig");
    out.addChild(ncOut);
    if (nc.hasNumCpus())
      ncOut.addChild(new OutputNode("NumCpus", nc.getNumCpus()));
    if (nc.hasAvailableMemory())
      ncOut.addChild(new OutputNode("AvailableMemory", nc.getAvailableMemory()));
    if (nc.hasNumSps())
      ncOut.addChild(new OutputNode("NumSps", nc.getNumSps()));
    if (nc.hasNumInstances())
      ncOut.addChild(new OutputNode("NumInstances", nc.getNumInstances()));
    if (nc.hasNumSSDSps())
      ncOut.addChild(new OutputNode("NumSSDSps", nc.getNumSSDSps()));
  }

  private void dumpMfsInstancesInfo(MfsInstancesInfo ii, OutputNode out)
  {
    OutputNode iiOut = new OutputNode("MfsInstancesInfo");
    out.addChild(iiOut);
    if (ii.hasNumSpsPerInstance())
      iiOut.addChild(new OutputNode("NumSpsPerInstance", ii.getNumSpsPerInstance()));
    if (ii.hasNumInstances())
      iiOut.addChild(new OutputNode("NumInstances", ii.getNumInstances()));
    if (ii.hasMemoryPerInstance())
      iiOut.addChild(new OutputNode("MemoryPerInstance", ii.getMemoryPerInstance()));
  }

  private void dumpFSProperties(ByteString val, OutputNode out) {
    try {
      FileServerProperties fsp = FileServerProperties.parseFrom(val);
      if (fsp.hasServerId())
        out.addChild(new OutputNode("serverId", fsp.getServerId()));
      if (fsp.hasTopology())
        out.addChild(new OutputNode("Topology", fsp.getTopology()));
      if (fsp.hasHostname())
        out.addChild(new OutputNode("Hostname", fsp.getHostname()));

      if (fsp.getIpsCount() != 0) {
        int k = 0;
        for (IPAddress ip : fsp.getIpsList()) {
          dumpIPAddress("ip" + k, ip, out);
          ++k;
        }
      }
      if (fsp.getSpIdsCount() != 0) {
        int k = 0;
        for (String spid : fsp.getSpIdsList()) {
          out.addChild(new OutputNode("spid" + k, Util.expandSpId(spid)));
          ++k;
        }
      }
      if (fsp.getSecondaryPortsCount() != 0) {
        int k = 0;
        for (int port : fsp.getSecondaryPortsList()) {
          out.addChild(new OutputNode("secPort" + k, port));
          ++k;
        }
      }

      if (fsp.hasNodeConfiguration())
        dumpMfsNodeConfiguration(fsp.getNodeConfiguration(), out);
      if (fsp.hasMfsInstancesInfo())
        dumpMfsInstancesInfo(fsp.getMfsInstancesInfo(), out);

      if (fsp.hasBuildVersion())
        out.addChild(new OutputNode("BuildVersion", fsp.getBuildVersion()));
      if (fsp.hasPatchVersion())
        out.addChild(new OutputNode("PatchVersion", fsp.getPatchVersion()));
      if (fsp.hasMarkMaintenanceTime())
        out.addChild(new OutputNode("MarkMaintenanceTime", fsp.getMarkMaintenanceTime()));
      if (fsp.hasMarkMaintenanceTimeOutMin())
        out.addChild(new OutputNode("MarkMaintenanceTimeOutMin", fsp.getMarkMaintenanceTimeOutMin()));
      if (fsp.hasBadServerId())
        out.addChild(new OutputNode("BadServerId", fsp.getBadServerId()));
      if (fsp.hasBlockMovesOut())
        out.addChild(new OutputNode("BlockMovesOut", fsp.getBlockMovesOut()));
      if (fsp.hasBlockMovesIn())
        out.addChild(new OutputNode("BlockMovesIn", fsp.getBlockMovesIn()));
      if (fsp.hasMaxContainers())
        out.addChild(new OutputNode("MaxContainers", fsp.getMaxContainers()));
      if (fsp.hasTopologyIncludesInstance())
        out.addChild(new OutputNode("TopologyIncludesInstance", fsp.getTopologyIncludesInstance()));
      if (fsp.hasPliId())
        out.addChild(new OutputNode("PliId", fsp.getPliId()));
    } catch (Throwable e) {
      out.addChild(new OutputNode("fsid", "UNKNOWN"));
    }
  }

  private void dumpSPProperties(ByteString val, OutputNode out) {
    try {
      StoragePoolProperties spp = StoragePoolProperties.parseFrom(val);
      if (spp.hasSpid())
        out.addChild(new OutputNode("spid", Util.expandSpId(spp.getSpid())));
      if (spp.hasServerId())
        out.addChild(new OutputNode("serverId", spp.getServerId()));
      if (spp.hasDeleteInProg())
        out.addChild(new OutputNode("deleteInProg", spp.getDeleteInProg()));
    } catch (Throwable e) {
      out.addChild(new OutputNode("spid", "UNKNOWN"));
    }
  }

  private void dumpKvStoreKey(KvStoreKey ksk, OutputNode out) {
    if (ksk.hasIntKey()) {
      out.addChild(new OutputNode("key", ksk.getIntKey()));
    } else if (ksk.hasLongKey()) {
      out.addChild(new OutputNode("key", ksk.getLongKey()));
      out.addChild(new OutputNode("hexKey", Long.toHexString(ksk.getLongKey())));
    } else {
      ByteString bs = ksk.getVarKey();
      out.addChild(new OutputNode("key", printableKey(bs.toByteArray())));
    }
  }

  private OutputNode formatKV(FidMsg fid, KVFormatType type, KvMsg kv) {
    OutputNode dout = new OutputNode();
    String ks = null;
    ByteString bks = null;

    // Add key.
    switch (type) {
    case XATTR:
      dout.addChild(new OutputNode("version", kv.getVersion()));
    case TABLE:
    case SCHEMA_INFO:
    case TABLET:
    case SEGMENT_MAP:
    case TABLET_MAP:
    case DEFER_MAP:
      ByteString bs = kv.getKey().getVarKey();
      dout.addChild(new OutputNode("key", printableKey(bs.toByteArray())));
      ks = bs.toStringUtf8();
      bks = bs;
      break;

    case GENERIC_KV:
      dumpKvStoreKey(kv.getKey(), dout);
      if (kv.getKey().hasVarKey())
        ks = kv.getKey().getVarKey().toStringUtf8();
      dout.addChild(new OutputNode("version", kv.getVersion()));
      break;

    case SPILL_MAP:
      dout.addChild(new OutputNode("key", kv.getKey().getIntKey()));
      break;

    case DIR:
      dout.addChild(new OutputNode("key", Long.toHexString(kv.getKey().getLongKey())));
      dout.addChild(new OutputNode("version", kv.getVersion()));
      break;

    case CLDB_CID_MAP:
      long longKey = kv.getKey().getLongKey();
      dout.addChild(new OutputNode("key", Long.toHexString(longKey)));
      dout.addChild(new OutputNode("hKey", Util.getHigherIntFromLong(longKey)));
      dout.addChild(new OutputNode("lKey", Util.getLowerIntFromLong(longKey)));
      break;

    case CLDB_SP_MAP:
      dumpSPMap(kv.getKey().getVarKey().toByteArray(), dout);
      break;

    case CLDB_CONTAINER_INFO:
    case CLDB_SNAPSHOT_INFO:
    case CLDB_VOLUME_PROP:
    case CLDB_CONTAINER_SIZE_INFO:
    case CLDB_FS_PROP:
    case CLDB_SP_PROP:
    case CLDB_VOLUME_NAME:
    default:
      dumpKvStoreKey(kv.getKey(), dout);
      break;
    }

    // Add value
    OutputNode vout = new OutputNode();
    switch (type) {
    case TABLE:
      try {
        FidMsg f = FidMsg.parseFrom(kv.getValue());
        vout.addChild(new OutputNode("fid", printableFid(f)));
      } catch (Throwable e) {
        vout.addChild(new OutputNode("fid", "UNKNOWN"));
      }
      break;

    case SCHEMA_INFO:
      try {
        DBInternalDefaults dbi = DBInternalDefaults.getDefaultInstance();
        if (ks == null) {
          vout.addChild(new OutputNode("raw", "UNKNOWN"));
        } else if (ks.startsWith(dbi.getColFamilyIdPrefix())) {
          SchemaFamily sf = SchemaFamily.parseFrom(kv.getValue());
          if (sf.hasName())
            vout.addChild(new OutputNode("cfname", sf.getName()));
          else
            vout.addChild(new OutputNode("cfname", "NONE"));
        } else if (ks.equals(dbi.getAttr())) {
          TableAttr attr = TableAttr.parseFrom(kv.getValue());
          if (attr.hasAutoSplit())
            vout.addChild(new OutputNode("attr:autoSplit",
                                         attr.getAutoSplit()));
          if (attr.hasBulkLoad())
            vout.addChild(new OutputNode("attr:bulkLoad",
                                         attr.getBulkLoad()));
          if (attr.hasDeleteTTL())
            vout.addChild(new OutputNode("attr:deleteTTL",
                                         attr.getDeleteTTL()));
          if (attr.hasSyncReplTimeoutMillis())
            vout.addChild(new OutputNode("attr:syncReplTimeoutMillis",
                                         attr.getSyncReplTimeoutMillis()));
          if (attr.hasRegionSizeMB())
            vout.addChild(new OutputNode("attr:regionSizeMB",
                                         attr.getRegionSizeMB()));
          if (attr.hasMaxValueSzInMemIndex())
            vout.addChild(new OutputNode("attr:maxValueSzInMemIndex",
                                         attr.getMaxValueSzInMemIndex()));
          if (attr.hasReclaimThreshPcntForPack())
            vout.addChild(new OutputNode("attr:reclaimThreshPcntForPack",
                                         attr.getReclaimThreshPcntForPack()));
          if (attr.hasIsMarlinTable())
            vout.addChild(new OutputNode("attr:isMarlinTable",
                                         attr.getIsMarlinTable()));
          if (attr.hasMarlinAttr()) {
            MarlinTableAttr mtAttr = attr.getMarlinAttr();

            if (mtAttr.hasAutoCreateTopics())
              vout.addChild(new OutputNode("marlinAttr:autoCreateTopics",
                                           mtAttr.getAutoCreateTopics()));
            if (mtAttr.hasDefaultNumFeedsPerTopic())
              vout.addChild(new OutputNode("marlinAttr:defaultNumFeedsPerTopic",
                                          mtAttr.getDefaultNumFeedsPerTopic()));
          }
        } else if (ks.startsWith(dbi.getReplIdxPrefix())) {
          TableReplicaDesc rd = TableReplicaDesc.parseFrom(kv.getValue());
          vout.addChild(new OutputNode("cluster", rd.getClusterName()));
          vout.addChild(new OutputNode("table", rd.getTablePath()));
          byte[] uuid = rd.getTableUuid().toByteArray();
          vout.addChild(new OutputNode("uuid",
                                       BinaryString.toUUIDString(uuid)));
          vout.addChild(new OutputNode("paused", rd.getIsPaused()));
          vout.addChild(new OutputNode("synchronous", rd.getSynchronous()));
          vout.addChild(new OutputNode("idx", rd.getIdx()));

          if (rd.getQualifiersCount() > 0) {
            String famList = "";
            int idx = 0;

            for (Qualifier qual : rd.getQualifiersList()) {
              if (qual.getQualifiersCount() > 0) {
                for (ByteString bstr : qual.getQualifiersList()) {
                  if (idx == 0) {
                    famList = qual.getFamily() + COLUMN_SEP + bstr.toStringUtf8();
                  } else {
                    famList = famList + "," + qual.getFamily() + COLUMN_SEP + bstr.toStringUtf8();
                  }
                  ++idx;
                }
              } else if (idx == 0) {
                famList = String.valueOf(qual.getFamily());
                ++idx;
              } else {
                famList = famList + "," + qual.getFamily();
                ++idx;
              }
            }
            vout.addChild(new OutputNode("columnfamilies", famList));
          }
        } else {
          vout.addChild(new OutputNode("raw", bytesToString(kv.getValue())));
        }
      } catch (Throwable e) {
        vout.addChild(new OutputNode("raw", "UNKNOWN"));
      }
      break;

    case TABLET:
      try {
        DBInternalDefaults dbi = DBInternalDefaults.getDefaultInstance();
        if (ks == null) {
          vout.addChild(new OutputNode("raw", "UNKNOWN"));

        } else if (ks.startsWith(dbi.getTabletKeyForPMap())) {
          PartitionMapEntry p = PartitionMapEntry.parseFrom(kv.getValue());
          if (p.hasSegmapFid())
            vout.addChild(new OutputNode("segfid",
                                         printableFid(p.getSegmapFid())));
          else
            vout.addChild(new OutputNode("segfid", "NONE"));

          // Add bucket list
          if (p.getBucketFidsList().size() > 0) {
            OutputNode buckets = new OutputNode("bucketfids");
            vout.addChild(buckets);
            for (FidMsg f : p.getBucketFidsList())
              buckets.addChild(new OutputNode("fid", printableFid(f)));
          }

          // Add frozen bit
          vout.addChild(new OutputNode("isFrozen", p.getIsFrozen()));

          // Add CidVN list
          if (p.getCidVNEntriesList().size() > 0) {
            OutputNode cidVNs = new OutputNode("CidVNs");
            vout.addChild(cidVNs);

            for (CidVNEntry ce : p.getCidVNEntriesList()) {
              cidVNs.addChild(new OutputNode("cid-minVN",
                                            ce.getCid() + "-" + ce.getMinVN()));
            }
          }

          // Add split state
          vout.addChild(new OutputNode("inSplit", p.getInSplit()));

          if (p.hasInImportBucket())
            dout.addChild(new OutputNode("inImportBucket",
                                         p.getInImportBucket()));

          // Add bucketDesc list
          if (p.getBucketDescsList().size() > 0) {
            OutputNode buckets = new OutputNode("bucketdescs");
            vout.addChild(buckets);
            for (PartitionMapEntry.BucketDesc bd : p.getBucketDescsList()) {
              buckets.addChild(new OutputNode("fid",
                                              printableFid(bd.getFid())));
              buckets.addChild(new OutputNode("needsRepl",
                                              bd.getNeedsRepl()));
              // buckets.addChild(new OutputNode("flushDone",
              //                                 bd.getFlushDone()));
            }
          }
          vout.addChild(new OutputNode("useBucketDesc", p.getUseBucketDesc()));

          if (p.hasLastFlushedBucketFid())
            vout.addChild(new OutputNode("lastFlushedBucketFid",
                                         printableFid(p.getLastFlushedBucketFid())));

          // Add space usage stats
          vout.addChild(new OutputNode("numLogicalBlocks",
                                       p.getUsage().getNumLogicalBlocks()));
          vout.addChild(new OutputNode("numPhysicalBlocks",
                                       p.getUsage().getNumPhysicalBlocks()));
          vout.addChild(new OutputNode("numRows", p.getUsage().getNumRows()));
          if (p.getUsage().hasNumRowsWithDelete())
            vout.addChild(new OutputNode("numRowsWithDelete",
                                         p.getUsage().getNumRowsWithDelete()));
          if (p.getUsage().hasNumRemoteBlocks())
            vout.addChild(new OutputNode("numRemoteBlocks",
                                         p.getUsage().getNumRemoteBlocks()));
          if (p.getUsage().hasNumSpills())
            vout.addChild(new OutputNode("numSpills",
                                         p.getUsage().getNumSpills()));
          if (p.getUsage().hasNumSegments())
            vout.addChild(new OutputNode("numSegments",
                                         p.getUsage().getNumSegments()));

          if (p.getUsage().hasTimeRange()) {
            TimeRange ts = p.getUsage().getTimeRange();
            vout.addChild(new OutputNode("minTimeStamp", ts.getMinTS()));
            vout.addChild(new OutputNode("maxTimeStamp", ts.getMaxTS()));
          }

          // Add marlin seq
          vout.addChild(new OutputNode("marlinMaxSeq",
                                       p.getMarlinMaxSeq()));

        } else if (ks.startsWith(dbi.getTabletKeyForSplit())) {
          SplitDesc sd = SplitDesc.parseFrom(kv.getValue());
          vout.addChild(new OutputNode("fid", printableFid(sd.getDstFid())));
          vout.addChild(new OutputNode("moveRightHalf", sd.getMoveRightHalf()));
          vout.addChild(new OutputNode("endGame", sd.getEndGame()));

        } else if (ks.startsWith(dbi.getTabletKeyForStartKey())) {
          // no value to print in this case

        } else if (ks.startsWith(dbi.getTabletKeyForEndKey())) {
          // no value to print in this case

        } else if (ks.startsWith(dbi.getTabletKeyForMerge())) {
          MergeDesc md = MergeDesc.parseFrom(kv.getValue());
          vout.addChild(new OutputNode("peerFid",
                                       printableFid(md.getPeerFid())));
          vout.addChild(new OutputNode("isDest", md.getIsDest()));
        } else {
          vout.addChild(new OutputNode("raw", bytesToString(kv.getValue())));
        }
      } catch (Throwable e) {
        vout.addChild(new OutputNode("raw", "UNKNOWN"));
      }
      break;

    case TABLET_MAP:
      try {
        TabletMapEntry t = TabletMapEntry.parseFrom(kv.getValue());
        vout.addChild(new OutputNode("fid", printableFid(t.getTabletFid())));
      } catch (Throwable e) {
        vout.addChild(new OutputNode("fid", "UNKNOWN"));
      }
      break;

    case SEGMENT_MAP:
      try {
        SegmentMapEntry s = SegmentMapEntry.parseFrom(kv.getValue());
        vout.addChild(new OutputNode("fid", printableFid(s.getFid())));
      } catch (Throwable e) {
        vout.addChild(new OutputNode("fid", "UNKNOWN"));
      }
      break;

    case SPILL_MAP:
      try {
        SpillMapEntry s = SpillMapEntry.parseFrom(kv.getValue());
        vout.addChild(new OutputNode("fid", printableFid(s.getFid())));
        if (s.hasSmeSize())
          vout.addChild(new OutputNode("smeSize", s.getSmeSize()));
        if (s.hasMinVN())
          vout.addChild(new OutputNode("minVN", s.getMinVN()));
        if (s.hasMaxMarlinSeq())
          vout.addChild(new OutputNode("maxMarlinSeq", s.getMaxMarlinSeq()));
        if (s.hasBackingBucketFid())
          vout.addChild(new OutputNode("backingBucketFid",
                                       printableFid(s.getBackingBucketFid())));
        vout.addChild(new OutputNode("keyIdxOffset", s.getKeyIdxOffset()));
        vout.addChild(new OutputNode("keyIdxLength", s.getKeyIdxLength()));
        vout.addChild(new OutputNode("ldbIdxLength", s.getLdbIdxLength()));
        vout.addChild(new OutputNode("bloomBitsPerKey",
                                     s.getBloomBitsPerKey()));
        vout.addChild(new OutputNode("numLogicalBlocks",
                                     s.getUsage().getNumLogicalBlocks()));
        vout.addChild(new OutputNode("numPhysicalBlocks",
                                     s.getUsage().getNumPhysicalBlocks()));
        vout.addChild(new OutputNode("numRows", s.getUsage().getNumRows()));
        if (s.getUsage().hasNumRowsWithDelete())
          vout.addChild(new OutputNode("numRowsWithDelete",
                                       s.getUsage().getNumRowsWithDelete()));

        OutputNode families = new OutputNode("families");
        vout.addChild(families);
        for (SpillMapEntry.FamilyEntry fe : s.getFamiliesList()) {
          families.addChild(new OutputNode("id", fe.getId()));
          if (fe.hasOffset())
            families.addChild(new OutputNode("offset", fe.getOffset()));
          if (fe.hasLength())
            families.addChild(new OutputNode("length", fe.getLength()));
          if (fe.hasTimeRange()) {
            TimeRange tr = fe.getTimeRange();
            if (tr.hasMinTS())
              families.addChild(new OutputNode("minTimeStamp", tr.getMinTS()));
            if (tr.hasMaxTS())
              families.addChild(new OutputNode("maxTimeStamp", tr.getMaxTS()));
          }
        }
      } catch (Throwable e) {
        vout.addChild(new OutputNode("keyfid", "UNKNOWN"));
      }
      break;

    case DEFER_MAP:
      try {
        if (ks.startsWith("b")) {
          ReplBucketDesc b = ReplBucketDesc.parseFrom(kv.getValue());
          vout.addChild(new OutputNode("bucketFid",
                                       printableFid(b.getBucketFid())));
          vout.addChild(new OutputNode("tableFid",
                                       printableFid(b.getTableFid())));
          if (b.getSendAfterList().size() != 0) {
            OutputNode sendAfter = new OutputNode("sendAfter");
            vout.addChild(sendAfter);
            for (FidMsg f : b.getSendAfterList())
              sendAfter.addChild(new OutputNode("fid", printableFid(f)));
          }
          if (b.getSendBeforeList().size() != 0) {
            OutputNode sendBefore = new OutputNode("sendBefore");
            vout.addChild(sendBefore);
            for (FidMsg f : b.getSendBeforeList())
              sendBefore.addChild(new OutputNode("fid", printableFid(f)));
          }

          vout.addChild(new OutputNode("flushed", b.getFlushed()));

          OutputNode replicas = new OutputNode("replicas");
          vout.addChild(replicas);
          for (ReplBucketDesc.ReplicaInfo rs : b.getReplList()) {
            replicas.addChild(new OutputNode("replicaIdx", rs.getReplicaIdx()));
            if (rs.hasDoneTillOffset())
              replicas.addChild(new OutputNode("doneTillOffset",
                                               rs.getDoneTillOffset()));
            if (rs.hasDone())
              replicas.addChild(new OutputNode("done", rs.getDone()));
          }
        } else {
          vout.addChild(new OutputNode("raw", "UNKNOWN"));
        }
      } catch (Throwable e) {
        vout.addChild(new OutputNode("raw", "UNKNOWN"));
      }
      break;

    case XATTR:
      try {
        XAttrValue x = XAttrValue.parseFrom(kv.getValue());
        ByteString xval;
        if (x.hasInum()) {
          vout.addChild(new OutputNode("inum", x.getInum()));
        }
        if (x.hasXattrNameLen()) {
          vout.addChild(new OutputNode("xattrNameLen", x.getXattrNameLen()));
          xval = bks.substring(1+x.getXattrNameLen()+1);
        } else {
          xval = x.getValue();
        }
        vout.addChild(new OutputNode("xval", xval.toStringUtf8()));
        if (ks.charAt(0) == 'A') {
          OutputNode aceOut = dumpFileACEs(xval);
          vout.addChild(aceOut);
        }
      } catch (Throwable e) {
        vout.addChild(new OutputNode("xattr", "UNKNOWN"));
      }
      break;

    case DIR:
      ByteString bs = kv.getValue();
      int offset = 0;

      while (offset < bs.size()) {
        NameEnt ent = getNameEnt(bs, offset);
        OutputNode entOut = new OutputNode("dirEntry");

        long cid = ent.cid;
        if (cid < 0) cid = fid.getCid();

        String fidStr = Long.toString(cid) + "." + Long.toString(ent.cinum) +
                     "." + Long.toString(ent.uniq);
        entOut.addChild(new OutputNode("fid", fidStr));
        entOut.addChild(new OutputNode("type", ent.ftype));
        entOut.addChild(new OutputNode("name", ent.name));
        entOut.addChild(new OutputNode("nameEnt",
                                       BinaryString.toStringHex(bs.toByteArray())));

        vout.addChild(entOut);
        offset += ent.nmlen;
        offset += 16;
      }

      break;

    case CLDB_CONTAINER_INFO:
      dumpContainerInfo(kv.getValue(), vout);
      break;
    case CLDB_CONTAINER_SIZE_INFO:
      dumpContainerSizeInfo(kv.getValue(), vout);
      break;
    case CLDB_SP_PROP:
      dumpSPProperties(kv.getValue(), vout);
      break;
    case CLDB_FS_PROP:
      dumpFSProperties(kv.getValue(), vout);
      break;
    case CLDB_SNAPSHOT_INFO:
      dumpSnapshotInfo(kv.getValue(), vout);
      break;
    case CLDB_VOLUME_PROP:
      dumpVolumeProperties(kv.getValue(), vout);
      break;
    case CLDB_VOLUME_QUOTA:
      dumpVolumeQuota(kv.getValue(), vout);
      break;
    case CLDB_VOLUME_ACES:
      dumpVolumeAces(kv.getValue(), vout);
      break;
    case CLDB_VOLUME_NAME:
      dumpVolumeTableId(kv.getValue(), vout);
      break;

    case MFS_SNAP_MAP:
      dumpUint32(kv.getValue(), vout);
      break;

    case CLDB_SP_MAP:
      // value is dummyProtoBuf
      vout.addChild(new OutputNode("value", "0"));
      break;

    default:
      vout.addChild(new OutputNode("raw", "UNKNOWN"));
      break;
    }
    dout.addChild(new OutputNode("value", vout));
    return dout;
  }

  private void dumpDir(FidMsg fid,
                       OutputHierarchy out) throws CLIProcessingException {
    MapRFileSystem fs = MapRCliUtil.getMapRFileSystem();
    MapRFileStatus dirents[];

    try {
      dirents = fs.scanDir(clusterName, "/.mapr::fid::" + printableFid(fid));
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, "scanDir failed"));
      return;
    }

    if (dirents == null)
      return;

    for (int i = 0; i < dirents.length; ++i)
      out.addNode(formatDirEntry(dirents[i]));
  }

  private void dumpRegKeyMap(
                       FidMsg fid,
                       Integer idxoffset,
                       Integer idxsize,
                       boolean dumpFullKeys,
                       int fmtVersion,
                       OutputHierarchy out) throws CLIProcessingException {
    long dbinding = getBindingForContainer(fid.getCid());
    TestScanResponse resp;
    KeyMapCookie cookie = null;

    LOG.info("Getting regular key-map with idxsize " + idxsize +
             " dumpFullKeys " + dumpFullKeys +
             " keyIdxVersion " + fmtVersion);
    if (dbinding == -1) {
      out.addError(new OutputError(Errno.ERPCFAILED,
                                   "container lookup failed"));
      return;
    }

    while (true) {
      TestScanRequest req;
      if (cookie == null) {
        req = TestScanRequest.newBuilder()
                             .setCmd(TestScanRequest.Cmd.KeyMapTable)
                             .setFid(fid)
                             .setOffset(idxoffset)
                             .setSize(idxsize)
                             .setCreds(getUserCredentials())
                             .setDumpFullKeys(dumpFullKeys)
                             .setKeyIdxFmtVersion(fmtVersion)
                             .build();
      } else {
        req = TestScanRequest.newBuilder()
                             .setCmd(TestScanRequest.Cmd.KeyMapTable)
                             .setFid(fid)
                             .setOffset(idxoffset)
                             .setSize(idxsize)
                             .setCreds(getUserCredentials())
                             .setKmapCookie(cookie)
                             .setDumpFullKeys(dumpFullKeys)
                             .setKeyIdxFmtVersion(fmtVersion)
                             .build();
      }

      try {
        byte[] replyData;
        replyData = Rpc.sendRequest(dbinding,
                              MapRProgramId.DBServerProgramId.getNumber(),
                              DBProg.TestScanProc.getNumber(), req);
        if (replyData == null) {
          out.addError(new OutputError(Errno.ERPCFAILED,
                                       "testscan rpc failed"));
          return;
        }

        resp = TestScanResponse.parseFrom(replyData);
        if (resp.getStatus() != 0) {
          out.addError(new OutputError(resp.getStatus(),
                       "TestScan failed, Error : " +
                       Errno.toString(resp.getStatus())));
          return;
        }

        for (KeyMapEntry kment : resp.getKmapentriesList()) {
          out.addNode(formatKeyMapEntry(kment));
        }
      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);
      } catch (Exception e) {
        LOG.error("Exception processing testscan command");
        out.addError(new OutputError(Errno.ERPCFAILED, "mfs rpc failed"));
        return;
      }

      if (!resp.getHasMoreKeys()) {
        LOG.info("Done with scan");
        break;
      }
      LOG.info("Setting cookie as " + resp.getKmapCookie().getOffset());
      cookie = resp.getKmapCookie();
    }
  }

  private void dumpKeyValues(FidMsg fid,
                             KVFormatType type,
                             KvStoreKey start,
                             KvStoreKey end,
                             int maxkeys,
                             OutputHierarchy out) throws CLIProcessingException
  {
    MapRFileSystem fs = MapRCliUtil.getMapRFileSystem();
    byte[] byteArr;

    try {
      // serialize the start and end key.
      byte[] kstart = null;
      if (start != null)
        kstart = start.toByteArray();

      byte[] kend = null;
      if (end != null)
        kend = end.toByteArray();

      byteArr = fs.scanKV(clusterName, "/.mapr::fid::" + printableFid(fid),
                          kstart, kend, maxkeys);
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, "scanKV failed"));
      return;
    }

    if (byteArr == null)
      return;

    try {
      KvList kvlist = KvList.parseFrom(byteArr);
      for (KvMsg kvp : kvlist.getEntriesList())
        out.addNode(formatKV(fid, type, kvp));
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED,
                                   "failed to parse key-value list"));
      return;
    }
  }

  private String printableFid(FidMsg f) {
    String cidStr;

    if (f.getCid() == -1)
      cidStr = "<parentCID>" + ".";
    else
      cidStr = f.getCid() + ".";
    return (cidStr + f.getCinum() + "." + f.getUniq());
  }

  private String printableSubType(FileType type, FileSubType subtype) {
    String stype = subtype.toString();

    if (type == FileType.FTKvstore) {
      if (subtype == FileSubType.FSTKvTable)
        stype = "FSTKvTable";
      else if (subtype == FileSubType.FSTKvTabletMap)
        stype = "FSTKvTabletMap";
      else if (subtype == FileSubType.FSTKvSchema)
        stype = "FSTKvSchema";
      else if (subtype == FileSubType.FSTKvTablet)
        stype = "FSTKvTablet";
      else if (subtype == FileSubType.FSTKvSegMap)
        stype = "FSTKvSegMap";
      else if (subtype == FileSubType.FSTKvSpillMap)
        stype = "FSTKvSpillMap";
      else if (subtype == FileSubType.FSTKvKeyMap)
        stype = "FSTKvKeyMap";
      else if (subtype == FileSubType.FSTKvXattr)
        stype = "FSTKvXattr";
    } else if (type == FileType.FTRegular) {
      if (subtype == FileSubType.FSTRegBucket)
        stype = "FSTRegBucket";
      else if (subtype == FileSubType.FSTRegSortedBucket)
        stype = "FSTRegSortedBucket";
      else if (subtype == FileSubType.FSTRegCF)
        stype = "FSTRegCF";
      else if (subtype == FileSubType.FSTRegSpill)
        stype = "FSTRegSpill";
    }
    return stype;
  }

  // constructs a byte array to a "pretty print" string.
  private String printableKey(final byte[] bytes) {
    return BinaryString.toStringBinary(bytes);
  }

  private String bytesToString(final ByteString bs) {
    return BinaryString.toStringBinary(bs.toByteArray());
  }

  private long getBindingForGateway(int gIp, int gPort) throws CLIProcessingException {
    if (gPort == 0)
      gPort = DefaultGatewayPort;
    return Rpc.createBindingFor(gIp, gPort, clusterName, ServerKeyType.ServerKey.getNumber());
  }

  private String getCompressionType(FileCompressionType ct) {
    if (ct == FileCompressionType.FCT_OFF)
      return "off";
    else if (ct == FileCompressionType.FCT_LZ4)
      return "lz4";
    else if (ct == FileCompressionType.FCT_LZF)
      return "lzf";
    else if (ct == FileCompressionType.FCT_ZLIB)
      return "zlib";
    else
      return "unknown";
 }

  private IPAddress buildIPFromString(String ipstr) {
    if (ipstr.equalsIgnoreCase("localhost") ||
        ipstr.equalsIgnoreCase("127.0.0.1") ) {
      ipstr = "127.0.0.1";
    } else {
      List<String> ips;
      ips = NodesCommonUtils.convertHostToIp(Collections.singletonList(ipstr));
      if (ips.isEmpty())
        return null;

      ipstr = ips.get(0);
    }

    int host = Util.ipToInt(ipstr);
    IPAddress server = IPAddress.newBuilder()
                                .setHost(host)
                                .build();
    return server;
  }
}
