/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.TextFormat;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.CommandOutput.*;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.*;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.License.*;
import org.apache.log4j.Logger;
import com.mapr.security.MaprSecurityException;

import java.io.IOException;
import java.util.List;
import java.util.Map;

/**
 * Eula support
 */
public class EulaCommands extends CLIBaseClass
{
  private static final Logger LOG = Logger.getLogger(EulaCommands.class);
  private static final String USER_PARAM_NAME = "user";
  
  public static Map<String, BaseInputParameter> baseParams =
    new ImmutableMap.Builder<String, BaseInputParameter>()
    .put(MapRCliUtil.CLUSTER_NAME_PARAM,
      new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
          "cluster name",
          CLIBaseClass.NOT_REQUIRED,
          null))
         .build();

  private static final CLICommand showacceptCommand =
    new CLICommand (
        "showaccept",
        "usage: eula showaccept -cluster clustername",
        EulaCommands.class,
        CLICommand.ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>().
          putAll(baseParams)
        .build(),
        null)
    .setShortUsage("eula showaccept -cluster <clustername>");
  
  private static final CLICommand acceptCommand =
    new CLICommand (
        "accept",
        "usage: eula accept -user username -cluster clustername",
        EulaCommands.class,
        CLICommand.ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(baseParams)
        .put(USER_PARAM_NAME, 
            new TextInputParameter(USER_PARAM_NAME,
                "username", 
                CLIBaseClass.REQUIRED, 
                null))
        .build(),
        null)
    .setShortUsage("eula accept -user <username> -cluster <clustername>");

  
  // main command
  public static final CLICommand eulaCommands =
    new CLICommand (
         "eula", "eula [accept|showaccept]",
         CLIUsageOnlyCommand.class,
         CLICommand.ExecutionTypeEnum.NATIVE,
         new CLICommand[] {
           acceptCommand, 
           showacceptCommand
         }
       )
    .setShortUsage("eula [accept|showaccept]")
    .setUsageInVisible(true);

  public EulaCommands (ProcessedInput input, CLICommand cliCommand)
  {
    super(input, cliCommand);
  }

  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException
  {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);

    if (!super.validateInput()) {
      return output;
    }

    if (cliCommand.getCommandName().equalsIgnoreCase("accept")) {
      acceptEula(out);
    } else if (cliCommand.getCommandName().equalsIgnoreCase("showaccept")) {
      showacceptEula(out);
    } 
    return output;
  }
  
  
  private void acceptEula (OutputHierarchy out) throws CLIProcessingException
  {
    String user = getParamTextValue(USER_PARAM_NAME, 0);
    if (user == null || user.trim().isEmpty()) {
      out.addError (new CommandOutput.OutputHierarchy.OutputError (
          Errno.EINVAL, "Invalid user  " + Errno.toString(Errno.EINVAL))
        );
      return;
    }
    EulaAcceptRequest.Builder b = EulaAcceptRequest.newBuilder();
    b.setAcceptedUser(user);
    
    EulaAcceptResponse resp;    
    byte[] data;

    try {
      if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        String cluster = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0);
        if (!CLDBRpcCommonUtils.getInstance().isValidClusterName(cluster)) {
          out.addError(new OutputError(Errno.EUCLUSTER, "Invalid cluster: " + cluster));
          return;
        }
        data = CLDBRpcCommonUtils.getInstance().sendRequest (
                    cluster,
                    Common.MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.EulaAcceptProc.getNumber(),
                    b.build(),
                    EulaAcceptResponse.class);
      } else {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(
                    Common.MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.EulaAcceptProc.getNumber(),
                    b.build(),
                    EulaAcceptResponse.class);
      }

      if (data == null) {
    	  out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
    	  return;
      }

      // success
      resp = EulaAcceptResponse.parseFrom(data);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      out.addError (new CommandOutput.OutputHierarchy.OutputError (
            Errno.EOPFAILED, "EulaAccept: " + Errno.toString(Errno.EOPFAILED)));
      LOG.error("Exception during EulaAccept", e);
      return;
    }

    int status = resp.getStatus();
    if (status != 0) {
      out.addError(new OutputError(status, Errno.toString(status)));
    }
  }
  
  private void showacceptEula (OutputHierarchy out) throws CLIProcessingException {
    String cluster = null;
    if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
      cluster = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0);
      if (!CLDBRpcCommonUtils.getInstance().isValidClusterName(cluster)) {
        out.addError(new OutputError(Errno.EUCLUSTER, "Invalid cluster: " + cluster));
        return;
      }
    }
    
    EulaShowAcceptResponse resp = fetchEula(cluster);
    if ( resp == null ) {
    	out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
    	return;
    }
    int status = resp.getStatus();
    if (status != 0) {
      out.addError(new OutputError(status, Errno.toString(status)));
      return;
    }    
    
    if (resp.hasAcceptedUser()) {
      OutputNode node = new OutputNode();
      node.addChild(new OutputNode("user", resp.getAcceptedUser()));
      node.addChild(new OutputNode("time", resp.getAcceptedTime()));
      out.addNode(node);    
    }
  }
  

  public static EulaShowAcceptResponse fetchEula (String cluster)
      throws CLIProcessingException
  {
    EulaShowAcceptRequest.Builder b = EulaShowAcceptRequest.newBuilder();
    EulaShowAcceptResponse resp;
    byte[] data;

    try {
      if (cluster != null) {
        data = CLDBRpcCommonUtils.getInstance().sendRequest (
                    cluster,
                    Common.MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.EulaShowAcceptProc.getNumber(),
                    b.build(),
                    EulaShowAcceptResponse.class
              );
      } else {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(
                    Common.MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.EulaShowAcceptProc.getNumber(),
                    b.build(),
                    EulaShowAcceptResponse.class
          );
      }

      if (data == null) {
    	  return null;
      }

      // success
      resp = EulaShowAcceptResponse.parseFrom(data);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      throw new CLIProcessingException ("Exception doing RPC to CLDB");
    }
    
    return resp;
  }
}
