/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import java.util.List;
import java.util.ArrayList;
import java.util.Map;
import java.util.Properties;

import org.apache.log4j.Logger;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.MessageLite;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.baseutils.Errno;
import com.mapr.cli.common.ListCommand;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.TextCommandOutput;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import com.mapr.cliframework.base.inputparams.IntegerInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.cliframework.util.FieldInfo;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.AeInfoFields;
import com.mapr.fs.cldb.proto.CLDBProto.AeKey;
import com.mapr.fs.cldb.proto.CLDBProto.AeLookupRequest;
import com.mapr.fs.cldb.proto.CLDBProto.AeLookupResponse;
import com.mapr.fs.cldb.proto.CLDBProto.AeModifyRequest;
import com.mapr.fs.cldb.proto.CLDBProto.AeModifyResponse;
import com.mapr.fs.cldb.proto.CLDBProto.AeProperties;
import com.mapr.fs.cldb.proto.CLDBProto.AeRemoveRequest;
import com.mapr.fs.cldb.proto.CLDBProto.AeRemoveResponse;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Common.AlarmId;
import com.mapr.fs.proto.Common.AlarmMsg;
import com.mapr.security.MaprSecurityException;

public class EntityCommands extends ListCommand implements CLIInterface {
    
    public static final String ENTITY_TYPE_PARAM_NAME = "type";
    public static final String ENTITY_NAME_PARAM_NAME = "name";
    public static final String ENTITIES_PARAM_NAME = "entities";
    public static final String ENTITY_EMAIL_PARAM_NAME = "email";
    public static final String ENTITY_QUOTA_PARAM_NAME = "quota";
    public static final String ENTITY_ADVISORY_QUOTA_PARAM_NAME = "advisoryquota";
    
    public static final String FILTER_PARAM_NAME   = "filter";
    public static final String COLUMNS_PARAM_NAME   = "columns";
    public static final String SORT_PARAM_NAME = "sort";
    public static final String SORT_DIRECTION_PARAM_NAME = "dir";
    public static final String OUTPUT_PARAM_NAME = "output";
    public static final String START_PARAM_NAME = "start";
    public static final String LIMIT_PARAM_NAME = "limit";
    public static final String ALARMEDENTITIES_PARAM_NAME = "alarmedentities";
    
    private static final int NUM_ENTITIES_PER_RPC = 100;
      
    Properties CLDBProperties = new Properties();
    String querysource;
    private static final Logger LOG = Logger.getLogger(EntityCommands.class);
    
    public EntityCommands(ProcessedInput input, CLICommand cliCommand) {
        super(input, cliCommand);
    }

    static final CLICommand listallCmd = new CLICommand(
            "listall",
            "",
            EntityCommands.class, 
            ExecutionTypeEnum.NATIVE,
            /* Add parameters in a hash map key is string,  value is BaseInputParameter */
            new ImmutableMap.Builder<String, BaseInputParameter>()
                .put(EntityCommands.OUTPUT_PARAM_NAME,
                    new TextInputParameter(EntityCommands.OUTPUT_PARAM_NAME,
                        "<terse|verbose>",
                        CLIBaseClass.NOT_REQUIRED,
                        "verbose"))
/*                .put(EntityCommands.SORT_PARAM_NAME,
                    new TextInputParameter(EntityCommands.SORT_PARAM_NAME,
                        "none",
                        CLIBaseClass.NOT_REQUIRED,
                        "none"))
                .put(EntityCommands.SORT_DIRECTION_PARAM_NAME,
                    new TextInputParameter(EntityCommands.SORT_DIRECTION_PARAM_NAME,
                        "none",
                        CLIBaseClass.NOT_REQUIRED,
                        "ASC"))
                .put(EntityCommands.START_PARAM_NAME,
                    new IntegerInputParameter(EntityCommands.START_PARAM_NAME,
                        "start",
                        CLIBaseClass.NOT_REQUIRED,
                        0))
                .put(EntityCommands.LIMIT_PARAM_NAME,
                    new IntegerInputParameter(EntityCommands.LIMIT_PARAM_NAME,
                        "limit",
                        CLIBaseClass.NOT_REQUIRED,
                        Integer.MAX_VALUE))
                .put(EntityCommands.FILTER_PARAM_NAME,
                    new TextInputParameter(EntityCommands.FILTER_PARAM_NAME,
                        "none",
                        CLIBaseClass.NOT_REQUIRED,
                        "none"))
                .put(EntityCommands.COLUMNS_PARAM_NAME,
                    new TextInputParameter(EntityCommands.COLUMNS_PARAM_NAME,
                        "none",
                        CLIBaseClass.NOT_REQUIRED,
                        "none"))
*/                .build(),
                null        /* sub-commands */
       ).setShortUsage("entity listall -output <terse|verbose>");


    static final CLICommand listCmd = new CLICommand(
            "list",
            "",
            EntityCommands.class, 
            ExecutionTypeEnum.NATIVE,
            /* Add parameters in a hash map key is string,  value is BaseInputParameter */
            new ImmutableMap.Builder<String, BaseInputParameter>()
                .put(EntityCommands.OUTPUT_PARAM_NAME,
                    new TextInputParameter(EntityCommands.OUTPUT_PARAM_NAME,
                        "<terse|verbose>",
                        CLIBaseClass.NOT_REQUIRED,
                        "verbose"))
                .put(EntityCommands.SORT_PARAM_NAME,
                    new TextInputParameter(EntityCommands.SORT_PARAM_NAME,
                        "none",
                        CLIBaseClass.NOT_REQUIRED,
                        "n").setInvisible(true))
                .put(EntityCommands.SORT_DIRECTION_PARAM_NAME,
                    new TextInputParameter(EntityCommands.SORT_DIRECTION_PARAM_NAME,
                        "none",
                        CLIBaseClass.NOT_REQUIRED,
                        "ASC").setInvisible(true))
                .put(EntityCommands.START_PARAM_NAME,
                    new IntegerInputParameter(EntityCommands.START_PARAM_NAME,
                        "start",
                        CLIBaseClass.NOT_REQUIRED,
                        0))
                .put(EntityCommands.LIMIT_PARAM_NAME,
                    new IntegerInputParameter(EntityCommands.LIMIT_PARAM_NAME,
                        "limit",
                        CLIBaseClass.NOT_REQUIRED,
                        Integer.MAX_VALUE))
                .put(EntityCommands.FILTER_PARAM_NAME,
                    new TextInputParameter(EntityCommands.FILTER_PARAM_NAME,
                        "none",
                        CLIBaseClass.NOT_REQUIRED,
                        "none"))
                .put(EntityCommands.COLUMNS_PARAM_NAME,
                    new TextInputParameter(EntityCommands.COLUMNS_PARAM_NAME,
                        "none",
                        CLIBaseClass.NOT_REQUIRED,
                        "none"))
                .put(EntityCommands.ALARMEDENTITIES_PARAM_NAME,
                   new BooleanInputParameter(EntityCommands.ALARMEDENTITIES_PARAM_NAME,
                       "true|false",
                       CLIBaseClass.NOT_REQUIRED,
                       false)) 
                .put(MapRCliUtil.CLUSTER_NAME_PARAM,
                    new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
                        "cluster name",
                        CLIBaseClass.NOT_REQUIRED,
                        null))
                .build(),
                null        /* sub-commands */
       ).setShortUsage("entity list");

    static final CLICommand infoCmd = new CLICommand(
            "info",
            "",
            EntityCommands.class, 
            ExecutionTypeEnum.NATIVE,
            /* Add parameters in a hash map key is string,  value is BaseInputParameter */
            new ImmutableMap.Builder<String, BaseInputParameter>()
                /* parameter 1 */
                .put(EntityCommands.ENTITY_TYPE_PARAM_NAME,
                   new BooleanInputParameter(
                       EntityCommands.ENTITY_TYPE_PARAM_NAME,
                       "type",
                       CLIBaseClass.REQUIRED,
                       0))
                /* parameter 2 */
               .put(EntityCommands.ENTITY_NAME_PARAM_NAME,
                   new TextInputParameter(
                       EntityCommands.ENTITY_NAME_PARAM_NAME,
                       "name",
                       CLIBaseClass.REQUIRED,
                       null))
               .put(EntityCommands.OUTPUT_PARAM_NAME,
                    new TextInputParameter(EntityCommands.OUTPUT_PARAM_NAME,
                        "<terse|verbose>",
                        CLIBaseClass.NOT_REQUIRED,
                        "verbose"))
               .put(MapRCliUtil.CLUSTER_NAME_PARAM,
                    new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
                        "cluster name",
                        CLIBaseClass.NOT_REQUIRED,
                        null))
               .build(),
                null        /* sub-commands */
       ).setShortUsage("entity info -type <0=user|1=group> -name <entityname>");

    static final CLICommand existsCmd = new CLICommand(
            "exists",
            "",
            EntityCommands.class, 
            ExecutionTypeEnum.NATIVE,
            /* Add parameters in a hash map key is string,  value is BaseInputParameter */
            new ImmutableMap.Builder<String, BaseInputParameter>()
               /* parameter 1 */
               .put(EntityCommands.ENTITY_TYPE_PARAM_NAME,
                   new BooleanInputParameter(
                       EntityCommands.ENTITY_TYPE_PARAM_NAME,
                       "type",
                       CLIBaseClass.REQUIRED,
                       0))
               /* parameter 2 */
               .put(EntityCommands.ENTITY_NAME_PARAM_NAME,
                    new TextInputParameter(
                            EntityCommands.ENTITY_NAME_PARAM_NAME,
                         "entityname",
                         CLIBaseClass.REQUIRED,
                         ""))
               .build(),
                null        /* sub-commands */
        ).setUsageInVisible(true).setShortUsage("entity exists -name <name>");

    static final CLICommand removeCmd = new CLICommand(
            "remove",
            "",
            EntityCommands.class,
            ExecutionTypeEnum.NATIVE,
            /* Add parameters in a hash map key is string,  value is BaseInputParameter */
            new ImmutableMap.Builder<String, BaseInputParameter>()
               /* parameter 1 */
               .put(EntityCommands.ENTITY_NAME_PARAM_NAME,
                    new TextInputParameter(
                            EntityCommands.ENTITY_NAME_PARAM_NAME,
                         "entityname",
                         CLIBaseClass.NOT_REQUIRED,
                         null))
               /* parameter 2 */
               .put(EntityCommands.ENTITY_TYPE_PARAM_NAME,
                   new BooleanInputParameter(
                       EntityCommands.ENTITY_TYPE_PARAM_NAME,
                       "type",
                       CLIBaseClass.NOT_REQUIRED,
                       null))
               /* parameter 3 */
               .put(EntityCommands.ENTITIES_PARAM_NAME,
                    new TextInputParameter(
                        EntityCommands.ENTITIES_PARAM_NAME,
                        "entities (0:<user1>,0:<user2>,1:<group1>,1:<group2>..)",
                        CLIBaseClass.NOT_REQUIRED,
                        null))
               .build(),
                null        /* sub-commands */
        ).setShortUsage("entity remove [-name <entityname> -type <entitytype>] [-entities ...]");

    static final CLICommand modifyCmd = new CLICommand(
            "modify",
            "",
            EntityCommands.class, 
            ExecutionTypeEnum.NATIVE,
            /* Add parameters in a hash map key is string,  value is BaseInputParameter */
            new ImmutableMap.Builder<String, BaseInputParameter>()
               /* parameter 1 */
               .put(EntityCommands.ENTITY_TYPE_PARAM_NAME,
                   new BooleanInputParameter(
                       EntityCommands.ENTITY_TYPE_PARAM_NAME,
                       "type",
                       CLIBaseClass.NOT_REQUIRED,
                       null))
               /* parameter 2 */
               .put(EntityCommands.ENTITY_NAME_PARAM_NAME,
                    new TextInputParameter(
                        EntityCommands.ENTITY_NAME_PARAM_NAME,
                        "entityname",
                        CLIBaseClass.NOT_REQUIRED,
                        null))
               /* parameter 3 */
               .put(EntityCommands.ENTITIES_PARAM_NAME,
                    new TextInputParameter(
                        EntityCommands.ENTITIES_PARAM_NAME,
                        "entities (0:<user1>,0:<user2>,1:<group1>,1:<group2>..)",
                        CLIBaseClass.NOT_REQUIRED,
                        null))         
               /* parameter 4 */
               .put(EntityCommands.ENTITY_EMAIL_PARAM_NAME,
                    new TextInputParameter(
                        EntityCommands.ENTITY_EMAIL_PARAM_NAME,
                        "email",
                        CLIBaseClass.NOT_REQUIRED,
                        null))
               /* parameter 5 */
               .put(EntityCommands.ENTITY_QUOTA_PARAM_NAME,
                    new TextInputParameter(
                        EntityCommands.ENTITY_QUOTA_PARAM_NAME,
                        "quota",
                        CLIBaseClass.NOT_REQUIRED,
                        null))
               /* parameter 6 */
               .put(EntityCommands.ENTITY_ADVISORY_QUOTA_PARAM_NAME,
                    new TextInputParameter(
                        EntityCommands.ENTITY_ADVISORY_QUOTA_PARAM_NAME,
                        "advisory quota",
                        CLIBaseClass.NOT_REQUIRED,
                        null))
               /* parameter 7 */                        
               .put(MapRCliUtil.CLUSTER_NAME_PARAM,
            		new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
            			  "cluster name",
            			   CLIBaseClass.NOT_REQUIRED,
            			   null))
               .build(),
                null        /* sub-commands */
       ).setShortUsage("entity modify");
    
    /* Define sub command */
    public static final CLICommand entityCmds = new CLICommand(
        "entity",
        "",
        CLIUsageOnlyCommand.class, 
        ExecutionTypeEnum.NATIVE,
        new CLICommand [] {         /* array of subcommands */
           listCmd,
           infoCmd,
           existsCmd,
           modifyCmd,
           removeCmd
        }
    ).setShortUsage("entity [list|info|modify|remove]");
    
    public static Map<AeInfoFields, FieldInfo> fieldTable = 
      new ImmutableMap.Builder<AeInfoFields, FieldInfo>()
      .put(AeInfoFields.aequota, 
           new FieldInfo(AeInfoFields.aequota.getNumber(), 
                        "qta", "EntityQuota", Long.class))
      .put(AeInfoFields.aeadvisoryquota, 
           new FieldInfo(AeInfoFields.aeadvisoryquota.getNumber(), 
                        "aqt", "EntityAdvisoryquota", Long.class))
      .put(AeInfoFields.aename, 
           new FieldInfo(AeInfoFields.aename.getNumber(), 
                         "n", "EntityName", String.class))    
      .put(AeInfoFields.aetype, 
           new FieldInfo(AeInfoFields.aetype.getNumber(), 
                         "t", "EntityType", Integer.class))
      .put(AeInfoFields.aevolumecount, 
           new FieldInfo(AeInfoFields.aevolumecount.getNumber(), 
                         "vct", "VolumeCount", Integer.class))                    
      .put(AeInfoFields.aediskusage, 
           new FieldInfo(AeInfoFields.aediskusage.getNumber(), 
                         "dsu", "DiskUsage", Long.class))                   
      .put(AeInfoFields.aeid, 
           new FieldInfo(AeInfoFields.aeid.getNumber(), 
                         "id", "EntityId", Integer.class))                         
      .put(AeInfoFields.aeemail, 
           new FieldInfo(AeInfoFields.aeemail.getNumber(), 
                         "em", "EntityEmail", String.class))
      .put(AeInfoFields.aequotaalarm, 
           new FieldInfo(AeInfoFields.aequotaalarm.getNumber(), 
                         "qe", "EntityQuotaExceededAlarm", String.class))                         
      .put(AeInfoFields.aeadvisoryquotaalarm, 
           new FieldInfo(AeInfoFields.aeadvisoryquotaalarm.getNumber(), 
                         "aqe", "EntityAdvisoryQuotaExceededAlarm", String.class))                         
      .build();
    
        
    /* This is a function that does the real work. */
    @Override
    public CommandOutput executeRealCommand() throws CLIProcessingException {
        OutputHierarchy out = new OutputHierarchy();  /* Root of comamnd output's hierarchy tree */  
        CommandOutput output = new CommandOutput();
        output.setOutput(out);
        
        String cmd = cliCommand.getCommandName();
        if (cmd.equalsIgnoreCase("listall")) {
            FetchAllEntities(out);
        } else if (cmd.equalsIgnoreCase("list")) {
          // First fetch CLDB Params to know the latest params
          String cluster = null;
          if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
            cluster = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0);
            if (!CLDBRpcCommonUtils.getInstance().isValidClusterName(cluster)) {
              out.addError(new OutputError(Errno.EUCLUSTER, "Invalid cluster: " + cluster));
              return output;
            }
          }
          MapRCliUtil.fetchCLDBParams(cluster, CLDBProperties, getUserCredentials());
          list(out);
        } else if (cmd.equalsIgnoreCase("info")) {
            String name = getParamTextValue(ENTITY_NAME_PARAM_NAME, 0);
            if ((name == null) || name.isEmpty()) {       
                /**
                * <MAPR_ERROR>
                * Message:Invalid Entity
                * Function:EntityCommands.executeRealCommand()
                * Meaning:The entity name was empty or NULL.
                * Resolution:Check the entity name and type, and the command syntax, and try again.
                * </MAPR_ERROR>
                */
                out.addError(new OutputError(Errno.EINVAL, "Invalid Entity")
                                   .setField(ENTITY_NAME_PARAM_NAME)
                                   .setFieldValue(name));
                return output;
            }     
            boolean type = getParamBooleanValue(ENTITY_TYPE_PARAM_NAME, 0);
            
            AeKey key = 
                 AeKey.newBuilder().setName(name).setType(type).build();     
            AeLookupRequest.Builder reqBuilder = AeLookupRequest.newBuilder().addKeys(key);            
            FetchEntities(out, reqBuilder);
        } else if (cmd.equalsIgnoreCase("exists")) {
            String name = getParamTextValue(ENTITY_NAME_PARAM_NAME, 0);
            if ((name == null) || name.isEmpty()) {
               /**
               * <MAPR_ERROR>
               * Message:Invalid Entity
               * Function:EntityCommands.executeRealCommand()
               * Meaning:The entity name was empty or NULL.
               * Resolution:Check the entity name and type, and the command syntax, and try again.
               * </MAPR_ERROR>
               */
               out.addError(new OutputError(Errno.EINVAL, "Invalid Entity")
                                 .setField(ENTITY_NAME_PARAM_NAME)
                                 .setFieldValue(name));
               return output;
            }
            boolean type = getParamBooleanValue(ENTITY_TYPE_PARAM_NAME, 0);
            
            if (checkEntityExists(type, name)) {
                out.addNode(new OutputNode());
            } else {
                /**
                * <MAPR_ERROR>
                * Message:Entity does not exist
                * Function:EntityCommands.executeRealCommand()
                * Meaning:The specified entity does not exist
                * Resolution:Check the entity name and type, and try again.
                * </MAPR_ERROR>
                */
                out.addError(new OutputError(Errno.ENOTEXIST, "Entity does not exist")
                .setField(ENTITY_NAME_PARAM_NAME)
                .setFieldValue(name));
            }   
        } else if (cmd.equalsIgnoreCase("modify")) {
          ModifyEntity(out, cmd);               
        } else if (cmd.equalsIgnoreCase("remove")) {
          RemoveEntity(out, cmd);
        }
        return output;
    }
    
    void FetchAllEntities (OutputHierarchy out) throws CLIProcessingException {
        // First fetch CLDB Params to know the latest params
        String cluster = null;
        if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
          cluster = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0);
          if (!CLDBRpcCommonUtils.getInstance().isValidClusterName(cluster)) {
            out.addError(new OutputError(Errno.EUCLUSTER, "Invalid cluster: " + cluster));
            return;
          }
        }
        MapRCliUtil.fetchCLDBParams(cluster, CLDBProperties, getUserCredentials());
        
        String [] db = {"passwd", "group"};
                
        boolean terse = getParamTextValue(OUTPUT_PARAM_NAME, 0).equalsIgnoreCase("terse");
        
        for (int i = 0; i < db.length; i++) {
            try {
                byte [] b = executeSimpleSHHCommand(6000L, "getent " + db[i]);
                if (b != null) {
                    TextCommandOutput txt = new TextCommandOutput(b);
                    String[] str = txt.toString().split("\n");
                    for (String s: str) {
                       String[] info = s.split(":");
                       OutputNode node = new OutputNode();                       
                       node.addChild(new OutputNode(
                           fieldTable.get(AeInfoFields.aetype).getName(terse), i));
                       node.addChild(new OutputNode(
                          fieldTable.get(AeInfoFields.aename).getName(terse), 
                          info[0]));
                       node.addChild(new OutputNode(
                           fieldTable.get(AeInfoFields.aeid).getName(terse),
                           Integer.valueOf(info[2])));
                       out.addNode(node);
                    }
                }
            } catch (Exception e) {
               /**
                * <MAPR_ERROR>
                * Message:Exception: <message>
                * Function:EntityCommands.FetchAllEntities()
                * Meaning:An exception occurred while executing the command.
                * Resolution:Contact technical support.
                * </MAPR_ERROR>
                */
                out.addError(new OutputError(Errno.EOPFAILED, "ERROR"));
                LOG.error("Exception: " + e.getLocalizedMessage());
            }   
        }
    }
    
    void FetchEntities (OutputHierarchy out, AeLookupRequest.Builder reqBuilder) 
        throws CLIProcessingException {

      reqBuilder.setCreds(getUserCredentials());

      // First fetch CLDB Params to know the latest params
      String cluster = null;
      if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        cluster = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0);
      }
      MapRCliUtil.fetchCLDBParams(cluster, CLDBProperties, getUserCredentials());
         
       // Then query CLDB to fetch Entity quota + static-email information
       try {
          byte[] replyData;
          if (cluster != null && !cluster.isEmpty()) {
        	  replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
                cluster, 
        			  Common.MapRProgramId.CldbProgramId.getNumber(),
        			  CLDBProto.CLDBProg.AeLookupProc.getNumber(),
                      reqBuilder.build(), AeLookupResponse.class);
          } else {
        	  replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
        			  Common.MapRProgramId.CldbProgramId.getNumber(),
        			  CLDBProto.CLDBProg.AeLookupProc.getNumber(),
                      reqBuilder.build(), AeLookupResponse.class);
          }
          
         if (replyData == null) {
           out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
           return;
         }
         
         boolean terse = getParamTextValue(OUTPUT_PARAM_NAME, 0).equalsIgnoreCase("terse");
              
         AeLookupResponse resp = AeLookupResponse.parseFrom(replyData);
         for (AeProperties props : resp.getAePropertyList()) {
          OutputNode node = new OutputNode();          
          boolean type = props.getAeKey().getType();
          String name = props.getAeKey().getName();
          node.addChild(new OutputNode(
               fieldTable.get(AeInfoFields.aetype).getName(terse), 
               type ? 1 : 0));
          node.addChild(new OutputNode(
              fieldTable.get(AeInfoFields.aename).getName(terse), 
              name));             
          node.addChild(new OutputNode(
              fieldTable.get(AeInfoFields.aevolumecount).getName(terse),
              props.getAeVolumeCount()));
          node.addChild(new OutputNode(
              fieldTable.get(AeInfoFields.aequota).getName(terse), 
              props.getQuotaSizeMB()));
          node.addChild(new OutputNode(
              fieldTable.get(AeInfoFields.aeadvisoryquota).getName(terse), 
              props.getQuotaAdvisorySizeMB()));
          node.addChild(new OutputNode(
              fieldTable.get(AeInfoFields.aediskusage).getName(terse),
              props.getUsedSizeMB()));
        
          // If message has statically configured email use that
          if (props.hasEmail()) {
             node.addChild(new OutputNode(
                 fieldTable.get(AeInfoFields.aeemail).getName(terse), 
                 props.getEmail()));
          } else {
             String email = MapRCliUtil.fetchEmail(CLDBProperties, name, type);
             if (email != null)
               node.addChild(new OutputNode(
                   fieldTable.get(AeInfoFields.aeemail).getName(terse), 
                   email));
          }        
          
          try {
            // Fetch uid/gid from getent
            byte [] b = executeSimpleSHHCommand(6000L, 
                             "getent " + (type ? "group " : "passwd ") + name);
            if (b != null) {
              TextCommandOutput txt = new TextCommandOutput(b);
              String[] info = txt.toString().split(":");
              node.addChild(new OutputNode(
                  fieldTable.get(AeInfoFields.aeid).getName(terse), 
                  Integer.valueOf(info[2])));
            }
          } catch (Exception e) {
           /**
            * <MAPR_ERROR>
            * Message:Could not fetch entity uid/gid <message>
            * Function:EntityCommands.FetchEntities()
            * Meaning:An exception occurred while attempting to retrieve the specified entity.
            * Resolution:Contact technical support.
            * </MAPR_ERROR>
            */
            LOG.error("Could not fetch entity uid/gid " + e.getLocalizedMessage());
          }
          
          out.addNode(node);
         }
       } catch (MaprSecurityException e) {
         throw new CLIProcessingException(
           "MaprSecurityException " + "Exception", e);
       } catch (Exception e) {
        /**
         * <MAPR_ERROR>
         * Message:Failed to fetch entity info
         * Function:EntityCommands.FetchEntities()
         * Meaning:An exception occurred while fetching the information for the specified entity.
         * Resolution:Contact technical support.
         * </MAPR_ERROR>
         */
         out.addError(new OutputError(Errno.EOPFAILED, "Failed to fetch entity info"));
         LOG.error("Exception: " + e.getLocalizedMessage());
       }
    }
    
    private void modify(OutputHierarchy out, List<String> names, List<Boolean> types) 
       throws CLIProcessingException{
    	boolean isModify = false;
      AeProperties.Builder propsBuilder = AeProperties.newBuilder();            
      
      if (isParamPresent(ENTITY_EMAIL_PARAM_NAME)) {
        String email = getParamTextValue(ENTITY_EMAIL_PARAM_NAME, 0);
        if (email == null) {
         /**
          * <MAPR_ERROR>
          * Message:Invalid Email
          * Function:EntityCommands.ModifyEntity()
          * Meaning:The specified email was NULL.
          * Resolution:Check the command syntax and try again.
          * </MAPR_ERROR>
          */
          out.addError(new OutputError(Errno.EINVAL, "Invalid Email")
                                  .setField(ENTITY_EMAIL_PARAM_NAME)
                                  .setFieldValue(email));
        }
        email = email.trim();
        if (!email.isEmpty() && !MapRCliUtil.validateEmail(email)) {
         /**
          * <MAPR_ERROR>
          * Message:Invalid Email
          * Function:EntityCommands.ModifyEntity()
          * Meaning:The specified email was invalid.
          * Resolution:Check the command syntax and try again.
          * </MAPR_ERROR>
          */
          out.addError(new OutputError(Errno.EINVAL, "Invalid Email")
                               .setField(ENTITY_EMAIL_PARAM_NAME)
                               .setFieldValue(email));
          return;
        }
        propsBuilder.setEmail(email);
        isModify = true;
      }
      
      if (isParamPresent(ENTITY_QUOTA_PARAM_NAME)) {
        String quotaStr = getParamTextValue(ENTITY_QUOTA_PARAM_NAME, 0);
        long quota = MapRCliUtil.quotaStringToMB(quotaStr);
        if (quota < 0) {
         /**
          * <MAPR_ERROR>
          * Message:Invalid Quota Value. Should be either an Integer, or a decimal value followed by one of (M,MB,G,GB,T,TB,P,PB)
          * Function:EntityCommands.ModifyEntity()
          * Meaning:The specified quota was invalid. 
          * Resolution:Check the command syntax and try again.
          * </MAPR_ERROR>
          */
          out.addError(new OutputError(Errno.EINVAL, "Invalid Quota Value. " +
              "Should be either an Integer, or a decimal value followed by one of (M,MB,G,GB,T,TB,P,PB)")
                               .setField(ENTITY_QUOTA_PARAM_NAME)
                               .setFieldValue(quotaStr));
          return;
        }
        propsBuilder.setQuotaSizeMB(quota);
        isModify = true;        
      }
      
      if (isParamPresent(ENTITY_ADVISORY_QUOTA_PARAM_NAME)) {
        String quotaStr = getParamTextValue(ENTITY_ADVISORY_QUOTA_PARAM_NAME, 0);
        long quota = MapRCliUtil.quotaStringToMB(quotaStr);
        if (quota < 0) {
         /**
          * <MAPR_ERROR>
          * Message:Invalid Advisory Quota Value. Should be either an Integer, or a decimal value followed by one of (M,MB,G,GB,T,TB,P,PB)
          * Function:EntityCommands.ModifyEntity()
          * Meaning:The specified advisory quota was invalid. 
          * Resolution:Check the command syntax and try again.
          * </MAPR_ERROR>
          */
          out.addError(new OutputError(Errno.EINVAL, "Invalid Advisory Quota Value." + 
              "Should be either an Integer, or a decimal value followed by one of (M,MB,G,GB,T,TB,P,PB)")
                              .setField(ENTITY_ADVISORY_QUOTA_PARAM_NAME)
                              .setFieldValue(quotaStr));
          return;
        }
        propsBuilder.setQuotaAdvisorySizeMB(quota);
        isModify = true;        
      }
      
      if (!isModify) {
        // Nothing to be modified
        return;
      }
      
      AeProperties props = propsBuilder.build();
      AeModifyRequest.Builder reqBuilder = 
      	AeModifyRequest.newBuilder().setCreds(getUserCredentials());
      
      for (int i = 0; i < names.size(); i++) {
      	AeKey key = AeKey.newBuilder().setName(names.get(i)).setType(types.get(i)).build();
      	AeProperties newprops = AeProperties.newBuilder(props).setAeKey(key).build();        
        reqBuilder.addAes(newprops);
      }

      // Then query CLDB to fetch Entity quota + static-email information
      try {
        byte[] replyData;
        if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
      	  replyData = CLDBRpcCommonUtils.getInstance()
      	     .sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
      			  Common.MapRProgramId.CldbProgramId.getNumber(),
      			CLDBProto.CLDBProg.AeModifyProc.getNumber(), reqBuilder.build(), AeModifyResponse.class);
        } else {
      	  replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
      			  Common.MapRProgramId.CldbProgramId.getNumber(),
      			CLDBProto.CLDBProg.AeModifyProc.getNumber(), reqBuilder.build(), AeModifyResponse.class);
        }
 
        if (replyData == null) {
         /**
          * <MAPR_ERROR>
          * Message:ERROR Contacting CLDB
          * Function:EntityCommands.ModifyEntity()
          * Meaning:An expected reply did not come back from the CLDB.
          * Resolution:Contact technical support.
          * </MAPR_ERROR>
          */
          out.addError(new OutputError(Errno.EOPFAILED, "ERROR Contacting CLDB"));
          return;
        }
        
        int status = AeModifyResponse.parseFrom(replyData).getStatus();
        if (status != 0) {
         /**
          * <MAPR_ERROR>
          * Message:ERROR Modifying Entity: <error>
          * Function:EntityCommands.ModifyEntity()
          * Meaning:An error occurred while attempting to modify the specified entity.
          * Resolution:Contact technical support.
          * </MAPR_ERROR>
          */
          out.addError(new OutputError(status, 
              "ERROR Modifying Entity: " + Errno.toString(status)));
          return;
        }
             
      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
      } catch (Exception e) {
       /**
        * <MAPR_ERROR>
        * Message:Exception: <exception>
        * Function:EntityCommands.ModifyEntity()
        * Meaning:An exception occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        out.addError(new OutputError(Errno.EOPFAILED, "ERROR"));
        LOG.error("Exception: " + e.getLocalizedMessage());
      }
    }

    void ModifyEntity (OutputHierarchy out, String cmd) 
      throws CLIProcessingException {

      if (isParamPresent(ENTITIES_PARAM_NAME)) {
        String names = getParamTextValue(ENTITIES_PARAM_NAME, 0);
        if ((names == null) || names.isEmpty()) {
          /**
           * <MAPR_ERROR>
           * Message:Invalid Entities
           * Function:EntityCommands.ModifyEntity()
           * Meaning:The entities string was empty or NULL.
           * Resolution:Check the format of entities, and the command syntax, and try again.
           * </MAPR_ERROR>
           */
          out.addError(new OutputError(Errno.EINVAL, "Invalid Entities")
              .setField(ENTITIES_PARAM_NAME)
              .setFieldValue(names));
          return;
        }

        List<String> namesArray = new ArrayList<String> ();
        List<Boolean> typesArray = new ArrayList<Boolean> ();

        for (String str: names.split(",")) {
          if (str == null || str.isEmpty()) {
            continue;
          }
          if (str.contains(":")) {
            String [] s = str.split(":");
            if (s.length != 2)
              continue;
            if (s[0] == null || s[0].isEmpty() || s[1] == null || s[1].isEmpty())
              continue;
            if (s[0].equalsIgnoreCase("0"))
              typesArray.add(false);
            else if (s[0].equalsIgnoreCase("1"))
              typesArray.add(true);
            else
              continue;
            namesArray.add(s[1]);    				
          } else {
            namesArray.add(str);
            typesArray.add(false); // Default user
          }
        }

        // Check if there was no valid entity in the list
        if (namesArray.isEmpty()) {
          /**
           * <MAPR_ERROR>
           * Message:Invalid Entities
           * Function:EntityCommands.ModifyEntity()
           * Meaning:The entities string was empty or NULL.
           * Resolution:Check the format of entities, and the command syntax, and try again.
           * </MAPR_ERROR>
           */
          out.addError(new OutputError(Errno.EINVAL, "Invalid Entities, no single valid entity")
              .setField(ENTITIES_PARAM_NAME)
              .setFieldValue(names));
        } else {
          modify(out, namesArray, typesArray);
        }
      } else if (isParamPresent(ENTITY_NAME_PARAM_NAME)) {
        String name = getParamTextValue(ENTITY_NAME_PARAM_NAME, 0);
        if ((name == null) || name.isEmpty()) {
          /**
           * <MAPR_ERROR>
           * Message:Invalid Entity
           * Function:EntityCommands.ModifyEntity()
           * Meaning:The entity name was empty or NULL.
           * Resolution:Check the entity name and type, and the command syntax, and try again.
           * </MAPR_ERROR>
           */
          out.addError(new OutputError(Errno.EINVAL, "Invalid Entity")
              .setField(ENTITY_NAME_PARAM_NAME)
              .setFieldValue(name));
          return;
        }
        boolean type = getParamBooleanValue(ENTITY_TYPE_PARAM_NAME, 0);

        List<String> namesArray = new ArrayList<String> ();
        List<Boolean> typesArray = new ArrayList<Boolean> ();

        namesArray.add(name);
        typesArray.add(type);

        modify(out, namesArray, typesArray);
      } else {
        out.addError(new OutputError(Errno.EINVAL, "Need either " +
              ENTITY_NAME_PARAM_NAME + " or " + ENTITIES_PARAM_NAME));
      }
    }

    private void remove (OutputHierarchy out, List<String> names, List<Boolean> types)
      throws CLIProcessingException {
      boolean isModify = false;

      AeRemoveRequest.Builder reqBuilder = AeRemoveRequest.newBuilder().setCreds(getUserCredentials());

      for (int i = 0; i < names.size(); i++) {
        AeKey key = AeKey.newBuilder().setName(names.get(i)).setType(types.get(i)).build();
        reqBuilder.addAeKeys(key);
      }

      try {
        byte[] replyData;
        if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
          replyData = CLDBRpcCommonUtils.getInstance()
            .sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
                Common.MapRProgramId.CldbProgramId.getNumber(),
                CLDBProto.CLDBProg.AeRemoveProc.getNumber(), reqBuilder.build(), AeRemoveResponse.class);
        } else {
          replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
              Common.MapRProgramId.CldbProgramId.getNumber(),
              CLDBProto.CLDBProg.AeRemoveProc.getNumber(), reqBuilder.build(), AeRemoveResponse.class);
        }

        if (replyData == null) {
          out.addError(new OutputError(Errno.EOPFAILED, "ERROR Contacting CLDB"));
          return;
        }

        int status = AeRemoveResponse.parseFrom(replyData).getStatus();
        if (status != 0) {
          String msg = AeRemoveResponse.parseFrom(replyData).getErrMsg();
          if (msg == null) {
            msg = Errno.toString(status);
          }
          out.addError(new OutputError(status, "ERROR Removing Entity: " + msg));
          return;
        }

      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);
      } catch (Exception e) {
        out.addError(new OutputError(Errno.EOPFAILED, "ERROR"));
        LOG.error("Exception: " + e.getLocalizedMessage());
      }
    }

    void RemoveEntity (OutputHierarchy out, String cmd) throws CLIProcessingException {
      if (isParamPresent(ENTITIES_PARAM_NAME)) {
        String names = getParamTextValue(ENTITIES_PARAM_NAME, 0);
        if ((names == null) || names.isEmpty()) {
          out.addError(new OutputError(Errno.EINVAL, "Invalid Entities")
              .setField(ENTITIES_PARAM_NAME)
              .setFieldValue(names));
          return;
        }

        List<String> namesArray = new ArrayList<String> ();
        List<Boolean> typesArray = new ArrayList<Boolean> ();

        for (String str: names.split(",")) {
          if (str == null || str.isEmpty()) {
            continue;
          }
          if (str.contains(":")) {
            String [] s = str.split(":");
            if (s.length != 2)
              continue;
            if (s[0] == null || s[0].isEmpty() || s[1] == null || s[1].isEmpty())
              continue;
            if (s[0].equalsIgnoreCase("0"))
              typesArray.add(false);
            else if (s[0].equalsIgnoreCase("1"))
              typesArray.add(true);
            else
              continue;
            namesArray.add(s[1]);
          } else {
            namesArray.add(str);
            typesArray.add(false); // Default user
          }
        }

        // Check if there was no valid entity in the list
        if (namesArray.isEmpty()) {
          out.addError(new OutputError(Errno.EINVAL, "Invalid Entities, no single valid entity")
              .setField(ENTITIES_PARAM_NAME)
              .setFieldValue(names));
        } else {
          remove(out, namesArray, typesArray);
        }
      } else if (isParamPresent(ENTITY_NAME_PARAM_NAME)) {
        String name = getParamTextValue(ENTITY_NAME_PARAM_NAME, 0);
        if ((name == null) || name.isEmpty()) {
          out.addError(new OutputError(Errno.EINVAL, "Invalid Entity")
              .setField(ENTITY_NAME_PARAM_NAME)
              .setFieldValue(name));
          return;
        }
        if (!isParamPresent(ENTITY_TYPE_PARAM_NAME)) {
          out.addError(new OutputError(Errno.EINVAL, "Entity type not provided for entity")
              .setField(ENTITY_NAME_PARAM_NAME)
              .setFieldValue(name));
          return;
        }
        boolean type = getParamBooleanValue(ENTITY_TYPE_PARAM_NAME, 0);

        List<String> namesArray = new ArrayList<String> ();
        List<Boolean> typesArray = new ArrayList<Boolean> ();

        namesArray.add(name);
        typesArray.add(type);

        remove(out, namesArray, typesArray);
      } else {
        out.addError(new OutputError(Errno.EINVAL,
              "Need either " + ENTITY_NAME_PARAM_NAME + " or " + ENTITIES_PARAM_NAME));
      }
    }

    boolean checkEntityExists(boolean type, String name) {
        String cmd = null;
        
        if (type == false) {
          cmd = "getent passwd " + name;
        } else {
          cmd = "getent group " + name;
        }
        
        try {
           String result = (new TextCommandOutput(executeSimpleSHHCommand(6000L, cmd))).toString();
           if (result == null || result.isEmpty()) {
               return false;
           }
           return true;
        } catch (Exception e) {
           return false;
        }
    }
    
    private AeLookupRequest.Builder getAeLookupRequestBuilder() throws CLIProcessingException {
      AeLookupRequest.Builder reqBuilder = AeLookupRequest.newBuilder();
      
      if (isParamPresent(ALARMEDENTITIES_PARAM_NAME)) {
        boolean alarmedAes = getParamBooleanValue(ALARMEDENTITIES_PARAM_NAME, 0);
        if (alarmedAes) {
          reqBuilder.setAlarmedAes(true);
        }
      }
      
      reqBuilder.addAllFilter(getFilters(fieldTable, FILTER_PARAM_NAME));
      reqBuilder.setCreds(getUserCredentials());
      reqBuilder.setLimiter(getNextLimiter(getParamIntValue(START_PARAM_NAME, 0), 0, 
          getParamIntValue(START_PARAM_NAME, 0), getParamIntValue(LIMIT_PARAM_NAME, 0),
          NUM_ENTITIES_PER_RPC));
      
      return reqBuilder;
    }

    @Override
    public AeLookupRequest buildNextRequest(MessageLite prevReq, MessageLite prevResp) throws CLIProcessingException {
      AeLookupRequest.Builder newReqBuilder = null;
      if (prevReq != null) {
        newReqBuilder = AeLookupRequest.newBuilder((AeLookupRequest) prevReq);
      } else {
        newReqBuilder = getAeLookupRequestBuilder();
      }
      
      if (prevResp != null) {
        int prevStart = newReqBuilder.getLimiter().getStart();
        int prevCount = ((AeLookupResponse) prevResp).getAePropertyCount();
        int origStart = getParamIntValue(START_PARAM_NAME, 0);
        int origLimit = getParamIntValue(LIMIT_PARAM_NAME, 0);
        newReqBuilder.setLimiter(getNextLimiter(prevStart, prevCount, 
            origStart, origLimit, NUM_ENTITIES_PER_RPC));
      }
      
      return newReqBuilder.build();
    }
    
    @Override
    public boolean hasMore(MessageLite prevReq, MessageLite prevResp) throws CLIProcessingException {
      return hasMore(getParamIntValue(START_PARAM_NAME, 0), 
          getParamIntValue(LIMIT_PARAM_NAME, 0), 
          ((AeLookupRequest) prevReq).getLimiter().getStart(), 
          ((AeLookupResponse) prevResp).getAePropertyCount());
    }

    @Override
    public void processResponse(OutputHierarchy out, MessageLite response) throws CLIProcessingException {
      AeLookupResponse resp = (AeLookupResponse) response;
      boolean terse = getParamTextValue(OUTPUT_PARAM_NAME, 0).equalsIgnoreCase("terse");
      
      if (resp.hasTotal()) {
        out.setTotal(resp.getTotal());
      }
           
      for (AeProperties props : resp.getAePropertyList()) {
       OutputNode node = new OutputNode();          
       boolean type = props.getAeKey().getType();
       String name = props.getAeKey().getName();
       node.addChild(new OutputNode(
            fieldTable.get(AeInfoFields.aetype).getName(terse), 
            type ? 1 : 0));
       node.addChild(new OutputNode(
           fieldTable.get(AeInfoFields.aename).getName(terse), 
           name));             
       node.addChild(new OutputNode(
           fieldTable.get(AeInfoFields.aevolumecount).getName(terse),
           props.getAeVolumeCount()));
       node.addChild(new OutputNode(
           fieldTable.get(AeInfoFields.aequota).getName(terse), 
           props.getQuotaSizeMB()));
       node.addChild(new OutputNode(
           fieldTable.get(AeInfoFields.aeadvisoryquota).getName(terse), 
           props.getQuotaAdvisorySizeMB()));
       node.addChild(new OutputNode(
           fieldTable.get(AeInfoFields.aediskusage).getName(terse),
           props.getUsedSizeMB()));
     
       // If message has statically configured email use that
       if (props.hasEmail()) {
          node.addChild(new OutputNode(
              fieldTable.get(AeInfoFields.aeemail).getName(terse), 
              props.getEmail()));
       } else {
          String email = MapRCliUtil.fetchEmail(CLDBProperties, name, type);
          if (email != null)
            node.addChild(new OutputNode(
                fieldTable.get(AeInfoFields.aeemail).getName(terse), 
                email));
       }      
       
       for (AlarmMsg alarm: props.getAeAlarmsList()) {
         if (alarm.getAlarmId() == AlarmId.AE_ALARM_AEQUOTA_EXCEEDED) {
           node.addChild(new OutputNode(
               fieldTable.get(AeInfoFields.aequotaalarm).getName(terse), 
               alarm.getAlarmTimeStamp()));
         } else if (alarm.getAlarmId() == AlarmId.AE_ALARM_AEADVISORY_QUOTA_EXCEEDED) {
           node.addChild(new OutputNode(
               fieldTable.get(AeInfoFields.aeadvisoryquotaalarm).getName(terse), 
               alarm.getAlarmTimeStamp()));
         }
       }       
              
       try {
         // Fetch uid/gid from getent
         byte [] b = executeSimpleSHHCommand(6000L, 
                          "getent " + (type ? "group " : "passwd ") + name);
         if (b != null) {
           TextCommandOutput txt = new TextCommandOutput(b);
           String[] info = txt.toString().split(":");
           node.addChild(new OutputNode(
               fieldTable.get(AeInfoFields.aeid).getName(terse), 
               Integer.valueOf(info[2])));
         }
       } catch (Exception e) {
        /**
         * <MAPR_ERROR>
         * Message:Could not fetch entity uid/gid <message>
         * Function:EntityCommands.processResponse()
         * Meaning:An exception occurred while attempting to retrieve the specified entity.
         * Resolution:Contact technical support.
         * </MAPR_ERROR>
         */
         LOG.error("Could not fetch entity uid/gid " + e.getLocalizedMessage());
       }
       
       out.addNode(node);
      }
    }

    @Override
    public AeLookupResponse sendRequest(MessageLite request) throws CLIProcessingException {
      AeLookupRequest req = (AeLookupRequest) request;
      byte[] replyData = null;
      try {
        if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
          replyData =  CLDBRpcCommonUtils.getInstance().sendRequest(
                          getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
                          Common.MapRProgramId.CldbProgramId.getNumber(),
                          CLDBProto.CLDBProg.AeLookupProc.getNumber(), 
                          req, AeLookupResponse.class);
        } else {
          replyData =  CLDBRpcCommonUtils.getInstance().sendRequest(
                          Common.MapRProgramId.CldbProgramId.getNumber(),
                          CLDBProto.CLDBProg.AeLookupProc.getNumber(), 
                          req, AeLookupResponse.class);
        }
      } catch (Exception e) {
        throw new CLIProcessingException(e);
      }
      
      if (replyData != null) {
        return getAeLookupResponse(replyData);
      } else {
       /**
        * <MAPR_ERROR>
        * Message:RPC Request to list Nodes failed. No data returned
        * Function:EntityCommands.sendRequest()
        * Meaning:A communication error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        LOG.error("RPC Request to list Nodes failed. No data returned");
        return null;
      }
    }
    
    private AeLookupResponse getAeLookupResponse(byte[] replyData) throws CLIProcessingException {
      try {
        return AeLookupResponse.parseFrom(replyData);
      } catch (InvalidProtocolBufferException ipbe) {
      /**
       * <MAPR_ERROR>
       * Message:Exception while parsing the RPC response data into FileServerListResponse proto object
       * Function:EntityCommands.getAeLookupResponse()
       * Meaning:An exception occurred.
       * Resolution:Contact technical support.
       * </MAPR_ERROR>
       */
        throw new CLIProcessingException("Exception while parsing the RPC " +
            "response data into FileServerListResponse proto object.", ipbe);
      }
    }
   
}
