/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.cli;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.ByteString;
import com.mapr.baseutils.BinaryString;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cli.common.FileclientRun;
import com.mapr.cli.common.JobExecutor;
import com.mapr.cli.table.RecentTablesListManager;
import com.mapr.cli.table.RecentTablesListManagers;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.db.MapRDB;
import com.mapr.db.impl.MapRDBTableImpl;
import com.mapr.db.exceptions.TableNotFoundException;
import com.mapr.fs.AceHelper;
import com.mapr.fs.MapRFileSystem;
import com.mapr.fs.MapRHTable;
import com.mapr.fs.Rpc;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.ContainerLookupRequest;
import com.mapr.fs.cldb.proto.CLDBProto.ContainerLookupResponse;
import com.mapr.fs.cldb.util.Util;
import com.mapr.fs.proto.Common.FidMsg;
import com.mapr.fs.proto.Common.FileCompressionType;
import com.mapr.fs.proto.Common.MapRProgramId;
import com.mapr.fs.proto.Common.Server;
import com.mapr.fs.proto.Dbserver.ColumnFamilyAttr;
import com.mapr.fs.proto.Dbserver.Qualifier;
import com.mapr.fs.proto.Dbserver.TableReplicaDesc;
import com.mapr.fs.proto.Dbserver.TableReplicaListResponse;
import com.mapr.fs.proto.Dbserver.TableReplicaStatus;
import com.mapr.fs.proto.Error.ExtendedError;
import com.mapr.fs.proto.Fileserver.FSProg;
import com.mapr.fs.proto.Fileserver.GetattrRequest;
import com.mapr.fs.proto.Fileserver.GetattrResponse;
import com.mapr.fs.proto.Security.ServerKeyType;
import com.mapr.security.MaprSecurityException;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.fs.permission.FsAction;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.log4j.Logger;
import org.yaml.snakeyaml.Yaml;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.SortedSet;
import java.util.TreeMap;

public class ElasticsearchCommands extends CLIBaseClass implements CLIInterface,AceHelper.DBPermission {
  private static final Logger LOG = Logger.getLogger(ElasticsearchCommands.class);

  private static final String NEW_LN = System.lineSeparator();
  private static final String COLUMN_SEP = ":";

  // ES files and directories
  private static final String ES_HOME =
    File.separator + "opt" + File.separator + "external" + File.separator + "elasticsearch";
  private static final String ES_REPLICAS = ES_HOME + File.separator + "replicas";
  private static final String ES_CLUSTERS = ES_HOME + File.separator + "clusters";
  private static final String LIB_DIR = "lib";
  private static final String CONF_DIR = "config";
  private static final String PLUGIN_DIR = "plugins";
  private static final String CONF_FILE = "elasticsearch.yml";
  private static final String TRANSPORT_NODE_FILE = "transport.yml";
  private static final String REPLICA_FILE = "config.es";
  private static final String TMP_REPLICA_FILE = "tmp.es";

  // Command hierarchy name
  private static final String REPLICA_FAMILY = "table replica elasticsearch";
  private static final String TARGET_FAMILY = "target elasticsearch";

  // Command names
  private static final String AUTOSETUP_CMD_NAME = "autosetup";
  private static final String ADD_CMD_NAME = "add";
  private static final String LIST_CMD_NAME = "list";
  private static final String EDIT_CMD_NAME = "edit";
  private static final String REMOVE_CMD_NAME = "remove";
  private static final String PAUSE_CMD_NAME = "pause";
  private static final String RESUME_CMD_NAME = "resume";
  private static final String CLEANUP_CMD_NAME = "cleanup";

  // Param names
  private static final String PATH_PARAM_NAME = "path";
  private static final String TARGET_PARAM_NAME = "target";
  private static final String ESCLUSTER_PARAM_NAME = "name";
  private static final String ESINDEX_PARAM_NAME = "index";
  private static final String ESTYPE_PARAM_NAME = "type";
  private static final String ESCONFIG_PARAM_NAME = "clusterconfig";
  private static final String ESLIB_PARAM_NAME = "lib";
  private static final String ESPLUGIN_PARAM_NAME = "plugins";
  private static final String ESTRANSPORT_PARAM_NAME = "transportnode";
  private static final String CONVCLASS_PARAM_NAME = "conversionclass";
  private static final String CONVJAR_PARAM_NAME = "conversionjar";
  private static final String COLUMNS_PARAM_NAME = "columns";
  private static final String INDEXED_COLUMNS_PARAM_NAME = "indexedcolumns";
  private static final String PAUSED_PARAM_NAME = "paused";
  private static final String THROTTLE_PARAM_NAME = "throttle";
  private static final String ENCRYPTION_PARAM_NAME = "networkencryption";
  private static final String COMPRESSION_PARAM_NAME = "networkcompression";
  private static final String REFRESH_PARAM_NAME = "refreshnow";
  private static final String TICKET_PARAM_NAME = "ticketpath";

  // Fields in yaml config files
  private static final String YAML_SINK_CLASSPATH = "sink.class.path";
  private static final String YAML_ES_TARGET = "es.target.name";
  private static final String YAML_ES_INDEX = "es.index.name";
  private static final String YAML_ES_TYPE = "es.index.type";
  private static final String YAML_CONV_CLASSPATH = "es.conversion.class.path";
  private static final String YAML_CONV_CLASS_NAME = "es.conversion.class.name";

  private static final CLICommand replicaSetupCommand =
    new CLICommand(
        AUTOSETUP_CMD_NAME,
        String.format("usage: %s %s -%s <srcpath> " +
            "-%s <escluster> -%s <esindex> -%s <estype> " +
            "-%s <conversionClass> -%s <conversionJar> " +
            "-%s <cf1[:col1],cf2,..> " +
            "-%s <cf1:col1,cf2:col2,..> " +
            "-%s <true|false> -%s <true|false> -%s <off|on|lzf|lz4|zlib>",
            REPLICA_FAMILY, AUTOSETUP_CMD_NAME, PATH_PARAM_NAME, TARGET_PARAM_NAME,
            ESINDEX_PARAM_NAME, ESTYPE_PARAM_NAME, CONVCLASS_PARAM_NAME,
            CONVJAR_PARAM_NAME, COLUMNS_PARAM_NAME, INDEXED_COLUMNS_PARAM_NAME, THROTTLE_PARAM_NAME,
            ENCRYPTION_PARAM_NAME, COMPRESSION_PARAM_NAME),
        ElasticsearchCommands.class,
        CLICommand.ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>()
            .put(PATH_PARAM_NAME,
                new TextInputParameter(PATH_PARAM_NAME,
                    "source table path",
                    CLIBaseClass.REQUIRED,
                    null))
            .put(TARGET_PARAM_NAME,
                new TextInputParameter(TARGET_PARAM_NAME,
                    "Target cluster name",
                    CLIBaseClass.REQUIRED,
                    null))
            .put(ESINDEX_PARAM_NAME,
                new TextInputParameter(ESINDEX_PARAM_NAME,
                    "Elasticsearch index name",
                    CLIBaseClass.REQUIRED,
                    null))
            .put(ESTYPE_PARAM_NAME,
                new TextInputParameter(ESTYPE_PARAM_NAME,
                    "Elasticsearch type name",
                    CLIBaseClass.REQUIRED,
                    null))
            .put(CONVCLASS_PARAM_NAME,
                new TextInputParameter(CONVCLASS_PARAM_NAME,
                    "Conversion class name",
                    CLIBaseClass.NOT_REQUIRED,
                    null))
            .put(CONVJAR_PARAM_NAME,
                new TextInputParameter(CONVJAR_PARAM_NAME,
                    "Path to conversion class JAR file",
                    CLIBaseClass.NOT_REQUIRED,
                    null))
            .put(COLUMNS_PARAM_NAME,
                new TextInputParameter(COLUMNS_PARAM_NAME,
                    "comma separated list of <family>[:<column>]",
                    CLIBaseClass.NOT_REQUIRED,
                    null))
            .put(INDEXED_COLUMNS_PARAM_NAME,
                new TextInputParameter(INDEXED_COLUMNS_PARAM_NAME,
                    "comma separated list of <family>:<column> pairs which are indexed",
                    CLIBaseClass.NOT_REQUIRED,
                    null))
            .put(THROTTLE_PARAM_NAME,
                new BooleanInputParameter(THROTTLE_PARAM_NAME,
                    "throttle replication ops",
                    CLIBaseClass.NOT_REQUIRED,
                    false))
            .put(ENCRYPTION_PARAM_NAME,
                new BooleanInputParameter(ENCRYPTION_PARAM_NAME,
                    "enable on-wire encryption",
                    CLIBaseClass.NOT_REQUIRED,
                    false))
            .put(COMPRESSION_PARAM_NAME,
                new TextInputParameter(COMPRESSION_PARAM_NAME,
                    "on-wire compression type: off|on|lzf|lz4|zlib",
                    CLIBaseClass.NOT_REQUIRED,
                    "on"))
            .put(TICKET_PARAM_NAME,
                new TextInputParameter(TICKET_PARAM_NAME,
                    "ticket path",
                    CLIBaseClass.NOT_REQUIRED,
                    null).setInvisible(true))
            .build(), null)
        .setShortUsage(String.format("usage: %s %s -%s <srcpath> -%s <escluster> -%s <esindex> -%s <estype>",
            REPLICA_FAMILY, AUTOSETUP_CMD_NAME, PATH_PARAM_NAME, TARGET_PARAM_NAME,
            ESINDEX_PARAM_NAME, ESTYPE_PARAM_NAME));

  private static final CLICommand replicaAddCommand =
    new CLICommand(
        ADD_CMD_NAME,
        String.format("usage: %s %s -%s <srcpath> " +
            "-%s <escluster> -%s <esindex> -%s <estype> " +
            "-%s <conversionClass> -%s <conversionJar> " +
            "-%s <cf1[:col1],cf2,..> -%s <true|false> " +
            "-%s <true|false> -%s <true|false> -%s <off|on|lzf|lz4|zlib>",
            REPLICA_FAMILY, ADD_CMD_NAME, PATH_PARAM_NAME, TARGET_PARAM_NAME,
            ESINDEX_PARAM_NAME, ESTYPE_PARAM_NAME, CONVCLASS_PARAM_NAME,
            CONVJAR_PARAM_NAME, COLUMNS_PARAM_NAME, PAUSED_PARAM_NAME,
            THROTTLE_PARAM_NAME, ENCRYPTION_PARAM_NAME, COMPRESSION_PARAM_NAME),
        ElasticsearchCommands.class,
        CLICommand.ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>()
            .put(PATH_PARAM_NAME,
                new TextInputParameter(PATH_PARAM_NAME,
                    "source table path",
                    CLIBaseClass.REQUIRED,
                    null))
            .put(TARGET_PARAM_NAME,
                new TextInputParameter(TARGET_PARAM_NAME,
                    "Target cluster name",
                    CLIBaseClass.REQUIRED,
                    null))
            .put(ESINDEX_PARAM_NAME,
                new TextInputParameter(ESINDEX_PARAM_NAME,
                    "Elasticsearch index name",
                    CLIBaseClass.REQUIRED,
                    null))
            .put(ESTYPE_PARAM_NAME,
                new TextInputParameter(ESTYPE_PARAM_NAME,
                    "Elasticsearch type name",
                    CLIBaseClass.REQUIRED,
                    null))
            .put(CONVCLASS_PARAM_NAME,
                new TextInputParameter(CONVCLASS_PARAM_NAME,
                    "Conversion class name",
                    CLIBaseClass.NOT_REQUIRED,
                    null))
            .put(CONVJAR_PARAM_NAME,
                new TextInputParameter(CONVJAR_PARAM_NAME,
                    "Path to conversion class JAR file",
                    CLIBaseClass.NOT_REQUIRED,
                    null))
            .put(COLUMNS_PARAM_NAME,
                new TextInputParameter(COLUMNS_PARAM_NAME,
                    "comma separated list of <family>[:<column>]",
                    CLIBaseClass.NOT_REQUIRED,
                    null))
            .put(PAUSED_PARAM_NAME,
                new BooleanInputParameter(PAUSED_PARAM_NAME,
                    "is replication paused",
                    CLIBaseClass.NOT_REQUIRED,
                    false))
            .put(THROTTLE_PARAM_NAME,
                new BooleanInputParameter(THROTTLE_PARAM_NAME,
                    "throttle replication ops",
                    CLIBaseClass.NOT_REQUIRED,
                    false))
            .put(ENCRYPTION_PARAM_NAME,
                new BooleanInputParameter(ENCRYPTION_PARAM_NAME,
                    "enable on-wire encryption",
                    CLIBaseClass.NOT_REQUIRED,
                    false))
            .put(COMPRESSION_PARAM_NAME,
                new TextInputParameter(COMPRESSION_PARAM_NAME,
                    "on-wire compression type: off|on|lzf|lz4|zlib",
                    CLIBaseClass.NOT_REQUIRED,
                    "on"))
            .put(TICKET_PARAM_NAME,
                new TextInputParameter(TICKET_PARAM_NAME,
                    "ticket path",
                    CLIBaseClass.NOT_REQUIRED,
                    null).setInvisible(true))
            .build(), null)
        .setUsageInVisible(true)
        .setShortUsage(String.format("usage: %s %s -%s <srcpath> -%s <escluster> -%s <esindex> -%s <estype>",
            REPLICA_FAMILY, ADD_CMD_NAME, PATH_PARAM_NAME, TARGET_PARAM_NAME,
            ESINDEX_PARAM_NAME, ESTYPE_PARAM_NAME));

  private static final CLICommand replicaEditCommand =
    new CLICommand(
        EDIT_CMD_NAME,
        String.format("usage: %s %s -%s <srcpath> " +
            "-%s <escluster> -%s <esindex> -%s <estype> " +
            "-%s <conversionClass> -%s <conversionJar> " +
            "-%s <cf1[:col1],cf2,..> " +
            "-%s <cf1:col1,cf2:col2,..> " +
            "-%s <true|false> -%s <true|false> -%s <off|on|lzf|lz4|zlib>",
            REPLICA_FAMILY, EDIT_CMD_NAME, PATH_PARAM_NAME, TARGET_PARAM_NAME,
            ESINDEX_PARAM_NAME, ESTYPE_PARAM_NAME, CONVCLASS_PARAM_NAME,
            CONVJAR_PARAM_NAME, COLUMNS_PARAM_NAME, INDEXED_COLUMNS_PARAM_NAME, THROTTLE_PARAM_NAME,
            ENCRYPTION_PARAM_NAME, COMPRESSION_PARAM_NAME),
        ElasticsearchCommands.class,
        CLICommand.ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>()
            .put(PATH_PARAM_NAME,
                new TextInputParameter(PATH_PARAM_NAME,
                    "source table path",
                    CLIBaseClass.REQUIRED,
                    null))
            .put(TARGET_PARAM_NAME,
                new TextInputParameter(TARGET_PARAM_NAME,
                    "Elasticsearch cluster name",
                    CLIBaseClass.REQUIRED,
                    null))
            .put(ESINDEX_PARAM_NAME,
                new TextInputParameter(ESINDEX_PARAM_NAME,
                    "Elasticsearch index name",
                    CLIBaseClass.REQUIRED,
                    null))
            .put(ESTYPE_PARAM_NAME,
                new TextInputParameter(ESTYPE_PARAM_NAME,
                    "Elasticsearch type name",
                    CLIBaseClass.REQUIRED,
                    null))
            .put(CONVCLASS_PARAM_NAME,
                new TextInputParameter(CONVCLASS_PARAM_NAME,
                    "Conversion class name",
                    CLIBaseClass.NOT_REQUIRED,
                    null))
            .put(CONVJAR_PARAM_NAME,
                new TextInputParameter(CONVJAR_PARAM_NAME,
                    "Path to conversion class JAR file",
                    CLIBaseClass.NOT_REQUIRED,
                    null))
            .put(COLUMNS_PARAM_NAME,
                new TextInputParameter(COLUMNS_PARAM_NAME,
                    "comma separated list of <family>[:<column>]",
                    CLIBaseClass.NOT_REQUIRED,
                    null))
            .put(INDEXED_COLUMNS_PARAM_NAME,
                new TextInputParameter(INDEXED_COLUMNS_PARAM_NAME,
                    "comma separated list of <family>:<column> pairs which are indexed",
                    CLIBaseClass.NOT_REQUIRED,
                    null))
            .put(THROTTLE_PARAM_NAME,
                new BooleanInputParameter(THROTTLE_PARAM_NAME,
                    "throttle replication ops",
                    CLIBaseClass.NOT_REQUIRED,
                    false))
            .put(ENCRYPTION_PARAM_NAME,
                new BooleanInputParameter(ENCRYPTION_PARAM_NAME,
                    "enable on-wire encryption",
                    CLIBaseClass.NOT_REQUIRED,
                    false))
            .put(COMPRESSION_PARAM_NAME,
                new TextInputParameter(COMPRESSION_PARAM_NAME,
                    "on-wire compression type: off|on|lzf|lz4|zlib",
                    CLIBaseClass.NOT_REQUIRED,
                    "on"))
            .build(), null)
        .setShortUsage(String.format("usage: %s %s -%s <srcpath> -%s <escluster> -%s <esindex> -%s <estype>",
            REPLICA_FAMILY, EDIT_CMD_NAME, PATH_PARAM_NAME, TARGET_PARAM_NAME,
            ESINDEX_PARAM_NAME, ESTYPE_PARAM_NAME));

  private static final CLICommand replicaListCommand =
    new CLICommand(
        LIST_CMD_NAME,
        String.format("usage: %s %s -%s <srcpath> -%s <true|false>",
            REPLICA_FAMILY, LIST_CMD_NAME, PATH_PARAM_NAME, REFRESH_PARAM_NAME),
        ElasticsearchCommands.class,
        CLICommand.ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>()
            .put(PATH_PARAM_NAME,
                new TextInputParameter(PATH_PARAM_NAME,
                    "source table path",
                    CLIBaseClass.REQUIRED,
                    null))
            .put(REFRESH_PARAM_NAME,
                new BooleanInputParameter(REFRESH_PARAM_NAME,
                    "refresh now",
                    CLIBaseClass.NOT_REQUIRED,
                    false))
            .build(), null)
        .setShortUsage(String.format("usage: %s %s -%s <srcpath>",
            REPLICA_FAMILY, LIST_CMD_NAME, PATH_PARAM_NAME));

  private static final CLICommand replicaPauseCommand =
    new CLICommand(
        PAUSE_CMD_NAME,
        String.format("usage: %s %s -%s <srcpath> -%s <escluster> -%s <esindex> -%s <estype>",
            REPLICA_FAMILY, PAUSE_CMD_NAME, PATH_PARAM_NAME, TARGET_PARAM_NAME,
            ESINDEX_PARAM_NAME, ESTYPE_PARAM_NAME),
        ElasticsearchCommands.class,
        CLICommand.ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>()
            .put(PATH_PARAM_NAME,
                new TextInputParameter(PATH_PARAM_NAME,
                    "source table path",
                    CLIBaseClass.REQUIRED,
                    null))
            .put(TARGET_PARAM_NAME,
                new TextInputParameter(TARGET_PARAM_NAME,
                    "Elasticsearch cluster name",
                    CLIBaseClass.REQUIRED,
                    null))
            .put(ESINDEX_PARAM_NAME,
                new TextInputParameter(ESINDEX_PARAM_NAME,
                    "Elasticsearch index name",
                    CLIBaseClass.REQUIRED,
                    null))
            .put(ESTYPE_PARAM_NAME,
                new TextInputParameter(ESTYPE_PARAM_NAME,
                    "Elasticsearch type name",
                    CLIBaseClass.REQUIRED,
                    null))
            .build(), null)
        .setShortUsage(String.format("usage: %s %s -%s <srcpath> -%s <escluster> -%s <esindex> -%s <estype>",
            REPLICA_FAMILY, PAUSE_CMD_NAME, PATH_PARAM_NAME, TARGET_PARAM_NAME,
            ESINDEX_PARAM_NAME, ESTYPE_PARAM_NAME));

  private static final CLICommand replicaResumeCommand =
    new CLICommand(
        RESUME_CMD_NAME,
        String.format("usage: %s %s -%s <srcpath> -%s <escluster> -%s <esindex> -%s <estype>",
            REPLICA_FAMILY, RESUME_CMD_NAME, PATH_PARAM_NAME, TARGET_PARAM_NAME,
            ESINDEX_PARAM_NAME, ESTYPE_PARAM_NAME),
        ElasticsearchCommands.class,
        CLICommand.ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>()
            .put(PATH_PARAM_NAME,
                new TextInputParameter(PATH_PARAM_NAME,
                    "source table path",
                    CLIBaseClass.REQUIRED,
                    null))
            .put(TARGET_PARAM_NAME,
                new TextInputParameter(TARGET_PARAM_NAME,
                    "Elasticsearch cluster name",
                    CLIBaseClass.REQUIRED,
                    null))
            .put(ESINDEX_PARAM_NAME,
                new TextInputParameter(ESINDEX_PARAM_NAME,
                    "Elasticsearch index name",
                    CLIBaseClass.REQUIRED,
                    null))
            .put(ESTYPE_PARAM_NAME,
                new TextInputParameter(ESTYPE_PARAM_NAME,
                    "Elasticsearch type name",
                    CLIBaseClass.REQUIRED,
                    null))
            .build(), null)
        .setShortUsage(String.format("usage: %s %s -%s <srcpath> -%s <escluster> -%s <esindex> -%s <estype>",
            REPLICA_FAMILY, RESUME_CMD_NAME, PATH_PARAM_NAME, TARGET_PARAM_NAME,
            ESINDEX_PARAM_NAME, ESTYPE_PARAM_NAME));

  private static final CLICommand replicaRemoveCommand =
    new CLICommand(
        REMOVE_CMD_NAME,
        String.format("usage: %s %s -%s <srcpath> -%s <escluster> -%s <esindex> -%s <estype>",
            REPLICA_FAMILY, REMOVE_CMD_NAME, PATH_PARAM_NAME, TARGET_PARAM_NAME,
            ESINDEX_PARAM_NAME, ESTYPE_PARAM_NAME),
        ElasticsearchCommands.class,
        CLICommand.ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>()
            .put(PATH_PARAM_NAME,
                new TextInputParameter(PATH_PARAM_NAME,
                    "source table path",
                    CLIBaseClass.REQUIRED,
                    null))
            .put(TARGET_PARAM_NAME,
                new TextInputParameter(TARGET_PARAM_NAME,
                    "Elasticsearch cluster name",
                    CLIBaseClass.REQUIRED,
                    null))
            .put(ESINDEX_PARAM_NAME,
                new TextInputParameter(ESINDEX_PARAM_NAME,
                    "Elasticsearch index name",
                    CLIBaseClass.REQUIRED,
                    null))
            .put(ESTYPE_PARAM_NAME,
                new TextInputParameter(ESTYPE_PARAM_NAME,
                    "Elasticsearch type name",
                    CLIBaseClass.REQUIRED,
                    null))
            .build(), null)
        .setShortUsage(String.format("usage: %s %s -%s <srcpath> -%s <escluster> -%s <esindex> -%s <estype>",
            REPLICA_FAMILY, REMOVE_CMD_NAME, PATH_PARAM_NAME, TARGET_PARAM_NAME,
            ESINDEX_PARAM_NAME, ESTYPE_PARAM_NAME));

  private static final CLICommand replicaCleanupCommand =
    new CLICommand(
        CLEANUP_CMD_NAME,
        String.format("usage: %s %s", REPLICA_FAMILY, CLEANUP_CMD_NAME),
        ElasticsearchCommands.class,
        CLICommand.ExecutionTypeEnum.NATIVE,
        null, null)
    .setUsageInVisible(true)
    .setShortUsage(String.format("usage: %s %s", REPLICA_FAMILY, CLEANUP_CMD_NAME));

  public static final CLICommand esReplicaCommands =
      new CLICommand(
          "elasticsearch",
          String.format("elasticsearch [%s|%s|%s|%s|%s|%s|%s]", AUTOSETUP_CMD_NAME,
            ADD_CMD_NAME, EDIT_CMD_NAME, LIST_CMD_NAME, PAUSE_CMD_NAME, RESUME_CMD_NAME,
            REMOVE_CMD_NAME),
          CLIUsageOnlyCommand.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new CLICommand[] {
              replicaSetupCommand,
              replicaAddCommand,
              replicaEditCommand,
              replicaListCommand,
              replicaPauseCommand,
              replicaResumeCommand,
              replicaRemoveCommand,
              replicaCleanupCommand
          }
      ).setShortUsage(String.format("%s [%s|%s|%s|%s|%s|%s|%s]", REPLICA_FAMILY,
          AUTOSETUP_CMD_NAME, ADD_CMD_NAME, EDIT_CMD_NAME, LIST_CMD_NAME, PAUSE_CMD_NAME,
          RESUME_CMD_NAME, REMOVE_CMD_NAME));

  private static final CLICommand targetAddCommand =
    new CLICommand(
        ADD_CMD_NAME,
        String.format("usage: %s %s -%s <clusterName> " +
            "-%s <clusterConfig> -%s <libDir> -%s <pluginDir> -%s <transportNode>",
            TARGET_FAMILY, ADD_CMD_NAME, ESCLUSTER_PARAM_NAME, ESCONFIG_PARAM_NAME,
            ESLIB_PARAM_NAME, ESPLUGIN_PARAM_NAME, ESTRANSPORT_PARAM_NAME),
        ElasticsearchCommands.class,
        CLICommand.ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>()
            .put(ESCLUSTER_PARAM_NAME,
                new TextInputParameter(ESCLUSTER_PARAM_NAME,
                    "Elasticsearch cluster name",
                    CLIBaseClass.REQUIRED,
                    null))
            .put(ESCONFIG_PARAM_NAME,
                new TextInputParameter(ESCONFIG_PARAM_NAME,
                    "path to Elasticsearch cluster config file",
                    CLIBaseClass.REQUIRED,
                    null))
            .put(ESLIB_PARAM_NAME,
                new TextInputParameter(ESLIB_PARAM_NAME,
                    "path to Elasticsearch JAR files",
                    CLIBaseClass.REQUIRED,
                    null))
            .put(ESPLUGIN_PARAM_NAME,
                new TextInputParameter(ESPLUGIN_PARAM_NAME,
                    "path to Elasticsearch plugin JAR files",
                    CLIBaseClass.NOT_REQUIRED,
                    null))
            .put(ESTRANSPORT_PARAM_NAME,
                new TextInputParameter(ESTRANSPORT_PARAM_NAME,
                    "name[:port] of Elasticsearch transport node",
                    CLIBaseClass.NOT_REQUIRED,
                    null))
            .build(), null)
        .setShortUsage(String.format("usage: %s %s -%s <es cluster name>",
            TARGET_FAMILY, ADD_CMD_NAME, ESCLUSTER_PARAM_NAME));

  private static final CLICommand targetEditCommand =
    new CLICommand(
        EDIT_CMD_NAME,
        String.format("usage: %s %s -%s <clusterName> " +
            "-%s <clusterConfig> -%s <libDir> -%s <pluginDir> -%s <transportNode>",
            TARGET_FAMILY, EDIT_CMD_NAME, ESCLUSTER_PARAM_NAME, ESCONFIG_PARAM_NAME,
            ESLIB_PARAM_NAME, ESPLUGIN_PARAM_NAME, ESTRANSPORT_PARAM_NAME),
        ElasticsearchCommands.class,
        CLICommand.ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>()
            .put(ESCLUSTER_PARAM_NAME,
                new TextInputParameter(ESCLUSTER_PARAM_NAME,
                    "Elasticsearch cluster name",
                    CLIBaseClass.REQUIRED,
                    null))
            .put(ESCONFIG_PARAM_NAME,
                new TextInputParameter(ESCONFIG_PARAM_NAME,
                    "path to Elasticsearch cluster config file",
                    CLIBaseClass.NOT_REQUIRED,
                    null))
            .put(ESLIB_PARAM_NAME,
                new TextInputParameter(ESLIB_PARAM_NAME,
                    "path to Elasticsearch JAR files",
                    CLIBaseClass.NOT_REQUIRED,
                    null))
            .put(ESPLUGIN_PARAM_NAME,
                new TextInputParameter(ESPLUGIN_PARAM_NAME,
                    "path to Elasticsearch plugin JAR files",
                    CLIBaseClass.NOT_REQUIRED,
                    null))
            .put(ESTRANSPORT_PARAM_NAME,
                new TextInputParameter(ESTRANSPORT_PARAM_NAME,
                    "name[:port] of Elasticsearch transport node",
                    CLIBaseClass.NOT_REQUIRED,
                    null))
            .build(), null)
        .setShortUsage(String.format("usage: %s %s -%s <es cluster name>",
            TARGET_FAMILY, EDIT_CMD_NAME, ESCLUSTER_PARAM_NAME));

  private static final CLICommand targetListCommand =
    new CLICommand(
        LIST_CMD_NAME,
        String.format("usage: %s %s", TARGET_FAMILY, LIST_CMD_NAME),
        ElasticsearchCommands.class,
        CLICommand.ExecutionTypeEnum.NATIVE,
        null, null)
        .setShortUsage(String.format("usage: %s %s", TARGET_FAMILY, LIST_CMD_NAME));

  private static final CLICommand targetRemoveCommand =
    new CLICommand(
        REMOVE_CMD_NAME,
        String.format("usage: %s %s -%s <clusterName> ",
            TARGET_FAMILY, REMOVE_CMD_NAME, ESCLUSTER_PARAM_NAME),
        ElasticsearchCommands.class,
        CLICommand.ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>()
            .put(ESCLUSTER_PARAM_NAME,
                new TextInputParameter(ESCLUSTER_PARAM_NAME,
                    "Elasticsearch cluster name",
                    CLIBaseClass.REQUIRED,
                    null))
            .build(), null)
        .setShortUsage(String.format("usage: %s %s -%s <es cluster name>",
            TARGET_FAMILY, REMOVE_CMD_NAME, ESCLUSTER_PARAM_NAME));

  public static final CLICommand esTargetCommands =
      new CLICommand(
          "elasticsearch",
          String.format("elasticsearch [%s|%s|%s|%s]",
            ADD_CMD_NAME, EDIT_CMD_NAME, LIST_CMD_NAME, REMOVE_CMD_NAME),
          CLIUsageOnlyCommand.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new CLICommand[] {
              targetAddCommand,
              targetEditCommand,
              targetListCommand,
              targetRemoveCommand
          }
      ).setShortUsage(String.format("%s [%s|%s|%s|%s]", TARGET_FAMILY,
          ADD_CMD_NAME, EDIT_CMD_NAME, LIST_CMD_NAME, REMOVE_CMD_NAME));

  private class ESTarget {
    String targetName;
    String clusterName;
    String esVersion;
    String transportNode;

    void setTargetName(String name) {
      targetName = name;
    }

    String getTargetName() {
      return targetName;
    }

    void setClusterName(String name) {
      clusterName = name;
    }

    String getClusterName() {
      return clusterName;
    }

    void setESVersion(Path esJar) {
      String name = esJar.getName();
      if (name.startsWith("elasticsearch-") && name.endsWith(".jar")) {
        esVersion = name.substring(14, name.length() - 4);
      }
    }

    String getESVersion() {
      return esVersion;
    }

    void setTransportNode(String node) {
      transportNode = node;
    }

    String getTransportNode() {
      return transportNode;
    }
  }

  public ElasticsearchCommands(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }

  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    if (!super.validateInput()) {
      return output;
    }

    String cname = cliCommand.getCommandName();
    String commandFamily = cliCommand.getParentCommand().getParentCommandNames();
    if (commandFamily.equalsIgnoreCase(REPLICA_FAMILY)) {
      if (cname.equalsIgnoreCase(replicaSetupCommand.getCommandName())) {
        replicaSetup(out);
      } else if (cname.equalsIgnoreCase(replicaAddCommand.getCommandName())) {
        replicaAdd(out);
      } else if (cname.equalsIgnoreCase(replicaEditCommand.getCommandName())) {
        replicaEdit(out);
      } else if (cname.equalsIgnoreCase(replicaListCommand.getCommandName())) {
        replicaList(out);
      } else if (cname.equalsIgnoreCase(replicaPauseCommand.getCommandName())) {
        replicaPauseOrResume(out, true);
      } else if (cname.equalsIgnoreCase(replicaResumeCommand.getCommandName())) {
        replicaPauseOrResume(out, false);
      } else if (cname.equalsIgnoreCase(replicaRemoveCommand.getCommandName())) {
        replicaRemove(out);
      } else if (cname.equalsIgnoreCase(replicaCleanupCommand.getCommandName())) {
        replicaCleanup(out);
      }
    } else if (commandFamily.equalsIgnoreCase(TARGET_FAMILY)) {
      if (cname.equalsIgnoreCase(targetAddCommand.getCommandName())) {
        targetAdd(out);
      } else if (cname.equalsIgnoreCase(targetEditCommand.getCommandName())) {
        targetEdit(out);
      } else if (cname.equalsIgnoreCase(targetListCommand.getCommandName())) {
        targetList(out);
      } else if (cname.equalsIgnoreCase(targetRemoveCommand.getCommandName())) {
        targetRemove(out);
      }
    }

    return output;
  }

  @Override
  public String getCliParam(String key) throws IOException {
    String ret = null;

    // If permissions map val is a passed as parameter, then add to ace list
    try {
      if (isParamPresent(key)) {
        ret = getParamTextValue(key, 0);
        if (ret != null) {
          ret = ret.trim();
        }
      }
    } catch (CLIProcessingException e) {
      throw new IOException(e);
    }
    return ret;
  }

  private void replicaSetup(OutputHierarchy out) throws CLIProcessingException {
    final String tableName = DbCommands.getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId());
    final String targetName = getTextParam(TARGET_PARAM_NAME, null);
    final String indexName = getTextParam(ESINDEX_PARAM_NAME, null);
    final String typeName = getTextParam(ESTYPE_PARAM_NAME, null);
    final String colList = getTextParam(COLUMNS_PARAM_NAME, null);
    final String indexedColList = getTextParam(INDEXED_COLUMNS_PARAM_NAME, null);
    final String convClassName = getTextParam(CONVCLASS_PARAM_NAME, null);
    final String convJar = getTextParam(CONVJAR_PARAM_NAME, null);
    final String ticketPath = getTextParam(TICKET_PARAM_NAME, null);

    final AceHelper.DBPermission dbPerm = this;
    final TableReplicaDesc.Builder replBuilder = TableReplicaDesc.newBuilder();

    try {
      final Path replicaPath = getReplicaFilePath(tableName, targetName, indexName, typeName);

      new FileclientRun(getUserLoginId()) {
        @Override
        public void runAsProxyUser() throws IOException, CLIProcessingException {
          MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();

          replicaCommonSetup(mfs, tableName, targetName, indexName, typeName,
              colList, indexedColList, convClassName, convJar, replicaPath, ticketPath, dbPerm,
              true /*isAuto*/, replBuilder);
          DbReplicaCommands.copyTable(tableName, replicaPath.toString(), 1, colList, getUserLoginId(),
              true, ticketPath, false, dbPerm, replBuilder);
        }
      };
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EINVALMISC, e.getMessage()));
    }
  }

  private void replicaAdd(OutputHierarchy out) throws CLIProcessingException {
    final String tableName = DbCommands.getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId());
    final String targetName = getTextParam(TARGET_PARAM_NAME, null);
    final String indexName = getTextParam(ESINDEX_PARAM_NAME, null);
    final String typeName = getTextParam(ESTYPE_PARAM_NAME, null);
    final String colList = getTextParam(COLUMNS_PARAM_NAME, null);
    final String convClassName = getTextParam(CONVCLASS_PARAM_NAME, null);
    final String convJar = getTextParam(CONVJAR_PARAM_NAME, null);
    final String ticketPath = getTextParam(TICKET_PARAM_NAME, null);

    final AceHelper.DBPermission dbPerm = this;
    final TableReplicaDesc.Builder replBuilder = TableReplicaDesc.newBuilder();

    try {
      final Path replicaPath = getReplicaFilePath(tableName, targetName, indexName, typeName);

      new FileclientRun(getUserLoginId()) {
        @Override
        public void runAsProxyUser() throws IOException, CLIProcessingException {
          MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
          replicaCommonSetup(mfs, tableName, targetName, indexName, typeName,
              colList, null/*indexedColList*/, convClassName, convJar, replicaPath, ticketPath, dbPerm,
              false /*isAuto*/, replBuilder);
        }
      };
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EINVALMISC, e.getMessage()));
    }
  }

  private void checkTableType(MapRFileSystem mfs, final Path tablePath) throws CLIProcessingException {
    try {
      if (!mfs.exists(tablePath)) {
        throw new CLIProcessingException("Path " + tablePath + " not found");
      }

      if (!mfs.isTable(tablePath)) {
        throw new CLIProcessingException("Elasticsearch commands are not supported for regular files");
      }

      if (mfs.isStream(tablePath)) {
        throw new CLIProcessingException("Elasticsearch commands are not yet supported for streams");
      }

      if (mfs.isJsonTable(tablePath)) {
        throw new CLIProcessingException("Elasticsearch commands are not yet supported for JSON tables");
      }
    } catch (IOException e) {
      throw new CLIProcessingException("Could not check type of " + tablePath);
    }
  }

  private void replicaCommonSetup(MapRFileSystem mfs, final String tableName,
      final String targetName, final String indexName, final String typeName,
      final String colList, final String indexedColList, final String convClassName, final String convJar,
      final Path replicaPath, final String ticketPath, final AceHelper.DBPermission dbPerm,
      boolean isAuto, final TableReplicaDesc.Builder replBuilder)
    throws CLIProcessingException, IOException {
    Path tablePath = new Path(tableName);

    // Source must be a table
    checkTableType(mfs, tablePath);

    // if replica config file exists, this relationship already exists
    if (mfs.exists(replicaPath)) {
      throw new CLIProcessingException(String.format("Replica %s:%s/%s for table %s already exists",
            targetName, indexName, typeName, tableName));
    }

    mfs.mkdirs(replicaPath.getParent());
    replicaMakeWritable(replicaPath.getParent().getParent());

    if (convJar != null && convClassName != null) {
      copyConversionJar(convJar, replicaPath.getParent());
    } else if (convJar == null && convClassName == null) {
      // No conversion class/jar specified - that's fine
    } else {
      throw new CLIProcessingException("Must specify " + CONVJAR_PARAM_NAME +
          " and " + CONVCLASS_PARAM_NAME);
    }
    createReplicaFile(replicaPath, targetName, indexName, typeName, convJar, convClassName);

    replBuilder.setExternal(true);
    if (isAuto) {
      replBuilder.setIsPaused(true);
    } else if (isParamPresent(PAUSED_PARAM_NAME)) {
      replBuilder.setIsPaused(getParamBooleanValue(PAUSED_PARAM_NAME, 0));
    }

    if (isParamPresent(THROTTLE_PARAM_NAME)) {
      replBuilder.setThrottle(getParamBooleanValue(THROTTLE_PARAM_NAME, 0));
    }

    if (isParamPresent(ENCRYPTION_PARAM_NAME)) {
      replBuilder.setEncryptonwire(getParamBooleanValue(ENCRYPTION_PARAM_NAME, 0));
    }

    if (isParamPresent(COMPRESSION_PARAM_NAME)) {
      FileCompressionType ctype =
        DbReplicaCommands.getCompressionType(getParamTextValue(COMPRESSION_PARAM_NAME, 0));
      if (ctype == null) {
        throw new CLIProcessingException("Invalid input values. " +
            "The entered value for compression is not a supported type.");
      }
      replBuilder.setCompressonwire(ctype);
    }

    String className = DbReplicaCommands.verifyExternalDstSanity(mfs, replicaPath.toString());
    DbReplicaCommands.setupReplication(dbPerm, replBuilder, tableName, replicaPath.toString(),
        colList, indexedColList, getUserLoginId(), mfs, className);
  }

  private void replicaEdit(OutputHierarchy out) throws CLIProcessingException {
    final String tableName = DbCommands.getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId());
    final String targetName = getTextParam(TARGET_PARAM_NAME, null);
    final String indexName = getTextParam(ESINDEX_PARAM_NAME, null);
    final String typeName = getTextParam(ESTYPE_PARAM_NAME, null);
    final String colList = getTextParam(COLUMNS_PARAM_NAME, null);
    final String indexedColList = getTextParam(INDEXED_COLUMNS_PARAM_NAME, null);
    final String convClassName = getTextParam(CONVCLASS_PARAM_NAME, null);
    final String convJar = getTextParam(CONVJAR_PARAM_NAME, null);

    final TableReplicaDesc.Builder replBuilder = TableReplicaDesc.newBuilder();
    final int replDescEmptySize = replBuilder.build().getSerializedSize();
    final RecentTablesListManager manager = RecentTablesListManagers.getRecentTablesListManagerForUser(getUserLoginId());

    try {
      final Path replicaPath = getReplicaFilePath(tableName, targetName, indexName, typeName);

      new FileclientRun(getUserLoginId()) {
        @Override
        public void runAsProxyUser() throws IOException, CLIProcessingException {
          MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
          Path tablePath = new Path(tableName);
          boolean isJson = mfs.isJsonTable(tablePath);
          String replicaClusterName = mfs.getClusterNameUnchecked(replicaPath.toString());
          String replicaMFSName = mfs.getNameStr(replicaPath.toString());

          // Source must be a table
          checkTableType(mfs, tablePath);

          if (!mfs.exists(replicaPath)) {
            throw new CLIProcessingException(String.format("Replica %s:%s/%s for table %s not found",
                  targetName, indexName, typeName, tableName));
          }

          // If the conversion class stuff changes, rewrite the replica config file
          if (convClassName != null || convJar != null) {
            if (convJar != null) {
              copyConversionJar(convJar, replicaPath.getParent());
            }
            createReplicaFile(replicaPath, targetName, indexName, typeName, convJar, convClassName);
          }

          if (isParamPresent(THROTTLE_PARAM_NAME)) {
            replBuilder.setThrottle(getParamBooleanValue(THROTTLE_PARAM_NAME, 0));
          }

          if (isParamPresent(ENCRYPTION_PARAM_NAME)) {
            replBuilder.setEncryptonwire(getParamBooleanValue(ENCRYPTION_PARAM_NAME, 0));
          }

          if (isParamPresent(COMPRESSION_PARAM_NAME)) {
            FileCompressionType ctype =
                DbReplicaCommands.getCompressionType(getParamTextValue(COMPRESSION_PARAM_NAME, 0));
            if (ctype == null) {
              throw new CLIProcessingException("Invalid input values. " +
                  "The entered value for compression is not a supported type.");
            }
            replBuilder.setCompressonwire(ctype);
          }

          String user = getUserLoginId();

          String clist = colList;
          if (indexedColList != null) {
            DbReplicaCommands.validateIndexedFields(indexedColList, tableName, user);
            replBuilder.setIndexedFields(indexedColList);
            if (clist != null && clist.trim().length() > 0) {
              // append the indexed columns to the list of replicated columns
              clist += "," + indexedColList.replace(';', ',');
            }
          }

          boolean allCfs = false;
          if (clist != null) {
            allCfs = (clist.trim().length() == 0);
            if (!allCfs) {
              TreeMap<Integer, SortedSet<String>> familyMap = new TreeMap<Integer, SortedSet<String>>();
              DbReplicaCommands.parseColList(clist, tableName, user, isJson, familyMap);
              DbReplicaCommands.addCfQualifiers(familyMap, replBuilder);
            }
          }

          // Nothing changed, just bail
          if (replBuilder.build().getSerializedSize() == replDescEmptySize && !allCfs) {
            return;
          }

          try {
            TableReplicaDesc replDesc = replBuilder.build();

            mfs.editTableReplica(tablePath, replicaClusterName, replicaMFSName, allCfs, replDesc);
            manager.moveToTop(tableName);
          } catch (IOException e) {
            manager.deleteIfNotExist(tableName, mfs);
            throw new CLIProcessingException(e.getMessage());
          }
        }
      };
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }
  }

  class ReplicaInfo {
    TableReplicaDesc rd;
    String targetName;
    String indexName;
    String typeName;

    public ReplicaInfo(TableReplicaDesc rd) {
      this.rd = rd;
    }

    public void init(MapRFileSystem mfs) {
      InputStream is = null;
      try {
        Path confPath = new Path(rd.getTablePath());
        Yaml yaml = new Yaml();
        is = mfs.open(confPath);
        Map confMap = (Map)yaml.load(is);

        targetName = (String)confMap.get(YAML_ES_TARGET);
        indexName = (String)confMap.get(YAML_ES_INDEX);
        typeName = (String)confMap.get(YAML_ES_TYPE);
      } catch (IOException e) {
        LOG.error("Error reading replica config file: " + e.getMessage());
      } finally {
        try {
          if (is != null) {
            is.close();
          }
        } catch (Exception e) {}
      }
    }

    public TableReplicaDesc getReplicaDesc() {
      return rd;
    }

    public String getTargetName() {
      return targetName;
    }

    public String getIndexName() {
      return indexName;
    }

    public String getTypeName() {
      return typeName;
    }
  }

  private void replicaList(OutputHierarchy out) throws CLIProcessingException {
    final String tableName = DbCommands.getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId());
    final boolean refreshNow = getBooleanParam(REFRESH_PARAM_NAME, false);
    final RecentTablesListManager manager = RecentTablesListManagers.getRecentTablesListManagerForUser(getUserLoginId());
    final List<ReplicaInfo> rdlist = new LinkedList<ReplicaInfo>();
    final List<TableReplicaListResponse> rlist = new ArrayList<TableReplicaListResponse>();

    try {
      // Call impersonation
      new FileclientRun(getUserLoginId()) {
        @Override
        public void runAsProxyUser() throws IOException,CLIProcessingException {
          MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();

          try {
            Path tablePath = new Path(tableName);

            // Source must be a table
            checkTableType(mfs, tablePath);

            TableReplicaListResponse resp = mfs.listTableReplicas(tablePath,
                true /* wantStats */, refreshNow);
            rlist.add(resp);
            for (TableReplicaDesc rd : resp.getReplicasList()) {
              if (!rd.hasExternal() || !rd.getExternal()) {
                continue;
              }

              if (!mfs.exists(new Path(rd.getTablePath()))) {
                // If the destination config file doesn't exist, this replica
                // doesn't really exist.  I'm not sure what to do about it right
                // now, but I'm leaving code here for the situation because it'd
                // be nice to do something intelligent.
              } else if (rd.getTablePath().startsWith(ES_REPLICAS)) {
                ReplicaInfo ri = new ReplicaInfo(rd);
                ri.init(mfs);
                rdlist.add(ri);
              }
            }
            manager.moveToTop(tableName);
          } catch (IOException e) {
            manager.deleteIfNotExist(tableName, mfs);
            throw new CLIProcessingException(e.getMessage());
          }
        }
      };

      List<ColumnFamilyAttr> familyAttrs = DbReplicaCommands.getAllFamilies(tableName, getUserLoginId());
      TableReplicaListResponse listResp = rlist.get(0);
      for (ReplicaInfo ri : rdlist) {
        TableReplicaDesc rd = ri.getReplicaDesc();
        OutputNode replNode = new OutputNode();

        replNode.addChild(new OutputNode("cluster", rd.getClusterName()));

        replNode.addChild(new OutputNode("target", ri.getTargetName()));
        replNode.addChild(new OutputNode("index", ri.getIndexName()));
        replNode.addChild(new OutputNode("type", ri.getTypeName()));

        replNode.addChild(new OutputNode("paused", rd.getIsPaused()));
        replNode.addChild(new OutputNode("throttle", rd.getThrottle()));
        replNode.addChild(new OutputNode("idx", rd.getIdx()));
        replNode.addChild(new OutputNode("networkencryption", rd.getEncryptonwire()));
        replNode.addChild(new OutputNode("networkcompression",
              DbReplicaCommands.getCompressionName(rd.getCompressonwire())));

        if (rd.getQualifiersCount() > 0) {
          String famList = "";
          int idx = 0;

          for (Qualifier qual : rd.getQualifiersList()) {
            String famName = DbReplicaCommands.familyIdToName(familyAttrs, qual.getFamily());
            if (famName == null) {
              LOG.error("failed to get family name for id " + qual.getFamily());
            }

            if (qual.getQualifiersCount() > 0) {
              for (ByteString bstr : qual.getQualifiersList()) {
                if (idx == 0) {
                  famList = famName + COLUMN_SEP + bstr.toStringUtf8();
                } else {
                  famList = famList + "," + famName + COLUMN_SEP + bstr.toStringUtf8();
                }
                ++idx;
              }
            } else if (idx == 0) {
              famList = famName;
              ++idx;
            } else {
              famList = famList + "," + famName;
              ++idx;
            }
          }
          replNode.addChild(new OutputNode("columns", famList));
        }

        if (rd.hasIndexedFields()) {
          replNode.addChild(new OutputNode("indexedcolumns", rd.getIndexedFields()));
        }

        //Show copy table status
        String replicaTablePath = DbReplicaCommands.getTransformedTablePath(rd.getClusterName(), rd.getTablePath());
        if (JobExecutor.getStatus(tableName + "_" + replicaTablePath) != null) {
          replNode.addChild(new OutputNode("copytablestatus",
                JobExecutor.getStatus(tableName + "_" + replicaTablePath)));
        }

        long minPendingTS = 0;
        long maxPendingTS = 0;
        long bytesPending = 0;
        long putsPending = 0;
        int bucketsPending = 0;
        int asyncBuckets = 0;
        List<ExtendedError> elist = null;

        for (TableReplicaStatus rs : listResp.getReplicaStatusList()) {
          if (rd.getIdx() == rs.getReplicaIdx()) {
            minPendingTS = rs.getMinPendingTS();
            maxPendingTS = rs.getMaxPendingTS();
            bytesPending = rs.getBytesPending();
            putsPending = rs.getPutsPending();
            bucketsPending = rs.getBucketsPending();
            asyncBuckets = rs.getAsyncBuckets();

            for (ExtendedError ee : rs.getEerrorsList()) {
              if (elist == null)
                elist = new ArrayList<ExtendedError>();
              elist.add(ee);
            }
            break;
          }
        }
        replNode.addChild(new OutputNode("isUptodate", (bucketsPending == 0 && !rd.getIsPaused())));
        replNode.addChild(new OutputNode("minPendingTS", minPendingTS));
        replNode.addChild(new OutputNode("maxPendingTS", maxPendingTS));
        replNode.addChild(new OutputNode("bytesPending", bytesPending));
        replNode.addChild(new OutputNode("putsPending", putsPending));
        replNode.addChild(new OutputNode("bucketsPending", bucketsPending));

        byte[] uuid = rd.getTableUuid().toByteArray();
        replNode.addChild(new OutputNode("uuid", BinaryString.toUUIDString(uuid)));

        if (elist != null) {
          for (ExtendedError ee : elist) {
            OutputNode errNodes = new OutputNode("errors");
            replNode.addChild(errNodes);

            if (ee.hasEcode()) {
              errNodes.addChild(new OutputNode("Code", ee.getEcode().name()));
            }
            if (ee.hasEhost()) {
              errNodes.addChild(new OutputNode("Host",
                                               Util.intToIp(ee.getEhost())));
            }
            if (ee.hasEmsg()) {
              errNodes.addChild(new OutputNode("Msg", ee.getEmsg()));
            }
          }
        }
        out.addNode(replNode);
      }
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }
  }

  private void replicaPauseOrResume(OutputHierarchy out, boolean isPaused) throws CLIProcessingException {
    final String tableName = DbCommands.getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId());
    final String targetName = getTextParam(TARGET_PARAM_NAME, null);
    final String indexName = getTextParam(ESINDEX_PARAM_NAME, null);
    final String typeName = getTextParam(ESTYPE_PARAM_NAME, null);

    try {
      final Path replicaPath = getReplicaFilePath(tableName, targetName, indexName, typeName);

      DbReplicaCommands.pauseOrResumeReplication(tableName, replicaPath.toString(), isPaused, getUserLoginId());
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }

  }

  private void replicaRemove(OutputHierarchy out) throws CLIProcessingException {
    final String tableName = DbCommands.getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId());
    final String targetName = getTextParam(TARGET_PARAM_NAME, null);
    final String indexName = getTextParam(ESINDEX_PARAM_NAME, null);
    final String typeName = getTextParam(ESTYPE_PARAM_NAME, null);

    final TableReplicaDesc.Builder replBuilder = TableReplicaDesc.newBuilder();
    final RecentTablesListManager manager = RecentTablesListManagers.getRecentTablesListManagerForUser(getUserLoginId());

    try {
      final Path replicaPath = getReplicaFilePath(tableName, targetName, indexName, typeName);

      new FileclientRun(getUserLoginId()) {
        @Override
        public void runAsProxyUser() throws IOException, CLIProcessingException{
          MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
          Path tablePath = new Path(tableName);
          String replicaName = replicaPath.toString();

          // Source must be a table
          checkTableType(mfs, tablePath);

          try {
            String clusterName = mfs.getClusterNameUnchecked(replicaName);
            replBuilder.setClusterName(clusterName);

            String replicaMFSName = mfs.getNameStr(replicaName);
            replBuilder.setTablePath(replicaMFSName);

            TableReplicaDesc replDesc = replBuilder.build();
            mfs.removeTableReplica(tablePath, replDesc);
            mfs.delete(replicaPath.getParent(), true);
            replicaRemoveTree(replicaPath.getParent().getParent());
            manager.moveToTop(tableName);
          } catch (IOException e) {
            manager.deleteIfNotExist(tableName, mfs);
            throw new CLIProcessingException(e.getMessage());
          }
        }
      };
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }
  }

  private void replicaCleanup(OutputHierarchy out) throws CLIProcessingException {
    try {
      new FileclientRun(getUserLoginId()) {
        @Override
        public void runAsProxyUser() throws IOException, CLIProcessingException{
          MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
          String clusterName = mfs.getClusterName((new Path(ES_REPLICAS)).toUri());

          FileStatus[] srcs = mfs.listStatus(new Path(ES_REPLICAS));
          for (FileStatus src : srcs) {
            if (!src.isDirectory()) {
              continue;
            }

            String fidStr = src.getPath().getName();
            FidMsg fid = stringToFid(fidStr);
            long dbinding = getBindingForContainer(clusterName, fid.getCid());
            if (dbinding == -1) {
              LOG.error("Container lookup failed for: " + src.getPath().getName());
              continue;
            }

            GetattrResponse resp;
            GetattrRequest req = GetattrRequest.newBuilder()
              .setNode(fid)
              .setCreds(getUserCredentials())
              .build();

            try {
              byte[] replyData = Rpc.sendRequest(dbinding,
                  MapRProgramId.FileServerProgramId.getNumber(),
                  FSProg.GetattrProc.getNumber(),
                  req);
              if (replyData == null) {
                LOG.error("RPC failed for: " + src.getPath().getName());
                continue;
              }

              resp = GetattrResponse.parseFrom(replyData);
              if (resp.getStatus() == Errno.ESTALE) {
                LOG.info("Removing entries for deleted table: " + src.getPath().getName());
                mfs.delete(src.getPath(), true);
              } else if (resp.getStatus() != 0) {
                LOG.error("GetAttr failed with error: " + Errno.toString(resp.getStatus()));
              }
            } catch (MaprSecurityException e) {
              throw new CLIProcessingException("Security exception for: " +
                  src.getPath().getName());
            } catch (Exception e) {
              throw new IOException(e.getMessage());
            }
          }
        }
      };
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }
  }

  /**
   * Construct the path to replica file.  The file might not exist - this is just
   * where the file should be.
   *
   * @param sourceTable name of source table
   * @param targetName internal name of target ES cluster
   * @param indexName name of ES index
   * @param typeName name of ES type
   */
  private Path getReplicaFilePath(final String sourceTable, final String targetName,
      final String indexName, final String typeName) throws IOException,CLIProcessingException {
    final List<Path> results = new LinkedList<Path>();

    new FileclientRun(getUserLoginId()) {
      @Override
      public void runAsProxyUser() throws IOException,CLIProcessingException {
        MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
        MapRHTable htable = new MapRHTable();
        htable.init(mfs.getConf(), new Path(sourceTable));

        String fid = mfs.openTable(new Path(sourceTable), htable).attrs().toString();
        Path result = new Path(ES_REPLICAS, fid);
        result = new Path(result, targetName);
        result = new Path(result, indexName);
        result = new Path(result, typeName);
        result = new Path(result, REPLICA_FILE);

        results.add(result);
      }
    };

    return results.get(0);
  }

  /**
   * Create, or recreate, the config.es file used to represent elasticsearch
   * replicas.  New config is written to a temporary file, and moved at the end
   * after all updates have completed, to avoid having a partially written
   * config file.
   *
   * @param replicaPath path to the replica config file
   * @param targetName name of the target Elasticsearch cluster
   * @param indexName name of Elasticsearch index
   * @param typeName name of the Elasticsearch type
   * @param convJar local path to the conversion class JAR file
   * @param convName name of the conversion class
   */
  private void createReplicaFile(Path replicaPath, String targetName, String indexName,
      String typeName, String convJar, String convName) throws CLIProcessingException, IOException {
    MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
    Map esMap = null;

    // If a config file already exists, we have to just update the pieces that
    // have changed (ie. arg is non-null).  Otherwise, we have to write a new
    // file.  Construct the file in a temporary location so that we don't get
    // stuck with a half written/modified file.
    Path tmpPath = new Path(replicaPath.getParent(), TMP_REPLICA_FILE);
    if (mfs.exists(replicaPath)) {
      FSDataInputStream is = mfs.open(replicaPath);
      Yaml yaml = new Yaml();
      esMap = (Map)yaml.load(is);
      is.close();
    } else {
      esMap = new HashMap();
    }

    FSDataOutputStream os = mfs.create(tmpPath, true);
    try {
      // The sink classpath is always the same, so just write a new one
      Path targetPath = new Path(ES_CLUSTERS, targetName);
      Path libPath = new Path(targetPath, LIB_DIR);
      Path pluginPath = new Path(targetPath, PLUGIN_DIR);

      os.writeBytes(YAML_SINK_CLASSPATH + ": " + libPath.toString());
      os.writeBytes(":" + pluginPath.toString());
      os.writeBytes(NEW_LN);

      // If there is an existing target name, just use - it can never be changed
      // The target, index, and type cannot be changed, so if we already have
      // one, just use it.  The path to the replica config file is defined by a
      // combination of the source table, target, index, and type.  We don't
      // want to deal with moving things around.  Besides, it doesn't look like
      // Elasticsearch allows index renames, so it doesn't really make sense to
      // change it on this side.
      if (esMap.get(YAML_ES_TARGET) != null) {
        os.writeBytes(YAML_ES_TARGET + ": " + (String)esMap.get(YAML_ES_TARGET) + NEW_LN);
      } else if (targetName != null) {
        os.writeBytes(YAML_ES_TARGET + ": " + targetName + NEW_LN);
      } else {
        throw new CLIProcessingException("No target name found");
      }

      if (esMap.get(YAML_ES_INDEX) != null) {
        os.writeBytes(YAML_ES_INDEX + ": " + (String)esMap.get(YAML_ES_INDEX) + NEW_LN);
      } else if (indexName != null) {
        os.writeBytes(YAML_ES_INDEX + ": " + indexName + NEW_LN);
      } else {
        throw new CLIProcessingException("No ES index name found");
      }

      if (esMap.get(YAML_ES_TYPE) != null) {
        os.writeBytes(YAML_ES_TYPE + ": " + (String)esMap.get(YAML_ES_TYPE) + NEW_LN);
      } else if (typeName != null) {
        os.writeBytes(YAML_ES_TYPE + ": " + typeName + NEW_LN);
      } else {
        throw new CLIProcessingException("No ES index name found");
      }

      boolean removeConv = false;
      if (convJar != null) {
        if (convJar.trim().length() > 0) {
          // convJar is a path in the local file system.  In here, we assume the
          // file will be copied into a particular location in MFS, but we do need
          // to build the new path before writing the config file.
          File convFile = new File(convJar);
          Path path = new Path(replicaPath.getParent(), convFile.getName());
          os.writeBytes(YAML_CONV_CLASSPATH + ": " + path.toString() + NEW_LN);
        } else {
          removeConv = true;
        }
      } else if (esMap.get(YAML_CONV_CLASSPATH) != null) {
        os.writeBytes(YAML_CONV_CLASSPATH + ": " + (String)esMap.get(YAML_CONV_CLASSPATH) + NEW_LN);
      }

      if (convName != null) {
        if (convName.trim().length() > 0) {
          os.writeBytes(YAML_CONV_CLASS_NAME + ": " + convName + NEW_LN);
        }
      } else if (esMap.get(YAML_CONV_CLASS_NAME) != null && !removeConv) {
        os.writeBytes(YAML_CONV_CLASS_NAME + ": " + (String)esMap.get(YAML_CONV_CLASS_NAME) + NEW_LN);
      }
    } finally {
      os.close();
    }
    mfs.rename(tmpPath, replicaPath);
  }

  private void copyConversionJar(String jarFileName, Path destPath) throws IOException, CLIProcessingException {
    MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
    File jarFile = new File(jarFileName);

    if (!jarFile.isFile()) {
      throw new CLIProcessingException("Could not find file: " + jarFileName);
    }

    mfs.copyFromLocalFile(false, true, new Path(jarFileName), destPath);
  }

  /**
   * Recursively remove a replica until we hit the ES_REPLICAS path.  Abort the
   * first time a directory isn't empty.
   */
  private void replicaRemoveTree(Path replicaPath) throws IOException, CLIProcessingException {
    MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();

    // We stop at the main replicas directory
    if (replicaPath.toString().compareTo(ES_REPLICAS) == 0) {
      return;
    }

    FileStatus[] listing = mfs.listStatus(replicaPath);
    if (listing != null && listing.length > 0) {
      return;
    }

    mfs.delete(replicaPath, true);
    replicaRemoveTree(replicaPath.getParent());
  }

  /**
   * Recursively make a replica path writable up to the ES_REPLICAS directory.
   * The mkdirs() family of functions does not do this the way you might guess.
   */
  private void replicaMakeWritable(Path replicaPath) throws IOException, CLIProcessingException {
    MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();

    // We stop at the main replicas directory
    if (replicaPath.toString().compareTo(ES_REPLICAS) == 0) {
      return;
    }

    mfs.setPermission(replicaPath, new FsPermission(FsAction.ALL, FsAction.ALL, FsAction.ALL));
    replicaMakeWritable(replicaPath.getParent());
  }

  private FidMsg stringToFid(String s) {
    String[] tmp = s.split("\\.");

    if (tmp.length != 3) {
      return null;
    }

    int cid = Integer.parseInt(tmp[0]);
    int cinum = Integer.parseInt(tmp[1]);
    int uniq = Integer.parseInt(tmp[2]);

    return FidMsg.newBuilder()
      .setCid(cid)
      .setCinum(cinum)
      .setUniq(uniq)
      .build();
  }

  private long getBindingForContainer(String clusterName, int cid) throws CLIProcessingException {
    // lookup container in cldb
    int dbHost, dbPort;
    byte[] replyData;
    ContainerLookupRequest creq = ContainerLookupRequest.newBuilder()
                                    .addContainerId(cid)
                                    .setCreds(getUserCredentials())
                                    .build();
    try {
      replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
                    MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.ContainerLookupProc.getNumber(),
                    creq,
                    ContainerLookupResponse.class);

      if (replyData == null) {
        // most likely can not connect to CLDB
        LOG.error("Couldn't connect to the CLDB service");
        return -1;
      }

      ContainerLookupResponse resp;
      resp = ContainerLookupResponse.parseFrom(replyData);
      if (resp.getStatus() == 0) {
        Server server = resp.getContainers(0).getMServer();
        dbHost = server.getIps(0).getHost();
        dbPort = server.getIps(0).getPort();
      } else {
        LOG.error("Container lookup failed : Error " + Errno.toString(resp.getStatus()));
        return -1;
      }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      LOG.error("Container lookup failed");
      return -1;
    }

    return Rpc.createBindingFor(dbHost, dbPort, clusterName, ServerKeyType.ServerKey.getNumber());
  }

  private void targetAdd(OutputHierarchy out) throws CLIProcessingException {
    final String targetName = getTextParam(ESCLUSTER_PARAM_NAME, null);
    final String configFileName = getTextParam(ESCONFIG_PARAM_NAME, null);
    final String libDirName = getTextParam(ESLIB_PARAM_NAME, null);
    final String pluginDirName = getTextParam(ESPLUGIN_PARAM_NAME, null);
    final String transportNode = getTextParam(ESTRANSPORT_PARAM_NAME, null);

    try {
      new FileclientRun(getUserLoginId()) {
        @Override
        public void runAsProxyUser() throws IOException, CLIProcessingException {
          MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
          Path targetPath = new Path(ES_CLUSTERS, targetName);

          File configFile = new File(configFileName);
          if (!configFile.isFile()) {
            throw new CLIProcessingException("Cannot find file " + configFileName);
          }

          File libDir = new File(libDirName);
          if (!libDir.isDirectory()) {
            throw new CLIProcessingException("Cannot find directory " + libDirName);
          }

          File pluginDir = null;
          if (pluginDirName != null) {
            pluginDir = new File(pluginDirName);
            if (!pluginDir.isDirectory()) {
              throw new CLIProcessingException("Cannot find directory " + pluginDirName);
            }
          }

          // Check for existing target cluster and create directory
          if (mfs.exists(targetPath)) {
            throw new CLIProcessingException("Target cluster " + targetName + " already exists");
          }
          mfs.mkdirs(targetPath);

          // Create config dir and copy ES cluster config file
          copyConfigFile(configFile, new Path(targetPath, CONF_DIR));

          // Create lib dir and copy JAR files
          copyJarDir(libDir, new Path(targetPath, LIB_DIR), false);

          // Copy JAR files from plugin dir.  This is done recursively, because
          // the default ES plugin directory doesn't have all of the JAR files
          // at the top level.
          if (pluginDir != null) {
            copyJarDir(pluginDir, new Path(targetPath, PLUGIN_DIR), true);
          }

          // If a transport node was provided, add it to the config file
          if (transportNode != null) {
            setTransportNode(transportNode, new Path(targetPath, CONF_DIR));
          }

          if (!mfs.isDirectory(new Path(ES_REPLICAS))) {
            mfs.mkdirs(new Path(ES_REPLICAS), new FsPermission(FsAction.ALL, FsAction.ALL, FsAction.ALL));
          }
        }
      };
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EINVALMISC, e.getMessage()));
    }
  }

  private void targetEdit(OutputHierarchy out) throws CLIProcessingException {
    final String targetName = getTextParam(ESCLUSTER_PARAM_NAME, null);
    final String configFileName = getTextParam(ESCONFIG_PARAM_NAME, null);
    final String libDirName = getTextParam(ESLIB_PARAM_NAME, null);
    final String pluginDirName = getTextParam(ESPLUGIN_PARAM_NAME, null);
    final String transportNode = getTextParam(ESTRANSPORT_PARAM_NAME, null);

    try {
      new FileclientRun(getUserLoginId()) {
        @Override
        public void runAsProxyUser() throws IOException, CLIProcessingException {
          MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
          Path targetPath = new Path(ES_CLUSTERS, targetName);
          File configFile = null;
          File libDir = null;
          File pluginDir = null;

          if (!mfs.isDirectory(targetPath)) {
            throw new CLIProcessingException("Could not find registered cluster: " + targetName);
          }

          if (configFileName != null) {
            configFile = new File(configFileName);
            if (!configFile.isFile()) {
              throw new CLIProcessingException("Cannot find file " + configFileName);
            }
          }

          if (libDirName != null) {
            libDir = new File(libDirName);
            if (!libDir.isDirectory()) {
              throw new CLIProcessingException("Cannot find directory " + libDirName);
            }
          }

          if (pluginDirName != null) {
            pluginDir = new File(pluginDirName);
            if (!pluginDir.isDirectory()) {
              throw new CLIProcessingException("Cannot find directory " + pluginDirName);
            }
          }

          if (configFile != null) {
            copyConfigFile(configFile, new Path(targetPath, CONF_DIR));
          }

          if (libDir != null) {
            copyJarDir(libDir, new Path(targetPath, LIB_DIR), false);
          }

          if (pluginDir != null) {
            copyJarDir(pluginDir, new Path(targetPath, PLUGIN_DIR), true);
          }

          if (transportNode != null) {
            setTransportNode(transportNode, new Path(targetPath, CONF_DIR));
          }
        }
      };
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EINVALMISC, e.getMessage()));
    }
  }

  private void targetList(OutputHierarchy out) throws CLIProcessingException {
    final List<ESTarget> targets = new ArrayList<ESTarget>();

    try {
      new FileclientRun(getUserLoginId()) {
        @Override
        public void runAsProxyUser() throws IOException, CLIProcessingException {
          MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
          Path clustersPath = new Path(ES_CLUSTERS);

          for (FileStatus file : mfs.listStatus(clustersPath)) {
            if (file.isDirectory()) {
              Path clusterPath = file.getPath();
              ESTarget target = new ESTarget();

              // Directory name is the target name for this ES cluster
              target.setTargetName(clusterPath.getName());

              // Look for elasticsearch-<version>.jar to get ES version
              Path libPath = new Path(clusterPath, LIB_DIR);
              FileStatus[] jarFiles = mfs.listStatus(libPath, new PathFilter() {
                    @Override
                    public boolean accept(Path path) {
                      if (path.getName().startsWith("elasticsearch-") &&
                          path.getName().endsWith(".jar")) {
                        return true;
                      }
                      return false;
                    }
                  });
              if (jarFiles.length > 0) {
                target.setESVersion(jarFiles[0].getPath());
              }

              // Open cluster conf file to get ES cluster name and transport node settings
              Path confPath = new Path(clusterPath, CONF_DIR);
              Yaml yaml = new Yaml();
              InputStream is = mfs.open(new Path(confPath, CONF_FILE));
              Map confMap = (Map)yaml.load(is);
              is.close();
              target.setClusterName((String)confMap.get("cluster.name"));

              Path transportPath = new Path(confPath, TRANSPORT_NODE_FILE);
              if (mfs.isFile(transportPath)) {
                yaml = new Yaml();
                is = mfs.open(transportPath);
                confMap = (Map)yaml.load(is);
                is.close();
                if (confMap.get("transport.client.initial_nodes") != null) {
                  ArrayList<String> nodes = (ArrayList<String>)confMap.get("transport.client.initial_nodes");
                  target.setTransportNode(nodes.get(0));
                }
              }

              targets.add(target);
            }
          }
        }
      };

      for (ESTarget target : targets) {
        OutputNode targetNode = new OutputNode();

        targetNode.addChild(new OutputNode("target", target.getTargetName()));
        targetNode.addChild(new OutputNode("clusterName", target.getClusterName()));
        if (target.getESVersion() != null) {
          targetNode.addChild(new OutputNode("esVersion", target.getESVersion()));
        }
        if (target.getTransportNode() != null) {
          targetNode.addChild(new OutputNode("transportNode", target.getTransportNode()));
        }

        out.addNode(targetNode);
      }
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EINVALMISC, e.getMessage()));
    }
  }

  private void targetRemove(OutputHierarchy out) throws CLIProcessingException {
    final String targetName = getTextParam(ESCLUSTER_PARAM_NAME, null);

    try {
      new FileclientRun(getUserLoginId()) {
        @Override
        public void runAsProxyUser() throws IOException, CLIProcessingException {
          MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
          Path targetPath = new Path(ES_CLUSTERS, targetName);

          if (!mfs.isDirectory(targetPath)) {
            throw new CLIProcessingException("Could not find registered cluster: " + targetName);
          }

          mfs.delete(targetPath, true);
        }
      };
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EINVALMISC, e.getMessage()));
    }
  }

  /**
   * @param localFile path to ES config file in the local file system
   * @param mfsPath path to config directory in MFS
   */
  private void copyConfigFile(File localFile, Path mfsPath) throws IOException, CLIProcessingException {
    MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
    Path confPath = new Path(mfsPath, CONF_FILE);

    mfs.mkdirs(mfsPath);
    mfs.copyFromLocalFile(false, true, new Path(localFile.getPath()), confPath);
  }

  /**
   * @param localDir path to directory containing jar files
   * @param mfsPath destination path in MFS
   * @param recursive search localDir recursively for jar files?
   */
  private void copyJarDir(final File localDir, Path mfsPath, final boolean recursive) throws IOException, CLIProcessingException {
    MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();

    if (mfs.exists(mfsPath)) {
      mfs.delete(mfsPath, true);
    }

    final ArrayList<Path> jarPaths = new ArrayList<Path>();
    Files.walkFileTree(localDir.toPath(), new SimpleFileVisitor<java.nio.file.Path>() {
          @Override
          public FileVisitResult visitFile(java.nio.file.Path file, BasicFileAttributes attrs) throws IOException {
            // We only want to copy JAR files, but we have the additional hack
            // that we know the marvel jar file causes problems with the
            // Transport Client, so we don't want to copy it.
            if (file.toFile().getName().endsWith(".jar") &&
                !file.toFile().getName().startsWith("marvel.")) {
              jarPaths.add(new Path(file.toFile().getPath()));
            }
            return FileVisitResult.CONTINUE;
          }

          @Override
          public FileVisitResult preVisitDirectory(java.nio.file.Path dir, BasicFileAttributes attrs) throws IOException {
            if (recursive ||
                dir.toString().compareTo(localDir.getPath()) == 0) {
              System.out.println("Visiting directory: " + dir.toString());
              return FileVisitResult.CONTINUE;
            }
            System.out.println("Skipping directory: " + dir.toString());
            return FileVisitResult.SKIP_SUBTREE;
          }
        });
    mfs.mkdirs(mfsPath);
    mfs.copyFromLocalFile(false, true, jarPaths.toArray(new Path[0]), mfsPath);
  }

  /**
   * @param transportNode name[:port] of a transport node
   * @param mfsPath path to config directory in MFS
   */
  private void setTransportNode(String transportNode, Path mfsPath) throws IOException, CLIProcessingException {
    MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
    Path transportPath = new Path(mfsPath, TRANSPORT_NODE_FILE);

    if (transportNode.trim().length() == 0) {
      mfs.delete(transportPath, false);
    } else {
      FSDataOutputStream os = mfs.create(transportPath, true);
      os.writeBytes("transport.client.initial_nodes: [ " + transportNode + " ]" + System.lineSeparator());
      os.close();
    }
  }

  private String getTextParam(String paramName, String defaultValue) throws CLIProcessingException {
    if (isParamPresent(paramName)) {
      return getParamTextValue(paramName, 0);
    }
    return defaultValue;
  }

  private boolean getBooleanParam(String paramName, boolean defaultValue) throws CLIProcessingException {
    if (isParamPresent(paramName)) {
      return getParamBooleanValue(paramName, 0);
    }
    return defaultValue;
  }
}
