/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.InvalidProtocolBufferException;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cli.common.NodesCommonUtils;
import com.mapr.baseutils.utils.GetStringsForAuditOps;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.TextCommandOutput;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import com.mapr.cliframework.base.inputparams.IntegerInputParameter;
import com.mapr.cliframework.base.inputparams.NoValueInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import com.mapr.fs.Rpc;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.fs.cldb.conf.CLDBConfiguration;
import com.mapr.fs.cldb.proto.CLDBProto.*;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.NativeMemoryInfo;
import com.mapr.fs.cldb.proto.dialhome.MetricsProto.DiskBalancerMetrics;
import com.mapr.fs.cldb.proto.dialhome.MetricsProto.RoleBalancerMetrics;
import com.mapr.fs.cldb.proto.dialhome.MetricsProto.RereplicationMetrics;
import com.mapr.fs.cldb.util.Util;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Common.IPAddress;
import com.mapr.fs.proto.Common.Server;
import com.mapr.fs.proto.Common.Server.ReplicaState;
import com.mapr.fs.proto.Security.ServerKeyType;
import com.mapr.fs.proto.Security.TicketAndKey;
import com.mapr.security.MutableInt;
import com.mapr.security.Security;
import com.mapr.security.JNISecurity;
import com.mapr.security.SecurityHelper;
import com.mapr.baseutils.Errno;

import org.apache.log4j.Logger;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.ZooDefs.Ids;
import org.apache.zookeeper.data.Stat;

import com.mapr.security.MaprSecurityException;


public class DumpCommands extends CLIBaseClass implements CLIInterface
{
  public static final String CHANGE_LOG_PARAM_NAME = "loglevel";
  public static final String CHANGE_LOG_CLASSNAME_PARAM_NAME = "classname";
  private static final String DUMP_CONTAINER_ID_PARAM_NAME = "ids";
  private static final String MEMORY_TRIM_PARAM_NAME = "trim";
  private static final String DUMP_CONTAINER_TYPE_PARAM_NAME = "type";
  private static final String DUMP_VOLUME_NAME_PARAM_NAME = "volumename";
  private static final String DUMP_REPLICATION_MANAGER_QUEUE_PARAM_NAME = "queue";
  private static final String DUMP_FILESERVER_IP_PARAM_NAME = "fileserverip";
  private static final String DUMP_FILESERVER_PORT_PARAM_NAME = "fileserverport";  
  public static final String ENODES_PARAM = "enodes";
  public static final String DUMP_PARAMS = "params";

  private static final String RBALINFO_STATS_OPT = "stats";
  private static final String RBALINFO_VOLUME_OPT = "volumeinfo";
  private static final String RBALINFO_NAMECNTRS_OPT = "namecntrinfo";
  
  public static final String MULTI_ARG_SEP = ",";
  private static final Logger LOG = Logger.getLogger(DumpCommands.class);
  
  static String dumpFileServerWorkInfoUsage = "fileserverworkinfo -cldbip " +
  		"cldbip -cldbport cldbport -fileserverip ip/host -fileserverport port";
  static String dumpContainerInfoUsage = "containerinfo -cldbip cldbip -cldbport " +
      " cldbport -ids id1,id2,id3";
  static String dumpReplicationManagerQueueInfoUsage = "replicationmanagerqueueinfo -queue <queue>";
  static String dumpVolumeInfoUsage = "volumeinfo -cldbip cldbip -cldbport cldbport " +
  		"-volumename <volume name>";
  static String dumpVolumeNodesUsage = "volumenodes -cldbip cldbip -cldbport cldbport " +
  "-volumename <volume name>";
  static String dumpReplicationManagerUsage = "replicationmanager -cldbip " +
  		"cldbip -cldbport cldbport";
  static String dumpRereplicationUsage = "rereplication -cldbip " +
  		"cldbip -cldbport cldbport";
  static String dumpBalancerUsage = "balancerinfo -cldbip " +
  		"cldbip -cldbport cldbport";
  static String dumpRoleBalancerUsage = "rolebalancerinfo "
  		+ "[-" + RBALINFO_STATS_OPT + "] "
  		+ "[-" + RBALINFO_VOLUME_OPT + " "
  		+ "-" + DUMP_VOLUME_NAME_PARAM_NAME + "<name>" + "]";
  static String dumpBalancerMetricsUsage = "balancermetrics -cldbip " +
  		"cldbip -cldbport cldbport";
  static String dumpRoleBalancerMetricsUsage = "rolebalancermetrics -cldbip " +
  		"cldbip -cldbport cldbport";
  static String dumpRereplicationMetricsUsage = "rereplicationmetrics " +
                "-cldbip cldbip -cldbport cldbport";
  static String cldbNodesUsage = " cldbnodes -zkconnect <zkconnectstring>";
  static String containersType = "<offline|resync|bm|waiting|unused>";
  static String dumpContainersUsage = "containers -type " + containersType;
  static String dumpCLDBMemoryInfoUsage = "cldbmemoryinfo ";
  static String dumpAssignVouchersUsage = "assignvouchers " +
                "-volumename volumename";
  static String dumpActiveWritersUsage = "activewriters";

  static final String[] mallocFileds = { "arena", "ordblks", 
    "smblks", "hblks", "hblkhd", "usmblks", "fsmblks",
    "uordblks", "fordblks", "keepcost" };
  /*
   * List of common parameters to all commands.
   */
  public static Map<String, BaseInputParameter> baseParams = 
    new ImmutableMap.Builder<String, BaseInputParameter>()
    .put(MapRCliUtil.CLUSTER_NAME_PARAM, new TextInputParameter(
        MapRCliUtil.CLUSTER_NAME_PARAM, "cluster_name",
         CLIBaseClass.NOT_REQUIRED, null))    
    .build();

  static final CLICommand cldbNodesCommand = new CLICommand(
      "cldbnodes",
      "usage : " + cldbNodesUsage,
      DumpCommands.class, 
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
          .putAll(DumpCommands.baseParams)
          .put(NodesCommonUtils.ZK_CONNECTSTRING, new TextInputParameter(
              NodesCommonUtils.ZK_CONNECTSTRING, "ZK Connect String",
              CLIBaseClass.REQUIRED, null))
          .build(),
       null       
    ).setShortUsage(cldbNodesUsage);  
  
  static final CLICommand dumpContainersCommand = new CLICommand(
      "containers",
      "",
      DumpCommands.class, ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(DumpCommands.baseParams)
      .put(DumpCommands.DUMP_CONTAINER_TYPE_PARAM_NAME,
           new TextInputParameter(
               DumpCommands.DUMP_CONTAINER_TYPE_PARAM_NAME,
               containersType,
               CLIBaseClass.REQUIRED,
               null))      
          .build(),
       null       
    ).setShortUsage(dumpContainersUsage);

  static final CLICommand dumpContainerInfo = new CLICommand(
      "containerinfo",
      "usage : " + dumpContainerInfoUsage,
      DumpCommands.class, 
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(DumpCommands.baseParams)
          
           .put(DumpCommands.DUMP_CONTAINER_ID_PARAM_NAME,
             new TextInputParameter(
                 DumpCommands.DUMP_CONTAINER_ID_PARAM_NAME,
                 "ids",
                 CLIBaseClass.REQUIRED,
                 null))      
          .build(),
       null       
    ).setShortUsage(dumpContainerInfoUsage);

  static final CLICommand dumpReplicationManagerInfo = new CLICommand(
      "replicationmanagerinfo",
      "usage : " + dumpContainerInfoUsage,
      DumpCommands.class, 
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(DumpCommands.baseParams)
          
          .put(DumpCommands.DUMP_VOLUME_NAME_PARAM_NAME,
               new TextInputParameter(
                   DumpCommands.DUMP_VOLUME_NAME_PARAM_NAME,
                   "volumename",
                   CLIBaseClass.REQUIRED,
                   null))   
           .build(),
       null       
    ).setShortUsage(dumpReplicationManagerUsage);
  
  
  
  static final CLICommand dumpReplicationManagerQueueInfo = new CLICommand(
      "replicationmanagerqueueinfo",
      "usage : " + dumpReplicationManagerQueueInfoUsage,
      DumpCommands.class, 
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(DumpCommands.baseParams)
          
          .put(DumpCommands.DUMP_REPLICATION_MANAGER_QUEUE_PARAM_NAME,
               new IntegerInputParameter(
                   DumpCommands.DUMP_REPLICATION_MANAGER_QUEUE_PARAM_NAME,
                   "queue",
                   CLIBaseClass.REQUIRED,
                   null))   
           .build(),
       null       
    ).setShortUsage(dumpReplicationManagerQueueInfoUsage);

  static final CLICommand dumpRereplicationInfo = new CLICommand(
      "rereplicationinfo",
      "usage : " + dumpRereplicationUsage,
      DumpCommands.class, 
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(DumpCommands.baseParams)
           .build(),
       null       
    ).setShortUsage(dumpRereplicationUsage);

  static final CLICommand dumpBalancerInfo = new CLICommand(
      "balancerinfo",
      "usage : " + dumpBalancerUsage,
      DumpCommands.class, 
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(DumpCommands.baseParams)
           .build(),
       null
    ).setShortUsage(dumpBalancerUsage);

  static final CLICommand dumpBalancerMetrics = new CLICommand(
      "balancermetrics",
      "usage : " + dumpBalancerMetricsUsage,
      DumpCommands.class, 
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(DumpCommands.baseParams)
           .build(),
       null       
    ).setShortUsage(dumpBalancerMetricsUsage);

  static final CLICommand dumpRoleBalancerMetrics = new CLICommand(
      "rolebalancermetrics",
      "usage : " + dumpRoleBalancerMetricsUsage,
      DumpCommands.class, 
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(DumpCommands.baseParams)
           .build(),
       null       
    ).setShortUsage(dumpRoleBalancerMetricsUsage);

  static final CLICommand dumpRereplicationMetrics = new CLICommand(
      "rereplicationmetrics",
      "usage : " + dumpRereplicationMetricsUsage,
      DumpCommands.class, 
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(DumpCommands.baseParams)
           .build(),
       null       
    ).setShortUsage(dumpRereplicationMetricsUsage);

  static final CLICommand dumpAssignVouchers = new CLICommand(
      "assignvouchers",
      "usage : " + dumpAssignVouchersUsage,
      DumpCommands.class, 
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(DumpCommands.baseParams)
      .put(DumpCommands.DUMP_VOLUME_NAME_PARAM_NAME,
          new TextInputParameter(
              DumpCommands.DUMP_VOLUME_NAME_PARAM_NAME,
              "volumename",
              CLIBaseClass.REQUIRED,
              null))
      .build(),
      null
    ).setShortUsage(dumpAssignVouchersUsage);

  static final CLICommand dumpActiveWriters = new CLICommand(
      "activewriters",
      "usage : " + dumpActiveWritersUsage,
      DumpCommands.class, 
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(DumpCommands.baseParams)
      .build(),
      null
    ).setShortUsage(dumpActiveWritersUsage);

  static final CLICommand dumpFileServerWorkInfo = new CLICommand(
      "fileserverworkinfo",
      "usage : " + dumpFileServerWorkInfoUsage,
      DumpCommands.class, 
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(DumpCommands.baseParams)
          
          .put(DumpCommands.DUMP_FILESERVER_IP_PARAM_NAME,
             new TextInputParameter(
                 DumpCommands.DUMP_FILESERVER_IP_PARAM_NAME,
                 "fileserverip/host",
                 CLIBaseClass.REQUIRED,
                 "127.0.0.1"))
          .put(DumpCommands.DUMP_FILESERVER_PORT_PARAM_NAME,
             new IntegerInputParameter(
                 DumpCommands.DUMP_FILESERVER_PORT_PARAM_NAME,
                 "fileserverport",
                 CLIBaseClass.REQUIRED,
                 5660))  
           .build(),
       null       
    ).setShortUsage(dumpFileServerWorkInfoUsage);
  
  static final CLICommand dumpVolumeInfo = new CLICommand(
	      "volumeinfo",
	      "usage : " + dumpVolumeInfoUsage,
	      DumpCommands.class, 
	      ExecutionTypeEnum.NATIVE,
	      new ImmutableMap.Builder<String, BaseInputParameter>()
	      .putAll(DumpCommands.baseParams)
	          
	           .put(DumpCommands.DUMP_VOLUME_NAME_PARAM_NAME,
	             new TextInputParameter(
	                 DumpCommands.DUMP_VOLUME_NAME_PARAM_NAME,
	                 "volumename",
	                 CLIBaseClass.REQUIRED,
	                 null))      
	          .build(),
	       null       
	    ).setShortUsage(dumpVolumeInfoUsage);


  static final CLICommand dumpVolumeNodes = new CLICommand(
        "volumenodes",
        "usage : " + dumpVolumeNodesUsage,
        DumpCommands.class, 
        ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(DumpCommands.baseParams)
            
             .put(DumpCommands.DUMP_VOLUME_NAME_PARAM_NAME,
               new TextInputParameter(
                   DumpCommands.DUMP_VOLUME_NAME_PARAM_NAME,
                   "volumename",
                   CLIBaseClass.REQUIRED,
                   null))      
            .build(),
         null       
      ).setShortUsage(dumpVolumeNodesUsage);

  static final CLICommand dumpZKInfo = new CLICommand(
		  "zkinfo", "Dumping whole ZK Hierarchy into output",
		  DumpCommands.class,
		  ExecutionTypeEnum.NATIVE,
		  new ImmutableMap.Builder<String, BaseInputParameter>()
		  .putAll(DumpCommands.baseParams)
		  .put(NodesCommonUtils.ZK_CONNECTSTRING, new TextInputParameter(
				  NodesCommonUtils.ZK_CONNECTSTRING, "ZK Connect String",
				  CLIBaseClass.NOT_REQUIRED, null)).build(), null).setShortUsage("Dumping ZK hierarchy info");
  
  static final CLICommand supportDumpTrigger = new CLICommand(
		  "supportdump", "Triggering nodes to gather support dump",
		  DumpCommands.class,
		  ExecutionTypeEnum.NATIVE,
		  new ImmutableMap.Builder<String, BaseInputParameter>()
		  .putAll(DumpCommands.baseParams)
		  .put(NodeServicesManagementCommand.NODES_PARAM, new TextInputParameter(NodeServicesManagementCommand.NODES_PARAM, "node names for which support dump is needed. Space separated. Will be all if parameter is not given", CLIBaseClass.NOT_REQUIRED, null))
		  .put(DUMP_PARAMS, new TextInputParameter(DUMP_PARAMS, "parameters string to create a dump", CLIBaseClass.NOT_REQUIRED, null))
//		  .put(ENODES_PARAM, new TextInputParameter(ENODES_PARAM, "node names for which support dump NOT needed. Space separated.", CLIBaseClass.NOT_REQUIRED, null))
		  .put(NodesCommonUtils.ZK_CONNECTSTRING, new TextInputParameter(
				  NodesCommonUtils.ZK_CONNECTSTRING, "ZK Connect String",
				  CLIBaseClass.NOT_REQUIRED, null)).build(), null).setShortUsage("Triggering nodes to gather support dump");

  public DumpCommands(ProcessedInput input, 
      CLICommand cliCommand) 
      throws CLIProcessingException {
    super(input, cliCommand);
  } 
  
  void init() throws CLIProcessingException {
   
    try {
      int port = Rpc.initialize (0, 0, null); // Client
      if (port < 0)
        throw new IOException("Failed to initalize RPC");
    } catch (Exception e) {
        /**
        * <MAPR_ERROR>
        * Message:Exception in Rpc.initialize <error>
        * Function:DumpCommands.init()
        * Meaning:An error occurred: failed to Initialze RPC.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        LOG.error("Exception in Rpc.initialize " + e);
    }
  }

  /**
   * Public function called to execute the sub-commands of the 'dump' command.
   * It is called by 'executeCommand' defined in the parent class.
   * 
   * @see CLIBaseClass
   */
  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    if (cliCommand.getCommandName().equalsIgnoreCase("zkinfo")) {
      return dumpZKInfo();
    }
    if (cliCommand.getCommandName().equalsIgnoreCase("supportdump")) {
      return supportDumpTrigger();
    }
    init();
    if (!super.validateInput()) {
      return output;
    }
    if (cliCommand.getCommandName().equalsIgnoreCase("containerinfo")) {
      try {
        return dumpContainerInfo();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      }
    } else if (cliCommand.getCommandName().equalsIgnoreCase("containers")) {
      try {
        return dumpContainers();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      }
    } else if (cliCommand.getCommandName().equalsIgnoreCase("volumenodes")) {
      try {
        return dumpVolumeNodes();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      }
    } else if (cliCommand.getCommandName().equalsIgnoreCase("volumeinfo")) {
      try {
        return dumpVolumeInfo();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      } 
    } else if (cliCommand.getCommandName().equalsIgnoreCase("replicationmanagerinfo")) {
      try {
        return dumpReplicationManagerInfo();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      } 
    } else if (cliCommand.getCommandName().equalsIgnoreCase("rereplicationinfo")) {
      try {
        return dumpRereplicationInfo();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      } 
    } else if (cliCommand.getCommandName().equalsIgnoreCase("replicationmanagerqueueinfo")) {
      try {
        return dumpReplicationManagerQueueInfo();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      } 
    } else if (cliCommand.getCommandName().equalsIgnoreCase("balancerinfo")) {
      try {
        return dumpBalancerInfo();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      } 
    } else if (cliCommand.getCommandName().equalsIgnoreCase("rolebalancerinfo")) {
      try {
        return dumpRoleBalancerInfo();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      } 
    } else if (cliCommand.getCommandName().equalsIgnoreCase("balancermetrics")) {
      try {
        return dumpBalancerMetrics();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      } 
    } else if (cliCommand.getCommandName().equalsIgnoreCase("rolebalancermetrics")) {
      try {
        return dumpRoleBalancerMetrics();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      } 
    } else if (cliCommand.getCommandName().equalsIgnoreCase("rereplicationmetrics")) {
      try {
        return dumpRereplicationMetrics();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      } 
    } else if (cliCommand.getCommandName().equalsIgnoreCase("assignvouchers")) {
      try {
        return dumpAssignVouchers();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      } 
    } else if (cliCommand.getCommandName().equalsIgnoreCase("activewriters")) {
      try {
        return dumpActiveWriters();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      }
    } else if (cliCommand.getCommandName().equalsIgnoreCase("fileserverworkinfo")) {
      try {
        return dumpFileServerWorkInfo();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      } 
      
    } else if (cliCommand.getCommandName().equalsIgnoreCase("cldbnodes")) {
      try {
        return dumpKvStoreCidInfo();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      }
    } else if (cliCommand.getCommandName().equalsIgnoreCase("cldbmemoryinfo")) {
      try {
        return dumpCLDBMemoryInfo();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      }
    } 
    return new TextCommandOutput(("Dump command failed").getBytes());
  } 
  

private CommandOutput dumpFileServerWorkInfo() 
                      throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy(); 
    CommandOutput output = new CommandOutput();
    output.setOutput(out);
   
    String fileServerIpString = getParamTextValue(DUMP_FILESERVER_IP_PARAM_NAME, 0);
    if ( fileServerIpString.equalsIgnoreCase("localhost") || fileServerIpString.equalsIgnoreCase("127.0.0.1") ) {
    	fileServerIpString = "127.0.0.1";
    } else {
		List<String> ips = NodesCommonUtils.convertHostToIp(Collections.singletonList(fileServerIpString));
		if ( ips.isEmpty() ) {
		  out.addError(new OutputError(Errno.EINVAL, "Can not get valid IP address out of provided name: " + fileServerIpString));
		  return output;
	    }
		fileServerIpString = ips.get(0);
    }
    int fileServerIp = Util.ipToInt(fileServerIpString);
    int fileServerPort = getParamIntValue(DUMP_FILESERVER_PORT_PARAM_NAME, 0);
    IPAddress fileServer = IPAddress.newBuilder()
                                .setHost(fileServerIp)
                                .setPort(fileServerPort)
                                .build();
    // FileServer Work Info 
    LOG.info("dumpFileServerWorkInfo : Dumping FileServerWorkInfo");  
    boolean hasMore = false;
    int lastContainerId = 0;
    String lastSpId = "";
    do {
      LOG.info("dumpFileServerWorkInfo : Dumping fileServer Work Units");
      
      DumpInfoRequest req = DumpInfoRequest.newBuilder()
              .setDumpOp(DumpInfoRequest.DumpOp.FILESERVER_WORK_QUEUE_OP)
              .setFileServer(fileServer)
              .setFileServerLastContainer(lastContainerId)
              .setCreds(getUserCredentials())
              .setFileServerLastSpId(lastSpId)
              .build();
      byte[] replyData;
      try {
        if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
          replyData = CLDBRpcCommonUtils.getInstance()
            .sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
                Common.MapRProgramId.CldbProgramId.getNumber(),
                CLDBProto.CLDBProg.DumpInfoProc.getNumber(), 
                req, 
                DumpInfoResponse.class);
        } else {
          replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
              Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg
                .DumpInfoProc.getNumber(), req,
                DumpInfoResponse.class);
        }
        if (replyData == null) {
          /**
          * <MAPR_ERROR>
          * Message:dump fileserverworkinfo RPC failed
          * Function:DumpCommands.dumpFileServerWorkInfo()
          * Meaning:An error occurred.
          * Resolution:Contact technical support.
          * </MAPR_ERROR>
          */
          out.addError(new OutputError(Errno.ERPCFAILED, 
          "Couldn't connect to the CLDB service. dump fileserverworkinfo RPC failed"));
          output.setOutput(out);
          return output;
        }
        DumpInfoResponse resp = DumpInfoResponse.parseFrom(replyData);
        if (resp.getStatus() != 0) {
          /**
          * <MAPR_ERROR>
          * Message:dump fileserverworkinfo failed with status : <status>
          * Function:DumpCommands.dumpFileServerWorkInfo()
          * Meaning:An error occurred.
          * Resolution:Contact technical support.
          * </MAPR_ERROR>
          */
          out.addError(new OutputError(resp.getStatus(), "dump " +
              "fileserverworkinfo failed with status : " + resp.getStatus()));
          output.setOutput(out);
          return output;
        }
        for (FileServerPendingWork cmd : resp.getFileServerCommandsList()) {
          out.addNode(formatFileServerCommand(cmd));
          lastContainerId = cmd.getCmd().getContainerId();
          lastSpId = cmd.getSpId();
        }
        hasMore = resp.getFileServerHasMore();        
      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
      } catch (Exception e) {
        /**
        * <MAPR_ERROR>
        * Message:dump fileServerWorkInfo failed
        * Function:DumpCommands.dumpFileServerWorkInfo()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        out.addError(new OutputError(Errno.ERPCFAILED, 
            "dump fileServerWorkInfo failed"));
        output.setOutput(out);
        return output;
      }
    } while (hasMore == true);
    output.setOutput(out); 
    return output;
  }

  private CommandOutput dumpReplicationManagerQueueInfo() 
    throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy(); 
    CommandOutput output = new CommandOutput();
    output.setOutput(out);
   
    int queueNumber = getParamIntValue(DUMP_REPLICATION_MANAGER_QUEUE_PARAM_NAME, 0);
    
    if ((queueNumber < 0) || (queueNumber > 5)){
      /**
       * <MAPR_ERROR>
       * Message:Invalid queue number <volume>
       * Function:DumpCommands.dumpReplicationManagerInfo()
       * Meaning:An error occurred.
       * Resolution:Contact technical support.
       * </MAPR_ERROR>
       */
      out.addError(new OutputError(Errno.EINVAL, 
          "Invalid queue number " + queueNumber + ". Valid values are 0 to 5"));
      output.setOutput(out);
      return output;
    }
    
    // Replication manager 
    LOG.info("dumpReplicationManagerQueueInfo : Dumping ReplicationManagerQueueInfo");  
    boolean hasMore = false;
    int lastContainerId = 0;
    do {
      LOG.info("dumpReplicationManagerQueueInfo: Request ReplicationmanagerQueue info");
      DumpInfoRequest req = DumpInfoRequest.newBuilder()
              .setDumpOp(DumpInfoRequest.DumpOp.REPLICATION_MANAGER_QUEUE_OP)
              .setReplicationQueue(queueNumber)
              .setReplicationLastContainer(lastContainerId)
              .setCreds(getUserCredentials())
              .build();
      byte[] replyData;
      try {
        if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
          replyData = CLDBRpcCommonUtils.getInstance()
            .sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
                Common.MapRProgramId.CldbProgramId.getNumber(),
                CLDBProto.CLDBProg.DumpInfoProc.getNumber(), 
                req, 
                DumpInfoResponse.class);
        } else {
          replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
              Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg
                .DumpInfoProc.getNumber(), req,
                DumpInfoResponse.class);
        }
        if (replyData == null) {
          /**
          * <MAPR_ERROR>
          * Message:dump replicationmanagerqueueinfo rpc failed
          * Function:DumpCommands.dumpReplicationManagerQueueInfo()
          * Meaning:An error occurred.
          * Resolution:Contact technical support.
          * </MAPR_ERROR>
          */
          out.addError(new OutputError(Errno.ERPCFAILED, 
          "Couldn't connect to the CLDB service. dump replicationmanagerqueueinfo rpc failed"));
          output.setOutput(out);
          return output;
        }
        DumpInfoResponse resp = DumpInfoResponse.parseFrom(replyData);
        if (resp.getStatus() != 0) {
          /**
          * <MAPR_ERROR>
          * Message:dump replicationmanagerinfo volume containersmap failed with status : <status>
          * Function:DumpCommands.dumpReplicationManagerInfo()
          * Meaning:An error occurred.
          * Resolution:Contact technical support.
          * </MAPR_ERROR>
          */
          out.addError(new OutputError(resp.getStatus(), "dump " +
              "replicationmanagerqueueinfo  " +
                "failed with status : " + resp.getStatus()));
          output.setOutput(out);
          return output;
        }
        for (ContainerInfo id : resp.getReplicationContainersList()) {
          out.addNode(formatContainerInfo(id, null));
          lastContainerId = id.getContainerId();
        }
        hasMore = resp.getReplicationHasMore();        
      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
      } catch (Exception e) {
        /**
        * <MAPR_ERROR>
        * Message:dump replicationmanager failed
        * Function:DumpCommands.dumpReplicationManagerInfo()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        out.addError(new OutputError(Errno.ERPCFAILED, "dump replicationmanagerqueueinfo failed"));
        output.setOutput(out);
        return output;
      }
    } while (hasMore == true);
     
    output.setOutput(out);
    return output;
  }  

  private CommandOutput dumpReplicationManagerInfo() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy(); 
    CommandOutput output = new CommandOutput();
    output.setOutput(out);
   
    String volumeName = getParamTextValue(DUMP_VOLUME_NAME_PARAM_NAME, 0);
    
    try {
      VolumeInfo volInfo = lookupVolumeInfo(volumeName);
      if (volInfo == null) {
        /**
        * <MAPR_ERROR>
        * Message:Could not get info for volume: <volume>
        * Function:DumpCommands.dumpReplicationManagerInfo()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        out.addError(new OutputError(Errno.ERPCFAILED, 
            "Could not get info for volume: " + volumeName));
        output.setOutput(out);
        return output;
      }
      out.addNode(formatVolumeInfo(volInfo));
    } catch (Exception e) {
      /**
      * <MAPR_ERROR>
      * Message:Could not get info for volume: <volume> <error>
      * Function:DumpCommands.dumpReplicationManagerInfo()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      out.addError(new OutputError(Errno.ERPCFAILED, 
          "Could not get info for volume: " + volumeName + 
          ", Error: " + e.getLocalizedMessage()));
      output.setOutput(out);
      return output;
    }
    
    // Replication manager 
    LOG.info("dumpReplicationManagerInfo : Dumping ReplicationManagerInfo");  
    boolean hasMore = false;
    int lastContainerId = 0;
    do {
      LOG.info("dumpReplicationManagerInfo: Request Replicationmanager info");
      DumpInfoRequest req = DumpInfoRequest.newBuilder()
              .setDumpOp(DumpInfoRequest.DumpOp.REPLICATION_MANAGER_OP)
              .setReplicationVolumeName(volumeName)
              .setReplicationLastContainer(lastContainerId)
              .setCreds(getUserCredentials())
              .build();
      byte[] replyData;
      try {
        if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
          replyData = CLDBRpcCommonUtils.getInstance()
            .sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
                Common.MapRProgramId.CldbProgramId.getNumber(),
                CLDBProto.CLDBProg.DumpInfoProc.getNumber(), 
                req, 
                DumpInfoResponse.class);
        } else {
          replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
              Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg
                .DumpInfoProc.getNumber(), req,
                DumpInfoResponse.class);
        }
        if (replyData == null) {
          /**
          * <MAPR_ERROR>
          * Message:dump replicationmanagerinfo rpc failed
          * Function:DumpCommands.dumpReplicationManagerInfo()
          * Meaning:An error occurred.
          * Resolution:Contact technical support.
          * </MAPR_ERROR>
          */
          out.addError(new OutputError(Errno.ERPCFAILED, 
          "Couldn't connect to the CLDB service. dump replicationmanagerinfo rpc failed"));
          output.setOutput(out);
          return output;
        }
        DumpInfoResponse resp = DumpInfoResponse.parseFrom(replyData);
        if (resp.getStatus() != 0) {
          /**
          * <MAPR_ERROR>
          * Message:dump replicationmanagerinfo volume containersmap failed with status : <status>
          * Function:DumpCommands.dumpReplicationManagerInfo()
          * Meaning:An error occurred.
          * Resolution:Contact technical support.
          * </MAPR_ERROR>
          */
          out.addError(new OutputError(resp.getStatus(), "dump " +
              "replicationmanagerinfo volume " +
                "containersmap failed with status : " + resp.getStatus()));
          output.setOutput(out);
          return output;
        }
        for (ContainerInfo id : resp.getReplicationContainersList()) {
          out.addNode(formatContainerInfo(id, null));
          lastContainerId = id.getContainerId();
        }
        hasMore = resp.getReplicationHasMore();        
      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
      } catch (Exception e) {
        /**
        * <MAPR_ERROR>
        * Message:dump replicationmanager failed
        * Function:DumpCommands.dumpReplicationManagerInfo()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        out.addError(new OutputError(Errno.ERPCFAILED, "dump replicationmanager failed"));
        output.setOutput(out);
        return output;
      }
    } while (hasMore == true);
     
    output.setOutput(out);
    return output;
  }
  
  private CommandOutput dumpRereplicationInfo() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy(); 
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    LOG.info("dumpRereplicationInfo : Dumping Rereplication info");
    boolean hasMore = false;
    int lastCid = 0;
    int ncontainers = 0;
    do {
      LOG.info("dumpReplicationInfo : Dumping Active Container Copy Creates");

      DumpInfoRequest req = DumpInfoRequest.newBuilder()
              .setDumpOp(DumpInfoRequest.DumpOp.ACTIVE_REREPL_OP)
              .setLastCopyingCid(lastCid)
              .setCreds(getUserCredentials())
              .build();
      byte[] replyData;
      try {
        if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
          replyData = CLDBRpcCommonUtils.getInstance()
            .sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
                Common.MapRProgramId.CldbProgramId.getNumber(),
                CLDBProto.CLDBProg.DumpInfoProc.getNumber(),
                req,
                DumpInfoResponse.class);
        } else {
          replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
              Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg
                .DumpInfoProc.getNumber(), req,
                DumpInfoResponse.class);
        }
        if (replyData == null) {
          /**
          * <MAPR_ERROR>
          * Message:dump rereplication RPC failed
          * Function:DumpCommands.dumpRereplicationInfo()
          * Meaning:An error occurred.
          * Resolution:Contact technical support.
          * </MAPR_ERROR>
          */
          out.addError(new OutputError(Errno.ERPCFAILED,
            "Couldn't connect to the CLDB service. dump rereplicationinfo " +
            " RPC failed"));
          output.setOutput(out);
          return output;
        }
        DumpInfoResponse resp = DumpInfoResponse.parseFrom(replyData);
        if (resp.getStatus() != 0) {
          /**
          * <MAPR_ERROR>
          * Message:dump rereplication failed with status : <status>
          * Function:DumpCommands.dumpRereplicationInfo()
          * Meaning:An error occurred.
          * Resolution:Contact technical support.
          * </MAPR_ERROR>
          */
          out.addError(new OutputError(resp.getStatus(), "dump " +
              "rereplication failed with status : " + resp.getStatus()));
          output.setOutput(out);
          return output;
        }
        for (ActiveContainerCopyCreate ac : resp.getActiveCopyCreatesList()) {
          out.addNode(formatActiveCopyCreate(ac));
          lastCid = ac.getContainerId();
          ++ncontainers;
        }
        hasMore = resp.getMoreActiveRoleSwitches();
      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
      } catch (Exception e) {
        /**
        * <MAPR_ERROR>
        * Function:DumpCommands.dumpRereplicationInfo()
          * Message:dump rereplication failed
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        out.addError(new OutputError(Errno.ERPCFAILED,
            "dump rereplication failed"));
        output.setOutput(out);
        return output;
      }
    } while (hasMore);

    if (ncontainers == 0) {
      return new TextCommandOutput(("No active rereplications").getBytes());
    } else {
      output.setOutput(out);
      return output;
    }
  }

  private CommandOutput dumpRereplicationMetrics() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy(); 
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    LOG.info("dumpRereplicationMetrics : Dumping Rereplication metrics");

    DumpInfoRequest req = DumpInfoRequest.newBuilder()
            .setDumpOp(DumpInfoRequest.DumpOp.METRICS_REREPLICATION_OP)
            .setCreds(getUserCredentials())
            .build();
    byte[] replyData;
    try {
      if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        replyData = CLDBRpcCommonUtils.getInstance()
          .sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
              Common.MapRProgramId.CldbProgramId.getNumber(),
              CLDBProto.CLDBProg.DumpInfoProc.getNumber(),
              req,
              DumpInfoResponse.class);
      } else {
        replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
            Common.MapRProgramId.CldbProgramId.getNumber(),
          CLDBProto.CLDBProg
              .DumpInfoProc.getNumber(), req,
              DumpInfoResponse.class);
      }
      if (replyData == null) {
        /**
        * <MAPR_ERROR>
        * Message:dump rereplicationmetrics RPC failed
        * Function:DumpCommands.dumpRereplicationMetrics()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        out.addError(new OutputError(Errno.ERPCFAILED,
          "Couldn't connect to the CLDB service. dump rereplicationmetrics RPC " +
          "failed"));
        output.setOutput(out);
        return output;
      }
      DumpInfoResponse resp = DumpInfoResponse.parseFrom(replyData);
      if (resp.getStatus() != 0) {
        /**
        * <MAPR_ERROR>
        * Message:dump rereplicationmetrics failed with status : <status>
        * Function:DumpCommands.dumpRereplicationMetrics()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        out.addError(new OutputError(resp.getStatus(), "dump " +
            "rereplicationmetrics failed with status : " + resp.getStatus()));
        output.setOutput(out);
        return output;
      }

      out.addNode(formatRereplicationMetrics(resp.getRereplicationMetrics()));
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      /**
      * <MAPR_ERROR>
      * Message:dump rereplicationInfo failed
      * Function:DumpCommands.dumpRereplicationMetrics()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      out.addError(new OutputError(Errno.ERPCFAILED,
          "dump rereplicationInfo failed"));
      output.setOutput(out);
      return output;
    }

    output.setOutput(out);
    return output;
  }

  private CommandOutput dumpAssignVouchers() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy(); 
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    LOG.debug("dumpAssignVouchers : Dumping assign vouchers");
    int nvouchers = 0;
    String volumeName = getParamTextValue(DUMP_VOLUME_NAME_PARAM_NAME, 0);
    DumpInfoRequest req = DumpInfoRequest.newBuilder()
            .setDumpOp(DumpInfoRequest.DumpOp.ASSIGN_VOUCHER_DUMP_OP)
            .setVolumeName(volumeName)
            .setCreds(getUserCredentials())
            .build();

    byte[] replyData;
    try {
      if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        replyData = CLDBRpcCommonUtils.getInstance()
          .sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
              Common.MapRProgramId.CldbProgramId.getNumber(),
              CLDBProto.CLDBProg.DumpInfoProc.getNumber(),
              req,
              DumpInfoResponse.class);
      } else {
        replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
            Common.MapRProgramId.CldbProgramId.getNumber(),
          CLDBProto.CLDBProg
              .DumpInfoProc.getNumber(), req,
              DumpInfoResponse.class);
      }
      if (replyData == null) {
        /**
        * <MAPR_ERROR>
        * Message:dump assignvouchers RPC failed
        * Function:DumpCommands.dumpAssignVouchers()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        out.addError(new OutputError(Errno.ERPCFAILED,
          "Couldn't connect to the CLDB service. dump assignvouchers RPC " +
          "failed"));
        output.setOutput(out);
        return output;
      }
      DumpInfoResponse resp = DumpInfoResponse.parseFrom(replyData);
      if (resp.getStatus() != 0) {
        /**
        * <MAPR_ERROR>
        * Message:dump assignvouchers failed with status : <status>
        * Function:DumpCommands.dumpAssignVouchers()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        out.addError(new OutputError(resp.getStatus(), "dump " +
            "assignvouchers failed with status : " + resp.getStatus()));
        output.setOutput(out);
        return output;
      }

      for (ActiveAssignVoucher av : resp.getVouchersList()) {
        out.addNode(formatActiveAssignVoucher(av));
        ++nvouchers;
      }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      /**
      * <MAPR_ERROR>
      * Message:dump assignvouchers failed
      * Function:DumpCommands.dumpAssignVouchers()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      out.addError(new OutputError(Errno.ERPCFAILED,
          "dump assignvouchers failed"));
      output.setOutput(out);
      return output;
    }

    if (nvouchers == 0) {
      return new TextCommandOutput(("No active assign vouchers").getBytes());
    } else {
      output.setOutput(out);
      return output;
    }
  }

  private CommandOutput dumpActiveWriters() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy(); 
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    LOG.debug("dumpActiveWriters : Dumping active writers");
    DumpInfoRequest req = DumpInfoRequest.newBuilder()
      .setDumpOp(DumpInfoRequest.DumpOp.ACTIVE_WRITERS_DUMP_OP)
      .setCreds(getUserCredentials())
      .build();

    byte[] replyData = null;
    int numWriters = 0;
    try {
      if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        replyData = CLDBRpcCommonUtils.getInstance()
          .sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
              Common.MapRProgramId.CldbProgramId.getNumber(),
              CLDBProto.CLDBProg.DumpInfoProc.getNumber(),
              req,
              DumpInfoResponse.class);
      } else {
        replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
            Common.MapRProgramId.CldbProgramId.getNumber(),
          CLDBProto.CLDBProg
              .DumpInfoProc.getNumber(), req,
              DumpInfoResponse.class);
      }
      if (replyData == null) {
        out.addError(new OutputError(Errno.ERPCFAILED,
          "Couldn't connect to the CLDB service. dump activewriters RPC " +
          "failed"));
        output.setOutput(out);
        return output;
      }
      DumpInfoResponse resp = DumpInfoResponse.parseFrom(replyData);
      if (resp.getStatus() != 0) {
        out.addError(new OutputError(resp.getStatus(), "dump " +
            "activewriters failed with status : " + resp.getStatus()));
        output.setOutput(out);
        return output;
      }

      for (ActiveWriter w : resp.getWritersList()) {
        out.addNode(formatActiveWriter(w));
        ++numWriters;
      }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      out.addError(new OutputError(Errno.ERPCFAILED,
          "dump activewriters failed"));
      output.setOutput(out);
      return output;
    }

    if (numWriters == 0) {
      return new TextCommandOutput(("No active writers").getBytes());
    } else {
      output.setOutput(out);
      return output;
    }
  }

  private CommandOutput dumpBalancerInfo() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy(); 
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    LOG.info("dumpBalancerInfo : Dumping Balancer info");
    boolean hasMore = false;
    String lastSpid = null;
    do {
      LOG.info("dumpBalancerInfo : Dumping Active Storage Pools");

      DumpInfoRequest.Builder reqBuilder = DumpInfoRequest.newBuilder()
        .setDumpOp(DumpInfoRequest.DumpOp.BALANCER_SP_OP)                 
        .setCreds(getUserCredentials());
      if (lastSpid != null)
        reqBuilder.setLastSpId(lastSpid);
      DumpInfoRequest req = reqBuilder.build();

      byte[] replyData;
      try {
        if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
          replyData = CLDBRpcCommonUtils.getInstance()
            .sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
                Common.MapRProgramId.CldbProgramId.getNumber(),
                CLDBProto.CLDBProg.DumpInfoProc.getNumber(),
                req,
                DumpInfoResponse.class);
        } else {
          replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
              Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg
                .DumpInfoProc.getNumber(), req,
                DumpInfoResponse.class);
        }
        if (replyData == null) {
          /**
          * <MAPR_ERROR>
          * Message:dump balancerinfo RPC failed
          * Function:DumpCommands.dumpBalancerInfo()
          * Meaning:An error occurred.
          * Resolution:Contact technical support.
          * </MAPR_ERROR>
          */
          out.addError(new OutputError(Errno.ERPCFAILED,
          "Couldn't connect to the CLDB service. dump balancerinfo RPC failed"));
          output.setOutput(out);
          return output;
        }
        DumpInfoResponse resp = DumpInfoResponse.parseFrom(replyData);
        if (resp.getStatus() != 0) {
          /**
          * <MAPR_ERROR>
          * Message:dump balancerinfo failed with status : <status>
          * Function:DumpCommands.dumpBalancerInfo()
          * Meaning:An error occurred.
          * Resolution:Contact technical support.
          * </MAPR_ERROR>
          */
          out.addError(new OutputError(resp.getStatus(), "dump " +
              "balancerinfo failed with status : " + resp.getStatus()));
          output.setOutput(out);
          return output;
        }
        for (ActiveStoragePool sp : resp.getActiveSpsList()) {
          out.addNode(formatActiveSp(sp));
          lastSpid = sp.getSpid();
        }
        hasMore = resp.getMoreActiveSps();
      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);
      } catch (Exception e) {
        /**
        * <MAPR_ERROR>
        * Message:dump balancerInfo failed
        * Function:DumpCommands.dumpBalancerInfo()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        out.addError(new OutputError(Errno.ERPCFAILED,
            "dump balancerInfo failed"));
        output.setOutput(out);
        return output;
      }
    } while (hasMore);

    hasMore = false;
    int lastCid = 0;
    int ncontainers = 0;
    do {
      LOG.info("dumpBalancerInfo : Dumping Active Container Moves");

      DumpInfoRequest req = DumpInfoRequest.newBuilder()
              .setDumpOp(DumpInfoRequest.DumpOp.BALANCER_MOVE_OP)
              .setLastMovingCid(lastCid)
              .setCreds(getUserCredentials())
              .build();
      byte[] replyData;
      try {
        if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
          replyData = CLDBRpcCommonUtils.getInstance()
            .sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
                Common.MapRProgramId.CldbProgramId.getNumber(),
                CLDBProto.CLDBProg.DumpInfoProc.getNumber(),
                req,
                DumpInfoResponse.class);
        } else {
          replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
              Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg
                .DumpInfoProc.getNumber(), req,
                DumpInfoResponse.class);
        }
        if (replyData == null) {
          /**
          * <MAPR_ERROR>
          * Message:dump balancerinfo RPC failed
          * Function:DumpCommands.dumpBalancerInfo()
          * Meaning:An error occurred.
          * Resolution:Contact technical support.
          * </MAPR_ERROR>
          */
          out.addError(new OutputError(Errno.ERPCFAILED,
          "Couldn't connect to the CLDB service. dump balancerinfo RPC failed"));
          output.setOutput(out);
          return output;
        }
        DumpInfoResponse resp = DumpInfoResponse.parseFrom(replyData);
        if (resp.getStatus() != 0) {
          /**
          * <MAPR_ERROR>
          * Message:dump balancerinfo failed with status : <status>
          * Function:DumpCommands.dumpBalancerInfo()
          * Meaning:An error occurred.
          * Resolution:Contact technical support.
          * </MAPR_ERROR>
          */
          out.addError(new OutputError(resp.getStatus(), "dump " +
              "balancerinfo failed with status : " + resp.getStatus()));
          output.setOutput(out);
          return output;
        }
        for (ActiveContainerMove am : resp.getActiveMovesList()) {
          out.addNode(formatActiveMove(am));
          lastCid = am.getContainerId();
          ++ncontainers;
        }
        hasMore = resp.getMoreActiveMoves();
      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);
      } catch (Exception e) {
        /**
        * <MAPR_ERROR>
        * Message:dump balancerInfo failed
        * Function:DumpCommands.dumpBalancerInfo()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        out.addError(new OutputError(Errno.ERPCFAILED,
            "dump balancerInfo failed"));
        output.setOutput(out);
        return output;
      }
    } while (hasMore);

    if (ncontainers == 0)
      out.addMessage("No active container moves");
    output.setOutput(out);
    return output;
  }
    
  private CommandOutput dumpBalancerMetrics() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy(); 
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    LOG.info("dumpBalancerMetrics : Dumping Balancer metrics");
    DumpInfoRequest req = DumpInfoRequest.newBuilder()
            .setDumpOp(DumpInfoRequest.DumpOp.METRICS_DISKBALANCER_OP)
            .setCreds(getUserCredentials())
            .build();
    byte[] replyData;
    try {
      if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        replyData = CLDBRpcCommonUtils.getInstance()
          .sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
              Common.MapRProgramId.CldbProgramId.getNumber(),
              CLDBProto.CLDBProg.DumpInfoProc.getNumber(),
              req,
              DumpInfoResponse.class);
      } else {
        replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
            Common.MapRProgramId.CldbProgramId.getNumber(),
          CLDBProto.CLDBProg
              .DumpInfoProc.getNumber(), req,
              DumpInfoResponse.class);
      }
      if (replyData == null) {
        /**
        * <MAPR_ERROR>
        * Message:dump balancermetrics RPC failed
        * Function:DumpCommands.dumpBalancerMetrics()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        out.addError(new OutputError(Errno.ERPCFAILED,
          "Couldn't connect to the CLDB service. dump balancermetrics RPC " +
          "failed"));
        output.setOutput(out);
        return output;
      }
      DumpInfoResponse resp = DumpInfoResponse.parseFrom(replyData);
      if (resp.getStatus() != 0) {
        /**
        * <MAPR_ERROR>
        * Message:dump balancermetrics failed with status : <status>
        * Function:DumpCommands.dumpBalancerInfo()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        out.addError(new OutputError(resp.getStatus(), "dump " +
            "balancermetrics failed with status : " + resp.getStatus()));
        output.setOutput(out);
        return output;
      }

      out.addNode(formatDiskBalancerMetrics(resp.getDiskBalancerMetrics()));
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      /**
      * <MAPR_ERROR>
      * Message:dump balancerInfo failed
      * Function:DumpCommands.dumpBalancerMetrics()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      out.addError(new OutputError(Errno.ERPCFAILED,
          "dump balancerInfo failed"));
      output.setOutput(out);
      return output;
    }

    output.setOutput(out);
    return output;
  }

  static final CLICommand dumpCLDBMemoryInfo = new CLICommand(
      "cldbmemoryinfo",
      "usage : " + dumpCLDBMemoryInfoUsage,
      DumpCommands.class, 
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(DumpCommands.baseParams)
      .put(MEMORY_TRIM_PARAM_NAME,
           new BooleanInputParameter(MEMORY_TRIM_PARAM_NAME,
             "shrink cldb native memory if possible",
             CLIBaseClass.NOT_REQUIRED, false)).build(),
      null).setShortUsage(dumpCLDBMemoryInfoUsage);

  private CommandOutput dumpCLDBMemoryInfo() throws CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);

    /* Set just the credentials here and set the rest while parsing the request */
    DumpInfoRequest.Builder dumpinfoReqBuilder = DumpInfoRequest.newBuilder()
        .setCreds(getUserCredentials())
        .setDumpOp(DumpInfoRequest.DumpOp.NATIVE_MEMORY_INFO_OP);

    boolean doTrim = ((isParamPresent(MEMORY_TRIM_PARAM_NAME)) ?
                      getParamBooleanValue(MEMORY_TRIM_PARAM_NAME, 0) : false);
    if (doTrim) {
      dumpinfoReqBuilder.setDumpOp(DumpInfoRequest.DumpOp.NATIVE_MEMORY_TRIM_OP);
    }

    DumpInfoRequest req = dumpinfoReqBuilder.build();

    /* Step 1: Send request and obtain response */
    byte[] replyData;
    try {
      if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
            getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0),
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg.DumpInfoProc.getNumber(),
            req, DumpInfoResponse.class);
      } else {
        replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg.DumpInfoProc.getNumber(),
            req, DumpInfoResponse.class);
      }
      if (replyData == null) {
        out.addError(new OutputError(Errno.ERPCFAILED,
            "Couldn't connect to the CLDB service. dump cldbmemoryinfo RPC failed"));
        return output;
      }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException("MaprSecurityException", e);
    } catch (Exception e) {
      out.addError(new OutputError(Errno.ERPCFAILED, "RPC Error"));
      return output;
    }

    DumpInfoResponse resp;
    try {
      resp = DumpInfoResponse.parseFrom(replyData);
      if (resp.getStatus() != 0) {
        out.addError(new OutputError(resp.getStatus(), "dump " +
            "cldbmemoryinfo failed with status : " + resp.getStatus()));
        return output;
      }
    } catch (InvalidProtocolBufferException e) {
      LOG.error("Exception while trying to parse DumpInfoResponse MemoryInfo  ", e);
      out.addError(new OutputError(Errno.EIO, "Error Parsing NativeMemoryInfo Response"));
      return output;
    }

    NativeMemoryInfo nmi = resp.getMemoryInfo();
    OutputNode cout = new OutputNode("memoryInfo");
    out.addNode(cout);
    cout.addChild(new OutputNode("NumContainersInCache", nmi.getNumContainers()));
    cout.addChild(new OutputNode("CachedContainer_Bytes", nmi.getCachedConBytes()));
    cout.addChild(new OutputNode("NumFSCommands", nmi.getNumFSCommands()));
    cout.addChild(new OutputNode("FSCmdHolder_Bytes", nmi.getNumFSCmdHolderBytes()));
    cout.addChild(new OutputNode("numFSCmdBytes", nmi.getNumFSCmdBytes()));
    cout.addChild(new OutputNode("NumContainerWorks", nmi.getNumContainerWorks()));
    cout.addChild(new OutputNode("ContainerWork_Bytes", nmi.getConWorkBytes()));
    cout.addChild(new OutputNode("ConLocDataBytes", nmi.getConLocDataBytes()));
    cout.addChild(new OutputNode("ConSizeDataBytes", nmi.getConSizeDataBytes()));
    cout.addChild(new OutputNode("ConAllocLocDataBytes", nmi.getConAllocLocDataBytes()));
    cout.addChild(new OutputNode("ConAllocSizeDataBytes", nmi.getConAllocSizeDataBytes()));
    cout.addChild(new OutputNode("TotalAllocBytes", nmi.getTotalAllocBytes()));
    if (nmi.getMallocInfoCount() != 0) {
      int i = 0; 
      OutputNode mout = new OutputNode("MallocInfo");
      cout.addNode(mout);
      for (Long val : nmi.getMallocInfoList()) {
        String name = ((i < mallocFileds.length) ? mallocFileds[i] : new
                       String("MallocInfo" + i));
        mout.addChild(new OutputNode(name, val));
        ++i;
      }
    }
    return output;
  }

  /* Initialize a CLICommand to dump rolebalancer info */
  static final CLICommand dumpRoleBalancerInfo = new CLICommand(
      "rolebalancerinfo" /* commandName */,
      "usage : " + dumpRoleBalancerUsage /* commandUsage */,
      DumpCommands.class /* class extending CLIInterface */, 
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(DumpCommands.baseParams)
      .put(RBALINFO_STATS_OPT, 
          new NoValueInputParameter(RBALINFO_STATS_OPT,
              "Gets RoleBalancer AcitveSwitches Info",
              false, false))
      .put(RBALINFO_VOLUME_OPT,
          new NoValueInputParameter(RBALINFO_VOLUME_OPT,
              "Gets Balancing Info for Volumes(s)",
              false, false))
      .put(DUMP_VOLUME_NAME_PARAM_NAME,
          new TextInputParameter(DUMP_VOLUME_NAME_PARAM_NAME,
              "Specifies the name of the volumes",
              false, null))
      .put(RBALINFO_NAMECNTRS_OPT,
          new NoValueInputParameter(RBALINFO_NAMECNTRS_OPT,
              "Get NameContainers Info",
              false, false)).build(),
              null   /* subCommands (CLICommand [])*/
  ).setShortUsage(dumpRoleBalancerUsage);
  
  /**
   * The function is called in response to 'maprcli dump rolebalancerinfo'
   * 
   * @return
   * @throws CLIProcessingException
   */
  private CommandOutput dumpRoleBalancerInfo() throws CLIProcessingException {
    if (isParamPresent(RBALINFO_VOLUME_OPT) || 
        isParamPresent(RBALINFO_NAMECNTRS_OPT)) {
      return getRBalBalancingInfo();
    }
    else {
      return getRoleBalancerStats();
    }
  }

  CommandOutput getRoleBalancerStats() throws CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);

    LOG.info("dumpRoleBalancerInfo : Dumping RoleBalancer info");
    boolean hasMore = false;
    int lastCid = 0;
    int ncontainers = 0;
    do {
      LOG.info("dumpRoleBalancerInfo : Dumping Active Container Role switches");

      DumpInfoRequest req = DumpInfoRequest.newBuilder()
              .setDumpOp(DumpInfoRequest.DumpOp.BALANCER_ROLESWITCH_OP)
              .setLastRoleSwitchCid(lastCid)
              .setCreds(getUserCredentials())
              .build();
      byte[] replyData;
      try {
        if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
          replyData = CLDBRpcCommonUtils.getInstance()
            .sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
                Common.MapRProgramId.CldbProgramId.getNumber(),
                CLDBProto.CLDBProg.DumpInfoProc.getNumber(),
                req,
                DumpInfoResponse.class);
        } else {
          replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
              Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg
                .DumpInfoProc.getNumber(), req,
                DumpInfoResponse.class);
        }
        if (replyData == null) {
          /**
          * <MAPR_ERROR>
          * Message:dump rolebalancerinfo RPC failed
          * Function:DumpCommands.dumpRoleBalancerInfo()
          * Meaning:An error occurred.
          * Resolution:Contact technical support.
          * </MAPR_ERROR>
          */
          out.addError(new OutputError(Errno.ERPCFAILED,
            "Couldn't connect to the CLDB service. dump rolebalancerinfo RPC failed"));
          output.setOutput(out);
          return output;
        }
        DumpInfoResponse resp = DumpInfoResponse.parseFrom(replyData);
        if (resp.getStatus() != 0) {
          /**
          * <MAPR_ERROR>
          * Message:dump rolebalancerinfo failed with status : <status>
          * Function:DumpCommands.dumpRoleBalancerInfo()
          * Meaning:An error occurred.
          * Resolution:Contact technical support.
          * </MAPR_ERROR>
          */
          out.addError(new OutputError(resp.getStatus(), "dump " +
              "rolebalancerinfo failed with status : " + resp.getStatus()));
          output.setOutput(out);
          return output;
        }
        for (ActiveContainerRoleSwitch am : resp.getActiveRoleSwitchesList()) {
          out.addNode(formatActiveRoleSwitch(am));
          lastCid = am.getContainerId();
          ++ncontainers;
        }
        hasMore = resp.getMoreActiveRoleSwitches();
      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
      } catch (Exception e) {
        /**
        * <MAPR_ERROR>
        * Message:dump rolebalancerInfo failed
        * Function:DumpCommands.dumpRoleBalancerInfo()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        out.addError(new OutputError(Errno.ERPCFAILED,
            "dump rolebalancerInfo failed"));
        output.setOutput(out);
        return output;
      }
    } while (hasMore);

    if (ncontainers == 0) {
      return new TextCommandOutput(("No active role switches").getBytes());
    } else {
      output.setOutput(out);
      return output;
    }
  }

  /*
   * The following function does two chief things:
   * 
   * (a) Constructs the input request to be sent to CLDB Server
   * (b) Interprets the results and constructs the CommandOutput
   */
  private CommandOutput getRBalBalancingInfo() throws CLIProcessingException
  {
    /* 
     * CommandOutput needs to be returned even if the request is
     * invalid, which is why we have to declare them no matter what.
     */
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);
    
    /* Set just the credentials here and set the rest while parsing the request */
    DumpInfoRequest.Builder dumpinfoReqBuilder = DumpInfoRequest.newBuilder()
        .setCreds(getUserCredentials())
        .setDumpOp(DumpInfoRequest.DumpOp.RBAL_BALANCING_INFO_OP);

    RBalBalancingInfoRequest.Builder balancingInfoReqBuilder = RBalBalancingInfoRequest.newBuilder();
    
    String volumeName = null;
    if (isParamPresent(RBALINFO_VOLUME_OPT)) {
      /* For missing information, we bail out immediately */
      if (!isParamPresent(DUMP_VOLUME_NAME_PARAM_NAME)) {
        out.addError(new OutputError(Errno.EINVAL, "Missing Volume Name Opt. Check usage info"));
        return output;
      }
      volumeName = getParamTextValue(DUMP_VOLUME_NAME_PARAM_NAME, 0);
      if (volumeName == null) {
        out.addError(new OutputError(Errno.EINVAL, "Missing Volume Name. Check usage info"));
        return output;
      }
      dumpinfoReqBuilder.setVolumeName(volumeName);
      balancingInfoReqBuilder.setNeedVolumeInfo(true);
    }
    
    if (isParamPresent(RBALINFO_NAMECNTRS_OPT)) {
      balancingInfoReqBuilder.setNeedNameCntrsInfo(true);
    }

    dumpinfoReqBuilder.setRbalBalancingInfoRequest(balancingInfoReqBuilder.build());
    DumpInfoRequest req = dumpinfoReqBuilder.build();

    /* Step 1: Send request and obtain response */
    byte[] replyData;
    try {
      if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
            getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0),
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg.DumpInfoProc.getNumber(),
            req, DumpInfoResponse.class);
      } else {
        replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg.DumpInfoProc.getNumber(),
            req, DumpInfoResponse.class);
      }
      if (replyData == null) {
        out.addError(new OutputError(Errno.ERPCFAILED,
            "Couldn't connect to the CLDB service. dump rolebalancerinfo RPC failed"));
        return output;
      }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException("MaprSecurityException", e);
    } catch (Exception e) {
      out.addError(new OutputError(Errno.ERPCFAILED, "RPC Error"));
      return output;
    }

    /* Step 2: Parse byte array response into the appropriate java object */
    DumpInfoResponse resp;
    try {
      resp = DumpInfoResponse.parseFrom(replyData);
      if (resp.getStatus() != 0) {
        out.addError(new OutputError(resp.getStatus(), "dump " +
            "rolebalancerinfo failed with status : " + resp.getStatus()));
        return output;
      }
    }
    catch (InvalidProtocolBufferException e) {
      LOG.error("Exception while trying to parse container info", e);
      out.addError(new OutputError(Errno.EIO, "Error Parsing Balacing Info Response"));
      return output;
    }
    
    /* Step 3: Build CommandOutput from RBalBalacning Info */
    if (resp.hasBalancingInfo() == false) {
      out.addError(new OutputError(Errno.ENOENT, "Missing Balancing Info in the response"));
      return output;
    }
    
    RBalBalancingInfo balancingInfo = resp.getBalancingInfo();

    /* StoragePool info of a Volume */
    if (balancingInfo.hasVolumeBalancingInfo()) {
      BalancingInfo volumeBalancingInfo = balancingInfo.getVolumeBalancingInfo();
      if (volumeBalancingInfo.getStatus() != 0) {
        out.addError(new OutputError(volumeBalancingInfo.getStatus(), 
            "Error Obtaining Volume Balancing Info"));
      }
      else {
        OutputNode volInfoOutputNode = new OutputNode();
        volInfoOutputNode.addChild(new OutputNode("Volume", volumeName));
        volInfoOutputNode.addChild(new OutputNode("Assign Cache Containers Count", 
            volumeBalancingInfo.getAssignCacheCntrsCount()));
        volInfoOutputNode.addChild(new OutputNode("Assign Cache Containers Size", 
            volumeBalancingInfo.getAssignCacheCntrsSize()));
        volInfoOutputNode.addChild(new OutputNode("Zero Size Containers Count",
            volumeBalancingInfo.getZeroSizeCntrsCount()));
        for (RBalSpBalancingInfo spInfo : volumeBalancingInfo.getSpBalancingInfoList()) {
          volInfoOutputNode.addChild(getFormattedSpInfo(spInfo));
        }
        out.addNode(new OutputNode("VolumeBalancingInfo", volInfoOutputNode));
      }
    }

    /* StoragePools info of Name Containers */
    if (balancingInfo.hasNameCntrsBalancingInfo()) {
      BalancingInfo nameCntrsBalancingInfo = balancingInfo.getNameCntrsBalancingInfo();
      if (nameCntrsBalancingInfo.getStatus() != 0) {
        out.addError(new OutputError(nameCntrsBalancingInfo.getStatus(), 
            "Error Obtaining NameCntrs Balancing Info"));
      }
      else {
        OutputNode nameCntrsInfoOutputNode = new OutputNode();
        for (RBalSpBalancingInfo spInfo : nameCntrsBalancingInfo.getSpBalancingInfoList()) {
          nameCntrsInfoOutputNode.addChild(getFormattedSpInfo(spInfo));
        }
        out.addNode(new OutputNode("NameCntrsBalancingInfo", nameCntrsInfoOutputNode));
      }
    }
    return output;
  } /* end of getRBalBalancingInfo */

  OutputNode getFormattedSpInfo(final RBalSpBalancingInfo spInfo)
  {
    return new OutputNode("Storage Pools") {{
      if (spInfo.hasSpId()) {
        addChild(new OutputNode("SpId", spInfo.getSpId()));
      }
      if (spInfo.hasHostAddress()) {
        addChild(new OutputNode("HostAddress", spInfo.getHostAddress()));        
      }
      
      /*
       * Info relevant when Balancing is based on the number of containers
       */
      if (spInfo.hasNumContainers()) {
        addChild(new OutputNode("NumContainers", new Integer(spInfo.getNumContainers()) ));
      }
      if (spInfo.hasNumMasters()) {
        addChild(new OutputNode("NumMasters", new Integer(spInfo.getNumMasters()) ));
      }
      if (spInfo.hasDesiredNumMasters()) {
        addChild(new OutputNode("DesiredNumMasters", new Integer(spInfo.getDesiredNumMasters()) ));        
      }
      if (spInfo.hasNumTails()) {
        addChild(new OutputNode("NumTails", new Integer(spInfo.getNumTails()) ));
      }
      if (spInfo.hasDesiredNumTails()) {
        addChild(new OutputNode("DesiredNumTails", new Integer(spInfo.getDesiredNumTails()) ));        
      }

      /*
       * Info relevant when Balancing is based on the size of containers
       */
      if (spInfo.hasSizeofContainers()) {
        addChild(new OutputNode("SizeOfContainers", new Long(spInfo.getSizeofContainers()) ));
      }
      if (spInfo.hasSizeofMasters()) {
        addChild(new OutputNode("SizeOfMasters", new Long(spInfo.getSizeofMasters()) ));
      }
      if (spInfo.hasDesiredSizeofMasters()) {
        addChild(new OutputNode("DesiredSizeOfMasters", new Long(spInfo.getDesiredSizeofMasters()) ));
      }
      if (spInfo.hasSizeofTails()) {
        addChild(new OutputNode("SizeOfTails", new Long(spInfo.getSizeofTails()) ));
      }
      if (spInfo.hasDesiredSizeofTails()) {
        addChild(new OutputNode("DesiredSizeOfTails", new Long(spInfo.getDesiredSizeofTails()) ));
      }
      
      if (spInfo.hasAssignCacheCntrsCount()) {
        addChild(new OutputNode("Assign Cache Containers Count", 
            spInfo.getAssignCacheCntrsCount()));
      }
      if (spInfo.hasAssignCacheCntrsSize()) {
      addChild(new OutputNode("Assign Cache Containers Size", 
          spInfo.getAssignCacheCntrsSize()));
      }
    }};
  }

  private CommandOutput dumpRoleBalancerMetrics() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy(); 
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    LOG.info("dumpRoleBalancerMetrics : Dumping RoleBalancer metrics");

    DumpInfoRequest req = DumpInfoRequest.newBuilder()
            .setDumpOp(DumpInfoRequest.DumpOp.METRICS_ROLEBALANCER_OP)
            .setCreds(getUserCredentials())
            .build();
    byte[] replyData;
    try {
      if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        replyData = CLDBRpcCommonUtils.getInstance()
          .sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
              Common.MapRProgramId.CldbProgramId.getNumber(),
              CLDBProto.CLDBProg.DumpInfoProc.getNumber(),
              req,
              DumpInfoResponse.class);
      } else {
        replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
            Common.MapRProgramId.CldbProgramId.getNumber(),
          CLDBProto.CLDBProg
              .DumpInfoProc.getNumber(), req,
              DumpInfoResponse.class);
      }
      if (replyData == null) {
        /**
        * <MAPR_ERROR>
        * Message:dump rolebalancermetrics RPC failed
        * Function:DumpCommands.dumpRoleBalancerMetrics()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        out.addError(new OutputError(Errno.ERPCFAILED,
          "Couldn't connect to the CLDB service. dump " +
          "rolebalancermetrics RPC failed"));
        output.setOutput(out);
        return output;
      }
      DumpInfoResponse resp = DumpInfoResponse.parseFrom(replyData);
      if (resp.getStatus() != 0) {
        /**
        * <MAPR_ERROR>
        * Message:dump balancerinfo failed with status : <status>
        * Function:DumpCommands.dumpBalancerMetrics()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        out.addError(new OutputError(resp.getStatus(), "dump " +
            "rolebalancermetrics failed with status : " + resp.getStatus()));
        output.setOutput(out);
        return output;
      }

      out.addNode(formatRoleBalancerMetrics(resp.getRoleBalancerMetrics()));
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      /**
      * <MAPR_ERROR>
      * Message:dump rolebalancerMetrics failed
      * Function:DumpCommands.dumpRoleBalancerMetrics()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      out.addError(new OutputError(Errno.ERPCFAILED,
          "dump rolebalancermetrics failed"));
      output.setOutput(out);
      return output;
    }

    output.setOutput(out);
    return output;
  }

  private CommandOutput dumpVolumeNodes() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy(); 
    CommandOutput output = new CommandOutput();
    output.setOutput(out);
    
   String volumeName = getParamTextValue(DUMP_VOLUME_NAME_PARAM_NAME, 0);
     
   try {
     VolumeInfo volInfo = lookupVolumeInfo(volumeName);
     if (volInfo == null) {
       /**
       * <MAPR_ERROR>
       * Message:Could not get volume nodes for <volume>
       * Function:DumpCommands.dumpVolumeInfo()
       * Meaning:An error occurred.
       * Resolution:Contact technical support.
       * </MAPR_ERROR>
       */
       out.addError(new OutputError(Errno.ERPCFAILED, 
           "Could not get info for volume: " + volumeName));
       output.setOutput(out);
       return output;
     }
     //out.addNode(formatVolumeInfo(volInfo));
   } catch (Exception e) {
     /**
     * <MAPR_ERROR>
     * Message:Could not get info for volume: <volume> <error>
     * Function:DumpCommands.dumpVolumeNodes()
     * Meaning:An error occurred.
     * Resolution:Contact technical support.
     * </MAPR_ERROR>
     */
     out.addError(new OutputError(Errno.ERPCFAILED, 
         "Could not get info for volume: " + volumeName + 
         ", Error:" + e.getLocalizedMessage()));
     output.setOutput(out);
     return output;
   }
   
   Map<Long, Server> nodes = new HashMap<Long, Server>();
   // Lookup volume containers
   boolean hasMore = false;
   int startId = 0;
   
   do {
     VolumeContainersMapRequest volReq = VolumeContainersMapRequest.newBuilder()
                                           .setVolumeName(volumeName)
                                           .setStartContainerId(startId)
                                           .setNumContainers(10)
                                           .setCreds(getUserCredentials())
                                           .build();
     byte[] replyData;
     try {
       if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
         replyData = CLDBRpcCommonUtils.getInstance()
           .sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
               Common.MapRProgramId.CldbProgramId.getNumber(),
               CLDBProto.CLDBProg.VolumeContainersMapProc.getNumber(), 
               volReq, 
               VolumeContainersMapResponse.class);
       } else {
         replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
             Common.MapRProgramId.CldbProgramId.getNumber(),
           CLDBProto.CLDBProg
               .VolumeContainersMapProc.getNumber(), volReq,
               VolumeContainersMapResponse.class);
       }
       
       if (replyData == null) {
         /**
         * <MAPR_ERROR>
         * Message:dump volumenodes rpc failed
         * Function:DumpCommands.dumpVolumeNodes()
         * Meaning:An error occurred.
         * Resolution:Contact technical support.
         * </MAPR_ERROR>
         */
         out.addError(new OutputError(Errno.ERPCFAILED, 
             "Couldn't connect to the CLDB service. dump volumenodes rpc failed"));
         output.setOutput(out);
         return output;
       }
       VolumeContainersMapResponse resp = VolumeContainersMapResponse
                                                       .parseFrom(replyData);
       if (resp.getStatus() != 0) {
         /**
         * <MAPR_ERROR>
         * Message:dump volumenodes volume containersmap failed with status : <status>
         * Function:DumpCommands.dumpVolumeNodes()
         * Meaning:An error occurred.
         * Resolution:Contact technical support.
         * </MAPR_ERROR>
         */
         out.addError(new OutputError(resp.getStatus(), "dump volumenodes volume " +
            "containersmap failed with status : " + resp.getStatus()));
         output.setOutput(out);
         return output;
       }
       int lastId = 0;
       for (ContainerInfo cInfo : resp.getContainersList()) {
         for (Server ip : cInfo.getAServersList()) {
           if (nodes.get(ip.getServerId()) == null)
             nodes.put(ip.getServerId(), ip);
         }
         for (Server ip : cInfo.getIServersList()) {
           if (nodes.get(ip.getServerId()) == null)
             nodes.put(ip.getServerId(), ip);
         }
         for (Server ip : cInfo.getUServersList()) {
           if (nodes.get(ip.getServerId()) == null)
             nodes.put(ip.getServerId(), ip);
         }
         lastId = cInfo.getContainerId();
       }
       hasMore = resp.getHasMoreContainers();
       if (hasMore) {
         startId = lastId + 1;
       }      
     } catch (MaprSecurityException e) {
       throw new CLIProcessingException(
         "MaprSecurityException " + "Exception", e);
     } catch (Exception e) {
       /**
       * <MAPR_ERROR>
       * Message:dump volumenodes failed
       * Function:DumpCommands.dumpVolumeNodes()
       * Meaning:An error occurred.
       * Resolution:Contact technical support.
       * </MAPR_ERROR>
       */
       out.addError(new OutputError(Errno.ERPCFAILED, "dump volumenodes failed"));
       output.setOutput(out);
       return output;
     }
   } while (hasMore == true);
   
   OutputNode empty = new OutputNode();
   OutputNode s = new OutputNode("Servers");
   for (Server n : nodes.values()) {
     s.addChild(new OutputNode("IP:Port", Util.printServerWithStateForCLI(n)));
   }
   empty.addChild(s);
   out.addNode(empty);
   output.setOutput(out);
   return output;
  }

  private CommandOutput dumpVolumeInfo() throws CLIProcessingException {
	  OutputHierarchy out = new OutputHierarchy(); 
    CommandOutput output = new CommandOutput();
    output.setOutput(out);
    
   String volumeName = getParamTextValue(DUMP_VOLUME_NAME_PARAM_NAME, 0);
     
   try {
     VolumeInfo volInfo = lookupVolumeInfo(volumeName);
     if (volInfo == null) {
       /**
       * <MAPR_ERROR>
       * Message:Could not get info for volume: <volume>
       * Function:DumpCommands.dumpVolumeInfo()
       * Meaning:An error occurred.
       * Resolution:Contact technical support.
       * </MAPR_ERROR>
       */
       out.addError(new OutputError(Errno.ERPCFAILED, 
           "Could not get info for volume: " + volumeName));
       output.setOutput(out);
       return output;
     }
     out.addNode(formatVolumeInfo(volInfo));
   } catch (Exception e) {
     /**
     * <MAPR_ERROR>
     * Message:Could not get info for volume: <volume> <error>
     * Function:DumpCommands.dumpVolumeInfo()
     * Meaning:An error occurred.
     * Resolution:Contact technical support.
     * </MAPR_ERROR>
     */
     out.addError(new OutputError(Errno.ERPCFAILED, 
         "Could not get info for volume: " + volumeName + 
         ", Error: " + e.getLocalizedMessage()));
     output.setOutput(out);
     return output;
   }
   
   // Lookup volume containers
   boolean hasMore = false;
   int startId = 0;
   do {
     LOG.info("dumpVolumeInfo : Requesting VolumeContainersMap");
     VolumeContainersMapRequest volReq = VolumeContainersMapRequest.newBuilder()
                                           .setVolumeName(volumeName)
                                           .setStartContainerId(startId)
                                           .setNumContainers(10)
                                           .setCreds(getUserCredentials())
                                           .build();
     byte[] replyData;
     try {
       if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
         replyData = CLDBRpcCommonUtils.getInstance()
           .sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
               Common.MapRProgramId.CldbProgramId.getNumber(),
               CLDBProto.CLDBProg.VolumeContainersMapProc.getNumber(), 
               volReq, 
               VolumeContainersMapResponse.class);
       } else {
         replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
             Common.MapRProgramId.CldbProgramId.getNumber(),
           CLDBProto.CLDBProg
               .VolumeContainersMapProc.getNumber(), volReq,
               VolumeContainersMapResponse.class);
       }
       
       if (replyData == null) {
         /**
         * <MAPR_ERROR>
         * Message:dump volumeinfo rpc failed
         * Function:DumpCommands.dumpVolumeInfo()
         * Meaning:An error occurred.
         * Resolution:Contact technical support.
         * </MAPR_ERROR>
         */
         out.addError(new OutputError(Errno.ERPCFAILED, 
             "Couldn't connect to the CLDB service. dump volumeinfo rpc failed"));
         output.setOutput(out);
         return output;
       }
       VolumeContainersMapResponse resp = VolumeContainersMapResponse
                                                       .parseFrom(replyData);
       if (resp.getStatus() != 0) {
         /**
         * <MAPR_ERROR>
         * Message:dump volumeinfo volume containersmap failed with status : <status>
         * Function:DumpCommands.dumpVolumeInfo()
         * Meaning:An error occurred.
         * Resolution:Contact technical support.
         * </MAPR_ERROR>
         */
         out.addError(new OutputError(resp.getStatus(), "dump volumeinfo volume " +
         		"containersmap failed with status : " + resp.getStatus()));
         output.setOutput(out);
         return output;
       }
       int lastId = 0;
       for (ContainerInfo cInfo : resp.getContainersList()) {
         out.addNode(formatContainerInfo(cInfo, null));
         lastId = cInfo.getContainerId();
       }
       hasMore = resp.getHasMoreContainers();
       if (hasMore) {
         startId = lastId + 1;
       }      
     } catch (MaprSecurityException e) {
       throw new CLIProcessingException(
         "MaprSecurityException " + "Exception", e);
     } catch (Exception e) {
       /**
       * <MAPR_ERROR>
       * Message:dump volumeinfo failed
       * Function:DumpCommands.dumpVolumeInfo()
       * Meaning:An error occurred.
       * Resolution:Contact technical support.
       * </MAPR_ERROR>
       */
       out.addError(new OutputError(Errno.ERPCFAILED, "dump volumeinfo failed"));
       output.setOutput(out);
       return output;
     }
   } while (hasMore == true);
     output.setOutput(out);
    return output;
  }
  
  private OutputNode formatContainerInfo(ContainerInfo cInfo,
                                                      VolumeInfo vInfo) {
    return formatContainerInfo(cInfo, vInfo, null);
  }
  private OutputNode formatContainerInfo(ContainerInfo cInfo,
                                                      VolumeInfo vInfo,
                                                      String type) {
    OutputNode cOut = new OutputNode();
    if (type != null 
        && (type.equalsIgnoreCase("waiting") || type.equalsIgnoreCase("unused")
          || type.equalsIgnoreCase("resync") || type.equalsIgnoreCase("bm") )) {
      int count = 0;
      if (type.equalsIgnoreCase("waiting")) {
        count = cInfo.getIServersCount();
      } else if (type.equalsIgnoreCase("unused")) {
        count = cInfo.getUServersCount();
      } else if (type.equalsIgnoreCase("resync")) {
        for (Server s : cInfo.getAServersList())
          if (s.getState() == ReplicaState.RESYNC)
            count++;
      } else {
        count = 1; // become-master cant be anything but 1
      }
      cOut.addChild(new OutputNode("InstanceCount", count));
    }

    cOut.addChild(new OutputNode("ContainerId", cInfo.getContainerId()));
    cOut.addChild(new OutputNode("Epoch", cInfo.getLatestEpoch()));
    cOut.addChild(new OutputNode("Master", Util.printServerWithStateForCLI(cInfo.getMServer())));
    
    OutputNode activeServers = new OutputNode("ActiveServers");
    cOut.addChild(activeServers);
    for (Server ip : cInfo.getAServersList()) {
      activeServers.addChild(new OutputNode("IP:Port", Util.printServerWithStateForCLI(ip)));
    }
    
    OutputNode inactiveServer = new OutputNode("InactiveServers");
    cOut.addChild(inactiveServer);
    for (Server ip : cInfo.getIServersList()) {
      inactiveServer.addChild(new OutputNode("IP:Port", Util.printServerWithNoStateForCLI(ip)));
    }

    OutputNode unUsedServer = new OutputNode("UnusedServers");
    cOut.addChild(unUsedServer);
    for (Server ip : cInfo.getUServersList()) {
      unUsedServer.addChild(new OutputNode("IP:Port", Util.printServerWithNoStateForCLI(ip)));
    }
    cOut.addChild(new OutputNode("OwnedSizeMB", 
        Util.readableSizeMB(cInfo.getOwnedSizeMB())));
    cOut.addChild(new OutputNode("SharedSizeMB", 
        Util.readableSizeMB(cInfo.getSharedSizeMB())));
    cOut.addChild(new OutputNode("LogicalSizeMB", 
        Util.readableSizeMB(cInfo.getLogicalSizeMB())));
    
    cOut.addChild(new OutputNode("TotalSizeMB", 
        Util.readableSizeMB(cInfo.getChainSizeMB())));

    if (cInfo.hasNumInumUsed()) {
      cOut.addChild(new OutputNode("NumInodesInUse", 
        cInfo.getNumInumUsed()));
    }
    
    if (cInfo.hasMtime()) {
      cOut.addChild(new OutputNode("Mtime",
          new Date(cInfo.getMtime()).toString()));
    }
    if (cInfo.getNameContainer()) {
      cOut.addChild(new OutputNode("NameContainer", "true"));
    } else {
      cOut.addChild(new OutputNode("NameContainer", "false"));
    }
    if (cInfo.hasRwContainerId()) {
      cOut.addChild(new OutputNode("RW ContainerId", cInfo.getRwContainerId()));
    }
    if (cInfo.hasRwVolumeId()) {
      cOut.addChild(new OutputNode("RW VolumeId", cInfo.getRwVolumeId()));
    }
    
    cOut.addChild(new OutputNode("CreatorContainerId", cInfo.getCreatorContainerId()));
    StringBuilder creatorVolUuid= new StringBuilder();
    if (cInfo.hasCreatorVolumeUuid()) {
      creatorVolUuid.append(cInfo.getCreatorVolumeUuid().getId640());
      creatorVolUuid.append(":");
      creatorVolUuid.append(cInfo.getCreatorVolumeUuid().getId641());
    }
    cOut.addChild(new OutputNode("CreatorVolumeUuid", creatorVolUuid.toString()));
    cOut.addChild(new OutputNode("UseActualCreatorId", cInfo.getUseActualCreatorId()));
    if (vInfo != null) {
      cOut.addChild(new OutputNode("VolumeName", vInfo.getVolProperties().getVolumeName()));
      cOut.addChild(new OutputNode("VolumeId", vInfo.getVolProperties().getVolumeId()));
      cOut.addChild(new OutputNode("VolumeReplication", 
          vInfo.getVolProperties().getReplicationPolicy().getNumReplicas()));
      cOut.addChild(new OutputNode("NameSpaceReplication", 
          vInfo.getVolProperties().getNumNamespaceReplicas()));
      cOut.addChild(new OutputNode("VolumeMounted", vInfo.getVolProperties().getMounted()));
      cOut.addChild(new OutputNode("AccessTime",
                        (vInfo.hasAtime() ?
                        (new Date(vInfo.getAtime()).toString()) : "Missing Atime")));

    }

    return cOut;
  }
  
  private OutputNode formatVolumeInfo(VolumeInfo volInfo) {
    OutputNode vOut = new OutputNode();
    vOut.addChild(new OutputNode("VolumeName", volInfo.getVolProperties().getVolumeName()));
    vOut.addChild(new OutputNode("VolumeId", volInfo.getVolProperties().getVolumeId()));
    vOut.addChild(new OutputNode("VolumeTopology", 
        volInfo.getVolProperties().getTopology().getTopologyRestricted()));
    if (volInfo.getVolProperties().hasLocalTopology()) {
      vOut.addChild(new OutputNode("VolumeLocalTopology", 
          volInfo.getVolProperties().getLocalTopology().getTopologyRestricted()));
    }
    vOut.addChild(new OutputNode("VolumeUsedSizeMB", 
        volInfo.getVolQuota().getVolumeUsedSizeMB()));
    vOut.addChild(new OutputNode("VolumeReplication", 
        volInfo.getVolProperties().getReplicationPolicy().getNumReplicas()));
    vOut.addChild(new OutputNode("VolumeMinReplication", 
        volInfo.getVolProperties().getReplicationPolicy().getGuaranteedMinReplicas()));
    vOut.addChild(new OutputNode("NameSpaceReplication", 
        volInfo.getVolProperties().getNumNamespaceReplicas()));
    vOut.addChild(new OutputNode("NameSpaceMinReplication", 
        volInfo.getVolProperties().getGuaranteedMinNamespaceReplicas()));
    vOut.addChild(new OutputNode("ReReplicationTimeOutSec", 
        volInfo.getVolProperties().getReReplicationTimeOutSec()));
    vOut.addChild(new OutputNode("MirrorThrottle",
        volInfo.getVolProperties().getMirrorThrottle()));
    vOut.addChild(new OutputNode("AccessTime",
        (volInfo.hasAtime() ?  (new Date(volInfo.getAtime()).toString()) : "Missing Atime")));
    vOut.addChild(new OutputNode("AuditVolume",
         volInfo.getVolProperties().getIsAuditVolume() ? "1" : "0"));
    vOut.addChild(new OutputNode("Audited",
         volInfo.getVolProperties().getAudited() ? "1" : "0"));
    vOut.addChild(new OutputNode("CoalesceInterval",
         volInfo.getVolProperties().getCoalesceInterval()));
    vOut.addChild(new OutputNode("EnabledDataAuditOperations",
         GetStringsForAuditOps.getEnabledOps(volInfo.getVolProperties().getFsAuditDisabledOperations())));
    vOut.addChild(new OutputNode("DisabledDataAuditOperations",
         GetStringsForAuditOps.getDisabledOps(volInfo.getVolProperties().getFsAuditDisabledOperations())));
    vOut.addChild(new OutputNode("limitSpread",
        (volInfo.getVolProperties().hasLimitVolumeSpread() ?
         volInfo.getVolProperties().getLimitVolumeSpread() : true)));
    vOut.addChild(new OutputNode("allowGrant",
        (volInfo.getVolProperties().hasAllowGrant() ?
         volInfo.getVolProperties().getAllowGrant() : false)));
    vOut.addChild(new OutputNode("fixCreatorId",
         volInfo.getVolProperties().getUseActualCreatorId()));
    return vOut;
  }
  
  private OutputNode formatFileServerCommand(FileServerPendingWork cmd) {
    // For now, just dump it to screen
    OutputNode fsOut = new OutputNode();
    fsOut.addChild(new OutputNode("ContainerId", cmd.getCmd().getContainerId()));
    fsOut.addChild(new OutputNode("Command", cmd.getCmd().getWork()));
    fsOut.addChild(new OutputNode("ExpectingReply", cmd.getExpectingReply()));
    fsOut.addChild(new OutputNode("StoragePool", cmd.getSpId()));
    return fsOut;
  }
  
  private OutputNode formatActiveSp(ActiveStoragePool sp)
  {
    OutputNode out = new OutputNode();
    out.addChild(new OutputNode("spid", sp.getSpid()));
    out.addChild(new OutputNode("fsid", sp.getFsid()));
    out.addChild(new OutputNode("ip:port", Util.printIPAddressesForCLI(sp.getServer())));
    out.addChild(new OutputNode("capacityMB", sp.getCapacityMB()));
    out.addChild(new OutputNode("usedMB", sp.getUsedMB()));
    out.addChild(new OutputNode("percentage", sp.getPercentUsed()));
    out.addChild(new OutputNode("fullnessLevel", sp.getFullness()));
    out.addChild(new OutputNode("inTransitMB", sp.getInTransitMB()));
    out.addChild(new OutputNode("outTransitMB", sp.getOutTransitMB()));
    out.addChild(new OutputNode("topology", sp.getTopology()));
    out.addChild(new OutputNode("lastHeartBeatSec", sp.getLastHeartBeat()));
    out.addChild(new OutputNode("inTransitContainers", sp.getInTransitContainersCount()));
    out.addChild(new OutputNode("outTransitContainers", sp.getOutTransitContainersCount()));
    return out;
  }

  private OutputNode formatActiveMove(ActiveContainerMove am) 
  {
    OutputNode out = new OutputNode();
    out.addChild(new OutputNode("containerid", am.getContainerId()));
    out.addChild(new OutputNode("sizeMB", am.getSizeMB()));
    if (am.hasFrom()) {
      out.addChild(new OutputNode("From fsid", am.getFrom().getServerId()));
      out.addChild(new OutputNode("From IP:Port", Util.printIPAddressesForCLI(am.getFrom())));
      out.addChild(new OutputNode("From SP", am.getFrom().getSpInfo().getSpId()));
    }
    if (am.hasTo()) {
      out.addChild(new OutputNode("To fsid", am.getTo().getServerId()));
      out.addChild(new OutputNode("To IP:Port", Util.printIPAddressesForCLI(am.getTo())));
      out.addChild(new OutputNode("To SP", am.getTo().getSpInfo().getSpId()));
    }
    if (am.hasUpdatesBlocked() && am.getUpdatesBlocked() &&
        am.hasUpdatesBlockedSince()) {
      out.addChild(new OutputNode("Updates blocked Since", new Date(am.getUpdatesBlockedSince()).toString()));
    }
    return out;
  }
  
  private OutputNode formatActiveRoleSwitch(ActiveContainerRoleSwitch am)
  {
    OutputNode out = new OutputNode();
    out.addChild(new OutputNode("containerid", am.getContainerId()));
    if (am.hasTail()) {
      out.addChild(new OutputNode("Tail IP:Port", Util.printIPAddressesForCLI(am.getTail())));
    }
    if (am.hasUpdatesBlockedSince() && (am.getUpdatesBlockedSince() != 0)) {
      out.addChild(new OutputNode("Updates blocked Since", new Date(am.getUpdatesBlockedSince()).toString()));
    }
    return out;
  }

  private OutputNode formatActiveCopyCreate(ActiveContainerCopyCreate am)
  {
    OutputNode out = new OutputNode();
    out.addChild(new OutputNode("containerid", am.getContainerId()));
    for (ActiveContainerCopyReplica repl : am.getReplicasList())
    {
      OutputNode rout = new OutputNode("replica");
      rout.addChild(new OutputNode("sizeMB", repl.getSizeMB()));
      if (repl.hasTo()) {
        rout.addChild(new OutputNode("To fsid", repl.getTo().getServerId()));
        rout.addChild(new OutputNode("To IP:Port", Util.printIPAddressesForCLI(repl.getTo())));
        rout.addChild(new OutputNode("To SP", repl.getTo().getSpInfo().getSpId()));
      }
      out.addChild(rout);
    }
    if (am.hasStartTime()) {
      out.addChild(new OutputNode("startTime", new Date(am.getStartTime()).toString()));
    }
    return out;
  }
  
  private OutputNode formatDiskBalancerMetrics(DiskBalancerMetrics dm)
  {
    OutputNode out = new OutputNode();
    if (dm.hasNumContainersMoved()) {
      out.addChild(new OutputNode("numContainersMoved", dm.getNumContainersMoved()));
    }
    if (dm.hasNumMBMoved()) {
      out.addChild(new OutputNode("numMBMoved", dm.getNumMBMoved()));
    }
    if (dm.hasTimeOfLastMove() && (dm.getTimeOfLastMove() > 0)) {
      out.addChild(new OutputNode("timeOfLastMove", new Date(dm.getTimeOfLastMove()).toString()));
    }
    return out;
  }

  private OutputNode formatRoleBalancerMetrics(RoleBalancerMetrics rm)
  {
    OutputNode out = new OutputNode();
    if (rm.hasNumNameContainerSwitches()) {
      out.addChild(new OutputNode("numNameContainerSwitches", rm.getNumNameContainerSwitches()));
    }
    if (rm.hasNumDataContainerSwitches()) {
      out.addChild(new OutputNode("numDataContainerSwitches", rm.getNumDataContainerSwitches()));
    }
    if (rm.hasTimeOfLastSwitch() && (rm.getTimeOfLastSwitch() > 0)) {
      out.addChild(new OutputNode("timeOfLastMove", new Date(rm.getTimeOfLastSwitch()).toString()));
    }
    return out;
  }

  private OutputNode formatRereplicationMetrics(RereplicationMetrics rr)
  {
    OutputNode out = new OutputNode();
    if (rr.hasNumContainersCopied()) {
      out.addChild(new OutputNode("numContainersCopied", rr.getNumContainersCopied()));
    }
    if (rr.hasNumMBCopied()) {
      out.addChild(new OutputNode("numMBCopied", rr.getNumMBCopied()));
    }
    return out;
  }

  private OutputNode formatActiveAssignVoucher(ActiveAssignVoucher av)
  {
    OutputNode out = new OutputNode();
    out.addChild(new OutputNode("containerid", av.getContainerId()));
    out.addChild(new OutputNode("activeSizeMB", av.getActiveSizeMB()));
    out.addChild(new OutputNode("inactiveSizeMB", av.getInactiveSizeMB()));
    return out;
  }

  private OutputNode formatActiveWriter(ActiveWriter w) {
    OutputNode out = new OutputNode();
    out.addChild(new OutputNode("hostname", w.getHostname()));
    out.addChild(new OutputNode("volume", w.getVolumeName()));
    out.addChild(new OutputNode("numAssigns", w.getNumContainerAssignCalls()));
    out.addChild(new OutputNode("numInactiveAssigns", w.getNumContainerAssignCallsInactive()));
    return out;
  }
  
  private VolumeInfo lookupVolumeInfo(int volumeId) 
                                            throws Exception {
    VolumeLookupRequest req = VolumeLookupRequest.newBuilder()
                                         .setVolumeId(volumeId)
                                         .setCreds(getUserCredentials())
                                         .build();
    return lookupVolumeInfoInternal(req);
  }

  private VolumeInfo lookupVolumeInfo(String volumeName)
                                            throws Exception {

    VolumeLookupRequest req = VolumeLookupRequest.newBuilder()
          .setVolumeName(volumeName)
          .setCreds(getUserCredentials())
          .build();
    return lookupVolumeInfoInternal(req);
  }
  
  private VolumeInfo lookupVolumeInfoInternal(
      VolumeLookupRequest req) throws Exception {
    
    byte[] data;
    if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
      data = CLDBRpcCommonUtils.getInstance()
                .sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
                             Common.MapRProgramId.CldbProgramId.getNumber(),
                             CLDBProto.CLDBProg.VolumeLookupProc.getNumber(), 
                             req, VolumeLookupResponse.class);
    } else {
      data = CLDBRpcCommonUtils.getInstance().sendRequest(
                   Common.MapRProgramId.CldbProgramId.getNumber(),
                   CLDBProto.CLDBProg.VolumeLookupProc.getNumber(), req,
                   VolumeLookupResponse.class);
    }
      
    if (data == null) {
    	throw new CLIProcessingException("Exception while processing RPC");
    }
      
    VolumeLookupResponse resp = VolumeLookupResponse.parseFrom(data);
    if (resp.getStatus() != 0) {
      throw new IOException((resp.getStatus() == Errno.ENOENT) ? 
      				"No such volume" : Errno.toString(resp.getStatus()));
    }
    return resp.getVolInfo();
  }
  
  private ContainerInfo lookupContainerInfo(int containerId) 
                                                throws Exception {
    LOG.info("Lookup for container " + containerId);
    DumpInfoRequest req = DumpInfoRequest.newBuilder()
                                    .setDumpOp(
                                      DumpInfoRequest.DumpOp.CONTAINER_DUMP_OP)
                                    .setContainerId(containerId)
                                    .setCreds(getUserCredentials())
                                    .build();
    byte[] data;
    try {
      if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        data = CLDBRpcCommonUtils.getInstance()
          .sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
              Common.MapRProgramId.CldbProgramId.getNumber(),
              CLDBProto.CLDBProg.DumpInfoProc.getNumber(), 
              req, 
              DumpInfoResponse.class);
      } else {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(
            Common.MapRProgramId.CldbProgramId.getNumber(),
          CLDBProto.CLDBProg
              .DumpInfoProc.getNumber(), req,
              DumpInfoResponse.class);
      }
      
      if (data == null) {
    	  // most likely can not connect to CLDB
    	  LOG.error("Couldn't connect to the CLDB service");
    	  throw new CLIProcessingException("Couldn't connect to the CLDB service");
      }
      DumpInfoResponse resp = DumpInfoResponse.parseFrom(data);
      if (resp.getStatus() == 0)
        return resp.getContainerInfo();
      else
        return null;
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      throw e;
    }
  }

  private CommandOutput dumpContainerInfo() 
        throws CLIProcessingException {        
    OutputHierarchy out = new OutputHierarchy();  /* Root of comamnd output's hierarchy tree */
    CommandOutput output = new CommandOutput();
    output.setOutput(out);
    
     String containerIds = getParamTextValue(DUMP_CONTAINER_ID_PARAM_NAME, 0);       
     List<Integer> cIds = new ArrayList<Integer>();
     
     if (!containerIds.contains(MULTI_ARG_SEP)) {
       cIds.add(Integer.parseInt(containerIds));
     } else {
       String[] cs = containerIds.split(MULTI_ARG_SEP);
       for (String cId : cs) {
         cIds.add(Integer.parseInt(cId));
       }
     }
     
     for (Integer cid : cIds) {
       try {
        ContainerInfo cInfo = lookupContainerInfo(cid);
        if (cInfo != null) {
          int volumeId = cInfo.hasRwVolumeId() ? 
              cInfo.getRwVolumeId() : cInfo.getVolumeId();
          VolumeInfo vInfo = lookupVolumeInfo(volumeId);
          out.addNode(formatContainerInfo(cInfo, vInfo));
        }
        
      } catch( CLIProcessingException e) {
    	  out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
    	  // no point of proceeding further
    	  return output;
     } catch (Exception e) {
        /**
        * <MAPR_ERROR>
        * Message:dump containerInfo failed
        * Function:DumpCommands.dumpContainerInfo()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        out.addError(new OutputError(Errno.ERPCFAILED, "dump containerInfo failed"));
        output.setOutput(out);
        return output;
      }
     }  
    return output;
  }

  private CommandOutput dumpContainers()
        throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    String type = getParamTextValue(DUMP_CONTAINER_TYPE_PARAM_NAME, 0);
    DumpInfoRequest.DumpOp dumpOp;
    long cookie = 0;
    if (type.equalsIgnoreCase("offline")) {
      dumpOp = DumpInfoRequest.DumpOp.LIST_OFFLINE_CONTAINERS_OP;
    } else if (type.equalsIgnoreCase("resync")) {
      dumpOp = DumpInfoRequest.DumpOp.LIST_RESYNCING_CONTAINERS_OP;
    } else if (type.equalsIgnoreCase("bm")) {
      dumpOp = DumpInfoRequest.DumpOp.LIST_BECOME_MASTER_CONTAINERS_OP;
    } else if (type.equalsIgnoreCase("waiting")) {
      dumpOp = DumpInfoRequest.DumpOp.LIST_WAITING_FOR_ROLE_CONTAINERS_OP;
    } else if (type.equalsIgnoreCase("unused")) {
      dumpOp = DumpInfoRequest.DumpOp.LIST_UNUSED_CONTAINERS_OP;
    } else {
      out.addError(new OutputError(Errno.EINVAL, "Invalid type " + type));
      return output;
    }

    boolean hasMore = false;
    do {
      DumpInfoRequest req = DumpInfoRequest.newBuilder()
                              .setCreds(getUserCredentials())
                              .setCookie(cookie)
                              .setDumpOp(dumpOp)
                              .build();
      byte[] replyData;
      try {
        if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
          replyData = CLDBRpcCommonUtils.getInstance()
            .sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
                Common.MapRProgramId.CldbProgramId.getNumber(),
                CLDBProto.CLDBProg.DumpInfoProc.getNumber(), 
                req, 
                DumpInfoResponse.class);
        } else {
          replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
              Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg
                .DumpInfoProc.getNumber(), req,
                DumpInfoResponse.class);
        }

        if (replyData == null) {
          out.addError(new OutputError(Errno.ERPCFAILED, 
            "Couldn't connect to the CLDB service. dump containers rpc failed"));
          return output;
        }
        DumpInfoResponse resp = DumpInfoResponse.parseFrom(replyData);
        if (resp.getStatus() != 0) {
          out.addError(new OutputError(resp.getStatus(),
              "dump containers failed with status : " + resp.getStatus()));
          return output;
        }
        boolean knownOp = resp.hasKnownOp();
        if (knownOp) knownOp = resp.getKnownOp();
        if (!knownOp) {
          out.addError(new OutputError(Errno.ENOSYS,
              "CLDB does not understand the dump containers command. Upgrade the CLDB and run the command again"));
          return output;
        }
        for (ContainerInfo cinfo : resp.getReplicationContainersList()) {
          out.addNode(formatContainerInfo(cinfo, null, type));
        }
        hasMore = resp.getMoreCids();        
        if (hasMore) cookie = resp.getCookie();
      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
      } catch (Exception e) {
        out.addError(new OutputError(Errno.ERPCFAILED, "dump containers failed"));
        return output;
      }
    } while (hasMore);

    return output;
  }
  
  private CommandOutput dumpZKInfo() throws CLIProcessingException {
	   OutputHierarchy out = new OutputHierarchy();  
	   CommandOutput output = new CommandOutput();
	   output.setOutput(out);

      ZooKeeper s_zk = null;
      String zkConnectString = CLDBRpcCommonUtils.getInstance().getZkConnect();
     	if ( (zkConnectString == null || zkConnectString.trim().isEmpty()) && isParamPresent(NodesCommonUtils.ZK_CONNECTSTRING) ) {
      		zkConnectString = getParamTextValue(NodesCommonUtils.ZK_CONNECTSTRING,0);
     	}
     	if ( LOG.isInfoEnabled()) {
     		LOG.info("ZKConnectString: " + zkConnectString);
     	}
  	  if ( zkConnectString == null ) {
  			/**
  			* <MAPR_ERROR>
  			* Message:ZK Connect string is null. Check if ZK Connect string was provided correctly
  			* Function:NodeServicesManagementCommand.executeRealCommand()
  			* Meaning:The ZooKeeper connect string was not specified.
  			* Resolution:Check the command syntax and try again.
  			* </MAPR_ERROR>
  			*/
  			String errorMsg = "Unable to obtain the ZooKeeper connection string";
  			out.addError(new OutputError(Errno.EZKCANTCONNECT, errorMsg));
  			return output;    		
  	  }

      if (isParamPresent(NodesCommonUtils.ZK_CONNECTSTRING)) {
        zkConnectString = getParamTextValue(NodesCommonUtils.ZK_CONNECTSTRING, 0); 
      } else {
        zkConnectString = CLDBRpcCommonUtils.getInstance().getZkConnect();
      }
      
       try {
        s_zk = NodesCommonUtils.connect(zkConnectString);
        OutputNode topNode = new OutputNode();
        getRecursiveZKInfo(s_zk, topNode, "/");
        out.addNode(topNode);
       } catch (KeeperException.NoAuthException e) {
         /**
        * <MAPR_ERROR>
        * Message:Pemission denied while trying to access ZooKeeper information.
        * Function:DumpCommands.dumpContainerInfo()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
      	LOG.error("Pemission denied while trying to access ZooKeeper information", e);
        /**
        * <MAPR_ERROR>
        * Message:"Permission denied to access zookeeper information."
        * Function:DumpCommands.dumpContainerInfo()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        out.addError(new OutputError(Errno.EACCES, "Permission denied to access zookeeper information."));          
       } catch(Throwable t) {
        /**
        * <MAPR_ERROR>
        * Message:Error while trying to dump info from ZK. Check ZK connection
        * Function:DumpCommands.dumpContainerInfo()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
      	LOG.error("Error while trying to dump info from ZK. Check ZK connection", t);
        /**
        * <MAPR_ERROR>
        * Message:Error while trying to dump info from ZK. Check ZK connection: <ZK connect string>
        * Function:DumpCommands.dumpContainerInfo()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        out.addError(new OutputError(Errno.EZKCANTCONNECT, "Error while trying to dump info from ZK. Check ZK connection: " + zkConnectString));
      } 
      return output;
  }
  
  private static String SUPPORT_DUMP_ACTION = "sdump";
  private CommandOutput supportDumpTrigger() throws CLIProcessingException {
	   OutputHierarchy oh = new OutputHierarchy();  
	   CommandOutput co = new CommandOutput();
	   co.setOutput(oh);
	    
    String zkConnectString = CLDBRpcCommonUtils.getInstance().getZkConnect();
   	if ( (zkConnectString == null || zkConnectString.trim().isEmpty()) && isParamPresent(NodesCommonUtils.ZK_CONNECTSTRING) ) {
    		zkConnectString = getParamTextValue(NodesCommonUtils.ZK_CONNECTSTRING,0);
   	}
   	if ( LOG.isInfoEnabled()) {
   		LOG.info("ZKConnectString: " + zkConnectString);
   	}
	  if ( zkConnectString == null ) {
			/**
			* <MAPR_ERROR>
			* Message:ZK Connect string is null. Check if ZK Connect string was provided correctly
			* Function:NodeServicesManagementCommand.executeRealCommand()
			* Meaning:The ZooKeeper connect string was not specified.
			* Resolution:Check the command syntax and try again.
			* </MAPR_ERROR>
			*/
			String errorMsg = "Unable to obtain the ZooKeeper connection string";
			oh.addError(new OutputError(Errno.EZKCANTCONNECT, errorMsg));
			return co;    		
	  }

		ZooKeeper s_zk = NodesCommonUtils.connect ( zkConnectString );
		List<String> nodeHostNames = new ArrayList<String>();
		
		if ( isParamPresent(NodeServicesManagementCommand.NODES_PARAM)) {
			nodeHostNames = input.getParameterByName(NodeServicesManagementCommand.NODES_PARAM).getParamValues();
			Set<String> hostNamesSet = new HashSet<String>(NodesCommonUtils.convertIpToHost(nodeHostNames));
			nodeHostNames.clear();
			nodeHostNames.addAll(hostNamesSet);			
		} else {
		  nodeHostNames = NodesCommonUtils.findNodeIps(zkConnectString);
		}
		try {
	    for ( String node : nodeHostNames ) {
	    	OutputNode pNode = new OutputNode();
	    	oh.addNode(pNode);
				try {
					s_zk.create(NodeServicesManagementCommand.NODES_SERVICES_MNGMNT_TOP_PATH, new byte[0], Ids.OPEN_ACL_UNSAFE, CreateMode.PERSISTENT);
				} catch (KeeperException.NodeExistsException e1) {
					// expected one
				} 
				try {
					s_zk.create(NodeServicesManagementCommand.NODES_SERVICES_MNGMNT_TOP_PATH + "/" + node, new byte[0], Ids.OPEN_ACL_UNSAFE, CreateMode.PERSISTENT);
				} catch (KeeperException.NodeExistsException e1) {
					// expected one
				} 
				try {
					// output time when node was created
					byte data[] = String.valueOf(System.currentTimeMillis()).getBytes();
					if ( isParamPresent(DUMP_PARAMS)) {
						String dumpParams = getParamTextValue(DUMP_PARAMS,0);
						data = dumpParams.getBytes();
					}
					s_zk.create(NodeServicesManagementCommand.NODES_SERVICES_MNGMNT_TOP_PATH + "/" + node + "/" + SUPPORT_DUMP_ACTION, data, Ids.OPEN_ACL_UNSAFE, CreateMode.PERSISTENT);
					pNode.addChild(new OutputNode("node", node));
				} catch (KeeperException.NodeExistsException e1) {
					// too bad - most likely somebody already issued a command to control service, but either it was not yet executed, or node was not removed for some reason
					LOG.warn("Please wait. The previous " + SUPPORT_DUMP_ACTION + " is executing.");
					// continue with next service
				}
		  }
		} catch (KeeperException e) {
			LOG.error("KeeperException while trying to create ZK struct for support dump", e);
			oh.addError(new OutputError(Errno.EZKCANTCONNECT, "KeeperException while trying to create ZK struct for support dump" + e.getLocalizedMessage()));
		} catch (InterruptedException e) {
			LOG.error("InterruptedException while trying to create ZK struct for support dump", e);
			oh.addError(new OutputError(Errno.EZKCANTCONNECT, "InterruptedException while trying to create ZK struct for support dump" + e.getLocalizedMessage()));
		}
		return co;
	}



  private CommandOutput dumpKvStoreCidInfo() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();  
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    ZooKeeper s_zk = null;
    String zkConnectString = CLDBRpcCommonUtils.getInstance().getZkConnect();
    if (zkConnectString == null || zkConnectString.trim().isEmpty()) {
      zkConnectString = getParamTextValue(NodesCommonUtils.ZK_CONNECTSTRING, 0);
    }
    try {
      s_zk = NodesCommonUtils.connect(zkConnectString);

      Stat stat = new Stat();
      byte[] data = s_zk.getData(CLDBConfiguration
          .ZK_ZNODE_KVSTORE_CONTAINER_INFO, false, stat);
      if (data != null) {
        ContainerInfo kvStoreContainerInfo = ContainerInfo.parseFrom(data);
        LOG.info("ContainerInfo : " + Util.printContainerInfoForCLI(kvStoreContainerInfo));
        for( Server s : kvStoreContainerInfo.getAServersList()) {
          out.addNode(new OutputNode("valid", Util.printIPAddressesForCLI(s)));
        }
        for (Server s : kvStoreContainerInfo.getIServersList()) {
          out.addNode(new OutputNode("invalid", Util.printIPAddressesForCLI(s)));
        }
      } else {
        /**
        * <MAPR_ERROR>
        * Message:Error while trying to trying to read ZooKeeper Information. Please Check ZooKeeper connection: <ZK connect string>
        * Function:DumpCommands.dumpKvStoreCidInfo()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        out.addError(new OutputError(Errno.EZKCANTCONNECT, 
            "Error while trying to trying to read ZooKeeper Information." +
         		  " Please Check ZooKeeper connection: " + zkConnectString));
        return output;
      }
    } catch(Throwable t) {
      /**
      * <MAPR_ERROR>
      * Message:Error while trying to dump info from ZK. Check ZK connection
      * Function:DumpCommands.dumpKvStoreCidInfo()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      LOG.error("Error while trying to dump info from ZK. Check ZK connection", t);
      /**
      * <MAPR_ERROR>
      * Message:Error while trying to dump info from ZooKeeper. Please Check ZooKeeper connection: <ZK connect string>
      * Function:DumpCommands.dumpKvStoreCidInfo()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      out.addError(new OutputError(Errno.EZKCANTCONNECT, 
          "Error while trying to dump info from ZooKeeper." +
          " Please Check ZooKeeper connection: " + zkConnectString));
    } 
    return output;
  }
  
   private void getRecursiveZKInfo(ZooKeeper s_zk, OutputNode parentNode, String zkParentPath) 
  	throws KeeperException, InterruptedException {
		List<String> children = s_zk.getChildren(zkParentPath, null);
		Stat stat = new Stat();
		byte[] data = s_zk.getData(zkParentPath, false, stat);
		parentNode.addChild(new OutputNode(zkParentPath+"_Stats", printStat(stat)));
		if ( children.isEmpty()) {
			// might be data node
			String dataStr = new String(data);
			if ( zkParentPath.equalsIgnoreCase(CLDBConfiguration.ZK_ZNODE_KVSTORE_CONTAINER_INFO)) {
		        ContainerInfo kvStoreContainerInfo;
				try {
					kvStoreContainerInfo = ContainerInfo.parseFrom(data);
			        dataStr = Util.printContainerInfoForCLI(kvStoreContainerInfo);
				} catch (InvalidProtocolBufferException e) {
				        /**
				        * <MAPR_ERROR>
				        * Message:Exception while trying to parse container info
				        * Function:DumpCommands.getRecursiveZKInfo()
				        * Meaning:An error occurred.
				        * Resolution:Contact technical support.
				        * </MAPR_ERROR>
				        */
					LOG.error("Exception while trying to parse container info", e);
				}
			} 		
			parentNode.addChild(new OutputNode(zkParentPath, dataStr));
		} else {
			parentNode.addChild(new OutputNode(zkParentPath));
			for ( String child : children ) {
				OutputNode node = new OutputNode(zkParentPath);
				String newPath = zkParentPath;
				if ( !zkParentPath.endsWith("/")) {
				  newPath = newPath + "/" + child;
				} else {
				  newPath = newPath + child;
				}
				getRecursiveZKInfo(s_zk, node, newPath);
				parentNode.addChild(node);
			}
		}
  }
  private static String printStat(Stat stat) {
	  StringBuilder statB = new StringBuilder();
      statB.append("\ncZxid = " + stat.getCzxid());
      statB.append(",ctime = " + new Date(stat.getCtime()).toString());
      statB.append(",mZxid = " + stat.getMzxid());
      statB.append(",mtime = " + new Date(stat.getMtime()).toString());
      statB.append(",pZxid = " + stat.getPzxid());
      statB.append(",cversion = " + stat.getCversion());
      statB.append(",dataVersion = " + stat.getVersion());
      statB.append(",aclVersion = " + stat.getAversion());
      statB.append(",ephemeralOwner = " + stat.getEphemeralOwner());
      statB.append(",dataLength = " + stat.getDataLength());
      statB.append(",numChildren = " + stat.getNumChildren());
      return statB.toString();
  }

}
