  /* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import java.util.ArrayList;
import java.util.Map;

import com.mapr.cli.common.RemoteCommandExecutor;

import com.mapr.fs.proto.clustermetrics.ClusterMetricsProto.CommandId;
import com.mapr.fs.proto.clustermetrics.ClusterMetricsProto.ExecuteCommandRequest;
import org.apache.log4j.Logger;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.MessageLite;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.baseutils.Errno;
import com.mapr.cli.common.ListCommand;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import com.mapr.cliframework.base.inputparams.IntegerInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.cliframework.util.FieldInfo;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.FileServerInfo;
import com.mapr.fs.cldb.proto.CLDBProto.FileServerListRequest;
import com.mapr.fs.cldb.proto.CLDBProto.FileServerListResponse;
import com.mapr.fs.cldb.proto.CLDBProto.NodeInfo;
import com.mapr.fs.proto.Common;


public class DiskCommands extends ListCommand implements CLIInterface {

     /* Define all the parameters used in Registry */
     public static final String HOST_PARAM_NAME = "host";
     public static final String STRIPEWIDTH_PARAM_NAME = "stripeWidth";
     public static final String DISK_PARAM_NAME = "disks";
     public static final String FORCE_PARAM_NAME   = "force";
     public static final String SYSTEM_PARAM_NAME   = "system";     
     public static final String OUTPUT_PARAM_NAME = "output";
     public static final String START_PARAM_NAME = "start";
     public static final String LIMIT_PARAM_NAME = "limit";
     /*public static final String COLUMNS_PARAM_NAME   = "columns";
     public static final String SORT_PARAM_NAME = "sort";
     public static final String SORT_DIRECTION_PARAM_NAME = "dir";*/

     public static final String INSTALL_DIR = MapRCliUtil.getMapRInstallDir();     
     public static final String DISKLIST_SH = INSTALL_DIR + "/server/disklist.sh ";
     public static final String DISKADD_SH = INSTALL_DIR + "/server/diskadd.sh ";
     public static final String DISKREMOVE_SH = INSTALL_DIR + "/server/diskremove ";
     private static final Logger LOG = Logger.getLogger(DiskCommands.class);
     private static final int ERR_CODE_VOLUMES_LOSS=150;
     private static final int ERR_CODE_METADATA_LOSS=151;
     private static final int ERR_CODE_GENERIC=152;

     static enum DiskPrintOption {
       LIST_SYSTEM,
       LIST_NONSYSTEM,
       LIST_ALL
     };
     
     /* Define sub-sub command */
     static final CLICommand listallCmd = new CLICommand(
        "listall",
        "",
        DiskCommands.class, 
        ExecutionTypeEnum.NATIVE,
        /* Add parameters in a hash map key is string,  value is BaseInputParameter */
        new ImmutableMap.Builder<String, BaseInputParameter>()
             /* parameter 1 */
             .put(MapRCliUtil.CLUSTER_NAME_PARAM,
                new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
                     "cluster name",
                     CLIBaseClass.NOT_REQUIRED,
                     null))
             .put(DiskCommands.START_PARAM_NAME,
                 new IntegerInputParameter(DiskCommands.START_PARAM_NAME,
                     "index of the first node (starting from 0)",
                     CLIBaseClass.NOT_REQUIRED,
                     0))
             .put(DiskCommands.LIMIT_PARAM_NAME,
                 new IntegerInputParameter(DiskCommands.LIMIT_PARAM_NAME,
                     "number of nodes to query",
                     CLIBaseClass.NOT_REQUIRED,
                     Integer.MAX_VALUE))
             .put(DiskCommands.OUTPUT_PARAM_NAME,
                 new TextInputParameter(DiskCommands.OUTPUT_PARAM_NAME,
                     "<terse|verbose>",
                     CLIBaseClass.NOT_REQUIRED,
                     "verbose"))
             .build(),
         null        /* sub-commands */
      ).setShortUsage("disk listall -cluster <clustername> -start <node index> -limit <number of nodes> -output terse|verbose");

     /* Define sub-sub command */
     static final CLICommand listCmd = new CLICommand(
        "list",
        "",
        DiskCommands.class, 
        ExecutionTypeEnum.NATIVE,
        /* Add parameters in a hash map key is string,  value is BaseInputParameter */
        new ImmutableMap.Builder<String, BaseInputParameter>()
             .put(DiskCommands.HOST_PARAM_NAME,
                 new TextInputParameter(
                     DiskCommands.HOST_PARAM_NAME,
                     "name/ip",
                     CLIBaseClass.REQUIRED,
                     null))
             .put(DiskCommands.SYSTEM_PARAM_NAME,
                new BooleanInputParameter(DiskCommands.SYSTEM_PARAM_NAME,
                     "1/0",
                     CLIBaseClass.NOT_REQUIRED,
                     null))
             .put(DiskCommands.OUTPUT_PARAM_NAME,
                 new TextInputParameter(DiskCommands.OUTPUT_PARAM_NAME,
                     "<terse|verbose>",
                     CLIBaseClass.NOT_REQUIRED,
                     "verbose"))       
             .build(),
         null        /* sub-commands */
      ).setShortUsage("disk list -host <name/ip> -output terse|verbose");

     /* Define sub-sub command */
     static final CLICommand addCmd = new CLICommand(
        "add",
        "",
        DiskCommands.class, 
        ExecutionTypeEnum.NATIVE,
        /* Add parameters in a hash map key is string,  value is BaseInputParameter */
        new ImmutableMap.Builder<String, BaseInputParameter>()
            /* parameter 1 */
            .put(DiskCommands.HOST_PARAM_NAME,
               new TextInputParameter(
                   DiskCommands.HOST_PARAM_NAME,
                   "name/ip",
                   CLIBaseClass.REQUIRED,
                   null))
            /* parameter 2 */
            .put(DiskCommands.DISK_PARAM_NAME,
               new TextInputParameter(
                   DiskCommands.DISK_PARAM_NAME,
                   "comma-separated list of disks",
                   CLIBaseClass.REQUIRED,
                   null))
            /* parameter 3 */
            .put(DiskCommands.STRIPEWIDTH_PARAM_NAME,
               new TextInputParameter(
                   DiskCommands.STRIPEWIDTH_PARAM_NAME,
                   "stripe-width",
                   CLIBaseClass.NOT_REQUIRED,
                   null))
            .build(),
         null        /* sub-commands */
      ).setShortUsage("disk add [-stripeWidth <stripe-width>] -host <name/ip> -disks <disks>");

     /* Define sub-sub command */
     static final CLICommand removeCmd = new CLICommand(
        "remove",
        "",
        DiskCommands.class, 
        ExecutionTypeEnum.NATIVE,
        /* Add parameters in a hash map key is string,  value is BaseInputParameter */
        new ImmutableMap.Builder<String, BaseInputParameter>()
            /* parameter 1 */
            .put(DiskCommands.HOST_PARAM_NAME,
                new TextInputParameter(
                    DiskCommands.HOST_PARAM_NAME,
                    "name/ip",
                    CLIBaseClass.REQUIRED,
                    null))
            /* parameter 2 */
            .put(DiskCommands.DISK_PARAM_NAME,
                new TextInputParameter(
                    DiskCommands.DISK_PARAM_NAME,
                    "comma-separated list of disks",
                    CLIBaseClass.REQUIRED,
                    null))
             /* parameter 3 */
            .put(DiskCommands.FORCE_PARAM_NAME,
                new BooleanInputParameter(
                    DiskCommands.FORCE_PARAM_NAME,
                    "<true|false OR 1|0>. Need this parameter to actually remove the disk, " + 
                    "otherwise this command behaves like a test remove",
                    CLIBaseClass.NOT_REQUIRED,
                    false))
            .put(MapRCliUtil.CLUSTER_NAME_PARAM, 
                new TextInputParameter(
                    MapRCliUtil.CLUSTER_NAME_PARAM, "cluster_name",
                    CLIBaseClass.NOT_REQUIRED, 
                    null))        
            .build(),
        null        /* sub-commands */
     ).setShortUsage("disk remove [-force <true|false OR 1|0>] -host <name/ip> -disks <disks> -cluster <clustername>");

     /* Define sub command */
     public static final CLICommand diskCommands = new CLICommand(
        "disk",
        "",
        CLIUsageOnlyCommand.class,
        ExecutionTypeEnum.NATIVE,
        new CLICommand [] {         /* array of subcommands */
         listallCmd,
	       listCmd,
	       addCmd,
	       removeCmd
	    }
     ).setShortUsage("disk [list|listall|add|remove]");
     

     static enum DiskEntryField {
    	 	hn,
    	 	n,
    	 	mt,
    	 	vn,
    	 	mn,
    	 	sn,
    	 	fw,
    	 	dst,
    	 	dsu,
    	 	dsa,
    	 	fs,
    	 	pst,
    	 	st,
    	 	err,
    	 	sp,
    	 	ft
     };
    
     static String [] DiskEntryFieldLongName = {
       "hostname",
    	 "diskname",
    	 "mount",
    	 "vendor",
    	 "modelnum",
    	 "serialnum",
    	 "firmwareversion",
    	 "totalspace",
    	 "usedspace",
    	 "availablespace",
    	 "fstype",
    	 "powerstatus",
    	 "status",
    	 "errormsg",
    	 "storagepoolid",
    	 "failuretime"
     };

     long columns = 0xffffffff;
     DiskEntryField sortField = null;
     boolean asc = true; // direction
     
     public static Map<DiskEntryField, FieldInfo> fieldTable = 
    	    new ImmutableMap.Builder<DiskEntryField, FieldInfo>()
    	    .put(DiskEntryField.hn, 
    	    		new FieldInfo(DiskEntryField.hn.ordinal(), 
    	    				DiskEntryField.hn.name(),
    	    				DiskEntryFieldLongName[DiskEntryField.hn.ordinal()],
    	    				String.class))
    	    .put(DiskEntryField.n, 
    	    		new FieldInfo(DiskEntryField.n.ordinal(), 
    	    				DiskEntryField.n.name(),
    	    				DiskEntryFieldLongName[DiskEntryField.n.ordinal()],
    	    				String.class))
    	    .put(DiskEntryField.mt, 
    	    		new FieldInfo(DiskEntryField.mt.ordinal(), 
    	    				DiskEntryField.mt.name(),
    	    				DiskEntryFieldLongName[DiskEntryField.mt.ordinal()],
    	    				String.class))
    	    .put(DiskEntryField.vn, 
    	    		new FieldInfo(DiskEntryField.vn.ordinal(), 
    	    				DiskEntryField.vn.name(),
    	    				DiskEntryFieldLongName[DiskEntryField.vn.ordinal()],
    	    				String.class))				
    	    .put(DiskEntryField.mn, 
    	    		new FieldInfo(DiskEntryField.mn.ordinal(), 
    	    				DiskEntryField.mn.name(),
    	    				DiskEntryFieldLongName[DiskEntryField.mn.ordinal()],
    	    				String.class))
    	    .put(DiskEntryField.sn, 
    	    		new FieldInfo(DiskEntryField.sn.ordinal(), 
    	    				DiskEntryField.sn.name(),
    	    				DiskEntryFieldLongName[DiskEntryField.sn.ordinal()],
    	    				String.class))
    	    .put(DiskEntryField.fw, 
    	    		new FieldInfo(DiskEntryField.fw.ordinal(), 
    	    				DiskEntryField.fw.name(),
    	    				DiskEntryFieldLongName[DiskEntryField.fw.ordinal()],
    	    				String.class))
    	    .put(DiskEntryField.dst, 
    	    		new FieldInfo(DiskEntryField.dst.ordinal(), 
    	    				DiskEntryField.dst.name(),
    	    				DiskEntryFieldLongName[DiskEntryField.dst.ordinal()],
    	    				Long.class))
    	    .put(DiskEntryField.dsu, 
    	    		new FieldInfo(DiskEntryField.dsu.ordinal(), 
    	    				DiskEntryField.dsu.name(),
    	    				DiskEntryFieldLongName[DiskEntryField.dsu.ordinal()],
    	    				Long.class))
    	    .put(DiskEntryField.dsa, 
    	    		new FieldInfo(DiskEntryField.dsa.ordinal(), 
    	    				DiskEntryField.dsa.name(),
    	    				DiskEntryFieldLongName[DiskEntryField.dsa.ordinal()],
    	    				Long.class))
    	    .put(DiskEntryField.fs, 
    	    		new FieldInfo(DiskEntryField.fs.ordinal(), 
    	    				DiskEntryField.fs.name(),
    	    				DiskEntryFieldLongName[DiskEntryField.fs.ordinal()],
    	    				String.class))
    	    .put(DiskEntryField.pst, 
    	    		new FieldInfo(DiskEntryField.pst.ordinal(), 
    	    				DiskEntryField.pst.name(),
    	    				DiskEntryFieldLongName[DiskEntryField.pst.ordinal()],
    	    				String.class))				
    	    .put(DiskEntryField.st, 
    	    		new FieldInfo(DiskEntryField.st.ordinal(), 
    	    				DiskEntryField.st.name(),
    	    				DiskEntryFieldLongName[DiskEntryField.st.ordinal()],
    	    				Integer.class))				
    	    .put(DiskEntryField.err, 
    	    		new FieldInfo(DiskEntryField.err.ordinal(), 
    	    				DiskEntryField.err.name(),
    	    				DiskEntryFieldLongName[DiskEntryField.err.ordinal()],
    	    				String.class))
    	    .put(DiskEntryField.ft, 
    	    		new FieldInfo(DiskEntryField.ft.ordinal(), 
    	    				DiskEntryField.ft.name(),
    	    				DiskEntryFieldLongName[DiskEntryField.ft.ordinal()],
    	    				Long.class))				
    	    .build();

     public DiskCommands(ProcessedInput input, CLICommand cliCommand) {
         super(input, cliCommand);
     }

  private boolean isSystemPartition(String fstype, int mount, String totalStr) {
       if (mount == 1)
         return true;
       
       try {
         if (!totalStr.startsWith("unknown") && Long.parseLong(totalStr) == 0)
           return true;
       } catch (Exception e) {
         // Do nothing
       }
       
       fstype = fstype.trim().toLowerCase();
       if (fstype.contains("swap") || fstype.contains("lvm") || fstype.contains("raid"))
         return true;
       
       return false;
     }
     
     private OutputNode PrintDiskInfo(String s, DiskPrintOption dp,
    		                              boolean verbose) {
    	 try {
    	   String[] str = s.split("\\s");
      	 
    	   String fstype = str[DiskEntryField.fs.ordinal()];
      	 int mount = Integer.parseInt(str[DiskEntryField.mt.ordinal()]);      	 
      	 String totalStr = str[DiskEntryField.dst.ordinal()];
      	 
      	 if ((dp == DiskPrintOption.LIST_SYSTEM && !isSystemPartition(fstype, mount, totalStr)) || 
             (dp == DiskPrintOption.LIST_NONSYSTEM && isSystemPartition(fstype, mount, totalStr))) {
           return null;
         }
      	 
      	 OutputNode node = new OutputNode();
      	 DiskEntryField[] arr = DiskEntryField.values();
      	 
      	 for (int i = 0; i < str.length; i++) {
      		if (str[i].startsWith("unknown"))
            continue;
      		if (str[i].startsWith("Run_`smartctl"))
            str[i] = str[i].replaceAll("_", " ");
      		if ((columns & (1L << i)) != 0) {
      			String columnName = verbose ? DiskEntryFieldLongName[i] : arr[i].toString();
      		  node.addChild(new OutputNode(columnName, str[i]));
      		}
      	 }
      	 return node;
    	 } catch (Exception e) {
    	   return null;
    	 }
     }
     
     private void PrintDisksInfo(OutputHierarchy out, ArrayList<String> ListOfDisks, 
                                 DiskPrintOption dp)
         throws CLIProcessingException {
    	 
    	 boolean verbose = true;
    	 if (isParamPresent(OUTPUT_PARAM_NAME) && 
    			 getParamTextValue(OUTPUT_PARAM_NAME, 0).equalsIgnoreCase("terse")) {
    		 verbose = false;
    	 }
    	  
       // Do Sorting and filtering here	 
       for (String s: ListOfDisks) {
         s = s.trim();
         if (s.isEmpty())
           continue;
         OutputNode node = PrintDiskInfo(s, dp, verbose);
    	   if (node != null) {
    	     out.addNode(node);
    	   }
       }
     }

  void FetchDisksInfo(String host, ArrayList<String> ListOfDisks) throws CLIProcessingException {
    ExecuteCommandRequest.Builder requestBuilder = ExecuteCommandRequest.newBuilder();
    requestBuilder.setCommandId(CommandId.DISK_LIST);
    requestBuilder.setCreds(getUserCredentials());

    try {
      for (String s : RemoteCommandExecutor.execute(host, requestBuilder.build())) {
        ListOfDisks.add(host + " " + s);
      }
    } catch(CLIProcessingException e) {
      throw new CLIProcessingException("Error trying to reach host \"" + host +"\". Check if hostname is valid and up");

    }
  }

     /*void SetupSortLimitColumns() throws CLIProcessingException {
    	
    	 // Sorting related parameters
     	 String sortFieldName = getParamTextValue(SORT_PARAM_NAME, 0);            
         if (sortFieldName != null && (!sortFieldName.equals("none"))) {
            FieldInfo info = FilterUtil.searchFieldByName(fieldTable, 
                                                          sortFieldName.trim());
            if (info != null) {
               sortField = DiskEntryField.valueOf(sortFieldName);
            } else {
               sortField = null;
            }
         }
         
         if (getParamTextValue(SORT_DIRECTION_PARAM_NAME, 0).equals("DESC")) {
       	    asc = false;
         }
         
         // Columns related parameters
         String columnsString = getParamTextValue(COLUMNS_PARAM_NAME, 0);
         if (columnsString != null && (!columnsString.equals("none"))) {
            columns = FilterUtil.getColumns(fieldTable, columnsString.trim());
         } else {
         	// send all columns
         	columns = 0xffffffff; 
         }
     }*/
     
     // Extract host from fs
     public static String getHostname(FileServerInfo fs) {
       if (fs == null) {
         return null;
       }
       
       try {
   	     Common.IPAddress ip = fs.getAddress(0);
         
         if (ip != null ) {
           return ip.getHostname();
         }
       } catch (Exception e) {
         return null;
       }
       return null;
     }

     void DiskAddRemove(OutputHierarchy out, boolean add) throws CLIProcessingException { 
    	 String host = getParamTextValue(HOST_PARAM_NAME, 0);
    	 if ((host == null) || host.isEmpty()) {
    	   out.addError(new OutputError(Errno.EINVAL, "Invalid host"));
    		 return;
    	 }
    
    	 String stripeWidth = null;
       if (isParamPresent(STRIPEWIDTH_PARAM_NAME))
         stripeWidth = getParamTextValue(STRIPEWIDTH_PARAM_NAME, 0);

    	 String disklist = getParamTextValue(DISK_PARAM_NAME, 0);
    	 if ((disklist == null) || (disklist.length() == 0)) {
    	   /**
    	   * <MAPR_ERROR>
    	   * Message:Invalid config
    	   * Function:DiskCommands.DiskAddRemove()
    	   * Meaning:An error occurred.
    	   * Resolution:Contact technical support.
    	   * </MAPR_ERROR>
    	   */
    	   out.addError(new OutputError(Errno.EINVAL, "Invalid config"));
    		 return;
    	 }
    	 if (disklist.startsWith("[")) {
    	   // Remove beginning and trailing '[' and ']'
    	   disklist = 
    	     disklist.substring(1, disklist.length() - 1);
    	 }
     	 disklist = disklist.replaceAll("'", "");
    	 disklist = disklist.replaceAll(",", " ");
    	 
       CommandId commandId;
       String commandArgs = "";

    	 if (add) {
         commandId = CommandId.DISK_ADD;
         if (stripeWidth != null) {
           commandArgs = " -W " + stripeWidth + " ";
         }
    		 commandArgs += disklist;
    	 } else {
         commandId = CommandId.DISK_REMOVE;
    	    StringBuffer options = new StringBuffer();
    	    if (isParamPresent(FORCE_PARAM_NAME) && 
    	      (getParamBooleanValue(FORCE_PARAM_NAME, 0))) {
    	      options.append("-f ");
    	    }
    	    
    	    CLDBRpcCommonUtils.IpPort ipPort = null;
    	    if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
    	      String cluster = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0);
            if (!CLDBRpcCommonUtils.getInstance().isValidClusterName(cluster)) {
              out.addError(new OutputError(Errno.EUCLUSTER, "Invalid cluster: " + cluster));
              return;
            }
    	      ipPort = CLDBRpcCommonUtils.getInstance().getCurrentValidIpPort(cluster);
    	    } else {
    	      ipPort = CLDBRpcCommonUtils.getInstance().getCurrentValidIpPort();
    	    }
    	    
    	    if (ipPort != null) {
            for (String cldbip : ipPort.getAddr()) {
               if (!cldbip.isEmpty()) {
                 options.append("-i " + cldbip + " ");
                 options.append("-p " + ipPort.getPort() + " ");
                 break;
               }
            }
    	    }
    	    commandArgs = options + disklist;
    	 }

       ExecuteCommandRequest.Builder requestBuilder = ExecuteCommandRequest.newBuilder();
       requestBuilder.setCommandId(commandId);
       requestBuilder.setArgs(commandArgs);
       requestBuilder.setCreds(getUserCredentials());

  	   for (String line : RemoteCommandExecutor.execute(host, requestBuilder.build())) {
  	      line = line.trim();
  	      if (line.isEmpty())
  	        continue;
  	      
  	      String [] words = line.split("\\s+");
          if (words.length < 2)
            continue;

  	      String disk = words[0];
  	      
    		  if (words[1].startsWith("added") || 
    		      words[1].startsWith("removed")) {
      		  // Success
      		  OutputNode node = new OutputNode();
      		  node.addChild(new OutputNode("host", host));
      		  node.addChild(new OutputNode("disk", disk));  		  
      		  node.addChild(new OutputNode("message", words[1]));
      		  out.addNode(node);
    		  } else {
    		    int err = Errno.EOPFAILED;
    		    String errMsg = null;
    		    
    		    if (words.length > 3 && words[3] != null && !words[3].isEmpty()) {
              words[3] = words[3].replaceAll("[^0-9]+$", "");
    		      try {
    		        err = Integer.parseInt(words[3]);
    		      } catch (Exception e) {
    		        err = Errno.EOPFAILED;
    		      }
            }
    		    
    		    /**
             * <MAPR_ERROR>
             * Message:Failed operation for disk <disk> <error>
             * Function:DiskCommands.DiskAddRemove()
             * Meaning:An error occurred.
             * Resolution:Contact technical support.
             * </MAPR_ERROR>
             */
      	     if (err == ERR_CODE_VOLUMES_LOSS) {
      	       int lostVols = 0;     
      	       if (words.length > 4 && words[4] != null && !words[4].isEmpty()) {
                 try {
                   lostVols = Integer.parseInt(words[4]);
                 } catch (Exception e) {
                 }
               }
      	       errMsg = "Operation may cause loss of data due to under-replication";
      	       if (lostVols > 0)
      	         errMsg += " of " + lostVols + " or more volumes";
               
             } else if (err == ERR_CODE_METADATA_LOSS) {
      	       errMsg = "Operation may bring down cluster due to loss of cluster meta-data";
               
      	     } else if (err == ERR_CODE_GENERIC) {
      	    	 errMsg = words[3];
             } else if (err == Errno.ENOENT) {
               // Disk has been removed from disktab if error returned is ENOENT (ret code 2)
               errMsg = Errno.toString(err) + "\nThe disk was either removed from /opt/mapr/conf/disktab earlier, or is invalid";
      	     } else {
      	    	 errMsg = Errno.toString(err);
             }
      	     
      	     out.addError(new OutputError(err, "Failed operation for disk " + disk + ", " + errMsg));
    		  }
  	   }
     }
          
     /* This is a function that does the real work. */
     @Override
     public CommandOutput executeRealCommand() throws CLIProcessingException {        
         OutputHierarchy out = new OutputHierarchy();  /* Root of comamnd output's hierarchy tree */
         CommandOutput output = new CommandOutput();
         output.setOutput(out);
         
         String cmd = cliCommand.getCommandName();
         if (cmd.equalsIgnoreCase("listall")) {
           try {
             list(out);
           } catch (Exception e) {
        	   out.addError(new OutputError(Errno.EOPFAILED, e.getLocalizedMessage()));
        	   return output;
           }
         } else if (cmd.equalsIgnoreCase("list")){
        	 String host = getParamTextValue(HOST_PARAM_NAME, 0);
        	 if ((host == null) || host.isEmpty()) {
        		 return output;
        	 }
        	 
        	 ArrayList<String> ListOfDisks = new ArrayList<String>();
        	 try {
        	   FetchDisksInfo(host, ListOfDisks);
        	 } catch (Exception e) {
        	   out.addError(new OutputError(Errno.ENOSYS, e.getLocalizedMessage()));
        	   return output;
        	 }
        	 DiskPrintOption dp = DiskPrintOption.LIST_ALL;
        	 if (isParamPresent(SYSTEM_PARAM_NAME)) {
        	   if (getParamBooleanValue(SYSTEM_PARAM_NAME, 0))
        	     dp = DiskPrintOption.LIST_SYSTEM;
        	   else
        	     dp = DiskPrintOption.LIST_NONSYSTEM;
        	 }
        	 
        	 PrintDisksInfo(out, ListOfDisks, dp);
         } else if (cmd.equalsIgnoreCase("add")) {
           try {
        	   DiskAddRemove(out, true);
           } catch (Exception e) {
             out.addError(new OutputError(Errno.ENOSYS, e.getLocalizedMessage()));
           }
         } else if (cmd.equalsIgnoreCase("remove")) {
           try {
        	   DiskAddRemove(out, false);
           } catch (Exception e) {
             /**
             * <MAPR_ERROR>
             * Message:
             * Function:
             * Meaning:
             * Resolution:
             * </MAPR_ERROR>
             */
             out.addError(new OutputError(Errno.ENOSYS, e.getLocalizedMessage()));
           }
         }
         return output;
     }
     
     private FileServerListRequest.Builder getFileServerListRequestBuilder() throws CLIProcessingException {
       return FileServerListRequest.newBuilder()
           .setCreds(getUserCredentials())
           .setColumns((1L << NodeInfo.Ip.getNumber()))
           .setLimiter(getNextLimiter(getParamIntValue(START_PARAM_NAME, 0), 0, 
               getParamIntValue(START_PARAM_NAME, 0), getParamIntValue(LIMIT_PARAM_NAME, 0),
               ServerCommands.NUM_NODES_PER_RPC));
     }
     
     @Override
     public FileServerListRequest buildNextRequest(MessageLite prevReq, MessageLite prevResp) throws CLIProcessingException {
       FileServerListRequest.Builder newReqBuilder = null;
       if (prevReq != null) {
         newReqBuilder = FileServerListRequest.newBuilder((FileServerListRequest) prevReq);
       } else {
         newReqBuilder = getFileServerListRequestBuilder();
       }
       
       if (prevResp != null) {
         int prevStart = newReqBuilder.getLimiter().getStart();
         int prevCount = ((FileServerListResponse) prevResp).getInfoCount();
         int origStart = getParamIntValue(START_PARAM_NAME, 0);
         int origLimit = getParamIntValue(LIMIT_PARAM_NAME, 0);
         newReqBuilder.setLimiter(getNextLimiter(prevStart, prevCount,
             origStart, origLimit, ServerCommands.NUM_NODES_PER_RPC));
       }
       
       return newReqBuilder.build();
     }
     
     @Override
     public boolean hasMore(MessageLite prevReq, MessageLite prevResp) throws CLIProcessingException {
       return hasMore(getParamIntValue(START_PARAM_NAME, 0), 
           getParamIntValue(LIMIT_PARAM_NAME, 0), 
           ((FileServerListRequest) prevReq).getLimiter().getStart(), 
           ((FileServerListResponse) prevResp).getInfoCount());
     }

     @Override
     public void processResponse(OutputHierarchy out, MessageLite response) throws CLIProcessingException {
       FileServerListResponse resp = (FileServerListResponse) response;
       ArrayList<String> ListOfDisks = new ArrayList<String>();

       String prevhost = null;
       for (int i = 0; i < resp.getInfoCount(); ++i) { 
          String host = getHostname(resp.getInfo(i));
          if (host == null || host.isEmpty()) {
             continue;
          }

          // Just in case, avoid duplicate servers
          if (prevhost != null && host.equalsIgnoreCase(prevhost)) {
           continue;
          }

          try {
             FetchDisksInfo(host, ListOfDisks);
          } catch (Exception e) {
            /**
            * <MAPR_ERROR>
            * Message:Exception in FetchDisksInfo <error>
            * Function:DiskCommands.processResponse()
            * Meaning:An error occurred.
            * Resolution:Contact technical support.
            * </MAPR_ERROR>
            */
            LOG.error("Exception in FetchDisksInfo " + e);
            // TODO: Add partial output
            continue;
          }
       }
       
       PrintDisksInfo(out, ListOfDisks, DiskPrintOption.LIST_ALL);
     }

     @Override
     public FileServerListResponse sendRequest(MessageLite request) throws CLIProcessingException {
       FileServerListRequest req = (FileServerListRequest) request;
       byte[] replyData = null;
       if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
         String cluster = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0);
         if (!CLDBRpcCommonUtils.getInstance().isValidClusterName(cluster)) {
           throw new CLIProcessingException("Invalid cluster: " + cluster);
         }
         try {
           replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
                         cluster,
                         Common.MapRProgramId.CldbProgramId.getNumber(),
                         CLDBProto.CLDBProg.FileServerListProc.getNumber(), 
                         req, FileServerListResponse.class);
         } catch (Exception e) {
           throw new CLIProcessingException(e);
         }
       } else {
         try {
           replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
                         Common.MapRProgramId.CldbProgramId.getNumber(),
                         CLDBProto.CLDBProg.FileServerListProc.getNumber(), 
                         req, FileServerListResponse.class);
         } catch (Exception e) {
           throw new CLIProcessingException(e);
         }
       }
       
       if (replyData != null) {
         return getFileServerListResponse(replyData);
       } else {
         /**
         * <MAPR_ERROR>
         * Message:RPC Request to list Nodes failed. No data returned
         * Function:DiskCommands.processResponse()
         * Meaning:A communication error occurred.
         * Resolution:Contact technical support.
         * </MAPR_ERROR>
         */
         throw new CLIProcessingException("Failed to get list of servers for this cluster");
       }
     }
     
     private FileServerListResponse getFileServerListResponse(byte[] replyData)
         throws CLIProcessingException {
       try {
         return FileServerListResponse.parseFrom(replyData);
       } catch (InvalidProtocolBufferException ipbe) {
         throw new CLIProcessingException("Exception while parsing the RPC " +
             "response data into FileServerListResponse proto object.", ipbe);
       }
     }
}
