package com.mapr.cli;

import java.io.IOException;
import java.text.DateFormat;
import java.text.ParseException;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.zip.Deflater;

import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.log4j.Logger;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.ByteString;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import com.mapr.cliframework.base.inputparams.DateInputParameter;
import com.mapr.fs.cldb.dialhome.metrics.MetricsReader;
import com.mapr.fs.cldb.dialhome.metrics.MetricsUtil;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.DialHomeEnableRequest;
import com.mapr.fs.cldb.proto.CLDBProto.DialHomeEnableResponse;
import com.mapr.fs.cldb.proto.CLDBProto.DialHomeLastDialedRequest;
import com.mapr.fs.cldb.proto.CLDBProto.DialHomeLastDialedResponse;
import com.mapr.fs.cldb.proto.CLDBProto.DialHomeStatus;
import com.mapr.fs.cldb.proto.CLDBProto.DialHomeStatusRequest;
import com.mapr.fs.cldb.proto.CLDBProto.DialHomeStatusResponse;
import com.mapr.fs.cldb.proto.CLDBProto.DialHomeSuccessfulAckRequest;
import com.mapr.fs.cldb.proto.CLDBProto.DialHomeSuccessfulAckResponse;
import com.mapr.fs.cldb.proto.dialhome.MetricsProto.DayMetrics;
import com.mapr.fs.cldb.proto.dialhome.MetricsProto.DayMetrics.Builder;
import com.mapr.fs.cldb.proto.dialhome.MetricsProto.MetricsData;
import com.mapr.fs.proto.Common;
import com.mapr.util.MapRFSUtil;
import com.mapr.security.MaprSecurityException;

public class DialHomeCommands extends CLIBaseClass {
  private static final Logger LOG = Logger.getLogger(DialHomeCommands.class);
  
  private static final String FOR_DAY = "forDay";
  private static final String ENABLE = "enable";

  static final CLICommand metricsCmd = new CLICommand("metrics", "get dialhome metrics",
      DialHomeCommands.class, ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>().put(
          DialHomeCommands.FOR_DAY,
          new DateInputParameter(DialHomeCommands.FOR_DAY,
              "Date for which metrics are being queried. Accepted values: UTC timestamp in millisecond or a UTC date in MM/DD/YY format", CLIBaseClass.REQUIRED,
              getYesterday())).build(), null)
      .setShortUsage("dialhome metrics -forDay <MM/DD/YY>");

  static final CLICommand enableCmd = new CLICommand("enable", "enable/disable dialhome",
      DialHomeCommands.class, ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>().put(
          DialHomeCommands.ENABLE,
          new BooleanInputParameter(DialHomeCommands.ENABLE,
              "enable/disable dialhome", CLIBaseClass.REQUIRED,
              true)).build(), null)
      .setShortUsage("dialhome metrics -enable <true/false>");

  static final CLICommand statusCmd = new CLICommand("status",
      "query if dial home is enabled/disabled", DialHomeCommands.class, ExecutionTypeEnum.NATIVE,
      null);

  static final CLICommand ackDialCmd = new CLICommand("ackdial",
      "ack a successful dial home", DialHomeCommands.class, ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>().put(
          DialHomeCommands.FOR_DAY,
          new DateInputParameter(DialHomeCommands.FOR_DAY,
              "Date for which the recorded metrics were successfully dialed home. Accepted values: UTC timestamp in millisecond or a UTC date in MM/DD/YY format", CLIBaseClass.REQUIRED,
              getYesterday())).build(), null)
      .setShortUsage("dialhome ackdial -forDay <MM/DD/YY>");
  
  static final CLICommand lastDialedCmd = new CLICommand("lastdialed",
      "query the last successful dial home date", DialHomeCommands.class, ExecutionTypeEnum.NATIVE,
      null);

  public static final CLICommand dialhomeCmds = new CLICommand("dialhome", "",
      CLIUsageOnlyCommand.class, ExecutionTypeEnum.NATIVE, new CLICommand[] { metricsCmd, enableCmd,
          statusCmd, ackDialCmd, lastDialedCmd }).setShortUsage("dialhome [metrics|enable|status|ackdial|lastdialed]");

  public DialHomeCommands(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }

  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    String cmd = cliCommand.getCommandName();
    if (cmd.equalsIgnoreCase("metrics")) {
      getDialHomeMetrics(out);
    } else if (cmd.equalsIgnoreCase("enable")) {
      enableDialHome(out, getParamBooleanValue(ENABLE, 0));
    } else if (cmd.equalsIgnoreCase("status")) {
      getDialHomeStatus(out);
    } else if (cmd.equalsIgnoreCase("ackdial")) {
      ackDialHome(out);
    } else if (cmd.equalsIgnoreCase("lastdialed")) {
      getLastDialedDate(out);
    }

    return output;
  }
    
  private void getDialHomeMetrics(OutputHierarchy out) throws CLIProcessingException {
    DialHomeStatusResponse resp = this.getDialHomeStatusResponse();
    if (resp == null || resp.getStatus() != Errno.SUCCESS) {
      out.addError(new OutputError(resp.getStatus(), "Couldn't retrieve metrics as the request to fetch dial home status failed"));
    } else if (!resp.hasDialHomeStatus()) {
      out.addError(new OutputError(resp.getStatus(), "Couldn't retrieve metrics as the dial home status could not be known"));
    } else if (resp.getDialHomeStatus().equals(DialHomeStatus.ENABLED)){
      Date forDay = getParamDateValue(FOR_DAY, 0);
      DayMetrics metricsForDay = this.getMetricsForDay(forDay.getTime());
      if (metricsForDay != null) {
        byte[] unCompressedDayMetrics = metricsForDay.toByteArray();
        byte[] compressedDayMetrics = compress(unCompressedDayMetrics);
        MetricsData metricsData = MetricsData.newBuilder()
                                   .setCompressedDayMetrics(ByteString.copyFrom(compressedDayMetrics))
                                   .setRawDayMetricsSize(unCompressedDayMetrics.length)
                                   .build();
        
        OutputNode outputNode = new OutputNode();
        outputNode.addChild(new OutputNode("metrics", metricsData.toByteArray()));
        out.addNode(outputNode);
      } else {
        out.addError(new OutputError(Errno.ENOMETRICSFORDAY, Errno.toString(Errno.ENOMETRICSFORDAY)));
      }
    } else {
      out.addError(new OutputError(Errno.EDIALHOMEDISABLED,
          "Couldn't retrieve metrics. " + Errno.toString(Errno.EDIALHOMEDISABLED)));
    }
  }

  private void enableDialHome(OutputHierarchy out, boolean enable) throws CLIProcessingException {
    DialHomeEnableRequest.Builder req = DialHomeEnableRequest.newBuilder()
        .setEnable(enable)
        .setCreds(getUserCredentials());

    byte[] replyData;
    try {
      if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        replyData =
            CLDBRpcCommonUtils.getInstance().sendRequest(
                getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0),
                Common.MapRProgramId.CldbProgramId.getNumber(),
                CLDBProto.CLDBProg.DialHomeEnableRequestProc.getNumber(), req.build(),
                DialHomeEnableResponse.class);
      } else {
        replyData =
            CLDBRpcCommonUtils.getInstance().sendRequest(
                Common.MapRProgramId.CldbProgramId.getNumber(),
                CLDBProto.CLDBProg.DialHomeEnableRequestProc.getNumber(), req.build(),
                DialHomeEnableResponse.class);
      }
      if (replyData == null) {
        out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
        return;
      }
      DialHomeEnableResponse resp = DialHomeEnableResponse.parseFrom(replyData);
      if (resp.getStatus() != Errno.SUCCESS) {
        if (resp.hasFailureReason()) {
          out.addError(new OutputError(resp.getStatus(), "Request to "
              + (enable ? "enable" : "disable") + " failed. Reason: " + resp.getFailureReason()));
        } else {
          out.addError(new OutputError(resp.getStatus(), "Request to "
              + (enable ? "enable" : "disable") + " failed."));
        }
      } else {
        out.addNode(new OutputNode("enabled", enable ? 1 : 0));
      }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      throw new CLIProcessingException("InvalidProtocolBufferException " + " Exception", e);
    }
  }

  private void getDialHomeStatus(OutputHierarchy out) throws CLIProcessingException {
    DialHomeStatusResponse resp = this.getDialHomeStatusResponse();
    if (resp == null || resp.getStatus() != Errno.SUCCESS) {
      out.addError(new OutputError(resp.getStatus(), "Request to fetch dial home status failed"));
    } else if (!resp.hasDialHomeStatus()) {
      out.addError(new OutputError(resp.getStatus(), "No dial home status in the response"));
    } else {
      OutputNode outputNode = new OutputNode();
      outputNode.addChild(new OutputNode("enabled", getStatusAsInt(resp.getDialHomeStatus())));
      out.addNode(outputNode);
    }
  }
  
  private void ackDialHome(OutputHierarchy out) throws CLIProcessingException {
    Date forDay = getParamDateValue(FOR_DAY, 0);
    
    if (!forDay.before(new Date())) {
      out.addError(new OutputError(Errno.EINVAL, "Date cannot be in future"));
      return;
    }
    
    try {
      if (!forDay.after(DateFormat.getDateInstance(DateFormat.SHORT).parse("1/1/70"))) {
        out.addError(new OutputError(Errno.EINVAL, "Date cannot be before unix epoch (1/1/70)"));
        return;
      }
    } catch (ParseException e) {
      LOG.error("Date cannot be before unix epoch (1/1/70)", e);
    }
    
    LOG.info("Sending RPC to CLDB to set lastdialed to " + forDay.getTime());
    DialHomeSuccessfulAckRequest req =
        DialHomeSuccessfulAckRequest.newBuilder().setForDay(forDay.getTime())
          .setCreds(getUserCredentials()).build();
    
    byte[] replyData;
    try {
      if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        replyData =
            CLDBRpcCommonUtils.getInstance().sendRequest(
                getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0),
                Common.MapRProgramId.CldbProgramId.getNumber(),
                CLDBProto.CLDBProg.DialHomeSuccessfulAckRequestProc.getNumber(), req,
                DialHomeSuccessfulAckResponse.class);
      } else {
        replyData =
            CLDBRpcCommonUtils.getInstance().sendRequest(
                Common.MapRProgramId.CldbProgramId.getNumber(),
                CLDBProto.CLDBProg.DialHomeSuccessfulAckRequestProc.getNumber(), req,
                DialHomeSuccessfulAckResponse.class);
      }
      
      if (replyData == null) {
        out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
        return;
      }
      DialHomeSuccessfulAckResponse resp = DialHomeSuccessfulAckResponse.parseFrom(replyData);
      if (resp.getStatus() != Errno.SUCCESS) {
        out.addError(new OutputError(resp.getStatus(), "Request to get metrics failed. Reason: "
            + Errno.toString(resp.getStatus())));
      }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      throw new CLIProcessingException("InvalidProtocolBufferException " + " Exception", e);
    }
  }
  
  private void getLastDialedDate(OutputHierarchy out) throws CLIProcessingException {
    DialHomeLastDialedRequest.Builder req = DialHomeLastDialedRequest.newBuilder()
                                              .setCreds(getUserCredentials());
    byte[] replyData;
    try {
      if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        replyData =
            CLDBRpcCommonUtils.getInstance().sendRequest(
                getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0),
                Common.MapRProgramId.CldbProgramId.getNumber(),
                CLDBProto.CLDBProg.DialHomeLastDialedRequestProc.getNumber(), req.build(),
                DialHomeLastDialedResponse.class);
      } else {
        replyData =
            CLDBRpcCommonUtils.getInstance().sendRequest(
                Common.MapRProgramId.CldbProgramId.getNumber(),
                CLDBProto.CLDBProg.DialHomeLastDialedRequestProc.getNumber(), req.build(),
                DialHomeLastDialedResponse.class);
      }
      if (replyData == null) {
        out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
        return;
      }
      DialHomeLastDialedResponse resp = DialHomeLastDialedResponse.parseFrom(replyData);
      if (resp.getStatus() != Errno.SUCCESS) {
        out.addError(new OutputError(resp.getStatus(),
            "Request to fetch last dialed date failed. Reason: " + Errno.toString(resp.getStatus())));
      } else {
        OutputNode outputNode = new OutputNode();
        outputNode.addChild(new OutputNode("date", resp.getLastDialedDate()));
        out.addNode(outputNode);
      }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      throw new CLIProcessingException("InvalidProtocolBufferException " + " Exception", e);
    }
  }
  
  private DialHomeStatusResponse getDialHomeStatusResponse() throws CLIProcessingException {
    DialHomeStatusRequest.Builder req = DialHomeStatusRequest.newBuilder()
                                          .setCreds(getUserCredentials());

    byte[] replyData;
    try {
      if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        replyData =
            CLDBRpcCommonUtils.getInstance().sendRequest(
                getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0),
                Common.MapRProgramId.CldbProgramId.getNumber(),
                CLDBProto.CLDBProg.DialHomeStatusRequestProc.getNumber(), req.build(),
                DialHomeEnableResponse.class);
      } else {
        replyData =
            CLDBRpcCommonUtils.getInstance().sendRequest(
                Common.MapRProgramId.CldbProgramId.getNumber(),
                CLDBProto.CLDBProg.DialHomeStatusRequestProc.getNumber(), req.build(),
                DialHomeStatusResponse.class);
      }
      if (replyData == null) {
        LOG.error("Couldn't connect to the CLDB service");
        return null;
      }
      return DialHomeStatusResponse.parseFrom(replyData);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      throw new CLIProcessingException("InvalidProtocolBufferException " + " Exception", e);
    }
  }
  
  private DayMetrics getMetricsForDay(long timeOfDay) {
    String metricsFilePath = MetricsUtil.getMetricsFilePath(timeOfDay);
    LOG.info("Retrieving metrics. Timestamp received: " + timeOfDay + ". Metrics file path: " + metricsFilePath);
    if (!MapRFSUtil.pathExists(metricsFilePath)) {
      LOG.error("Read Failed. Unable to find metrics file with path: " + metricsFilePath);
      return null;
    }

    long fileSize = 0;
    try {
      FileStatus fileStatus = MapRFSUtil.getMapRFileSystem().getFileStatus(new Path(metricsFilePath));
      fileSize = fileStatus.getLen();
    } catch (IOException e) {
      LOG.error("Failed to fetch the size of the metrics file. Path: " + metricsFilePath);
      return null;
    }


    if (fileSize > 0) {
      Builder dayMetricsBuilder = MetricsReader.readDayMetricsFromFile(metricsFilePath);
      if (dayMetricsBuilder == null) {
        return null;
      } else {
        return dayMetricsBuilder.build();
      }
    } else {
      LOG.warn("Size of the metrics file is zero. Path: " + metricsFilePath);
      return null;
    }
  }
  
  private static Date getYesterday() {
    Calendar cal = Calendar.getInstance();
    cal.setTime(new Date());
    cal.add(Calendar.DAY_OF_MONTH, -1);
    return cal.getTime();
  }

  private int getStatusAsInt(DialHomeStatus status) {
    switch (status) {
    case ENABLED:
      return 1;
    case DISABLED:
      return 0;
    }
    return -1;
  }
  
  private byte[] compress(byte[] input) {
    byte[] output = new byte[input.length];
    Deflater compresser = new Deflater(Deflater.BEST_COMPRESSION);
    compresser.setInput(input);
    compresser.finish();
    int compressedDataLength = compresser.deflate(output);
    if (LOG.isDebugEnabled()) {
      LOG.debug("Compression. Input size: " + input.length + ", output size: " + compressedDataLength);
    }
    return Arrays.copyOf(output, compressedDataLength);
  }
}
