/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Formatter;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import java.util.regex.Matcher;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.ByteString;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cli.common.NodesCommonUtils;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.TextCommandOutput;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import com.mapr.cliframework.base.inputparams.IntegerInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.cliframework.base.inputparams.LongInputParameter;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.fs.Rpc;
import com.mapr.fs.cldb.conf.CLDBConfiguration;
import com.mapr.fs.cldb.proto.CLDBProto.*;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.util.Util;
import com.mapr.fs.CompressionHelper;
import com.mapr.fs.proto.Fileserver.MiniInodeMsg;
import com.mapr.fs.proto.Security.ServerKeyType;
import com.mapr.fs.proto.Common.FidMsg;
import com.mapr.fs.proto.Common.FileType;
import com.mapr.fs.proto.Common.FileSubType;
import com.mapr.fs.proto.Common.FSKeyType;
import com.mapr.fs.proto.Common.Server;
import com.mapr.fs.proto.Common.MapRProgramId;
import com.mapr.fs.proto.Common.FileCompressionType;
import com.mapr.fs.proto.Common.IPAddress;
import com.mapr.fs.proto.Dbserver;
import com.mapr.fs.proto.Fileserver.DeleteAsyncEntry;
import com.mapr.fs.proto.Fileserver.KvMsg;
import com.mapr.fs.proto.Fileserver.KvList;
import com.mapr.fs.proto.Fileserver.KvStoreKey;
import com.mapr.fs.proto.Fileserver.GetattrRequest;
import com.mapr.fs.proto.Fileserver.GetattrResponse;
import com.mapr.fs.proto.Fileserver.ScanFileClustersRequest;
import com.mapr.fs.proto.Fileserver.ScanFileClustersResponse ;
import com.mapr.fs.proto.Fileserver.TabletRangeCheckRequest;
import com.mapr.fs.proto.Fileserver.TabletRangeCheckResponse;
import com.mapr.fs.proto.Fileserver.TabletRangeCheckResponse;
import com.mapr.fs.proto.Fileserver.KvStoreMultiOpDBRequest;
import com.mapr.fs.proto.Fileserver.KvStoreMultiOpDBResponse;
import com.mapr.fs.proto.Fileserver.FSProg;
import com.mapr.fs.proto.Dbserver.DBInternalDefaults;
import com.mapr.fs.proto.Dbserver.SplitDesc;
import com.mapr.fs.proto.Dbserver.TabletMapEntry;
import com.mapr.fs.proto.Dbserver.SegmentMapEntry;
import com.mapr.fs.proto.Dbserver.SpillMapEntry;
import com.mapr.fs.proto.Dbserver.TimeRange;
import com.mapr.fs.proto.Dbserver.SpillKeyEntry;
import com.mapr.fs.proto.Dbserver.PartitionMapEntry;
import com.mapr.fs.proto.Dbserver.PartitionMapEntry.BucketDesc;
import com.mapr.fs.proto.Dbserver.TabletCraftRequest;
import com.mapr.fs.proto.Dbserver.TabletCraftResponse;
import com.mapr.fs.proto.Dbserver.TestScanRequest;
import com.mapr.fs.proto.Dbserver.TestScanResponse;
import com.mapr.fs.proto.Dbserver.RawSpillScanRequest;
import com.mapr.fs.proto.Dbserver.RawSpillScanResponse;
import com.mapr.fs.proto.Dbserver.KeyMapEntry;
import com.mapr.fs.proto.Dbserver.KeyMapCookie;
import com.mapr.fs.proto.Dbserver.SchemaFamily;
import com.mapr.fs.proto.Dbserver.TableAttr;
import com.mapr.fs.proto.Dbserver.TableReplicaDesc;
import com.mapr.fs.proto.Dbserver.Qualifier;
import com.mapr.fs.proto.Dbserver.DBProg;
import com.mapr.fs.proto.Dbserver.TabletStatRequest;
import com.mapr.fs.proto.Dbserver.TabletStatResponse;
import com.mapr.fs.proto.Dbserver.GetPartitionSplitsRequest;
import com.mapr.fs.proto.Dbserver.GetPartitionSplitsResponse;
import com.mapr.fs.proto.Dbserver.SpaceUsage;
import com.mapr.fs.proto.Dbserver.TabletDesc;
import com.mapr.fs.proto.Dbserver.MergeDesc;
import com.mapr.fs.proto.Dbserver.CidVNEntry;
import com.mapr.fs.proto.Dbserver.ReplBucketDesc;
import com.mapr.fs.proto.Dbreplicator.DBReplicatorProg;
import com.mapr.fs.proto.Dbreplicator.GetHostNamesRequest;
import com.mapr.fs.proto.Dbreplicator.GetHostNamesResponse;
import com.mapr.fs.MapRFileSystem;
import com.mapr.fs.MapRFileStatus;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.BinaryString;

import org.apache.log4j.Logger;

import com.mapr.security.MaprSecurityException;
import com.mapr.cli.table.TabletStats;

public class DebugDbCommands extends CLIBaseClass implements CLIInterface {
  private static final String DEBUGDB_FID_PARAM_NAME = "fid";
  private static final String DEBUGDB_HOST_PARAM_NAME = "host";
  private static final String DEBUGDB_PORT_PARAM_NAME = "port";
  private static final String DEBUGDB_TABLE_PARAM_NAME = "path";
  private static final String DEBUGDB_CID_PARAM_NAME = "cid";
  private static final String DEBUGDB_FTYPE_PARAM_NAME = "ftype";
  private static final String DEBUGDB_FTYPE_PARAM_DESC =
    "table|tabletmap|tablet|segmap|spillmap|spill";
  private static final String DEBUGDB_FROMGFSCK_PARAM_NAME = "fromGfsck";
  private static final String DEBUGDB_GFSCKREPAIR_PARAM_NAME = "repair";
  private static final String DEBUGDB_STARTKEY_PARAM_NAME = "startkey";
  private static final String DEBUGDB_ENDKEY_PARAM_NAME = "endkey";
  private static final String DEBUGDB_MAXKEYS_PARAM_NAME = "maxkeys";
  private static final String DEBUGDB_INDEXOFFSET_PARAM_NAME = "idxoffset";
  private static final String DEBUGDB_INDEXSIZE_PARAM_NAME = "idxsize";
  private static final String DEBUGDB_DUMPFULLKEYS_PARAM_NAME = "dumpfullkeys";
  private static final String DEBUGDB_KEYIDXVERSION_PARAM_NAME = "keyidxversion";
  private static final String DEBUGDB_DIRFID_PARAM_NAME = "dirfid";
  private static final String DEBUGDB_TABLEFID_PARAM_NAME = "tablefid";
  private static final String DEBUGDB_TABLENM_PARAM_NAME = "tablename";
  private static final String DEBUGDB_NTABLETS_PARAM_NAME = "ntablets";
  private static final String DEBUGDB_IPLIST_PARAM_NAME = "iplist";
  private static final String DEBUGDB_IP_PARAM_NAME = "ip";
  private static final String DEBUGDB_TRACEFILE_PARAM_NAME = "tracefile";
  private static final String DEBUGDB_FROMFS_PARAM_NAME = "fromfileserverid";
  private static final String DEBUGDB_TOFS_PARAM_NAME = "tofileserverid";
  private static final String DEBUGDB_CTYPE_PARAM_NAME = "type";
  private static final String DEBUGDB_CTYPE_PARAM_DESC =
    "default|postsplit";
  private static final String DEBUGDB_DUMPFILE_PARAM_NAME = "dumpfile";
  private static final String DEBUGDB_KVFID_PARAM_NAME = "kvfid";
  private static final String DEBUGDB_KEYTYPE_PARAM_NAME = "keytype";
  private static final String DEBUGDB_DELKEYS_PARAM_NAME = "delkeys";
  private static final String DEBUGDB_DELFIDS_PARAM_NAME = "delfids";
  public static final String MULTI_ARG_SEP = ",";
  public static final String COLUMN_SEP = ":";
  // Ideally share with DbCommands.java - TODO (BB)
  private static final String TABLE_TYPE_DEF_BIN = "binary";
  private static final String TABLE_TYPE_JSON = "json";
  private static final Logger LOG = Logger.getLogger(DumpCommands.class);
  private static Pattern printableStringName =
                Pattern.compile("\\p{Print}+");
  // Usage
  static String statUsage = "stat -fid fidx";
  static String cdscanUsage = "cdscan -fid fid";
  static String bmapUsage = "bmap -fid fid";
  static String dumpUsage = "dump -fid fid";
  static String dumpTableUsage = "dump -path tablePath";
  static String switchMasterUsage = "switchMaster -cid cid";
  static String checkTabletUsage = "checkTablet -fid fid -startkey key1 -endkey key2 -tracefile file";
  static String statTabletUsage = "statTablet -fid fid";
  static String partitionSplitsUsage = "partitionSplits -fid fid";
  static String rawScanUsage = "rawScan -fid tabletfid -startkey key1 -maxkeys maxkeys -dumpfile file";
  static String multiOpUsage = "multiOp -kvfid kvfid -keytype <int|string> -delkeys <keyList> -delfids <fidList>";
  static String getHostNamesUsage = "getHostNames -host gatewayIP -port gatewayPort";

  private static final int DefaultGatewayPort = 7660;

  public static enum KVFormatType {
    TABLE,
    TABLET_MAP,
    SCHEMA_INFO,
    TABLET,
    SEGMENT_MAP,
    SPILL_MAP,
    SPILL,
    DEFER_MAP,
    GENERIC_KV,
    DIR,
    INVALID;
  }

  public static Map<String, BaseInputParameter> baseParams =
    new ImmutableMap.Builder<String, BaseInputParameter>()
    .put(MapRCliUtil.CLUSTER_NAME_PARAM, new TextInputParameter(
        MapRCliUtil.CLUSTER_NAME_PARAM, "cluster_name",
        CLIBaseClass.NOT_REQUIRED, null))
    .build();

  // Cmds
  static final CLICommand dump = new CLICommand(
      "dump",
      "usage : " + dumpUsage,
      DebugDbCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(DebugDbCommands.baseParams)
        .put(DebugDbCommands.DEBUGDB_FID_PARAM_NAME,
             new TextInputParameter(
                 DebugDbCommands.DEBUGDB_FID_PARAM_NAME,
                 DebugDbCommands.DEBUGDB_FID_PARAM_NAME,
                 CLIBaseClass.REQUIRED,
                 null))
        .put(DebugDbCommands.DEBUGDB_FTYPE_PARAM_NAME,
             new TextInputParameter(
                 DebugDbCommands.DEBUGDB_FTYPE_PARAM_NAME,
                 DebugDbCommands.DEBUGDB_FTYPE_PARAM_DESC,
                 CLIBaseClass.NOT_REQUIRED,
                 null).setInvisible(true))
        .put(DebugDbCommands.DEBUGDB_STARTKEY_PARAM_NAME,
             new TextInputParameter(
                 DebugDbCommands.DEBUGDB_STARTKEY_PARAM_NAME,
                 DebugDbCommands.DEBUGDB_STARTKEY_PARAM_NAME,
                 CLIBaseClass.NOT_REQUIRED,
                 null))
        .put(DebugDbCommands.DEBUGDB_ENDKEY_PARAM_NAME,
             new TextInputParameter(
                 DebugDbCommands.DEBUGDB_ENDKEY_PARAM_NAME,
                 DebugDbCommands.DEBUGDB_ENDKEY_PARAM_NAME,
                 CLIBaseClass.NOT_REQUIRED,
                 null))
        .put(DebugDbCommands.DEBUGDB_MAXKEYS_PARAM_NAME,
             new IntegerInputParameter(
                 DebugDbCommands.DEBUGDB_MAXKEYS_PARAM_NAME,
                 DebugDbCommands.DEBUGDB_MAXKEYS_PARAM_NAME,
                 CLIBaseClass.NOT_REQUIRED,
                 null))
        .put(DebugDbCommands.DEBUGDB_INDEXOFFSET_PARAM_NAME,
             new IntegerInputParameter(
                 DebugDbCommands.DEBUGDB_INDEXOFFSET_PARAM_NAME,
                 DebugDbCommands.DEBUGDB_INDEXOFFSET_PARAM_NAME,
                 CLIBaseClass.NOT_REQUIRED,
                 null))
        .put(DebugDbCommands.DEBUGDB_INDEXSIZE_PARAM_NAME,
             new IntegerInputParameter(
                 DebugDbCommands.DEBUGDB_INDEXSIZE_PARAM_NAME,
                 DebugDbCommands.DEBUGDB_INDEXSIZE_PARAM_NAME,
                 CLIBaseClass.NOT_REQUIRED,
                 null))
        .put(DebugDbCommands.DEBUGDB_DUMPFULLKEYS_PARAM_NAME,
             new BooleanInputParameter(
                 DebugDbCommands.DEBUGDB_DUMPFULLKEYS_PARAM_NAME,
                 DebugDbCommands.DEBUGDB_DUMPFULLKEYS_PARAM_NAME,
                 CLIBaseClass.NOT_REQUIRED,
                 null))
        .put(DebugDbCommands.DEBUGDB_KEYIDXVERSION_PARAM_NAME,
             new IntegerInputParameter(
                 DebugDbCommands.DEBUGDB_KEYIDXVERSION_PARAM_NAME,
                 DebugDbCommands.DEBUGDB_KEYIDXVERSION_PARAM_NAME,
                 CLIBaseClass.NOT_REQUIRED,
                 null))
        .build(),
       null
    ).setShortUsage(dumpUsage);

    // Cmds
  static final CLICommand dumpTable= new CLICommand(
      "dumpTable",
      "usage : " + dumpTableUsage,
      DebugDbCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(DebugDbCommands.baseParams)
        .put(DebugDbCommands.DEBUGDB_TABLE_PARAM_NAME,
             new TextInputParameter(
                 DebugDbCommands.DEBUGDB_TABLE_PARAM_NAME,
                 DebugDbCommands.DEBUGDB_TABLE_PARAM_NAME,
                 CLIBaseClass.REQUIRED,
                 null))
          .build(),
       null
    ).setShortUsage(dumpTableUsage)
     .setUsageInVisible(true);

  static final CLICommand rawScan= new CLICommand(
      "rawScan",
      "usage : " + rawScanUsage,
      DebugDbCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(DebugDbCommands.baseParams)
        .put(DebugDbCommands.DEBUGDB_FID_PARAM_NAME,
             new TextInputParameter(
                 DebugDbCommands.DEBUGDB_FID_PARAM_NAME,
                 DebugDbCommands.DEBUGDB_FID_PARAM_NAME,
                 CLIBaseClass.REQUIRED,
                 null))
        .put(DebugDbCommands.DEBUGDB_STARTKEY_PARAM_NAME,
             new TextInputParameter(
                 DebugDbCommands.DEBUGDB_STARTKEY_PARAM_NAME,
                 DebugDbCommands.DEBUGDB_STARTKEY_PARAM_NAME,
                 CLIBaseClass.REQUIRED,
                 null))
        .put(DebugDbCommands.DEBUGDB_MAXKEYS_PARAM_NAME,
             new IntegerInputParameter(
                 DebugDbCommands.DEBUGDB_MAXKEYS_PARAM_NAME,
                 DebugDbCommands.DEBUGDB_MAXKEYS_PARAM_NAME,
                 CLIBaseClass.NOT_REQUIRED,
                 1))
        .put(DebugDbCommands.DEBUGDB_DUMPFILE_PARAM_NAME,
             new TextInputParameter(
                 DebugDbCommands.DEBUGDB_DUMPFILE_PARAM_NAME,
                 DebugDbCommands.DEBUGDB_DUMPFILE_PARAM_NAME,
                 CLIBaseClass.REQUIRED,
                 null))
        .build(),
       null
    ).setShortUsage(rawScanUsage);


  static final CLICommand bmap= new CLICommand(
      "bmap",
      "usage : " + bmapUsage,
      DebugDbCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .put(DebugDbCommands.DEBUGDB_FID_PARAM_NAME,
             new TextInputParameter(
                 DebugDbCommands.DEBUGDB_FID_PARAM_NAME,
                 DebugDbCommands.DEBUGDB_FID_PARAM_NAME,
                 CLIBaseClass.REQUIRED,
                 null))
        .build(),
       null
    ).setShortUsage(bmapUsage);

  static final CLICommand cdscan = new CLICommand(
      "cdscan",
      "usage : " + cdscanUsage,
      DebugDbCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .put(DebugDbCommands.DEBUGDB_FID_PARAM_NAME,
             new TextInputParameter(
                 DebugDbCommands.DEBUGDB_FID_PARAM_NAME,
                 DebugDbCommands.DEBUGDB_FID_PARAM_NAME,
                 CLIBaseClass.REQUIRED,
                 null))
        .build(),
       null
    ).setShortUsage(cdscanUsage);

  static final CLICommand stat = new CLICommand(
      "stat",
      "usage : " + statUsage,
      DebugDbCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(DebugDbCommands.baseParams)
        .put(DebugDbCommands.DEBUGDB_FID_PARAM_NAME,
             new TextInputParameter(
                 DebugDbCommands.DEBUGDB_FID_PARAM_NAME,
                 DebugDbCommands.DEBUGDB_FID_PARAM_NAME,
                 CLIBaseClass.REQUIRED,
                 null))
        .build(),
       null
    ).setShortUsage(statUsage)
     .setUsageInVisible(true);

  static final CLICommand getHostNames = new CLICommand(
      "getHostNames",
      "usage : " + getHostNamesUsage,
      DebugDbCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(DebugDbCommands.baseParams)
        .put(DebugDbCommands.DEBUGDB_HOST_PARAM_NAME,
             new TextInputParameter(
                 DebugDbCommands.DEBUGDB_HOST_PARAM_NAME,
                 "gatewayHost/ip",
                 CLIBaseClass.REQUIRED,
                 null))
        .put(DebugDbCommands.DEBUGDB_PORT_PARAM_NAME,
             new IntegerInputParameter(
                 DebugDbCommands.DEBUGDB_PORT_PARAM_NAME,
                 "gatewayPort",
                 CLIBaseClass.NOT_REQUIRED,
                 DefaultGatewayPort))
        .build(),
       null
    ).setShortUsage(getHostNamesUsage)
     .setUsageInVisible(true);

  static final CLICommand multiOp = new CLICommand(
      "multiOp",
      "usage : " + multiOpUsage,
      DebugDbCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(DebugDbCommands.baseParams)
        .put(DebugDbCommands.DEBUGDB_KVFID_PARAM_NAME,
             new TextInputParameter(
                 DebugDbCommands.DEBUGDB_KVFID_PARAM_NAME,
                 DebugDbCommands.DEBUGDB_KVFID_PARAM_NAME,
                 CLIBaseClass.NOT_REQUIRED,
                 null))
        .put(DebugDbCommands.DEBUGDB_DELKEYS_PARAM_NAME,
             new TextInputParameter(
                 DebugDbCommands.DEBUGDB_DELKEYS_PARAM_NAME,
                 "keyList",
                 CLIBaseClass.NOT_REQUIRED,
                 null))
        .put(DebugDbCommands.DEBUGDB_KEYTYPE_PARAM_NAME,
             new TextInputParameter(
                 DebugDbCommands.DEBUGDB_KEYTYPE_PARAM_NAME,
                 "int|string",
                 CLIBaseClass.NOT_REQUIRED,
                 "int"))
        .put(DebugDbCommands.DEBUGDB_DELFIDS_PARAM_NAME,
             new TextInputParameter(
                 DebugDbCommands.DEBUGDB_DELFIDS_PARAM_NAME,
                 "fidList",
                 CLIBaseClass.NOT_REQUIRED,
                 null))
        .build(),
       null
    ).setShortUsage(multiOpUsage);

  static final CLICommand switchMaster = new CLICommand(
      "switchMaster",
      "usage : " + switchMasterUsage,
      DebugDbCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(DebugDbCommands.baseParams)
        .put(DebugDbCommands.DEBUGDB_CID_PARAM_NAME,
             new IntegerInputParameter(
                 DebugDbCommands.DEBUGDB_CID_PARAM_NAME,
                 DebugDbCommands.DEBUGDB_CID_PARAM_NAME,
                 CLIBaseClass.REQUIRED,
                 null))
        .build(),
       null
    ).setShortUsage(switchMasterUsage)
     .setUsageInVisible(true);

  static final CLICommand checkTablet = new CLICommand(
      "checkTablet",
      "usage : " + dumpUsage,
      DebugDbCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(DebugDbCommands.baseParams)
        .put(DebugDbCommands.DEBUGDB_FID_PARAM_NAME,
             new TextInputParameter(
                 DebugDbCommands.DEBUGDB_FID_PARAM_NAME,
                 DebugDbCommands.DEBUGDB_FID_PARAM_NAME,
                 CLIBaseClass.REQUIRED,
                 null))
        .put(DebugDbCommands.DEBUGDB_STARTKEY_PARAM_NAME,
             new TextInputParameter(
                 DebugDbCommands.DEBUGDB_STARTKEY_PARAM_NAME,
                 DebugDbCommands.DEBUGDB_STARTKEY_PARAM_NAME,
                 CLIBaseClass.NOT_REQUIRED,
                 null))
        .put(DebugDbCommands.DEBUGDB_ENDKEY_PARAM_NAME,
             new TextInputParameter(
                 DebugDbCommands.DEBUGDB_ENDKEY_PARAM_NAME,
                 DebugDbCommands.DEBUGDB_ENDKEY_PARAM_NAME,
                 CLIBaseClass.NOT_REQUIRED,
                 null))
        .put(DebugDbCommands.DEBUGDB_FROMGFSCK_PARAM_NAME,
             new BooleanInputParameter(
                 DebugDbCommands.DEBUGDB_FROMGFSCK_PARAM_NAME,
                 DebugDbCommands.DEBUGDB_FROMGFSCK_PARAM_NAME,
                 CLIBaseClass.NOT_REQUIRED,
                 null).setInvisible(true))
        .put(DebugDbCommands.DEBUGDB_GFSCKREPAIR_PARAM_NAME,
             new BooleanInputParameter(
                 DebugDbCommands.DEBUGDB_GFSCKREPAIR_PARAM_NAME,
                 DebugDbCommands.DEBUGDB_GFSCKREPAIR_PARAM_NAME,
                 CLIBaseClass.NOT_REQUIRED,
                 null).setInvisible(true))
        .put(DebugDbCommands.DEBUGDB_TRACEFILE_PARAM_NAME,
             new TextInputParameter(
                 DebugDbCommands.DEBUGDB_TRACEFILE_PARAM_NAME,
                 DebugDbCommands.DEBUGDB_TRACEFILE_PARAM_NAME,
                 CLIBaseClass.REQUIRED,
                 null))
        .build(),
       null
    ).setShortUsage(checkTabletUsage);

  static final CLICommand statTablet = new CLICommand(
      "statTablet",
      "usage : " + statTabletUsage,
      DebugDbCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(DebugDbCommands.baseParams)
        .put(DebugDbCommands.DEBUGDB_FID_PARAM_NAME,
             new TextInputParameter(
                 DebugDbCommands.DEBUGDB_FID_PARAM_NAME,
                 DebugDbCommands.DEBUGDB_FID_PARAM_NAME,
                 CLIBaseClass.REQUIRED,
                 null))
        .build(),
       null
    ).setShortUsage(statTabletUsage);

  static final CLICommand partitionSplits = new CLICommand(
      "partitionSplits",
      "usage : " + partitionSplitsUsage,
      DebugDbCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(DebugDbCommands.baseParams)
        .put(DebugDbCommands.DEBUGDB_FID_PARAM_NAME,
             new TextInputParameter(
                 DebugDbCommands.DEBUGDB_FID_PARAM_NAME,
                 DebugDbCommands.DEBUGDB_FID_PARAM_NAME,
                 CLIBaseClass.REQUIRED,
                 null))
        .build(),
       null
    ).setShortUsage(partitionSplitsUsage);

  public DebugDbCommands(ProcessedInput input, CLICommand cliCommand)
      throws CLIProcessingException {
    super(input, cliCommand);
  }

  String clusterName = null;

  void init() throws CLIProcessingException {
    try {
      int port = Rpc.initialize(0, 0, null); // Client
      if (port < 0)
        throw new IOException("Failed to initalize RPC");
    } catch (Exception e) {
        /**
        * <MAPR_ERROR>
        * Message:Exception in Rpc.initialize <error>
        * Function:DebugDbCommands.init()
        * Meaning:An error occurred: failed to Initialze RPC.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        LOG.error("Exception in Rpc.initialize " + e);
    }
  }

  public CommandOutput executeRealCommand() throws CLIProcessingException {
    init();
    if (!super.validateInput()) {
      return output;
    }

    if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
      clusterName = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0);
    } else {
      clusterName = CLDBRpcCommonUtils.getInstance().getCurrentClusterName();
    }

    try {
      if (cliCommand.getCommandName().equalsIgnoreCase("dump")) {
        return dump();
      } else if (cliCommand.getCommandName().equalsIgnoreCase("dumpTable")) {
        return dumpTable();
      } else if (cliCommand.getCommandName().equalsIgnoreCase("getHostNames")) {
        return getHostNames();
      } else if (cliCommand.getCommandName().equalsIgnoreCase("stat")) {
        return stat();
      } else if (cliCommand.getCommandName().equalsIgnoreCase("cdscan")) {
        return cdscan();
      } else if (cliCommand.getCommandName().equalsIgnoreCase("bmap")) {
        return bmap();
      } else if (cliCommand.getCommandName().equalsIgnoreCase("switchmaster")) {
        return switchMaster();
      } else if (cliCommand.getCommandName().equalsIgnoreCase("checkTablet")) {
        return checkTablet();
      } else if (cliCommand.getCommandName().equalsIgnoreCase("statTablet")) {
        return statTablet();
      } else if (cliCommand.getCommandName().equalsIgnoreCase("partitionSplits")) {
        return partitionSplits();
      } else if (cliCommand.getCommandName().equalsIgnoreCase("rawScan")) {
        return rawScan();
      } else if (cliCommand.getCommandName().equalsIgnoreCase("multiOp")) {
        return multiOp();
      } else {
        return new TextCommandOutput("DebugDb unknown sub-command".getBytes());
      }
    } catch (Exception e) {
      throw new CLIProcessingException("Send request Exception", e);
    }
  }

  // Convert string to fidmsg.
  private FidMsg stringToFid(String s) {
    String[] temp = s.split("\\.");

    if (temp.length != 3)
      return null;

    int cid = Integer.parseInt(temp[0]);
    int cinum = Integer.parseInt(temp[1]);
    int uniq = Integer.parseInt(temp[2]);
    return FidMsg.newBuilder()
                 .setCid(cid)
                 .setCinum(cinum)
                 .setUniq(uniq)
                 .build();
  }

  private String printableFid(FidMsg f) {
    String cidStr;

    if (f.getCid() == -1)
      cidStr = "<parentCID>" + ".";
    else
      cidStr = f.getCid() + ".";
    return (cidStr + f.getCinum() + "." + f.getUniq());
  }

  private String printableSubType(FileType type, FileSubType subtype) {
    String stype = subtype.toString();

    if (type == FileType.FTKvstore) {
      if (subtype == FileSubType.FSTKvTable)
        stype = "FSTKvTable";
      else if (subtype == FileSubType.FSTKvTabletMap)
        stype = "FSTKvTabletMap";
      else if (subtype == FileSubType.FSTKvSchema)
        stype = "FSTKvSchema";
      else if (subtype == FileSubType.FSTKvTablet)
        stype = "FSTKvTablet";
      else if (subtype == FileSubType.FSTKvSegMap)
        stype = "FSTKvSegMap";
      else if (subtype == FileSubType.FSTKvSpillMap)
        stype = "FSTKvSpillMap";
      else if (subtype == FileSubType.FSTKvKeyMap)
        stype = "FSTKvKeyMap";
    } else if (type == FileType.FTRegular) {
      if (subtype == FileSubType.FSTRegBucket)
        stype = "FSTRegBucket";
      else if (subtype == FileSubType.FSTRegSortedBucket)
        stype = "FSTRegSortedBucket";
      else if (subtype == FileSubType.FSTRegCF)
        stype = "FSTRegCF";
      else if (subtype == FileSubType.FSTRegSpill)
        stype = "FSTRegSpill";
    }
    return stype;
  }

  // constructs a byte array to a "pretty print" string.
  private String printableKey(final byte[] bytes) {
    return BinaryString.toStringBinary(bytes);
  }

  private String bytesToString(final ByteString bs) {
    return BinaryString.toStringBinary(bs.toByteArray());
  }

  // constructs a "pretty printed binary string" to its raw format.
  private ByteString binaryKeyToByteString(String raws) {
    byte[] bArr = BinaryString.toBytesBinary(raws);
    return ByteString.copyFrom(bArr);
  }

  private ByteString prepareVarKey(String raws) {
    return binaryKeyToByteString(raws);
  }

  private OutputNode formatDirEntry(MapRFileStatus d) {
    String childFid = d.getCid() + "." + d.getCinum() + "." + d.getUniq();
    String s = d.getPath().toString();
    int l = s.lastIndexOf('/');

    OutputNode dout = new OutputNode();
    dout.addChild(new OutputNode("name", s.substring(l+1)));
    dout.addChild(new OutputNode("fid", childFid));
    dout.addChild(new OutputNode("isDir", d.isDir() ? "true" : "false"));
    return dout;
  }

  private OutputNode formatKeyMapEntry(KeyMapEntry ent) {
    OutputNode dout = new OutputNode();
    ByteString bs = null;

    dout.addChild(new OutputNode("shared", ent.getSharedLen()));
    dout.addChild(new OutputNode("nonshared", ent.getNonsharedLen()));
    dout.addChild(new OutputNode("value", ent.getValueLen()));

    if (ent.hasBlkOff()) {
      dout.addChild(new OutputNode("childblockoffset", ent.getBlkOff()));
    }
    if (ent.hasBlkSize()) {
      dout.addChild(new OutputNode("childblocksize", ent.getBlkSize()));
    }

    if (!ent.hasBlkOff() && !ent.hasBlkSize()) {
      try {
        SpillKeyEntry spillVal;

        if (ent.hasKvalue())
          spillVal = ent.getKvalue();
        else
          spillVal = SpillKeyEntry.parseFrom(ent.getValue());

        if (spillVal.hasInlineValue()) {
          dout.addChild(new OutputNode("inlineValue",
                        printableKey(spillVal.getInlineValue().toByteArray())));
          dout.addChild(new OutputNode("inlineValueSz",
                        spillVal.getInlineValue().size()));
        }

        int cnt = spillVal.getValuesCount();
        for (int index = 0; index < cnt; ++index) {
          OutputNode valueout = new OutputNode("familyData");
          boolean addFamily = false;

          SpillKeyEntry.FamilyValue fv = spillVal.getValues(index);
          if (fv.hasId()) {
            valueout.addChild(new OutputNode("id", fv.getId()));
            addFamily = true;
          }
          if (fv.hasOffset()) {
            valueout.addChild(new OutputNode("offset", fv.getOffset()));
            addFamily = true;
          }
          if (fv.hasLength()) {
            valueout.addChild(new OutputNode("length", fv.getLength()));
            addFamily = true;
          }
          if (fv.hasInlined()) {
            valueout.addChild(new OutputNode("inlined", fv.getInlined()));
            addFamily = true;
          }

          if (addFamily )
            dout.addChild(valueout);
        }
      } catch (Throwable e) {
        dout.addChild(new OutputNode("error", "PARSE ERROR"));
      }
    }

    dout.addChild(new OutputNode("key",
                  printableKey(ent.getKey().toByteArray())));
    dout.addChild(new OutputNode("value",
                  printableKey(ent.getValue().toByteArray())));

    return dout;
  }

  private OutputNode formatKV(KVFormatType type, KvMsg kv) {
    OutputNode dout = new OutputNode();
    String ks = null;

    // Add key.
    switch (type) {
    case TABLE:
    case SCHEMA_INFO:
    case TABLET:
    case SEGMENT_MAP:
    case TABLET_MAP:
    case DEFER_MAP:
    case SPILL:
    case GENERIC_KV:
      ByteString bs = kv.getKey().getVarKey();
      dout.addChild(new OutputNode("key", printableKey(bs.toByteArray())));
      ks = bs.toStringUtf8();
      break;

    case SPILL_MAP:
      dout.addChild(new OutputNode("key", kv.getKey().getIntKey()));
      break;

    default:
      dout.addChild(new OutputNode("key", "UNKNOWN"));
      break;
    }

    // Add value
    OutputNode vout = new OutputNode();
    switch (type) {
    case TABLE:
      try {
        FidMsg f = FidMsg.parseFrom(kv.getValue());
        vout.addChild(new OutputNode("fid", printableFid(f)));
      } catch (Throwable e) {
        vout.addChild(new OutputNode("fid", "UNKNOWN"));
      }
      break;

    case SCHEMA_INFO:
      try {
        DBInternalDefaults dbi = DBInternalDefaults.getDefaultInstance();
        if (ks == null) {
          vout.addChild(new OutputNode("raw", "UNKNOWN"));
        } else if (ks.startsWith(dbi.getColFamilyIdPrefix())) {
          SchemaFamily sf = SchemaFamily.parseFrom(kv.getValue());
          if (sf.hasName())
            vout.addChild(new OutputNode("cfname", sf.getName()));
          else
            vout.addChild(new OutputNode("cfname", "NONE"));
        } else if (ks.equals(dbi.getAttr())) {
          TableAttr attr = TableAttr.parseFrom(kv.getValue());
          if (attr.hasAutoSplit())
            vout.addChild(new OutputNode("attr:autoSplit",
                                         attr.getAutoSplit()));
          if (attr.hasBulkLoad())
            vout.addChild(new OutputNode("attr:bulkLoad",
                                         attr.getBulkLoad()));
          if (attr.hasDeleteTTL())
            vout.addChild(new OutputNode("attr:deleteTTL",
                                         attr.getDeleteTTL()));
          if (attr.hasSyncReplTimeoutMillis())
            vout.addChild(new OutputNode("attr:syncReplTimeoutMillis",
                                         attr.getSyncReplTimeoutMillis()));
          if (attr.hasJson()) {
            String tblType = (attr.getJson() ? TABLE_TYPE_JSON :
                              TABLE_TYPE_DEF_BIN);
            dout.addChild(new OutputNode("attr:tableType", tblType));
          }
          if (attr.hasRegionSizeMB())
            vout.addChild(new OutputNode("attr:regionSizeMB",
                                         attr.getRegionSizeMB()));
          if (attr.hasMaxValueSzInMemIndex())
            vout.addChild(new OutputNode("attr:maxValueSzInMemIndex",
                                         attr.getMaxValueSzInMemIndex()));
          if (attr.hasReclaimThreshPcntForPack())
            vout.addChild(new OutputNode("attr:reclaimThreshPcntForPack",
                                         attr.getReclaimThreshPcntForPack()));
          if (attr.hasDropLargeRows())
            vout.addChild(new OutputNode("attr:dropLargeRows",
                                         attr.getDropLargeRows()));
          if (attr.hasTtlCompaction())
            vout.addChild(new OutputNode("attr:ttlCompaction",
                                         attr.getTtlCompaction()));
          if (attr.hasTtlCompactionHrs())
            vout.addChild(new OutputNode("attr:ttlCompactionHrs",
                                         attr.getTtlCompactionHrs()));
        } else if (ks.startsWith(dbi.getReplIdxPrefix())) {
          TableReplicaDesc rd = TableReplicaDesc.parseFrom(kv.getValue());
          vout.addChild(new OutputNode("cluster", rd.getClusterName()));
          vout.addChild(new OutputNode("table", rd.getTablePath()));

          byte[] uuid = rd.getTableUuid().toByteArray();
          vout.addChild(new OutputNode("uuid",
                                       BinaryString.toUUIDString(uuid)));
          vout.addChild(new OutputNode("paused", rd.getIsPaused()));
          vout.addChild(new OutputNode("synchronous", rd.getSynchronous()));
          vout.addChild(new OutputNode("idx", rd.getIdx()));

          if (rd.getQualifiersCount() > 0) {
            String famList = "";
            int idx = 0;

            for (Qualifier qual : rd.getQualifiersList()) {
              if (qual.getQualifiersCount() > 0) {
                for (ByteString bstr : qual.getQualifiersList()) {
                  if (idx == 0) {
                    famList = qual.getFamily() + COLUMN_SEP + bstr.toStringUtf8();
                  } else {
                    famList = famList + "," + qual.getFamily() + COLUMN_SEP + bstr.toStringUtf8();
                  }
                  ++idx;
                }
              } else if (idx == 0) {
                famList = String.valueOf(qual.getFamily());
                ++idx;
              } else {
                famList = famList + "," + qual.getFamily();
                ++idx;
              }
            }
            vout.addChild(new OutputNode("columnfamilies", famList));
          }
        } else {
          vout.addChild(new OutputNode("raw", bytesToString(kv.getValue())));
        }
      } catch (Throwable e) {
        vout.addChild(new OutputNode("raw", "UNKNOWN"));
      }
      break;

    case TABLET:
      try {
        DBInternalDefaults dbi = DBInternalDefaults.getDefaultInstance();
        if (ks == null) {
          vout.addChild(new OutputNode("raw", "UNKNOWN"));

        } else if (ks.startsWith(dbi.getTabletKeyForPMap())) {
          PartitionMapEntry p = PartitionMapEntry.parseFrom(kv.getValue());
          if (p.hasSegmapFid())
            vout.addChild(new OutputNode("segfid",
                                         printableFid(p.getSegmapFid())));
          else
            vout.addChild(new OutputNode("segfid", "NONE"));

          // Add bucket list
          if (p.getBucketFidsList().size() > 0) {
            OutputNode buckets = new OutputNode("bucketfids");
            vout.addChild(buckets);
            for (FidMsg f : p.getBucketFidsList())
              buckets.addChild(new OutputNode("fid", printableFid(f)));
          }

          // Add frozen bit
          vout.addChild(new OutputNode("isFrozen", p.getIsFrozen()));

          // Add CidVN list
          if (p.getCidVNEntriesList().size() > 0) {
            OutputNode cidVNs = new OutputNode("CidVNs");
            vout.addChild(cidVNs);

            for (CidVNEntry ce : p.getCidVNEntriesList()) {
              cidVNs.addChild(new OutputNode("cid-minVN",
                                            ce.getCid() + "-" + ce.getMinVN()));
            }
          }

          // Add split state
          vout.addChild(new OutputNode("inSplit", p.getInSplit()));

          if (p.hasInImportBucket())
            dout.addChild(new OutputNode("inImportBucket",
                                         p.getInImportBucket()));

          // Add bucketDesc list
          if (p.getBucketDescsList().size() > 0) {
            OutputNode buckets = new OutputNode("bucketdescs");
            vout.addChild(buckets);
            for (PartitionMapEntry.BucketDesc bd : p.getBucketDescsList()) {
              buckets.addChild(new OutputNode("fid",
                                              printableFid(bd.getFid())));
              buckets.addChild(new OutputNode("needsRepl",
                                              bd.getNeedsRepl()));
              // buckets.addChild(new OutputNode("flushDone",
              //                                 bd.getFlushDone()));
            }
          }
          vout.addChild(new OutputNode("useBucketDesc", p.getUseBucketDesc()));

          if (p.hasLastFlushedBucketFid())
            vout.addChild(new OutputNode("lastFlushedBucketFid",
                                         printableFid(p.getLastFlushedBucketFid())));

          // Add space usage stats
          vout.addChild(new OutputNode("numLogicalBlocks",
                                       p.getUsage().getNumLogicalBlocks()));
          vout.addChild(new OutputNode("numPhysicalBlocks",
                                       p.getUsage().getNumPhysicalBlocks()));
          vout.addChild(new OutputNode("numRows", p.getUsage().getNumRows()));
          if (p.getUsage().hasNumRowsWithDelete())
            vout.addChild(new OutputNode("numRowsWithDelete",
                                         p.getUsage().getNumRowsWithDelete()));
          if (p.getUsage().hasNumRemoteBlocks())
            vout.addChild(new OutputNode("numRemoteBlocks",
                                         p.getUsage().getNumRemoteBlocks()));
          if (p.getUsage().hasNumSpills())
            vout.addChild(new OutputNode("numSpills",
                                         p.getUsage().getNumSpills()));
          if (p.getUsage().hasNumSegments())
            vout.addChild(new OutputNode("numSegments",
                                         p.getUsage().getNumSegments()));
        } else if (ks.startsWith(dbi.getTabletKeyForSplit())) {
          SplitDesc sd = SplitDesc.parseFrom(kv.getValue());
          vout.addChild(new OutputNode("fid", printableFid(sd.getDstFid())));
          vout.addChild(new OutputNode("moveRightHalf", sd.getMoveRightHalf()));
          vout.addChild(new OutputNode("endGame", sd.getEndGame()));

        } else if (ks.startsWith(dbi.getTabletKeyForStartKey())) {
          // no value to print in this case

        } else if (ks.startsWith(dbi.getTabletKeyForEndKey())) {
          // no value to print in this case

        } else if (ks.startsWith(dbi.getTabletKeyForMerge())) {
          MergeDesc md = MergeDesc.parseFrom(kv.getValue());
          vout.addChild(new OutputNode("peerFid",
                                       printableFid(md.getPeerFid())));
          vout.addChild(new OutputNode("isDest", md.getIsDest()));
        } else {
          vout.addChild(new OutputNode("raw", bytesToString(kv.getValue())));
        }
      } catch (Throwable e) {
        vout.addChild(new OutputNode("raw", "UNKNOWN"));
      }
      break;

    case TABLET_MAP:
      try {
        TabletMapEntry t = TabletMapEntry.parseFrom(kv.getValue());
        vout.addChild(new OutputNode("fid", printableFid(t.getTabletFid())));
      } catch (Throwable e) {
        vout.addChild(new OutputNode("fid", "UNKNOWN"));
      }
      break;

    case SEGMENT_MAP:
      try {
        SegmentMapEntry s = SegmentMapEntry.parseFrom(kv.getValue());
        vout.addChild(new OutputNode("fid", printableFid(s.getFid())));
      } catch (Throwable e) {
        vout.addChild(new OutputNode("fid", "UNKNOWN"));
      }
      break;

    case SPILL_MAP:
      try {
        SpillMapEntry s = SpillMapEntry.parseFrom(kv.getValue());
        vout.addChild(new OutputNode("fid", printableFid(s.getFid())));
        if (s.hasSmeSize())
          vout.addChild(new OutputNode("smeSize", s.getSmeSize()));
        if (s.hasMinVN())
          vout.addChild(new OutputNode("minVN", s.getMinVN()));
        vout.addChild(new OutputNode("keyIdxOffset", s.getKeyIdxOffset()));
        vout.addChild(new OutputNode("keyIdxLength", s.getKeyIdxLength()));
        vout.addChild(new OutputNode("ldbIdxLength", s.getLdbIdxLength()));
        vout.addChild(new OutputNode("bloomBitsPerKey",
                                     s.getBloomBitsPerKey()));
        vout.addChild(new OutputNode("numLogicalBlocks",
                                     s.getUsage().getNumLogicalBlocks()));
        vout.addChild(new OutputNode("numPhysicalBlocks",
                                     s.getUsage().getNumPhysicalBlocks()));
        vout.addChild(new OutputNode("numRows", s.getUsage().getNumRows()));
        if (s.getUsage().hasNumRowsWithDelete())
          vout.addChild(new OutputNode("numRowsWithDelete",
                                       s.getUsage().getNumRowsWithDelete()));
        if (s.getUsage().hasNumRemoteBlocks())
          dout.addChild(new OutputNode("numRemoteBlocks",
                                       s.getUsage().getNumRemoteBlocks()));
        if (s.getUsage().hasNumSpills())
          dout.addChild(new OutputNode("numSpills",
                                       s.getUsage().getNumSpills()));
        if (s.getUsage().hasNumSegments())
          dout.addChild(new OutputNode("numSegments",
                                       s.getUsage().getNumSegments()));

        OutputNode families = new OutputNode("families");
        vout.addChild(families);
        for (SpillMapEntry.FamilyEntry fe : s.getFamiliesList()) {
          families.addChild(new OutputNode("id", fe.getId()));
          if (fe.hasOffset())
            families.addChild(new OutputNode("offset", fe.getOffset()));
          if (fe.hasLength())
            families.addChild(new OutputNode("length", fe.getLength()));
          if (fe.hasTimeRange()) {
            TimeRange tr = fe.getTimeRange();
            if (tr.hasMinTS())
              families.addChild(new OutputNode("minTimeStamp", tr.getMinTS()));
            if (tr.hasMaxTS())
              families.addChild(new OutputNode("maxTimeStamp", tr.getMaxTS()));
          }
        }
      } catch (Throwable e) {
        vout.addChild(new OutputNode("keyfid", "UNKNOWN"));
      }
      break;

    case DEFER_MAP:
      try {
        ReplBucketDesc b = ReplBucketDesc.parseFrom(kv.getValue());
        vout.addChild(new OutputNode("bucketFid",
                                     printableFid(b.getBucketFid())));
        vout.addChild(new OutputNode("tableFid",
                                     printableFid(b.getTableFid())));
        if (b.getSendAfterList().size() != 0) {
          OutputNode sendAfter = new OutputNode("sendAfter");
          vout.addChild(sendAfter);
          for (FidMsg f : b.getSendAfterList())
            sendAfter.addChild(new OutputNode("fid", printableFid(f)));
        }
        if (b.getSendBeforeList().size() != 0) {
          OutputNode sendBefore = new OutputNode("sendBefore");
          vout.addChild(sendBefore);
          for (FidMsg f : b.getSendBeforeList())
            sendBefore.addChild(new OutputNode("fid", printableFid(f)));
        }

        vout.addChild(new OutputNode("flushed", b.getFlushed()));

        OutputNode replicas = new OutputNode("replicas");
        vout.addChild(replicas);
        for (ReplBucketDesc.ReplicaInfo rs : b.getReplList()) {
          replicas.addChild(new OutputNode("replicaIdx", rs.getReplicaIdx()));
          if (rs.hasDoneTillOffset())
            replicas.addChild(new OutputNode("doneTillOffset",
                                             rs.getDoneTillOffset()));
          if (rs.hasDone())
            replicas.addChild(new OutputNode("done", rs.getDone()));
        }
      } catch (Throwable e) {
        vout.addChild(new OutputNode("fid", "UNKNOWN"));
      }
      break;

    default:
      vout.addChild(new OutputNode("raw", "UNKNOWN"));
      break;
    }
    dout.addChild(new OutputNode("value", vout));
    return dout;
  }

  private void dumpDir(FidMsg fid,
                       OutputHierarchy out) throws CLIProcessingException {
    MapRFileSystem fs = MapRCliUtil.getMapRFileSystem();
    MapRFileStatus dirents[];

    try {
      dirents = fs.scanDir(clusterName, "/.mapr::fid::" + printableFid(fid));
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, "scanDir failed"));
      return;
    }

    if (dirents == null)
      return;

    for (int i = 0; i < dirents.length; ++i)
      out.addNode(formatDirEntry(dirents[i]));
  }

  private void dumpRegKeyMap(
                       FidMsg fid,
                       Integer idxoffset,
                       Integer idxsize,
                       boolean dumpFullKeys,
                       int fmtVersion,
                       OutputHierarchy out) throws CLIProcessingException {
    long dbinding = getBindingForContainer(fid.getCid());
    TestScanResponse resp;
    KeyMapCookie cookie = null;

    LOG.info("Getting regular key-map with idxsize " + idxsize +
             " dumpFullKeys " + dumpFullKeys +
             " keyIdxVersion " + fmtVersion);
    if (dbinding == -1) {
      out.addError(new OutputError(Errno.ERPCFAILED,
                                   "container lookup failed"));
      return;
    }

    while (true) {
      TestScanRequest req;
      if (cookie == null) {
        req = TestScanRequest.newBuilder()
                             .setCmd(TestScanRequest.Cmd.KeyMapTable)
                             .setFid(fid)
                             .setOffset(idxoffset)
                             .setSize(idxsize)
                             .setCreds(getUserCredentials())
                             .setDumpFullKeys(dumpFullKeys)
                             .setKeyIdxFmtVersion(fmtVersion)
                             .build();
      } else {
        req = TestScanRequest.newBuilder()
                             .setCmd(TestScanRequest.Cmd.KeyMapTable)
                             .setFid(fid)
                             .setOffset(idxoffset)
                             .setSize(idxsize)
                             .setCreds(getUserCredentials())
                             .setKmapCookie(cookie)
                             .setDumpFullKeys(dumpFullKeys)
                             .setKeyIdxFmtVersion(fmtVersion)
                             .build();
      }

      try {
        byte[] replyData;
        replyData = Rpc.sendRequest(dbinding,
                              MapRProgramId.DBServerProgramId.getNumber(),
                              DBProg.TestScanProc.getNumber(), req);
        if (replyData == null) {
          out.addError(new OutputError(Errno.ERPCFAILED,
                                       "testscan rpc failed"));
          return;
        }

        resp = TestScanResponse.parseFrom(replyData);
        if (resp.getStatus() != 0) {
          out.addError(new OutputError(resp.getStatus(),
                       "TestScan failed, Error : " +
                       Errno.toString(resp.getStatus())));
          return;
        }

        for (KeyMapEntry kment : resp.getKmapentriesList()) {
          out.addNode(formatKeyMapEntry(kment));
        }
      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);
      } catch (Exception e) {
        LOG.error("Exception processing testscan command");
        out.addError(new OutputError(Errno.ERPCFAILED, "mfs rpc failed"));
        return;
      }

      if (!resp.getHasMoreKeys()) {
        LOG.info("Done with scan");
        break;
      }
      LOG.info("Setting cookie as " + resp.getKmapCookie().getOffset());
      cookie = resp.getKmapCookie();
    }
  }

  private void dumpKeyValues(FidMsg fid,
                             KVFormatType type,
                             KvStoreKey start,
                             KvStoreKey end,
                             int maxkeys,
                             OutputHierarchy out) throws CLIProcessingException
  {
    MapRFileSystem fs = MapRCliUtil.getMapRFileSystem();
    byte[] byteArr;

    try {
      // serialize the start and end key.
      byte[] kstart = null;
      if (start != null)
        kstart = start.toByteArray();

      byte[] kend = null;
      if (end != null)
        kend = end.toByteArray();

      byteArr = fs.scanKV(clusterName, "/.mapr::fid::" + printableFid(fid),
                          kstart, kend, maxkeys);
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, "scanKV failed"));
      return;
    }

    if (byteArr == null)
      return;

    try {
      KvList kvlist = KvList.parseFrom(byteArr);
      for (KvMsg kvp : kvlist.getEntriesList())
        out.addNode(formatKV(type, kvp));
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED,
                                   "failed to parse key-value list"));
      return;
    }
  }

  private CommandOutput getDumpFromFid(FidMsg fid, OutputHierarchy out) throws CLIProcessingException {

    String cluster = null;
    if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
      cluster = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0);
    }

    KVFormatType ftype = KVFormatType.INVALID;
    if (isParamPresent(DEBUGDB_FTYPE_PARAM_NAME)) {
      String ftypestr = getParamTextValue(DEBUGDB_FTYPE_PARAM_NAME, 0);
      if (ftypestr.equalsIgnoreCase("table")) {
        ftype = KVFormatType.TABLE;
      } else if (ftypestr.equalsIgnoreCase("tabletmap")) {
        ftype = KVFormatType.TABLET_MAP;
      } else if (ftypestr.equalsIgnoreCase("tablet")) {
        ftype = KVFormatType.TABLET;
      } else if (ftypestr.equalsIgnoreCase("segmap")) {
        ftype = KVFormatType.SEGMENT_MAP;
      } else if (ftypestr.equalsIgnoreCase("spillmap")) {
        ftype = KVFormatType.SPILL_MAP;
      } else if (ftypestr.equalsIgnoreCase("spill")) {
        ftype = KVFormatType.SPILL;
      } else if (ftypestr.equalsIgnoreCase("dir")) {
        ftype = KVFormatType.DIR;
      } else if (ftypestr.equalsIgnoreCase("defermap")) {
        ftype = KVFormatType.DEFER_MAP;
      }
      if (ftype == KVFormatType.INVALID) {
        out.addError(new OutputError(Errno.EINVAL,"Invalid ftype " + ftypestr));
        output.setOutput(out);
        return output;
      }
    } else {
      // lookup container in cldb
      long dbinding = getBindingForContainer(fid.getCid());
      GetattrResponse resp;

      if (dbinding == -1) {
        out.addError(new OutputError(Errno.ERPCFAILED,
                                     "container lookup failed"));
        return output;
      }

      // Send GetAttr to mfs
      GetattrRequest req = GetattrRequest.newBuilder()
                                         .setNode(fid)
                                         .setCreds(getUserCredentials())
                                         .build();
      try {
        byte[] replyData;
        replyData = Rpc.sendRequest(dbinding,
                                  MapRProgramId.FileServerProgramId.getNumber(),
                                  FSProg.GetattrProc.getNumber(),
                                  req);
        if (replyData == null) {
          out.addError(new OutputError(Errno.ERPCFAILED, "getattr rpc failed"));
          return output;
        }

        resp = GetattrResponse.parseFrom(replyData);
        if (resp.getStatus() != 0) {
          out.addError(new OutputError(resp.getStatus(),
                       "GetAttr failed, Error : " +
                       Errno.toString(resp.getStatus())));
          return output;
        }
      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);
      } catch (Exception e) {
        LOG.error("Exception processing stat command");
        out.addError(new OutputError(Errno.ERPCFAILED, "mfs rpc failed"));
        return output;
      }

      FileType itype = resp.getAttr().getType();
      FileSubType subtype = resp.getAttr().getSubtype();
      if (itype == FileType.FTDirectory) {
        ftype = KVFormatType.DIR;
      } else if (itype == FileType.FTRegular) {
        if (subtype == FileSubType.FSTRegSpill) {
          ftype = KVFormatType.SPILL;
        } else {
          out.addError(new OutputError(Errno.EINVAL,
                                       "Cannot dump files of itype " + itype +
                                       " subtype " + subtype));
          return output;
        }
      } else if (itype == FileType.FTKvstore) {
        if (subtype == FileSubType.FSTKvTable) {
          ftype = KVFormatType.TABLE;
        } else if (subtype == FileSubType.FSTKvTabletMap) {
          ftype = KVFormatType.TABLET_MAP;
        } else if (subtype == FileSubType.FSTKvSchema) {
          ftype = KVFormatType.SCHEMA_INFO;
        } else if (subtype == FileSubType.FSTKvTablet) {
          ftype = KVFormatType.TABLET;
        } else if (subtype == FileSubType.FSTKvSegMap) {
          ftype = KVFormatType.SEGMENT_MAP;
        } else if (subtype == FileSubType.FSTKvSpillMap) {
          ftype = KVFormatType.SPILL_MAP;
        } else if (fid.getCinum() == 22) {
          ftype = KVFormatType.DEFER_MAP;
        } else {
          ftype = KVFormatType.GENERIC_KV;
        }
      } else {
        out.addError(new OutputError(Errno.EINVAL, "Cannot dump files of itype "
                                     + itype));
        return output;
      }
    }

    KvStoreKey kstart = null;
    if (isParamPresent(DEBUGDB_STARTKEY_PARAM_NAME)) {
      String kstartstr = getParamTextValue(DEBUGDB_STARTKEY_PARAM_NAME, 0);

      if (ftype == KVFormatType.SPILL_MAP) {
        kstart = KvStoreKey.newBuilder()
                           .setType(FSKeyType.UintKey)
                           .setIntKey(Integer.parseInt(kstartstr))
                           .build();
      } else {
        ByteString bstr = prepareVarKey(kstartstr);
        if (bstr == null) {
          out.addError(new OutputError(Errno.EINVAL,
                                       "Invalid startkey " + kstartstr));
          output.setOutput(out);
          return output;
        }

        kstart = KvStoreKey.newBuilder()
                           .setType(FSKeyType.VarKey)
                           .setVarKey(bstr)
                           .build();
      }
    }

    KvStoreKey kend = null;
    if (isParamPresent(DEBUGDB_ENDKEY_PARAM_NAME)) {
      String kendstr = getParamTextValue(DEBUGDB_ENDKEY_PARAM_NAME, 0);

      if (ftype == KVFormatType.SPILL_MAP) {
        kend = KvStoreKey.newBuilder()
                           .setType(FSKeyType.UintKey)
                           .setIntKey(Integer.parseInt(kendstr))
                           .build();
      } else {
        ByteString bstr = prepareVarKey(kendstr);
        if (bstr == null) {
          out.addError(new OutputError(Errno.EINVAL,
                                       "Invalid endkey " + kendstr));
          output.setOutput(out);
          return output;
        }

        kend = KvStoreKey.newBuilder()
                           .setType(FSKeyType.VarKey)
                           .setVarKey(bstr)
                           .build();
      }
    }

    int maxkeys = 0;
    if (isParamPresent(DEBUGDB_MAXKEYS_PARAM_NAME))
      maxkeys = getParamIntValue(DEBUGDB_MAXKEYS_PARAM_NAME, 0);

    int idxoffset = 0;
    if (isParamPresent(DEBUGDB_INDEXOFFSET_PARAM_NAME)) {
      idxoffset = getParamIntValue(DEBUGDB_INDEXOFFSET_PARAM_NAME, 0);
    } else if (ftype == KVFormatType.SPILL) {
      out.addError(new OutputError(Errno.EINVAL,
                                   "Dump for Spill requires param " +
                                   DEBUGDB_INDEXOFFSET_PARAM_NAME));
      output.setOutput(out);
      return output;
    }

    int idxsize = 0;
    if (isParamPresent(DEBUGDB_INDEXSIZE_PARAM_NAME)) {
      idxsize = getParamIntValue(DEBUGDB_INDEXSIZE_PARAM_NAME, 0);
    } else if (ftype == KVFormatType.SPILL) {
      out.addError(new OutputError(Errno.EINVAL,
                                   "Dump for Spill requires param " +
                                   DEBUGDB_INDEXSIZE_PARAM_NAME));
      output.setOutput(out);
      return output;
    }

    if (ftype == KVFormatType.DIR) {
      dumpDir(fid, out);
    } else if (ftype == KVFormatType.SPILL) {
      boolean dumpFullKeys = false;
      int idxVersion = 1;
      if (isParamPresent(DEBUGDB_DUMPFULLKEYS_PARAM_NAME)) {
        dumpFullKeys = getParamBooleanValue(DEBUGDB_DUMPFULLKEYS_PARAM_NAME, 0);
      }
      if (isParamPresent(DEBUGDB_KEYIDXVERSION_PARAM_NAME)) {
        idxVersion = getParamIntValue(DEBUGDB_KEYIDXVERSION_PARAM_NAME, 0);
      }
      dumpRegKeyMap(fid, idxoffset, idxsize, dumpFullKeys, idxVersion,
                    out);
    } else {
      dumpKeyValues(fid, ftype, kstart, kend, maxkeys, out);
    }
    return output;

  }

  private CommandOutput dump() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    String fidstr = getParamTextValue(DEBUGDB_FID_PARAM_NAME, 0);
    FidMsg fid = stringToFid(fidstr);
    if (fid == null) {
      out.addError(new OutputError(Errno.EINVAL, "Invalid fid " + fidstr));
      output.setOutput(out);
      return output;
    }

    getDumpFromFid(fid, out);

    return output;
  }

  private void RawScanFromFid(FidMsg fid,
                              String dumpFile,
                              ByteString bstr,
                              int maxkeys,
                              OutputHierarchy out) throws CLIProcessingException {

    RawSpillScanRequest req;
    RawSpillScanResponse resp;
    long dbinding = getBindingForContainer(fid.getCid());

    if (dbinding == -1) {
      out.addError(new OutputError(Errno.ERPCFAILED,
                                   "container lookup failed"));
      return;
    }

    req = RawSpillScanRequest.newBuilder()
                         .setFid(fid)
                         .setDumpFile(dumpFile)
                         .setStartKey(bstr)
                         .setMaxKeys(maxkeys)
                         .setCreds(getUserCredentials())
                         .build();

    try {
      byte[] replyData;
      replyData = Rpc.sendRequest(dbinding,
                            MapRProgramId.DBServerProgramId.getNumber(),
                            DBProg.RawSpillScanProc.getNumber(), req);
      if (replyData == null) {
        out.addError(new OutputError(Errno.ERPCFAILED,
                                     "testspillscan rpc failed"));
        return;
      }

      resp = RawSpillScanResponse.parseFrom(replyData);
      if (resp.getStatus() != 0) {
        out.addError(new OutputError(resp.getStatus(),
                     "RawSpillScan failed, Error : " +
                     Errno.toString(resp.getStatus())));
        return;
      }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      LOG.error("Exception processing testspillscan command");
      out.addError(new OutputError(Errno.ERPCFAILED, "mfs rpc failed"));
      return;
    }

    return;
  }

  private CommandOutput rawScan() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    // Fid
    String fidstr = getParamTextValue(DEBUGDB_FID_PARAM_NAME, 0);
    FidMsg fid = stringToFid(fidstr);
    if (fid == null) {
      out.addError(new OutputError(Errno.EINVAL, "Invalid fid " + fidstr));
      output.setOutput(out);
      return output;
    }

    // DumpFile
    String dumpFile = getParamTextValue(DEBUGDB_DUMPFILE_PARAM_NAME, 0);

    // StartKey
    ByteString bstr = ByteString.EMPTY;
    if (isParamPresent(DEBUGDB_STARTKEY_PARAM_NAME)) {
      String kstartstr = getParamTextValue(DEBUGDB_STARTKEY_PARAM_NAME, 0);
      bstr = prepareVarKey(kstartstr);
      if (bstr == null) {
        out.addError(new OutputError(Errno.EINVAL,
                                     "Invalid startkey " + kstartstr));
        output.setOutput(out);
        return output;
      }
    }

    // MaxKeys
    int maxkeys = 1;
    if (isParamPresent(DEBUGDB_MAXKEYS_PARAM_NAME)) {
      maxkeys = getParamIntValue(DEBUGDB_MAXKEYS_PARAM_NAME, 0);
    }
    RawScanFromFid(fid, dumpFile, bstr, maxkeys, out);

    return output;
  }

  private CommandOutput dumpTable() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    String path = getParamTextValue(DEBUGDB_TABLE_PARAM_NAME, 0);
    TabletStats tabletStats = new TabletStats(path, getUserLoginId());

    // Iterate through all tablets, 50 records at a time
    for (int i = 0; ; i += 50) {
      List<TabletDesc> tablets = tabletStats.getTablets(out, i, 50);
      if (tablets == null || tablets.isEmpty()) {
        // No more tablets found, or error encountered
        break;
      }

      for (TabletDesc tablet : tablets) {
        // Iterate through all tablets and get their dump using FID
        getDumpFromFid(tablet.getFid(), out);
      }
    }

    return output;
  }

  private long getBindingForContainer(int cid) throws CLIProcessingException {
    // lookup container in cldb
    int dbHost, dbPort;
    byte[] replyData;
    ContainerLookupRequest creq = ContainerLookupRequest.newBuilder()
                                    .addContainerId(cid)
                                    .setCreds(getUserCredentials())
                                    .build();
    try {
      replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
                    MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.ContainerLookupProc.getNumber(),
                    creq,
                    ContainerLookupResponse.class);

      if (replyData == null) {
        // most likely can not connect to CLDB
        LOG.error("Couldn't connect to the CLDB service");
        return -1;
      }

      ContainerLookupResponse resp;
      resp = ContainerLookupResponse.parseFrom(replyData);
      if (resp.getStatus() == 0) {
        Server server = resp.getContainers(0).getMServer();
        dbHost = server.getIps(0).getHost();
        dbPort = server.getIps(0).getPort();
      } else {
        LOG.error("Container lookup failed : Error " + Errno.toString(resp.getStatus()));
        return -1;
      }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      LOG.error("Container lookup failed");
      return -1;
    }

    return Rpc.createBindingFor(dbHost, dbPort, clusterName, ServerKeyType.ServerKey.getNumber());
  }

  private long getBindingForGateway(int gIp, int gPort) throws CLIProcessingException {
    if (gPort == 0)
      gPort = DefaultGatewayPort;
    return Rpc.createBindingFor(gIp, gPort, clusterName, ServerKeyType.ServerKey.getNumber());
  }

  private CommandOutput bmap()  throws CLIProcessingException {
    return cdscanrpc( true /* print block map only */ );
  }

  private CommandOutput cdscan( )  throws CLIProcessingException {
    return cdscanrpc( false /* print file cluster information */);
  }

  private CommandOutput cdscanrpc( boolean printBlockMapOnly )  throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    String fidstr = getParamTextValue(DEBUGDB_FID_PARAM_NAME, 0);
    FidMsg fid = stringToFid(fidstr);
    if (fid == null) {
      out.addError(new OutputError(Errno.EINVAL, "Invalid fid " + fidstr));
      return output;
    }

    // lookup container in cldb
    long dbinding = getBindingForContainer(fid.getCid());
    if (dbinding == -1) {
      out.addError(new OutputError(Errno.ERPCFAILED,"container lookup failed"));
      return output;
    }

    ArrayList<Long> blockAddrList = new ArrayList<Long>();
    boolean  readCompleteClusterData = false;
    long startId = 0;

    while( !readCompleteClusterData  ) {
      ScanFileClustersRequest req = ScanFileClustersRequest.newBuilder()
                                         .setFid(fid)
                                         .setCreds(getUserCredentials())
                                         .setIdx( startId )
                                         .build();
      try {
        byte[] replyData;
        replyData = Rpc.sendRequest(dbinding,
                                    MapRProgramId.FileServerProgramId.getNumber(),
                                    FSProg.FileClusterScanProc .getNumber(),
                                    req);
        if (replyData == null) {
          LOG.error("Got null reply from RPC");
          out.addError(new OutputError(Errno.ERPCFAILED, "cdscan rpc failed"));
          return output;
        }

        ScanFileClustersResponse resp = ScanFileClustersResponse.parseFrom(replyData);
        if (resp.getStatus() == 0) {
          int nClusters = resp.getClustersCount();
          for( int pos = 0; pos < nClusters; ++pos ) {
            ScanFileClustersResponse.Cluster fileCluster = resp.getClusters( pos );
            if ( printBlockMapOnly ) {
              int nBlocks = fileCluster.getBlocksCount();
              for( int blockPos = 0; blockPos < nBlocks; ++blockPos ) {
                blockAddrList.add( fileCluster.getBlocks( blockPos ) );
              }
            } else {
              OutputNode dout = new OutputNode();
              dout.addChild(new OutputNode("id",
                                         fileCluster.getIdx()));

              dout.addChild(new OutputNode("compressiontype",
                                               fileCluster.getCtype().toString()));
              dout.addChild(new OutputNode("packed",fileCluster.getPacked()));
              OutputNode blockOut = new OutputNode( "blocks" );
              int nBlocks = fileCluster.getBlocksCount();
              for( int blockPos = 0; blockPos < nBlocks; ++blockPos ) {
                OutputNode blockAddrOut = new OutputNode("blockaddr",
                                                fileCluster.getBlocks( blockPos ) );
                blockOut.addChild( blockAddrOut );
              }
              dout.addChild( blockOut );
              OutputNode fileClusterBrepOut = new OutputNode("brep" );
              int nBrep = fileCluster.getBrepCount();
              for( int brepPos = 0; brepPos < nBrep; ++brepPos ) {
                ScanFileClustersResponse.Cluster.BlockRep fileClusterBlockRep =
                                                 fileCluster.getBrep( brepPos );
                OutputNode brepOut = new OutputNode("");
                long crcVal = fileClusterBlockRep.getCrc();
                if( crcVal < 0 ) {
                  crcVal = -crcVal;
                  crcVal += ((long)1 << 31 );
                }
                brepOut.addChild(new OutputNode("crc", crcVal));
                brepOut.addChild(new OutputNode("compressed",
                                             fileClusterBlockRep.getCompressed()));
                brepOut.addChild(new OutputNode("notCompressible",
                                        fileClusterBlockRep.getNotCompressible()));
                brepOut.addChild(new OutputNode("clen",
                                        fileClusterBlockRep.getClen()));
                fileClusterBrepOut.addChild( brepOut );
              }
              dout.addChild( fileClusterBrepOut);
              out.addNode(dout);
            }

            startId = fileCluster.getIdx() + 1;
          }

          // see if it needs to make another RPC call
          readCompleteClusterData = !resp.getHasMore();

        } else {
          out.addError(new OutputError(resp.getStatus(),
                       "CdScan failed, Error : " +
                       Errno.toString(resp.getStatus())));
          // should we retry?
          return output;
        }
      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);
      } catch (Exception e) {
        LOG.error("Exception processing cdscan command");
        out.addError(new OutputError(Errno.ERPCFAILED, "db rpc failed"));
        return output;
      }
    }

    if( printBlockMapOnly ) {
      int nTotalAddrs = blockAddrList.size();
      long prevBlock = -99;
      long size = 0;
      OutputNode count = null;
      OutputNode blockOut = null;
      for( int addrPos = 0; addrPos < nTotalAddrs; ++addrPos ) {
        long curBlock = blockAddrList.get( addrPos );

        if( curBlock == prevBlock + 1 ) {
          size++;
        } else {

          if( size > 0 ) {
            count.setValue( size );
            out.addNode( blockOut );
          }

          blockOut = new OutputNode("block");
          blockOut.addChild( new OutputNode("addr", curBlock ) );
          count = new OutputNode("count");
          blockOut.addChild( count );
          size = 1;
        }

        prevBlock = curBlock;
      }

      if( size > 0 ) {
        count.setValue( size );
        out.addNode( blockOut );
      }
    }

    return output;
  }

  private CommandOutput stat()  throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    String fidstr = getParamTextValue(DEBUGDB_FID_PARAM_NAME, 0);
    FidMsg fid = stringToFid(fidstr);
    if (fid == null) {
      out.addError(new OutputError(Errno.EINVAL, "Invalid fid " + fidstr));
      output.setOutput(out);
      return output;
    }

    // lookup container in cldb
    long dbinding = getBindingForContainer(fid.getCid());
    if (dbinding == -1) {
      out.addError(new OutputError(Errno.ERPCFAILED,"container lookup failed"));
      return output;
    }

    // Send GetAttr to mfs
    GetattrRequest req = GetattrRequest.newBuilder()
                                       .setNode(fid)
                                       .setCreds(getUserCredentials())
                                       .build();
    try {
      byte[] replyData;
      replyData = Rpc.sendRequest(dbinding,
                                  MapRProgramId.FileServerProgramId.getNumber(),
                                  FSProg.GetattrProc.getNumber(),
                                  req);
      if (replyData == null) {
        LOG.error("Got null reply from RPC");
        out.addError(new OutputError(Errno.ERPCFAILED, "getattr rpc failed"));
        return output;
      }

      GetattrResponse resp = GetattrResponse.parseFrom(replyData);
      OutputNode dout = new OutputNode();
      if (resp.getStatus() == 0) {
        dout.addChild(new OutputNode("type",
                                     resp.getAttr().getType().toString()));

        String subtype = printableSubType(resp.getAttr().getType(),
                                          resp.getAttr().getSubtype());
        dout.addChild(new OutputNode("subtype", subtype));
        dout.addChild(new OutputNode("parent",
                                     printableFid(resp.getAttr().getParent())));
        dout.addChild(new OutputNode("size", resp.getAttr().getSize()));
        dout.addChild(new OutputNode("nblocks", resp.getAttr().getNblocks()));
        String cs = "off";
        if (resp.getAttr().getCanCompress())
          cs = CompressionHelper.getCompressionName(FileCompressionType.valueOf(resp.getAttr().getCompressorType()));

        dout.addChild(new OutputNode("compression", cs));
        dout.addChild(new OutputNode("deleteFlags",
                                     resp.getAttr().getDeleteFlags()));
        dout.addChild(new OutputNode("atime",
                                     resp.getAttr().getAtime().getSec()));
        dout.addChild(new OutputNode("mtime",
                                     resp.getAttr().getMtime().getSec()));
        int mode = resp.getAttr().getMode();
        dout.addChild(new OutputNode("mode",
                                     Integer.toOctalString(mode)));
        dout.addChild(new OutputNode("audit", resp.getAttr().getAudit()));
        dout.addChild(new OutputNode("uid", resp.getAttr().getUid()));
        dout.addChild(new OutputNode("gid", resp.getAttr().getGid()));
        if (resp.getAttr().hasVersion())
          dout.addChild(new OutputNode("version",
                                       resp.getAttr().getVersion()));
        dout.addChild(new OutputNode("networkencryption",
                                     resp.getAttr().getWireSecurityEnabled()));
        out.addNode(dout);
      } else {
        out.addError(new OutputError(resp.getStatus(),
                     "GetAttr failed, Error : " +
                     Errno.toString(resp.getStatus())));
      }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      LOG.error("Exception processing stat command");
      out.addError(new OutputError(Errno.ERPCFAILED, "db rpc failed"));
    }
    output.setOutput(out);
    return output;
  }

  private CommandOutput getHostNames()  throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    String ipStr = getParamTextValue(DEBUGDB_HOST_PARAM_NAME, 0);
    IPAddress gatewayIp = buildIPFromString(ipStr);
    if (gatewayIp == null) {
      out.addError(new OutputError(Errno.EINVAL, "Invalid gateway host " + ipStr));
      output.setOutput(out);
      return output;
    }

    int gatewayPort = getParamIntValue(DEBUGDB_PORT_PARAM_NAME, 0);
    long dbinding = getBindingForGateway(gatewayIp.getHost(), gatewayPort);
    if (dbinding == -1) {
      out.addError(new OutputError(Errno.ERPCFAILED,"Gateway connect failed"));
      return output;
    }

    // Send GetAttr to mfs
    GetHostNamesRequest req = GetHostNamesRequest.newBuilder()
                                       .setCreds(getUserCredentials())
                                       .build();
    try {
      byte[] replyData;
      replyData = Rpc.sendRequest(dbinding,
                                  MapRProgramId.DBReplicatorServerProgramId.getNumber(),
                                  DBReplicatorProg.GetHostNamesProc.getNumber(),
                                  req);
      if (replyData == null) {
        LOG.error("Got null reply from RPC");
        out.addError(new OutputError(Errno.ERPCFAILED, "gethostnames rpc failed"));
        return output;
      }

      GetHostNamesResponse resp = GetHostNamesResponse.parseFrom(replyData);
      OutputNode dout = new OutputNode();
      if (resp.getStatus() == 0) {
        for (String name : resp.getNamesList()) {
          dout.addChild(new OutputNode("HostNames", name));
        }
        out.addNode(dout);
      } else {
        out.addError(new OutputError(resp.getStatus(),
                     "GetHostNames failed, Error : " +
                     Errno.toString(resp.getStatus())));
      }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      LOG.error("Exception processing getHostNames command");
      out.addError(new OutputError(Errno.ERPCFAILED, "db rpc failed"));
    }
    output.setOutput(out);
    return output;
  }

  private KvMsg keyToKvMsg(String key, boolean keyIsStringType)
      throws CLIProcessingException {

    KvStoreKey toDelKey;
    if (keyIsStringType) {
      ByteString bs = prepareVarKey(key);

      toDelKey = KvStoreKey.newBuilder()
                           .setType(FSKeyType.VarKey)
                           .setVarKey(bs)
                           .build();
    } else {
      int kval = Integer.parseInt(key);

      toDelKey = KvStoreKey.newBuilder()
                           .setType(FSKeyType.UintKey)
                           .setIntKey(kval)
                           .build();
    }
    return KvMsg.newBuilder()
                .setKey(toDelKey)
                .build();
  }

  private CommandOutput multiOp()  throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);
    boolean done = false;

    int cid = 0;
    FidMsg kvfid = null;
    KvList.Builder delKeys = null;
    List<DeleteAsyncEntry> delFids = null;

    if (isParamPresent(DEBUGDB_KVFID_PARAM_NAME)) {
      String fidstr = getParamTextValue(DEBUGDB_KVFID_PARAM_NAME, 0);
      kvfid = stringToFid(fidstr);
      if (kvfid == null) {
        out.addError(new OutputError(Errno.EINVAL, "Invalid kvfid " + fidstr));
        output.setOutput(out);
        return output;
      }

      cid = kvfid.getCid();
    }

    boolean keyIsStringType = false;
    if (isParamPresent(DEBUGDB_KEYTYPE_PARAM_NAME)) {
      String keytype = getParamTextValue(DEBUGDB_KEYTYPE_PARAM_NAME, 0);
      if (keytype.equalsIgnoreCase("int")) {
        keyIsStringType = false;
      } else if (keytype.equalsIgnoreCase("string") ||
                 keytype.equalsIgnoreCase("bytes")) {
        keyIsStringType = true;
      } else {
        out.addError(new OutputError(Errno.EINVAL,
                                     "Invalid keyType " + keytype));
        output.setOutput(out);
        return output;
      }
    }

    if (isParamPresent(DEBUGDB_DELKEYS_PARAM_NAME)) {
      if (kvfid == null) {
        out.addError(new OutputError(Errno.EINVAL,
                                    "keylist present, but no kvfid specified"));
        output.setOutput(out);
        return output;
      }

      delKeys = KvList.newBuilder();
      String klist = getParamTextValue(DEBUGDB_DELKEYS_PARAM_NAME, 0);
      if (klist.contains(MULTI_ARG_SEP)) {
        String[] ks = klist.split(MULTI_ARG_SEP);
        for (String kone : ks)
          delKeys.addEntries(keyToKvMsg(kone, keyIsStringType));
      } else {
        delKeys.addEntries(keyToKvMsg(klist, keyIsStringType));
      }
    }

    if (isParamPresent(DEBUGDB_DELFIDS_PARAM_NAME)) {
      delFids = new ArrayList<DeleteAsyncEntry>();
      String flist = getParamTextValue(DEBUGDB_DELFIDS_PARAM_NAME, 0);

      if (flist.contains(MULTI_ARG_SEP)) {
        String[] fs = flist.split(MULTI_ARG_SEP);
        for (String fone : fs) {
          FidMsg fid = stringToFid(fone);
          if (fid == null) {
            out.addError(new OutputError(Errno.EINVAL, "Invalid fid " + fone));
            output.setOutput(out);
            return output;
          }
          if (cid == 0)
            cid = fid.getCid();

          DeleteAsyncEntry del = DeleteAsyncEntry.newBuilder()
                                                 .setFid(fid)
                                                 .setDelay(false)
                                                 .build();
          delFids.add(del);
        }
      } else {
        FidMsg fid = stringToFid(flist);
        if (fid == null) {
          out.addError(new OutputError(Errno.EINVAL, "Invalid fid " + flist));
          output.setOutput(out);
          return output;
        }
        if (cid == 0)
          cid = fid.getCid();

        DeleteAsyncEntry del = DeleteAsyncEntry.newBuilder()
                                               .setFid(fid)
                                               .setDelay(false)
                                               .build();
        delFids.add(del);
      }
    }

    if (delKeys == null && delFids == null) {
      out.addError(new OutputError(Errno.EINVAL,
                   "both delkeys and dellist are empty"));
      output.setOutput(out);
      return output;
    }

    // lookup container in cldb
    long dbbinding = getBindingForContainer(cid);
    if (dbbinding == -1) {
      out.addError(new OutputError(Errno.ERPCFAILED,
                                   "container " + cid + " lookup failed"));
      return output;
    }

    // Send unlink to mfs

    KvStoreMultiOpDBRequest.Builder req = KvStoreMultiOpDBRequest.newBuilder();
    if (kvfid != null)
      req.setFid(kvfid);
    if (delKeys != null)
      req.setDelList(delKeys);
    if (delFids != null) {
      for (DeleteAsyncEntry dae : delFids)
        req.addDeleteAsyncs(dae);
    }
    req.setCreds(getUserCredentials());

    try {
      byte[] replyData;
      replyData = Rpc.sendRequest(dbbinding,
                                  MapRProgramId.FileServerProgramId.getNumber(),
                                  FSProg.KvStoreMultiOpDBProc.getNumber(),
                                  req.build());
      if (replyData == null) {
        LOG.error("Got null reply from RPC");
        out.addError(new OutputError(Errno.ERPCFAILED, "MultiOpDB failed"));
        return output;
      }

      KvStoreMultiOpDBResponse resp
          = KvStoreMultiOpDBResponse.parseFrom(replyData);
      OutputNode dout = new OutputNode();
      if (resp.getStatus() == 0) {
        done = true;
      } else {
        out.addError(new OutputError(resp.getStatus(),
                     "MultiOpDB failed, Error : " +
                     Errno.toString(resp.getStatus())));
      }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      LOG.error("Exception processing MultiOp command");
      out.addError(new OutputError(Errno.ERPCFAILED, "fs rpc failed"));
    }
    if (done)
      return new TextCommandOutput(("MultiOp done").getBytes());
    else
    return output;
  }

  private CommandOutput switchMaster() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    int cid = getParamIntValue(DEBUGDB_CID_PARAM_NAME, 0);

    // lookup container in cldb
    int dbHost, dbPort;
    byte[] replyData;
    String spid;
    ContainerLookupRequest creq = ContainerLookupRequest.newBuilder()
                                    .addContainerId(cid)
                                    .setCreds(getUserCredentials())
                                    .build();
    try {
      replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
                    MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.ContainerLookupProc.getNumber(),
                    creq,
                    ContainerLookupResponse.class);

      if (replyData == null) {
        // most likely can not connect to CLDB
        LOG.error("Couldn't connect to the CLDB service");
        out.addError(new OutputError(Errno.ERPCFAILED, "cldb rpc failed"));
        return output;
      }

      ContainerLookupResponse resp;
      resp = ContainerLookupResponse.parseFrom(replyData);
      if (resp.getStatus() == 0) {
        spid = resp.getContainers(0).getMServer().getSpInfo().getSpId();
      } else {
        out.addError(new OutputError(resp.getStatus(),
                     "Container lookup failed, Error : " +
                     Errno.toString(resp.getStatus())));
        return output;
      }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      LOG.error("Container lookup failed");
      out.addError(new OutputError(Errno.ERPCFAILED, "cldb rpc failed"));
      return output;
    }


    // Send switch rpc to cldbdb
    boolean done = false;
    ContainerSwitchMasterRequest req = ContainerSwitchMasterRequest.newBuilder()
                                    .setCid(cid)
                                    .setMasterSpid(spid)
                                    .setCreds(getUserCredentials())
                                    .build();
    try {
      replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
                    MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.ContainerSwitchMasterProc.getNumber(),
                    req,
                    ContainerSwitchMasterResponse.class);

      if (replyData == null) {
        // most likely can not connect to CLDB
        LOG.error("Couldn't connect to the CLDB service");
        out.addError(new OutputError(Errno.ERPCFAILED, "cldb rpc failed"));
        return output;
      }

      ContainerSwitchMasterResponse resp;
      resp = ContainerSwitchMasterResponse.parseFrom(replyData);

      if (resp.getStatus() == 0) {
        done = true;
      } else {
        out.addError(new OutputError(resp.getStatus(),
                     "switchMaster failed, Error : " +
                     Errno.toString(resp.getStatus())));
      }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      LOG.error("Container switchMaster failed");
      out.addError(new OutputError(Errno.ERPCFAILED, "cldb rpc failed"));
      return output;
    }
    if (done)
      return new TextCommandOutput(("Switched master").getBytes());
    else
      return output;
  }

  private IPAddress buildIPFromString(String ipstr) {
    if (ipstr.equalsIgnoreCase("localhost") ||
        ipstr.equalsIgnoreCase("127.0.0.1") ) {
      ipstr = "127.0.0.1";
    } else {
      List<String> ips;
      ips = NodesCommonUtils.convertHostToIp(Collections.singletonList(ipstr));
      if (ips.isEmpty())
        return null;

      ipstr = ips.get(0);
    }

    int host = Util.ipToInt(ipstr);
    IPAddress server = IPAddress.newBuilder()
                                .setHost(host)
                                .build();
    return server;
  }

  private CommandOutput checkTablet() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    String fidstr = getParamTextValue(DEBUGDB_FID_PARAM_NAME, 0);
    FidMsg fid = stringToFid(fidstr);
    if (fid == null) {
      out.addError(new OutputError(Errno.EINVAL, "Invalid fid " + fidstr));
      output.setOutput(out);
      return output;
    }

    boolean fromGfsck = false;
    if (isParamPresent(DEBUGDB_FROMGFSCK_PARAM_NAME))
      fromGfsck = getParamBooleanValue(DEBUGDB_FROMGFSCK_PARAM_NAME, 0);

    boolean isRepair = false;
    if (isParamPresent(DEBUGDB_GFSCKREPAIR_PARAM_NAME)) {
      isRepair = getParamBooleanValue(DEBUGDB_GFSCKREPAIR_PARAM_NAME, 0);
      if (isRepair)
        fromGfsck = true;
    }

    ByteString bstart = null;
    if (isParamPresent(DEBUGDB_STARTKEY_PARAM_NAME)) {
      String kstartstr = getParamTextValue(DEBUGDB_STARTKEY_PARAM_NAME, 0);
      bstart = prepareVarKey(kstartstr);
      if (bstart == null) {
        out.addError(new OutputError(Errno.EINVAL,
                                     "Invalid startkey " + kstartstr));
        output.setOutput(out);
        return output;
      }
    } else {
      bstart = ByteString.EMPTY;
    }

    ByteString bend = null;
    if (isParamPresent(DEBUGDB_ENDKEY_PARAM_NAME)) {
      String kendstr = getParamTextValue(DEBUGDB_ENDKEY_PARAM_NAME, 0);
      bend = prepareVarKey(kendstr);
      if (bend == null) {
        out.addError(new OutputError(Errno.EINVAL,
                                     "Invalid endkey " + kendstr));
        output.setOutput(out);
        return output;
      }
    } else {
      bend = ByteString.EMPTY;
    }

    String traceFile = getParamTextValue(DEBUGDB_TRACEFILE_PARAM_NAME, 0);
    
    
    // lookup container in cldb
    long dbinding = getBindingForContainer(fid.getCid());
    GetattrResponse resp;

    if (dbinding == -1) {
      out.addError(new OutputError(Errno.ERPCFAILED,
                                   "container lookup failed"));
      return output;
    }

    //before we send rpc to checkTablet, we need to extract table type
    //associated with the current tablet. This is a 3 step process :
    //a) find table fid.
    //b) find schema fid for the table fid.
    //c) find tabletype attribute with schema fid and pass it to checkTablet.

    //A) Send GetAttr to mfs to get table fid
    GetattrRequest req = GetattrRequest.newBuilder()
                                       .setNode(fid)
                                       .setCreds(getUserCredentials())
                                       .build();
    try {
      byte[] replyData;
      replyData = Rpc.sendRequest(dbinding,
                                MapRProgramId.FileServerProgramId.getNumber(),
                                FSProg.GetattrProc.getNumber(),
                                req);
      if (replyData == null) {
        out.addError(new OutputError(Errno.ERPCFAILED, "getattr rpc failed"));
        return output;
      }

      resp = GetattrResponse.parseFrom(replyData);
      if (resp.getStatus() != 0) {
        out.addError(new OutputError(resp.getStatus(),
                     "GetAttr failed, Error : " +
                     Errno.toString(resp.getStatus())));
        return output;
      }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      LOG.error("checkTablet : Exception processing GetAttr RPC to extract table fid");
      out.addError(new OutputError(Errno.ERPCFAILED, "mfs rpc failed"));
      return output;
    }

    FidMsg parent = resp.getAttr().getParent();

    //B) get schema fid
    MapRFileSystem fs = MapRCliUtil.getMapRFileSystem();
    byte [] byteArr; 
    FidMsg schemaFid = null;
    try {

      KvStoreKey kStart = KvStoreKey.newBuilder()
                          .setType(FSKeyType.VarKey)
                          .setVarKey(DBInternalDefaults.getDefaultInstance().getTableKeyForSchemaBytes())
                          .build();
      byteArr = fs.scanKV(clusterName, "/.mapr::fid::" + printableFid(parent),
                          kStart.toByteArray(), null, 1);
      KvList kvList = KvList.parseFrom(byteArr);
      
      //get schema fid
      FidMsg f = null;
      KvMsg msg = kvList.getEntries(0);
      f = FidMsg.parseFrom(msg.getValue());
      FidMsg.Builder sFid = FidMsg.newBuilder();
      schemaFid = sFid.setCid(parent.getCid())
                      .setCinum(f.getCinum())
                      .setUniq(f.getUniq())
                      .build();

      if (schemaFid != null) {
        LOG.info("Schema fid for tablet "+fidstr+" is "+printableFid(schemaFid));
      }

    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, "scanKV failed to get schemaFid."));
      return null;
    }
    
    MiniInodeMsg.TableType tableType = MiniInodeMsg.TableType.BINARY;
    try {
      KvStoreKey kStart = KvStoreKey.newBuilder()
          .setType(FSKeyType.VarKey)
          .setVarKey(DBInternalDefaults.getDefaultInstance().getAttrBytes())
          .build();

      byteArr = fs.scanKV(clusterName, "/.mapr::fid::"+printableFid(schemaFid), kStart.toByteArray(), null, 1);

      KvList kvList = KvList.parseFrom(byteArr);
      KvMsg msg = kvList.getEntries(0);
      TableAttr attr = TableAttr.parseFrom(msg.getValue());

      if (attr.hasJson()) {
        tableType = (attr.getJson() == true)? MiniInodeMsg.TableType.JSON : MiniInodeMsg.TableType.BINARY;
      }      
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, "scanKV failed to get tabletype."));
      return null;
    }


    // lookup container in cldb
    long fsinding = getBindingForContainer(fid.getCid());
    if (fsinding == -1) {
      out.addError(new OutputError(Errno.ERPCFAILED,"container lookup failed"));
      return output;
    }

    // Send split rpc to db
    boolean done = false;
    TabletRangeCheckRequest treq = TabletRangeCheckRequest.newBuilder()
                                                .setTabletFid(fid)
                                                .setCreds(getUserCredentials())
                                                .setStartKey(bstart)
                                                .setEndKey(bend)
                                                .setTraceFile(traceFile)
                                                .setIsGfsckRepair(isRepair)
                                                .setFromGfsck(fromGfsck)
                                                .setTableType(tableType)
                                                .build();
    try {
      byte[] replyData;
      replyData = Rpc.sendRequest(fsinding,
                                  MapRProgramId.FileServerProgramId.getNumber(),
                                  FSProg.TabletRangeCheckProc.getNumber(),
                                  treq);
      if (replyData == null) {
        LOG.error("Got null reply from RPC");
        out.addError(new OutputError(Errno.ERPCFAILED, "fs rpc failed"));
        return output;
      }

      TabletRangeCheckResponse tresp;
      tresp = TabletRangeCheckResponse.parseFrom(replyData);
      if (tresp.getStatus() == 0) {
        done = true;
      } else {
        out.addError(new OutputError(tresp.getStatus(),
                       "TabletRangeCheck failed, Error : " +
                       Errno.toString(tresp.getStatus())));
      }
      output.setOutput(out);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      LOG.error("Exception processing RangeCheck command");
      out.addError(new OutputError(Errno.ERPCFAILED, "fs rpc failed"));
    }
    if (done)
      System.out.println("TabletRangeCheck done");

    return output;
  }

  private CommandOutput statTablet() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    String fidstr = getParamTextValue(DEBUGDB_FID_PARAM_NAME, 0);
    FidMsg fid = stringToFid(fidstr);
    if (fid == null) {
      out.addError(new OutputError(Errno.EINVAL, "Invalid fid " + fidstr));
      output.setOutput(out);
      return output;
    }

    // lookup container in cldb
    long dbinding = getBindingForContainer(fid.getCid());
    if (dbinding == -1) {
      out.addError(new OutputError(Errno.ERPCFAILED,"container lookup failed"));
      return output;
    }

    // Send stat rpc to db
    TabletStatRequest treq = TabletStatRequest.newBuilder()
                                              .setTablet(fid)
                                              .setCreds(getUserCredentials())
                                              .build();
    try {
      byte[] replyData;
      replyData = Rpc.sendRequest(dbinding,
                                  MapRProgramId.DBServerProgramId.getNumber(),
                                  DBProg.TabletStatProc.getNumber(),
                                  treq);
      if (replyData == null) {
        LOG.error("Got null reply from RPC");
        out.addError(new OutputError(Errno.ERPCFAILED, "db rpc failed"));
        return output;
      }

      TabletStatResponse tresp = TabletStatResponse.parseFrom(replyData);
      if (tresp.getStatus() == 0) {
        OutputNode dout = new OutputNode();
        SpaceUsage usage = tresp.getUsage();

        dout.addChild(new OutputNode("numPhysicalBlocks",
                                     usage.getNumPhysicalBlocks()));
        dout.addChild(new OutputNode("numLogicalBlocks",
                                     usage.getNumLogicalBlocks()));
        dout.addChild(new OutputNode("numRows",
                                     usage.getNumRows()));
        if (usage.hasNumRowsWithDelete())
          dout.addChild(new OutputNode("numRowsWithDelete",
                                       usage.getNumRowsWithDelete()));
        if (usage.hasNumRemoteBlocks())
          dout.addChild(new OutputNode("numRemoteBlocks",
                                       usage.getNumRemoteBlocks()));
        if (usage.hasNumSpills())
          dout.addChild(new OutputNode("numSpills",
                                       usage.getNumSpills()));
        if (usage.hasNumSegments())
          dout.addChild(new OutputNode("numSegments",
                                       usage.getNumSegments()));
        out.addNode(dout);
      } else {
        out.addError(new OutputError(tresp.getStatus(),
                     "TabletStat failed, Error : " +
                     Errno.toString(tresp.getStatus())));
      }
      output.setOutput(out);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      LOG.error("Exception processing compact command");
      out.addError(new OutputError(Errno.ERPCFAILED, "db rpc failed"));
    }
    return output;
  }

  private CommandOutput partitionSplits() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    String fidstr = getParamTextValue(DEBUGDB_FID_PARAM_NAME, 0);
    FidMsg fid = stringToFid(fidstr);
    if (fid == null) {
      out.addError(new OutputError(Errno.EINVAL, "Invalid fid " + fidstr));
      output.setOutput(out);
      return output;
    }

    // lookup container in cldb
    long dbinding = getBindingForContainer(fid.getCid());
    if (dbinding == -1) {
      out.addError(new OutputError(Errno.ERPCFAILED,"container lookup failed"));
      return output;
    }

    // Send stat rpc to db
    GetPartitionSplitsRequest treq = GetPartitionSplitsRequest.newBuilder()
                                              .setTablet(fid)
                                              .setCreds(getUserCredentials())
                                              .build();
    try {
      byte[] replyData;
      replyData = Rpc.sendRequest(dbinding,
                                  MapRProgramId.DBServerProgramId.getNumber(),
                                  DBProg.GetPartitionSplitsProc.getNumber(),
                                  treq);
      if (replyData == null) {
        LOG.error("Got null reply from RPC");
        out.addError(new OutputError(Errno.ERPCFAILED, "db rpc failed"));
        return output;
      }

      GetPartitionSplitsResponse tresp = GetPartitionSplitsResponse.parseFrom(replyData);
      if (tresp.getStatus() == 0) {
        for (ByteString bs : tresp.getStartKeysList()) {
          OutputNode dout = new OutputNode();
          dout.addChild(new OutputNode("startKey", printableKey(bs.toByteArray())));
          out.addNode(dout);
        }
      } else {
        out.addError(new OutputError(tresp.getStatus(),
                     "GetPartitionSplitsProc failed, Error : " +
                     Errno.toString(tresp.getStatus())));
      }
      output.setOutput(out);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      LOG.error("Exception processing compact command");
      out.addError(new OutputError(Errno.ERPCFAILED, "db rpc failed"));
    }
    return output;
  }
}
