/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.ByteString;
import com.mapr.baseutils.BinaryString;
import com.mapr.baseutils.Errno;
import com.mapr.cli.table.RecentTablesListManager;
import com.mapr.cli.table.RecentTablesListManagers;
import com.mapr.cliframework.base.*;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.inputparams.*;
import com.mapr.cli.common.FileclientRun;
import com.mapr.fs.MapRFileSystem;
import com.mapr.fs.proto.Dbserver.ColumnFamilyAttr;
import com.mapr.fs.proto.Dbserver.TableUpstreamDesc;
import com.mapr.fs.proto.Dbserver.TableUpstreamListResponse;
import com.mapr.fs.tables.TableProperties;
import com.mapr.fs.MapRFileSystem;

import org.apache.hadoop.fs.Path;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.json.JSONTokener;
import org.apache.log4j.Logger;

import java.io.IOException;
import java.util.HashSet;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Collections;

public class DbUpstreamCommands extends CLIBaseClass implements CLIInterface {

  private static final Logger LOG = Logger.getLogger(DbUpstreamCommands.class);

  public static final String COLUMNS_PARAM_NAME = "columns";
  public static final String OUTPUT_PARAM_NAME = "output";
  public static final String START_PARAM_NAME = "start";
  public static final String LIMIT_PARAM_NAME = "limit";

  private static final String PATH_PARAM_NAME = "path";
  private static final String UPSTREAM_PARAM_NAME = "upstream";
  private static final String PAUSED_PARAM_NAME = "paused";

  private static final CLICommand listCommand =
      new CLICommand(
          "list",
          "usage: table upstream list -path <tablepath>",
          DbUpstreamCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "table path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(COLUMNS_PARAM_NAME,
                  new TextInputParameter(COLUMNS_PARAM_NAME,
                      "columns",
                      CLIBaseClass.NOT_REQUIRED,
                      "all").setInvisible(true))
              .put(OUTPUT_PARAM_NAME,
                  new TextInputParameter(OUTPUT_PARAM_NAME,
                      "verbose|terse",
                      CLIBaseClass.NOT_REQUIRED,
                      "verbose").setInvisible(true))
              .put(START_PARAM_NAME,
                  new IntegerInputParameter(START_PARAM_NAME,
                      "start",
                      CLIBaseClass.NOT_REQUIRED,
                      0).setInvisible(true))
              .put(LIMIT_PARAM_NAME,
                  new IntegerInputParameter(LIMIT_PARAM_NAME,
                      "limit",
                      CLIBaseClass.NOT_REQUIRED,
                      Integer.MAX_VALUE).setInvisible(true))              .build(), null)
          .setShortUsage("table upstream list -path <tablepath>");

  private static final CLICommand addCommand =
      new CLICommand(
          "add",
          "usage: table upstream add -path <tablePath> -upstream <upstreamPath>",
          DbUpstreamCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "table path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(UPSTREAM_PARAM_NAME,
                  new TextInputParameter(UPSTREAM_PARAM_NAME,
                      "upstream table path",
                      CLIBaseClass.REQUIRED,
                      null))
             .build(), null)
          .setShortUsage("table upstream add -path <tablePath> -upstream <upstreamPath>");

  private static final CLICommand removeCommand =
      new CLICommand(
          "remove",
          "usage: table upstream remove -path <tablePath> -upstream <upstreamPath>",
          DbUpstreamCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "table path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(UPSTREAM_PARAM_NAME,
                  new TextInputParameter(UPSTREAM_PARAM_NAME,
                      "upstream table path",
                      CLIBaseClass.REQUIRED,
                      null))
             .build(), null)
          .setShortUsage("table upstream remove -path <tablePath> -upstream <upstreamPath>");

  // main command
  public static final CLICommand upstreamCommands =
      new CLICommand(
          "upstream", "upstream [add|list|remove]",
          CLIUsageOnlyCommand.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new CLICommand[]{
              addCommand,
              listCommand,
              removeCommand,
          }
      ).setShortUsage("table upstream [add|list|remove]");

  public DbUpstreamCommands(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }

  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    if (!super.validateInput()) {
      return output;
    }

    String cname = cliCommand.getCommandName();
    if (cname.equalsIgnoreCase(addCommand.getCommandName())) {
      addUpstream(out);
    } else if (cname.equalsIgnoreCase(listCommand.getCommandName())) {
      listUpstream(out);
    } else if (cname.equalsIgnoreCase(removeCommand.getCommandName())) {
      removeUpstream(out);
    }

    return output;
  }

  public String entityName() {
    return "table";
  }

  // constructs a byte array to a "pretty print" string.
  private String bytesToString(ByteString bstr) {
    return BinaryString.toStringBinary(bstr.toByteArray());
  }

  // constructs a "pretty printed binary string" to its raw format.
  private ByteString stringToByteString(String raws) {
    byte[] bArr = BinaryString.toBytesBinary(raws);
    return ByteString.copyFrom(bArr);
  }

  protected void addUpstream(OutputHierarchy out) throws CLIProcessingException {
    final String tablePath = DbCommands.getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId());
    final String upstreamFullPath = DbCommands.getTransformedPath(getParamTextValue(UPSTREAM_PARAM_NAME, 0), getUserLoginId());
   
    try {
      addUpStream(tablePath, upstreamFullPath,getUserLoginId());

    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }
  }

  public static void addUpStream(final String tablePath, final String upstreamFullPath, String user) throws CLIProcessingException, IOException{
    // Set as final to be called in impersonation
    final RecentTablesListManager manager = RecentTablesListManagers.getRecentTablesListManagerForUser(user);
    final TableUpstreamDesc.Builder upstreamBuilder = TableUpstreamDesc.newBuilder();
    final boolean validatePeer = true; // TODO: this should be a param

    // Call impersonation
    new FileclientRun(user) {
      @Override
      public void runAsProxyUser() throws IOException, CLIProcessingException{
        MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
        try {
          Path upath = new Path(upstreamFullPath);
          String clusterName = mfs.getClusterName(upath.toUri());
          upstreamBuilder.setClusterName(clusterName);

          String upstreamPath = mfs.getNameStr(upstreamFullPath);
          upstreamBuilder.setTablePath(upstreamPath);

          Path path = new Path(tablePath);
          if (!mfs.isTable(path)) {
            throw new CLIProcessingException("Table not found. " +
                                             "Path: " + tablePath);
          }

          if (clusterName.equals(mfs.getClusterNameUnchecked(tablePath)) &&
              upstreamPath.equals(mfs.getNameStr(tablePath))) {
            throw new CLIProcessingException("upstream and current tables " +
                                             "cannot be same");
          } 

          TableProperties props = mfs.getTableProperties(path);
          TableProperties uprops = mfs.getTableProperties(upath);
          if (validatePeer) {
             if (!mfs.isTable(upath))
               throw new CLIProcessingException("Upstream not found. " +
                                                "Path: " + upstreamPath);

            if (props.getAttr().getIsMarlinTable() !=
                uprops.getAttr().getIsMarlinTable()) {
              String msg = tablePath + " is " +
                           (props.getAttr().getIsMarlinTable() ? "" : " not ") +
                           "a stream, but upstream " +
                           upstreamPath + " is " +
                           (uprops.getAttr().getIsMarlinTable() ? "" : "not ") +
                           "a stream";
              throw new CLIProcessingException(msg);
            }

            if (props.getAttr().getJson() !=  uprops.getAttr().getJson()) {
              String msg = tablePath + " is " +
                           (props.getAttr().getJson() ? "" : " not ") +
                           "a json table, but upstream " +
                           upstreamPath + " is " +
                           (uprops.getAttr().getJson() ? "" : "not ") +
                           "a json table";
              throw new CLIProcessingException(msg);
            }
          }

          // Get the uuid of upstream table
          if (uprops.getUuid() == null || uprops.getUuid().length == 0)
            throw new IOException("upstream table does not have a uuid");

          upstreamBuilder.setTableUuid(ByteString.copyFrom(uprops.getUuid()));
          TableUpstreamDesc upstreamDesc = upstreamBuilder.build();

          mfs.addTableUpstream(path, upstreamDesc);
          manager.moveToTop(tablePath);
        } catch (IOException e) {
          manager.deleteIfNotExist(tablePath, mfs);
          throw new CLIProcessingException(e.getMessage());
        }
      }
    };
  }

  protected void listUpstream(OutputHierarchy out) throws CLIProcessingException {
    final String tablePath = DbCommands.getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId());
    final RecentTablesListManager manager = RecentTablesListManagers.getRecentTablesListManagerForUser(getUserLoginId());
    final List<TableUpstreamListResponse> rlist = new ArrayList<TableUpstreamListResponse>();
    try {
      // Call impersonation
      new FileclientRun(getUserLoginId()) {
        @Override
        public void runAsProxyUser() throws IOException, CLIProcessingException{
          MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
          try {
            if (!mfs.isTable(new Path(tablePath))) {
              throw new CLIProcessingException("Table not found. Path: " + tablePath);
            }

            TableUpstreamListResponse resp = mfs.listTableUpstreams(new Path(tablePath));
            rlist.add(resp);
            manager.moveToTop(tablePath);
          } catch (IOException e) {
            manager.deleteIfNotExist(tablePath, mfs);
            throw new CLIProcessingException(e.getMessage());
          }
        }
      };

      TableUpstreamListResponse listResp = rlist.get(0);
      for (TableUpstreamDesc ud : listResp.getUpstreamsList()) {
        OutputNode upstreamNode = new OutputNode();
        upstreamNode.addChild(new OutputNode("cluster", ud.getClusterName()));
        upstreamNode.addChild(new OutputNode(entityName(), ud.getTablePath()));
        upstreamNode.addChild(new OutputNode("idx", ud.getIdx()));
        upstreamNode.addChild(new OutputNode("uuid",
                   BinaryString.toUUIDString(ud.getTableUuid().toByteArray())));

        out.addNode(upstreamNode);
      }
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }
  }

  protected void removeUpstream(OutputHierarchy out) throws CLIProcessingException {
    final String tablePath = DbCommands.getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId());
    final String upstreamFullPath = DbCommands.getTransformedPath(getParamTextValue(UPSTREAM_PARAM_NAME, 0), getUserLoginId());
    final TableUpstreamDesc.Builder upstreamBuilder = TableUpstreamDesc.newBuilder();

    // Set as final to be called in impersonation
    final RecentTablesListManager manager = RecentTablesListManagers.getRecentTablesListManagerForUser(getUserLoginId());
    try {
      // Call impersonation
      new FileclientRun(getUserLoginId()) {
        @Override
        public void runAsProxyUser() throws IOException, CLIProcessingException{
          MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
          try {
            Path rpath = new Path(upstreamFullPath);
            String clusterName = mfs.getClusterNameUnchecked(upstreamFullPath);
            upstreamBuilder.setClusterName(clusterName);

            String upstreamPath = mfs.getNameStr(upstreamFullPath);
            upstreamBuilder.setTablePath(upstreamPath);

            TableUpstreamDesc upstreamDesc = upstreamBuilder.build();

            if (!mfs.isTable(new Path(tablePath))) {
              throw new CLIProcessingException("Table not found. Path: " + tablePath);
            }

            mfs.removeTableUpstream(new Path(tablePath), upstreamDesc);
            manager.moveToTop(tablePath);
          } catch (IOException e) {
            manager.deleteIfNotExist(tablePath, mfs);
            throw new CLIProcessingException(e.getMessage());
          }
        }
      };
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }   
  }
}
