/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.cli;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;

import org.apache.hadoop.fs.Path;
import org.apache.log4j.Logger;

import org.ojai.FieldPath;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.ByteString;
import com.mapr.baseutils.BinaryString;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cli.common.CopyTableCallable;
import com.mapr.cli.common.FileclientRun;
import com.mapr.cli.common.JobExecutor;
import com.mapr.cli.table.RecentTablesListManager;
import com.mapr.cli.table.RecentTablesListManagers;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import com.mapr.cliframework.base.inputparams.IntegerInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.db.impl.AdminImpl;
import com.mapr.db.impl.MapRDBTableImplHelper;
import com.mapr.fs.AceHelper;
import com.mapr.fs.MapRFileSystem;
import com.mapr.fs.cldb.util.Util;
import com.mapr.fs.gateway.external.GatewaySink;
import com.mapr.fs.hbase.CopyMetaHelper;
import com.mapr.fs.hbase.TableUtil;
import com.mapr.streams.impl.admin.MarlinAdminImpl;
import com.mapr.fs.proto.Common.FileCompressionType;
import com.mapr.fs.proto.Dbserver.ColumnFamilyAttr;
import com.mapr.fs.proto.Dbserver.Qualifier;
import com.mapr.fs.proto.Dbserver.TableReplicaDesc;
import com.mapr.fs.proto.Dbserver.TableReplicaListResponse;
import com.mapr.fs.proto.Dbserver.TableReplicaStatus;
import com.mapr.fs.proto.Error.ExtendedError;
import com.mapr.fs.proto.Marlinserver.MarlinInternalDefaults;
import com.mapr.fs.tables.MapRAdmin;
import com.mapr.fs.tables.TableProperties;

public class DbReplicaCommands extends CLIBaseClass implements CLIInterface,AceHelper.DBPermission {

  private static final Logger LOG = Logger.getLogger(DbReplicaCommands.class);

  public static final String COLUMNS_PARAM_NAME = "columns";
  public static final String OUTPUT_PARAM_NAME = "output";
  public static final String START_PARAM_NAME = "start";
  public static final String LIMIT_PARAM_NAME = "limit";

  private static final String PATH_PARAM_NAME = "path";
  private static final String REPLICA_PARAM_NAME = "replica";
  private static final String NEWREPLICA_PARAM_NAME = "newreplica";
  private static final String PAUSED_PARAM_NAME = "paused";
  private static final String THROTTLE_PARAM_NAME = "throttle";
  private static final String ENCRYPTION_PARAM_NAME = "networkencryption";
  private static final String SYNCHRONOUS_PARAM_NAME = "synchronous";
  private static final String COMPRESSION_PARAM_NAME = "networkcompression";
  private static final String MULTI_ARG_SEP = ",";
  private static final String MULTI_FIELD_SEP = ";";
  private static final String COLUMN_SEP = ":";
  private static final String JSON_PATH_SEP = ".";
  private static final String REFRESH_PARAM_NAME = "refreshnow";
  private static final String WAIT_FOR_COMPLETION_PARAM_NAME="waitforcompletion";
  private static final String MULTI_MASTER_PARAM_NAME="multimaster";
  private static final String TICKET_PATH_PARAM_NAME="ticketpath";
  private static final String MAPRDB_REPLICA_CLASS_NAME="MapRDB";
  private static final String MAPRSTREAM_REPLICA_CLASS_NAME="MapRStream";

  private static final CLICommand listCommand =
      new CLICommand(
          "list",
          "usage: table replica list -path <tablepath>",
          DbReplicaCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "table path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(REFRESH_PARAM_NAME,
                  new BooleanInputParameter(REFRESH_PARAM_NAME,
                      REFRESH_PARAM_NAME,
                      CLIBaseClass.NOT_REQUIRED,
                      false))
              .put(COLUMNS_PARAM_NAME,
                  new TextInputParameter(COLUMNS_PARAM_NAME,
                      "columns",
                      CLIBaseClass.NOT_REQUIRED,
                      "all").setInvisible(true))
              .put(OUTPUT_PARAM_NAME,
                  new TextInputParameter(OUTPUT_PARAM_NAME,
                      "verbose|terse",
                      CLIBaseClass.NOT_REQUIRED,
                      "verbose").setInvisible(true))
              .put(START_PARAM_NAME,
                  new IntegerInputParameter(START_PARAM_NAME,
                      "start",
                      CLIBaseClass.NOT_REQUIRED,
                      0).setInvisible(true))
              .put(LIMIT_PARAM_NAME,
                  new IntegerInputParameter(LIMIT_PARAM_NAME,
                      "limit",
                      CLIBaseClass.NOT_REQUIRED,
                      Integer.MAX_VALUE).setInvisible(true))              .build(), null)
          .setShortUsage("table replica list -path <tablepath>");

  private static final CLICommand addCommand =
      new CLICommand(
          "add",
          "usage: table replica add -path <tablePath> -replica <replicaPath> " +
          "-columns <cf1[:col1],cf2,..> -paused <true|false> " +
          "-throttle <true|false> -synchronous <true|false> " +
          "-networkencryption <true|false> -networkcompression <off|on|lzf|lz4|zlib> ",
          DbReplicaCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "table path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(REPLICA_PARAM_NAME,
                  new TextInputParameter(REPLICA_PARAM_NAME,
                      "replica table path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(COLUMNS_PARAM_NAME,
                  new TextInputParameter(COLUMNS_PARAM_NAME,
                      "comma separated list of <family>[:<column>]",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PAUSED_PARAM_NAME,
                  new BooleanInputParameter(PAUSED_PARAM_NAME,
                      "is replication paused",
                      CLIBaseClass.NOT_REQUIRED,
                      false))
              .put(THROTTLE_PARAM_NAME,
                  new BooleanInputParameter(THROTTLE_PARAM_NAME,
                      "throttle replication ops",
                      CLIBaseClass.NOT_REQUIRED,
                      false))
              .put(ENCRYPTION_PARAM_NAME,
                  new BooleanInputParameter(ENCRYPTION_PARAM_NAME,
                      "enable on-wire encryption",
                      CLIBaseClass.NOT_REQUIRED,
                      false))
              .put(SYNCHRONOUS_PARAM_NAME,
                  new BooleanInputParameter(SYNCHRONOUS_PARAM_NAME,
                      "is synchronous replication",
                      CLIBaseClass.NOT_REQUIRED,
                      false))
              .put(COMPRESSION_PARAM_NAME,
                  new TextInputParameter(COMPRESSION_PARAM_NAME,
                      "on-wire compression type: off|on|lzf|lz4|zlib",
                      CLIBaseClass.NOT_REQUIRED,
                      "on"))
             .build(), null)
          .setShortUsage("table replica add -path <tablePath> -replica <replicaPath>");

  private static final CLICommand autoSetupCommand =
      new CLICommand(
          "autosetup",
          "usage: table replica autosetup -path <tablePath> -replica <replicaPath> " +
          "-columns <cf1[:col1],cf2,..>" +
          "-multimaster <true/false>" +
          "-waitforcompletion <true/false>" +
          "-throttle <true|false> -synchronous <true|false> " +
          "-networkencryption <true|false> -networkcompression <off|on|lzf|lz4|zlib> " +
          "-ticketpath <path>" ,
          DbReplicaCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "table path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(REPLICA_PARAM_NAME,
                  new TextInputParameter(REPLICA_PARAM_NAME,
                      "replica table path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(COLUMNS_PARAM_NAME,
                  new TextInputParameter(COLUMNS_PARAM_NAME,
                      "comma separated list of <family>[:<column>]",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(SYNCHRONOUS_PARAM_NAME,
                  new BooleanInputParameter(SYNCHRONOUS_PARAM_NAME,
                      "is synchronous replication",
                      CLIBaseClass.NOT_REQUIRED,
                      false))
              .put(MULTI_MASTER_PARAM_NAME,
                  new BooleanInputParameter(MULTI_MASTER_PARAM_NAME,
                      "is multi master replication",
                      CLIBaseClass.NOT_REQUIRED,
                      false))
              .put(THROTTLE_PARAM_NAME,
                  new BooleanInputParameter(THROTTLE_PARAM_NAME,
                      "throttle replication ops",
                      CLIBaseClass.NOT_REQUIRED,
                      false))
              .put(ENCRYPTION_PARAM_NAME,
                  new BooleanInputParameter(ENCRYPTION_PARAM_NAME,
                      "enable on-wire encryption",
                      CLIBaseClass.NOT_REQUIRED,
                      false))
              .put(COMPRESSION_PARAM_NAME,
                  new TextInputParameter(COMPRESSION_PARAM_NAME,
                      "on-wire compression type: off|on|lzf|lz4|zlib",
                      CLIBaseClass.NOT_REQUIRED,
                      "on"))
              .put(WAIT_FOR_COMPLETION_PARAM_NAME,
                  new BooleanInputParameter(WAIT_FOR_COMPLETION_PARAM_NAME,
                      "wait for completion",
                      CLIBaseClass.NOT_REQUIRED,
                      true).setInvisible(true))
              .put(TICKET_PATH_PARAM_NAME,
                  new TextInputParameter(TICKET_PATH_PARAM_NAME,
                      "ticket path",
                      CLIBaseClass.NOT_REQUIRED,
                      null).setInvisible(true))
             .build(), null)
          .setShortUsage("table replica autosetup -path <tablePath> -replica <replicaPath>");

  private static final CLICommand editCommand =
      new CLICommand(
          "edit",
          "usage: table replica edit -path <tablePath> -replica <replicaPath>" +
          " -newreplica <newReplicaPath> -columns <cf1[:col1],cf2,..> " +
          "-paused <true|false> -allowallcfs <true> -throttle <true|false>" +
          " -synchronous <true|false> -networkencryption <true|false> " +
          "-networkcompression <off|on|lzf|lz4|zlib> ",
          DbReplicaCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "table path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(REPLICA_PARAM_NAME,
                  new TextInputParameter(REPLICA_PARAM_NAME,
                      "replica table path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(NEWREPLICA_PARAM_NAME,
                  new TextInputParameter(NEWREPLICA_PARAM_NAME,
                      "renamed table path",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(COLUMNS_PARAM_NAME,
                  new TextInputParameter(COLUMNS_PARAM_NAME,
                      "comma separated list of <family>[:<column>]",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(THROTTLE_PARAM_NAME,
                  new BooleanInputParameter(THROTTLE_PARAM_NAME,
                      "throttle replication ops",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(ENCRYPTION_PARAM_NAME,
                  new BooleanInputParameter(ENCRYPTION_PARAM_NAME,
                      "enable on-wire encryption",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(SYNCHRONOUS_PARAM_NAME,
                  new BooleanInputParameter(SYNCHRONOUS_PARAM_NAME,
                      "is synchronous replication",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(COMPRESSION_PARAM_NAME,
                  new TextInputParameter(COMPRESSION_PARAM_NAME,
                      "on-wire compression type: off|on|lzf|lz4|zlib",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
             .build(), null)
          .setShortUsage("table replica edit -path <tablePath> " +
                         "-replica <replicaPath> -newreplica <newReplicaPath>" +
                         " -columns <cf1[:col1],cf2,..> -paused <true|false> " +
                         "-allowallcfs <true> -throttle <true|false>  " +
                         "-synchronous <true|false> -networkencryption <true|false " +
                         "-networkcompression <off|on|lzf|lz4|zlib>");

  private static final CLICommand pauseCommand =
      new CLICommand(
          "pause",
          "usage: table replica pause -path <tablePath> -replica <replicaPath>",
          DbReplicaCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "table path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(REPLICA_PARAM_NAME,
                  new TextInputParameter(REPLICA_PARAM_NAME,
                      "replica table path",
                      CLIBaseClass.REQUIRED,
                      null))
             .build(), null)
           .setShortUsage("table replica pause -path <tablePath> -replica <replicaPath>");

  private static final CLICommand resumeCommand =
      new CLICommand(
          "resume",
          "usage: table replica resume -path <tablePath> -replica <replicaPath>",
          DbReplicaCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "table path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(REPLICA_PARAM_NAME,
                  new TextInputParameter(REPLICA_PARAM_NAME,
                      "replica table path",
                      CLIBaseClass.REQUIRED,
                      null))
             .build(), null)
           .setShortUsage("table replica resume -path <tablePath> -replica <replicaPath>");


  private static final CLICommand removeCommand =
      new CLICommand(
          "remove",
          "usage: table replica remove -path <tablePath> -replica <replicaPath>",
          DbReplicaCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "table path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(REPLICA_PARAM_NAME,
                  new TextInputParameter(REPLICA_PARAM_NAME,
                      "replica table path",
                      CLIBaseClass.REQUIRED,
                      null))
             .build(), null)
          .setShortUsage("table replica remove -path <tablePath> -replica <replicaPath>");

  // main command
  public static final CLICommand replicaCommands =
      new CLICommand(
          "replica", "replica [add|edit|list|remove|pause|resume|elasticsearch]",
          CLIUsageOnlyCommand.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new CLICommand[]{
              addCommand,
              editCommand,
              listCommand,
              removeCommand,
              pauseCommand,
              resumeCommand,
              autoSetupCommand,
              ElasticsearchCommands.esReplicaCommands
          }
      ).setShortUsage("table replica [add|edit|list|remove|pause|resume|elasticsearch]");

  public DbReplicaCommands(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }

  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    if (!super.validateInput()) {
      return output;
    }

    String cname = cliCommand.getCommandName();
    if (cname.equalsIgnoreCase(addCommand.getCommandName())) {
      addReplica(out);
    } else if (cname.equalsIgnoreCase(editCommand.getCommandName())) {
      editReplica(out);
    } else if (cname.equalsIgnoreCase(listCommand.getCommandName())) {
      listReplica(out);
    } else if (cname.equalsIgnoreCase(removeCommand.getCommandName())) {
      removeReplica(out);
    } else if (cname.equalsIgnoreCase(pauseCommand.getCommandName())) {
      pauseOrResumeReplica(out, true);
    } else if (cname.equalsIgnoreCase(resumeCommand.getCommandName())) {
      pauseOrResumeReplica(out, false);
    } else if (cname.equalsIgnoreCase(autoSetupCommand.getCommandName())) {
      autoSetup(out);
    }
    return output;
  }

  // constructs a byte array to a "pretty print" string.
  private String bytesToString(ByteString bstr) {
    return BinaryString.toStringBinary(bstr.toByteArray());
  }

  // constructs a "pretty printed binary string" to its raw format.
  private ByteString stringToByteString(String raws) {
    byte[] bArr = BinaryString.toBytesBinary(raws);
    return ByteString.copyFrom(bArr);
  }

  public static FileCompressionType getCompressionType(String cs) {
    if (cs.equalsIgnoreCase("off"))
      return FileCompressionType.FCT_OFF;
    else if (cs.equalsIgnoreCase("lz4") || cs.equalsIgnoreCase("on"))
      return FileCompressionType.FCT_LZ4;
    else if (cs.equalsIgnoreCase("lzf"))
      return FileCompressionType.FCT_LZF;
    else if (cs.equalsIgnoreCase("zlib"))
      return FileCompressionType.FCT_ZLIB;
    else
      return null;
  }

  public static String getCompressionName(FileCompressionType ct) {
    if (ct == FileCompressionType.FCT_OFF)
      return "off";
    else if (ct == FileCompressionType.FCT_LZ4)
      return "lz4";
    else if (ct == FileCompressionType.FCT_ZLIB)
      return "zlib";
    else if (ct == FileCompressionType.FCT_LZF ||
             ct == FileCompressionType.FCT_OLDLZF)
      return "lzf";
    else
      return "unknown";
  }

  public String entityName() {
    return "table";
  }

  //TODO JJS Temporary - replace with the JSON DB Table Admin API (Bug 19295)
  private void createJSONTableForCopy(String replicaPath,
                                      String srcPath,
                                      List<String> cfList,
                                      boolean isMarlinStream,  //TODO JJS: handle inside the fn with admin api
                                      boolean isBulkload) throws Exception {
    if (!isMarlinStream) {
      AdminImpl.createTableForCopy(replicaPath, srcPath, cfList, isBulkload);
    } else {
      //For Marlin, ignore the projections
      MarlinAdminImpl.createStreamForCopy(replicaPath, srcPath);
    }
  }

  private String getStreamDefaultJsonPathsForRepl(String streamDefaultCF) {
    if (streamDefaultCF == null || streamDefaultCF.isEmpty()) {
      return null;
    }

    StringBuilder defJP = new StringBuilder();
    String[] defCFs = streamDefaultCF.split(",", -1);
    int len = defCFs.length;
    for (int i = 0; i < len; ++i) {
      if (defCFs[i].equals("default")) {
        defCFs[i] = new String("");
      }
      defJP.append(defCFs[i]);
      if (i < (len - 1)) {
        defJP.append(",");
      }
    }
    return defJP.toString();
  }

  protected void autoSetup(OutputHierarchy out) throws CLIProcessingException {
    final String tablePath = DbCommands.getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId());
    final String replicaFullPath = DbCommands.getTransformedPath(getParamTextValue(REPLICA_PARAM_NAME, 0), getUserLoginId());
    final String colList = isParamPresent(COLUMNS_PARAM_NAME) ? getParamTextValue(COLUMNS_PARAM_NAME, 0) : null;
    final boolean isMultiMaster = isParamPresent(MULTI_MASTER_PARAM_NAME)?getParamBooleanValue(MULTI_MASTER_PARAM_NAME,0):false;
    final boolean waitForCompletion = isParamPresent(WAIT_FOR_COMPLETION_PARAM_NAME)?getParamBooleanValue(WAIT_FOR_COMPLETION_PARAM_NAME,0):false;
    final boolean isSync = isParamPresent(SYNCHRONOUS_PARAM_NAME) ? getParamBooleanValue(SYNCHRONOUS_PARAM_NAME, 0) : false;
    final String ticketPath = isParamPresent(TICKET_PATH_PARAM_NAME) ? getParamTextValue(TICKET_PATH_PARAM_NAME, 0) : null;
    final AceHelper.DBPermission dbPerm = this;
    final TableReplicaDesc.Builder replBuilder = TableReplicaDesc.newBuilder();

    try {
      // Call impersonation
      new FileclientRun(getUserLoginId()) {
        @Override
        public void runAsProxyUser() throws IOException, CLIProcessingException{
          final MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
          final boolean isReplicaTypeTable = !isExternalDestination(mfs, replicaFullPath);
          final int maxVersions = isReplicaTypeTable ? Integer.MAX_VALUE : 1;
          String rCName = null;
          TableProperties rtableProp;
          //Check if source table exists
          Path path = new Path(tablePath);
          if (!mfs.isTable(path)) {
            throw new CLIProcessingException("Table not found. Path: " + tablePath);
          }

          TableProperties props = mfs.getTableProperties(path);
          boolean isJson = props.getAttr().getJson();
          boolean isMarlinStream = props.getAttr().getIsMarlinTable();

          //Pause the replication
          replBuilder.setIsPaused(true);

          if (isParamPresent(THROTTLE_PARAM_NAME)) {
            replBuilder.setThrottle(getParamBooleanValue(THROTTLE_PARAM_NAME, 0));
          }

          if (isParamPresent(ENCRYPTION_PARAM_NAME)) {
            replBuilder.setEncryptonwire(getParamBooleanValue(ENCRYPTION_PARAM_NAME, 0));
          }

          if (isParamPresent(COMPRESSION_PARAM_NAME)) {
            FileCompressionType ctype =
                getCompressionType(getParamTextValue(COMPRESSION_PARAM_NAME, 0));
            if (ctype == null) {
              throw new CLIProcessingException("Invalid input values. " +
                  "The entered value for compression is not a supported type.");
            }
            replBuilder.setCompressonwire(ctype);
          } else if (props.getAttr().getIsMarlinTable()) {
            // For marlin streams, use the compression of 'messages' family
            // if compression type is not specified.
            FileCompressionType ctype = getStreamCompression(mfs, path);
            replBuilder.setCompressonwire(ctype);
          }

          if (isSync) {
            if (isReplicaTypeTable)
              replBuilder.setSynchronous(getParamBooleanValue(SYNCHRONOUS_PARAM_NAME, 0));
            else
              throw new CLIProcessingException("Can setup synchronous replication only when the replica is a MapRDB table.");
          }

          if (isMultiMaster) {
            if (!isReplicaTypeTable)
              throw new CLIProcessingException("Can setup multi-master replication only when the replica is a MapRDB table.");
          }

          if (isReplicaTypeTable) {
            //Check if destination table exists, else create the destination table
            if (!mfs.isTable(new Path(replicaFullPath))) {
              LOG.info("Creating the destination table: "+replicaFullPath+" as user: "+getUserLoginId());
              //Call  CLI Create table command to create table with bulkload set to true
              //Get the list of column families
              boolean isBulkLoad = true;
              if (!isJson) {
                List<String> cfList = TableUtil.getColumnFamiliesList(colList);
                if (props.getAttr().getIsMarlinTable())
                  isBulkLoad = false;

                DbCommands.createTable(replicaFullPath, false /*autoSplit*/,
                    isBulkLoad, null /*regionSize*/, false /*dropLarge*/,
                    false /*ttlCompact*/, null /*ttlCompactHrs*/, getUserLoginId(),
                    tablePath, dbPerm, CopyMetaHelper.METATYPE_ALL,cfList,
                    -1 /*auditValue*/, isJson, false/*insertOrder*/);
              } else {
                List<String> cfList = MapRDBTableImplHelper.getColumnFamiliesList(colList);

                try {
                  createJSONTableForCopy(replicaFullPath, tablePath, cfList, isMarlinStream, isBulkLoad);
                } catch (Exception e) {
                  throw new CLIProcessingException("Failed to create replica " +
                                                   replicaFullPath + " due to an exception: " + e.getMessage());
                }
              }
              LOG.info("Finished creating destination table");
            } else {
              LOG.info("Destination table: "+replicaFullPath+" exists already. Replication cannot be setup in auto mode");
              throw new CLIProcessingException("Destination table: "+replicaFullPath+" exists already. Replication cannot be setup in auto mode");
            }

          } else {
            replBuilder.setExternal(true);
            rCName = verifyExternalDstSanity(mfs, replicaFullPath);
          }

          String clist = colList;
          if (clist == null && isMarlinStream) {
            clist = getStreamDefaultCFsForRepl();
          }

          //Create replica
          setupReplication(dbPerm,replBuilder,tablePath,replicaFullPath,clist,(String)null,getUserLoginId(),mfs,rCName);

          //Call create upstream command
          if (isReplicaTypeTable) {
            LOG.info("Creating upstream");
            DbUpstreamCommands.addUpStream(replicaFullPath, tablePath, getUserLoginId());
            LOG.info("Finished creating upstream");
          }

          if (!isReplicaTypeTable && isMultiMaster) {
            throw new CLIProcessingException("Can setup multi-master replication only when the replica is a MapRDB table.");
          }
          copyTable(tablePath, replicaFullPath, maxVersions, getStreamDefaultJsonPathsForRepl(clist), getUserLoginId(), waitForCompletion, ticketPath,isMultiMaster,dbPerm,replBuilder);
        }
      };
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EINVALMISC, e.getMessage()));
    }
  }

  public static String verifyExternalDstSanity(final MapRFileSystem mfs, final String replicaFullPathStr)
    throws CLIProcessingException, IOException {
    String className = null;

    try {
      className = GatewaySink.verifyDstSanity(replicaFullPathStr);
    } catch (Exception e) {
      throw new CLIProcessingException(e.getMessage());
    }

    return className;
  }

  public static void setupMultiMaster(final AceHelper.DBPermission dbPerm, final TableReplicaDesc.Builder replBuilder, final String tablePath,
      final String replicaFullPath, final String colList, final String user) throws IOException, CLIProcessingException {
    //Setup replication from destination to source
    // Get the uuid of replica table
    MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
    TableProperties rtableProp = mfs.getTableProperties(new Path(tablePath));
    if (rtableProp.getUuid() == null || rtableProp.getUuid().length == 0)
      throw new IOException("replica table does not have a uuid");
    replBuilder.setTableUuid(ByteString.copyFrom(rtableProp.getUuid()));
    //Remove isPaused on replBuilder
    replBuilder.setIsPaused(false);
    setupReplication(dbPerm,replBuilder,replicaFullPath,tablePath,(String)null,(String)null,user,mfs,null);
    //Call create upstream command
    LOG.info("Creating upstream for source table in multi-master replication");
    DbUpstreamCommands.addUpStream(tablePath, replicaFullPath, user);
    LOG.info("Finished creating upstream for source table in multi-master replication");
  }

  public static String verifyExternalDstSanity(final String dstTable) throws
    CLIProcessingException, IOException {
      return verifyExternalDstSanity(MapRCliUtil.getMapRFileSystem(), dstTable);
    }

  public static boolean isExternalDestination(final MapRFileSystem mfs, final String dstTable) throws IOException {
    Path replicaPath = new Path(dstTable);
    return (mfs.isFile(replicaPath) && !mfs.isTable(replicaPath));
  }

  public static boolean isExternalDestination(final String dstTable) throws
    IOException, CLIProcessingException {
    return isExternalDestination(MapRCliUtil.getMapRFileSystem(), dstTable);
  }

  protected void addReplica(OutputHierarchy out) throws CLIProcessingException {
    final String tablePath = DbCommands.getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId());
    final String replicaFullPath = DbCommands.getTransformedPath(getParamTextValue(REPLICA_PARAM_NAME, 0), getUserLoginId());
    final String colList = isParamPresent(COLUMNS_PARAM_NAME) ? getParamTextValue(COLUMNS_PARAM_NAME, 0) : null;
    final boolean isSync = isParamPresent(SYNCHRONOUS_PARAM_NAME) ? getParamBooleanValue(SYNCHRONOUS_PARAM_NAME, 0) : false;
    final boolean validatePeer = true; // TODO : this should be a param
    final TableReplicaDesc.Builder replBuilder = TableReplicaDesc.newBuilder();
    final AceHelper.DBPermission dbPerm = this;

    try {
      // Call impersonation
      new FileclientRun(getUserLoginId()) {
        @Override
        public void runAsProxyUser() throws IOException, CLIProcessingException{
          final MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
          final boolean isReplicaTypeTable = !isExternalDestination(mfs, replicaFullPath);
          Path path = new Path(tablePath);
          String rCName = null;

          //Check if source table exists
          if (!mfs.isTable(path)) {
            throw new CLIProcessingException("Table not found. Path: " +
                                             tablePath);
          }

          TableProperties props = mfs.getTableProperties(path);
          boolean isJson = props.getAttr().getJson();
          boolean isMarlinStream = props.getAttr().getIsMarlinTable();

          //Check if destination table exists (but not for external destination)
          if (isReplicaTypeTable && validatePeer) {
            Path rpath = new Path(replicaFullPath);

            if (!mfs.isTable(rpath))
              throw new CLIProcessingException("Replica table does not exist." +
                                               " Path: " + replicaFullPath);

            TableProperties rprops = mfs.getTableProperties(rpath);
            if (props.getAttr().getIsMarlinTable() !=
                rprops.getAttr().getIsMarlinTable()) {
              String msg = tablePath + " is " +
                           (props.getAttr().getIsMarlinTable() ? "" : " not ") +
                           "a stream, but replica " +
                           replicaFullPath + " is " +
                           (rprops.getAttr().getIsMarlinTable() ? "" : "not ") +
                           "a stream";
              throw new CLIProcessingException(msg);
            }

            if (props.getAttr().getJson() !=  rprops.getAttr().getJson()) {
              String msg = tablePath + " is " +
                           (props.getAttr().getJson() ? "" : " not ") +
                           "a json table, but replica " +
                           replicaFullPath + " is " +
                           (rprops.getAttr().getJson() ? "" : "not ") +
                           "a json table";
              throw new CLIProcessingException(msg);
            }
          }

          if (isParamPresent(PAUSED_PARAM_NAME)) {
            replBuilder.setIsPaused(getParamBooleanValue(PAUSED_PARAM_NAME, 0));
          }

          if (isParamPresent(THROTTLE_PARAM_NAME)) {
            replBuilder.setThrottle(getParamBooleanValue(THROTTLE_PARAM_NAME, 0));
          }

          if (isParamPresent(ENCRYPTION_PARAM_NAME)) {
            replBuilder.setEncryptonwire(getParamBooleanValue(ENCRYPTION_PARAM_NAME, 0));
          }

          if (isParamPresent(COMPRESSION_PARAM_NAME)) {
            FileCompressionType ctype =
                getCompressionType(getParamTextValue(COMPRESSION_PARAM_NAME, 0));
            if (ctype == null) {
              throw new CLIProcessingException("Invalid input values. " +
                  "The entered value for compression is not a supported type.");
            }
            replBuilder.setCompressonwire(ctype);
          } else if (props.getAttr().getIsMarlinTable()) {
            // For marlin streams, use the compression of 'messages' family
            // if not specified.
            FileCompressionType ctype = getStreamCompression(mfs, path);
            replBuilder.setCompressonwire(ctype);
          }

          if (isSync) {
            if (isReplicaTypeTable)
              replBuilder.setSynchronous(getParamBooleanValue(SYNCHRONOUS_PARAM_NAME, 0));
            else
              throw new CLIProcessingException("Can use synchronous replication only when the replica is a MapRDB table.");
          }

          if(!isReplicaTypeTable) {
            rCName = verifyExternalDstSanity(mfs, replicaFullPath);
            replBuilder.setExternal(true);
          }

          String clist = colList;
          if (colList == null && isMarlinStream) {
            clist = getStreamDefaultCFsForRepl();
          }

          //Create replica
          setupReplication(dbPerm,replBuilder,tablePath,replicaFullPath,clist,(String)null,getUserLoginId(),mfs,rCName);
        }
      };
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }
  }

  public static void validateIndexedFields(String indexedColList,
      String tablePath, String user) throws CLIProcessingException {
    List<ColumnFamilyAttr> familyAttrs;
    try {
      familyAttrs = getAllFamilies(tablePath, user);
    } catch (IOException e) {
      throw new CLIProcessingException(e.getMessage());
    }

    for (String indexedFields : indexedColList.split(MULTI_ARG_SEP, -1)) {
      for (String indexedField : indexedFields.split(MULTI_FIELD_SEP)) {
        int separatorIndex = indexedField.indexOf(COLUMN_SEP);
        if (separatorIndex == -1) {
          throw new CLIProcessingException(String.format("Invalid indexed column '%s'.\n%s", indexedField,
              "Indexed column must be specified in <column_family>:<column_qualifier> format."));
        }
        String familyName = indexedField.substring(0, separatorIndex);
        if (familyNameToId(familyAttrs, familyName) == 0) {
          throw new CLIProcessingException(
              String.format("The family '%s' specified in indexed column '%s' does not exist",
                  familyName, indexedField));
        }
      }
    }
  }

  // Parse the columns list argument from the CLI
  public static void parseColList(String colList,
                                  final String tablePath,
                                  final String user,
                                  boolean isJsonTable,
                                  Map<Integer, SortedSet<String>> familyMap /*out*/)
    throws CLIProcessingException {
    if (colList == null || familyMap == null) {
      return;
    }

    String cfColList = colList;

    // Find all the families in the source table
    List<ColumnFamilyAttr> familyAttrs;
    try {
      familyAttrs = getAllFamilies(tablePath, user);
    } catch (IOException e) {
      throw new CLIProcessingException(e.getMessage());
    }

    //Translate JSON Paths list to CF:Qualifier list
    if (isJsonTable) {
      String[] jsonPaths = colList.split(MULTI_ARG_SEP, -1);
      String jsonColList = null;
      for (String jsonPath : jsonPaths) {
      String cfQual = MapRDBTableImplHelper.jsonPathToCfQualifier(jsonPath, familyAttrs);
        if (jsonColList != null) {
          jsonColList += MULTI_ARG_SEP;
          jsonColList += cfQual;
        } else {
          jsonColList = cfQual;
        }
      }
      cfColList = jsonColList;
    }

    LOG.trace("Replication Columns: " + cfColList);

    for (String entry : cfColList.split(MULTI_ARG_SEP)) {
      String[] names = entry.split(COLUMN_SEP, 2);
      int famId = familyNameToId(familyAttrs, names[0]);
      SortedSet<String> cols = familyMap.get(famId);

      if (famId == 0) {
        throw new CLIProcessingException("counld not get familyId for " +
            "columnFamily " + names[0]);
      }

      if (cols == null) {
        cols = new TreeSet<String>();
        familyMap.put(famId, cols);
        //add column to list
        if (names.length > 1) {
          cols.add(names[1]);
        }
      } else {  //list exists
        if (names.length > 1 && !cols.isEmpty()) {
          cols.add(names[1]);
        } else {
          //entire column family needs to be included,
          //not specific columns under it
          if (!cols.isEmpty()) {
            cols.clear();
          }
        }
      }
    }
  }

  // For each family in the map, create a Qualifier with the family ID and
  // a sorted list of all the column names.  If no column names are given,
  // all columns in this family will be replicated.
  public static void addCfQualifiers(Map<Integer, SortedSet<String>> familyMap,
      TableReplicaDesc.Builder replBuilder) {
    for (Map.Entry<Integer, SortedSet<String>> entry : familyMap.entrySet()) {
      final Qualifier.Builder qualBuilder = Qualifier.newBuilder();
      qualBuilder.setFamily(entry.getKey());
      for (String colName : entry.getValue()) {
        qualBuilder.addQualifiers(ByteString.copyFrom(colName.getBytes()));
      }
      replBuilder.addQualifiers(qualBuilder.build());
    }
  }

  private static String getStreamDefaultCFsForRepl() {
    MarlinInternalDefaults mdef = MarlinInternalDefaults.getDefaultInstance();

    return mdef.getCfTopicMeta() + "," +
           mdef.getCfMessages() + "," +
           mdef.getCfCursors();
  }

  private FileCompressionType getStreamCompression(MapRFileSystem mfs,
                                                   Path stream)
      throws IOException {

    MapRAdmin madmin = new MapRAdmin(mfs);
    MarlinInternalDefaults mdef = MarlinInternalDefaults.getDefaultInstance();

    ColumnFamilyAttr cfAttr = madmin.getColumnFamily(stream,
                                                     mdef.getCfMessages(),
                                                     false /*wantAce*/);
    return cfAttr.getSchFamily().getCompression();
  }

  public static void setupReplication(final AceHelper.DBPermission dbPerm, final TableReplicaDesc.Builder replBuilder,
      final String tablePath, final String replicaFullPath, final String clist, final String indexedColList,
      final String user, final MapRFileSystem mfs, String replicaClassName) throws CLIProcessingException, IOException {

    final RecentTablesListManager manager = RecentTablesListManagers.getRecentTablesListManagerForUser(user);
    try {

      TableProperties tableProp = mfs.getTableProperties(new Path(tablePath));
      if (tableProp.getUuid() == null || tableProp.getUuid().length == 0) {
        // table before current release.
        throw new IOException("Table " + tablePath + " does not have a uuid." +
                              " Set replperm and retry operation");
      }

      final boolean isReplicaTypeTable = !isExternalDestination(mfs, replicaFullPath);
      if (isReplicaTypeTable) {
        TableProperties replicaTableProp = mfs.getTableProperties(new Path(replicaFullPath));
        if (replicaTableProp.getUuid() == null || replicaTableProp.getUuid().length == 0) {
          // replica table before current release.
          throw new IOException("Replica table " + replicaFullPath + " does not have a uuid." +
                                " Set replperm and retry operation");
        }

        // Set replica table uuid
        replBuilder.setTableUuid(ByteString.copyFrom(replicaTableProp.getUuid()));
      }

      if (tableProp.getAttr().getBulkLoad()) {
        throw new CLIProcessingException("source cannot be in bulkload mode." +
                                         " Path: " + tablePath);
      }

      String colList = clist;
      if (indexedColList != null) {
        validateIndexedFields(indexedColList, tablePath, user);
        replBuilder.setIndexedFields(indexedColList);
        if (colList != null) {
          // if the replicated column list is non-empty,
          // append the indexed columns to it.
          colList += MULTI_ARG_SEP + indexedColList.replace(MULTI_FIELD_SEP, MULTI_ARG_SEP);
        }
      }

      // For marlin streams, if the colList isn't specified, replicate all cfs
      // except "partitionAssigns"
      boolean isJson = tableProp.getAttr().getJson();
      boolean isStream = tableProp.getAttr().getIsMarlinTable();
      if (colList != null && colList.length() > 0) {
        TreeMap<Integer, SortedSet<String>> familyMap = new TreeMap<Integer, SortedSet<String>>();
        parseColList(colList, tablePath, user, (isJson && !isStream), familyMap);

        if (familyMap.isEmpty()) {
          throw new CLIProcessingException("empty list invalid for " +
              "param " + COLUMNS_PARAM_NAME);
        }

        addCfQualifiers(familyMap, replBuilder);
      }

      if (replicaClassName == null) {
        if (tableProp.getAttr().getIsMarlinTable())
          replBuilder.setReplicaClassName(MAPRSTREAM_REPLICA_CLASS_NAME);
        else
          replBuilder.setReplicaClassName(MAPRDB_REPLICA_CLASS_NAME);
      } else {
        replBuilder.setReplicaClassName(replicaClassName);
      }


      Path rpath = new Path(replicaFullPath);
      String clusterName = mfs.getClusterName(rpath.toUri());
      replBuilder.setClusterName(clusterName);

      String replicaPath = mfs.getNameStr(replicaFullPath);
      replBuilder.setTablePath(replicaPath);

      if ((!replBuilder.hasExternal() || !replBuilder.getExternal()) && !mfs.isTable(rpath)) {
        throw new CLIProcessingException("Replica should be a table. Path: " + replicaPath);
      }

      if (clusterName.equals(mfs.getClusterNameUnchecked(tablePath)) &&
          replicaPath.equals(mfs.getNameStr(tablePath))) {
        throw new CLIProcessingException("source and replica tables cannot be same");
      }

      TableReplicaDesc replDesc = replBuilder.build();
      mfs.addTableReplica(new Path(tablePath), replDesc);
      manager.moveToTop(tablePath);
    } catch (Exception e) {
      manager.deleteIfNotExist(tablePath, mfs);
      throw new CLIProcessingException(e.getMessage());
    }
  }

  public static void copyTable(final String tablePath,
      final String replicaFullPath,
      final int maxVersions,
      final String colList, final String user, final boolean waitForCompletion,
      final String ticketPath, final boolean isMultiMaster, final AceHelper.DBPermission dbPerm, final TableReplicaDesc.Builder replBuilder) throws CLIProcessingException {
    //Call copy table command, resume the replication
    ExecutorService execService = JobExecutor.getExecutorService();
    CopyTableCallable copyTableTask = new CopyTableCallable(tablePath,replicaFullPath,user, maxVersions, colList,ticketPath,isMultiMaster,dbPerm,replBuilder);
    LOG.info("Calling copy table for tables: "+tablePath+","+replicaFullPath+" by user: "+user);
    Future<Integer> status = execService.submit(copyTableTask);
    if(waitForCompletion) {
      try {
        if(status.get()==null) throw new CLIProcessingException("Copy Table failed for tables: "+tablePath+","+replicaFullPath);
      } catch(Exception e) {
        throw new CLIProcessingException(e.getMessage());
      }
    }
  }

  private boolean isExternalReplicaClass(String className) {
    if (className.equals(MAPRDB_REPLICA_CLASS_NAME) ||
        className.equals(MAPRSTREAM_REPLICA_CLASS_NAME))
      return false;
    else
      return true;
  }

  protected void listReplica(OutputHierarchy out) throws CLIProcessingException {
    final boolean refreshNow = (isParamPresent(REFRESH_PARAM_NAME) ?
                                  getParamBooleanValue(REFRESH_PARAM_NAME, 0) :
                                  false);
    final String tablePath = DbCommands.getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId());
    final RecentTablesListManager manager = RecentTablesListManagers.getRecentTablesListManagerForUser(getUserLoginId());
    final List<TableReplicaListResponse> rlist = new ArrayList<TableReplicaListResponse>();
    final MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
    try {
      // Call impersonation
      new FileclientRun(getUserLoginId()) {
        @Override
        public void runAsProxyUser() throws IOException, CLIProcessingException{
          MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
          try {
            if (!mfs.isTable(new Path(tablePath))) {
              throw new CLIProcessingException("Table not found. Path: " + tablePath);
            }

            TableReplicaListResponse resp = mfs.listTableReplicas(new Path(tablePath),
                                                                  true /*wantStats*/,
                                                                  refreshNow);
            rlist.add(resp);
            manager.moveToTop(tablePath);
          } catch (IOException e) {
            manager.deleteIfNotExist(tablePath, mfs);
            throw new CLIProcessingException(e.getMessage());
          }
        }
      };

      final boolean isJson = mfs.isJsonTable(new Path(tablePath));
      TableReplicaListResponse listResp = rlist.get(0);
      List<ColumnFamilyAttr> familyAttrs = getAllFamilies(tablePath,getUserLoginId());
      for (TableReplicaDesc rd : listResp.getReplicasList()) {
        boolean isExternal = isExternalReplicaClass(rd.getReplicaClassName());
        OutputNode replNode = new OutputNode();

        replNode.addChild(new OutputNode("cluster", rd.getClusterName()));
        if (!isExternal) {
          replNode.addChild(new OutputNode(entityName(), rd.getTablePath()));
          replNode.addChild(new OutputNode("type", rd.getReplicaClassName()));
        } else {
          replNode.addChild(new OutputNode(entityName(), GatewaySink.getDestinationName(rd.getTablePath())));
          replNode.addChild(new OutputNode("type", GatewaySink.getDestinationType(rd.getTablePath())));
        }
        replNode.addChild(new OutputNode("realTablePath", rd.getTablePath()));
        replNode.addChild(new OutputNode("paused", rd.getIsPaused()));
        replNode.addChild(new OutputNode("throttle", rd.getThrottle()));
        replNode.addChild(new OutputNode("idx", rd.getIdx()));
        replNode.addChild(new OutputNode("networkencryption", rd.getEncryptonwire()));
        replNode.addChild(new OutputNode("synchronous", rd.getSynchronous()));
        replNode.addChild(new OutputNode("networkcompression",
                                         getCompressionName(rd.getCompressonwire())));

        if (entityName().equals("table") && //don't expose cf details in streams
            rd.getQualifiersCount() > 0) {
          String famList = "";
          int idx = 0;

          for (Qualifier qual : rd.getQualifiersList()) {
            String famName = familyIdToName(familyAttrs, qual.getFamily());
            if (famName == null) {
              LOG.error("failed to get family name for id " + qual.getFamily());
            }

            if (qual.getQualifiersCount() > 0) {
              for (ByteString bstr : qual.getQualifiersList()) {
                if (idx == 0) {
                  if (isJson) {
                    FieldPath jsonFP = MapRDBTableImplHelper.cfQualifierToJsonPath(famName, bstr.toStringUtf8(), familyAttrs);
                    famList = jsonFP.asPathString(false /*quoted*/);
                  } else {
                    famList = famName + COLUMN_SEP + bstr.toStringUtf8();
                  }
                } else {
                  famList = famList + MULTI_ARG_SEP +
                    (isJson ? MapRDBTableImplHelper.cfQualifierToJsonPath(famName, bstr.toStringUtf8(), familyAttrs) :
                     (famName + COLUMN_SEP + bstr.toStringUtf8()));
                }
                ++idx;
              }
            } else if (idx == 0) {
              if (isJson) {
                FieldPath jsonFP = MapRDBTableImplHelper.cfQualifierToJsonPath(famName, null, familyAttrs);
                famList = jsonFP.asPathString(false /*quoted*/);
              } else {
                famList = famName;
              }
              ++idx;
            } else {
              famList = famList + MULTI_ARG_SEP + (isJson ? MapRDBTableImplHelper.cfQualifierToJsonPath(famName, null, familyAttrs) : famName);
              ++idx;
            }
          }
          replNode.addChild(new OutputNode("Columns", famList));
        }

        String replicaTablePath = DbReplicaCommands.getTransformedTablePath(rd.getClusterName(), rd.getTablePath());
        //Show copy table status
        if(JobExecutor.getStatus(tablePath+"_"+replicaTablePath) != null)
        {
          replNode.addChild(new OutputNode("copytablestatus",JobExecutor.getStatus(tablePath+"_"+replicaTablePath)));
        }

        long minPendingTS = 0;
        long maxPendingTS = 0;
        long bytesPending = 0;
        long putsPending = 0;
        int bucketsPending = 0;
        int asyncBuckets = 0;
        List<ExtendedError> elist = null;

        for (TableReplicaStatus rs : listResp.getReplicaStatusList()) {
          if (rd.getIdx() == rs.getReplicaIdx()) {
            minPendingTS = rs.getMinPendingTS();
            maxPendingTS = rs.getMaxPendingTS();
            bytesPending = rs.getBytesPending();
            putsPending = rs.getPutsPending();
            bucketsPending = rs.getBucketsPending();
            asyncBuckets = rs.getAsyncBuckets();

            for (ExtendedError ee : rs.getEerrorsList()) {
              if (elist == null)
                elist = new ArrayList<ExtendedError>();
              elist.add(ee);
            }
            break;
          }
        }
        replNode.addChild(new OutputNode("isUptodate", (bucketsPending == 0 &&
                                                        !rd.getIsPaused())));
        replNode.addChild(new OutputNode("minPendingTS", minPendingTS));
        replNode.addChild(new OutputNode("maxPendingTS", maxPendingTS));
        replNode.addChild(new OutputNode("bytesPending", bytesPending));
        if (entityName().equals("table")) {
          //puts don't make sense for streams
          replNode.addChild(new OutputNode("putsPending", putsPending));
        }
        replNode.addChild(new OutputNode("bucketsPending", bucketsPending));

        byte[] uuid = rd.getTableUuid().toByteArray();
        replNode.addChild(new OutputNode("uuid",
                                         BinaryString.toUUIDString(uuid)));
        if (rd.getSynchronous()) {
          replNode.addChild(new OutputNode("numAsyncBuckets", asyncBuckets));
          replNode.addChild(new OutputNode("switchedToAsync", (asyncBuckets != 0)));
        }

        if (elist != null) {
          for (ExtendedError ee : elist) {
            OutputNode errNodes = new OutputNode("errors");
            replNode.addChild(errNodes);

            if (ee.hasEcode()) {
              errNodes.addChild(new OutputNode("Code", ee.getEcode().name()));
            }
            if (ee.hasEhost()) {
              errNodes.addChild(new OutputNode("Host",
                                               Util.intToIp(ee.getEhost())));
            }
            if (ee.hasEmsg()) {
              errNodes.addChild(new OutputNode("Msg", ee.getEmsg()));
            }
          }
        }
        out.addNode(replNode);
      }
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }
  }

  public static String getTransformedTablePath(String clusterName, String tableName)
  {
    if (!clusterName.equals(CLDBRpcCommonUtils.getInstance().getCurrentClusterName()))
        return "/mapr/" + clusterName + tableName;
    else
        return tableName;
  }

  protected void editReplica(OutputHierarchy out) throws CLIProcessingException {
    final String tablePath = DbCommands.getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId());
    final String replicaFullPath = DbCommands.getTransformedPath(getParamTextValue(REPLICA_PARAM_NAME, 0), getUserLoginId());
    final TableReplicaDesc.Builder replBuilder = TableReplicaDesc.newBuilder();
    final int replDescSerializedSize = replBuilder.build().getSerializedSize();
    final boolean isSync = isParamPresent(SYNCHRONOUS_PARAM_NAME) ? getParamBooleanValue(SYNCHRONOUS_PARAM_NAME, 0) : false;

    try {
      // Call impersonation
      new FileclientRun(getUserLoginId()) {
        @Override
        public void runAsProxyUser() throws IOException, CLIProcessingException {
          MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
          String replicaClusterName = mfs.getClusterNameUnchecked(replicaFullPath);
          String replicaPath = mfs.getNameStr(replicaFullPath);
          final boolean isReplicaTypeTable = !isExternalDestination(mfs, replicaFullPath);

          if (isParamPresent(NEWREPLICA_PARAM_NAME)) {

            /*
             * If existing replica is a not a table and we are replicating to an
             * external sink config file, then such a replica can not be renamed.
             */
            TableReplicaDesc rd = getReplica(tablePath, replicaClusterName,
                replicaPath);
            if (rd == null)
              throw new IOException("Replica not found");

            if (rd.hasExternal() && rd.getExternal()) {
              throw new CLIProcessingException("Can not rename an external replica");
            }

            final String newReplicaFullPath = DbCommands.getTransformedPath(getParamTextValue(NEWREPLICA_PARAM_NAME, 0), getUserLoginId());
            Path newRpath = new Path(newReplicaFullPath);
            String newReplicaClusterName = mfs.getClusterName(newRpath.toUri());
            replBuilder.setClusterName(newReplicaClusterName);

            String newReplicaPath = mfs.getNameStr(newReplicaFullPath);
            replBuilder.setTablePath(newReplicaPath);

            if (!mfs.isTable(new Path(tablePath))) {
              throw new CLIProcessingException("Table not found. Path: " + tablePath);
            }

            if (!mfs.isTable(newRpath)) {
              throw new CLIProcessingException("New Replica not found. Path: " + newReplicaPath);
            }

            if (newReplicaClusterName.equals(mfs.getClusterNameUnchecked(tablePath)) &&
                newReplicaPath.equals(mfs.getNameStr(tablePath))) {
              throw new CLIProcessingException("source and replica tables cannot be same");
            }

            // Get the uuid of replica table
            TableProperties rtableProp = mfs.getTableProperties(newRpath);
            if (rtableProp.getUuid() == null || rtableProp.getUuid().length == 0)
              throw new IOException("replica table does not have a uuid");

            replBuilder.setTableUuid(ByteString.copyFrom(rtableProp.getUuid()));
          }

          if (isParamPresent(COMPRESSION_PARAM_NAME)) {
            boolean updateCompression = false;
            String compressionStr = getParamTextValue(COMPRESSION_PARAM_NAME, 0);
            if (compressionStr.equals("on")) {
              TableReplicaDesc rd = getReplica(tablePath, replicaClusterName,
                  replicaPath);
              if (rd == null)
                throw new IOException("Replica not found");

              if (!rd.hasCompressonwire() ||
                  rd.getCompressonwire() == FileCompressionType.FCT_OFF)
                updateCompression = true;
            } else {
              updateCompression = true;
            }

            if (updateCompression) {
              FileCompressionType ctype = getCompressionType(compressionStr);
              if (ctype == null) {
                throw new CLIProcessingException("Invalid input values. " +
                    "The entered value for compression is not a supported type.");
              }
              replBuilder.setCompressonwire(ctype);
            }
          }

          // ' ' for COLUMNS_PARAM_NAME denotes allowAllCfs
          final boolean allowAllCfs = isParamPresent(COLUMNS_PARAM_NAME) ?
              getParamTextValue(COLUMNS_PARAM_NAME, 0).trim().length() == 0 : false;
          if (!allowAllCfs && isParamPresent(COLUMNS_PARAM_NAME)) {
            String colList = getParamTextValue(COLUMNS_PARAM_NAME, 0);
            TreeMap<Integer, SortedSet<String>> familyMap = new TreeMap<Integer, SortedSet<String>>();
            String user = getUserLoginId();
            final String tablePath = DbCommands.getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), user);
            TableProperties srcTableProp = mfs.getTableProperties(new Path(tablePath));
            boolean isJson = srcTableProp.getAttr().getJson();
            boolean isStream = srcTableProp.getAttr().getIsMarlinTable();
            parseColList(colList,tablePath,user, (isJson && !isStream), familyMap);
            addCfQualifiers(familyMap, replBuilder);
          }

          if (isParamPresent(THROTTLE_PARAM_NAME))
            replBuilder.setThrottle(getParamBooleanValue(THROTTLE_PARAM_NAME, 0));

          if (isParamPresent(ENCRYPTION_PARAM_NAME))
            replBuilder.setEncryptonwire(getParamBooleanValue(ENCRYPTION_PARAM_NAME, 0));

          if (isParamPresent(SYNCHRONOUS_PARAM_NAME)) {
            if (isSync && !isReplicaTypeTable)
              throw new CLIProcessingException("Can setup synchronous replication only when the replica is a MapRDB table.");

            replBuilder.setSynchronous(isSync);
          }

          if (replDescSerializedSize == replBuilder.build().getSerializedSize() && !allowAllCfs) {
            // No paramter changed. Silently return
            return;
          }

          // Set as final to be called in impersonation
          final RecentTablesListManager manager = RecentTablesListManagers.getRecentTablesListManagerForUser(getUserLoginId());
          try {
            if (!mfs.isTable(new Path(tablePath))) {
              throw new CLIProcessingException("Table not found. Path: " + tablePath);
            }

            TableReplicaDesc replDesc = replBuilder.build();

            mfs.editTableReplica(new Path(tablePath), replicaClusterName,
                                 replicaPath, allowAllCfs, replDesc);
            manager.moveToTop(tablePath);
          } catch (IOException e) {
            manager.deleteIfNotExist(tablePath, mfs);
            throw new CLIProcessingException(e.getMessage());
          }
        }
      };
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
      return;
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
      return;
    }
  }

  protected void removeReplica(OutputHierarchy out) throws CLIProcessingException {
    final String tablePath = DbCommands.getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId());
    final String replicaFullPath = DbCommands.getTransformedPath(getParamTextValue(REPLICA_PARAM_NAME, 0), getUserLoginId());
    final TableReplicaDesc.Builder replBuilder = TableReplicaDesc.newBuilder();

    // Set as final to be called in impersonation
    final RecentTablesListManager manager = RecentTablesListManagers.getRecentTablesListManagerForUser(getUserLoginId());
    try {
      // Call impersonation
      new FileclientRun(getUserLoginId()) {
        @Override
        public void runAsProxyUser() throws IOException, CLIProcessingException{
          MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
          try {
            Path rpath = new Path(replicaFullPath);
            String clusterName = mfs.getClusterNameUnchecked(replicaFullPath);
            replBuilder.setClusterName(clusterName);

            String replicaPath = mfs.getNameStr(replicaFullPath);
            replBuilder.setTablePath(replicaPath);

            TableReplicaDesc replDesc = replBuilder.build();

            if (!mfs.isTable(new Path(tablePath))) {
              throw new CLIProcessingException("Table not found. Path: " + tablePath);
            }

            mfs.removeTableReplica(new Path(tablePath), replDesc);
            manager.moveToTop(tablePath);
          } catch (IOException e) {
            manager.deleteIfNotExist(tablePath, mfs);
            throw new CLIProcessingException(e.getMessage());
          }
        }
      };
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }
  }

  public static void pauseOrResumeReplication(final String tablePath, final String replicaFullPath, boolean isPaused,String loginId) throws IOException,CLIProcessingException
  {
    final TableReplicaDesc.Builder replBuilder = TableReplicaDesc.newBuilder();
    replBuilder.setIsPaused(isPaused);

    // Set as final to be called in impersonation
    final RecentTablesListManager manager = RecentTablesListManagers.getRecentTablesListManagerForUser(loginId);
    try {
      // Call impersonation
      new FileclientRun(loginId) {
        @Override
        public void runAsProxyUser() throws IOException, CLIProcessingException {
          final MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
          try {
            Path rpath = new Path(replicaFullPath);
            final String replicaClusterName = mfs.getClusterNameUnchecked(replicaFullPath);
            final String replicaPath = mfs.getNameStr(replicaFullPath);

            if (!mfs.isTable(new Path(tablePath))) {
              throw new CLIProcessingException("Table not found. Path: " + tablePath);
            }

            TableReplicaDesc replDesc = replBuilder.build();
            mfs.editTableReplica(new Path(tablePath), replicaClusterName,
                                 replicaPath, false, replDesc);
          } catch (IOException e) {
            manager.deleteIfNotExist(tablePath, mfs);
            throw new CLIProcessingException(e.getMessage());
          }
        }
      };
    }catch(Exception e)
    {
      throw new CLIProcessingException(e.getMessage());
    }
  }

  protected void pauseOrResumeReplica(OutputHierarchy out, boolean isPaused) throws CLIProcessingException {
    final String tablePath = DbCommands.getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId());
    final String replicaFullPath = DbCommands.getTransformedPath(getParamTextValue(REPLICA_PARAM_NAME, 0), getUserLoginId());
        try {
            DbReplicaCommands.pauseOrResumeReplication(tablePath, replicaFullPath, isPaused, getUserLoginId());
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }
  }

  private TableReplicaDesc getReplica(final String tablePath, String replClusterName,
                                      String replTablePath) throws IOException,CLIProcessingException {
    final RecentTablesListManager manager = RecentTablesListManagers.getRecentTablesListManagerForUser(getUserLoginId());
    final List<TableReplicaListResponse> rlist = new ArrayList<TableReplicaListResponse>();
    try {
      // Call impersonation
      new FileclientRun(getUserLoginId()) {
        @Override
        public void runAsProxyUser() throws IOException, CLIProcessingException{
          MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
          try {
            if (!mfs.isTable(new Path(tablePath))) {
              throw new CLIProcessingException("Table not found. Path: " + tablePath);
            }

            TableReplicaListResponse resp = mfs.listTableReplicas(new Path(tablePath),
                                                                  false /*wantStats*/,
                                                                  false /*refreshNow*/);
            rlist.add(resp);
            manager.moveToTop(tablePath);
          } catch (IOException e) {
            manager.deleteIfNotExist(tablePath, mfs);
            throw new CLIProcessingException(e.getMessage());
          }
        }
      };

      TableReplicaListResponse listResp = rlist.get(0);
      for (TableReplicaDesc rd : listResp.getReplicasList()) {
        if (!replClusterName.equals(rd.getClusterName()) ||
            !replTablePath.equals(rd.getTablePath()))
          continue;

        return rd;
      }

      return null;
    } catch (CLIProcessingException e) {
      throw new CLIProcessingException(e.getMessage());
    } catch (IOException e) {
      throw new IOException(e.getMessage());
    }
  }

  public static List<ColumnFamilyAttr> getAllFamilies(String tablePath, final String user) throws IOException, CLIProcessingException {
    final String tablePathFinal = tablePath;
    final List<ColumnFamilyAttr> cfListFinal = new ArrayList<ColumnFamilyAttr>();

    new FileclientRun(user) {
      @Override
      public void runAsProxyUser() throws CLIProcessingException, IOException {
        MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
        List<ColumnFamilyAttr> colList;
        colList = mfs.listColumnFamily(new Path(tablePathFinal), false /*ace*/);
        cfListFinal.addAll(colList);
      }
    };
    return cfListFinal;
  }

  private static int familyNameToId(List<ColumnFamilyAttr> cfAttrList, String familyName) {
    for (ColumnFamilyAttr cf : cfAttrList) {
      if (cf.getSchFamily().getName().equals(familyName))
        return cf.getSchFamily().getId();
    }
    return 0;
  }

  public static String familyIdToName(List<ColumnFamilyAttr> cfAttrList, int familyId) {
    for (ColumnFamilyAttr cf : cfAttrList) {
      if (cf.getSchFamily().getId() == familyId)
        return cf.getSchFamily().getName();
    }
    return null;
  }

  @Override
  public String getCliParam(String key) throws IOException {
    String ret = null;
    // If permissions map val is a passed as parameter, then add to ace list
    try {
      if (isParamPresent(key)) {
        ret = getParamTextValue(key, 0);
      }
    } catch (CLIProcessingException e) {
      throw new IOException(e);
    }
    return ret;
  }
}
