/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cli.table.RecentTablesListManager;
import com.mapr.cli.table.RecentTablesListManagers;
import com.mapr.cli.table.TabletStats;
import com.mapr.cliframework.base.*;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.inputparams.*;
import com.mapr.db.impl.IdCodec;
import com.mapr.fs.MapRFileSystem;
import com.mapr.fs.cldb.proto.CLDBProto.*;
import com.mapr.fs.cldb.util.Util;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Common.Server;
import com.mapr.fs.proto.Dbserver.TabletDesc;
import com.mapr.fs.proto.Dbserver.TabletStatResponse;
import com.mapr.fs.proto.Dbserver.SpaceUsage;
import com.mapr.fs.proto.Dbserver.ForcedCompactionType;
import com.mapr.org.apache.hadoop.hbase.util.Bytes;
import com.mapr.security.MaprSecurityException;

import org.apache.log4j.Logger;
import org.apache.hadoop.fs.Path;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Arrays;
import java.util.HashSet;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Callable;


public class DbRegionCommands extends CLIBaseClass implements CLIInterface {

  private static final Logger LOG = Logger.getLogger(DbRegionCommands.class);
  private static final String PATH_PARAM_NAME = "path";
  private static final String FID_PARAM_NAME = "fid";
  private static final String CTYPE_PARAM_NAME = "type";
  private static final String CTYPE_PARAM_DESC =
    "default|postsplit|sync|partitionSplit";
  private static final String NTHREADS_PARAM_NAME = "nthreads";

  public static final String COLUMNS_PARAM_NAME = "columns";
  public static final String OUTPUT_PARAM_NAME = "output";
  public static final String START_PARAM_NAME = "start";
  public static final String LIMIT_PARAM_NAME = "limit";

  public static final int NUM_REGION_INFOS_PER_RPC = 25;

  private static final CLICommand splitCommand =
    new CLICommand(
         "split",
         "usage: table region split -path <tablepath> -fid <fid>",
         DbRegionCommands.class,
         CLICommand.ExecutionTypeEnum.NATIVE,
         new ImmutableMap.Builder<String, BaseInputParameter>()
             .put(PATH_PARAM_NAME,
                 new TextInputParameter(PATH_PARAM_NAME,
                     "table path",
                     CLIBaseClass.REQUIRED,
                     null))
             .put(FID_PARAM_NAME,
                 new TextInputParameter(FID_PARAM_NAME,
                     "fid",
                     CLIBaseClass.REQUIRED,
                     null))
             .build(), null)
         .setShortUsage("table region split -path <tablepath> -fid <fid>");

  private static final CLICommand packCommand =
    new CLICommand(
         "pack",
         "usage: table region pack -path <tablepath> -fid <fid>",
         DbRegionCommands.class,
         CLICommand.ExecutionTypeEnum.NATIVE,
         new ImmutableMap.Builder<String, BaseInputParameter>()
             .put(PATH_PARAM_NAME,
                 new TextInputParameter(PATH_PARAM_NAME,
                     "table path",
                     CLIBaseClass.REQUIRED,
                     null))
             .put(FID_PARAM_NAME,
                 new TextInputParameter(FID_PARAM_NAME,
                     "fid|all",
                     CLIBaseClass.REQUIRED,
                     null))
             .put(CTYPE_PARAM_NAME,
                  new TextInputParameter(CTYPE_PARAM_NAME,
                      CTYPE_PARAM_DESC,
                      CLIBaseClass.NOT_REQUIRED,
                      null).setInvisible(true))
             .put(NTHREADS_PARAM_NAME,
                  new IntegerInputParameter(NTHREADS_PARAM_NAME,
                      "nthreads",
                      CLIBaseClass.NOT_REQUIRED,
                      16))
             .build(), null)
         .setShortUsage("table region pack -path <tablepath> -fid <fid>");

  private static final CLICommand listCommand =
      new CLICommand(
          "list",
          "usage: table region list -path <tablepath>",
          DbRegionCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "table path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(COLUMNS_PARAM_NAME,
                  new TextInputParameter(COLUMNS_PARAM_NAME,
                      "columns",
                      CLIBaseClass.NOT_REQUIRED,
                      "all").setInvisible(true))
              .put(OUTPUT_PARAM_NAME,
                  new TextInputParameter(OUTPUT_PARAM_NAME,
                      "verbose|terse",
                      CLIBaseClass.NOT_REQUIRED,
                      "verbose").setInvisible(true))
              .put(START_PARAM_NAME,
                  new IntegerInputParameter(START_PARAM_NAME,
                      "start",
                      CLIBaseClass.NOT_REQUIRED,
                      0))
              .put(LIMIT_PARAM_NAME,
                  new IntegerInputParameter(LIMIT_PARAM_NAME,
                      "limit",
                      CLIBaseClass.NOT_REQUIRED,
                      Integer.MAX_VALUE))
              .build(), null)
          .setShortUsage("table region list -path <tablepath>");

  private static final CLICommand mergeCommand =
    new CLICommand(
         "merge",
         "usage: table region merge -path <tablepath> -fid <fid>",
         DbRegionCommands.class,
         CLICommand.ExecutionTypeEnum.NATIVE,
         new ImmutableMap.Builder<String, BaseInputParameter>()
             .put(PATH_PARAM_NAME,
                 new TextInputParameter(PATH_PARAM_NAME,
                     "table path",
                     CLIBaseClass.REQUIRED,
                     null))
             .put(FID_PARAM_NAME,
                 new TextInputParameter(FID_PARAM_NAME,
                     "fid",
                     CLIBaseClass.REQUIRED,
                     null))
             .build(), null)
         .setShortUsage("table region -path <tablepath> merge -fid <fid>");

  // main command
  public static final CLICommand regionCommands =
      new CLICommand(
          "region", "region [split|pack|merge|list]",
          CLIUsageOnlyCommand.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new CLICommand[]{
              splitCommand,
              packCommand,
              mergeCommand,
              listCommand
          }
      ).setShortUsage("table region [split|pack|merge|list]");

  private static final Map<String, String> verboseToTerseMap = new ImmutableMap.Builder<String, String>()
      .put("startkey", "sk")
      .put("endkey", "ek")
      .put("size", "s")
      .put("numrows", "nr")
      .put("numberofrowswithdelete", "nrd")
      .put("primarymfs", "pn")
      .put("secondarymfs", "sn")
      .put("lastheartbeat", "lhb")
      .put("physicalsize", "ps")
      .put("logicalsize", "ls")
      .put("numberofrows", "nr")
      .put("fid", "fid")
      .put("copypendingsize", "cps")
      .put("numberofspills", "nsp")
      .put("numberofsegments", "nsg")
      .build();


  public DbRegionCommands(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }

  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    if (!super.validateInput()) {
      return output;
    }

    if (cliCommand.getCommandName().equalsIgnoreCase(splitCommand.getCommandName())) {
      splitRegion(out);
    } else if (cliCommand.getCommandName().equalsIgnoreCase(packCommand.getCommandName())) {
      packRegion(out);
    } else if (cliCommand.getCommandName().equalsIgnoreCase(mergeCommand.getCommandName())) {
      mergeRegion(out);
    } else if (cliCommand.getCommandName().equalsIgnoreCase(listCommand.getCommandName())) {
      listRegions(out);
    }

    return output;
  }

  private void splitRegion(OutputHierarchy out) throws CLIProcessingException {
    String tablePath =
      DbCommands.getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0),
                                    getUserLoginId());
    String fidstr = getParamTextValue(FID_PARAM_NAME, 0);
    MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
    RecentTablesListManager manager =
      RecentTablesListManagers.getRecentTablesListManagerForUser(getUserLoginId());
    try {
      mfs.splitTableRegion(new Path(tablePath), fidstr, false /*ignoreRegionTooSmallError*/);
      manager.moveToTop(tablePath);
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
      manager.deleteIfNotExist(tablePath, mfs);
    }
  }

  private static OutputError invokePackRegionOnTabletFid(
		                                   final String tablePath,
		                                   final String fidstr,
		                                   int ctype)
		  throws CLIProcessingException {
    MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
    try {
      mfs.packTableRegion(new Path(tablePath), fidstr, ctype);
    } catch (IOException e) {
      return new OutputError(Errno.EOPFAILED, e.getMessage());
    }
    return new OutputError(Errno.SUCCESS, null);
  }

  private static List<TabletDesc> getAllTabletDescriptors(String tablePath,
                                                          String username,
                                                          OutputHierarchy out) 
          throws CLIProcessingException {
    //get all tablets
    int start = 0;
    int limit = Integer.MAX_VALUE;
    TabletStats tabletStats = new TabletStats(tablePath, username);
    List<TabletDesc> tablets = tabletStats.getTablets(out, start, limit);
    return tablets;
  }

  public static OutputError packAllRegionsOfTable(final String tablePath,
                                                  final int ctype,
                                                  final int nthreads,
                                                  final String username)
      throws CLIProcessingException {

    OutputHierarchy out = new OutputHierarchy();
    List<TabletDesc> tablets = getAllTabletDescriptors(tablePath, username,
                                                       out);
    if (tablets == null) {
      if (out.getOutputErrors() != null && out.getOutputErrors().size() > 1)
        return out.getOutputErrors().get(0);
      else
        return new OutputError(Errno.EOPFAILED,
                               "Error fetching regions for " + tablePath);
    }
      
    ExecutorService executor = Executors.newFixedThreadPool(nthreads);
    List<Future<OutputError>> fObj = new ArrayList<Future<OutputError>>();

    for (TabletDesc tablet : tablets) {
      final String fid = MapRCliUtil.getFidAsString(tablet.getFid());
      Callable<OutputError> packCommand = new Callable<OutputError>() {
        public OutputError call() throws CLIProcessingException {
          OutputError retval = null;
          try {
            retval = invokePackRegionOnTabletFid(tablePath, fid, ctype);
          } catch(CLIProcessingException e) {
            throw e;
          }
          return retval;
        }
      };
      Future<OutputError> fRet = executor.submit(packCommand);
      fObj.add(fRet);
    }

    int fObjSize = fObj.size();
    //in this list, we keep all returned OutputError object. When we are done,
    //we will check for error code in each and return the first error
    //condition we find.
    List<OutputError> errorCodes = new ArrayList<OutputError>();
    try {
      for (int i = 0; i < fObjSize; i++) {
        OutputError retval = fObj.get(i).get();
        if (retval.getErrorCode() != Errno.SUCCESS) {
           errorCodes.add(retval);
        }
      }
    } catch(Exception e) {
      return new OutputError(Errno.EOPFAILED,
                            " wait for threadpool to finish failed " +
                            e.getMessage());
    }
    executor.shutdown();

    if (errorCodes.isEmpty()) {
      return null;
    } else {
      return errorCodes.get(0);
    }
  }

  private void packRegion(OutputHierarchy out) throws CLIProcessingException {
    String tablePath =
      DbCommands.getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0),
                                    getUserLoginId());
    String fidstr = null;
    fidstr = getParamTextValue(FID_PARAM_NAME, 0);

    int ctype = ForcedCompactionType.ForcedCompactionDefault.getNumber();
    if (isParamPresent(CTYPE_PARAM_NAME)) {
      String ctypeStr = getParamTextValue(CTYPE_PARAM_NAME, 0);
      if (ctypeStr.equalsIgnoreCase("default")) {
        ctype = ForcedCompactionType.ForcedCompactionDefault.getNumber();
      } else if (ctypeStr.equalsIgnoreCase("postSplit")) {
        ctype = ForcedCompactionType.ForcedCompactionPostSplit.getNumber();
      } else if (ctypeStr.equalsIgnoreCase("sync")) {
        ctype = ForcedCompactionType.ForcedCompactionSync.getNumber();
      } else if (ctypeStr.equalsIgnoreCase("partitionSplit")) {
        ctype = ForcedCompactionType.ForcedCompactionPartitionSplit.getNumber();
      } else if (ctypeStr.equalsIgnoreCase("ttl")) {
        ctype = ForcedCompactionType.ForcedCompactionTTL.getNumber();
      } else {
        out.addError(new OutputError(Errno.EINVAL, "unknown type " + ctypeStr));
        return;
      }
    }

    RecentTablesListManager manager =
    RecentTablesListManagers.getRecentTablesListManagerForUser(getUserLoginId());
    manager.moveToTop(tablePath);
    OutputError err = null;
    if (fidstr.equals("all")) {
      int nthreads = getParamIntValue(NTHREADS_PARAM_NAME, 0);
      err = packAllRegionsOfTable(tablePath, ctype, nthreads, getUserLoginId());
    } else {
      err = invokePackRegionOnTabletFid(tablePath, fidstr, ctype);
    }
    if ((err != null) && (err.getErrorCode() != Errno.SUCCESS)) {
      MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
      manager.deleteIfNotExist(tablePath, mfs);
      out.addError(err);
    }

  }

  private boolean showColumn(Set<String> columns, String verboseName) {
      if (columns == null)
          return false;
      String terseName = verboseToTerseMap.get(verboseName);
      return columns.contains("all") || columns.contains("verboseName") || columns.contains(terseName);
  }

  private void listRegions(OutputHierarchy out) throws CLIProcessingException {
    final String path = DbCommands.getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId());
    int start = getParamIntValue(START_PARAM_NAME, 0);
    int limit = getParamIntValue(LIMIT_PARAM_NAME, 0);

    final MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
    RecentTablesListManager manager = RecentTablesListManagers.getRecentTablesListManagerForUser(getUserLoginId());
    // Call helper methods to populate data

    TabletStats tabletStats = new TabletStats(path, getUserLoginId());

    List<TabletDesc> tablets = tabletStats.getTablets(out, start, limit);
    // Tablets had an error, return
    if (tablets == null) {
      return;
    }

    boolean isJson = isJsonTable(path);
    String columnsString = getParamTextValue(COLUMNS_PARAM_NAME, 0);
    Set<String> columns = null;
    if (columnsString != null)
        columns = new HashSet<String>(Arrays.asList(columnsString.split(",")));

    Map<Integer, TabletInfo> cidMap = getTabletsInfo(out, tablets);
    for (TabletDesc tablet : tablets) {

      TabletInfo tabletInfo = cidMap.get(tablet.getFid().getCid());
      String primaryMfs = tabletInfo.getMaster().getIpsList().get(0).getHostname() + ":" +
    		  tabletInfo.getMaster().getIpsList().get(0).getPort();
      String secondaryMfs = getSecondaryMfs(tabletInfo);

      byte[] startKey = tablet.getStartKey().toByteArray();
      byte[] endKey = tablet.getEndKey().toByteArray();
      String startKeyStr = "-INFINITY";
      String endKeyStr = "INFINITY";
      if (startKey.length != 0) {
        startKeyStr = isJson ? IdCodec.asString(IdCodec.decode(startKey)) : Bytes.toStringBinary(startKey);
      }
      if (endKey.length != 0) {
        endKeyStr = isJson ? IdCodec.asString(IdCodec.decode(endKey)) : Bytes.toStringBinary(endKey);
      }

      OutputNode tabletNode = new OutputNode();
      if(showColumn(columns, "primarymfs"))
        tabletNode.addChild(new OutputNode(getOutputFieldName("primarymfs"), primaryMfs));
      if(showColumn(columns, "secondarymfs"))
        tabletNode.addChild(new OutputNode(getOutputFieldName("secondarymfs"), secondaryMfs));
      if(showColumn(columns, "startkey"))
        tabletNode.addChild(new OutputNode(getOutputFieldName("startkey"), startKeyStr));
      if(showColumn(columns, "endkey"))
        tabletNode.addChild(new OutputNode(getOutputFieldName("endkey"), endKeyStr));
      if(showColumn(columns, "lastheartbeat"))
        tabletNode.addChild(new OutputNode(getOutputFieldName("lastheartbeat"), tabletInfo.getMastersLastHBSec()));
      if(showColumn(columns, "fid"))
        tabletNode.addChild(new OutputNode(getOutputFieldName("fid"), MapRCliUtil.getFidAsString(tablet.getFid())));

      try {
        // blocks until a response is available
        TabletStatResponse tsr = tabletStats.getTabletStatResponse(tablet);
        if (tsr != null && tsr.hasUsage()) {
          SpaceUsage su = tsr.getUsage();
          long blockSize = 8 * 1024L;
          if(showColumn(columns, "logicalsize"))
            tabletNode.addChild(new OutputNode(getOutputFieldName("logicalsize"), su.getNumLogicalBlocks() * blockSize));
          if(showColumn(columns, "physicalsize"))
            tabletNode.addChild(new OutputNode(getOutputFieldName("physicalsize"), su.getNumPhysicalBlocks() * blockSize));
          if (showColumn(columns, "copypendingsize") && su.hasNumRemoteBlocks())
            tabletNode.addChild(new OutputNode(getOutputFieldName("copypendingsize"), su.getNumRemoteBlocks() * blockSize));
          if(showColumn(columns, "numberofrows"))
            tabletNode.addChild(new OutputNode(getOutputFieldName("numberofrows"), su.getNumRows()));
          if (showColumn(columns, "numberofrowswithdelete") && su.hasNumRowsWithDelete())
            tabletNode.addChild(new OutputNode(getOutputFieldName("numberofrowswithdelete"), su.getNumRowsWithDelete()));
          if (showColumn(columns, "numberofspills") && su.hasNumSpills())
            tabletNode.addChild(new OutputNode(getOutputFieldName("numberofspills"), su.getNumSpills()));
          if (showColumn(columns, "numberofsegments") && su.hasNumSegments())
            tabletNode.addChild(new OutputNode(getOutputFieldName("numberofsegments"), su.getNumSegments()));
        }
      } catch (Exception e) {
        LOG.error("Error fetching tablet stats for fid: " + MapRCliUtil.getFidAsString(tablet.getFid()), e);
      }
      out.addNode(tabletNode);
    }
    manager.moveToTop(path);
  }

  private boolean isJsonTable(String path) throws CLIProcessingException {
    try {
      return MapRCliUtil.getMapRFileSystem().getTableProperties(new Path(path)).getAttr().getJson();
    } catch (IllegalArgumentException | IOException e) {
      throw new CLIProcessingException(e);
    }
  }

  private Map<Integer, TabletInfo> getTabletsInfo(OutputHierarchy out, List<TabletDesc> tablets)
      throws CLIProcessingException {
    Set<Integer> cidSet = Sets.newHashSet();
    for (TabletDesc tablet : tablets) {
      cidSet.add(tablet.getFid().getCid());
    }

    Map<Integer, TabletInfo> cidToTabletInfoMap = Maps.newHashMap();
    int from = 0;
    int to = NUM_REGION_INFOS_PER_RPC;

    List<Integer> cidList = Lists.newArrayList(cidSet);
    while (to < cidList.size()) {
      TabletInfoRequest.Builder tabletInfoReqBuilder = TabletInfoRequest.newBuilder()
        .setCreds(getUserCredentials());
      tabletInfoReqBuilder.addAllContainerId(cidList.subList(from, to));
      fetchTabletsInfoFromCLDB(tabletInfoReqBuilder, out, cidToTabletInfoMap);

      if (!out.getOutputErrors().isEmpty()) {
        return cidToTabletInfoMap; // fail fast at first failure.
      }

      from = to;
      to = from + NUM_REGION_INFOS_PER_RPC;
    }

    TabletInfoRequest.Builder tabletInfoReqBuilder = TabletInfoRequest.newBuilder()
        .setCreds(getUserCredentials());
    tabletInfoReqBuilder.addAllContainerId(cidList.subList(from, cidList.size()));
    fetchTabletsInfoFromCLDB(tabletInfoReqBuilder, out, cidToTabletInfoMap);

    return cidToTabletInfoMap;
  }

  private void fetchTabletsInfoFromCLDB(TabletInfoRequest.Builder request, OutputHierarchy out,
                                        Map<Integer, TabletInfo> results) throws CLIProcessingException {
    try {
      byte[] replyData;
      String cluster = MapRCliUtil.extractClusterNameFromFullyQualifiedPath(getParamTextValue(PATH_PARAM_NAME, 0));
      if (cluster != null) {
        replyData = CLDBRpcCommonUtils.getInstance().sendRequest(cluster,
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProg.TabletInfoProc.getNumber(), request.build(),
            TabletInfoResponse.class);
      } else {
        replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProg.TabletInfoProc.getNumber(), request.build(),
            TabletInfoResponse.class);
      }

      if (replyData == null) {
        out.addError(new OutputError(Errno.ERPCFAILED, "Could not retrieve the list of regions. " +
            "Reason: Could not connect to the CLDB service"));
        return;
      }

      TabletInfoResponse resp = TabletInfoResponse.parseFrom(replyData);
      if (resp.getStatus() != Errno.SUCCESS) {
        out.addError(new OutputError(resp.getStatus(), "Could not retrieve the list of regions. " +
            "Reason: Region lookup failed. Error: " + Errno.toString(resp.getStatus())));
        return;
      }

      List<TabletInfo> tabletsInfoList = resp.getTabletsList();
      if (tabletsInfoList.isEmpty()) {
        out.addError(new OutputError(resp.getStatus(), "Could not retrieve the list of regions. " +
            "Reason: Region info response did not contain regions information in it."));
        return;
      }
      for (TabletInfo tabletInfo : tabletsInfoList) {
        results.put(tabletInfo.getContainerId(), tabletInfo);
      }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      throw new CLIProcessingException("InvalidProtocolBufferException Exception", e);
    }
  }

  private String getSecondaryMfs(TabletInfo tablet) {
    StringBuilder secondaryMfsBuilder = new StringBuilder();
    boolean first = true;
    for (Server secondaryServers : tablet.getReplicasList()) {
      if (!Util.compareIPAddress(tablet.getMaster().getIps(0), secondaryServers.getIps(0))) {
        if (!first) {
          secondaryMfsBuilder.append(", ");
        }

        String mfs = null;
        if (secondaryServers.hasHostname()) {
          mfs = secondaryServers.getHostname() + ":" + secondaryServers.getIps(0).getPort();
        }
        if (mfs == null || mfs.isEmpty()) {
          mfs = secondaryServers.getIps(0).getHostname()+ ":" + secondaryServers.getIps(0).getPort();
        }
        secondaryMfsBuilder.append(mfs);
        first = false;
      }
    }
    return secondaryMfsBuilder.toString();
  }

  private String getOutputFieldName(String verboseName) throws CLIProcessingException {
    return "terse".equals(getParamTextValue(OUTPUT_PARAM_NAME, 0)) ? verboseToTerseMap.get(verboseName) : verboseName;
  }

  private void mergeRegion(OutputHierarchy out) throws CLIProcessingException {
    String tablePath =
      DbCommands.getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0),
                                    getUserLoginId());
    String fidstr = getParamTextValue(FID_PARAM_NAME, 0);
    MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
    RecentTablesListManager manager =
      RecentTablesListManagers.getRecentTablesListManagerForUser(getUserLoginId());
    try {
      mfs.mergeTableRegion(new Path(tablePath), fidstr);
      manager.moveToTop(tablePath);
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
      manager.deleteIfNotExist(tablePath, mfs);
    }
  }
}
