/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.ByteString;
import com.mapr.baseutils.BinaryString;
import com.mapr.baseutils.Errno;
import com.mapr.cli.table.RecentTablesListManager;
import com.mapr.cli.table.RecentTablesListManagers;
import com.mapr.cliframework.base.*;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.inputparams.*;
import com.mapr.cli.common.FileclientRun;
import com.mapr.cli.common.NodesCommonUtils;
import com.mapr.security.MaprSecurityException;

import com.mapr.fs.Rpc;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.util.Util;
import com.mapr.fs.MapRFileSystem;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Common.ServiceData;
import com.mapr.fs.proto.Common.MapRProgramId;
import com.mapr.fs.proto.Common.IPAddress;
import com.mapr.fs.proto.Dbreplicator.DBReplicatorProg;
import com.mapr.fs.proto.Dbreplicator.GetHostNamesRequest;
import com.mapr.fs.proto.Dbreplicator.GetHostNamesResponse;
import com.mapr.fs.proto.Security.ServerKeyType;
import com.mapr.fs.proto.Error.ExtendedError;
import com.mapr.fs.MapRFileSystem;
import com.mapr.fs.ClusterConf;
import com.google.protobuf.InvalidProtocolBufferException;
import com.mapr.fs.cldb.proto.CLDBProto.DBReplGatewayConfigGetRequest;
import com.mapr.fs.cldb.proto.CLDBProto.DBReplGatewayConfigGetResponse;
import com.mapr.fs.cldb.proto.CLDBProto.DBReplGatewayConfigSetRequest;
import com.mapr.fs.cldb.proto.CLDBProto.DBReplGatewayConfigSetResponse;
import com.mapr.fs.cldb.proto.CLDBProto.DBReplGatewayConfigDeleteRequest;
import com.mapr.fs.cldb.proto.CLDBProto.DBReplGatewayConfigDeleteResponse;
import com.mapr.fs.cldb.proto.CLDBProto.DBReplGatewayConfigListRequest;
import com.mapr.fs.cldb.proto.CLDBProto.DBReplGatewayConfigListResponse;

import com.mapr.fs.proto.Dbserver.GatewayLookupSource;

import org.apache.hadoop.fs.Path;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.json.JSONTokener;
import org.apache.log4j.Logger;

import java.io.IOException;
import java.security.PrivilegedExceptionAction;
import java.util.HashSet;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Collections;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.fs.jni.IPPort;
import com.mapr.fs.jni.GatewaySource;

public class DbGatewayCommands extends CLIBaseClass implements CLIInterface {

  private static final int DEFAULT_GATEWAY_PORT = 7660;
  private static final Logger LOG = Logger.getLogger(DbGatewayCommands.class);
  private static final String IPADDRESS_PATTERN =
                                   "^([01]?\\d\\d?|2[0-4]\\d|25[0-5])\\." +
                                   "([01]?\\d\\d?|2[0-4]\\d|25[0-5])\\." +
                                   "([01]?\\d\\d?|2[0-4]\\d|25[0-5])\\." +
                                   "([01]?\\d\\d?|2[0-4]\\d|25[0-5])$";

  public static final String MULTI_ADDR_SEPARATOR = ";";
  public static final String HOSTNAME_IP_SEPARATOR = ",";
  public static final String IP_PORT_SEPARATOR = ":";
  public static final int MAX_PORT_NUMBER = 65535;

  public static final String DST_CLUSTER_PARAM_NAME = "dstcluster";
  public static final String CLUSTER_PARAM_NAME = "cluster";
  public static final String GATEWAYS_PARAM_NAME = "gateways";
  public static final String FORMAT_PARAM_NAME = "format";

  private static final CLICommand getCommand = 
      new CLICommand(
          "get",
          "usage: cluster gateway get -dstcluster <clustername>",
          DbGatewayCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(DST_CLUSTER_PARAM_NAME,
                   new TextInputParameter(DST_CLUSTER_PARAM_NAME,
                   "cluster name",
                   CLIBaseClass.REQUIRED,
                   null))
              .put(CLUSTER_PARAM_NAME,
                   new TextInputParameter(CLUSTER_PARAM_NAME,
                   "cluster on which command to be run",
                   CLIBaseClass.NOT_REQUIRED,
                   null))
              .build(), null)
      .setShortUsage("cluster gateway get -dstcluster <clustername>");

  private static final CLICommand listCommand = 
      new CLICommand(
          "list",
          "usage: cluster gateway list",
          DbGatewayCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(CLUSTER_PARAM_NAME,
                   new TextInputParameter(CLUSTER_PARAM_NAME,
                   "cluster on which command to be run",
                   CLIBaseClass.NOT_REQUIRED,
                   null))
              .build(), null)
      .setShortUsage("cluster gateway list");

 private static final CLICommand setCommand =
      new CLICommand(
          "set",
          "usage: cluster gateway set -dstcluster <clustername> -gateways <gatewayList>",
          DbGatewayCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(DST_CLUSTER_PARAM_NAME,
                   new TextInputParameter(DST_CLUSTER_PARAM_NAME,
                   "cluster name",
                   CLIBaseClass.REQUIRED,
                   null))
              .put(GATEWAYS_PARAM_NAME,
                   new TextInputParameter(GATEWAYS_PARAM_NAME,
                   "space-separated list of hostnames",
                   CLIBaseClass.REQUIRED,
                   null))
              .put(CLUSTER_PARAM_NAME,
                   new TextInputParameter(CLUSTER_PARAM_NAME,
                   "cluster on which command to be run",
                   CLIBaseClass.NOT_REQUIRED,
                   null))
              .build(), null)
      .setShortUsage("cluster gateway set -dstcluster <clustername> -gateways <gatewayList>");

 private static final CLICommand localCommand =
      new CLICommand(
          "local",
          "usage: cluster gateway local -format <dns/text>",
          DbGatewayCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(FORMAT_PARAM_NAME,
                   new TextInputParameter(FORMAT_PARAM_NAME,
                   "dns/text",
                   CLIBaseClass.NOT_REQUIRED,
                   "text"))
              .put(CLUSTER_PARAM_NAME,
                   new TextInputParameter(CLUSTER_PARAM_NAME,
                   "cluster on which command to be run",
                   CLIBaseClass.NOT_REQUIRED,
                   null))
              .build(), null)
       .setShortUsage("cluster gateway local -format <dns/text>");

  private static final CLICommand deleteCommand = 
      new CLICommand(
          "delete",
          "usage: cluster gateway delete -dstcluster <clustername>",
          DbGatewayCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(DST_CLUSTER_PARAM_NAME,
                   new TextInputParameter(DST_CLUSTER_PARAM_NAME,
                   "cluster name",
                   CLIBaseClass.REQUIRED,
                   null))
              .put(CLUSTER_PARAM_NAME,
                   new TextInputParameter(CLUSTER_PARAM_NAME,
                   "cluster on which command to be run",
                   CLIBaseClass.NOT_REQUIRED,
                   null))
              .build(), null)
      .setShortUsage("cluster gateway delete -dstcluster <clustername>");

  private static final CLICommand resolveCommand = 
      new CLICommand(
          "resolve",
          "usage: cluster gateway resolve -dstcluster <clustername>",
          DbGatewayCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(DST_CLUSTER_PARAM_NAME,
                   new TextInputParameter(DST_CLUSTER_PARAM_NAME,
                   "cluster name",
                   CLIBaseClass.REQUIRED,
                   null))
              .put(CLUSTER_PARAM_NAME,
                   new TextInputParameter(CLUSTER_PARAM_NAME,
                   "cluster on which command to be run",
                   CLIBaseClass.NOT_REQUIRED,
                   null))
              .build(), null)
      .setShortUsage("cluster gateway resolve -dstcluster <clustername>");

  // main command
  public static final CLICommand gatewayCommands =
      new CLICommand(
          "gateway", "gateway [get|set|delete|list|local]",
          CLIUsageOnlyCommand.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new CLICommand[]{
              getCommand,
              setCommand,
              deleteCommand,
              listCommand,
              resolveCommand,
              localCommand
          }
      ).setShortUsage("cluster gateway [get|set|delete|list|resolve|local]");

 public DbGatewayCommands(ProcessedInput input, CLICommand cliCommand) {
   super(input, cliCommand);
 }

 @Override
 public CommandOutput executeRealCommand() throws CLIProcessingException {
   OutputHierarchy out = new OutputHierarchy();
   CommandOutput output = new CommandOutput();
   output.setOutput(out);

   if (!super.validateInput()) {
     return output;
   }

   String cname = cliCommand.getCommandName();
   if (cname.equalsIgnoreCase(getCommand.getCommandName())) {
     getGateway(out);
   } else if (cname.equalsIgnoreCase(setCommand.getCommandName())) {
     setGateway(out);
   } else if (cname.equalsIgnoreCase(deleteCommand.getCommandName())) {
     deleteGateway(out);
   } else if (cname.equalsIgnoreCase(listCommand.getCommandName())) {
     listGateway(out);
   } else if (cname.equalsIgnoreCase(resolveCommand.getCommandName())) {
     resolveGateway(out);
   } else if (cname.equalsIgnoreCase(localCommand.getCommandName())) {
     localGateway(out);
   }

   return output;
 }

 private void getGateway(OutputHierarchy out) throws CLIProcessingException {
   String clusterName = getParamTextValue(DST_CLUSTER_PARAM_NAME, 0);

   DBReplGatewayConfigGetRequest.Builder gwBuilder = DBReplGatewayConfigGetRequest.newBuilder();
   gwBuilder.setClusterName(clusterName);
   gwBuilder.setCreds(getUserCredentials());

   byte[] data = null;
   try {
     if (isParamPresent(CLUSTER_PARAM_NAME)) {
       String targetClusterName = getParamTextValue(CLUSTER_PARAM_NAME, 0);
       ClusterConf clusterConf =
         MapRCliUtil.getMapRFileSystem().getClusterConf();
       if (clusterConf.getClusterEntryByName(targetClusterName) == null)
         throw new IOException("Cluster " + targetClusterName + " does not exist");

       data = CLDBRpcCommonUtils.getInstance().sendRequest(
           targetClusterName,
           Common.MapRProgramId.CldbProgramId.getNumber(),
           CLDBProto.CLDBProg.DBReplGatewayConfigGetProc.getNumber(),
           gwBuilder.build(), DBReplGatewayConfigGetResponse.class);
     } else {
       data = CLDBRpcCommonUtils.getInstance().sendRequest(
           Common.MapRProgramId.CldbProgramId.getNumber(),
           CLDBProto.CLDBProg.DBReplGatewayConfigGetProc.getNumber(),
           gwBuilder.build(), DBReplGatewayConfigGetResponse.class);
     }

     if (data == null) {
       /**
       * <MAPR_ERROR>
       * Message: Exception while processing RPC
       * Function: DbGatewayCommands.getGateway()
       * Meaning: An error occurred.
       * Resolution: Contact technical support.
       * </MAPR_ERROR>
       */
       out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
       return;
     }

     DBReplGatewayConfigGetResponse resp = DBReplGatewayConfigGetResponse.parseFrom(data);

     if (resp.getStatus() == Errno.ENOENT) {
       out.addError(new OutputError(Errno.ENOENT,
           "gateway not defined for cluster " + clusterName));
       return;
     } else if (resp.getStatus() == 0) {
       OutputNode gwNode = new OutputNode();
       gwNode.addChild(new OutputNode("cluster", clusterName));
       gwNode.addChild(new OutputNode("gatewayConfig", resp.getGatewayConfig()));
       out.addNode(gwNode);
     } else {
       out.addError(new OutputError(Errno.EOPFAILED, "Getting gateway config failed with error " + resp.getStatus()));
       return;
     }
   } catch (InvalidProtocolBufferException e) {
      /**
      * <MAPR_ERROR>
      * Message:InvalidProtocolBufferException <error>
      * Function: DbGatewayCommands.setGateway()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      throw new CLIProcessingException("InvalidProtocolBufferException " + e);
   } catch (CLIProcessingException e) {
     out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
   } catch (IOException e) {
     out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
   } catch (Exception e) {
     out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
     LOG.error("Exception during gateway get " + e);
   }
 }

 private void listGateway(OutputHierarchy out) throws CLIProcessingException {

   String startClusterName = null;
   boolean hasMore = false;

   do {
     DBReplGatewayConfigListRequest.Builder gwBuilder = DBReplGatewayConfigListRequest.newBuilder();
     gwBuilder.setCreds(getUserCredentials());
     if (startClusterName != null)
       gwBuilder.setStartClusterName(startClusterName);

     byte[] data = null;
     try {
       if (isParamPresent(CLUSTER_PARAM_NAME)) {
         String targetClusterName = getParamTextValue(CLUSTER_PARAM_NAME, 0);
         ClusterConf clusterConf =
           MapRCliUtil.getMapRFileSystem().getClusterConf();
         if (clusterConf.getClusterEntryByName(targetClusterName) == null)
           throw new IOException("Cluster " + targetClusterName + " does not exist");

         data = CLDBRpcCommonUtils.getInstance().sendRequest(
             targetClusterName,
             Common.MapRProgramId.CldbProgramId.getNumber(),
             CLDBProto.CLDBProg.DBReplGatewayConfigListProc.getNumber(),
             gwBuilder.build(), DBReplGatewayConfigListResponse.class);
       } else {
         data = CLDBRpcCommonUtils.getInstance().sendRequest(
             Common.MapRProgramId.CldbProgramId.getNumber(),
             CLDBProto.CLDBProg.DBReplGatewayConfigListProc.getNumber(),
             gwBuilder.build(), DBReplGatewayConfigListResponse.class);
       }

       if (data == null) {
         /**
         * <MAPR_ERROR>
         * Message: Exception while processing RPC
         * Function: DbGatewayCommands.listGateway()
         * Meaning: An error occurred.
         * Resolution: Contact technical support.
         * </MAPR_ERROR>
         */
         out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
         return;
       }

       DBReplGatewayConfigListResponse resp = DBReplGatewayConfigListResponse.parseFrom(data);

       if (resp.getStatus() == 0) {
         if (resp.getGatewayConfigsCount() == 0) {
           out.addError(new OutputError(Errno.ENOENT, "No gateways configured"));
           return;
         }

         for (int i = 0; i < resp.getGatewayConfigsCount(); ++i) {
           OutputNode gwNode = new OutputNode();
           gwNode.addChild(new OutputNode("cluster", resp.getGatewayConfigs(i).getClusterName()));
           gwNode.addChild(new OutputNode("gatewayConfig", resp.getGatewayConfigs(i).getGatewayConfig()));
           out.addNode(gwNode);
         }

         if (resp.hasCookie() && resp.getCookie() != null) {
           startClusterName = resp.getCookie();
           hasMore = true;
         } else {
           hasMore = false;
         }
       } else {
         out.addError(new OutputError(Errno.EOPFAILED, "Listing gateway configs failed with error " + resp.getStatus()));
         return;
       }
     } catch (InvalidProtocolBufferException e) {
        /**
        * <MAPR_ERROR>
        * Message:InvalidProtocolBufferException <error>
        * Function: DbGatewayCommands.listGateway()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        throw new CLIProcessingException("InvalidProtocolBufferException " + e);
     } catch (CLIProcessingException e) {
       out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
       return;
     } catch (IOException e) {
       out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
       return;
     } catch (Exception e) {
       out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
       LOG.error("Exception during gateway list " + e);
       return;
     }
   } while (hasMore == true);
 }

  private void resolveGateway(final OutputHierarchy out) throws CLIProcessingException {
    final String dstCluster = getParamTextValue(DST_CLUSTER_PARAM_NAME, 0);

    try {
      // Call impersonation
      new FileclientRun(getUserLoginId()) {
        @Override
        public void runAsProxyUser() throws IOException, CLIProcessingException {
          final GatewaySource sourceObj = new GatewaySource();
          final MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
          final String varFilePath = ((isParamPresent(CLUSTER_PARAM_NAME) &&
                                       !CLDBRpcCommonUtils.getInstance().getCurrentClusterName().equals(getParamTextValue(CLUSTER_PARAM_NAME, 0))) ? new String("/mapr/" + getParamTextValue(CLUSTER_PARAM_NAME, 0) + "/var") : new String("/var"));

          IPPort[] ipPorts = mfs.getGatewayIps(varFilePath, dstCluster,
                                               true /*skipCache*/, sourceObj);

          StringBuffer sb = new StringBuffer();
          String delimiter1 = "";
          for (IPPort ipPort : ipPorts) {
            int port = ipPort.port;
            String delimiter2 = "";

            sb.append(delimiter1);
            for (int host : ipPort.ips) {
              sb.append(delimiter2);
              sb.append(Util.intToIp(host));
              sb.append(":");
              sb.append(port);
              delimiter2 = ";";
            }

            delimiter1 = " ";
          }

          OutputNode gwSourceNode = new OutputNode();
          gwSourceNode.addChild(new OutputNode("GatewayHosts", sb.toString()));
          gwSourceNode.addChild(new OutputNode("Source",
                                               mfs.GatewaySourceToString(sourceObj.source)));
          out.addNode(gwSourceNode);
        }
      };
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
      LOG.error("Exception during gateway resolve " + e);
    }
  }

 private boolean isGatewayConfigValid(String config, StringBuilder error) {
   String[] tokens = config.split("[\\s]+");
   Pattern pattern = Pattern.compile(IPADDRESS_PATTERN);

   for (int i = 0; i < tokens.length; i++) {
     for (String token: tokens[i].split(MULTI_ADDR_SEPARATOR)) {
       token = token.trim();
       if (token.isEmpty())
         return false;

       String addr = null;
       if (token.contains(IP_PORT_SEPARATOR)) {
         String[] arr = token.split(IP_PORT_SEPARATOR);

         if (arr.length < 2 || arr[0].isEmpty() || arr[1].isEmpty()) {
           error.append("Empty host/port");
           return false;
         }

         addr = arr[0];
         // in case both hostname and IP provided.
         int index = addr.indexOf(',');
         if (index != -1 ) {
           if (index < (addr.length() - 1)) {
             addr = addr.substring(index+1);
           } else {
             addr = addr.substring(0, index);
           }
         }

         int port = Integer.parseInt(arr[1]);
         if (port < 0 || port > MAX_PORT_NUMBER) {
           error.append("Invalid port: " + port);
           return false;
         }
       } else {
         addr = token;
       }

       if (!addr.matches(".*[a-zA-Z]+.*")) {
         // ip address
         Matcher matcher = pattern.matcher(addr);
         if (!matcher.matches()) {
           error.append("Invalid ipaddress " + addr);
           return false;
         }
       }
     }
   }

   return true;
 }

 private void setGateway(OutputHierarchy out) throws CLIProcessingException {
   String clusterName = getParamTextValue(DST_CLUSTER_PARAM_NAME, 0);
   String confTxt = getParamTextValue(GATEWAYS_PARAM_NAME, 0);
   StringBuilder error = new StringBuilder("");

   if (!isGatewayConfigValid(confTxt, error)) {
     out.addError(new OutputError(Errno.EOPFAILED, "Incorrect gateway config provided: " + error));
     return;
   }

   DBReplGatewayConfigSetRequest.Builder gwBuilder = DBReplGatewayConfigSetRequest.newBuilder();
   gwBuilder.setClusterName(clusterName);
   gwBuilder.setGatewayConfig(confTxt);
   gwBuilder.setCreds(getUserCredentials());

   byte[] data = null;
   try {
     if (isParamPresent(CLUSTER_PARAM_NAME)) {
       String targetClusterName = getParamTextValue(CLUSTER_PARAM_NAME, 0);
       ClusterConf clusterConf =
         MapRCliUtil.getMapRFileSystem().getClusterConf();
       if (clusterConf.getClusterEntryByName(targetClusterName) == null)
         throw new IOException("Cluster " + targetClusterName + " does not exist");

       data = CLDBRpcCommonUtils.getInstance().sendRequest(
           targetClusterName,
           Common.MapRProgramId.CldbProgramId.getNumber(),
           CLDBProto.CLDBProg.DBReplGatewayConfigSetProc.getNumber(),
           gwBuilder.build(), DBReplGatewayConfigSetResponse.class);
     } else {
       data = CLDBRpcCommonUtils.getInstance().sendRequest(
           Common.MapRProgramId.CldbProgramId.getNumber(),
           CLDBProto.CLDBProg.DBReplGatewayConfigSetProc.getNumber(),
           gwBuilder.build(), DBReplGatewayConfigSetResponse.class);
     }

     if (data == null) {
       /**
       * <MAPR_ERROR>
       * Message:Exception while processing RPC
       * Function: DbGatewayCommands.setGateway()
       * Meaning:An error occurred.
       * Resolution:Contact technical support.
       * </MAPR_ERROR>
       */
       out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
       return;
     }

     DBReplGatewayConfigSetResponse resp = DBReplGatewayConfigSetResponse.parseFrom(data);

     if (resp.getStatus() == Errno.E_NOT_ENOUGH_PRIVILEGES) {
       out.addError(new OutputError(Errno.E_NOT_ENOUGH_PRIVILEGES,
           Errno.toString(Errno.E_NOT_ENOUGH_PRIVILEGES) + " : " +
           "User not authorized on the cluster"));
       return;
     } else if (resp.getStatus() != 0) {
       out.addError(new OutputError(Errno.EOPFAILED, "Setting gateway config failed with error " + resp.getStatus()));
       return;
     }

   } catch (InvalidProtocolBufferException e) {
      /**
      * <MAPR_ERROR>
      * Message:InvalidProtocolBufferException <error>
      * Function: DbGatewayCommands.setGateway()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      throw new CLIProcessingException("InvalidProtocolBufferException " + e);
   } catch (CLIProcessingException e) {
     out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
   } catch (IOException e) {
     out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
   } catch (Exception e) {
     out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
     LOG.error("Exception during gateway get " + e);
   }
 }

 private void deleteGateway(OutputHierarchy out) throws CLIProcessingException {
   String clusterName = getParamTextValue(DST_CLUSTER_PARAM_NAME, 0);


   DBReplGatewayConfigDeleteRequest.Builder gwBuilder = DBReplGatewayConfigDeleteRequest.newBuilder();
   gwBuilder.setClusterName(clusterName);
   gwBuilder.setCreds(getUserCredentials());

   byte[] data = null;
   try {
     if (isParamPresent(CLUSTER_PARAM_NAME)) {
       String targetClusterName = getParamTextValue(CLUSTER_PARAM_NAME, 0);
       ClusterConf clusterConf =
         MapRCliUtil.getMapRFileSystem().getClusterConf();
       if (clusterConf.getClusterEntryByName(targetClusterName) == null)
         throw new IOException("Cluster " + targetClusterName + " does not exist");

       data = CLDBRpcCommonUtils.getInstance().sendRequest(
           targetClusterName,
           Common.MapRProgramId.CldbProgramId.getNumber(),
           CLDBProto.CLDBProg.DBReplGatewayConfigDeleteProc.getNumber(),
           gwBuilder.build(), DBReplGatewayConfigDeleteResponse.class);
     } else {
       data = CLDBRpcCommonUtils.getInstance().sendRequest(
           Common.MapRProgramId.CldbProgramId.getNumber(),
           CLDBProto.CLDBProg.DBReplGatewayConfigDeleteProc.getNumber(),
           gwBuilder.build(), DBReplGatewayConfigDeleteResponse.class);
     }

     if (data == null) {
       /**
       * <MAPR_ERROR>
       * Message:Exception while processing RPC
       * Function: DbGatewayCommands.deleteGateway()
       * Meaning:An error occurred.
       * Resolution:Contact technical support.
       * </MAPR_ERROR>
       */
       out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
       return;
     }

     DBReplGatewayConfigDeleteResponse resp = DBReplGatewayConfigDeleteResponse.parseFrom(data);

     if (resp.getStatus() == Errno.E_NOT_ENOUGH_PRIVILEGES) {
       out.addError(new OutputError(Errno.E_NOT_ENOUGH_PRIVILEGES,
           Errno.toString(Errno.E_NOT_ENOUGH_PRIVILEGES) + " : " +
           "User not authorized on the cluster"));
       return;
     } else if (resp.getStatus() == Errno.ENOENT) {
       out.addError(new OutputError(Errno.ENOENT,
           "gateway not defined for cluster " + clusterName));
       return;
     } else if (resp.getStatus() != 0) {
       out.addError(new OutputError(Errno.EOPFAILED, "Deleting gateway config failed with error " + resp.getStatus()));
       return;
     }
   } catch (InvalidProtocolBufferException e) {
      /**
      * <MAPR_ERROR>
      * Message:InvalidProtocolBufferException <error>
      * Function: DbGatewayCommands.deleteGateway()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      throw new CLIProcessingException("InvalidProtocolBufferException " + e);
   } catch (CLIProcessingException e) {
     out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
   } catch (IOException e) {
     out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
   } catch (Exception e) {
     out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
     LOG.error("Exception during gateway delete " + e);
   }
 }

 private void localGateway(OutputHierarchy out) throws CLIProcessingException {
   String clusterName = null;
   String format = getParamTextValue(FORMAT_PARAM_NAME, 0);

   try {
     if (!format.equals("text") && !format.equals("dns"))
       throw new CLIProcessingException("Invalid format, supported: text/dns");

     if (isParamPresent(CLUSTER_PARAM_NAME))
       clusterName = getParamTextValue(CLUSTER_PARAM_NAME, 0);
     else
       clusterName = CLDBRpcCommonUtils.getInstance().getCurrentClusterName();

     getLocalGatewayInfo(clusterName, format, out);
   } catch (CLIProcessingException e) {
     out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
   }
 }

  private void getLocalGatewayInfo(String clusterName, String format,
                                   OutputHierarchy out) throws CLIProcessingException {
    String zkConnectString = CLDBRpcCommonUtils.getInstance().getZkConnect(clusterName);
    if (zkConnectString == null) {
      throw new CLIProcessingException("Invalid cluster provided: " + clusterName);
    }

    Map<String, ServiceData> gatewayServiceData = NodesCommonUtils.getServiceNodeData(zkConnectString, "gateway");
    if (gatewayServiceData == null) {
      throw new CLIProcessingException("Gateway service not found on cluster: " + clusterName);
    }

    gatewayServiceData.remove("master");

    StringBuffer sb = new StringBuffer();
    String delimiter = "";
    for(ServiceData nodeServiceData : gatewayServiceData.values()) {
        if(nodeServiceData.hasIsRunning() && nodeServiceData.getIsRunning()) {
            String host = null;
            int port = 0;
            if(nodeServiceData.hasHost())
                host = nodeServiceData.getHost();
            if(nodeServiceData.hasPort())
                port = new Integer(nodeServiceData.getPort());

            if(host != null) {
                if (port == 0 )
                    port = DEFAULT_GATEWAY_PORT; // Assume gateway service default port and try.
                // Connect to the gateway service and look up its hostnames.
                IPAddress hostIP = buildIPFromString(host);
                if (hostIP == null) {
                   LOG.error("Cannot find valid hostname to IPAddress mapping in " + 
                             "file/dns for host " + host);
                   out.addError(new OutputError(Errno.EINVAL,
                                                "Cannot get valid IP address out of hostname: " + host));
                   return;
                }
                long gwbinding = Rpc.createBindingFor(hostIP.getHost(), port,
                        clusterName, ServerKeyType.ServerKey.getNumber());
                // Send GetAttr to mfs
                GetHostNamesRequest req = GetHostNamesRequest.newBuilder()
                        .setCreds(getUserCredentials())
                        .build();
                try {
                    byte[] replyData;
                    replyData = Rpc.sendRequest(gwbinding,
                            MapRProgramId.DBReplicatorServerProgramId.getNumber(),
                            DBReplicatorProg.GetHostNamesProc.getNumber(),
                            req);
                    if (replyData == null) {
                        LOG.error("Got null reply from RPC");
                        out.addError(new OutputHierarchy.OutputError(Errno.ERPCFAILED, "gethostnames rpc failed"));
                        return;
                    }

                    GetHostNamesResponse resp = GetHostNamesResponse.parseFrom(replyData);
                    OutputNode dout = new OutputNode();
                    if (resp.getStatus() == 0) {
                        for (String name : resp.getNamesList()) {
                            sb.append(delimiter);
                            sb.append(name);
                            if(port != DEFAULT_GATEWAY_PORT) // Append the port if different from default value.
                                sb.append(":" + port);
                            delimiter = ";"; // Within the same gateway service, next name to be separated by ';'
                        }
                    } else {
                        out.addError(new OutputHierarchy.OutputError(resp.getStatus(),
                                "GetHostNames failed, Error : " +
                                        Errno.toString(resp.getStatus())));
                        return;
                    }
                } catch (MaprSecurityException e) {
                    throw new CLIProcessingException(
                            "MaprSecurityException " + "Exception", e);
                } catch (Exception e) {
                    LOG.error("Exception processing getHostNames command", e);
                    out.addError(new OutputHierarchy.OutputError(Errno.ERPCFAILED, "db rpc failed"));
                    return;
                }
            }
        }
        delimiter = " "; // Next gateway service records to be separated by a space.
    }

    StringBuffer outsb = null;
    if (format.equals("dns")) {
      outsb = new StringBuffer("\n; TXT Record addresses\n");
      outsb.append("gateway."+clusterName+" IN TXT \"");
      outsb.append(sb);
      outsb.append("\"");
      out.addNode(new OutputNode("gatewaydnsinfo", outsb.toString()));
    } else {
      out.addNode(new OutputNode("gatewayinfo", sb.toString()));
    }
  }

  
  private IPAddress buildIPFromString(String ipstr) {
      if (ipstr.equalsIgnoreCase("localhost") ||
              ipstr.equalsIgnoreCase("127.0.0.1") ) {
          ipstr = "127.0.0.1";
      } else {
          List<String> ips;
          ips = NodesCommonUtils.convertHostToIp(Collections.singletonList(ipstr));
          if (ips.isEmpty())
              return null;

          ipstr = ips.get(0);
      }

      int host = Util.ipToInt(ipstr);
      IPAddress server = IPAddress.newBuilder()
              .setHost(host)
              .build();
      return server;
  }
}
