/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.ByteString;
import com.mapr.baseutils.BinaryString;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cli.common.FileclientRun;
import com.mapr.cli.common.JobExecutor;
import com.mapr.cli.common.NodesCommonUtils;
import com.mapr.cli.common.SecurityTokenHelper;
import com.mapr.cli.common.ServicesEnum;
import com.mapr.cli.table.RecentTablesListManager;
import com.mapr.cli.table.RecentTablesListManagers;
import com.mapr.cli.table.TabletStats;
import com.mapr.cliframework.base.*;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.inputparams.*;

import com.mapr.db.mapreduce.tools.CopyTable;

import com.mapr.fs.AceHelper;
import com.mapr.fs.MapRFileSystem;
import com.mapr.fs.proto.Common.FileCompressionType;
import com.mapr.fs.proto.Dbserver.AccessControlExpression;
import com.mapr.fs.proto.Dbserver.ColumnFamilyAttr;
import com.mapr.fs.proto.Dbserver.DBInternalDefaults;
import com.mapr.fs.proto.Dbserver.SchemaFamily;
import com.mapr.fs.proto.Dbserver.SpaceUsage;
import com.mapr.fs.proto.Dbserver.TableAces;
import com.mapr.fs.proto.Dbserver.TableAttr;
import com.mapr.fs.proto.Dbserver.TabletDesc;
import com.mapr.fs.proto.Dbserver.TabletStatResponse;
import com.mapr.fs.tables.TableProperties;

import org.apache.hadoop.fs.Path;
import org.apache.log4j.Logger;

import com.mapr.fs.cli.proto.CLIProto;
import com.mapr.fs.hbase.CopyMetaHelper;

import org.apache.hadoop.security.UserGroupInformation;

import java.lang.String;
import java.lang.reflect.Method;
import java.security.PrivilegedExceptionAction;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Map;
import java.util.List;

public class DbCommands extends CLIBaseClass implements CLIInterface, AceHelper.DBPermission {
  private static final Logger LOG = Logger.getLogger(DbCommands.class);
  public static final String COLUMNS_PARAM_NAME = "columns";
  public static final String OUTPUT_PARAM_NAME = "output";
  public static final String START_PARAM_NAME = "start";
  public static final String LIMIT_PARAM_NAME = "limit";

  private static final String PATH_PARAM_NAME = "path";
  private static final String BULKLOAD_PARAM_NAME = "bulkload";

  private static final String TABLE_TYPE_PARAM_NAME = "tabletype";
  private static final String TABLE_TYPE_DEF_BIN = "binary";
  private static final String TABLE_TYPE_JSON = "json";
  private static final String INSERTION_ORDER_JSON = "insertionorder";

  private static final String AUTOSPLIT_PARAM_NAME = "autosplit";
  private static final String MAX_VALUES_SIZE_IN_MEM_NAME = "maxvalueszinmemindex";
  private static final String REGION_SIZE_MB_NAME = "regionsizemb";
  private static final String RECLAIM_THRESH_PCNT_FOR_PACK_NAME = "reclaimthreshpcntforpack";
  private static final String SIZE_THRESH_PCNT_FOR_PACK_NAME = "sizethreshpcntforpack";
  private static final String MAX_SPILLS_PARAM_NAME = "maxspills";
  private static final String MINI_PACK_PARAM_NAME = "minipack";
  private static final String DELETE_TTL_PARAM_NAME = "deletettl";
  private static final String HAS_REPLICATION_PARAM_NAME = "hasreplication";
  private static final String SYNC_REPL_TIMEOUT_PARAMNAME = "syncrepltimeout";
  private static final String DROP_LARGE_ROWS_PARAM_NAME = "droplargerows";
  private static final String TTL_COMPACT_PARAM_NAME = "ttlcompact";
  private static final String TTL_COMPACT_HRS_PARAM_NAME = "ttlcompacthrs";
  private static final String COPY_META_FROM_PARAM_NAME = "copymetafrom";
  private static final String COPY_META_TYPE_PARAM_NAME = "copymetatype";
  private static final String MULTI_ARG_SEP = ",";

  private static final String PERM_PACK_PARAM_NAME = "packperm";
  private static final String PERM_BULKLOAD_PARAM_NAME = "bulkloadperm";
  private static final String PERM_SPLITMERGE_PARAM_NAME = "splitmergeperm";
  private static final String PERM_CREATERENAMEFAMILY_PARAM_NAME = "createrenamefamilyperm";
  private static final String PERM_DELETEFAMILY_PARAM_NAME = "deletefamilyperm";
  private static final String PERM_ACEADMINACCESS_PARAM_NAME = "adminaccessperm";
  private static final String PERM_REPL_PARAM_NAME = "replperm";
  private static final String PERM_VERSIONS_PARAM_NAME = "defaultversionperm";
  private static final String PERM_COMPRESSION_PARAM_NAME = "defaultcompressionperm";
  private static final String PERM_MEMORY_PARAM_NAME = "defaultmemoryperm";
  private static final String PERM_WRITE_PARAM_NAME = "defaultwriteperm";
  private static final String PERM_TRAVERSE_PARAM_NAME = "defaulttraverseperm";
  private static final String PERM_APPEND_PARAM_NAME = "defaultappendperm";
  private static final String PERM_READ_PARAM_NAME = "defaultreadperm";
  private static final String PERM_ENCRYPT_PARAM_NAME = "defaultencryptperm";
  // NOTE: These permission variables need to be replicated in AceHelper in the immutable maps, in order to be parsed
  // To be sent to the server
  private static final String SOURCE_TABLE_NAME = "src";
  private static final String DESTINATION_TABLE_NAME = "dst";
  private static final String REPLICA_PARAM_NAME = "replica";
  private static final String USER_PARAM_NAME = "user";
  private static final String MAX_VER_PARAM_NAME = "maxversions";
  private static final String START_TIME_PARAM_NAME = "starttime";
  private static final String END_TIME_PARAM_NAME = "endtime";
  private static final String USE_MR_PARAM_NAME = "mapreduce";
  private static final String AUDIT_PARAM_NAME = "audit";

  // Json table record insertion order
  private static final boolean INSERTION_ORDER_DEFAULT = true;

  private int metaTypeMask = 0;
  private TableProperties tableProp;

  private static final CLICommand createCommand =
      new CLICommand(
          "create",
          "usage: table create -path <path>",
          DbCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(COPY_META_FROM_PARAM_NAME,
                   new TextInputParameter(COPY_META_FROM_PARAM_NAME,
                       "SrcTablePath",
                       CLIBaseClass.NOT_REQUIRED,
                       null))
              .put(COPY_META_TYPE_PARAM_NAME,
                   new TextInputParameter(COPY_META_TYPE_PARAM_NAME,
                       "all|cfs|aces|splits|attrs",
                       CLIBaseClass.NOT_REQUIRED,
                       null))
              .put(REGION_SIZE_MB_NAME,
                  new LongInputParameter(REGION_SIZE_MB_NAME,
                      "Region Size in MB",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(AUTOSPLIT_PARAM_NAME,
                  new BooleanInputParameter(AUTOSPLIT_PARAM_NAME,
                      "Auto Split table",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(BULKLOAD_PARAM_NAME,
                  new BooleanInputParameter(BULKLOAD_PARAM_NAME,
                      "Bulk load",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(AUDIT_PARAM_NAME,
                  new TextInputParameter(AUDIT_PARAM_NAME,
                      "Enable Audit",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(TABLE_TYPE_PARAM_NAME,
                  new TextInputParameter(TABLE_TYPE_PARAM_NAME,
                      "Table Type - json or binary. default: binary",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(INSERTION_ORDER_JSON,
                  new BooleanInputParameter(INSERTION_ORDER_JSON,
                      "Retain Insertion Order within document - only for JSON table type. default : true",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_PACK_PARAM_NAME,
                  new TextInputParameter(PERM_PACK_PARAM_NAME,
                      "Pack Permission settings",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_BULKLOAD_PARAM_NAME,
                  new TextInputParameter(PERM_BULKLOAD_PARAM_NAME,
                      "Bulk load Permission settings",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_SPLITMERGE_PARAM_NAME,
                  new TextInputParameter(PERM_SPLITMERGE_PARAM_NAME,
                      "Split and Merge Permission settings",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_CREATERENAMEFAMILY_PARAM_NAME,
                  new TextInputParameter(PERM_CREATERENAMEFAMILY_PARAM_NAME,
                      "Add/Rename Family Permission settings",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_DELETEFAMILY_PARAM_NAME,
                  new TextInputParameter(PERM_DELETEFAMILY_PARAM_NAME,
                      "Delete Family Permission settings",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_ACEADMINACCESS_PARAM_NAME,
                  new TextInputParameter(PERM_ACEADMINACCESS_PARAM_NAME,
                      "Ace Admin Permission settings",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_REPL_PARAM_NAME,
                  new TextInputParameter(PERM_REPL_PARAM_NAME,
                      "Replication Admin Permission settings",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_VERSIONS_PARAM_NAME,
                  new TextInputParameter(PERM_VERSIONS_PARAM_NAME,
                      "CF Versions Default Permission for binary tabletype",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
               .put(PERM_COMPRESSION_PARAM_NAME,
                   new TextInputParameter(PERM_COMPRESSION_PARAM_NAME,
                       "CF Compression Default Permission",
                       CLIBaseClass.NOT_REQUIRED,
                       null))
              .put(PERM_MEMORY_PARAM_NAME,
                  new TextInputParameter(PERM_MEMORY_PARAM_NAME,
                      "CF Memory Default Permission",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
               .put(PERM_READ_PARAM_NAME,
                   new TextInputParameter(PERM_READ_PARAM_NAME,
                       "CF Read Default Permission",
                       CLIBaseClass.NOT_REQUIRED,
                       null))
              .put(PERM_WRITE_PARAM_NAME,
                  new TextInputParameter(PERM_WRITE_PARAM_NAME,
                      "CF Write Default Permission",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_TRAVERSE_PARAM_NAME,
                  new TextInputParameter(PERM_TRAVERSE_PARAM_NAME,
                      "CF Traverse Default Permission for json tabletype",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_APPEND_PARAM_NAME,
                  new TextInputParameter(PERM_APPEND_PARAM_NAME,
                      "CF Append Default Permission for binary tabletype",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_ENCRYPT_PARAM_NAME,
                  new TextInputParameter(PERM_ENCRYPT_PARAM_NAME,
                      "CF Encrypt Default Permission",
                      CLIBaseClass.NOT_REQUIRED,
                      null).setInvisible(true))
              .put(DROP_LARGE_ROWS_PARAM_NAME,
                  new BooleanInputParameter(DROP_LARGE_ROWS_PARAM_NAME,
                      "Drop large rows in table",
                      CLIBaseClass.NOT_REQUIRED,
                      null).setInvisible(true))
              .put(TTL_COMPACT_PARAM_NAME,
                  new BooleanInputParameter(TTL_COMPACT_PARAM_NAME,
                      "Enable TTL-based compactions",
                      CLIBaseClass.NOT_REQUIRED,
                      null).setInvisible(true))
              .put(TTL_COMPACT_HRS_PARAM_NAME,
                  new IntegerInputParameter(TTL_COMPACT_HRS_PARAM_NAME,
                      "Minimum hours between TTL compactions",
                      CLIBaseClass.NOT_REQUIRED,
                      null).setInvisible(true))
              .build(),
          null)
          .setShortUsage("table create -path <path>");

  private static final CLICommand editCommand =
      new CLICommand(
          "edit",
          "usage: table edit -path <path>",
          DbCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(AUTOSPLIT_PARAM_NAME,
                  new BooleanInputParameter(AUTOSPLIT_PARAM_NAME,
                      "Auto Split table",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(MAX_VALUES_SIZE_IN_MEM_NAME,
                  new IntegerInputParameter(MAX_VALUES_SIZE_IN_MEM_NAME,
                      "Max Values Size in Memory",
                      CLIBaseClass.NOT_REQUIRED,
                      null).setInvisible(true))
              .put(REGION_SIZE_MB_NAME,
                  new LongInputParameter(REGION_SIZE_MB_NAME,
                      "Region Size in MB",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(BULKLOAD_PARAM_NAME,
                  new BooleanInputParameter(BULKLOAD_PARAM_NAME,
                      "Bulk load",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(AUDIT_PARAM_NAME,
                  new BooleanInputParameter(AUDIT_PARAM_NAME,
                      "Enable Audit",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(RECLAIM_THRESH_PCNT_FOR_PACK_NAME,
                  new IntegerInputParameter(RECLAIM_THRESH_PCNT_FOR_PACK_NAME,
                      "Space reclaim percentage threshold to pack",
                      CLIBaseClass.NOT_REQUIRED,
                      null).setInvisible(true))
              .put(SIZE_THRESH_PCNT_FOR_PACK_NAME,
                  new IntegerInputParameter(SIZE_THRESH_PCNT_FOR_PACK_NAME,
                      "Size percentage threshold to pack",
                      CLIBaseClass.NOT_REQUIRED,
                      null).setInvisible(true))
              .put(MAX_SPILLS_PARAM_NAME,
                  new IntegerInputParameter(MAX_SPILLS_PARAM_NAME,
                      "Max Spills",
                      CLIBaseClass.NOT_REQUIRED,
                      null).setInvisible(true))
              .put(MINI_PACK_PARAM_NAME,
                  new BooleanInputParameter(MINI_PACK_PARAM_NAME,
                      "Mini Pack",
                      CLIBaseClass.NOT_REQUIRED,
                      null).setInvisible(true))
              .put(DELETE_TTL_PARAM_NAME,
                  new LongInputParameter(DELETE_TTL_PARAM_NAME,
                      "delete TTL in secs",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(HAS_REPLICATION_PARAM_NAME,
                  new BooleanInputParameter(HAS_REPLICATION_PARAM_NAME,
                      "Table has replicaton with other tables",
                      CLIBaseClass.NOT_REQUIRED,
                      null).setInvisible(true))
              .put(SYNC_REPL_TIMEOUT_PARAMNAME,
                  new LongInputParameter(SYNC_REPL_TIMEOUT_PARAMNAME,
                      "sync replica timeout in milli secs",
                      CLIBaseClass.NOT_REQUIRED,
                      null).setInvisible(true))
              .put(PERM_PACK_PARAM_NAME,
                  new TextInputParameter(PERM_PACK_PARAM_NAME,
                      "Pack Permission settings",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_BULKLOAD_PARAM_NAME,
                  new TextInputParameter(PERM_BULKLOAD_PARAM_NAME,
                      "Bulk load Permission settings",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_SPLITMERGE_PARAM_NAME,
                  new TextInputParameter(PERM_SPLITMERGE_PARAM_NAME,
                      "Split and Merge Permission settings",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_CREATERENAMEFAMILY_PARAM_NAME,
                  new TextInputParameter(PERM_CREATERENAMEFAMILY_PARAM_NAME,
                      "Add/Rename Family Permission settings",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_DELETEFAMILY_PARAM_NAME,
                  new TextInputParameter(PERM_DELETEFAMILY_PARAM_NAME,
                      "Delete Family Permission settings",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_ACEADMINACCESS_PARAM_NAME,
                  new TextInputParameter(PERM_ACEADMINACCESS_PARAM_NAME,
                      "Ace Admin Permission settings",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_REPL_PARAM_NAME,
                  new TextInputParameter(PERM_REPL_PARAM_NAME,
                      "Replication Admin Permission settings",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_VERSIONS_PARAM_NAME,
                  new TextInputParameter(PERM_VERSIONS_PARAM_NAME,
                      "CF Versions Default Permission for binary tabletype",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
               .put(PERM_COMPRESSION_PARAM_NAME,
                   new TextInputParameter(PERM_COMPRESSION_PARAM_NAME,
                       "CF Compression Default Permission",
                       CLIBaseClass.NOT_REQUIRED,
                       null))
              .put(PERM_MEMORY_PARAM_NAME,
                  new TextInputParameter(PERM_MEMORY_PARAM_NAME,
                      "CF Memory Default Permission",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
               .put(PERM_READ_PARAM_NAME,
                   new TextInputParameter(PERM_READ_PARAM_NAME,
                       "CF Read Default Permission",
                       CLIBaseClass.NOT_REQUIRED,
                       null))
              .put(PERM_WRITE_PARAM_NAME,
                  new TextInputParameter(PERM_WRITE_PARAM_NAME,
                      "CF Write Default Permission",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_TRAVERSE_PARAM_NAME,
                  new TextInputParameter(PERM_TRAVERSE_PARAM_NAME,
                      "CF Traverse Default Permission for json tabletype",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_APPEND_PARAM_NAME,
                  new TextInputParameter(PERM_APPEND_PARAM_NAME,
                      "CF Append Default Permission for binary tabletype",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_ENCRYPT_PARAM_NAME,
                  new TextInputParameter(PERM_ENCRYPT_PARAM_NAME,
                      "CF Encrypt Default Permission",
                      CLIBaseClass.NOT_REQUIRED,
                      null).setInvisible(true))
              .put(DROP_LARGE_ROWS_PARAM_NAME,
                  new BooleanInputParameter(DROP_LARGE_ROWS_PARAM_NAME,
                      "Drop large rows in table",
                      CLIBaseClass.NOT_REQUIRED,
                      null).setInvisible(true))
              .put(TTL_COMPACT_PARAM_NAME,
                  new BooleanInputParameter(TTL_COMPACT_PARAM_NAME,
                      "Enable TTL-based compactions",
                      CLIBaseClass.NOT_REQUIRED,
                      null).setInvisible(true))
              .put(TTL_COMPACT_HRS_PARAM_NAME,
                  new IntegerInputParameter(TTL_COMPACT_HRS_PARAM_NAME,
                      "Minimum hours between TTL compactions",
                      CLIBaseClass.NOT_REQUIRED,
                      null).setInvisible(true))
              .build(),
          null)
          .setShortUsage("table edit -path <path>");

  private static final CLICommand deleteCommand =
      new CLICommand(
          "delete",
          "usage: table delete -path <path>",
          DbCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "path",
                      CLIBaseClass.REQUIRED,
                      null))
              .build(),
          null)
          .setShortUsage("table delete -path <path>");

  private static final CLICommand genUuidCommand =
      new CLICommand(
          "genuuid",
          "usage: table genuuid -path <path>",
          DbCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "path",
                      CLIBaseClass.REQUIRED,
                      null))
              .build(),
          null)
          .setShortUsage("table genuuid -path <path>")
          .setUsageInVisible(true);

  private static final CLICommand listRecentCommand =
      new CLICommand(
          "listrecent",
          "usage: table listrecent -columns <columns> -start <start> -limit <limit>",
          DbCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "table path",
                      CLIBaseClass.NOT_REQUIRED,
                      null).setInvisible(true))
              .put(COLUMNS_PARAM_NAME,
                  new TextInputParameter(COLUMNS_PARAM_NAME,
                      "columns",
                      CLIBaseClass.NOT_REQUIRED,
                      "all").setInvisible(true))
              .put(OUTPUT_PARAM_NAME,
                  new TextInputParameter(OUTPUT_PARAM_NAME,
                      "verbose|terse",
                      CLIBaseClass.NOT_REQUIRED,
                      "verbose").setInvisible(true))
              .put(START_PARAM_NAME,
                  new IntegerInputParameter(START_PARAM_NAME,
                      "start",
                      CLIBaseClass.NOT_REQUIRED,
                      0).setInvisible(true))
              .put(LIMIT_PARAM_NAME,
                  new IntegerInputParameter(LIMIT_PARAM_NAME,
                      "limit",
                      CLIBaseClass.NOT_REQUIRED,
                      Integer.MAX_VALUE).setInvisible(true))
              .build(), null)
          .setShortUsage("table listrecent -columns <columns> -start <start> -limit <limit>")
          .setUsageInVisible(true);

    private static final CLICommand infoCommand =
      new CLICommand(
          "info",
          "usage: table info -path <path> -columns <columns>",
          DbCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "table path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(COLUMNS_PARAM_NAME,
                  new TextInputParameter(COLUMNS_PARAM_NAME,
                      "columns",
                      CLIBaseClass.NOT_REQUIRED,
                      "all").setInvisible(true))
              .put(OUTPUT_PARAM_NAME,
                  new TextInputParameter(OUTPUT_PARAM_NAME,
                      "verbose|terse",
                      CLIBaseClass.NOT_REQUIRED,
                      "verbose").setInvisible(true))
              .build(), null)
          .setShortUsage("table info -path <path> -columns <columns>");

  private static final CLICommand copyCommand =
     new CLICommand(
         "copy",
         "usage: table copy -src <path> -dst <path> -columns <cf1[:col1],cf2,...> -maxversions <num> -starttime <start> -endtime <end>",
         DbCommands.class,
         CLICommand.ExecutionTypeEnum.NATIVE,
         new ImmutableMap.Builder<String, BaseInputParameter>()
             .put(SOURCE_TABLE_NAME,
                 new TextInputParameter(SOURCE_TABLE_NAME,
                     "sourcetable", CLIBaseClass.REQUIRED, null))
             .put(DESTINATION_TABLE_NAME,
                 new TextInputParameter(DESTINATION_TABLE_NAME,
                     "destinationTable", CLIBaseClass.REQUIRED,
                     null))
             .put(COLUMNS_PARAM_NAME,
                 new TextInputParameter(COLUMNS_PARAM_NAME,
                     "comma separated list of <family>[:<column>]",
                     CLIBaseClass.NOT_REQUIRED,
                     null))
             .put(MAX_VER_PARAM_NAME,
                 new IntegerInputParameter(MAX_VER_PARAM_NAME,
                     "max versions to copy", CLIBaseClass.NOT_REQUIRED, null))
             .put(START_TIME_PARAM_NAME,
                 new LongInputParameter(START_TIME_PARAM_NAME,
                     "only copy columns from starttime (inclusive)",
                     CLIBaseClass.NOT_REQUIRED, null))
             .put(END_TIME_PARAM_NAME,
                 new LongInputParameter(END_TIME_PARAM_NAME,
                     "only copy columns until endtime (exclusive)",
                     CLIBaseClass.NOT_REQUIRED, null))
             .put(USE_MR_PARAM_NAME,
                 new BooleanInputParameter(USE_MR_PARAM_NAME,
                     "enable (if resources required to run MapReduce are available) or disable use of MapReduce",
                     CLIBaseClass.NOT_REQUIRED, true))
             .put(BULKLOAD_PARAM_NAME,
                 new BooleanInputParameter(BULKLOAD_PARAM_NAME,
                     "enable or disable use of BulkLoad with MapReduce is used",
                     CLIBaseClass.NOT_REQUIRED, true))
              .put(INSERTION_ORDER_JSON,
                  new BooleanInputParameter(INSERTION_ORDER_JSON,
                      "Retain Insertion Order for JSON table documents",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(TABLE_TYPE_PARAM_NAME,
                  new TextInputParameter(TABLE_TYPE_PARAM_NAME,
                      "Table Type - json or binary. default: binary",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
             .put(USER_PARAM_NAME, new TextInputParameter(USER_PARAM_NAME,
                 "user", CLIBaseClass.NOT_REQUIRED,
                 null))
             .build(), null).setUsageInVisible(true)
        .setShortUsage("table copy -sourceTable <path> -destinationTable <path>");

  private static final CLICommand copyStatusCommand =
      new CLICommand(
          "copystatus",
          "usage: table copystatus -srctable <path> -dsttable <path>",
          DbCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(SOURCE_TABLE_NAME,
                  new TextInputParameter(SOURCE_TABLE_NAME,
                      "sourcetable", CLIBaseClass.REQUIRED, null))
              .put(DESTINATION_TABLE_NAME,
                  new TextInputParameter(DESTINATION_TABLE_NAME,
                      "destinationTable", CLIBaseClass.REQUIRED,
                      null))
              .build(), null).setUsageInVisible(true)
         .setShortUsage("table copystatus -sourceTable <path> -destinationTable <path>");

  public static final CLICommand tableCommands =
      new CLICommand("table",
          "table",
          CLIUsageOnlyCommand.class,
          ExecutionTypeEnum.NATIVE,
          new CLICommand[]{
              createCommand,
              editCommand,
              deleteCommand,
              genUuidCommand,
              listRecentCommand,
              infoCommand,
              copyCommand,
              copyStatusCommand,
              DbCfCommands.cfCommands,
              DbRegionCommands.regionCommands,
              DbReplicaCommands.replicaCommands,
              DbUpstreamCommands.upstreamCommands,
          })
          .setShortUsage("table [create|edit|delete|listrecent|info|cf|region|replica|upstream]");

  private static final Map<String, String> verboseToTerseMap = new ImmutableMap.Builder<String, String>()
      .put("path", "p")
      .put("autosplit", "as")
      .put("bulkload", "bl")
      .put("tabletype", "tt")
      .put("insertionorder", "inso")
      .put("maxvalueszinmemindex", "mvmi")
      .put("regionsizemb", "rsmb")
      .put("reclaimthreshpcntforpack", "rtfp")
      .put("droplargerows", "dlr")
      .put("ttlcompact", "ttlc")
      .put("ttlcompacthrs", "ttlhrs")
      .put("packperm", "ppack")
      .put("bulkloadperm", "blperm")
      .put("adminaccessperm", "aaperm")
      .put("replperm", "rperm")
      .put("splitmergeperm", "smperm")
      .put("createrenamefamilyperm", "crfperm")
      .put("deletefamilyperm", "dfperm")
      .put("defaultversionperm", "pver")
      .put("defaultcompressionperm", "pcomp")
      .put("defaultmemoryperm", "pmem")
      .put("defaultreadperm", "pread")
      .put("defaultwriteperm", "pwrite")
      .put("defaultencryptperm", "pencrypt")
      .put("defaultappendperm", "pappend")
      .put("totalrows", "tr")
      .put("totalphysicalsize", "tps")
      .put("totallogicalsize", "tls")
      .put("totalcopypendingsize", "tcps")
      .put("totalnumberofspills", "tsp")
      .put("totalnumberofsegments", "tsg")
      .build();


  public DbCommands(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }

  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    if (!super.validateInput()) {
      return output;
    }

    if (cliCommand.getCommandName().equalsIgnoreCase(createCommand.getCommandName())) {
      createTable(out);
    } else if (cliCommand.getCommandName().equalsIgnoreCase(editCommand.getCommandName())) {
      editTable(out);
    } else if (cliCommand.getCommandName().equalsIgnoreCase(deleteCommand.getCommandName())) {
      deleteTable(out);
    } else if (cliCommand.getCommandName().equalsIgnoreCase(genUuidCommand.getCommandName())) {
      genTableUuid(out);
    } else if (cliCommand.getCommandName().equalsIgnoreCase(listRecentCommand.getCommandName())) {
      listRecentTables(out);
    } else if (cliCommand.getCommandName().equalsIgnoreCase(infoCommand.getCommandName())) {
      tableInfo(out);
    } else if(cliCommand.getCommandName().equalsIgnoreCase(copyCommand.getCommandName())) {
      copyTable(out);
    } else if(cliCommand.getCommandName().equalsIgnoreCase(copyStatusCommand.getCommandName())) {
      copyTableStatus(out);
    }
    return output;
  }

  private void createTable(OutputHierarchy out) throws CLIProcessingException {
    try {
      int auditValue = -1;
      if (isParamPresent(AUDIT_PARAM_NAME)) {
        String auditStr = getParamTextValue(AUDIT_PARAM_NAME, 0);

        if (auditStr.equals("true") || auditStr.equals("1")) {
          auditValue = 1;
        } else if (auditStr.equals("false") || auditStr.equals("0")) {
          auditValue = 0;
        } else if (!auditStr.equals(" ")) {
          out.addError(new OutputError(Errno.EINVAL,
                                       "audit value can only be true/false"));
          return;
        }
      }

      final String path = getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId());
      Boolean autosplit = null, bulkload = null, droplarge = null, ttlcompact = null, isJson = null,
              insertionOrder=INSERTION_ORDER_DEFAULT;
      Integer ttlcompacthrs = null;
      Long regionsize = null;
      if (isParamPresent(AUTOSPLIT_PARAM_NAME)) {
        autosplit = getParamBooleanValue(AUTOSPLIT_PARAM_NAME, 0);
      }
      if (isParamPresent(BULKLOAD_PARAM_NAME)) {
        bulkload = getParamBooleanValue(BULKLOAD_PARAM_NAME, 0);
      }
      if (isParamPresent(TABLE_TYPE_PARAM_NAME)) {
        String tableType = getParamTextValue(TABLE_TYPE_PARAM_NAME, 0);
        if (!tableType.toLowerCase().equals(TABLE_TYPE_DEF_BIN) &&
            !tableType.toLowerCase().equals(TABLE_TYPE_JSON)) {
          out.addError(new OutputError(Errno.EOPFAILED,
                           "Create table -> Path: " +
                           path + " - table type cannot be " +
                           tableType + ". Choose one of " +
                           TABLE_TYPE_DEF_BIN + " or " +
                           TABLE_TYPE_JSON + "."));
          return;
        }
        isJson = (tableType.toLowerCase().equals(TABLE_TYPE_JSON) ?
                  true : false);
      }
      else {
        isJson = false;
      }
      if (isParamPresent(INSERTION_ORDER_JSON)) {
        insertionOrder = getParamBooleanValue(INSERTION_ORDER_JSON, 0);
      }
      if (isParamPresent(REGION_SIZE_MB_NAME)) {
        regionsize = getParamLongValue(REGION_SIZE_MB_NAME, 0);
        if (regionsize < 256) {
          out.addError(new OutputError(Errno.EINVAL,
                       "attr " + REGION_SIZE_MB_NAME +
                       " value must be >= 256"));
          return;
        }
      }
      if (isParamPresent(DROP_LARGE_ROWS_PARAM_NAME)) {
        droplarge = getParamBooleanValue(DROP_LARGE_ROWS_PARAM_NAME, 0);
      }
      if (isParamPresent(TTL_COMPACT_PARAM_NAME)) {
        ttlcompact = getParamBooleanValue(TTL_COMPACT_PARAM_NAME, 0);
      }
      if (isParamPresent(TTL_COMPACT_HRS_PARAM_NAME)) {
        if (!ttlcompact) {
          out.addError(new OutputError(Errno.EINVAL,
                       "attr " + TTL_COMPACT_HRS_PARAM_NAME +
                       " can only be set when " + TTL_COMPACT_PARAM_NAME +
                       " is set to true"));
          return;
        }
        ttlcompacthrs = getParamIntValue(TTL_COMPACT_HRS_PARAM_NAME, 0);
      }

      final String srcPath =
        isParamPresent(COPY_META_FROM_PARAM_NAME) ? getParamTextValue(COPY_META_FROM_PARAM_NAME, 0) : null;

      if (isParamPresent(COPY_META_TYPE_PARAM_NAME)) {
        if (!isParamPresent(COPY_META_FROM_PARAM_NAME)) {
          out.addError(new OutputError(Errno.EINVAL, "copyMetaType provided "
                                       + "without specifying copyMetaFrom"));
          return;
        }

        String metaStr = getParamTextValue(COPY_META_TYPE_PARAM_NAME, 0);
        for (String metaType : metaStr.split(MULTI_ARG_SEP)) {
          if (metaType.equals("all")) {
            metaTypeMask = CopyMetaHelper.METATYPE_ALL;
            break;
          } else if (metaType.equals("cfs")) {
            metaTypeMask |= CopyMetaHelper.METATYPE_CFS;
          } else if (metaType.equals("attrs")) {
            metaTypeMask |= CopyMetaHelper.METATYPE_ATTRS;
          } else if (metaType.equals("aces")) {
            metaTypeMask |= CopyMetaHelper.METATYPE_ACES;
          } else if (metaType.equals("splits")) {
            metaTypeMask |= CopyMetaHelper.METATYPE_SPLITS;
          } else {
            out.addError(new OutputError(Errno.EINVAL, "Invalid MetaType "
                                         + "specified " + metaType));
            return;
          }
        }
      } else {
        metaTypeMask = CopyMetaHelper.METATYPE_ALL;
      }
      createTable(path,autosplit,bulkload,regionsize, droplarge, ttlcompact, ttlcompacthrs,
                  getUserLoginId(),srcPath,this, metaTypeMask,null /*cflist*/,
                  auditValue,isJson,insertionOrder);
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }
  }

  public static final ColumnFamilyAttr JSON_DEFAULT_COL_FAM_ATTR;
  static {
    SchemaFamily.Builder schemaFamily =  SchemaFamily.newBuilder()
        .setInMemory(false)
        .setMinVersions(0)
        .setMaxVersions(1)
        .setCompression(FileCompressionType.FCT_LZ4)
       .setName(DBInternalDefaults.getDefaultInstance().getJsonDefaultCFName());
    JSON_DEFAULT_COL_FAM_ATTR = ColumnFamilyAttr.newBuilder()
        .setSchFamily(schemaFamily)
        .build();
  }

 public static void createTable(final String path, final Boolean autoSplit,final Boolean bulkLoad,
                                final Long regionSize, final Boolean dropLarge,
                                final Boolean ttlCompact, final Integer ttlCompactHrs,
                                final String user, final String srcPath,
                                final AceHelper.DBPermission dbPerm, final int metaTypeMask,
                                final List<String> cfList, final int auditValue, final Boolean isJson,
                                final Boolean insertionOrder) throws IOException, CLIProcessingException
  {
    if (LOG.isDebugEnabled()) {
      LOG.debug("Attempting to create table for user: " + user);
    }
    TableAttr.Builder attrBuilder = TableAttr.newBuilder();
    if (autoSplit != null)
      attrBuilder.setAutoSplit(autoSplit);
    if (bulkLoad != null)
      attrBuilder.setBulkLoad(bulkLoad);
    if (regionSize!= null)
      attrBuilder.setRegionSizeMB(regionSize);
    if (dropLarge != null)
      attrBuilder.setDropLargeRows(dropLarge);
    if (ttlCompact != null) {
      attrBuilder.setTtlCompaction(ttlCompact);
      if (ttlCompactHrs != null) {
        attrBuilder.setTtlCompactionHrs(ttlCompactHrs);
      }
    }
    if (isJson != null)
      attrBuilder.setJson(isJson);
    if (insertionOrder != null)
      attrBuilder.setInsertionOrder(insertionOrder);

    // Set as final for impersonation
    final TableAttr attr = attrBuilder.build();
    // Run impersonation
    new FileclientRun(user) {
      @Override
      public void runAsProxyUser() throws IOException, CLIProcessingException{
        if (srcPath != null) {
          CopyMetaHelper.createTable(srcPath, path, metaTypeMask, attr,cfList);
        } else {
          MapRCliUtil.getMapRFileSystem().createTable(new Path(path),
                                                      user, attr,
                                                      dbPerm, auditValue);

          // Add default CF also if json tabletype
          if (isJson) {
            MapRCliUtil.getMapRFileSystem().createColumnFamily(new Path(path),
                 DBInternalDefaults.getDefaultInstance().getJsonDefaultCFName(),
                 JSON_DEFAULT_COL_FAM_ATTR);
          }
        }

        RecentTablesListManagers.getRecentTablesListManagerForUser(user).add(path);
      }
    };
  }

  private void editTable(final OutputHierarchy out) throws CLIProcessingException {
    try {
      final String path = getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId());


      TableAttr.Builder builder = TableAttr.newBuilder();

      if (isParamPresent(AUTOSPLIT_PARAM_NAME)) {
        builder.setAutoSplit(getParamBooleanValue(AUTOSPLIT_PARAM_NAME, 0));
      }
      if (isParamPresent(BULKLOAD_PARAM_NAME)) {
        builder.setBulkLoad(getParamBooleanValue(BULKLOAD_PARAM_NAME, 0));
      }
      if (isParamPresent(TABLE_TYPE_PARAM_NAME)) {
        String tableType = getParamTextValue(TABLE_TYPE_PARAM_NAME, 0);
        if (!tableType.toLowerCase().equals(TABLE_TYPE_DEF_BIN) &&
            !tableType.toLowerCase().equals(TABLE_TYPE_JSON)) {
          out.addError(new OutputError(Errno.EOPFAILED,
                           "Edit table -> Path: " +
                           path + " - table type cannot be " +
                           tableType + ". Choose one of " +
                           TABLE_TYPE_DEF_BIN + " or " +
                           TABLE_TYPE_JSON + "."));
          return;
        }
        boolean isJson = tableType.toLowerCase().equals(TABLE_TYPE_JSON);
        builder.setJson(isJson);
      }
      if (isParamPresent(INSERTION_ORDER_JSON)) {
        out.addError(new OutputError(Errno.EOPFAILED, "Insertion order cannot"
                                     + " be altered dynamically."));
        return;
      }

      if (isParamPresent(MAX_VALUES_SIZE_IN_MEM_NAME)) {
          int maxval = getParamIntValue(MAX_VALUES_SIZE_IN_MEM_NAME, 0);
          if (maxval <= 0) {
              out.addError(new OutputError(Errno.EINVAL,
                      "attr " + MAX_VALUES_SIZE_IN_MEM_NAME +
                              " value must be >= 0"));
              return;
          }

          builder.setMaxValueSzInMemIndex(maxval);

      }
      if (isParamPresent(REGION_SIZE_MB_NAME)) {
          long regionsize = getParamLongValue(REGION_SIZE_MB_NAME, 0);
          if (regionsize < 256) {
              out.addError(new OutputError(Errno.EINVAL,
                      "attr " + REGION_SIZE_MB_NAME +
                              " value must be >= 256"));
              return;
          }
          builder.setRegionSizeMB(regionsize);
      }
      if (isParamPresent(RECLAIM_THRESH_PCNT_FOR_PACK_NAME)) {
          int val = getParamIntValue(RECLAIM_THRESH_PCNT_FOR_PACK_NAME, 0);
          if (val < 0 || val > 100) {
              out.addError(new OutputError(Errno.EINVAL,
                      "attr " + RECLAIM_THRESH_PCNT_FOR_PACK_NAME +
                      " value must be >=0 and <= 100"));
              return;
          }
          builder.setReclaimThreshPcntForPack(val);
      }
      if (isParamPresent(SIZE_THRESH_PCNT_FOR_PACK_NAME)) {
          int val = getParamIntValue(SIZE_THRESH_PCNT_FOR_PACK_NAME, 0);
          if ((val != 0) && (val <= 100)) {
              out.addError(new OutputError(Errno.EINVAL,
                      "attr " + SIZE_THRESH_PCNT_FOR_PACK_NAME +
                      " value must be 0 or > 100"));
              return;
          }
          builder.setSizeThreshPcntForPack(val);
      }
      if (isParamPresent(MAX_SPILLS_PARAM_NAME)) {
          int val = getParamIntValue(MAX_SPILLS_PARAM_NAME, 0);
          if ((val != 0) && (val < 3 || val > 6)) {
              out.addError(new OutputError(Errno.EINVAL,
                      "attr " + MAX_SPILLS_PARAM_NAME +
                      " value must be >=3 and <=6"));
              return;
          }
          builder.setMaxSpills(val);
      }
      if (isParamPresent(MINI_PACK_PARAM_NAME))
        builder.setMiniPack(getParamBooleanValue(MINI_PACK_PARAM_NAME, 0));
      if (isParamPresent(DELETE_TTL_PARAM_NAME))
        builder.setDeleteTTL(getParamLongValue(DELETE_TTL_PARAM_NAME, 0));

      if (isParamPresent(HAS_REPLICATION_PARAM_NAME))
        builder.setHasReplication(getParamBooleanValue(HAS_REPLICATION_PARAM_NAME, 0));

      if (isParamPresent(SYNC_REPL_TIMEOUT_PARAMNAME))
        builder.setSyncReplTimeoutMillis(getParamLongValue(SYNC_REPL_TIMEOUT_PARAMNAME, 0));

      if (isParamPresent(DROP_LARGE_ROWS_PARAM_NAME))
        builder.setDropLargeRows(getParamBooleanValue(DROP_LARGE_ROWS_PARAM_NAME, 0));

      if (isParamPresent(TTL_COMPACT_PARAM_NAME))
        builder.setTtlCompaction(getParamBooleanValue(TTL_COMPACT_PARAM_NAME, 0));

      if (isParamPresent(TTL_COMPACT_HRS_PARAM_NAME)) {
        builder.setTtlCompactionHrs(getParamIntValue(TTL_COMPACT_HRS_PARAM_NAME, 0));
      }

      if (LOG.isDebugEnabled()) {
        LOG.debug("Attempting to edit table for user: " + getUserLoginId());
      }

      // Set as final to use in impersonation
      final RecentTablesListManager manager = RecentTablesListManagers.getRecentTablesListManagerForUser(getUserLoginId());
      final AceHelper.DBPermission self = this;
      final TableAttr attr = builder.build();

      // Run impersonation
      new FileclientRun(getUserLoginId()) {
        @Override
        public void runAsProxyUser() throws IOException, CLIProcessingException{
          boolean needUuidGeneration = false;
          if (isParamPresent(PERM_REPL_PARAM_NAME)) {
            getTableProps(new Path(path));
            if (tableProp.getUuid() == null || tableProp.getUuid().length == 0) {
              needUuidGeneration = true;
            }
          }

          MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
          mfs.modifyTableAttr(new Path(path), attr, self,
                              needUuidGeneration);
          manager.moveToTop(path);

          if (!isParamPresent(AUDIT_PARAM_NAME))
            return;

          boolean audit = getParamBooleanValue(AUDIT_PARAM_NAME, 0);
          int err = mfs.modifyAudit(new Path(path), audit);
          if (err != 0) {
            out.addError(new OutputError(err, "Failed to modify audit for" +
                                         " table " + path));
            return;
          }
        }
      };
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }
  }

  private void genTableUuid(OutputHierarchy out) throws CLIProcessingException {
    final String path = getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId());
    final AceHelper.DBPermission dbPerm = this;

    try {
      genTableUuid(path, dbPerm, getUserLoginId());
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }
  }

  public static void genTableUuid(final String tablePath,
                                  final AceHelper.DBPermission dbPerm,
                                  String user) throws CLIProcessingException, Exception {
    // Set as final to use in impersonation
    final RecentTablesListManager manager = RecentTablesListManagers.getRecentTablesListManagerForUser(user);
    final TableAttr attr = TableAttr.newBuilder().build();

    // Run impersonation
    new FileclientRun(user) {
      @Override
      public void runAsProxyUser() throws IOException, CLIProcessingException{
        MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
        mfs.modifyTableAttr(new Path(tablePath), attr, dbPerm, true /*genUuid*/);
        manager.moveToTop(tablePath);

      }
    };
  }


  private void deleteTable(OutputHierarchy out) throws CLIProcessingException {
    final String path = getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId());
    try {

      // Run impersonation
      new FileclientRun(getUserLoginId()) {
        @Override
        public void runAsProxyUser() throws IOException, CLIProcessingException {
          MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
          if (mfs.getMapRFileStatus(new Path(path)).isTable()) {
            if (LOG.isDebugEnabled()) {
              LOG.debug("Attempting to delete table from File System");
            }

            // Check result of delete
            boolean result = mfs.delete(new Path(path));
            if (!result) {
              // Throw exception if delete had a problem
              throw new CLIProcessingException("Table delete failed for path: " + path);
            } else {
              if (LOG.isDebugEnabled()) {
                LOG.debug("Deleting table from recent table list");
              }
              // Remove table from recent list
              RecentTablesListManagers.getRecentTablesListManagerForUser(getUserLoginId()).delete(path);
            }
          } else {
            // Throw exception for table not existing
            throw new CLIProcessingException("Table delete failed. Path '" + path + "' is not a MapR table.");
          }
        }
      };

    } catch (CLIProcessingException e) {
      // Catch exception
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }
  }

  private void listRecentTables(OutputHierarchy out) throws CLIProcessingException {

    RecentTablesListManager manager = RecentTablesListManagers.getRecentTablesListManagerForUser(getUserLoginId());
    boolean hasHomeDir = manager.hasHomeDir();

    if (LOG.isDebugEnabled() && hasHomeDir) {
        LOG.debug("Home Directory /user/" + getUserLoginId() + " found");
    }

    if (isParamPresent(PATH_PARAM_NAME)) {
      String path = getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId());
      try {
        if (MapRCliUtil.getMapRFileSystem().getMapRFileStatus(new Path(path)).isTable()) {
          OutputNode tableNode = new OutputNode();
          tableNode.addChild(new OutputNode(getOutputFieldName("path"), path));
          out.addNode(tableNode);
        } else {
          // delete from recent tables, if it's not already deleted
          if (LOG.isDebugEnabled()) {
              LOG.debug("Deleting table from recent table list");
          }

          manager.delete(path);
          out.addError(new OutputError(Errno.EINVAL, "Table not found. Path: " + path));
        }
      } catch (IOException e) {
        out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
      }
    } else if (!hasHomeDir) {
      out.addError(new OutputError(Errno.EINVAL, "Home directory (/user/" + getUserLoginId() + ") is missing for " +
          "this user. Please create it in order to cache the recent tables administered by this user."));
    } else {
        HashSet<String> recentPaths = new HashSet<String>();
        String currentClusterName = CLDBRpcCommonUtils.getInstance().getCurrentClusterName();
        for (String path : manager.getListFromFile()) {
            String recentPath = null;
            String[] splitPaths = path.split("/");
            if (path.startsWith("/mapr") && splitPaths.length > 2 && path.split("/")[2].equals(currentClusterName))
                recentPath = path.substring(path.lastIndexOf('/'));
            else
                recentPath = path;

            if (!recentPaths.contains(recentPath)) {
                OutputNode tableNode = new OutputNode();
                tableNode.addChild(new OutputNode(getOutputFieldName("path"), recentPath));
                out.addNode(tableNode);
                recentPaths.add(recentPath);
            }
        }
    }
  }

  private void getTableProps(final Path tablePath) throws IOException, CLIProcessingException {
    // Call impersonation
    new FileclientRun(getUserLoginId()) {
      @Override
      public void runAsProxyUser() throws IOException, CLIProcessingException{
        final MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
        // Get tablet data information
        TableProperties tableProp = mfs.getTableProperties(tablePath);
        setTableProps(tableProp);
      }
    };
  }

  private void setTableProps(TableProperties tableProp)
  {
    this.tableProp = tableProp;
  }

  private void tableInfo(OutputHierarchy out) throws CLIProcessingException {
    RecentTablesListManager manager = RecentTablesListManagers.getRecentTablesListManagerForUser(getUserLoginId());
    boolean hasHomeDir = manager.hasHomeDir();

    if (LOG.isDebugEnabled() && hasHomeDir) {
        LOG.debug("Home Directory /user/" + getUserLoginId() + " found");
    }

    String path = getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId());
    TabletStats tabletStats = new TabletStats(path, getUserLoginId());
    try {
      final Path tablePath = new Path(path);
      if (MapRCliUtil.getMapRFileSystem().getMapRFileStatus(tablePath).isTable()) {

        getTableProps(tablePath);
        long physicalSize = 0;
        long logicalSize = 0;
        long numRows = 0;
        long remoteSize = 0;
        long numSpills = 0;
        long numSegments = 0;
        int numRegions = 0;

        // These counters are valid only if all tablets are tracking them.
        // (they were not tracked in older releases).
        boolean numSegmentsValid = true;
        boolean numSpillsValid = true;
        boolean remoteSizeValid = true;

        // TODO - is numRowsWithDelete not printed on purpose here?

        List<TabletDesc> tablets = tabletStats.getTablets(out, 0, Integer.MAX_VALUE);
        if (tablets == null) {
          // Tablets had an error, return
          return;
        }

        numRegions += tablets.size();
        // Iterate through each tablet to get size
        for (TabletDesc tablet : tablets) {
          try {
            // blocks until a response is available
            TabletStatResponse tsr = tabletStats.getTabletStatResponse(tablet);
            if (tsr != null && tsr.hasUsage()) {
              SpaceUsage su = tsr.getUsage();
              long blockSize = 8 * 1024L;
              physicalSize += su.getNumPhysicalBlocks() * blockSize;
              logicalSize += su.getNumLogicalBlocks() * blockSize;
              numRows += su.getNumRows();
              if (su.hasNumRemoteBlocks())
                remoteSize += su.getNumRemoteBlocks() * blockSize;
              else
                remoteSizeValid = false;

              if (su.hasNumSpills())
                numSpills += su.getNumSpills();
              else
                numSpillsValid = false;

              if (su.hasNumSegments())
                numSegments += su.getNumSegments();
              else
                numSegmentsValid = false;
            }
          } catch (Exception e) {
            LOG.error("Error fetching tablet stats for fid: " +
                      MapRCliUtil.getFidAsString(tablet.getFid()), e);
          }
        }

        TableAttr attr = tableProp.getAttr();
        TableAces aces = tableProp.getAces();

        // Add all items to output
        OutputNode tableNode = new OutputNode();
        tableNode.addChild(new OutputNode(getOutputFieldName("path"), path));
        tableNode.addChild(new OutputNode(getOutputFieldName("numregions"), numRegions));
        tableNode.addChild(new OutputNode(getOutputFieldName("totallogicalsize"), logicalSize));
        tableNode.addChild(new OutputNode(getOutputFieldName("totalphysicalsize"), physicalSize));
        if (remoteSizeValid)
          tableNode.addChild(new OutputNode(getOutputFieldName("totalcopypendingsize"), remoteSize));
        tableNode.addChild(new OutputNode(getOutputFieldName("totalrows"), numRows));
        if (numSpillsValid)
          tableNode.addChild(new OutputNode(getOutputFieldName("totalnumberofspills"), numSpills));
        if (numSegmentsValid)
          tableNode.addChild(new OutputNode(getOutputFieldName("totalnumberofsegments"), numSegments));
        tableNode.addChild(new OutputNode(getOutputFieldName(AUTOSPLIT_PARAM_NAME), attr.getAutoSplit()));
        tableNode.addChild(new OutputNode(getOutputFieldName(BULKLOAD_PARAM_NAME), attr.getBulkLoad()));
        tableNode.addChild(new OutputNode(getOutputFieldName(INSERTION_ORDER_JSON), attr.getInsertionOrder()));

        String tblType = attr.getJson() ? TABLE_TYPE_JSON : TABLE_TYPE_DEF_BIN;
        tableNode.addChild(new OutputNode(getOutputFieldName(TABLE_TYPE_PARAM_NAME), tblType));

        tableNode.addChild(new OutputNode(getOutputFieldName(REGION_SIZE_MB_NAME), attr.getRegionSizeMB()));
        tableNode.addChild(new OutputNode(getOutputFieldName(AUDIT_PARAM_NAME),
                                          tableProp.getAuditEnabled()));
        if (attr.hasMaxValueSzInMemIndex())
          tableNode.addChild(new OutputNode(getOutputFieldName(MAX_VALUES_SIZE_IN_MEM_NAME), attr.getMaxValueSzInMemIndex()));
        if (attr.hasReclaimThreshPcntForPack())
          tableNode.addChild(new OutputNode(getOutputFieldName(RECLAIM_THRESH_PCNT_FOR_PACK_NAME), attr.getReclaimThreshPcntForPack()));
        if (attr.hasSizeThreshPcntForPack())
          tableNode.addChild(new OutputNode(getOutputFieldName(SIZE_THRESH_PCNT_FOR_PACK_NAME), attr.getSizeThreshPcntForPack()));
        if (attr.hasMaxSpills())
          tableNode.addChild(new OutputNode(getOutputFieldName(MAX_SPILLS_PARAM_NAME), attr.getMaxSpills()));
        if (attr.hasMiniPack())
          tableNode.addChild(new OutputNode(getOutputFieldName(MINI_PACK_PARAM_NAME), attr.getMiniPack()));
        if (attr.hasDeleteTTL())
          tableNode.addChild(new OutputNode(getOutputFieldName(DELETE_TTL_PARAM_NAME), attr.getDeleteTTL()));
        if (attr.hasHasReplication())
          tableNode.addChild(new OutputNode(getOutputFieldName(HAS_REPLICATION_PARAM_NAME), attr.getHasReplication()));
        if (attr.hasSyncReplTimeoutMillis())
          tableNode.addChild(new OutputNode(getOutputFieldName(SYNC_REPL_TIMEOUT_PARAMNAME), attr.getSyncReplTimeoutMillis()));
        if (attr.hasDropLargeRows())
          tableNode.addChild(new OutputNode(getOutputFieldName(DROP_LARGE_ROWS_PARAM_NAME), attr.getDropLargeRows()));
        if (attr.hasTtlCompaction()) {
          tableNode.addChild(new OutputNode(getOutputFieldName(TTL_COMPACT_PARAM_NAME), attr.getTtlCompaction()));
          if (attr.getTtlCompaction() && attr.hasTtlCompactionHrs()) {
            tableNode.addChild(new OutputNode(getOutputFieldName(TTL_COMPACT_HRS_PARAM_NAME),
                               attr.getTtlCompactionHrs()));
          }
        }

        for (AccessControlExpression ace : aces.getAcesList()) {
          tableNode.addChild(new OutputNode(
              getOutputFieldName(AceHelper.tblPermissionMap.get(ace.getAccessType())),
              AceHelper.toInfix(ace.getBooleanExpression().toStringUtf8())));
        }
        for (AccessControlExpression ace : aces.getDefaultColumnFamilyAcesList()) {
          tableNode.addChild(new OutputNode(
              getOutputFieldName(AceHelper.cfDefPermissionMap.get(ace.getAccessType())),
              AceHelper.toInfix(ace.getBooleanExpression().toStringUtf8())));
        }

        byte[] uuid = tableProp.getUuid();
        if (uuid != null && uuid.length != 0) {
          tableNode.addChild(new OutputNode(getOutputFieldName("uuid"),
                             BinaryString.toUUIDString(uuid)));
        }

        if (attr.hasIsMarlinTable())
          tableNode.addChild(new OutputNode("ismarlintable", attr.getIsMarlinTable()));

        out.addNode(tableNode);
      } else {
        // delete from recent tables, if it's not already deleted
        if (LOG.isDebugEnabled()) {
          LOG.debug("Deleting table from recent table list");
        }

        manager.delete(path);
        out.addError(new OutputError(Errno.EINVAL, "Table not found. Path: " + path));
      }
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    }

  }

  private void copyTable(OutputHierarchy out) throws CLIProcessingException {
    try
     {
        if (LOG.isDebugEnabled())
        {
          LOG.debug("Home Directory /user/" + getUserLoginId() + " found");
        }

        // Get arguments
        String srcTable = getTransformedPath(getParamTextValue(SOURCE_TABLE_NAME, 0), getUserLoginId());
        String dstTable = getTransformedPath(getParamTextValue(DESTINATION_TABLE_NAME, 0),getUserLoginId());
        String colList = isParamPresent(COLUMNS_PARAM_NAME) ? getParamTextValue(COLUMNS_PARAM_NAME, 0) : null;
        String user = isParamPresent(USER_PARAM_NAME) ? getParamTextValue(USER_PARAM_NAME, 0) : getUserLoginId();
        int maxVersions = isParamPresent(MAX_VER_PARAM_NAME) ? getParamIntValue(MAX_VER_PARAM_NAME, 0) : Integer.MAX_VALUE;
        long startTime = isParamPresent(START_TIME_PARAM_NAME) ? getParamLongValue(START_TIME_PARAM_NAME, 0) : 0;
        long endTime = isParamPresent(END_TIME_PARAM_NAME) ? getParamLongValue(END_TIME_PARAM_NAME, 0) : Long.MAX_VALUE;
        boolean useMR = getParamBooleanValue(USE_MR_PARAM_NAME, 0);
        boolean useBulkLoad = getParamBooleanValue(BULKLOAD_PARAM_NAME, 0);
        boolean isExternal = DbReplicaCommands.isExternalDestination(dstTable);

        getTableProps(new Path(srcTable));
        boolean isJson = tableProp.getAttr().getJson();
        if (isJson && (isParamPresent(MAX_VER_PARAM_NAME) ||
                       isParamPresent(START_TIME_PARAM_NAME) ||
                       isParamPresent(END_TIME_PARAM_NAME))) {
          throw new CLIProcessingException("CopyTable for JSON-DB tables does not support multiple versions.");
        }

        // Build the list of arguments that will be passed to the CopyTable job
        ArrayList<String> args = new ArrayList<String>();
        args.add("-" + SOURCE_TABLE_NAME); args.add(srcTable);
        args.add("-" + DESTINATION_TABLE_NAME); args.add(dstTable);
        if (maxVersions < Integer.MAX_VALUE) {
          args.add("-" + MAX_VER_PARAM_NAME); args.add(String.valueOf(maxVersions));
        }
        if (startTime > 0) {
          args.add("-" + START_TIME_PARAM_NAME); args.add(String.valueOf(startTime));
        }
        if (endTime < Long.MAX_VALUE) {
          args.add("-" + END_TIME_PARAM_NAME); args.add(String.valueOf(endTime));
        }
        if (colList != null) {
          args.add("-" + COLUMNS_PARAM_NAME); args.add(colList);
        }
        // Normal and external destinations don't recognize all the same options
        if (!isExternal) {
          args.add("-" + BULKLOAD_PARAM_NAME); args.add(String.valueOf(useBulkLoad));
        }

        String zkConnectString = CLDBRpcCommonUtils.getInstance().getZkConnect();
        LOG.info("Looking up ticket for user: "+user+" in location "+System.getenv("MAPR_TICKETFILE_LOCATION"));
        //Figure out if JT/RM is running
        if (useMR &&
            (NodesCommonUtils.isServiceAvailable(zkConnectString, ServicesEnum.jobtracker.name()) ||
            NodesCommonUtils.isServiceAvailable(zkConnectString, ServicesEnum.resourcemanager.name()))) {
          ClusterCommands.VersionFileContents fileContents = ClusterCommands.readHadoopVersionFile();
          //Check if mode is yarn and RM is running
          if(fileContents.default_mode.equalsIgnoreCase("yarn")
              && !(NodesCommonUtils.isServiceAvailable(zkConnectString, ServicesEnum.resourcemanager.name()))) {
            throw new CLIProcessingException("Please check if map-reduce mode is yarn and RM is installed and running");
          }
          //Check if mode is classic and JT is running
          if (fileContents.default_mode.equalsIgnoreCase("classic")
              && !(NodesCommonUtils.isServiceAvailable(zkConnectString, ServicesEnum.jobtracker.name()))) {

            throw new CLIProcessingException("Please check if map-reduce mode is classic and JT is installed and running");
          }

          boolean result = false;
          LOG.info("Calling copytable job for tables: "+srcTable+","+dstTable+" by user: "+user);
          //set map-reduce to true
          args.add("-" + USE_MR_PARAM_NAME); args.add(String.valueOf(true));
          if (isExternal) {
            DbReplicaCommands.verifyExternalDstSanity(dstTable);
            result = copyTableExternal(args.toArray(new String[args.size()]), user, true, isJson);
          } else {
            result = copyTable(args.toArray(new String[args.size()]), user, isJson);
          }

          if (!result) {
            throw new CLIProcessingException("CopyTable job failed for tables: "+srcTable+" "+dstTable);
          }
          LOG.info("Finished copy table from table: "+srcTable+" to "+dstTable+" by user: "+user);

        } else {
          boolean result = false;
          LOG.info("Job tracker or Resource Manager not found. Running non-mapreduce copytable for tables: "+srcTable+","+dstTable+" by user: "+user);
          //set map-reduce to true
          args.add("-" + USE_MR_PARAM_NAME); args.add(String.valueOf(false));
          if (isExternal) {
            DbReplicaCommands.verifyExternalDstSanity(dstTable);
            result = copyTableExternal(args.toArray(new String[args.size()]), user, false, isJson);
          } else {
            result = copyTableNoMR(args.toArray(new String[args.size()]), user, isJson);
          }
          if (!result) {
            throw new CLIProcessingException("Non-mapreduce copyTable job failed for tables: "+srcTable+" "+dstTable);
          }
          LOG.info("Finished non-mapreduce copy table from table: "+srcTable+" to "+dstTable+" by user: "+user);
        }
     } catch (IOException e) {
       // Catch exception
       out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
     } catch (Exception e) {
       out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
     }
   }

  public static boolean copyTable(String[] args, String user, boolean isJson) throws IOException, CLIProcessingException,Exception
  {
    if (!isJson) {
      return com.mapr.fs.hbase.tools.mapreduce.CopyTable.copy(args, user);
    } else {
      return com.mapr.db.mapreduce.tools.CopyTable.copy(args, user);
    }
  }

  // To avoid putting a dependency on the External package, we attempt to load
  // the CopyTable code on the fly from a known location.  The External package
  // does not have to be installed, so we can't have a static reference to in
  // this code.  This function handles mapreduce and non-mapreduce CopyTable
  // calls.
  private static boolean copyTableExternal(String[] args, String user, boolean useMR, boolean isJson)
    throws IOException, CLIProcessingException {
    final String BINARY_COPYTABLE_NAME = "com.mapr.fs.gateway.external.tools.CopyTableExt";
    final String JSON_COPYTABLE_NAME = "com.mapr.fs.gateway.external.tools.CopyTableJsonExt";
    final String COPY_METHOD_NAME = "copy";

    String className = isJson ? JSON_COPYTABLE_NAME : BINARY_COPYTABLE_NAME;
    boolean result = false;

    try {
      ClassLoader parentLoader = ClassLoader.getSystemClassLoader();
      Class copyClass = parentLoader.loadClass(className);
      Class[] copyArgs = new Class[3];
      copyArgs[0] = String[].class;
      copyArgs[1] = String.class;
      copyArgs[2] = boolean.class;
      Method copyMethod = copyClass.getDeclaredMethod(COPY_METHOD_NAME, copyArgs);
      result = (Boolean) copyMethod.invoke(null, args, user, useMR);
    } catch (ClassNotFoundException e) {
      LOG.error("Could not find class " + className);
      throw new CLIProcessingException(e.getMessage());
    } catch (NoSuchMethodException e) {
      LOG.error("Copy class " + className + " does not contain a " + COPY_METHOD_NAME + " method");
      throw new CLIProcessingException(e.getMessage());
    } catch (Exception e) {
      throw new CLIProcessingException(e.getMessage());
    }

    return result;
  }

  public static boolean copyTableNoMR(String[] args, String user, boolean isJson) throws IOException, CLIProcessingException, Exception {
    if (!isJson) {
      return com.mapr.fs.hbase.tools.mapreduce.CopyTable.copy(args, user);
    } else {
      return com.mapr.db.mapreduce.tools.CopyTable.copy(args, user);
    }
  }

  private void copyTableStatus(OutputHierarchy out) throws CLIProcessingException {
    try
     {
        if (LOG.isDebugEnabled())
        {
          LOG.debug("Home Directory /user/" + getUserLoginId() + " found");
        }
        // Get source and destination table paths
        String srcTable = getTransformedPath(getParamTextValue(SOURCE_TABLE_NAME, 0), getUserLoginId());
        String dstTable = getTransformedPath(getParamTextValue(DESTINATION_TABLE_NAME, 0),getUserLoginId());
        LOG.info("Retrieving the copy table status for tables: "+srcTable+","+dstTable);
        if(JobExecutor.getStatus(srcTable+"_"+dstTable) != null)
        {
          LOG.info("Status: "+JobExecutor.getStatus(srcTable+"_"+dstTable));
          out.addNode(new OutputNode("Status",JobExecutor.getStatus(srcTable+"_"+dstTable)));
        }

     } catch (Exception e) {
       out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
     }
   }

  private String getOutputFieldName(String verboseName) throws CLIProcessingException {
    return "terse".equals(getParamTextValue(OUTPUT_PARAM_NAME, 0)) ? verboseToTerseMap.get(verboseName) : verboseName;
  }

  public static String getTransformedPath(String path, String user) {
    if (path.contains("\\")) {
      path = path.replace('\\', '/');
    }

    if (!path.startsWith("/")) {
      path = "/user/" + user + "/" + path;
    }
    if (LOG.isDebugEnabled()) {
        LOG.debug("Query path: " + path + ", User: " + user);
    }

    return path;
  }

  @Override
  public String getCliParam(String key) throws IOException{
    String ret = null;
    // If permissions map val is a passed as parameter, then add to ace list
    try {
      if (isParamPresent(key)) {
        ret = getParamTextValue(key, 0);
        if (ret != null) {
          ret = ret.trim();
        }
      }
    } catch (CLIProcessingException e) {
      throw new IOException(e);
    }
    return ret;
  }

  private UserGroupInformation getProxyUser() throws IOException {
    UserGroupInformation currUser = UserGroupInformation.getLoginUser();
    if (currUser.getUserName().equals(getUserLoginId())) {
      return currUser;
    }
    return UserGroupInformation.createProxyUser(getUserLoginId(), currUser);
  }

}

