/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.ByteString;
import com.mapr.baseutils.Errno;
import com.mapr.cli.table.RecentTablesListManager;
import com.mapr.cli.table.RecentTablesListManagers;
import com.mapr.cliframework.base.*;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.inputparams.*;
import com.mapr.cli.common.FileclientRun;
import com.mapr.fs.MapRFileSystem;
import com.mapr.fs.AceHelper;
import com.mapr.fs.CompressionHelper;
import com.mapr.fs.proto.Common.FileCompressionType;
import com.mapr.fs.proto.Dbserver;
import com.mapr.fs.proto.Dbserver.SchemaFamily;
import com.mapr.fs.proto.Dbserver.DBAccessType;
import com.mapr.fs.proto.Dbserver.ColumnAttr;
import com.mapr.fs.proto.Dbserver.ColumnFamilyAttr;
import com.mapr.fs.proto.Dbserver.AccessControlExpression;

import org.apache.hadoop.fs.Path;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.json.JSONTokener;
import org.ojai.FieldPath;
import org.ojai.FieldSegment;

import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.Iterator;

import com.mapr.fs.proto.Dbserver.DBInternalDefaults;
import com.mapr.fs.tables.TableProperties;

import org.apache.log4j.Logger;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

public class DbCfCommands extends CLIBaseClass implements CLIInterface,AceHelper.DBPermission {

  private static final Logger LOG = Logger.getLogger(DbCfCommands.class);
  private static final String PATH_PARAM_NAME = "path";
  private static final String MIN_VERSIONS_PARAM_NAME = "minversions";
  private static final String MAX_VERSIONS_PARAM_NAME = "maxversions";
  private static final String TTL_PARAM_NAME = "ttl";
  private static final String IN_MEMORY_PARAM_NAME= "inmemory";
  private static final String COMPRESSION_PARAM_NAME = "compression";
  private static final String CFNAME_PARAM_NAME = "cfname";
  private static final String NEWCFNAME_PARAM_NAME = "newcfname";
  private static final String PERM_VERSIONS_PARAM_NAME = "versionperm";
  private static final String PERM_COMPRESSION_PARAM_NAME = "compressionperm";
  private static final String PERM_MEMORY_PARAM_NAME = "memoryperm";
  private static final String PERM_READ_PARAM_NAME = "readperm";
  private static final String PERM_WRITE_PARAM_NAME = "writeperm";
  private static final String PERM_APPEND_PARAM_NAME = "appendperm";
  private static final String PERM_ENCRYPT_PARAM_NAME = "encryptperm";
  private static final String PERM_TRAVERSE_PARAM_NAME = "traverseperm";
  private static final String COLUMN_PERMISSIONS_PARAM_NAME = "columnperm";
  private static final String JSON_FAMILY_PATH = "jsonpath";
  private static final String JSON_FORCE_CF_CREATE = "force";

  public static final String COLUMNS_PARAM_NAME = "columns";
  public static final String OUTPUT_PARAM_NAME = "output";
  public static final String START_PARAM_NAME = "start";
  public static final String LIMIT_PARAM_NAME = "limit";
  public static final String SHOW_COLUMNS_NAME = "showcol";

  public static final int DEFAULT_TTL = Integer.MAX_VALUE;


  private static final CLICommand listCommand =
      new CLICommand(
          "list",
          "usage: table cf list -path <tablepath>",
          DbCfCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "Table path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(CFNAME_PARAM_NAME,
                  new TextInputParameter(CFNAME_PARAM_NAME,
                      "Column family name",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(COLUMNS_PARAM_NAME,
                  new TextInputParameter(COLUMNS_PARAM_NAME,
                      "columns",
                      CLIBaseClass.NOT_REQUIRED,
                      "all").setInvisible(true))
              .put(OUTPUT_PARAM_NAME,
                  new TextInputParameter(OUTPUT_PARAM_NAME,
                      "verbose|terse",
                      CLIBaseClass.NOT_REQUIRED,
                      "verbose").setInvisible(true))
              .put(START_PARAM_NAME,
                  new IntegerInputParameter(START_PARAM_NAME,
                      "start",
                      CLIBaseClass.NOT_REQUIRED,
                      0).setInvisible(true))
              .put(LIMIT_PARAM_NAME,
                  new IntegerInputParameter(LIMIT_PARAM_NAME,
                      "limit",
                      CLIBaseClass.NOT_REQUIRED,
                      Integer.MAX_VALUE).setInvisible(true))
              .put(SHOW_COLUMNS_NAME,
                    new BooleanInputParameter(SHOW_COLUMNS_NAME,
                      "showcol",
                      CLIBaseClass.NOT_REQUIRED,
                      false).setInvisible(true))
              .build(), null)
          .setShortUsage("table cf list -path <tablepath>");

  private static final CLICommand createCommand =
      new CLICommand(
          "create",
          "usage: table cf create -path <tablepath> -cfname <cfname> -minversions <minversions> -maxversions<maxversions> -ttl <ttl> -inmemory <inmemory> -compression <off|lzf|lz4|zlib> -jsonpath <dot.separated.path>",
          DbCfCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "Table path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(CFNAME_PARAM_NAME,
                  new TextInputParameter(CFNAME_PARAM_NAME,
                      "Column family name",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(MIN_VERSIONS_PARAM_NAME,
                  new IntegerInputParameter(MIN_VERSIONS_PARAM_NAME,
                      "Min versions to keep",
                      CLIBaseClass.NOT_REQUIRED,
                      0))
              .put(MAX_VERSIONS_PARAM_NAME,
                  new IntegerInputParameter(MAX_VERSIONS_PARAM_NAME,
                      "Max versions to keep",
                      CLIBaseClass.NOT_REQUIRED,
                      1))
              .put(TTL_PARAM_NAME,
                  new LongInputParameter(TTL_PARAM_NAME,
                      "Time to live. Enter 0 for forever. Otherwise enter time in seconds. default: 0",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(IN_MEMORY_PARAM_NAME,
                  new BooleanInputParameter(IN_MEMORY_PARAM_NAME,
                      "In-memory",
                      CLIBaseClass.NOT_REQUIRED,
                      false))
              .put(COMPRESSION_PARAM_NAME,
                  new TextInputParameter(COMPRESSION_PARAM_NAME,
                      "off|lzf|lz4|zlib. default: table's compression setting is applied.",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_VERSIONS_PARAM_NAME,
                  new TextInputParameter(PERM_VERSIONS_PARAM_NAME,
                      "Version Permissions for binary tabletype",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
               .put(PERM_COMPRESSION_PARAM_NAME,
                  new TextInputParameter(PERM_COMPRESSION_PARAM_NAME,
                      "Compression Permissions",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_MEMORY_PARAM_NAME,
                  new TextInputParameter(PERM_MEMORY_PARAM_NAME,
                      "Memory Permissions",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
               .put(PERM_READ_PARAM_NAME,
                  new TextInputParameter(PERM_READ_PARAM_NAME,
                      "Read Permissions",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_WRITE_PARAM_NAME,
                  new TextInputParameter(PERM_WRITE_PARAM_NAME,
                      "Write Permissions",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_APPEND_PARAM_NAME,
                  new TextInputParameter(PERM_APPEND_PARAM_NAME,
                      "Append Permissions for binary tabletype",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_TRAVERSE_PARAM_NAME,
                      new TextInputParameter(PERM_TRAVERSE_PARAM_NAME,
                          "Traverse Permissions for json tabletype",
                          CLIBaseClass.NOT_REQUIRED,
                          null))
              .put(JSON_FAMILY_PATH,
                  new TextInputParameter(JSON_FAMILY_PATH,
                      "Json Family Path - needed for JSON column family, like a.b.c",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(JSON_FORCE_CF_CREATE,
                  new TextInputParameter(JSON_FORCE_CF_CREATE,
                      "Force create non-default column family for json tabletype. default: false",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_ENCRYPT_PARAM_NAME,
                  new TextInputParameter(PERM_ENCRYPT_PARAM_NAME,
                      "Encrypt Permissions",
                      CLIBaseClass.NOT_REQUIRED,
                      null).setInvisible(true))
              .put(COLUMN_PERMISSIONS_PARAM_NAME,
                  new TextInputParameter(COLUMN_PERMISSIONS_PARAM_NAME,
                      "Column Specific Permissions",
                      CLIBaseClass.NOT_REQUIRED,
                      null).setInvisible(true))
              .build(), null)
          .setShortUsage("table cf create -path <tablepath> -cfname <cfname>");

  private static final CLICommand editCommand =
      new CLICommand(
          "edit",
          "usage: table cf edit -path <tablepath> -cfname <cfname> -minversions <minversions> -maxversions<maxversions> -ttl <ttl> -inmemory <inmemory> -compression <off|lzf|lz4|zlib>",
          DbCfCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "Table path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(CFNAME_PARAM_NAME,
                  new TextInputParameter(CFNAME_PARAM_NAME,
                      "Column family name",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(NEWCFNAME_PARAM_NAME,
                  new TextInputParameter(NEWCFNAME_PARAM_NAME,
                      "New column family name",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(MIN_VERSIONS_PARAM_NAME,
                  new IntegerInputParameter(MIN_VERSIONS_PARAM_NAME,
                      "Min versions to keep",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(MAX_VERSIONS_PARAM_NAME,
                  new IntegerInputParameter(MAX_VERSIONS_PARAM_NAME,
                      "Max versions to keep",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(TTL_PARAM_NAME,
                  new LongInputParameter(TTL_PARAM_NAME,
                      "Time to live (in seconds)",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(IN_MEMORY_PARAM_NAME,
                  new BooleanInputParameter(IN_MEMORY_PARAM_NAME,
                      "In-memory",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(COMPRESSION_PARAM_NAME,
                  new TextInputParameter(COMPRESSION_PARAM_NAME,
                      "off|lzf|lz4|zlib",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_VERSIONS_PARAM_NAME,
                  new TextInputParameter(PERM_VERSIONS_PARAM_NAME,
                      "Version Permissions for binary tabletype",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
               .put(PERM_COMPRESSION_PARAM_NAME,
                  new TextInputParameter(PERM_COMPRESSION_PARAM_NAME,
                      "Compression Permissions",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_MEMORY_PARAM_NAME,
                  new TextInputParameter(PERM_MEMORY_PARAM_NAME,
                      "Memory Permissions",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
               .put(PERM_READ_PARAM_NAME,
                  new TextInputParameter(PERM_READ_PARAM_NAME,
                      "Read Permissions",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_WRITE_PARAM_NAME,
                  new TextInputParameter(PERM_WRITE_PARAM_NAME,
                      "Write Permissions",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_APPEND_PARAM_NAME,
                  new TextInputParameter(PERM_APPEND_PARAM_NAME,
                      "Append Permissions for binary tabletype",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PERM_ENCRYPT_PARAM_NAME,
                  new TextInputParameter(PERM_ENCRYPT_PARAM_NAME,
                      "Encrypt Permissions",
                      CLIBaseClass.NOT_REQUIRED,
                      null).setInvisible(true))
              .put(COLUMN_PERMISSIONS_PARAM_NAME,
                  new TextInputParameter(COLUMN_PERMISSIONS_PARAM_NAME,
                      "Column Specific Permissions",
                      CLIBaseClass.NOT_REQUIRED,
                      null).setInvisible(true))
              .put(PERM_TRAVERSE_PARAM_NAME,
                  new TextInputParameter(PERM_TRAVERSE_PARAM_NAME,
                      "Traverse Permissions for json tabletype",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .build(), null)
          .setShortUsage("table cf edit -path <tablepath> -cfname <cfname> -minversions <minversions> -maxversions<maxversions> -ttl <ttl> -inmemory <inmemory> -compression <off|lzf|lz4|zlib>");

  private static final CLICommand deleteCommand =
      new CLICommand(
          "delete",
          "usage: table cf delete -path <tablepath>",
          DbCfCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "Table path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(CFNAME_PARAM_NAME,
                  new TextInputParameter(CFNAME_PARAM_NAME,
                      "Column family name",
                      CLIBaseClass.REQUIRED,
                      null))
              .build(), null)
          .setShortUsage("table cf delete -path <tablepath> -cfname <cfname>");

  // main command
  public static final CLICommand cfCommands =
      new CLICommand(
          "cf", "cf [create|edit|delete|list|colperm]",
          CLIUsageOnlyCommand.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new CLICommand[]{
              createCommand,
              editCommand,
              deleteCommand,
              listCommand,
              DbCfColCommands.cfColCommands
          }
      ).setShortUsage("table cf [create|edit|delete|list|col]");

  private static final Map<String, String> verboseToTerseMap = new ImmutableMap.Builder<String, String>()
      .put("cfname", "n")
      .put("maxversions", "vmax")
      .put("minversions", "vmin")
      .put("ttl", "ttl")
      .put("inmemory", "inmem")
      .put("compression", "comp")
      .put("versionperm", "pver")
      .put("compressionperm", "pcomp")
      .put("memoryperm", "pmem")
      .put("readperm", "pread")
      .put("writeperm", "pwrite")
      .put("columnperm", "pcol")
      .put("appendperm", "papp")
      .put("encryptperm", "penc")
      .put("traverseperm", "ptraverse")
      .put("jsonfamilypath", "jpath")
      .build();

  public DbCfCommands(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }

  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    if (!super.validateInput()) {
      return output;
    }

    if (cliCommand.getCommandName().equalsIgnoreCase(createCommand.getCommandName())) {
      createCf(out);
    } else if (cliCommand.getCommandName().equalsIgnoreCase(editCommand.getCommandName())) {
      editCf(out);
    } else if (cliCommand.getCommandName().equalsIgnoreCase(deleteCommand.getCommandName())) {
      deleteCf(out);
    } else if (cliCommand.getCommandName().equalsIgnoreCase(listCommand.getCommandName())) {
      listCf(out);
    }

    return output;
  }

  private FileCompressionType getCompressionType(String cs) {
    if (cs.equalsIgnoreCase("off"))
      return FileCompressionType.FCT_OFF;
    else if (cs.equalsIgnoreCase("lz4"))
      return FileCompressionType.FCT_LZ4;
    else if (cs.equalsIgnoreCase("lzf"))
      return FileCompressionType.FCT_LZF;
    else if (cs.equalsIgnoreCase("zlib"))
      return FileCompressionType.FCT_ZLIB;
    else
      return null;
  }

  private String getCompressionName(FileCompressionType ct) {
    if (ct == FileCompressionType.FCT_OFF)
      return "off";
    else if (ct == FileCompressionType.FCT_LZ4)
      return "lz4";
    else if (ct == FileCompressionType.FCT_ZLIB)
      return "zlib";
    else if (ct == FileCompressionType.FCT_LZF ||
             ct == FileCompressionType.FCT_OLDLZF)
      return "lzf";
    else
      return "unknown";
 }

 // Function validates that a given string is a valid json family path
 // it has to be in the form a.b.c and it can't contain array element in it
 private boolean validateFamilyPath(String s) {
  try {
    // Zero size string is not a valid field path for CF create
    if ((s != null) && s.length() == 0) {
      return false;
    }
    FieldPath fieldPath = FieldPath.parseFrom(s);
    Iterator<FieldSegment> iter = fieldPath.iterator();
    while (iter.hasNext()) {
      FieldSegment seg = iter.next();
      if (seg.isArray()) {
        return false;
      }
    }
  } catch (Exception e) {
    return false;
  }
  return true;
 }


 private void createCf(OutputHierarchy out) throws CLIProcessingException {
   final String tablePath = DbCommands.getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId());
   final String cfName = getParamTextValue(CFNAME_PARAM_NAME, 0);

   Integer minVersions = getParamIntValue(MIN_VERSIONS_PARAM_NAME, 0);
   Integer maxVersions = getParamIntValue(MAX_VERSIONS_PARAM_NAME, 0);

   JSONArray columns = null;

   Long ttl = isParamPresent(TTL_PARAM_NAME) ? getParamLongValue(TTL_PARAM_NAME, 0) : null;
    Boolean inMem = isParamPresent(IN_MEMORY_PARAM_NAME) ? getParamBooleanValue(IN_MEMORY_PARAM_NAME, 0) : null;

    FileCompressionType ctype = null;
    if (isParamPresent(COMPRESSION_PARAM_NAME)) {
      ctype = CompressionHelper.getCompressionType(getParamTextValue(COMPRESSION_PARAM_NAME, 0));
      if (ctype == null) {
        out.addError(new OutputError(Errno.EINVAL, "Invalid input values. " +
          "The entered value for compression is not a supported type."));
        return;
      }
    }

    final AtomicBoolean isJsonTable = new AtomicBoolean(false);
    final AtomicBoolean isStream = new AtomicBoolean(false);
    boolean isDefaultCf =
      cfName.equals(DBInternalDefaults.getDefaultInstance().getJsonDefaultCFName());

    // Call impersonation
    try {
      new FileclientRun(getUserLoginId()) {
        @Override
        public void runAsProxyUser() throws IOException, CLIProcessingException {
          final MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
          // Get tablet data // information
          TableProperties tableProp = mfs.getTableProperties(new Path(tablePath));
          isJsonTable.set(tableProp.getAttr().getJson());
          isStream.set(tableProp.getAttr().getIsMarlinTable());
        }
      };
    } catch (IOException e) {
      throw new CLIProcessingException(e.getMessage());
    }

    if (minVersions != null && maxVersions != null && minVersions > maxVersions) {
      out.addError(new OutputError(Errno.EINVAL, "Invalid input values. The entered value for min versions: " +
          minVersions + " is greater than the value for max versions: " + maxVersions + "."));
      return;
    }

    ColumnFamilyAttr.Builder builder = ColumnFamilyAttr.newBuilder();

    SchemaFamily.Builder cfbuilder = SchemaFamily.newBuilder();
    cfbuilder.setName(cfName);
    if (maxVersions != null) {
      cfbuilder.setMaxVersions(maxVersions);
    }
    if (minVersions != null) {
      cfbuilder.setMinVersions(minVersions);
    }
    if (ttl != null) {
      if (isJsonTable.get() && !isStream.get()) {
        final AtomicInteger numCF = new AtomicInteger(0);
        // We don't have a mechanism to allow TTL for the full document when its
        // split into multiple CF. If multiple CF is there for document then we
        // don't support setting TTL on the default CF.
        try {
          new FileclientRun(getUserLoginId()) {
            @Override
            public void runAsProxyUser() throws IOException, CLIProcessingException {
              final MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
              // Get tablet data
              List<ColumnFamilyAttr> cfa = mfs.listColumnFamily(new Path(tablePath), false /*wantACES*/);
              numCF.set(cfa.size());
            }
          };
        } catch (IOException e) {
          throw new CLIProcessingException(e.getMessage());
        }

        if (numCF.get() >= 1) {
          out.addError(new OutputError(Errno.EINVAL, "TTL cannot be set when " +
                                       "multiple column families present in " +
                                       "json type of table."));
          return;
        }
      }

      cfbuilder.setTtl(ttl * 1000); // store the ttl value in ms
    }
    if (inMem != null) {
      cfbuilder.setInMemory(inMem);
    }
    if (ctype != null) {
      cfbuilder.setCompression(ctype);
    }

    builder.setSchFamily(cfbuilder.build());

    boolean isForceCreate = false;
    if (isParamPresent(JSON_FORCE_CF_CREATE)) {
      if (isJsonTable.get() == false) {
        out.addError(new OutputError(Errno.EINVAL, "Parameter " +
                                     JSON_FORCE_CF_CREATE + " can only be set "
                                     + "for json type of table"));
        return;
      }

      isForceCreate = true;
    }

    /*
     * Don't allow setting of family path for non json table
     * and also not for the default column family in json table
     */
    if (isParamPresent(JSON_FAMILY_PATH)) {
      String jsonFamilyPath = getParamTextValue(JSON_FAMILY_PATH, 0);

      if (isJsonTable.get() == false) {
        out.addError(new OutputError(Errno.EINVAL, "Parameter " +
                                     JSON_FAMILY_PATH + " can only be set for "
                                     + "json type of table"));
        return;
      }

      // If the default column family is getting created then the path must be
      if (isDefaultCf) {
        out.addError(new OutputError(Errno.EINVAL, "Parameter " +
                                     JSON_FAMILY_PATH + " can only be set for "
                                     + "non default column family in json type of table"));
        return;
      }

      if (!isDefaultCf && !isForceCreate) {
        out.addError(new OutputError(Errno.EINVAL, "ERROR: Please ensure that "+
                                     "none of the existing rows in the table " +
                                     tablePath + " contain data at path " +
                                     jsonFamilyPath + ". Use '-force true' option " +
                                     "to create the new column family " +
                                     "after ensuring no data exists at the " +
                                     "jsonpath. Creating a new column family " +
                                     "may leave any existing data at that path "+
                                     "inaccessible or cause unexpected " +
                                     "behavior."));
        return;
      }

      // Validate the family path. It must not contain an array element
      // or it must not be malformed path
      if (validateFamilyPath(jsonFamilyPath) == false) {
        out.addError(new OutputError(Errno.EINVAL, "Malformed path \""
                                     + jsonFamilyPath + "\", valid format "
                                     + "is like \"a.b.c\"."));
        return;
      }
      builder.setJsonFamilyPath(jsonFamilyPath);
    } else {
      if (isJsonTable.get() && !isDefaultCf) {
        out.addError(new OutputError(Errno.EINVAL, "Parameter " +
                                     JSON_FAMILY_PATH + " is required for "
                                     + "non default column family " + cfName +
                                     " in json table " + tablePath));
        return;
      }
    }


   try {
     if (isParamPresent(COLUMN_PERMISSIONS_PARAM_NAME)) {
       builder.addAllColumnAttr(getColumnList(toJSON(getParamTextValue(COLUMN_PERMISSIONS_PARAM_NAME, 0)), out));
     }
   } catch (CLIProcessingException e) {
     out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
     return;
   } catch (JSONException e) {
     out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
     return;
   }

   if (isJsonTable.get() && !isDefaultCf && !isStream.get()) {
     SchemaFamily defSF;

     try {
      ColumnFamilyAttr cfa = getColumnFamily(tablePath, DBInternalDefaults.getDefaultInstance().getJsonDefaultCFName());
      if (cfa == null) {
        out.addError(new OutputError(Errno.ENOTEXIST, "Column family default" +
                                     " is not defined for table '" +
                                     tablePath + "'."));
        return;
      }
      defSF = cfa.getSchFamily();
    } catch (IOException ioe) {
      out.addError(new OutputError(Errno.EOPFAILED, ioe.getMessage()));
      return;
    }

     if (defSF.hasTtl()) {
       out.addError(new OutputError(Errno.EINVAL, "Multiple column families " +
                                    "cannot be created when TTL is present in "+
                                    "default column family of json table."));

       return;
     }
   }

   // Set as final to be called in impersination
   final RecentTablesListManager manager = RecentTablesListManagers.getRecentTablesListManagerForUser(getUserLoginId());
   final ColumnFamilyAttr cfattr = builder.build();
   final AceHelper.DBPermission self = this;

   try {
     if (LOG.isDebugEnabled()) {
       LOG.debug("Attempting to create new column family -> Path: " + tablePath + ", Name: " + cfName);
     }
     // Call impersination
     new FileclientRun(getUserLoginId()) {
       @Override
       public void runAsProxyUser() throws IOException, CLIProcessingException{
         MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
         try {
           mfs.createColumnFamily(new Path(tablePath), cfName, cfattr, self);
           manager.moveToTop(tablePath);
         } catch (IOException e) {
           manager.deleteIfNotExist(tablePath, mfs);
           throw new CLIProcessingException(e.getMessage());
         }
       }
     };
   } catch (CLIProcessingException e) {
     out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
   } catch (IOException e) {
     out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
   }
  }

  private void editCf(OutputHierarchy out) throws CLIProcessingException {
    final String tablePath = DbCommands.getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId());
    final String currentCfName = getParamTextValue(CFNAME_PARAM_NAME, 0);
    String newCfName = isParamPresent(NEWCFNAME_PARAM_NAME) ? getParamTextValue(NEWCFNAME_PARAM_NAME, 0) : null;
    ColumnFamilyAttr.Builder builder;

    Integer minVersions = isParamPresent(MIN_VERSIONS_PARAM_NAME) ? getParamIntValue(MIN_VERSIONS_PARAM_NAME, 0) : null;
    Integer maxVersions = isParamPresent(MAX_VERSIONS_PARAM_NAME) ? getParamIntValue(MAX_VERSIONS_PARAM_NAME, 0) : null;

    List<ColumnAttr> colPerm = null;

    JSONArray columns = null;

    ColumnFamilyAttr cfa;
    SchemaFamily currentCf;

    try {
      cfa = getColumnFamily(tablePath, currentCfName);
      if (cfa == null) {
        out.addError(new OutputError(Errno.ENOTEXIST, "Column family '" +
                                     currentCfName + "' is not defined for table '" +
                                     tablePath + "'."));
        return;
      }
      currentCf = cfa.getSchFamily();
    } catch (IOException ioe) {
      out.addError(new OutputError(Errno.EOPFAILED, ioe.getMessage()));
      return;
    }

    // if only one of minversions or maxversions is entered on the CLI, retrieve the existing value for the other
    if ((minVersions != null && maxVersions == null) ||
        (minVersions == null && maxVersions != null)) {
      minVersions = minVersions != null ? minVersions : currentCf.getMinVersions();
      maxVersions = maxVersions != null ? maxVersions : currentCf.getMaxVersions();
    }

    //colPerm = cfa.getColumnAttrList();

    Long ttl = isParamPresent(TTL_PARAM_NAME) ? getParamLongValue(TTL_PARAM_NAME, 0) : null;
    Boolean inMem = isParamPresent(IN_MEMORY_PARAM_NAME) ? getParamBooleanValue(IN_MEMORY_PARAM_NAME, 0) : null;
    FileCompressionType ctype = null;

    if (isParamPresent(COMPRESSION_PARAM_NAME)) {
      ctype = CompressionHelper.getCompressionType(getParamTextValue(COMPRESSION_PARAM_NAME, 0));
      if (ctype == null) {
        out.addError(new OutputError(Errno.EINVAL, "Invalid input values. " +
          "The entered value for compression is not a supported type."));
       return;
     }
   }

    if (minVersions != null && maxVersions != null && minVersions > maxVersions) {
      out.addError(new OutputError(Errno.EINVAL, "The value for min versions is greater than the value for " +
          "max versions. Min versions: " + minVersions + ", Max versions: " + maxVersions + "."));
      return;
    }

    final AtomicBoolean isJsonTable = new AtomicBoolean(false);
    final AtomicBoolean isStream = new AtomicBoolean(false);
    try {
      new FileclientRun(getUserLoginId()) {
        @Override
        public void runAsProxyUser() throws IOException, CLIProcessingException{
          final MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
          // Get tablet data
          TableProperties tableProp = mfs.getTableProperties(new Path(tablePath));
          isJsonTable.set(tableProp.getAttr().getJson());
          isStream.set(tableProp.getAttr().getIsMarlinTable());
        }
      };
    } catch (IOException e) {
      throw new CLIProcessingException(e.getMessage());
    }

    SchemaFamily.Builder newCfBuilder = SchemaFamily.newBuilder();
    if (newCfName != null && !newCfName.equals(currentCfName)) {
      boolean isDefaultCf =
          currentCfName.equals(DBInternalDefaults.getDefaultInstance().getJsonDefaultCFName());

      // Do not allow rename of default CF for json tables
      if (isDefaultCf) {
        if (isJsonTable.get() == true) {
          out.addError(new OutputError(Errno.EINVAL, "'default' column family "+
                                   "of a JSON type table cannot be renamed."));
          return;
        }
      }

      newCfBuilder.setName(newCfName);
    }
    if (minVersions != null) {
      newCfBuilder.setMinVersions(minVersions);
    }
    if (maxVersions != null) {
      newCfBuilder.setMaxVersions(maxVersions);
    }
    if (ttl != null) {
      if (isJsonTable.get() && !isStream.get()) {
        final AtomicInteger numCF = new AtomicInteger(0);
        // We don't have a mechanism to allow TTL for the full document when its
        // split into multiple CF. If multiple CF is there for document then we
        // don't support setting TTL on the default CF.
        try {
          new FileclientRun(getUserLoginId()) {
            @Override
            public void runAsProxyUser() throws IOException, CLIProcessingException {
              final MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
              // Get tablet data
              List<ColumnFamilyAttr> cfa = mfs.listColumnFamily(new Path(tablePath), false /*wantACES*/);
              numCF.set(cfa.size());
            }
          };
        } catch (IOException e) {
          throw new CLIProcessingException(e.getMessage());
        }

        if (numCF.get() > 1) {
          out.addError(new OutputError(Errno.EINVAL, "TTL cannot be edited when " +
                                       "multiple column families present in " +
                                       "json type of table."));
          return;
        }
      }

      newCfBuilder.setTtl(ttl * 1000); // store the ttl value in ms
    }
    if (inMem != null) {
      newCfBuilder.setInMemory(inMem);
    }
    if (ctype != null) {
      newCfBuilder.setCompression(ctype);
    }



    try {
      if (isParamPresent(COLUMN_PERMISSIONS_PARAM_NAME)) {
        colPerm = getColumnList(toJSON(getParamTextValue(COLUMN_PERMISSIONS_PARAM_NAME, 0)), out);
        if (colPerm.isEmpty()) {
          // Delete all columns
          for (int i = 0; i < cfa.getColumnAttrCount(); ++i) {
            ColumnAttr.Builder colAttrBuilder = cfa.getColumnAttr(i).toBuilder();
            colAttrBuilder.clearAces();
            colPerm.add(colAttrBuilder.build());
          }
        }
      }


    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
      return;
    } catch (JSONException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
      return;
    }
    builder = ColumnFamilyAttr.newBuilder()
        .setSchFamily(newCfBuilder.build());

    if (colPerm != null)
      builder.addAllColumnAttr(colPerm);

    // Set as final to be called in impersination
    final RecentTablesListManager manager = RecentTablesListManagers.getRecentTablesListManagerForUser(getUserLoginId());
    final ColumnFamilyAttr cfattr = builder.build();
    final AceHelper.DBPermission self = this;
    try {
      if (LOG.isDebugEnabled()) {
        LOG.debug("Attempting to modify column family -> Path: " + tablePath + ", Name: " + currentCfName);
      }
      // Call impersination
      new FileclientRun(getUserLoginId()) {
        @Override
        public void runAsProxyUser() throws CLIProcessingException, IOException {
          MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
          try {
            mfs.modifyColumnFamily(new Path(tablePath), currentCfName, cfattr, self);
            manager.moveToTop(tablePath);
          } catch (IOException e) {
            manager.deleteIfNotExist(tablePath, mfs);
            throw new CLIProcessingException(e.getMessage());
          }

        }
      };
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    } catch (IOException e) {
     out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }

  }

  private void deleteCf(OutputHierarchy out) throws CLIProcessingException {
    final String tablePath = DbCommands.getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId());
    final String cfName = getParamTextValue(CFNAME_PARAM_NAME, 0);

    boolean isDefaultCf =
        cfName.equals(DBInternalDefaults.getDefaultInstance().getJsonDefaultCFName());
    final AtomicBoolean isJsonTable = new AtomicBoolean(false);
    try {
      new FileclientRun(getUserLoginId()) {
        @Override
        public void runAsProxyUser() throws IOException, CLIProcessingException {
          final MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
          // Get tablet data // information
          TableProperties tableProp = mfs.getTableProperties(new Path(tablePath));
          isJsonTable.set(tableProp.getAttr().getJson());
        }
      };
    } catch (IOException e) {
      throw new CLIProcessingException(e.getMessage());
    }

    // Cannot delete default CF in json table
    if (isDefaultCf && (isJsonTable.get() == true)) {
      out.addError(new OutputError(Errno.EINVAL, "'default' column family " +
                                   "of a JSON type table cannot be deleted."));
      return;
    }

    final RecentTablesListManager manager = RecentTablesListManagers.getRecentTablesListManagerForUser(getUserLoginId());
    try {
      if (LOG.isDebugEnabled()) {
        LOG.debug("Attempting to delete column family -> Path: " + tablePath + ", Name: " + cfName);
      }

      // Call impersination
      new FileclientRun(getUserLoginId()) {
        @Override
        public void runAsProxyUser() throws CLIProcessingException, IOException {
          MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
          try {
          mfs.deleteColumnFamily(new Path(tablePath), cfName);
          manager.moveToTop(tablePath);
          } catch (IOException e) {
            manager.deleteIfNotExist(tablePath, mfs);
            throw new CLIProcessingException(e.getMessage());
          }
        }
      };
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }
  }

  private void listCf(OutputHierarchy out) throws CLIProcessingException {
    String tablePath = DbCommands.getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId());
    MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
    RecentTablesListManager manager = RecentTablesListManagers.getRecentTablesListManagerForUser(getUserLoginId());
    try {

      if (isParamPresent(CFNAME_PARAM_NAME)) { // if a specific cfname is specified, just output the info for that specific cf.
        String cfName = getParamTextValue(CFNAME_PARAM_NAME, 0);
        ColumnFamilyAttr cfa = getColumnFamily(tablePath, cfName);
        if (cfa == null) {
          out.addError(new OutputError(Errno.ENOTEXIST, "Column family '" +
                                       cfName + "' is not defined for table '" +
                                       tablePath + "'."));
          return;
        }
        SchemaFamily cf = cfa.getSchFamily();
        if (cf != null) {
          OutputNode cfNode = new OutputNode();
          cfNode.addChild(new OutputNode(getOutputFieldName("cfname"), cf.getName()));
          cfNode.addChild(new OutputNode(getOutputFieldName("maxversions"), cf.getMaxVersions()));
          cfNode.addChild(new OutputNode(getOutputFieldName("minversions"), cf.getMinVersions()));
          cfNode.addChild(new OutputNode(getOutputFieldName("ttl"), getTtl(cf)));
          cfNode.addChild(new OutputNode(getOutputFieldName("inmemory"), cf.getInMemory()));
          cfNode.addChild(new OutputNode(getOutputFieldName("compression"), getCompressionName(cf.getCompression())));
          if (cfa.hasJsonFamilyPath()) {
            cfNode.addChild(new OutputNode(getOutputFieldName("jsonfamilypath"),
                                           cfa.getJsonFamilyPath()));
          }

          for (AccessControlExpression ace : cfa.getAcesList()) {
            cfNode.addChild(new OutputNode(
                getOutputFieldName(AceHelper.cfPermissionMap.get(ace.getAccessType())),
                AceHelper.toInfix(ace.getBooleanExpression().toStringUtf8())));
          }

          if (getParamBooleanValue(SHOW_COLUMNS_NAME, 0)) {

              for(int i = 0; i < cfa.getColumnAttrCount(); i++) {
                  ColumnAttr col = cfa.getColumnAttr(i);
                  // If colName is null, get every record. If set, then get only the record that is queried for
                  OutputNode colNode = new OutputNode();
                  colNode.addChild(new OutputNode(getOutputFieldName("name"),
                              AceHelper.toInfix(col.getQualifier().toStringUtf8())));

                  for (AccessControlExpression ace : col.getAcesList()) {
                      colNode.addChild(new OutputNode(
                                  getOutputFieldName(AceHelper.colPermissionMap.get(ace.getAccessType())),
                                  AceHelper.toInfix(ace.getBooleanExpression().toStringUtf8())));
                  }

                  cfNode.addChild(new OutputNode(getOutputFieldName("columnperm"), colNode));
              }
          }

          out.addNode(cfNode);
        } else {
          out.addError(new OutputError(Errno.ENOTEXIST, "Column family '" + cfName +
              "' is not defined for table '" + tablePath + "'."));
          return;
        }
      } else {
        for (ColumnFamilyAttr cfa : mfs.listColumnFamily(new Path(tablePath),
                                                         true /*ace*/)) {
          SchemaFamily cf = cfa.getSchFamily();
          OutputNode cfNode = new OutputNode();
          cfNode.addChild(new OutputNode(getOutputFieldName("cfname"), cf.getName()));
          cfNode.addChild(new OutputNode(getOutputFieldName("maxversions"), cf.getMaxVersions()));
          cfNode.addChild(new OutputNode(getOutputFieldName("minversions"), cf.getMinVersions()));
          cfNode.addChild(new OutputNode(getOutputFieldName("ttl"), getTtl(cf)));
          cfNode.addChild(new OutputNode(getOutputFieldName("inmemory"), cf.getInMemory()));
          cfNode.addChild(new OutputNode(getOutputFieldName("compression"), getCompressionName(cf.getCompression())));
          if (cfa.hasJsonFamilyPath()) {
            cfNode.addChild(new OutputNode(getOutputFieldName("jsonfamilypath"),
                                           cfa.getJsonFamilyPath()));
          }

          for (AccessControlExpression ace : cfa.getAcesList()) {
            cfNode.addChild(new OutputNode(
                getOutputFieldName(AceHelper.cfPermissionMap.get(ace.getAccessType())),
                AceHelper.toInfix(ace.getBooleanExpression().toStringUtf8())));
          }

          out.addNode(cfNode);
        }
      }
      manager.moveToTop(tablePath);
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
      manager.deleteIfNotExist(tablePath, mfs);
    }
  }

  private long getTtl(SchemaFamily cf) {
    long ttl = DEFAULT_TTL;
    if (cf.hasTtl()) {
      ttl = cf.getTtl() / 1000; // convert ms to seconds
    }
    return ttl;
  }

  private ColumnFamilyAttr getColumnFamily(String tablePath, String cfName) throws IOException, CLIProcessingException {

    final ColumnFamilyAttr.Builder ret = ColumnFamilyAttr.newBuilder();
    final String cfNameFinal = cfName;
    final String tablePathFinal = tablePath;


    if (LOG.isDebugEnabled()) {
      LOG.debug("Searching for column family " + cfName);
    }

    new FileclientRun(getUserLoginId()) {
      @Override
      public void runAsProxyUser() throws CLIProcessingException, IOException {
        MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
        for (ColumnFamilyAttr cf : mfs.listColumnFamily(new Path(tablePathFinal), true /*ace*/)) {
          if (cf.getSchFamily().getName().equals(cfNameFinal)) {
            ret.setSchFamily(cf.getSchFamily());
            ret.addAllAces(cf.getAcesList());
            ret.addAllColumnAttr(cf.getColumnAttrList());
            if (cf.hasJsonFamilyPath()) {
              ret.setJsonFamilyPath(cf.getJsonFamilyPath());
            }
          }
        }
      }
    };

    if (ret.getSchFamily().getName().isEmpty()) {
        // Name is empty, invalid CF
        return null;
    } else {
        return ret.build();
    }
  }

  private String getOutputFieldName(String verboseName) throws CLIProcessingException {
    return "terse".equals(getParamTextValue(OUTPUT_PARAM_NAME, 0)) ? verboseToTerseMap.get(verboseName) : verboseName;
  }

  /**
   * Converts string into JSON array
   * @param json The string to parse
   * @return JSON Array object
   * @throws JSONException
   */
  private JSONArray toJSON(String json) throws JSONException {
    JSONTokener token = new JSONTokener(json);
    return new JSONArray(token);
  }

  /**
   * Method that saves columns in JSON format, based on column family
   * @param columns JSON Array object of all the columns and permissions to be added
   * @param out Output method, to output errors
   */
  private ArrayList<ColumnAttr> getColumnList(JSONArray columns, OutputHierarchy out) {
    JSONObject column; // Object taken from Array
    String name;

    ColumnAttr.Builder col;

    ArrayList<AccessControlExpression> ace;

    ArrayList<ColumnAttr> columnPermissions = new ArrayList<ColumnAttr>();

    // Get column list for given tablename and column name
    try {
      for (int i = 0; i < columns.length(); i++) {
        column = columns.getJSONObject(i); // Get next JSON object

        col = ColumnAttr.newBuilder();

        col.setQualifier(ByteString.copyFromUtf8(column.getString("name")));

        ace = AceHelper.getColumnPermission(column);

        if (ace.size() > 0) {
          col.addAllAces(ace);
         }

        columnPermissions.add(col.build());
       }

    } catch (IOException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (JSONException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    }

    return columnPermissions;
  }

  @Override
  public String getCliParam(String key) throws IOException {
    String ret = null;
    // If permissions map val is a passed as parameter, then add to ace list
    try {
      if (isParamPresent(key)) {
        ret = getParamTextValue(key, 0);
      }
    } catch (CLIProcessingException e) {
      throw new IOException(e);
    }
    return ret;
  }

}
