/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.ByteString;
import com.mapr.baseutils.Errno;
import com.mapr.cli.table.RecentTablesListManager;
import com.mapr.cli.table.RecentTablesListManagers;
import com.mapr.cliframework.base.*;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.fs.AceHelper;
import com.mapr.fs.proto.Dbserver.DBAccessType;
import com.mapr.cliframework.base.inputparams.*;
import com.mapr.fs.MapRFileSystem;
import com.mapr.fs.MapRFsUtil;
import com.mapr.fs.proto.Dbserver.SchemaFamily;
import com.mapr.fs.proto.Dbserver.ColumnFamilyAttr;
import com.mapr.fs.proto.Dbserver.ColumnAttr;
import com.mapr.fs.proto.Dbserver.AccessControlExpression;
import com.mapr.fs.tables.TableProperties;
import org.apache.hadoop.fs.Path;

import org.apache.log4j.Logger;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class DbCfColCommands extends CLIBaseClass implements CLIInterface,AceHelper.DBPermission {

  private static final Logger LOG = Logger.getLogger(DbCfColCommands.class);
  private static final String PATH_PARAM_NAME = "path";
  private static final String CFNAME_PARAM_NAME = "cfname";
  private static final String COLNAME_PARAM_NAME = "name";
  private static final String READ_PARAM_NAME = "readperm";
  private static final String WRITE_PARAM_NAME = "writeperm";
  private static final String APPEND_PARAM_NAME = "appendperm";
  private static final String ENCRYPT_PARAM_NAME = "encryptperm";
  private static final String TRAVERSE_PARAM_NAME = "traverseperm";


  public static final String COLUMNS_PARAM_NAME = "columns";
  public static final String OUTPUT_PARAM_NAME = "output";
  public static final String START_PARAM_NAME = "start";
  public static final String LIMIT_PARAM_NAME = "limit";

  public static final int DEFAULT_TTL = Integer.MAX_VALUE;

  private static final CLICommand getCommand =
      new CLICommand(
          "get",
          "usage: table cf colperm get -path <tablepath>",
          DbCfColCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "Table path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(CFNAME_PARAM_NAME,
                  new TextInputParameter(CFNAME_PARAM_NAME,
                      "Column family name",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(COLNAME_PARAM_NAME,
                  new TextInputParameter(COLNAME_PARAM_NAME,
                      "Column name",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(COLUMNS_PARAM_NAME,
                  new TextInputParameter(COLUMNS_PARAM_NAME,
                      "columns",
                      CLIBaseClass.NOT_REQUIRED,
                      "all").setInvisible(true))
              .put(OUTPUT_PARAM_NAME,
                  new TextInputParameter(OUTPUT_PARAM_NAME,
                      "verbose|terse",
                      CLIBaseClass.NOT_REQUIRED,
                      "verbose").setInvisible(true))
              .put(START_PARAM_NAME,
                  new IntegerInputParameter(START_PARAM_NAME,
                      "start",
                      CLIBaseClass.NOT_REQUIRED,
                      0).setInvisible(true))
              .put(LIMIT_PARAM_NAME,
                  new IntegerInputParameter(LIMIT_PARAM_NAME,
                      "limit",
                      CLIBaseClass.NOT_REQUIRED,
                      Integer.MAX_VALUE).setInvisible(true))
              .build(), null)
          .setShortUsage("table cf colperm list -path <tablepath> -cfname <cf name>");

  private static final CLICommand setCommand =
      new CLICommand(
          "set",
          "usage: table cf colperm set -path <tablepath> -cfname <cfname> -colname <column name> -traverse <traverse permission> -read <read permission> -write <write permission> -append <append permission> -encrypt <encrypt permission> " +
          "(-traverse is supported for tables with tabletype=json only.)",
          DbCfColCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "Table path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(CFNAME_PARAM_NAME,
                  new TextInputParameter(CFNAME_PARAM_NAME,
                      "Column family name",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(COLNAME_PARAM_NAME,
                  new TextInputParameter(COLNAME_PARAM_NAME,
                      "Column name",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(READ_PARAM_NAME,
                  new TextInputParameter(READ_PARAM_NAME,
                      "Read column permission settings",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(WRITE_PARAM_NAME,
                  new TextInputParameter(WRITE_PARAM_NAME,
                      "Write column permission settings",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
               .put(APPEND_PARAM_NAME,
                  new TextInputParameter(APPEND_PARAM_NAME,
                      "Append column permission settings",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
                .put(ENCRYPT_PARAM_NAME,
                  new TextInputParameter(ENCRYPT_PARAM_NAME,
                      "Encrypt column permission settings",
                      CLIBaseClass.NOT_REQUIRED,
                      null).setInvisible(true))
                .put(TRAVERSE_PARAM_NAME,
                  new TextInputParameter(TRAVERSE_PARAM_NAME,
                      "Traverse column permission settings",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .build(), null)
          .setShortUsage("table cf colperm create -path <tablepath> -cfname <cfname> -colname <column name> -traverse <traverse permission> -read <read permission> -write <write permission> -append <append permission> -encrypt <encrypt permission>");


  private static final CLICommand deleteCommand =
      new CLICommand(
          "delete",
          "usage: table cf colperm delete -path <tablepath> -cfname <cfname> -colname <colname>",
          DbCfColCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "Table path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(CFNAME_PARAM_NAME,
                  new TextInputParameter(CFNAME_PARAM_NAME,
                      "Column family name",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(COLNAME_PARAM_NAME,
                  new TextInputParameter(COLNAME_PARAM_NAME,
                      "Column name",
                      CLIBaseClass.REQUIRED,
                      null))
              .build(), null)
          .setShortUsage("table cf col delete -path <tablepath> -cfname <cfname> -colname <colname>");

  // main command
  public static final CLICommand cfColCommands =
      new CLICommand(
          "colperm", "colperm [get|set|delete]",
          CLIUsageOnlyCommand.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new CLICommand[]{
              getCommand,
              setCommand,
              deleteCommand
          }
      ).setShortUsage("table cf colperm [get|set|delete]");

  private static final Map<String, String> verboseToTerseMap = new ImmutableMap.Builder<String, String>()
      .put("path", "p")
      .put("cfname", "cfn")
      .put("name", "n")
      .put("read", "r")
      .put("write", "w")
      .put("append", "a")
      .put("encrypt", "e")
      .put("traverse", "t")
      .build();

  public DbCfColCommands(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }

  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    if (!super.validateInput()) {
      return output;
    }

    if (cliCommand.getCommandName().equalsIgnoreCase(setCommand.getCommandName())) {
      setCol(out);
    } else if (cliCommand.getCommandName().equalsIgnoreCase(deleteCommand.getCommandName())) {
      deleteCol(out);
    } else if (cliCommand.getCommandName().equalsIgnoreCase(getCommand.getCommandName())) {
      getCol(out);
    }

    return output;
  }

  /**
   * Create Column method, run on create command
   * @param out Output
   * @throws CLIProcessingException
   */
  private void setCol(OutputHierarchy out) throws CLIProcessingException {
    final MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
    String tablePath = DbCommands.getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId()); // Get table name
    RecentTablesListManager manager = RecentTablesListManagers.getRecentTablesListManagerForUser(getUserLoginId());
    ColumnFamilyAttr cfAttr = null;
    SchemaFamily schFamily = null;
    String cfName = getParamTextValue(CFNAME_PARAM_NAME, 0); // Get column family name
    String colName = null;
    TableProperties tableProp = null;
    boolean isJson = false;

    try {
      tableProp = mfs.getTableProperties(new Path(tablePath));
      isJson = tableProp.getAttr().getJson();
      colName =  (isJson && !cfName.equals("default")) ?
          MapRFsUtil.getPrefixedColName(getParamTextValue(COLNAME_PARAM_NAME, 0)) : getParamTextValue(COLNAME_PARAM_NAME, 0);

      ByteString colByteName = ByteString.copyFromUtf8(colName);
      cfAttr = getColumnFamily(tablePath, cfName);
      if (cfAttr == null) {
        out.addError(new OutputError(Errno.ENOTEXIST, "Column family '" + cfName
                                     + "' is not defined for table '" + 
                                     tablePath + "'."));
        return;
      }
      schFamily = cfAttr.getSchFamily();

      if (schFamily == null) {
        out.addError(new OutputError(Errno.EINVAL, "Column family " + cfName + " not found under Table " + tablePath));
        return;
      }

      // Find index of item
      int index = findColumnPos(colByteName, cfAttr.getColumnAttrList());
      ColumnAttr.Builder builder = ColumnAttr.newBuilder();

      ArrayList<AccessControlExpression> ace = AceHelper.getColumnPermission(this);

      ColumnFamilyAttr.Builder cfBuilder = cfAttr.toBuilder();

      if (index == -1) {
        // Item does not exist, add item to end of list
        builder.setQualifier(colByteName);

        if (ace.size() > 0) {
          builder.addAllAces(ace);
        }
        cfBuilder.addColumnAttr(builder.build());

      } else {
        // Item exists, set item

        HashMap<DBAccessType, AccessControlExpression> col = new HashMap<DBAccessType, AccessControlExpression>();

        ColumnAttr cAttr = cfBuilder.getColumnAttr(index);

        for (AccessControlExpression a : cAttr.getAcesList()) {
          col.put(a.getAccessType(), a);
        }
        for (AccessControlExpression a : ace) {
          col.put(a.getAccessType(), a);
        }

        builder = cAttr.toBuilder().clearAces().addAllAces(col.values());

        cfBuilder.setColumnAttr(index, builder.build());
      }
      cfAttr = cfBuilder
          .clearAces()
          .clearSchFamily()
          .clearJsonFamilyPath()
          .build();

    } catch (IOException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    }
    modifyColumnSettings(tablePath, cfName, cfAttr, out);

  }

  /**
   * Delete column method, run on delete
   * @param out Output
   * @throws CLIProcessingException
   */
  private void deleteCol(OutputHierarchy out) throws CLIProcessingException {
    final MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
    String tablePath = DbCommands.getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId()); // Get table name
    RecentTablesListManager manager = RecentTablesListManagers.getRecentTablesListManagerForUser(getUserLoginId());
    String cfName = getParamTextValue(CFNAME_PARAM_NAME, 0); // Get column family name
    TableProperties tableProp = null;
    boolean isJson = false;
    String colName = null;
    ByteString colByteName = null;
    ColumnAttr.Builder builder = null;

    try {
      tableProp = mfs.getTableProperties(new Path(tablePath));
      isJson = tableProp.getAttr().getJson();
      colName = (isJson && !cfName.equals("default")) ?
          MapRFsUtil.getPrefixedColName(getParamTextValue(COLNAME_PARAM_NAME, 0)) : getParamTextValue(COLNAME_PARAM_NAME, 0);

      colByteName = ByteString.copyFromUtf8(colName);
      builder = ColumnAttr.newBuilder();
      ColumnFamilyAttr cfAttr = getColumnFamily(tablePath, cfName);

      int pos = findColumnPos(colByteName, cfAttr.getColumnAttrList());

      if (pos == -1) {
        out.addError(new OutputError(Errno.EINVAL, "Column " + colName + " not found in " + cfName + " ColumnFamily under Table " + tablePath));
        return;
      }

      builder = cfAttr.getColumnAttr(pos).toBuilder();
      builder.clearAces();
      cfAttr = cfAttr
          .toBuilder()
          .setColumnAttr(pos, builder.build())
          .clearAces()
          .clearSchFamily()
          .clearJsonFamilyPath()
          .build();

      modifyColumnSettings(tablePath, cfName, cfAttr, out);
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    }
  }

  /**
   * List column method. Run on list
   * @param out Output
   * @throws CLIProcessingException
   */
  private void getCol(OutputHierarchy out) throws CLIProcessingException {
    String tablePath = DbCommands.getTransformedPath(getParamTextValue(PATH_PARAM_NAME, 0), getUserLoginId()); // Get table name
    MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
    RecentTablesListManager manager = RecentTablesListManagers.getRecentTablesListManagerForUser(getUserLoginId());
    String cfName = getParamTextValue(CFNAME_PARAM_NAME, 0);
    TableProperties tableProp = null;
    boolean isJson = false;

    ByteString colName = null;
    try {
        tableProp = mfs.getTableProperties(new Path(tablePath));
        isJson = tableProp.getAttr().getJson();

        ColumnFamilyAttr cfAttr = getColumnFamily(tablePath, cfName);
        SchemaFamily cf = cfAttr.getSchFamily();

        if (cf != null) {
          ColumnAttr col;
          boolean cfPrefix = isJson && !cfName.equals("default");
          if (isParamPresent(COLNAME_PARAM_NAME)) { // if a specific cfname is specified, just output the info for that specific cf.
            String colNameStr = getParamTextValue(COLNAME_PARAM_NAME, 0);
            colName = cfPrefix ?
                ByteString.copyFromUtf8(MapRFsUtil.getPrefixedColName(colNameStr)) : ByteString.copyFromUtf8(colNameStr);
          }
          for(int i = 0; i < cfAttr.getColumnAttrCount(); i++) {
            col = cfAttr.getColumnAttr(i);
            // If colName is null, get every record. If set, then get only the record that is queried for
            if (colName == null || colName.equals(col.getQualifier())) {
              OutputNode colNode = new OutputNode();
              String postFixCF = cfPrefix ? MapRFsUtil.getUnPrefixedColName(col.getQualifier().toStringUtf8()) : col.getQualifier().toStringUtf8();
              colNode.addChild(new OutputNode(getOutputFieldName("name"), postFixCF));

              for (AccessControlExpression ace : col.getAcesList()) {
                colNode.addChild(new OutputNode(
                    getOutputFieldName(AceHelper.colPermissionMap.get(ace.getAccessType())),
                    AceHelper.toInfix(ace.getBooleanExpression().toStringUtf8())));
              }

              out.addNode(colNode);
            }
          }
        } else {
          out.addError(new OutputError(Errno.ENOTEXIST, "Column family '" + cfName +
              "' is not defined for table '" + tablePath + "'."));
          return;
        }
      manager.moveToTop(tablePath);
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
      manager.deleteIfNotExist(tablePath, mfs);
    }

  }


  /**
   * Private method that calls column family edit to modify column settings
   * @param tablePath The table's path
   * @param cfName The column family's name
   * @param cfAttr The Column Family attribute to modify, should only contain changes to Column settings
   * @param out Out, for displaying errors
   * @throws CLIProcessingException
   */
  private void modifyColumnSettings(String tablePath, String cfName, ColumnFamilyAttr cfAttr, OutputHierarchy out) throws CLIProcessingException {
    if (cfAttr != null) {
      MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
      RecentTablesListManager manager = RecentTablesListManagers.getRecentTablesListManagerForUser(getUserLoginId());
      try {
          if (LOG.isDebugEnabled()) {
            LOG.debug("Attempting to modify column settings -> Path: " + tablePath + ", CfName: " + cfName);
          }
          mfs.modifyColumnFamily(new Path(tablePath), cfName, cfAttr);
          manager.moveToTop(tablePath);
      } catch (IOException e) {
        out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
        manager.deleteIfNotExist(tablePath, mfs);
      }

    }
  }

  /**
   * Finds the position of the column in the given list
   * @param colName The name to search for (as key)
   * @param colPerm The list to search
   * @return
   */
  private int findColumnPos(ByteString colName, List<ColumnAttr> colPerm) {
    int i = 0;
    if (colPerm != null) {
      for (ColumnAttr cp : colPerm) {
        if (cp.getQualifier().equals(colName)) {
          return i;
        }
        i++;
      }
    }
    return -1;
  }

  private static ColumnFamilyAttr getColumnFamily(String tablePath, String cfName) throws IOException, CLIProcessingException {

    if (LOG.isDebugEnabled()) {
      LOG.debug("Searching for column family " + cfName);
    }

    MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
    for (ColumnFamilyAttr cf : mfs.listColumnFamily(new Path(tablePath), true /*ace*/)) {
      if (cf.getSchFamily().getName().equals(cfName)) {
        return cf;
      }
    }
    return null;
  }

  private String getOutputFieldName(String verboseName) throws CLIProcessingException {
    return "terse".equals(getParamTextValue(OUTPUT_PARAM_NAME, 0)) ? verboseToTerseMap.get(verboseName) : verboseName;
  }

  @Override
  public String getCliParam(String key) throws IOException {
    String ret = null;
    // If permissions map val is a passed as parameter, then add to ace list
    try {
      if (isParamPresent(key)) {
        ret = getParamTextValue(key, 0);
      }
    } catch (CLIProcessingException e) {
      throw new IOException(e);
    }
    return ret;
  }


}
