/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.concurrent.Callable;

import com.mapr.security.JNISecurity;
import org.apache.hadoop.mapred.ClusterStatus;
import org.apache.hadoop.mapred.JobClient;
import org.apache.hadoop.mapred.JobStatus;
import org.apache.hadoop.yarn.api.records.ApplicationReport;
import org.apache.hadoop.yarn.api.records.NodeReport;
import org.apache.hadoop.yarn.api.records.Resource;
import org.apache.hadoop.yarn.api.records.YarnApplicationState;
import org.apache.hadoop.yarn.api.records.YarnClusterMetrics;
import org.apache.hadoop.yarn.client.api.YarnClient;
import org.apache.log4j.Logger;

import com.mapr.baseutils.Errno;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils.IpPort;
import com.mapr.cli.common.NodesCommonUtils;
import com.mapr.cli.common.ServicesEnum;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.AlarmLookupRequest;
import com.mapr.fs.cldb.proto.CLDBProto.AlarmLookupResponse;
import com.mapr.fs.cldb.proto.CLDBProto.ClusterInfoRequest;
import com.mapr.fs.cldb.proto.CLDBProto.ClusterInfoResponse;
import com.mapr.fs.cldb.proto.CLDBProto.VolumeInfoSummary;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Common.AlarmId;
import com.mapr.fs.proto.Common.AlarmMsg;
import com.mapr.fs.proto.Common.AlarmType;
import com.mapr.fs.proto.License.LicenseIdResponse;
import com.mapr.security.MaprSecurityException;

public class Dashboard extends CLIBaseClass implements CLIInterface {
  
  private String zkConnectString = null;
  private static final Logger LOG = Logger.getLogger(Dashboard.class);
  
  private static final String VERSION_PARAM = "version";
  private static final String MULTICLUSTER_PARAM = "multi_cluster_info";

  public Dashboard(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }

  static final CLICommand infoCmd = new CLICommand(
      "info",
      "get node info for dashboard",
      Dashboard.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
          .put(MapRCliUtil.CLUSTER_NAME_PARAM,
              new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
                  "cluster name", 
                  CLIBaseClass.NOT_REQUIRED, 
                  null))
          .put(NodesCommonUtils.ZK_CONNECTSTRING,
              new TextInputParameter(
                  NodesCommonUtils.ZK_CONNECTSTRING,
                  "ZooKeeper Connect String: 'host:port,host:port,host:port,...'",
                  CLIBaseClass.NOT_REQUIRED, null))
          .put(VERSION_PARAM,
              new BooleanInputParameter(VERSION_PARAM,
                  "true|false", 
                  CLIBaseClass.NOT_REQUIRED, 
                  false))
          .put(MULTICLUSTER_PARAM,
              new BooleanInputParameter(MULTICLUSTER_PARAM,
                  "true|false", 
                  CLIBaseClass.NOT_REQUIRED, 
                  false)) 
          .build(), 
      null);

  /* Define sub command */
  public static final CLICommand dashboardCmds = new CLICommand(
      "dashboard",
      "", 
      CLIUsageOnlyCommand.class, 
      ExecutionTypeEnum.NATIVE,
        new CLICommand[] { /* array of subcommands */
          infoCmd 
        }
      ).setShortUsage("dashboard [info] -[version|multi_cluster_info]");
  
  void addVolumeSummary(ClusterInfoResponse resp, OutputNode finalNode) {

    if (!resp.hasVolSummary()) {
      return;
    }

    VolumeInfoSummary vSum = resp.getVolSummary();

    OutputNode mounted = new OutputNode("mounted");
    mounted.addChild(new OutputNode("total", vSum.getMountedCount()));
    mounted.addChild(new OutputNode("size", vSum.getMountedSizeMB()));

    OutputNode unmounted = new OutputNode("unmounted");
    unmounted.addChild(new OutputNode("total", vSum.getUnmountedCount()));
    unmounted.addChild(new OutputNode("size", vSum.getUnmountedSizeMB()));
    
    OutputNode vNode = new OutputNode("volumes");
    vNode.addChild(mounted);
    vNode.addChild(unmounted);

    finalNode.addChild(vNode);
  }

  void addUtilization(ClusterInfoResponse resp, OutputNode finalNode) {
    OutputNode cpuNode = new OutputNode("cpu");
    int numServers = resp.getClusterNumLiveFileServers();
    cpuNode.addChild(new OutputNode("util",
        ((numServers != 0) ? resp.getClusterCpuUtilization()/numServers : 0)));
    cpuNode.addChild(new OutputNode("total", resp.getClusterCpuCount()));
    cpuNode.addChild(new OutputNode("active", new Double(resp.getClusterCpuUsed()).longValue()));

    OutputNode memNode = new OutputNode("memory");
    memNode.addChild(new OutputNode("total",
        resp.getClusterMemCapacitySizeMB()));
    memNode.addChild(new OutputNode("active", 
        resp.getClusterMemOccupiedSizeMB()));

    OutputNode diskNode = new OutputNode("disk_space");
    diskNode.addChild(new OutputNode("total",
        resp.getClusterCapacitySizeMB()/1024));
    diskNode.addChild(new OutputNode("active",
        resp.getClusterOccupiedSizeMB()/1024));
    
    OutputNode compressNode = new OutputNode("compression");
    compressNode.addChild(new OutputNode("compressed",
        resp.getVolSummary().getTotalSizeMB()/1024));
    compressNode.addChild(new OutputNode("uncompressed",
        resp.getVolSummary().getTotalLogicalSizeMB()/1024));
    

    OutputNode uNode = new OutputNode("utilization");
    uNode.addChild(new OutputNode("cpu", cpuNode));
    uNode.addChild(new OutputNode("memory", memNode));
    uNode.addChild(new OutputNode("disk_space", diskNode));
    uNode.addChild(new OutputNode("compression", compressNode));

    finalNode.addChild(uNode);
  }


  void addReplication(ClusterInfoResponse resp, OutputNode finalNode) {

    OutputNode uNode = new OutputNode("clusterReplication");
    uNode.addChild(new OutputNode("bytesReceived", resp.getClusterReplicationBytesReceived()));
    uNode.addChild(new OutputNode("bytesSend", resp.getClusterReplicationBytesSent()));
    finalNode.addChild(uNode);
  }

  void addStream(ClusterInfoResponse resp, OutputNode finalNode) {

    OutputNode uNode = new OutputNode("streamThroughput");
    uNode.addChild(new OutputNode("bytesProduced",
                     resp.getStreamBytesProduced()));
    uNode.addChild(new OutputNode("bytesConsumed",
                     resp.getStreamBytesConsumed()));
    finalNode.addChild(uNode);
  }

  void addVersion(ClusterInfoResponse resp, OutputNode finalNode) {
    OutputNode vNode = new OutputNode("version", resp.getBuildVersion());
    finalNode.addChild(vNode);
  }

  void addClusterInfo(ClusterInfoResponse resp, OutputNode finalNode, String clusterName) throws CLIProcessingException {
    OutputNode cNode = new OutputNode("cluster");
    String curCluster = CLDBRpcCommonUtils.getInstance().getCurrentClusterName();
    boolean flag = false;
    if (clusterName != null && !curCluster.equals(clusterName)) {
        flag = true;
        curCluster = clusterName;
    }
    cNode.addChild(new OutputNode("name", curCluster));
    
    boolean isSecureCluster = JNISecurity.IsSecurityEnabled(curCluster);
    cNode.addChild(new OutputNode("secure", isSecureCluster));

    Map<String, List<IpPort>> clustersMap = CLDBRpcCommonUtils.getInstance().getClusterMap();
    if (flag) {
      try {
          for (String cldbip : clustersMap.get(curCluster).get(0).getAddr()) {
              if (!cldbip.isEmpty()) {
                  cNode.addChild(new OutputNode("ip", cldbip));
                  break;
              }
          }
      } catch (Exception e) {
          // Do nothing
          LOG.error("Exception while getting CLDB ip for cluster: " +
                  curCluster + ", " + e.getMessage());
      }
    } else {
        for (String cldbip : CLDBRpcCommonUtils.getInstance().getCurrentValidIpPort().getAddr()) {
            if (!cldbip.isEmpty()) {
                cNode.addChild(new OutputNode("ip", cldbip));
                break;
            }
        }
    }

    LicenseIdResponse licId = LicenseCommands.fetchClusterID(curCluster,
                                                             getUserCredentials());
    int licenseNodesTotal = 0;
    int licenseNodesUsed = 0;
    String clusterId = null;
    if (licId != null) {
      if (licId.getStatus() == 0) {
        clusterId = licId.getClusterid();
        licenseNodesUsed = licId.getNodesUsed();
        licenseNodesTotal = licId.getNodesTotal();

      } else if (licId.hasMessage()) {
        clusterId = licId.getMessage();
      }
    }
    cNode.addChild(new OutputNode("id", clusterId));
    cNode.addChild(new OutputNode("nodesUsed", licenseNodesUsed));
    cNode.addChild(new OutputNode("totalNodesAllowed", licenseNodesTotal));

    finalNode.addChild(cNode);
        
    for (String cluster: clustersMap.keySet()) {
      // Ignore current cluster
      if (cluster.equalsIgnoreCase(curCluster))
        continue;
      OutputNode mcNode = new OutputNode();
      mcNode.addChild(new OutputNode("name", cluster));
      try {
        for (String cldbip : clustersMap.get(cluster).get(0).getAddr()) {
           if (!cldbip.isEmpty()) {
             mcNode.addChild(new OutputNode("ip", cldbip));
             break;
           }
        }
      } catch (Exception e) {
        // Do nothing
        LOG.error("Exception while getting CLDB ip for cluster: " + 
                  cluster + ", " + e.getMessage());
      }
      finalNode.addChild(new OutputNode("multi_cluster_info", mcNode));
    }
  }

  void addMapreduce(OutputNode finalNode) {
    if ( LOG.isDebugEnabled() ) {
      LOG.debug("addMapreduce start"); 
    }

    try {
      if (isParamPresent(NodesCommonUtils.ZK_CONNECTSTRING)) {
        zkConnectString = getParamTextValue(NodesCommonUtils.ZK_CONNECTSTRING, 0);
      } else {
        zkConnectString = CLDBRpcCommonUtils.getInstance().getZkConnect();
      }
    } catch (CLIProcessingException e) {
      LOG.error("Could not find Zookeeper Connect String", e);
      return;
    }
    
    final JobClient jc = MapRCliUtil.getJobClient(zkConnectString);
    if (jc == null)
      return;

    try {
      Callable<ClusterStatus> callableClusterStatus = new Callable<ClusterStatus>() {

        @Override
        public ClusterStatus call() throws Exception {
          return jc.getClusterStatus();
        }
      };

      ClusterStatus cs = MapRCliUtil.asyncInvoke(callableClusterStatus, "Getting cluster status from Jobtracker");

      if (cs == null)
        return;

      Callable<JobStatus[]> callableJobStatus = new Callable<JobStatus[]>() {

        @Override
        public JobStatus[] call() throws Exception {
          return jc.getAllJobs();
        }
      };

      JobStatus[] jsArray = MapRCliUtil.asyncInvoke(callableJobStatus, "Getting all jobs Status from Jobtracker");

      if (jsArray == null)
        return;


      if (LOG.isDebugEnabled()) {
        LOG.debug("addMapreduce jt returned clusterstatus");
      }

      long running = 0;
      long queued = 0;
      for (JobStatus js : jsArray) {
        switch (js.getRunState()) {
          case JobStatus.RUNNING:
            running++;
            break;

          case JobStatus.PREP:
            queued++;
            break;
        }
      }

      OutputNode mrNode = new OutputNode("mapreduce");
      //mrNode.addChild(new OutputNode("cumulative_cpu", 50)); /* TODO: Please provide valid cumu. CPU */
      //mrNode.addChild(new OutputNode("cumulative_mem", 2588189)); /* TODO: Please provide valid cumu. RAM */
      //mrNode.addChild(new OutputNode("queue_time", 0.0)); /* TODO: Please provide valid queue Time */
      mrNode.addChild(new OutputNode("running_jobs", running));
      mrNode.addChild(new OutputNode("queued_jobs", queued));
      mrNode.addChild(new OutputNode("running_tasks", cs.getMapTasks() + cs.getReduceTasks()));
      mrNode.addChild(new OutputNode("running_map_tasks", cs.getMapTasks()));
      mrNode.addChild(new OutputNode("running_reduce_tasks", cs.getReduceTasks()));
      mrNode.addChild(new OutputNode("map_task_capacity", cs.getMaxMapTasks() - cs.getMaxPrefetchMapTasks()));
      mrNode.addChild(new OutputNode("reduce_task_capacity", cs.getMaxReduceTasks()));
      mrNode.addChild(new OutputNode("map_task_prefetch_capacity", cs.getMaxPrefetchMapTasks()));
      mrNode.addChild(new OutputNode("blacklisted", cs.getBlacklistedTrackers()));
      finalNode.addChild(mrNode);

    } catch(Exception e) {
      LOG.error("Exception while fetching MapReduce Stats: ", e);
    }

    if ( LOG.isDebugEnabled() ) {
      LOG.debug("addMapreduce end");
    }
  }

  void addYarnStats(OutputNode finalNode) {
    if( LOG.isDebugEnabled() ) {
      LOG.debug("addYarnStats start");
    }

    try {
      if (isParamPresent(NodesCommonUtils.ZK_CONNECTSTRING)) {
        zkConnectString = getParamTextValue(NodesCommonUtils.ZK_CONNECTSTRING, 0);
      } else {
        zkConnectString = CLDBRpcCommonUtils.getInstance().getZkConnect();
      }
    } catch (CLIProcessingException e) {
      LOG.error("Could not find Zookeeper Connect String", e);
      return;
    }
    
    final YarnClient yc = MapRCliUtil.getYarnClient(zkConnectString);
    // This could be because either Resource Manager is not configured or is not running.
    // Actual error would have been logged from the method above.
    if (yc == null)
      return;    

    try {
        int countRunningApps = 0;
        int countQueuedApps = 0;
        int totalMemoryMB = 0;
        int totalVCores = 0;
        int usedMemoryMB = 0;
        int usedVCores = 0;
        int numNodeManagers = 0;
        double totalDisks = 0;
        double usedDisks = 0;

      Callable<YarnClusterMetrics> callableYarnClusterMetrics = new Callable<YarnClusterMetrics>() {
        @Override
        public YarnClusterMetrics call() throws Exception {
          return yc.getYarnClusterMetrics();
        }
      };

	    YarnClusterMetrics ycm = MapRCliUtil.asyncInvoke(callableYarnClusterMetrics, "Getting Yarn Cluster Metrics from ResourceManager");

      if(ycm == null)
        return;

      Callable<List<ApplicationReport>> callableAppReports = new Callable<List<ApplicationReport>>() {
        @Override
        public List<ApplicationReport> call() throws Exception {
          return yc.getApplications();
        }
      };

      List<ApplicationReport> allAppReports = MapRCliUtil.asyncInvoke(callableAppReports, "Getting Application Reports from ResourceManager");

      if(allAppReports == null)
        return;

      Callable<List<NodeReport>> callableNodeReports = new Callable<List<NodeReport>>() {
        @Override
        public List<NodeReport> call() throws Exception {
          return yc.getNodeReports();
        }
      };

	    List<NodeReport> allNodeReports = MapRCliUtil.asyncInvoke(callableNodeReports, "Getting Node Reports from ResourceManager");

      if(allNodeReports == null)
        return;

	    numNodeManagers = ycm.getNumNodeManagers();
	    
	    for(ApplicationReport ar : allAppReports) {
	    	YarnApplicationState appstate = ar.getYarnApplicationState();
	    	switch(appstate) {
	    		case RUNNING:
	    			countRunningApps++;
	    			break;
	    		case NEW:
	    		case NEW_SAVING:
	    		case SUBMITTED:
	    		case ACCEPTED:
	    			countQueuedApps++;
	    			break;
	    	}
	    }
	    
      LOG.debug("Number of Node reports received = " + allNodeReports.size());

	    for(NodeReport nr : allNodeReports) {
	      
	      LOG.debug(nr);
	      
	      // Skip if this node is not usable right now.
	      if(nr.getNodeState().isUnusable()) {
          	LOG.debug("Skipping Node: " + nr.getNodeId().getHost() + ":" + nr.getNodeId().getPort() + " as Node is in " + nr.getNodeState() + " State" );
          	continue;
              } else {
          	LOG.debug("Counting Node: " + nr.getNodeId().getHost() + ":" + nr.getNodeId().getPort() + " as Node is in " + nr.getNodeState() + " State" );
              }

	      Resource capability = nr.getCapability();
	    	
	    	totalMemoryMB += capability.getMemory();
	    	totalVCores += capability.getVirtualCores();
        totalDisks += capability.getDisks();

	    	Resource used = nr.getUsed();
	    	// If no job has ever run yet, its returning null for used Resources. Seems a bug in NM/RM.
	    	if(used != null) {
	    		usedMemoryMB += used.getMemory();
	    		usedVCores += used.getVirtualCores();
          usedDisks += used.getDisks();
	    	}
	    }
	    
	    OutputNode yarnNode = new OutputNode("yarn");
	    yarnNode.addChild(new OutputNode("running_applications", countRunningApps));
	    yarnNode.addChild(new OutputNode("queued_applications", countQueuedApps));
	    yarnNode.addChild(new OutputNode("num_node_managers", numNodeManagers));
	    yarnNode.addChild(new OutputNode("total_memory_mb", totalMemoryMB));
	    yarnNode.addChild(new OutputNode("total_vcores", totalVCores));
      yarnNode.addChild(new OutputNode("total_disks", totalDisks));
	    yarnNode.addChild(new OutputNode("used_memory_mb", usedMemoryMB));
	    yarnNode.addChild(new OutputNode("used_vcores", usedVCores));
      yarnNode.addChild(new OutputNode("used_disks", usedDisks));

	    finalNode.addChild(yarnNode);
	
	} catch (Exception e) {
		LOG.error("Exception while fetching Yarn Stats: ", e);
	}
    
    if( LOG.isDebugEnabled() ) {
      LOG.debug("addYarnStats end");
    }
  }

  void addServices(OutputNode finalNode) {

    Map<String, List<String>> runningServices = new HashMap<String, List<String>>();
    Map<String, List<String>> configuredServices = new HashMap<String, List<String>>();
    try {
      runningServices = NodesCommonUtils.findServicesRunningHierarchy(zkConnectString);
      configuredServices = NodesCommonUtils.findConfiguredServicesByServiceHierarchy(zkConnectString);
    } catch (Throwable t) {
  	/**
  	* <MAPR_ERROR>
  	* Message:Exception during services fetch operation
  	* Function:Dashboard.addMapreduce()
  	* Meaning:An error occurred.
  	* Resolution:Contact technical support.
  	* </MAPR_ERROR>
  	*/
    	LOG.error("Exception during services fetch operation", t);
     }

    // summarize stuff by service
    Map<String, Integer> runningServiceNodes = new HashMap<String, Integer>();

    for (Map.Entry<String, List<String>> nodeRunningServices : runningServices
        .entrySet()) {
      for (String service : nodeRunningServices.getValue()) {
        Integer serviceCounter = runningServiceNodes.get(service);
        if (serviceCounter == null) {
          runningServiceNodes.put(service, 0);
        }
        int counter = runningServiceNodes.get(service).intValue();
        runningServiceNodes.put(service, Integer.valueOf(++counter));
      }
    }

    OutputNode sNode = new OutputNode("services");

    for ( Map.Entry<String, List<String>> configuredServiceEntry : configuredServices.entrySet()) {
    	String serviceName = configuredServiceEntry.getKey();
    	OutputNode nodeG = new OutputNode();
    	Integer runningTotal = runningServiceNodes.get(serviceName);
    	Integer standby = null;
    	// I am just commenting it for now. Will remove later when proven that it is working
      // Integer failed = getNumNodesWithServiceFailedAlarm(serviceName);
    	String maxRunning = getMaxRunningInstancesMap(serviceName).get(serviceName);
     	Map<String, Integer> actionCountMap = getActionCountMap(serviceName, runningServices);
      Integer failed = actionCountMap.get("failed");
      Integer stopped = actionCountMap.get("stop");
      List<String> configuredNodesList = configuredServiceEntry.getValue();

      if ( failed == null ) {
        failed = 0;
      }
      if ( stopped == null ) {
        stopped = 0;
      }

    	if (maxRunning != null) {
        int configured = configuredNodesList.size();
        int running = runningTotal == null ? 0 : runningTotal;
        if (!maxRunning.equalsIgnoreCase("all")) {
          standby = configured - running - stopped - failed;
          if ( standby < 0 ) {
            LOG.error("Number of standby nodes is negative: " + standby 
                + ". configured - running - stopped - failed = " + configured + ", " 
                + running  + ", " + stopped  + ", " + failed);
            standby = 0;
          }
        } 
      }
    	
        if ( runningTotal != null ) {
      	  nodeG.addChild(new OutputNode("active", runningTotal));
        } else {
      	  nodeG.addChild(new OutputNode("active", 0));
        }
        if (standby != null) {
          nodeG.addChild(new OutputNode("standby", standby));
        }
        if (stopped != null) {
          nodeG.addChild(new OutputNode("stopped", stopped));
        }
        nodeG.addChild(new OutputNode("failed", failed));
        if ( configuredNodesList != null) {
      	  nodeG.addChild(new OutputNode("total", configuredNodesList.size()));
        } else {
      	  nodeG.addChild(new OutputNode("total", 0));
        }
        sNode.addChild(new OutputNode(serviceName, nodeG));
    }

    finalNode.addChild(sNode);
    return;
  }
  
  // services=webserver:all:cldb;jobtracker:1:cldb;tasktracker:all:jobtracker;nfs:all:cldb;kvstore:all;cldb:all:kvstore;hoststats:all:kvstore
  private Map<String, String> getMaxRunningInstancesMap(String service) {
    Map<String, String> maxInstances = Maps.newHashMap();
    Map<String, Properties> serviceNodesProperties = NodesCommonUtils.getServiceNodesProperties(zkConnectString, service);
    if (serviceNodesProperties != null && !serviceNodesProperties.isEmpty()) {
      Properties wardenProps = Lists.newArrayList(serviceNodesProperties.values()).get(0); // get props from 1st node on which service is running
      String servicesString = (String) wardenProps.get("services");
      if (servicesString != null && !servicesString.isEmpty()) {
        StringTokenizer tokenizer = new StringTokenizer(servicesString, ";");
        while (tokenizer.hasMoreTokens()) {
          String token = tokenizer.nextToken();
          String[] parts = token.split(":");
          maxInstances.put(parts[0], parts[1]);
        }
      }
    }
    return maxInstances;
  }
  
  private Map<String, Integer> getActionCountMap(String serviceName, Map<String, List<String>> runningServices) {
    Map<String, Integer> actionCountMap = new HashMap<String, Integer>();
    Map<String, Properties> serviceNodesProperties = NodesCommonUtils.getServiceNodesProperties(zkConnectString, serviceName);
    for (Map.Entry<String, Properties> nodeProps : serviceNodesProperties.entrySet() ) {
      String nodeName = nodeProps.getKey();
      Properties props = nodeProps.getValue();
      String actionPropValue = props.getProperty("last.action");
       if ( actionPropValue != null ) {
         if ( "failed".equalsIgnoreCase(actionPropValue) || "stop".equalsIgnoreCase(actionPropValue)) {
           List<String> services = runningServices.get(nodeName);
           if ( services == null || services.isEmpty() || !services.contains(serviceName) ) {
             Integer count = actionCountMap.get(actionPropValue);
             if ( count == null ) {
               actionCountMap.put(actionPropValue, 0);
             }
             actionCountMap.put(actionPropValue, actionCountMap.get(actionPropValue) + 1);
           } else {
             // it is not failed/stopped
           }
         }
      }
    }
    return actionCountMap;
  }
  
  
  private int getNumNodesWithServiceFailedAlarm(String service) throws CLIProcessingException {
    int numFailed = 0;
    AlarmLookupRequest.Builder reqBuilder = AlarmLookupRequest.newBuilder()
        .setCreds(getUserCredentials());
    AlarmMsg.Builder msgBuilder = AlarmMsg.newBuilder();
    msgBuilder.setAlarmType(AlarmType.NODE_ALARM);
    msgBuilder.setAlarmId(getAlarmIdForService(service));
    reqBuilder.addAlarms(msgBuilder);
    AlarmCommands ac = new AlarmCommands(this.getInput(), this.cliCommand);
      try {
        AlarmLookupResponse response = (AlarmLookupResponse) ac.sendRequest(reqBuilder.build());
        if ( response != null ) {
          numFailed = response.getAlarmsCount();
        }
      } catch (CLIProcessingException e) {
        LOG.error("Exception while fetching alarm information.", e);
      }
    
    return numFailed;
  }
  
  private AlarmId getAlarmIdForService(String serviceName) {
  	try {
	    ServicesEnum service = ServicesEnum.valueOf(serviceName);
	    switch(service) {
	      case cldb:
	        return AlarmId.NODE_ALARM_SERVICE_CLDB_DOWN;
	      case fileserver:
	        return AlarmId.NODE_ALARM_SERVICE_FILESERVER_DOWN;
	      case hoststats:
	        return AlarmId.NODE_ALARM_SERVICE_HOSTSTATS_DOWN;
	      case nfs:
	        return AlarmId.NODE_ALARM_SERVICE_NFS_DOWN;
	      case hbmaster:
	        return AlarmId.NODE_ALARM_SERVICE_HBMASTER_DOWN;
	      case hbregionserver:
	        return AlarmId.NODE_ALARM_SERVICE_HBREGION_DOWN;
	      case jobtracker:
	        return AlarmId.NODE_ALARM_SERVICE_JT_DOWN;
	      case tasktracker:
	        return AlarmId.NODE_ALARM_SERVICE_TT_DOWN;
	      case webserver:
	        return AlarmId.NODE_ALARM_SERVICE_WEBSERVER_DOWN;
        default:
          String alarmKey = "NODE_ALARM_SERVICE_" +
              serviceName.toUpperCase() +
              "_DOWN";
          return AlarmId.valueOf(alarmKey);
	    }
  	} catch (IllegalArgumentException e) {
  		LOG.error("Unknown service: " + serviceName);
  	}
    return null;
  }
  
  /* This is a function that does the real work. */
  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    OutputNode finalNode = new OutputNode();

    // Fetch some sections from CLDB
    try {
      byte[] replyData;

      ClusterInfoRequest req = ClusterInfoRequest.newBuilder()
          .setCreds(getUserCredentials()).setColumns(0xffffffff).build();
      String cluster = null;
      if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        cluster = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0);
        if (!CLDBRpcCommonUtils.getInstance().isValidClusterName(cluster)) {
          out.addError(new OutputError(Errno.EUCLUSTER, "Invalid cluster: " + cluster));
          return output;
        }
        replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
            cluster,
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg.ClusterInfoProc.getNumber(), req,
            ClusterInfoResponse.class);
      } else {
        replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg.ClusterInfoProc.getNumber(), req,
            ClusterInfoResponse.class);
      }

      if ( replyData == null ) {
    	  /**
    	  * <MAPR_ERROR>
    	  * Message:Could not get cluster info data from CLDB
    	  * Function:Dashboard.addMapreduce()
    	  * Meaning:An error occurred.
    	  * Resolution:Contact technical support.
    	  * </MAPR_ERROR>
    	  */
    	  LOG.error("Could not get cluster info data from CLDB");
        //out.addError(new OutputError(Errno.INTERROR, "Could not get cluster info data from CLDB"));
      } else {
	      ClusterInfoResponse resp = ClusterInfoResponse.parseFrom(replyData);
	      boolean printAll = true;
	      if (getParamBooleanValue(VERSION_PARAM, 0)) {
	      	addVersion(resp, finalNode);
	      	printAll = false;
	      }
	      
	      if (getParamBooleanValue(MULTICLUSTER_PARAM, 0)) {
	      	addClusterInfo(resp, finalNode, cluster);
	      	printAll = false;
	      }
	      
	      if (!printAll) {
	      	out.addNode(finalNode);
	      	return output;
	      }
	      
	      addVersion(resp, finalNode);
	      addClusterInfo(resp, finalNode, cluster);
	      addVolumeSummary(resp, finalNode);
	      addUtilization(resp, finalNode);
              addReplication(resp, finalNode);
              addStream(resp, finalNode);
      }
    } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      // log error
      /**
      * <MAPR_ERROR>
      * Message:Exception doing RPC to CLDB <error>
      * Function:Dashboard.addMapreduce()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      LOG.error("Exception doing RPC to CLDB " + e);
    }

    if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        out.addNode(finalNode);
        return output;
    }
      
    if (isParamPresent(NodesCommonUtils.ZK_CONNECTSTRING)) {
      zkConnectString = getParamTextValue(NodesCommonUtils.ZK_CONNECTSTRING, 0);
    } else {  
      zkConnectString = CLDBRpcCommonUtils.getInstance().getZkConnect();
    }
    if (zkConnectString != null && !zkConnectString.trim().isEmpty()) {
      // Fetch services info from warden
      addServices(finalNode);          
      // Fetch Mapreduce info from JT
      addMapreduce(finalNode);
      // Fetch Yarn specific stats from RM
      addYarnStats(finalNode);
    }      
   
    out.addNode(finalNode);
    return output;
  }
  
}
