/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.StringWriter;
import java.util.Calendar;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;
import java.util.TreeSet;

import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.log4j.Logger;

import com.google.common.collect.ImmutableMap;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.IntegerInputParameter;
import com.mapr.cliframework.base.inputparams.JSONObjectInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.CLDBConfigParams;
import com.mapr.fs.cldb.proto.CLDBProto.CLDBConfigResponse;
import com.mapr.fs.cldb.proto.CLDBProto.CLDBConfigParams.*;
import com.mapr.fs.cldb.proto.CLDBProto.CLDBConfigRequest;
import com.mapr.fs.proto.Common;
import com.mapr.util.MapRFSUtil;
import com.mapr.security.MaprSecurityException;

import javax.crypto.Cipher;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.DESKeySpec;
import org.apache.commons.codec.binary.Base64;
import org.json.JSONException;
import org.json.JSONObject;


public class ConfigCommands extends CLIBaseClass implements CLIInterface {

  public static final String dbUrlParamName = "jm_db.url";
  public static final String dbUserParamName = "jm_db.user";
  public static final String dbPasswdParamName = "jm_db.passwd";
  public static final String dbSchemaParamName = "jm_db.schema";
  public static final String dbJobMaxRowsLimit = "jm_db.jobmaxrecords";
  public static final String dbTaskMaxRowsLimit = "jm_db.taskmaxrecords";
  public static final String jmConfigured = "jm_configured";
  public static final String centralConfigPath = "/var/mapr/configuration/default";

  public static final String KEYS_PARAM_NAME = "keys";
  public static final String VALUES_PARAM_NAME = "values";
  public static final String TEST_PARAM_NAME = "test";  
  
  public final String SUPER_PARAM_PREFIX = "mapr.fs.permissions.";
  public final String ParamSuperUser = SUPER_PARAM_PREFIX + "superuser";
  
  private static final String PassPhrase = "MapR_Random_$$_Phrase";
  private Properties CLDBProperties = new Properties();

  private static final Logger LOG = Logger.getLogger(ConfigCommands.class);
  
  public ConfigCommands(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }

  static final CLICommand loadCmd = new CLICommand(
      "load",
      "",
      ConfigCommands.class, 
      ExecutionTypeEnum.NATIVE,
      /* Add parameters in a hash map key is string,  value is BaseInputParameter */
      new ImmutableMap.Builder<String, BaseInputParameter>()
         /* parameter 1 */
         .put(ConfigCommands.KEYS_PARAM_NAME,
              new TextInputParameter(
                   ConfigCommands.KEYS_PARAM_NAME,
                   "list of keys",
                   CLIBaseClass.REQUIRED,
                   ""))
         .put(MapRCliUtil.CLUSTER_NAME_PARAM,
        		 new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
          "cluster name",
          CLIBaseClass.NOT_REQUIRED,
          null))
         .build(),
          null        /* sub-commands */
  ).setShortUsage("config load -keys <keys>");

  static final CLICommand saveCmd = new CLICommand(
      "save",
      "",
      ConfigCommands.class, 
      ExecutionTypeEnum.NATIVE,
      /* Add parameters in a hash map key is string,  value is BaseInputParameter */
      new ImmutableMap.Builder<String, BaseInputParameter>()
         /* parameter 1 */
           .put(ConfigCommands.VALUES_PARAM_NAME, 
        		   new JSONObjectInputParameter(ConfigCommands.VALUES_PARAM_NAME, 
        		   "JSON Object to comprise all config properties to save", 
        		   CLIBaseClass.REQUIRED))
                   /* parameter 3 */
           .put(ConfigCommands.TEST_PARAM_NAME,
                new IntegerInputParameter(
                    ConfigCommands.TEST_PARAM_NAME,
                    "test only",
                    CLIBaseClass.REQUIRED,
                    0))  
           .put(MapRCliUtil.CLUSTER_NAME_PARAM,
        		   new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
          "cluster name",
          CLIBaseClass.NOT_REQUIRED,
          null))
         .build(),
          null        /* sub-commands */
  ).setShortUsage("config save -values <keyvalues>");
   
  /* Define sub command */
  public static final CLICommand configCmds = new CLICommand(
      "config",
      "",
      CLIUsageOnlyCommand.class, 
      ExecutionTypeEnum.NATIVE,
      new CLICommand [] {         /* array of subcommands */
         loadCmd,
         saveCmd
    }
  ).setShortUsage("config [load|save]");
  

  String encryptPassword(String p) throws Exception {    
    DESKeySpec keySpec = new DESKeySpec(PassPhrase.getBytes("UTF8"));
    SecretKey PassKey = SecretKeyFactory.getInstance("DES").generateSecret(
        keySpec);
    Base64 base64encoder = new Base64();

    byte[] cleartext = p.getBytes("UTF8");
    Cipher cipher = Cipher.getInstance("DES"); // cipher is not thread safe
    cipher.init(Cipher.ENCRYPT_MODE, PassKey);
    return new String(base64encoder.encode(cipher.doFinal(cleartext)));
  }

  boolean PushCLDBParams(OutputHierarchy out) throws CLIProcessingException {
    try {
   
      CLDBConfigRequest.Builder reqbuilder = CLDBConfigRequest.newBuilder();
      reqbuilder.setLoad(false);

      if (getParamIntValue(TEST_PARAM_NAME, 0) == 1) {
        reqbuilder.setTest(true);
      }

      CLDBConfigParams.Builder pBuilder = CLDBConfigParams.newBuilder();
      for (Object o : CLDBProperties.keySet()) {
        String key = (String) o;
        CLDBConfigParam param = CLDBConfigParam.newBuilder()
                                               .setKeys(key)
                                               .setValues(CLDBProperties.getProperty(key))
                                               .build();
        pBuilder.addParams(param);
        
      }
      reqbuilder.setParams(pBuilder);
      
      CLDBConfigRequest req = reqbuilder.setCreds(getUserCredentials()).build();
      
      byte[] replyData;
      if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        String cluster = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0);
    	  replyData = CLDBRpcCommonUtils.getInstance().sendRequest(cluster,
   			                       Common.MapRProgramId.CldbProgramId.getNumber(),
          	                   CLDBProto.CLDBProg.CLDBConfigProc.getNumber(), req, 
                               CLDBConfigResponse.class);
      } else {
    	  replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
    			                    Common.MapRProgramId.CldbProgramId.getNumber(),
    	                        CLDBProto.CLDBProg.CLDBConfigProc.getNumber(), req, 
                              CLDBConfigResponse.class);
      }
      
      // Do some error handling
      if (replyData == null) {
        out.addError(new OutputError(
            Errno.EOPFAILED, "Invalid response to RPC, from CLDB"));
        return false;
      }

      CLDBConfigResponse resp = CLDBConfigResponse.parseFrom(replyData);
      if (resp.getStatus() != 0) {
        String errorMsg = "Failed to save config: " + Errno.toString(resp.getStatus()) + ". ";
        if (resp.hasErrorString()) {
          errorMsg += resp.getErrorString();
        }
        out.addError(new OutputError(resp.getStatus(), errorMsg));
        return false;
      }
      
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      /**
      * <MAPR_ERROR>
      * Message:Exception: <error>
      * Function:ConfigCommands.PushCLDBParams()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      out.addError(new OutputError(
          Errno.EOPFAILED, "Exception during config save"));
      LOG.error("Exception: " + e.getLocalizedMessage());
      return false;
    }
    return true;
  }

  void LoadParam(String key, String val, OutputNode node) {
    if ((val == null) || val.isEmpty()) {
      return;
    }
    
    if (val.equalsIgnoreCase("true")) {
      val = "1";
    } else if (val.equalsIgnoreCase("false")) {
      val = "0";
    }
    if ( key.indexOf(PASSWORD_PREFIX) >= 0 ) {
      node.addChild(new OutputNode(key, "****"));
    } else {
      node.addChild(new OutputNode(key, val));
    }
  }

  void LoadParam(String key, OutputNode node) throws CLIProcessingException {
    
    try {
      if (key.equalsIgnoreCase("smtp") || key.equalsIgnoreCase("ldap")
          || key.equalsIgnoreCase("quota") || key.equalsIgnoreCase("super")) {
        for (Object o : CLDBProperties.keySet()) {
          String k = (String) o;
          if (k.contains(key.subSequence(0, key.length() - 1))) {
            LoadParam(k, CLDBProperties.getProperty(k), node);
          }
        }
      } else {
        LoadParam(key, CLDBProperties.getProperty(key), node);
      }
    } catch (Exception e) {
      /**
      * <MAPR_ERROR>
      * Message:Exception: <error>
      * Function:ConfigCommands.LoadParam()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      LOG.error("Exception: " + e.getLocalizedMessage());
    }
  }

  /* This is a function that does the real work. */
  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    String cluster = null;
    if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
      cluster = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0);
      if (!CLDBRpcCommonUtils.getInstance().isValidClusterName(cluster)) {
        out.addError(new OutputError(Errno.EUCLUSTER, "Invalid cluster: " + cluster));
        return output;
      }
    }

    String cmd = cliCommand.getCommandName();
    if (cmd.equalsIgnoreCase("load")) {
      String keys = getParamTextValue(KEYS_PARAM_NAME, 0);
      if ((keys == null) || keys.isEmpty()) {
        // Load all params
        int status =
             MapRCliUtil.fetchCLDBParams(cluster, CLDBProperties, getUserCredentials());
        if (status < 0) {
          out.addError(new OutputError(Errno.EOPFAILED,
                                       "Unable to reach CLDB"));
          return output;
        } else if (status != 0) {
          out.addError(new OutputError(Errno.EOPFAILED,
                                       "Failed to load configuration - " + Errno.toString(status)));
          return output;
        }
      
        OutputNode node = new OutputNode();
        boolean added = false;
        
        TreeSet ts = new TreeSet(CLDBProperties.keySet());
        for (Object o: ts) {
          String key = (String)o;
          LoadParam(key, node);
          added = true;
        }
        if (added)
          out.addNode(node);
        return output;
      } 
      
      if ( keys.contains(dbUrlParamName) ||
           keys.contains(dbUserParamName) ||
           keys.contains(dbPasswdParamName) ||
           keys.contains(dbJobMaxRowsLimit) ||
           keys.contains(dbTaskMaxRowsLimit) ||
           keys.contains(dbSchemaParamName) ) {
       // read /opt/mapr/conf/db.conf to get default values
       String maprInstallPath = MapRCliUtil.getMapRInstallDir();
       String jmrolePath = maprInstallPath + "/roles/metrics";
       File jmRoleFile = new File(jmrolePath);
       if ( jmRoleFile.isFile() && jmRoleFile.exists() ) {
         String dbConfPath = maprInstallPath + "/conf/db.conf";
         Properties dbProperties = new Properties();
         try {
           dbProperties.load(new FileInputStream(dbConfPath));
           String dbUrl = dbProperties.getProperty("db.url");
           if ( dbUrl != null )
             CLDBProperties.setProperty(dbUrlParamName, dbUrl);
           String dbUser = dbProperties.getProperty("db.user");
           if ( dbUser != null )
             CLDBProperties.setProperty(dbUserParamName, dbUser);
           String dbPasswd = dbProperties.getProperty("db.passwd");
           if ( dbPasswd != null )
             CLDBProperties.setProperty(dbPasswdParamName, dbPasswd);
           String dbSchema = dbProperties.getProperty("db.schema");
           if ( dbSchema != null )
             CLDBProperties.setProperty(dbSchemaParamName, dbSchema);
           String dbJobMaxRows = dbProperties.getProperty("db.job.histogram.query.records.max");
           if ( dbJobMaxRows != null )
             CLDBProperties.setProperty(dbJobMaxRowsLimit, dbJobMaxRows);
           String dbTaskMaxRows = dbProperties.getProperty("db.task.histogram.query.records.max");
           if ( dbTaskMaxRows != null )
             CLDBProperties.setProperty(dbTaskMaxRowsLimit, dbTaskMaxRows);

         } catch (Throwable e) {
           LOG.error("Exception while trying to read: " + dbConfPath, e);
         }         
       }
     }

      // Fetch all params store in CLDB once and store it
      int status = 
          MapRCliUtil.fetchCLDBParams(cluster, CLDBProperties, getUserCredentials());
      if (status < 0) {
        out.addError(new OutputError(Errno.EOPFAILED,
                                     "Unable to reach CLDB"));
        return output;
      } else if (status != 0) {
        out.addError(new OutputError(Errno.EOPFAILED,
                                     "Failed to load configuration - " + Errno.toString(status)));
        return output;
      }
      
      OutputNode node = new OutputNode();
      for (String key : keys.split(",")) {
        LoadParam(key, node);
      }
      out.addNode(node);
    } else if (cmd.equalsIgnoreCase("save")) {
    	// validation is already done by this point - no point to revalidate here again
  		String jSonString = input.getParameterByName(VALUES_PARAM_NAME).getParamValues().get(0);
  		// parse key value pairs out of jSonString
	    try {
  			Map<String, String> keyValues = parseParamsFromJSONObject(jSonString);
  			
  			for ( Map.Entry<String, String> keyValuesEntry : keyValues.entrySet() ) {
  				String key = keyValuesEntry.getKey();
  				String value = keyValuesEntry.getValue().trim();
  				
  				if (key.equalsIgnoreCase(ParamSuperUser)) {
	          out.addError(new OutputError(Errno.EPERM, "Cannot modify " + key));
	          continue;
  			  }

          // If it is a password, encrypt it
          if (key.contains("password")) {
            try {
              value = encryptPassword(value);
            } catch (Exception e) {
              /**
               * <MAPR_ERROR>
               * Message:Exception: <error>
               * Function:ConfigCommands.executeRealCommand()
               * Meaning:An error occurred.
               * Resolution:Contact technical support.
               * </MAPR_ERROR>
               */
              LOG.error("Exception: " + e.getLocalizedMessage());
              out.addError(new OutputError(Errno.EOPFAILED,
                  "Exception during password encryption" + e.getLocalizedMessage()));
              return output;
            }
          }
          CLDBProperties.setProperty(key, value);
       	}
  			
	      if (!CLDBProperties.isEmpty()) {
	        boolean isNeedDBStuff = isToAddCLDBDBCredentials();
          if ( PushCLDBParams(out) && isNeedDBStuff) {
            handleDBCredentials();
          }
	      }
  		} catch (JSONException e1) {
  			LOG.error("JSON Exception during " + VALUES_PARAM_NAME + " parsing", e1);
  			out.addError(new OutputError(Errno.EINVAL, "JSON Exception during " + VALUES_PARAM_NAME + " parsing"));
  			return output;
  		}
		
    }
    return output;
  }
  
  public static Map<String, String> parseParamsFromJSONObject(String jsonString)
  throws JSONException {
	JSONObject jO = new JSONObject(jsonString);
	
	Map<String, String> jM = new HashMap<String, String>();
	
	Iterator<String> iter = jO.keys();
	while (iter.hasNext()) {
	  String jK = iter.next();
	  String value = jO.getString(jK);
	  jM.put(jK, value);
	}
	
	return jM;
  }

  private boolean isToAddCLDBDBCredentials() {
    String dbUrl = CLDBProperties.getProperty(dbUrlParamName);
    String dbUser = CLDBProperties.getProperty(dbUserParamName);
    String dbPasswd = CLDBProperties.getProperty(dbPasswdParamName);
    
    if ( dbUrl == null || dbUser == null || dbPasswd == null || 
        dbUrl.isEmpty() || dbUser.isEmpty() || dbPasswd.isEmpty() ) {
      // nothing to do here
      // No need to reset jmConfigured
      //CLDBProperties.setProperty(jmConfigured, "0");
      return false;
    } 
    CLDBProperties.setProperty(jmConfigured, "1");
    return true;
  }
  
  private void handleDBCredentials() {
    String dbUrl = CLDBProperties.getProperty(dbUrlParamName);
    String dbUser = CLDBProperties.getProperty(dbUserParamName);
    String dbPasswd = CLDBProperties.getProperty(dbPasswdParamName);
    String dbSchema = CLDBProperties.getProperty(dbSchemaParamName);
    String dbJobMaxRows = CLDBProperties.getProperty(dbJobMaxRowsLimit);
    String dbTaskMaxRows = CLDBProperties.getProperty(dbTaskMaxRowsLimit);
    
    if ( dbUrl == null || dbUser == null || dbPasswd == null ) {
      // nothing to do here
      return;
    } 
    
    // createfilesuffix
    Calendar cal = Calendar.getInstance();
    StringBuilder suffix = new StringBuilder(".");
    suffix.append(cal.get(Calendar.YEAR));
    suffix.append("-");
    suffix.append(cal.get(Calendar.MONTH)+1);
    suffix.append("-");
    suffix.append(cal.get(Calendar.DAY_OF_MONTH));
    suffix.append(".");
    suffix.append(cal.get(Calendar.HOUR_OF_DAY));
    suffix.append("-");
    suffix.append(cal.get(Calendar.MINUTE));
    String suffixStr = suffix.toString();
    
    // update /opt/mapr/conf/db.conf and put it into central config
    String maprInstallPath = MapRCliUtil.getMapRInstallDir();
    String dbConfPath = maprInstallPath + "/conf/db.conf";
    String hibernateConfPath = maprInstallPath + "/conf/hibernate.cfg.xml";
    String jmrolePath = maprInstallPath + "/roles/metrics";
    String webRolePath = maprInstallPath + "/roles/webserver";
    File jmRoleFile = new File(jmrolePath);
    if ( !jmRoleFile.isFile() || !jmRoleFile.exists() ) {
      // jmrole is not even there no action should be done
      LOG.warn("mapr-metrics package is not installed. No DB params will be saved");
      return;
    }
    
    // do regex
    try {
      BufferedReader fr = new BufferedReader(new FileReader(dbConfPath));
      BufferedWriter bw = new BufferedWriter(new FileWriter(dbConfPath + suffixStr));
      StringWriter strW = new StringWriter();
      String fileLine;
      while ( (fileLine = fr.readLine()) != null ) {
        String modfileLine = fileLine.replaceFirst("db.url=.*", "db.url="+dbUrl);
        modfileLine = modfileLine.replaceFirst("db.user=.*", "db.user="+dbUser);
        modfileLine = modfileLine.replaceFirst("db.passwd=.*", "db.passwd="+dbPasswd);
        if ( dbSchema != null ) {
          modfileLine = modfileLine.replaceFirst("db.schema=.*", "db.schema="+dbSchema);
        }
        if ( dbJobMaxRows != null ) {
          modfileLine = modfileLine.replaceFirst("db.job.histogram.query.records.max=.*", "db.job.histogram.query.records.max="+dbJobMaxRows);
        }
        if ( dbTaskMaxRows != null ) {
          modfileLine = modfileLine.replaceFirst("db.task.histogram.query.records.max=.*", "db.task.histogram.query.records.max="+dbTaskMaxRows);
        }


        strW.write(modfileLine);
        strW.write('\n');
        // writing unmodified file into dbConfPath + suffixStr
        bw.write(fileLine);
        bw.write('\n');
      }
      fr.close();
      bw.close();
      // overwrite original file
      FileOutputStream fos = new FileOutputStream(new File(dbConfPath));
      fos.write(strW.getBuffer().toString().getBytes());
      fos.close();
     } catch (IOException e) {
      LOG.error("IOException while handling DB Credentials replacement in: " + dbConfPath, e);
     }
     try {
       BufferedReader fr = new BufferedReader(new FileReader(hibernateConfPath));
       BufferedWriter bw = new BufferedWriter(new FileWriter(hibernateConfPath + suffixStr));
       StringWriter strW = new StringWriter();
       String fileLine;
       while ( (fileLine = fr.readLine()) != null ) {
         String modfileLine = fileLine.replaceFirst("jdbc:mysql://[a-zA-Z_0-9:.]*/", "jdbc:mysql://" + dbUrl + "/");
         if ( dbSchema != null ) {
           // replace schema as well
           modfileLine = modfileLine.replaceFirst(dbUrl + "/.*</property>", dbUrl + "/" + dbSchema + "</property>");
         }
         modfileLine = modfileLine.replaceFirst("\"connection.username\">.*<", "\"connection.username\">" + dbUser + "<");
         modfileLine = modfileLine.replaceFirst("\"connection.password\">.*<", "\"connection.password\">" + dbPasswd.replaceAll("&","&amp;") + "<");
         strW.write(modfileLine);
         strW.write('\n');
         // writing unmodified file into hibernateConfPath + suffixStr
         bw.write(fileLine);
         bw.write('\n');
       }
       fr.close();
       bw.close();
       // overwrite original file
       FileOutputStream fos = new FileOutputStream(new File(hibernateConfPath));
       fos.write(strW.getBuffer().toString().getBytes());
       fos.close();
     } catch (IOException e) {
       LOG.error("IOException while handling DB Credentials replacement in: " + hibernateConfPath, e);
     }
    // place db.conf and hibernate.cfg.xml on maprfs central config path
    try {
      Path dbConfFSPath = new Path(centralConfigPath + "/conf/db.conf");
      Path hibernateConfFSPath = new Path(centralConfigPath + "/conf/hibernate.cfg.xml");
      boolean isPathCreated = MapRFSUtil.getMapRFileSystem().mkdirs(new Path(centralConfigPath + "/conf"));
      if ( isPathCreated ) {
        MapRFSUtil.getMapRFileSystem().copyFromLocalFile(false, true, new Path(dbConfPath), dbConfFSPath);
        MapRFSUtil.getMapRFileSystem().setPermission(dbConfFSPath, FsPermission.createImmutable((short) 0700));
        File webRoleFile = new File(webRolePath);
        if ( webRoleFile.isFile() && webRoleFile.exists() ) {
          MapRFSUtil.getMapRFileSystem().copyFromLocalFile(false, true, new Path(hibernateConfPath), hibernateConfFSPath);
          MapRFSUtil.getMapRFileSystem().setPermission(hibernateConfFSPath, FsPermission.createImmutable((short) 0700));
        }
      }
    } catch (IOException e) {
      LOG.error("IOException while trying to update db.conf in centralconfig", e);
    } catch (Throwable t) {
      LOG.error("Exception while trying to process DB credentials", t);
    }
  }
}
