package com.mapr.cli;

import com.google.common.collect.ImmutableMap;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cli.common.NodesCommonUtils;
import com.mapr.cliframework.base.*;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import com.mapr.cliframework.base.inputparams.NoValueInputParameter;
import com.mapr.security.MaprSecurityException;
import com.mapr.util.MapRFSUtil;
import org.apache.hadoop.fs.Path;
import org.apache.log4j.Logger;

import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.MessageLite;

import com.mapr.fs.cldb.util.Util;
import com.mapr.fs.Rpc;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Common.MapRProgramId;
import com.mapr.fs.proto.Common.IPAddress;
import com.mapr.fs.proto.Dbreplicator.DBReplicatorProg;
import com.mapr.fs.proto.Dbreplicator.GetHostNamesRequest;
import com.mapr.fs.proto.Dbreplicator.GetHostNamesResponse;
import com.mapr.fs.proto.Security.ServerKeyType;
import com.mapr.fs.proto.Common.ServiceData;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.CLDBProg;
import com.mapr.fs.cldb.proto.CLDBProto.MaprFeatureInfo;
import com.mapr.fs.cldb.proto.CLDBProto.FeatureEnableRequest;
import com.mapr.fs.cldb.proto.CLDBProto.FeatureEnableResponse;
import com.mapr.fs.cldb.proto.CLDBProto.FeatureListRequest;
import com.mapr.fs.cldb.proto.CLDBProto.FeatureListResponse;
import com.mapr.fs.proto.Security.CredentialsMsg;


import java.io.*;
import java.util.Properties;
import java.util.Collections;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import static com.mapr.cliframework.base.CommandOutput.*;
import static com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;

public class ClusterCommands extends CLIBaseClass implements CLIInterface {

    public static class VersionFileContents {
        String classic_version;
        String yarn_version;
        String default_mode;
    }

    private static final String SETMODE_PARAM = "mode";

    private static final String FEATURE_NAME = "name";
    private static final String FEATURE_FORCE = "force";
    private static final String FEATURE_ENABLED = "enabled";
    private static final String FEATURE_DISABLED = "disabled";
    private static final String FEATURE_ALL = "all";
    private static final String MULTI_ARG_SEP = ",";


    public static final String centralConfigPath = "/var/mapr/configuration/default";
    public static final String versionFilePath = "/conf/hadoop_version";
    private static final String maprInstallPath = MapRCliUtil.getMapRInstallDir();
    private static final String hadoopVersionLocalFile = maprInstallPath + versionFilePath;
    private static final String daemonConfFile = maprInstallPath + "/conf/daemon.conf";
    private static final String CLASSIC_VERSION_TOKEN = "classic_version";
    private static final String YARN_VERSION_TOKEN = "yarn_version";
    private static final String DEFAULT_MODE_TOKEN = "default_mode";

    private static final Logger LOG = Logger.getLogger(ClusterCommands.class);

    /* 
     * The following three are the sub commands for the command
     * maprcli cluster mapreduce
     */
    private static final CLICommand mapreduceGetAllVersionCommand =
        new CLICommand("getall",
            "Get MapReduce Version values",
            ClusterCommands.class, 
            CLICommand.ExecutionTypeEnum.NATIVE,
            new ImmutableMap.Builder<String, BaseInputParameter>().build(),
            null).setShortUsage("Shows the MapReduce version info")
        .setUsageInVisible(true);

    private static final CLICommand mapreduceGetVersionCommand =
        new CLICommand("get",
            "Get Cluster wide MapReduce default mode",
            ClusterCommands.class, 
            CLICommand.ExecutionTypeEnum.NATIVE,
            new ImmutableMap.Builder<String, BaseInputParameter>().build(),
            null)
        .setShortUsage("Shows the current cluster MapReduce default version");

    private static final CLICommand mapreduceSetVersionCommand =
        new CLICommand("set",
            "Set Cluster wide MapReduce default mode",
            ClusterCommands.class, 
            CLICommand.ExecutionTypeEnum.NATIVE,
            new ImmutableMap.Builder<String, BaseInputParameter>()
            .put(SETMODE_PARAM, new TextInputParameter(
                    SETMODE_PARAM, "Sets the default MapReduce default version <classic|yarn>",
                    CLIBaseClass.REQUIRED, null))
            .build(),
            null);

    private static final CLICommand mapreduceCommands = 
        new CLICommand("mapreduce", 
            "Get or Set Cluster wide MapReduce defaults such as version",
            CLIUsageOnlyCommand.class, 
            CLICommand.ExecutionTypeEnum.NATIVE, 
            new CLICommand[] {mapreduceGetAllVersionCommand, mapreduceGetVersionCommand, mapreduceSetVersionCommand});

    static final CLICommand featureEnableCommand =
      new CLICommand("enable", "usage: cluster feature enable [-name <feature name> | -force <true|false> | -all ]",
          ClusterCommands.class, CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
          .put(ClusterCommands.FEATURE_NAME,
            new TextInputParameter(ClusterCommands.FEATURE_NAME, "feature name",
              CLIBaseClass.NOT_REQUIRED, null /* default param value */))
          .put(ClusterCommands.FEATURE_FORCE,
            new BooleanInputParameter(ClusterCommands.FEATURE_FORCE, "<true|false> true enables depedency features too",
              CLIBaseClass.NOT_REQUIRED, null /* default param value */))
          .put(ClusterCommands.FEATURE_ALL,
            new NoValueInputParameter(ClusterCommands.FEATURE_ALL, "all features",
              CLIBaseClass.NOT_REQUIRED, CLIBaseClass.NOT_REQUIRED /* value  not required */))
          .build(),
          null/* no subcommands */)
      .setShortUsage("Enables feature");

    static final CLICommand featureListCommand =
      new CLICommand("list", "usage: cluster feature list [ -name <featurename> -type <cldb|mfs> -state <enabled|disabled> -class <v2|v3> ]",
          ClusterCommands.class, CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
          .put(ClusterCommands.FEATURE_NAME,
            new TextInputParameter(ClusterCommands.FEATURE_NAME, "feature name",
              CLIBaseClass.NOT_REQUIRED, null /* default param value */))
          .put(ClusterCommands.FEATURE_ENABLED,
            new NoValueInputParameter(ClusterCommands.FEATURE_ENABLED, "enabled features only",
              CLIBaseClass.NOT_REQUIRED, CLIBaseClass.NOT_REQUIRED /* value  not required */))
          .put(ClusterCommands.FEATURE_DISABLED,
            new NoValueInputParameter(ClusterCommands.FEATURE_DISABLED, "disabled features only",
              CLIBaseClass.NOT_REQUIRED, CLIBaseClass.NOT_REQUIRED /* value  not required */))
          .build(),
          null/* no subcommands */)
      .setShortUsage("Lists features on the cluster");

    public static final CLICommand featureCommand = new CLICommand("feature", "Enable or List features on the cluster",
            CLIUsageOnlyCommand.class, CLICommand.ExecutionTypeEnum.NATIVE, new CLICommand[] {featureEnableCommand, featureListCommand});

    public static final CLICommand clusterCommands = new CLICommand("cluster", "cluster wide info such as mapreduce version",
            CLIUsageOnlyCommand.class, CLICommand.ExecutionTypeEnum.NATIVE, new CLICommand[] {mapreduceCommands, DbGatewayCommands.gatewayCommands, featureCommand})
            .setShortUsage("cluster [mapreduce [get|set] | gateway | feature [enable|list]]");

    public ClusterCommands(ProcessedInput input, CLICommand cliCommand) {
        super(input, cliCommand);
    }

    @Override
    public CommandOutput executeRealCommand() throws CLIProcessingException {

        OutputHierarchy oh = new OutputHierarchy();
        CommandOutput output = new CommandOutput();
        output.setOutput(oh);

        String cmd = cliCommand.getCommandName();

        if(cmd.equalsIgnoreCase("set")) {
            // Modify the current version.
            final String default_mode = getParamTextValue(SETMODE_PARAM, 0);
            if ( ! default_mode.equals("classic") && ! default_mode.equals("yarn")) {
                LOG.error("Default MapReduce specified should be classic or yarn");
                oh.addError(new OutputHierarchy.OutputError(Errno.EINVAL, "Invalid value for Default Mapreduce Mode. It should be classic or yarn"));
                return output;
            }
            VersionFileContents newContents = setClusterDefaultMapReduceMode(default_mode);
            if (newContents != null) {
                addContentsToOutput(oh, newContents);
            } else {
                oh.addError(new OutputHierarchy.OutputError(Errno.EOPFAILED, "Unable to set Mapreduce Mode. Check maprcli/adminuiapp log for more details."));
            }
        } else if (cmd.equalsIgnoreCase("get")) {
            // Display the current hadoop version configured.
            VersionFileContents contents = readHadoopVersionFile();
            if(contents != null) {
                addContentsToOutput(oh, contents);
            }
        } else if (cmd.equalsIgnoreCase("getall")) {
            VersionFileContents contents = readHadoopVersionFile();
            if(contents != null)
                addAllContentsToOutput(oh, contents);
        } else if (cmd.equalsIgnoreCase("enable")) {
          // cluster feature enable
            enableFeature(oh);
            LOG.error("enable: cluster feature " + cmd);
        } else if (cmd.equalsIgnoreCase("list")) {
          // cluster feature list
            listFeatures(oh);
            LOG.error("list: cluster feature " + cmd);
        }

        return output;
    }

    private void enableFeature(OutputHierarchy oh) throws CLIProcessingException {
      if (!isParamPresent(ClusterCommands.FEATURE_NAME) && 
          !isParamPresent(ClusterCommands.FEATURE_ALL)) {
        oh.addMessage(getCommandUsage());
        oh.addError(new OutputHierarchy.OutputError(Errno.EMISSING, "feature enable arguments missing. " +
              "Pass either featurename or 'all'"));
        return;
      }

      if (isParamPresent(ClusterCommands.FEATURE_NAME) && 
          isParamPresent(ClusterCommands.FEATURE_ALL)) {
        oh.addMessage(getCommandUsage());
        oh.addError(new OutputHierarchy.OutputError(Errno.EINVAL, "feature enable arguments must not provide 'name' and 'all' together."));
        return;
      }

      boolean force = false;
      if (isParamPresent(ClusterCommands.FEATURE_FORCE)) {
        force = getParamBooleanValue(FEATURE_FORCE, 0);
      }

      FeatureEnableRequest.Builder featureEnableReqBuilder = FeatureEnableRequest.newBuilder()
                                                                                 .setCreds(getUserCredentials())
                                                                                 .setForce(force);

      if (isParamPresent(ClusterCommands.FEATURE_NAME)) {
        String featureNames = getParamTextValue(FEATURE_NAME, 0);
        List<String> names = new ArrayList<String>();

        if (featureNames.contains(MULTI_ARG_SEP)) {
          names = Arrays.asList(featureNames.split(MULTI_ARG_SEP));
        } else {
          names.add(featureNames);
        }
        featureEnableReqBuilder.addAllFeatures(names);

      } else if (isParamPresent(ClusterCommands.FEATURE_ALL)) {
        // nothing to do. CLDB enables all features if no feature is mentioned.
        featureEnableReqBuilder.addFeatures(ClusterCommands.FEATURE_ALL);
      }

      byte[] data = sendRequestInternal(CLDBProg.FeatureEnableProc, featureEnableReqBuilder.build(), oh);

      FeatureEnableResponse resp = null;
      try {
        resp = FeatureEnableResponse.parseFrom(data);
      } catch (InvalidProtocolBufferException ipbe) {
        oh.addError(new OutputHierarchy.OutputError(Errno.EINVAL,
              "error parsing feature enable response."));
        LOG.error("Feature enable failed. Exception parsing feature enable response.", ipbe);
        return;
      }

      if (resp.getStatus() == 0) {
        OutputHierarchy toh = formatMaprFeatureInfoList(resp.getFeaturesList());
        for (OutputNode a : toh.getOutputNodes()) {
          oh.addNode(a);
        }
      } else {
        oh.addError(new OutputHierarchy.OutputError(resp.getStatus(), resp.getErrMsg()));
        LOG.error("Feature enable failed. status: " + resp.getStatus()  + " " + resp.getErrMsg());
      }
    }

    private void listFeatures(OutputHierarchy oh) throws CLIProcessingException {
      FeatureListRequest.Builder featureInfoRequestBuilder = FeatureListRequest.newBuilder()
        .setCreds(getUserCredentials());

      List<String> names = new ArrayList<String>();
      if (isParamPresent(ClusterCommands.FEATURE_NAME)) {
        String featureNames = getParamTextValue(FEATURE_NAME, 0);

        if (!featureNames.contains(MULTI_ARG_SEP)) {
          names.add(featureNames);
        } else {
          names = Arrays.asList(featureNames.split(MULTI_ARG_SEP));
        }
      } else {
        names.add("all");
      }
      featureInfoRequestBuilder.addAllFeatures(names);

      byte[] data = sendRequestInternal(CLDBProg.FeatureListProc, featureInfoRequestBuilder.build(), oh);

      FeatureListResponse resp = null;
      try {
        resp = FeatureListResponse.parseFrom(data);
      } catch (InvalidProtocolBufferException ipbe) {
        oh.addError(new OutputHierarchy.OutputError(Errno.EINVAL,
              "error parsing feature list response."));
        LOG.error("Feature list failed. Exception parsing feature list response.", ipbe);
        return;
      }

      if (resp.getStatus() == 0) {
        OutputHierarchy toh = formatMaprFeatureInfoList(resp.getFeaturesList());
        for (OutputNode a : toh.getOutputNodes()) {
          oh.addNode(a);
        }
      } else {
        oh.addError(new OutputHierarchy.OutputError(resp.getStatus(), "Feature list failed: " + resp.getErrMsg()));
        LOG.error("Feature list failed. status: " + resp.getStatus()  + " " + resp.getErrMsg());
      }

    }

    private OutputHierarchy formatMaprFeatureInfoList(List<MaprFeatureInfo> mfList) throws CLIProcessingException {
     OutputHierarchy oh = new OutputHierarchy();
     for (MaprFeatureInfo mf : mfList) {
       if ((isParamPresent(ClusterCommands.FEATURE_ENABLED) && mf.getEnabled()) ||
           (isParamPresent(ClusterCommands.FEATURE_DISABLED) && !mf.getEnabled()) ||
           (!isParamPresent(ClusterCommands.FEATURE_ENABLED) && !isParamPresent(ClusterCommands.FEATURE_DISABLED))) {
         oh.addNode( formatMaprFeatureInfo(mf));
       }
     }
     return oh;
    }

    private OutputNode formatMaprFeatureInfo(MaprFeatureInfo mf) throws CLIProcessingException {
      OutputNode out = new OutputNode();

      if (mf.hasFeatureName()) {
        out.addChild(new OutputNode("name", mf.getFeatureName()));
      }

      if (mf.hasEnabled()) {
        out.addChild(new OutputNode("enabled", mf.getEnabled()));
      }

      if (mf.hasDescription()) {
        out.addChild(new OutputNode("description", mf.getDescription()));
      }

      if (mf.getDependenceInfoList().size() > 0) {
        OutputHierarchy oh = formatMaprFeatureInfoList(mf.getDependenceInfoList());
        for (OutputNode a : oh.getOutputNodes()) {
          out.addChild(new OutputNode("dependency", a));
        }        
      }

      return out;
    }


    private byte[] sendRequestInternal(CLDBProg procId,
                                       MessageLite req,
                                       OutputHierarchy out) throws CLIProcessingException {
      byte[] data = null;
      Class<? extends MessageLite> responseClass = null;

      switch (procId) {
        case FeatureEnableProc:
          responseClass = FeatureEnableResponse.class;
          break;
        case FeatureListProc:
          responseClass = FeatureListResponse.class;
          break;
      }

      if (responseClass == null) {
        LOG.error("Unknown procId for send request " + procId);
        out.addError(new OutputHierarchy.OutputError(Errno.EINVAL,
              "Unknown procId for send request " + procId));
        return data;
      }

      try {
        if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
          data = CLDBRpcCommonUtils.getInstance().sendRequest(
              getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0),
              Common.MapRProgramId.CldbProgramId.getNumber(),
              procId.getNumber(), req, responseClass);
        } else {
          data = CLDBRpcCommonUtils.getInstance().sendRequest(
              Common.MapRProgramId.CldbProgramId.getNumber(),
              procId.getNumber(), req, responseClass);
        }
      } catch (Exception e) {
        throw new CLIProcessingException(e);
      }
      return data;
    }

    private IPAddress buildIPFromString(String ipstr) {
        if (ipstr.equalsIgnoreCase("localhost") ||
                ipstr.equalsIgnoreCase("127.0.0.1") ) {
            ipstr = "127.0.0.1";
        } else {
            List<String> ips;
            ips = NodesCommonUtils.convertHostToIp(Collections.singletonList(ipstr));
            if (ips.isEmpty())
                return null;

            ipstr = ips.get(0);
        }

        int host = Util.ipToInt(ipstr);
        IPAddress server = IPAddress.newBuilder()
                .setHost(host)
                .build();
        return server;
    }

    private void addAllContentsToOutput(OutputHierarchy oh, VersionFileContents contents) {
        OutputNode node = new OutputNode();
        node.addChild(new OutputNode("classic_version", contents.classic_version));
        node.addChild(new OutputNode("yarn_version", contents.yarn_version));
        node.addChild(new OutputNode("default_mode", contents.default_mode));
        oh.addNode(node);
    }

    private void addContentsToOutput(OutputHierarchy oh, VersionFileContents contents) {
        OutputNode node = new OutputNode();
        node.addChild(new OutputNode("default_mode", contents.default_mode));
        String version = "";
        if (contents.default_mode != null) {
          version = contents.default_mode.equals("classic") ? contents.classic_version : contents.yarn_version;
        }
        node.addChild(new OutputNode("mapreduce_version", version));
        oh.addNode(node);
    }

    // Sets the default hadoop version and writes the local and central config files.
    private VersionFileContents setClusterDefaultMapReduceMode(String defaultMode) {

        // Read the local file.
        VersionFileContents contents = readHadoopVersionFile();

        if(contents == null) {
            LOG.error("Error while reading Local Hadoop Version file");
            return null;
        }

        // Set the new default mode.
        contents.default_mode = defaultMode;

        // Write to the local file. If success, copy it to the central file.
        if( writeHadoopVersionFile(contents) ) {
            copyToCentralConfig();
            return contents;
        }

        return null;

    }

    // Copies the local file to the Central Config location.
    private void copyToCentralConfig() {
        Path versionFileFSPath = new Path (centralConfigPath + versionFilePath);
        String mapr_user;
        String mapr_group;
        try {
            Properties userInfoProps = new Properties();
            userInfoProps.load(new BufferedInputStream(new FileInputStream(daemonConfFile)));
            mapr_user = userInfoProps.getProperty("mapr.daemon.user");
            mapr_group = userInfoProps.getProperty("mapr.daemon.group");
        } catch (IOException e) {
            LOG.error("Exception while trying to open " + daemonConfFile, e);
            return;
        }
        try {
            boolean isPathCreated = MapRFSUtil.getMapRFileSystem().mkdirs(new Path(centralConfigPath + "/conf"));
            if(isPathCreated) {
                MapRFSUtil.getMapRFileSystem().setOwner(new Path(centralConfigPath + "/conf"), mapr_user, mapr_group);
                MapRFSUtil.getMapRFileSystem().delete(versionFileFSPath, false);
                MapRFSUtil.getMapRFileSystem().copyFromLocalFile(false, true, new Path(hadoopVersionLocalFile), versionFileFSPath);
                MapRFSUtil.getMapRFileSystem().setOwner(versionFileFSPath, mapr_user, mapr_group);
            }
        } catch(IOException e) {
            LOG.error("IOException while copying to Central config location " + centralConfigPath, e);
        }
    }

    // Overwrites the local hadoop_version file with the contents provided.
    private boolean writeHadoopVersionFile(VersionFileContents contents) {
        if(contents == null || contents.classic_version == null || contents.yarn_version == null) {
            LOG.error("Malformed hadoop_version local file. Please check the contents of " + hadoopVersionLocalFile);
            return false;
        }
        BufferedWriter bw = null;
        try {
            bw = new BufferedWriter(new FileWriter(hadoopVersionLocalFile));
            bw.write(CLASSIC_VERSION_TOKEN + "=" + contents.classic_version + "\n");
            bw.write(YARN_VERSION_TOKEN + "=" + contents.yarn_version + "\n");
            bw.write(DEFAULT_MODE_TOKEN + "=" + contents.default_mode + "\n");
            bw.close();
        } catch (IOException e) {
            LOG.error("IOException while trying to write Hadoop version file " + hadoopVersionLocalFile, e);
            return false;
        } finally {
            if (bw != null) {
                try {
                    bw.close();
                } catch (IOException e) {
                }
            }
        }
        return true;
    }

    // Parses the local hadoop_version file and returns its contents.
    public static VersionFileContents readHadoopVersionFile() {
        VersionFileContents contents = new VersionFileContents();
        BufferedReader br = null;
        try {
            br = new BufferedReader(new FileReader(hadoopVersionLocalFile));
            String line;
            while ( (line = br.readLine()) != null) {
                String[] tok = line.split("=");
                if (tok.length > 1)
                  if(tok[0].equals(CLASSIC_VERSION_TOKEN))
                      contents.classic_version = tok[1];
                  else if(tok[0].equals(YARN_VERSION_TOKEN))
                      contents.yarn_version = tok[1];
                  else if(tok[0].equals(DEFAULT_MODE_TOKEN))
                      contents.default_mode = tok[1];
            }

        } catch(IOException e) {
            LOG.error("IOException while trying to read Hadoop version file " + hadoopVersionLocalFile, e);
            return null;
        } finally {
            if (br != null) {
                try {
                    br.close();
                } catch (IOException e) {
                }
            }
        }
        return contents;
    }
}
