package com.mapr.cli;

import java.io.IOException;
import java.util.Map;

import org.apache.log4j.Logger;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.ByteString;
import com.google.protobuf.InvalidProtocolBufferException;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.fs.AceHelper;
import com.mapr.fs.cldb.proto.Accesscontrol.GetClusterAcesRequest;
import com.mapr.fs.cldb.proto.Accesscontrol.GetClusterAcesResponse;
import com.mapr.fs.cldb.proto.Accesscontrol.SetClusterAcesRequest;
import com.mapr.fs.cldb.proto.Accesscontrol.SetClusterAcesResponse;
import com.mapr.fs.cldb.proto.CLDBProto.CLDBProg;
import com.mapr.fs.cldb.proto.CLDBProto.ClusterAceEntry;
import com.mapr.fs.cldb.proto.CLDBProto.ClusterAces;
import com.mapr.fs.cldb.proto.CLDBProto.ClusterActions;
import com.mapr.fs.proto.Common;
import com.mapr.security.MaprSecurityException;

public class ClusterAceCommands extends CLIBaseClass implements CLIInterface {

  private static final Logger LOG = Logger.getLogger(ClusterAceCommands.class);

  public ClusterAceCommands(final ProcessedInput input, final CLICommand cliCommand) {
    super(input, cliCommand);
  }

  /* Command and Sub Commands pertaining to ACEs */

  private static final String READCONFIGACE_PARAM = "readConfigAce";  /* allows one to view config info */
  private static final String SERVICESACE_PARAM = "servicesAce";  /* allows start/stop services */
  private static final String VOLUMECREATEACE_PARAM = "volumeCreateAce";
  private static final String ADMINACE_PARAM = "adminAce";  /* controls the change of ACEs */
  private static final String FCACE_PARAM = "fcAce"; /* controls all operations except change of ACEs */
  private static final String AUDITADMINACE_PARAM = "auditAce"; /* controls access to audit settings */

  private static final Map<String, ClusterActions> clusterAcesParameterMap =
      new ImmutableMap.Builder<String, ClusterActions>()
      .put(READCONFIGACE_PARAM, ClusterActions.CLUSTER_READ_ONLY)
      .put(SERVICESACE_PARAM, ClusterActions.CLUSTER_START_STOP_SERVICES)
      .put(VOLUMECREATEACE_PARAM, ClusterActions.CLUSTER_CREATE_VOLUMES)
      .put(ADMINACE_PARAM, ClusterActions.CLUSTER_ADMIN)
      .put(FCACE_PARAM, ClusterActions.CLUSTER_FULL_CONTROL)
      .put(AUDITADMINACE_PARAM, ClusterActions.CLUSTER_AUDIT_ADMIN)
      .build();

  private static final CLICommand setAceCommand =
      new CLICommand("set",
          "Sets Aces for Cluster Actions",
          ClusterAceCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
          .put(READCONFIGACE_PARAM, new TextInputParameter(READCONFIGACE_PARAM,
              "Ace to Control Viewing of Config Info",
              CLIBaseClass.NOT_REQUIRED, null))
          .put(SERVICESACE_PARAM, new TextInputParameter(SERVICESACE_PARAM,
              "Ace to Control Starting and Stopping Services",
              CLIBaseClass.NOT_REQUIRED, null))
          .put(VOLUMECREATEACE_PARAM, new TextInputParameter(VOLUMECREATEACE_PARAM,
              "Ace to Control Volume Creation",
              CLIBaseClass.NOT_REQUIRED, null))
          .put(ADMINACE_PARAM, new TextInputParameter(ADMINACE_PARAM,
              "Ace to Control Ace administration",
              CLIBaseClass.NOT_REQUIRED, null))
          .put(FCACE_PARAM, new TextInputParameter(FCACE_PARAM,
              "Ace to Control administering of all operations except aces",
              CLIBaseClass.NOT_REQUIRED, null))
          .put(AUDITADMINACE_PARAM, new TextInputParameter(AUDITADMINACE_PARAM,
              "Ace to Control Toggling of Audit",
              CLIBaseClass.NOT_REQUIRED, null)).build(), /* end of building Immutable Map */
          null);

  private static final CLICommand getAceCommand =
      new CLICommand("get",
          "Get Aces for All Cluster Actions",
          ClusterAceCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>().build(),
          null);

  public static final CLICommand aceCommands =
      new CLICommand("ace",
      "Set or Get ACE on Cluster Actions",
      CLIUsageOnlyCommand.class,
      CLICommand.ExecutionTypeEnum.NATIVE,
      new CLICommand[] {setAceCommand, getAceCommand});

  /* End of declaration for ACE-related commands */
  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    if (cliCommand.getCommandName().equalsIgnoreCase("set")) {
      return processSetAceCommand();
    }
    else if (cliCommand.getCommandName().equalsIgnoreCase("get")) {
      return processGetAceCommand();
    }
    else {
      final OutputHierarchy out = new OutputHierarchy() {{
        addError(new OutputError(Errno.EINVAL, "Unrecognized argument to Cluster Aces Command"));
      }};
      return new CommandOutput() {{
        setOutput(out);
      }};
    }
  }

  /**
   * Obtains Cluster ACEs info from the CLDB and formats it into CommandOutput.
   *
   * @return
   * @throws CLIProcessingException
   */
  private CommandOutput processGetAceCommand() throws CLIProcessingException {

    final OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput() {{
      setOutput(out);
    }};

    /* 1. Construct the Request */
    GetClusterAcesRequest request = GetClusterAcesRequest.newBuilder()
        .setCreds(getUserCredentials())
        .build();

    /* 2. Issue the RPC to fetch the Cluster Aces */
    byte[] data;
    try {
      if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(
            getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0),
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProg.GetClusterAcesProc.getNumber(),
            request,
            GetClusterAcesResponse.class);
      }
      else {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProg.GetClusterAcesProc.getNumber(),
            request,
            GetClusterAcesResponse.class);
      }
    } catch (InvalidProtocolBufferException | MaprSecurityException e) {
      throw new CLIProcessingException(e);
    } catch (Exception e) {
      out.addError(new OutputError(Errno.ERPCFAILED, "RPC Error"));
      return output;
    }
    if (data == null) {
      out.addError(new OutputError(Errno.ENODATA, "Empty RPC Response"));
      output.setOutput(out);
      return output;
    }

    /* 3. Check for the status in the response and return in case of errors */
    GetClusterAcesResponse resp = null;
    try {
      resp = GetClusterAcesResponse.parseFrom(data);
      if (resp.getStatus() != 0) {
        String errorMsg = resp.hasErrorString() ? resp.getErrorString() :
          "Non Zero Status While Setting Cluster Aces";
        out.addError(new OutputError(resp.getStatus(), errorMsg));
        return output;
      }
    }
    catch (InvalidProtocolBufferException e) {
      throw new CLIProcessingException("InvalidProtocolBufferException", e);
    }

    /* 4. Format the Cluster Aces and populate them in CommandOutput */
    formatClusterAces(resp, out);

    return output;
  }

  private void formatClusterAces(final GetClusterAcesResponse resp,
      final OutputHierarchy out) {
    if (resp.hasAces() == false) {
      /* This should not happen as there are always the default Cluster Aces */
      return;
    }
    OutputNode clusterAcesNode = new OutputNode("clusterAces");
    try {
      for (ClusterAceEntry aceEntry : resp.getAces().getAcesList()) {
        clusterAcesNode.addNode(new OutputNode(
              aceEntry.getClusterAction().name(),
              AceHelper.toInfix(aceEntry.getExpr().toStringUtf8())));
      }
    }
    catch (IOException e) {
      LOG.error("Unable to parse cluster aces");
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
      return;
    }
    out.addNode(clusterAcesNode);
  }

  private CommandOutput processSetAceCommand() throws CLIProcessingException {
    final OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput() {{
      setOutput(out);
    }};

    SetClusterAcesRequest.Builder reqBuilder = SetClusterAcesRequest.newBuilder();

    /* 1. Parse the input and construct the request */
    ClusterAces clusterAces = null;
    try {
      clusterAces = buildClusterAcesProtobuf(clusterAcesParameterMap);
    }
    catch (IOException e) {
      LOG.error("Unable to parse volume ace arguments");
      out.addError(new OutputError(Errno.EINVAL, "Incorrect expression of Volume Ace(s)"));
      return output;
    }
    if (clusterAces == null) {
      out.addError(new OutputError(Errno.EINVAL, "No Cluster Aces specified in the Command"));
      return output;
    }

    reqBuilder.setAces(clusterAces);
    reqBuilder.setCreds(getUserCredentials());

    /* 2. Issue the RPC to set the cluster ACEs */
    byte[] data;
    try {
      if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(
            getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0),
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProg.SetClusterAcesProc.getNumber(),
            reqBuilder.build(),
            SetClusterAcesResponse.class);
      }
      else {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProg.SetClusterAcesProc.getNumber(),
            reqBuilder.build(),
            SetClusterAcesResponse.class);
      }
    } catch (InvalidProtocolBufferException | MaprSecurityException e) {
      throw new CLIProcessingException(e);
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, "Error Setting Cluster ACEs"));
      return output;
    }
    if (data == null) {
      out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
      output.setOutput(out);
      return output;
    }

    /* 3. Parse the response from the CLDB */
    try {
      SetClusterAcesResponse resp = SetClusterAcesResponse.parseFrom(data);
      int status = resp.getStatus();
      if (status != 0) {
        String errorMsg = resp.hasErrorString() ? resp.getErrorString() :
          "Non Zero Status While Setting Cluster Aces";
        out.addError(new OutputError(status, errorMsg));
        return output;
      }
    }
    catch (InvalidProtocolBufferException e) {
      throw new CLIProcessingException("InvalidProtocolBufferException", e);
    }

    /* Everything is good. Return an empty Output */
    return output;
  }

  /**
   * Constructs the cluster aces protobuf from the 'maprcli' command.
   *
   * @param volumeAceParameterMap
   * @return
   * @throws IOException If any of the cluster aces argument is incorrectly specified
   */
  private ClusterAces buildClusterAcesProtobuf(
      final Map<String, ClusterActions> clusterAcesParameterMap)
      throws IOException, CLIProcessingException {

    ClusterAces.Builder acesBuilder = ClusterAces.newBuilder();
    for (String param : clusterAcesParameterMap.keySet()) {
      if (isParamPresent(param)) {
        String arg = getParamTextValue(param, 0);
        acesBuilder.addAces(ClusterAceEntry.newBuilder()
            .setClusterAction(clusterAcesParameterMap.get(param))
            .setExpr(ByteString.copyFromUtf8(AceHelper.toPostfix(arg)))
            .build());
      }
    }
    return ((acesBuilder.getAcesCount() > 0) ? acesBuilder.build() : null);
  }
}
