  /* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import java.util.ArrayList;
import java.util.Map;

import org.apache.log4j.Logger;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.MessageLite;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.baseutils.Errno;
import com.mapr.cli.common.ListCommand;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.TextCommandOutput;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import com.mapr.cliframework.base.inputparams.IntegerInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.cliframework.util.FieldInfo;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.ContainerInfo;
import com.mapr.fs.cldb.proto.CLDBProto.ContainerLookupRequest;
import com.mapr.fs.cldb.proto.CLDBProto.ContainerLookupResponse;
import com.mapr.fs.cldb.proto.CLDBProto.NodeInfo;
import com.mapr.fs.proto.Common;
import com.mapr.security.MaprSecurityException;

public class CheckContainerRepl extends CLIBaseClass implements CLIInterface {

     /* Define all the parameters used in Registry */
     public static final String CONTAINERS_PARAM_NAME = "containers";     
     private static final int ERR_CODE_VOLUMES_LOSS=150;
     private static final int ERR_CODE_METADATA_LOSS=151;
     
     private static final int MAX_CONTAINERS_PER_LOOKUP = 100;

     /* Define sub command */
     static final CLICommand checkContainerReplCmd = new CLICommand(
        "checkcontainerrepl",
        "",
        CheckContainerRepl.class, 
        ExecutionTypeEnum.NATIVE,
        /* Add parameters in a hash map key is string,  value is BaseInputParameter */
        new ImmutableMap.Builder<String, BaseInputParameter>()
             /* parameter 1 */
             .put(MapRCliUtil.CLUSTER_NAME_PARAM,
                new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
                     "cluster name",
                     CLIBaseClass.NOT_REQUIRED,
                     null))
             .put(CheckContainerRepl.CONTAINERS_PARAM_NAME,
                 new TextInputParameter(CheckContainerRepl.CONTAINERS_PARAM_NAME,
                     "<comma-separated list of containers>",
                     CLIBaseClass.REQUIRED,
                     null))
             .build(),
         null        /* sub-commands */
      ).setUsageInVisible(true)       
       .setShortUsage("checkcontainerrepl -cluster <clustername> -containers <comma-separated list of containers>");
     
     public CheckContainerRepl(ProcessedInput input, CLICommand cliCommand) {
         super(input, cliCommand);
     }
     
     private boolean containersLost(OutputHierarchy out, ContainerLookupRequest.Builder req) throws CLIProcessingException {
    	 
    	 try {
     	  byte [] data = null;
     	 
      	  if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
            String clusterName = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0);
            if (!CLDBRpcCommonUtils.getInstance().isValidClusterName(clusterName)) { 	               
              String msg = "Invalid cluster name " + clusterName;           
              out.addError(new OutputError(Errno.EUCLUSTER, msg));
              return true;
            }
            data = CLDBRpcCommonUtils.getInstance().sendRequest(clusterName,
              Common.MapRProgramId.CldbProgramId.getNumber(),
              CLDBProto.CLDBProg.ContainerLookupProc.getNumber(),
              req.build(), ContainerLookupResponse.class);
         } else {
         	data = CLDBRpcCommonUtils.getInstance().sendRequest(
         			Common.MapRProgramId.CldbProgramId.getNumber(),
               CLDBProto.CLDBProg.ContainerLookupProc.getNumber(),
               req.build(), ContainerLookupResponse.class);
         }          
                  
         if (data == null) {
           /**
             * <MAPR_ERROR>
	            * Message:Exception while processing RPC
             * Function:VolumeCommands.volumeUnMount()
             * Meaning:An error occurred.
             * Resolution:Contact technical support.
             * </MAPR_ERROR>
             */
           out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
           return true;
         }
         
         ContainerLookupResponse resp = ContainerLookupResponse.parseFrom(data);
         for (ContainerInfo cinfo: resp.getContainersList()) {
         	 if (cinfo.getAServersCount() <= 1) {
         		 out.addNode(new OutputNode(
         				 "One or more volumes will be lost due to under replication of containers"));
         		 return true;
         	 }
         }
         
      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
      } catch (Exception e) {
        /**
         * <MAPR_ERROR>
         * Message:Exception while sending RPC to cluster <cluster>
         * Function:VolumeCommands.volumeUnMount()
         * Meaning:An error occurred.
         * Resolution:Contact technical support.
         * </MAPR_ERROR>
         */
         out.addError(new OutputError(Errno.EOPFAILED, 
         		"Exception while running command " + e.getMessage()));
         return true;
      }
      
      return false;
     }
     
     private void containersLost(OutputHierarchy out, String [] containers) throws CLIProcessingException {
    	 
	       ContainerLookupRequest.Builder req = 
	      	 ContainerLookupRequest.newBuilder().setCreds(getUserCredentials());
    	                     
         for (int i = 0; i < containers.length;) {
        	 req.addContainerId(Integer.parseInt(containers[i]));
        	 i++;
        	 if (i % MAX_CONTAINERS_PER_LOOKUP != 0)
        		 continue;
        	 
        	 // Do lookup for every 100 containers
        	 if (containersLost(out, req))
        		 return;
           
           // Clear containers and proceed with the next set
       		 req.clearContainerId();
       	 }
         
         // Any remaining containers
         if (req.getContainerIdCount() > 0)
        	 containersLost(out, req);
     }
     
     /* This is a function that does the real work. */
     @Override
     public CommandOutput executeRealCommand() throws CLIProcessingException {        
         OutputHierarchy out = new OutputHierarchy();  /* Root of comamnd output's hierarchy tree */
         CommandOutput output = new CommandOutput();
         output.setOutput(out);
         
         String cmd = cliCommand.getCommandName();
         if (cmd.equalsIgnoreCase("checkcontainerrepl")) {
           String containers[] = getParamTextValue(CONTAINERS_PARAM_NAME, 0).split(",");
           if (containers == null || containers.length < 1) {
          	 out.addError(new OutputError(Errno.EINVAL, "Invalid list of containers"));
          	 return output;
           }
           containersLost(out, containers);
         }
         
         return output;         
     }
}
