package com.mapr.cli;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;

import java.net.InetAddress;

import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import org.apache.log4j.Logger;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.InvalidProtocolBufferException;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandLineOutput;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.TextCommandOutput;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.NoValueInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.fs.cldb.proto.CLDBProto.CLDBProg;
import com.mapr.fs.cldb.proto.CLDBProto.ClusterActions;
import com.mapr.fs.cldb.proto.CLDBProto.SecureObjectType;
import com.mapr.fs.cldb.proto.CLDBProto.BlacklistAddRequest;
import com.mapr.fs.cldb.proto.CLDBProto.BlacklistAddResponse;
import com.mapr.fs.cldb.proto.CLDBProto.BlacklistListRequest;
import com.mapr.fs.cldb.proto.CLDBProto.BlacklistListResponse;
import com.mapr.fs.cldb.proto.CLDBProto.SecurityModifyAclRequest;
import com.mapr.fs.cldb.proto.CLDBProto.SecurityModifyAclResponse;
import com.mapr.fs.proto.Common.VolumeActions;
import com.mapr.fs.cldb.proto.CLDBProto.BlacklistedAeMsg;
import com.mapr.fs.cldb.security.ACL;
import com.mapr.baseutils.acls.SecurityCommandHelper;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Security.AccessControlList;
import com.mapr.fs.proto.Security.AclEntry;
import com.mapr.fs.proto.Security.CredentialsMsg;
import com.mapr.fs.proto.Security.SecurityPrincipal;
import com.mapr.security.UnixUserGroupHelper;
import com.mapr.security.UserInformation;
import com.mapr.security.MaprSecurityException;

public class BlacklistCommands extends CLIBaseClass implements CLIInterface {
  // commands
  private static final String USER_BLACKLIST_CMD = "user";
  private static final String LIST_USERS_BLACKLIST_CMD = "listusers";
  //private static final String LIST_ALL_BLACKLIST_CMD = "listall";
  
  private static final String TYPE_PARAM_NAME = "type";
  private static final String CLUSTER_PARAM_NAME = "cluster";
  private static final String NAME_PARAM = "name";
  
  private static final Logger LOG = Logger.getLogger(BlacklistCommands.class);
 
  private UnixUserGroupHelper userInfo;
 
  static final CLICommand userBlacklistCmd = new CLICommand(
      USER_BLACKLIST_CMD,
      "Adds the entity to blacklist",
      BlacklistCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .put(BlacklistCommands.NAME_PARAM,
            new TextInputParameter(BlacklistCommands.NAME_PARAM,
                "username to be blacklisted",
                CLIBaseClass.REQUIRED,
                null))
        .put(BlacklistCommands.CLUSTER_PARAM_NAME,
            new TextInputParameter(BlacklistCommands.CLUSTER_PARAM_NAME,
                "name of the cluster",
                CLIBaseClass.NOT_REQUIRED,
                null))
        .build(),
      null)
    .setShortUsage("user -name usernameToBlacklist -cluster cluster name");
  
  static final CLICommand listUsersBlacklistCmd = new CLICommand(
      LIST_USERS_BLACKLIST_CMD,
      "lists blacklisted users",
      BlacklistCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .put(BlacklistCommands.CLUSTER_PARAM_NAME,
            new TextInputParameter(BlacklistCommands.CLUSTER_PARAM_NAME,
                "name of the cluster",
                CLIBaseClass.NOT_REQUIRED,
                null))
        .build(),
        null)
  .setShortUsage("listusers -cluster clustername");
  
  /*static final CLICommand listAllBlacklistCmd = new CLICommand(
      "listall",
      "listall blacklisted entities",
      BlacklistCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .put(BlacklistCommands.CLUSTER_PARAM_NAME,
            new TextInputParameter(BlacklistCommands.CLUSTER_PARAM_NAME,
                "name of the cluster",
                CLIBaseClass.NOT_REQUIRED,
                null))
        .build(),
        null)
  .setShortUsage("listall -cluster clustername");*/
  
  static final CLICommand[] blacklistSubCommands = {
    userBlacklistCmd,
    listUsersBlacklistCmd,
    //listAllBlacklistCmd
  };
  
  static final String usageStr = "blacklist [user|listusers]";
  public static final CLICommand blacklistCommands = 
    new CLICommand(
        "blacklist",
        "usage: " + usageStr,
        CLIUsageOnlyCommand.class,
        ExecutionTypeEnum.NATIVE,
        blacklistSubCommands)
    .setShortUsage(usageStr);

  public BlacklistCommands(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }
  
  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    if (!super.validateInput()) {
      CommandOutput output = new CommandOutput();
      OutputHierarchy out = new OutputHierarchy();
      output.setOutput(out);
      return output;
    }

    String command = cliCommand.getCommandName();
    if (command.equalsIgnoreCase("user")) {
      return addUserToBlacklist();
    }
    
    if (command.equalsIgnoreCase("listusers")) {
      return listBlacklistedUsers();
    }
    
    /*if (command.equalsIgnoreCase("listall")) {
      return listAllBlacklistEntities();
    }*/
    
    return new TextCommandOutput(("Blacklist command failed: unknown command " + 
        command + " received.").getBytes());
  }
 
  private CommandOutput addUserToBlacklist() throws 
                CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();

    try {
      BlacklistAddRequest.Builder addRequestBuilder = 
           BlacklistAddRequest.newBuilder();

      CredentialsMsg creds = getUserCredentials();
      addRequestBuilder.setCreds(creds);

      // set the username to be blackliseted
      String username = getParamTextValue(NAME_PARAM, 0);
      addRequestBuilder.setName(username);
      addRequestBuilder.setType(BlacklistAddRequest.EntityType.USER);
    
      // send the request to respective cluster
      byte[] data = null;
      if (isParamPresent(CLUSTER_PARAM_NAME)) {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(
                      getParamTextValue(CLUSTER_PARAM_NAME, 0),
                      Common.MapRProgramId.CldbProgramId.getNumber(),
                      CLDBProg.BlacklistAddProc.getNumber(),
                      addRequestBuilder.build(),
                      BlacklistAddResponse.class);
      } else {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(
                      Common.MapRProgramId.CldbProgramId.getNumber(),
                      CLDBProg.BlacklistAddProc.getNumber(),
                      addRequestBuilder.build(),
                      BlacklistAddResponse.class);
      }

      if (data == null) {
        out.addError(new OutputError(Errno.ERPCFAILED, "Could not connect to CLDB service"));
        output.setOutput(out);
        return output;
      }

      BlacklistAddResponse resp = BlacklistAddResponse.parseFrom(data);

      int status = resp.getStatus();
      if (status != 0) {
        /**
         * <MAPR_ERROR>                                                              
         * Message:Error while trying to blacklist user                                   
         * Function:BlacklistCommands.addToBlacklist()                                              
         * Meaning:An error occurred.                                                  
         * Resolution:Check the command syntax and try again.                          
         * </MAPR_ERROR>
         */
        out.addError(new OutputError(Errno.EOPFAILED,                             
          resp.getErrMsg()));               
        output.setOutput(out);
        return output;                                                              
      } 
  
    } catch (InvalidProtocolBufferException e) {                                    
      throw new CLIProcessingException(                                           
          "InvalidProtocolBufferException " + "Exception", e);                    
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {                                                         
      /**                                                                           
       * <MAPR_ERROR>                                                                
       * Message:Error while trying to blacklist user                                   
       * Function:BlacklistCommands.addToBlacklist()                                              
       * Meaning:An error occurred.                                                  
       * Resolution:Check the command syntax and try again.                          
       * </MAPR_ERROR>                                                               
       */                                                                            
                                                                                  
      out.addError(new OutputError(Errno.EOPFAILED,                                 
        "Error while trying to blacklist a user"));                                       
        output.setOutput(out);                                                      
        return output;                                                              
    }
  
    // TODO for node and service blacklist operations
    return output;
  }
   
  private CommandOutput listBlacklistedUsers() throws 
                CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    userInfo = new UnixUserGroupHelper();

    try {
      BlacklistListRequest.Builder listRequestBuilder =
           BlacklistListRequest.newBuilder();

      CredentialsMsg creds = getUserCredentials();
      listRequestBuilder.setCreds(creds);

      listRequestBuilder.setType(BlacklistListRequest.EntityType.USER);

      // send the request to respective cluster
      byte[] data = null;
      if (isParamPresent(CLUSTER_PARAM_NAME)) {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(
                      getParamTextValue(CLUSTER_PARAM_NAME, 0),
                      Common.MapRProgramId.CldbProgramId.getNumber(),
                      CLDBProg.BlacklistListProc.getNumber(),
                      listRequestBuilder.build(),
                      BlacklistListResponse.class);
      } else {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(
                      Common.MapRProgramId.CldbProgramId.getNumber(),
                      CLDBProg.BlacklistListProc.getNumber(),
                      listRequestBuilder.build(),
                      BlacklistListResponse.class);
      }

      if (data == null) {
        out.addError(new OutputError(Errno.ERPCFAILED, "Could not connect to CLDB service"));
        output.setOutput(out);
        return output;
      }

      BlacklistListResponse resp = BlacklistListResponse.parseFrom(data);

      int status = resp.getStatus();
      if (status != 0) {
        /**
         * <MAPR_ERROR>                                                              
         * Message:Error while trying to blacklist user                                   
         * Function:BlacklistCommands.addToBlacklist()                                              
         * Meaning:An error occurred.                                                  
         * Resolution:Check the command syntax and try again.                          
         * </MAPR_ERROR>
         */
        out.addError(new OutputError(Errno.EOPFAILED,
          resp.getErrMsg()));
        output.setOutput(out);
        return output;
      }

      String[] headers = {"Username","BlacklistTime"};
      output.setNodeOrder(headers);

      int blacklistedAeCount = resp.getBlacklistedAesCount();
      OutputNode node = null;

      for(int index = 0; index < blacklistedAeCount; index++) {
        node = new OutputNode();;
        BlacklistedAeMsg entry = resp.getBlacklistedAes(index);
        try {
          node.addNode(new OutputNode("Username",
            userInfo.getUsername(entry.getId())));
        } catch (SecurityException ignore) {
          // Using unknown UID as user id is not known.
          node.addNode(new OutputNode("Username",
            "Unknown user id " + entry.getId()));
        }
        node.addNode(new OutputNode("BlacklistTime",
          new Date(entry.getBlacklistTime())));
        out.addNode(node);
      }
      
      output.setOutput(out);

    } catch (InvalidProtocolBufferException e) {
      throw new CLIProcessingException(
          "InvalidProtocolBufferException " + "Exception", e);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      /**                                                                           
       * <MAPR_ERROR>                                                                
       * Message:Error while trying to blacklist user                                   
       * Function:BlacklistCommands.addToBlacklist()                                              
       * Meaning:An error occurred.                                                  
       * Resolution:Check the command syntax and try again.                          
       * </MAPR_ERROR>                                                               
       */

      out.addError(new OutputError(Errno.EOPFAILED,
        e.getMessage()));
        output.setOutput(out);
        return output;
    }

    // TODO for node and service blacklist operations
    return output;
  }

  /*private CommandOutput listAllBlacklistEntities() throws 
                CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    out.addError(new OutputError(Errno.EOPFAILED,
                   "Opertion not implemented"));
    return output;
  }*/
   
  @Override
  public String getCommandUsage() {
    return usageStr;
  }
}
