package com.mapr.cli;

import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.log4j.Logger;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.InvalidProtocolBufferException;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandLineOutput;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.TextCommandOutput;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.NoValueInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import com.mapr.cliframework.base.inputparams.IntegerInputParameter;
import com.mapr.cliframework.base.inputparams.LongInputParameter;
import com.mapr.fs.proto.Common;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.CLDBConfigParams;
import com.mapr.fs.cldb.proto.CLDBProto.CLDBConfigResponse;
import com.mapr.fs.cldb.proto.CLDBProto.CLDBConfigParams.*;
import com.mapr.fs.cldb.proto.CLDBProto.CLDBConfigRequest;
import com.mapr.fs.cldb.proto.CLDBProto.ClusterActions;
import com.mapr.fs.cldb.proto.CLDBProto.SecureObjectType;
import com.mapr.fs.proto.Security.CredentialsMsg;
import com.mapr.fs.proto.Security.SecurityPrincipal;
import com.mapr.security.MaprSecurityException;

public class AuditCommands extends CLIBaseClass implements CLIInterface {
  // commands
  private static final String AUDIT_DATA = "data";
  private static final String AUDIT_CLUSTER = "cluster";
  private static final String AUDIT_INFO = "info";

  private static final String ENABLED_PARAM_NAME = "enabled";
  private static final String MAXSIZE_PARAM_NAME = "maxsize";
  private static final String RETENTION_PARAM_NAME = "retention";
  private static final String REPLICAS_PARAM_NAME = "replicas";
  private static final String CLUSTER_PARAM_NAME = "cluster";

  private static final Logger LOG = Logger.getLogger(AuditCommands.class);

  // Audit related constants
  private final String ParamAuditDataAccess =
    "mapr.audit.data.access";
  private final String ParamAuditClusterMgmtOps =
    "mapr.audit.cluster.mgmt.ops";
  private final String ParamAuditDataRetentionPeriod =
    "mapr.audit.data.retention.days";
  private final String ParamAuditClusterRetentionPeriod =
    "mapr.audit.cluster.retention.days";
  private final String ParamAuditDataVolumeSize =
    "mapr.audit.data.volume.size.mb";
  private final String ParamAuditClusterVolumeSize =
    "mapr.audit.cluster.volume.size.mb";

  private static Map<String, BaseInputParameter> baseParams = 
    new ImmutableMap.Builder<String, BaseInputParameter>()
    .put(MapRCliUtil.CLUSTER_NAME_PARAM, new TextInputParameter(
        MapRCliUtil.CLUSTER_NAME_PARAM, "cluster_name",
         CLIBaseClass.NOT_REQUIRED, null))    
    .build();

  static final CLICommand dataAuditCmd = new CLICommand(
      AUDIT_DATA,
      "audit data access operations",
      AuditCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(baseParams)
        .put(AuditCommands.ENABLED_PARAM_NAME,
            new BooleanInputParameter(AuditCommands.ENABLED_PARAM_NAME,
                "enable audit for data access",
                CLIBaseClass.NOT_REQUIRED,
                null))
        .put(AuditCommands.MAXSIZE_PARAM_NAME,
            new IntegerInputParameter(AuditCommands.MAXSIZE_PARAM_NAME,
                "size of audit volume per node",
                CLIBaseClass.NOT_REQUIRED,
                null))
        .put(AuditCommands.RETENTION_PARAM_NAME,
            new IntegerInputParameter(AuditCommands.RETENTION_PARAM_NAME,
                "retention period of audit logs in days",
                CLIBaseClass.NOT_REQUIRED,
                null))
        .put(AuditCommands.REPLICAS_PARAM_NAME,
            new IntegerInputParameter(AuditCommands.REPLICAS_PARAM_NAME,
                "number of replicas for audit data",
                CLIBaseClass.NOT_REQUIRED,
                null).setInvisible(true))
        .build(),
      null)
    .setShortUsage("data -enabled <true|false> -maxsize <GB>  -retention <days>");

  static final CLICommand clusterAuditCmd = new CLICommand(
      AUDIT_CLUSTER,
      "audit cluster management operations",
      AuditCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .put(AuditCommands.ENABLED_PARAM_NAME,
            new BooleanInputParameter(AuditCommands.ENABLED_PARAM_NAME,
                "enable audit for cluster mgmt ops",
                CLIBaseClass.NOT_REQUIRED,
                null))
        /*
        .put(AuditCommands.MAXSIZE_PARAM_NAME,
            new IntegerInputParameter(AuditCommands.MAXSIZE_PARAM_NAME,
                "size of audit volume per node",
                CLIBaseClass.NOT_REQUIRED,
                null))
        .put(AuditCommands.RETENTION_PARAM_NAME,
            new IntegerInputParameter(AuditCommands.RETENTION_PARAM_NAME,
                "retention period of audit logs in days",
                CLIBaseClass.NOT_REQUIRED,
                null))
        .put(AuditCommands.REPLICAS_PARAM_NAME,
            new IntegerInputParameter(AuditCommands.REPLICAS_PARAM_NAME,
                "number of replicas for audit data",
                CLIBaseClass.NOT_REQUIRED,
                null).setInvisible(true))
                */
        .build(),
      null)
    .setShortUsage("cluster -enabled <true|false> -maxsize <GB>  -retention <days> -replicas <count>");

  static final CLICommand auditInfoCmd = new CLICommand(
      AUDIT_INFO,
      "information about audit params",
      AuditCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(baseParams)
        .build(),
      null)
    .setShortUsage("info");

  static final CLICommand[] auditSubCommands = {
    dataAuditCmd,
    clusterAuditCmd,
    auditInfoCmd
  };

  static final String usageStr = "audit [data|cluster|info]";
  public static final CLICommand auditCommands = 
    new CLICommand(
        "audit",
        "usage: " + usageStr,
        CLIUsageOnlyCommand.class,
        ExecutionTypeEnum.NATIVE,
        auditSubCommands)
    .setShortUsage(usageStr);

  public AuditCommands(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }

  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    if (!super.validateInput()) {
      CommandOutput output = new CommandOutput();
      OutputHierarchy out = new OutputHierarchy();
      output.setOutput(out);
      return output;
    }

    String command = cliCommand.getCommandName();
    if (command.equalsIgnoreCase("data")) {
      return auditDataAccess();
    }

    if (command.equalsIgnoreCase("cluster")) {
      return auditClusterMgmt();
    }

    if (command.equalsIgnoreCase("info")) {
      return auditInfo();
    }

    return new TextCommandOutput(("Audit command failed: unknown command " +
        command + " received.").getBytes());
  }

  private CommandOutput auditInfo() throws
                CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);

    Properties cldbProps = new Properties();
    
    String cluster = null;
    if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
      cluster = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0);
      if (!CLDBRpcCommonUtils.getInstance().isValidClusterName(cluster)) {
        out.addError(new OutputError(Errno.EUCLUSTER, "Invalid cluster: " + cluster));
        return output;
      }
    }

    int status = 
      MapRCliUtil.fetchCLDBParams(cluster, cldbProps, getUserCredentials());
    if (status < 0) {
      out.addError(new OutputError(Errno.EOPFAILED,
                                   "Unable to reach CLDB"));
      return output;
    } else if (status != 0) {
      out.addError(new OutputError(Errno.EOPFAILED,
                                   "Failed to load configuration - " + Errno.toString(status)));
      return output;
    }

    OutputNode finalNode = new OutputNode();

    String value;
    String emptyValue = "-";
    // Data access related params
    OutputNode dNode = new OutputNode("data");
    
    value = cldbProps.getProperty(ParamAuditDataAccess);
    if (value == null)
      value = emptyValue;
    dNode.addChild(new OutputNode("enabled", value));
   
    value = cldbProps.getProperty(ParamAuditDataVolumeSize);
    if (value == null) {
     value = "-";
    } else {
      try {
        int sizeInMB = Integer.parseInt(value);
        value = Integer.toString(sizeInMB / 1024);
      } catch (NumberFormatException e) {
        out.addError(new OutputError(Errno.EINVAL,
               "Invalid value for max size for data access logs"));
      }
    }
    dNode.addChild(new OutputNode("maxSizeGB", value));
    
    value = cldbProps.getProperty(ParamAuditDataRetentionPeriod);
    if (value == null)
      value = "-";
    dNode.addChild(new OutputNode("retentionDays", value));
    finalNode.addChild(dNode);

    // Cluster mgmt related params
    OutputNode cNode = new OutputNode("cluster");
    
    value = cldbProps.getProperty(ParamAuditClusterMgmtOps);
    if (value == null)
      value = emptyValue;
    cNode.addChild(new OutputNode("enabled", value));
   
    value = cldbProps.getProperty(ParamAuditClusterVolumeSize);
    if (value == null) {
     value = "-";
    } else {
      try {
        int sizeInMB = Integer.parseInt(value);
        value = Integer.toString(sizeInMB / 1024);
      } catch (NumberFormatException e) {
        out.addError(new OutputError(Errno.EINVAL,
               "Invalid value for max size for data access logs"));
      }
    }
    cNode.addChild(new OutputNode("maxSizeGB", "NA")); // not supported yet
    
    value = cldbProps.getProperty(ParamAuditClusterRetentionPeriod);
    if (value == null)
      value = "-";
    cNode.addChild(new OutputNode("retentionDays", "NA")); // not supported yet
    finalNode.addChild(cNode);

    out.addNode(finalNode);

    return output;
  }

  private CommandOutput auditDataAccess() throws
                CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);

    String cluster = null;
    if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
      cluster = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0);
      if (!CLDBRpcCommonUtils.getInstance().isValidClusterName(cluster)) {
        out.addError(new OutputError(Errno.EUCLUSTER, "Invalid cluster: " + cluster));
        return output;
      }
    }

    try {
      CLDBConfigRequest.Builder configReqBuilder = CLDBConfigRequest.newBuilder();
      configReqBuilder.setLoad(false);

      CredentialsMsg creds = getUserCredentials();
      configReqBuilder.setCreds(creds);

      CLDBConfigParams.Builder pBuilder = CLDBConfigParams.newBuilder();

      if (isParamPresent(ENABLED_PARAM_NAME)) {
        boolean enableDataAudit = getParamBooleanValue(ENABLED_PARAM_NAME, 0);
        CLDBConfigParam param = CLDBConfigParam.newBuilder()
                                   .setKeys(ParamAuditDataAccess)
                                   .setValues(enableDataAudit ? "1" : "0")
                                   .build();
        pBuilder.addParams(param);
      }

      if (isParamPresent(MAXSIZE_PARAM_NAME)) {
        int maxSize = getParamIntValue(MAXSIZE_PARAM_NAME, 0) * 1024;
        CLDBConfigParam param = CLDBConfigParam.newBuilder()
                                   .setKeys(ParamAuditDataVolumeSize)
                                   .setValues(Integer.toString(maxSize))
                                   .build();
        pBuilder.addParams(param);
      }

      if (isParamPresent(RETENTION_PARAM_NAME)) {
        int retention = getParamIntValue(RETENTION_PARAM_NAME, 0);
        CLDBConfigParam param = CLDBConfigParam.newBuilder()
                                   .setKeys(ParamAuditDataRetentionPeriod)
                                   .setValues(Integer.toString(retention))
                                   .build();
        pBuilder.addParams(param);
      }
/*
      if (isParamPresent(REPLICAS_PARAM_NAME)) {
        int replicas = getParamIntValue(REPLICAS_PARAM_NAME, 0);
        CLDBConfigParam param = CLDBConfigParam.newBuilder()
                                   .setKeys()
                                   .setValues(replicas)
                                   .build();
        pBuilder.addParams(param);
        paramSpecified = true;
      }
*/
      if (pBuilder.getParamsCount() == 0) {
        out.addError(new OutputError(Errno.EINVAL,
             "Specify the param that needs change"));
        return output;
      }

      configReqBuilder.setParams(pBuilder);

      // send the request to respective cluster
      byte[] data = null;
      if (cluster != null && !cluster.isEmpty()) {
    	  data = CLDBRpcCommonUtils.getInstance().sendRequest(
                      cluster,
    			            Common.MapRProgramId.CldbProgramId.getNumber(),
    	                CLDBProto.CLDBProg.CLDBConfigProc.getNumber(),
                      configReqBuilder.build(), CLDBConfigResponse.class);
      } else {
    	  data = CLDBRpcCommonUtils.getInstance().sendRequest(
    			            Common.MapRProgramId.CldbProgramId.getNumber(),
    	                CLDBProto.CLDBProg.CLDBConfigProc.getNumber(),
                      configReqBuilder.build(), CLDBConfigResponse.class);
      }

      if (data == null) {
        out.addError(new OutputError(Errno.ERPCFAILED, "Could not connect to CLDB service"));
        return output;
      }

      CLDBConfigResponse resp = CLDBConfigResponse.parseFrom(data);

      int status = resp.getStatus();
      if (status != 0) {
        out.addError(new OutputError(status,
          (resp.hasErrorString() ? resp.getErrorString() :
          "Failed to save audit configuration : " + Errno.toString(status))));
        return output;
      }

    } catch (InvalidProtocolBufferException e) {
      throw new CLIProcessingException(
          "InvalidProtocolBufferException " + "Exception", e);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED,
        "Error while trying to save audit data config"));
    }

    return output;
  }

  private CommandOutput auditClusterMgmt() throws
                CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);

    try {
      CLDBConfigRequest.Builder configReqBuilder = CLDBConfigRequest.newBuilder();
      configReqBuilder.setLoad(false);

      CredentialsMsg creds = getUserCredentials();
      configReqBuilder.setCreds(creds);

      CLDBConfigParams.Builder pBuilder = CLDBConfigParams.newBuilder();

      if (isParamPresent(ENABLED_PARAM_NAME)) {
        boolean enableClusterAudit = getParamBooleanValue(ENABLED_PARAM_NAME, 0);
        CLDBConfigParam param = CLDBConfigParam.newBuilder()
                                   .setKeys(ParamAuditClusterMgmtOps)
                                   .setValues(enableClusterAudit ? "1" : "0")
                                   .build();
        pBuilder.addParams(param);
      }

/*
      if (isParamPresent(MAXSIZE_PARAM_NAME)) {
        int maxSize = getParamIntValue(MAXSIZE_PARAM_NAME, 0) * 1024;
        CLDBConfigParam param = CLDBConfigParam.newBuilder()
                                   .setKeys(ParamAuditClusterVolumeSize)
                                   .setValues(Integer.toString(maxSize))
                                   .build();
        pBuilder.addParams(param);
      }

      if (isParamPresent(RETENTION_PARAM_NAME)) {
        int retention = getParamIntValue(RETENTION_PARAM_NAME, 0);
        CLDBConfigParam param = CLDBConfigParam.newBuilder()
                                   .setKeys(ParamAuditClusterRetentionPeriod)
                                   .setValues(Integer.toString(retention))
                                   .build();
        pBuilder.addParams(param);
      }
      if (isParamPresent(REPLICAS_PARAM_NAME)) {
        int replicas = getParamIntValue(REPLICAS_PARAM_NAME, 0);
        CLDBConfigParam param = CLDBConfigParam.newBuilder()
                                   .setKeys()
                                   .setValues(replicas)
                                   .build();
        pBuilder.addParams(param);
        paramSpecified = true;
      }
*/
      if (pBuilder.getParamsCount() == 0) {
        out.addError(new OutputError(Errno.EINVAL,
             "Specify the param that needs to be changed"));
        return output;
      }

      configReqBuilder.setParams(pBuilder);

      // send the request to respective cluster
      byte[] data = null;
/*      if (cluster != null && !cluster.isEmpty()) {
    	  data = CLDBRpcCommonUtils.getInstance().sendRequest(
    			            cluster,
                      Common.MapRProgramId.CldbProgramId.getNumber(),
    	                CLDBProto.CLDBProg.CLDBConfigProc.getNumber(),
                      configReqBuilder.build(), CLDBConfigResponse.class);
      } else {*/
    	  data = CLDBRpcCommonUtils.getInstance().sendRequest(
                      Common.MapRProgramId.CldbProgramId.getNumber(),
    	                CLDBProto.CLDBProg.CLDBConfigProc.getNumber(),
                      configReqBuilder.build(), CLDBConfigResponse.class);
//      }

      if (data == null) {
        out.addError(new OutputError(Errno.ERPCFAILED, "Could not connect to CLDB service"));
        return output;
      }

      CLDBConfigResponse resp = CLDBConfigResponse.parseFrom(data);

      int status = resp.getStatus();
      if (status != 0) {
        out.addError(new OutputError(status,
          (resp.hasErrorString() ? resp.getErrorString() :
          "Failed to save audit configuration : " + Errno.toString(status))));
        return output;
      }

    } catch (InvalidProtocolBufferException e) {
      throw new CLIProcessingException(
          "InvalidProtocolBufferException " + "Exception", e);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED,
        "Error while trying to save audit cluster config"));
    }

    return output;
  }

  @Override
  public String getCommandUsage() {
    return usageStr;
  }
}
