/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;


import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import com.mapr.cli.common.NodeField;
import com.mapr.cli.common.NodesCommonUtils;
import org.apache.log4j.Logger;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.MessageLite;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.baseutils.acls.SecurityCommandHelper;
import com.mapr.cli.common.AuthManager;
import com.mapr.cli.common.ListCommand;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import com.mapr.cliframework.base.inputparams.IntegerInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.cliframework.util.FieldInfo;
import com.mapr.fs.cldb.security.ACL;
import com.mapr.fs.cldb.proto.CLDBProto.CLDBProg;
import com.mapr.fs.cldb.proto.CLDBProto.SecureObjectType;
import com.mapr.fs.cldb.proto.CLDBProto.AlarmLookupRequest;
import com.mapr.fs.cldb.proto.CLDBProto.AlarmLookupResponse;
import com.mapr.fs.cldb.proto.CLDBProto.AlarmUpdateRequest;
import com.mapr.fs.cldb.proto.CLDBProto.AlarmUpdateResponse;
import com.mapr.fs.cldb.proto.Accesscontrol.ClientAuthorizationRequest;
import com.mapr.fs.cldb.proto.Accesscontrol.ClientAuthorizationResponse;
import com.mapr.fs.cldb.proto.CLDBProto.AlarmAddRequest;
import com.mapr.fs.cldb.proto.CLDBProto.AlarmAddResponse;
import com.mapr.fs.cldb.proto.CLDBProto.AlarmDeleteRequest;
import com.mapr.fs.cldb.proto.CLDBProto.AlarmDeleteResponse;
import com.mapr.fs.cldb.proto.CLDBProto.AlarmEditRequest;
import com.mapr.fs.cldb.proto.CLDBProto.AlarmEditResponse;
import com.mapr.fs.cldb.proto.CLDBProto.SecurityGetAclRequest;
import com.mapr.fs.cldb.proto.CLDBProto.SecurityGetAclResponse;
import com.mapr.fs.MapRFileSystem;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Common.AlarmConfigMsg;
import com.mapr.fs.proto.Common.AlarmId;
import com.mapr.fs.proto.Common.AlarmMsg;
import com.mapr.fs.proto.Common.AlarmType;
import com.mapr.security.MaprSecurityException;
import com.mapr.fs.proto.Common.PluggableAlarm;
import com.mapr.fs.proto.Security.CredentialsMsg;
import com.mapr.fs.proto.Security.AccessControlList;
import com.mapr.cli.common.PluggableAlarmUtil;

public class AlarmCommands extends ListCommand implements CLIInterface {

  private static final int NUM_ALARMS_PER_RPC = 100;

  public static final String ALARM_NAME_PARAM_NAME = "alarm";
  public static final String ALARM_NEWNAME_PARAM_NAME = "newname";
  public static final String ALARM_TERSE_NAME_PARAM_NAME = "terse";
  public static final String ALARM_TYPE_PARAM_NAME = "type";
  public static final String ALARM_DISPLAY_NAME_PARAM_NAME = "displayName";
  public static final String ALARM_SERVICE_DISPLAY_NAME = "serviceName";
  public static final String ALARM_SERVICE_KEY_NAME = "service";
  public static final String ALARM_BASE_SERVICE_NAME = "baseService";
  public static final String ALARM_UI_DISPLAY_NAME = "uiDisplayName";
  public static final String ALARM_TOOLTIP_NAME = "tooltip";

  public static final String SUMMARY_PARAM_NAME = "summary";
  public static final String ENTITY_PARAM_NAME = "entity";
  public static final String VALUES_PARAM_NAME = "values";
  public static final String DESC_PARAM_NAME = "description";

  public static final String START_PARAM_NAME = "start";
  public static final String LIMIT_PARAM_NAME = "limit";
  public static final String FILTER_PARAM_NAME = "filter";
  public static final String OUTPUT_PARAM_NAME = "output";


  private static final Logger LOG = Logger.getLogger(AlarmCommands.class);

  private AuthManager authManager;

  public AlarmCommands(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
    authManager = AuthManager.getInstance();
  }

  public static final Map<String, AlarmType> alarmNameMap =
      new ImmutableMap.Builder<String, Common.AlarmType>()
      .put("cluster", AlarmType.CLUSTER_ALARM)
      .put("node", AlarmType.NODE_ALARM).put("volume", AlarmType.VOLUME_ALARM)
      .put("ae", AlarmType.AE_ALARM).build();

  public static Map<String, BaseInputParameter> baseParams =
    new ImmutableMap.Builder<String, BaseInputParameter>()
    .put(MapRCliUtil.CLUSTER_NAME_PARAM, new TextInputParameter(
        MapRCliUtil.CLUSTER_NAME_PARAM, "cluster_name",
         CLIBaseClass.NOT_REQUIRED, null))
    .build();


  static final CLICommand addCmd = new CLICommand(
      "add",
      "",
      AlarmCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(AlarmCommands.baseParams)
      .put(ALARM_NAME_PARAM_NAME,
          new TextInputParameter(ALARM_NAME_PARAM_NAME,
              "alarm name",
              CLIBaseClass.REQUIRED,
              null))
      .put(ALARM_TERSE_NAME_PARAM_NAME,
          new TextInputParameter(ALARM_TERSE_NAME_PARAM_NAME,
              "alarm terse name",
              CLIBaseClass.REQUIRED,
              null))
      .put(ALARM_DISPLAY_NAME_PARAM_NAME,
          new TextInputParameter(ALARM_DISPLAY_NAME_PARAM_NAME,
              "alarm display name",
              CLIBaseClass.NOT_REQUIRED,
              null))
      .put(ALARM_SERVICE_KEY_NAME,
          new TextInputParameter(ALARM_SERVICE_KEY_NAME,
              "alarm service key",
              CLIBaseClass.NOT_REQUIRED,
              null))
      .put(ALARM_SERVICE_DISPLAY_NAME,
          new TextInputParameter(ALARM_SERVICE_DISPLAY_NAME,
              "alarm service name",
              CLIBaseClass.NOT_REQUIRED,
              null))
          .put(ALARM_BASE_SERVICE_NAME,
          new BooleanInputParameter(ALARM_BASE_SERVICE_NAME,
              "alarm base service",
              CLIBaseClass.NOT_REQUIRED,
              0))
      .put(ALARM_UI_DISPLAY_NAME,
          new TextInputParameter(ALARM_UI_DISPLAY_NAME,
              "alarm ui display name",
              CLIBaseClass.NOT_REQUIRED,
              null))
      .put(ALARM_TOOLTIP_NAME,
          new TextInputParameter(ALARM_TOOLTIP_NAME,
              "alarm ui tooltip content",
              CLIBaseClass.NOT_REQUIRED,
              null))
          .build(),
      null
  ).setUsageInVisible(true);
  //).setShortUsage("add");

  // Unused at the moment
  static final CLICommand editCmd = new CLICommand(
      "edit",
      "",
      AlarmCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(AlarmCommands.baseParams)
      .put(ALARM_NAME_PARAM_NAME,
          new TextInputParameter(ALARM_NAME_PARAM_NAME,
              "alarm name",
              CLIBaseClass.REQUIRED,
              null))
      .put(ALARM_NEWNAME_PARAM_NAME,
          new TextInputParameter(ALARM_NEWNAME_PARAM_NAME,
              "new alarm name",
              CLIBaseClass.NOT_REQUIRED,
              null))
      .put(ALARM_TERSE_NAME_PARAM_NAME,
          new TextInputParameter(ALARM_TERSE_NAME_PARAM_NAME,
              "alarm terse name",
              CLIBaseClass.REQUIRED,
              null))
      .put(ALARM_DISPLAY_NAME_PARAM_NAME,
          new TextInputParameter(ALARM_DISPLAY_NAME_PARAM_NAME,
              "alarm display name",
              CLIBaseClass.NOT_REQUIRED,
              null))
      .build(),
      null
  ).setShortUsage("edit");


  static final CLICommand deleteCmd = new CLICommand(
      "delete",
      "",
      AlarmCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(AlarmCommands.baseParams)
      .put(ALARM_NAME_PARAM_NAME,
          new TextInputParameter(ALARM_NAME_PARAM_NAME,
              "alarm name",
              CLIBaseClass.REQUIRED,
              null))
      .build(),
      null
  //).setShortUsage("delete");
  ).setUsageInVisible(true);

  static final CLICommand viewCmd = new CLICommand(
      "view",
      "",
      AlarmCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(AlarmCommands.baseParams)
      .put(AlarmCommands.ALARM_NAME_PARAM_NAME,
          new TextInputParameter(AlarmCommands.ALARM_NAME_PARAM_NAME,
              "alarm name",
              CLIBaseClass.NOT_REQUIRED,
              ""))
      .put(AlarmCommands.START_PARAM_NAME,
          new IntegerInputParameter(AlarmCommands.START_PARAM_NAME,
              "start",
               CLIBaseClass.NOT_REQUIRED,
               0))
      .put(AlarmCommands.LIMIT_PARAM_NAME,
          new IntegerInputParameter(AlarmCommands.LIMIT_PARAM_NAME,
              "limit",
              CLIBaseClass.NOT_REQUIRED,
              Integer.MAX_VALUE))
      .put(AlarmCommands.OUTPUT_PARAM_NAME,
          new TextInputParameter(AlarmCommands.OUTPUT_PARAM_NAME,
              "output",
              CLIBaseClass.NOT_REQUIRED,
              "verbose"))
      .put(NodeServicesManagementCommand.ZK_CONNECTSTRING, new TextInputParameter(NodeServicesManagementCommand.ZK_CONNECTSTRING, "ZooKeeper Connect String: 'host:port,host:port,host:port,...'", CLIBaseClass.NOT_REQUIRED, null))
      .build(),
      null
  ).setUsageInVisible(true);
  //).setShortUsage("view");



  static final CLICommand listCmd = new CLICommand(
      "list",
      "",
      AlarmCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(AlarmCommands.baseParams)
      .put(AlarmCommands.SUMMARY_PARAM_NAME,
          new BooleanInputParameter(AlarmCommands.SUMMARY_PARAM_NAME,
              "summary",
              CLIBaseClass.NOT_REQUIRED,
              0))
      .put(AlarmCommands.ALARM_NAME_PARAM_NAME,
          new TextInputParameter(AlarmCommands.ALARM_NAME_PARAM_NAME,
              "alarm name",
              CLIBaseClass.NOT_REQUIRED,
              null))
      .put(AlarmCommands.ALARM_TYPE_PARAM_NAME,
          new TextInputParameter(AlarmCommands.ALARM_TYPE_PARAM_NAME,
              "type (CLUSTER OR NODE OR VOLUME OR AE)",
              CLIBaseClass.NOT_REQUIRED,
              null))
      .put(AlarmCommands.ENTITY_PARAM_NAME,
          new TextInputParameter(AlarmCommands.ENTITY_PARAM_NAME,
              "entity (hostname OR volume name OR Ae name)",
              CLIBaseClass.NOT_REQUIRED,
              null))
      .put(AlarmCommands.START_PARAM_NAME,
          new IntegerInputParameter(AlarmCommands.START_PARAM_NAME,
              "start",
               CLIBaseClass.NOT_REQUIRED,
               0))
      .put(AlarmCommands.LIMIT_PARAM_NAME,
          new IntegerInputParameter(AlarmCommands.LIMIT_PARAM_NAME,
              "limit",
              CLIBaseClass.NOT_REQUIRED,
              Integer.MAX_VALUE))
      .put(AlarmCommands.OUTPUT_PARAM_NAME,
          new TextInputParameter(AlarmCommands.OUTPUT_PARAM_NAME,
              "output",
              CLIBaseClass.NOT_REQUIRED,
              "verbose"))
      .build(),
      null
  ).setShortUsage("list");


  static final CLICommand raiseCmd = new CLICommand(
      "raise",
      "",
      AlarmCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(AlarmCommands.baseParams)
      .put(AlarmCommands.ALARM_NAME_PARAM_NAME,
          new TextInputParameter(AlarmCommands.ALARM_NAME_PARAM_NAME,
              "alarm",
              CLIBaseClass.REQUIRED,
              null))
      .put(AlarmCommands.ENTITY_PARAM_NAME,
          new TextInputParameter(AlarmCommands.ENTITY_PARAM_NAME,
              "entity (hostname OR volume name OR Ae name)",
              CLIBaseClass.NOT_REQUIRED,
              null))
      .put(AlarmCommands.DESC_PARAM_NAME,
          new TextInputParameter(AlarmCommands.DESC_PARAM_NAME,
              "brief description",
              CLIBaseClass.NOT_REQUIRED,
              null))
      .build(),
      null
  ).setShortUsage("raise");


  static final CLICommand clearCmd = new CLICommand(
      "clear",
      "",
      AlarmCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(AlarmCommands.baseParams)
      .put(AlarmCommands.ALARM_NAME_PARAM_NAME,
          new TextInputParameter(AlarmCommands.ALARM_NAME_PARAM_NAME,
              "alarm",
              CLIBaseClass.REQUIRED,
              null))
      .put(AlarmCommands.ENTITY_PARAM_NAME,
          new TextInputParameter(AlarmCommands.ENTITY_PARAM_NAME,
              "entity (hostname OR volume name OR Ae name)",
              CLIBaseClass.NOT_REQUIRED,
              null))
      .build(),
      null
  ).setShortUsage("clear");

  static final CLICommand clearAllCmd = new CLICommand(
      "clearall",
      "",
      AlarmCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(AlarmCommands.baseParams)
      .build(),
      null
  ).setShortUsage("clearall");

  static final CLICommand configLoadCmd = new CLICommand(
      "load",
      "",
      AlarmCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(AlarmCommands.baseParams)
      .put(AlarmCommands.OUTPUT_PARAM_NAME,
          new TextInputParameter(AlarmCommands.OUTPUT_PARAM_NAME,
              "output",
              CLIBaseClass.NOT_REQUIRED,
              "verbose"))
      .build(),
      null
  ).setShortUsage("config load");

  static final CLICommand configSaveCmd = new CLICommand(
      "save",
      "",
      AlarmCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .putAll(AlarmCommands.baseParams)
      .put(AlarmCommands.VALUES_PARAM_NAME,
          new TextInputParameter(AlarmCommands.VALUES_PARAM_NAME,
              "values",
              CLIBaseClass.REQUIRED,
              ""))
      .build(),
      null
  ).setShortUsage("config save");

  static final CLICommand namesCmd = new CLICommand(
      "names",
      "list of alarm names",
      AlarmCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .build(),
      null
  ).setShortUsage("names");

  public static final CLICommand[] configCmdsArray = {
    configLoadCmd,
    configSaveCmd
  };

  public static CLICommand configCmds = new CLICommand(
      "config",
      "",
      CLIUsageOnlyCommand.class,
      ExecutionTypeEnum.NATIVE,
      configCmdsArray
  ).setShortUsage("config [load|save]");

  /* Define sub command */
  public static final CLICommand alarmCmds = new CLICommand(
      "alarm",
      "",
      CLIUsageOnlyCommand.class,
      ExecutionTypeEnum.NATIVE,
      new CLICommand [] {         /* array of subcommands */
         addCmd,
         viewCmd,
         deleteCmd,
         listCmd,
         raiseCmd,
         clearCmd,
         clearAllCmd,
         configCmds,
         namesCmd
      }
  ).setShortUsage("alarm [list|raise|clear|clearall|config|names]");

  public static enum AlarmEntryField {
    an,
    et,
    st,
    sct,
    des,
    en
  };

  public static String [] AlarmEntryFieldLongName = {
    "alarm name",
    "alarm type",
    "alarm state",
    "alarm statechange time",
    "description",
    "entity"
  };

  public static Map<AlarmEntryField, FieldInfo> fieldTable =
   new ImmutableMap.Builder<AlarmEntryField, FieldInfo>()
   .put(AlarmEntryField.an,
       new FieldInfo(AlarmEntryField.an.ordinal(),
           AlarmEntryField.an.name(),
           AlarmEntryFieldLongName[AlarmEntryField.an.ordinal()],
           String.class))
   .put(AlarmEntryField.et,
       new FieldInfo(AlarmEntryField.et.ordinal(),
           AlarmEntryField.et.name(),
           AlarmEntryFieldLongName[AlarmEntryField.et.ordinal()],
           String.class))
   .put(AlarmEntryField.st,
       new FieldInfo(AlarmEntryField.st.ordinal(),
           AlarmEntryField.st.name(),
           AlarmEntryFieldLongName[AlarmEntryField.st.ordinal()],
           String.class))
   .put(AlarmEntryField.sct,
       new FieldInfo(AlarmEntryField.sct.ordinal(),
           AlarmEntryField.sct.name(),
           AlarmEntryFieldLongName[AlarmEntryField.sct.ordinal()],
           String.class))
   .put(AlarmEntryField.des,
       new FieldInfo(AlarmEntryField.des.ordinal(),
           AlarmEntryField.des.name(),
           AlarmEntryFieldLongName[AlarmEntryField.des.ordinal()],
           String.class))
   .put(AlarmEntryField.en,
       new FieldInfo(AlarmEntryField.en.ordinal(),
           AlarmEntryField.en.name(),
           AlarmEntryFieldLongName[AlarmEntryField.en.ordinal()],
           String.class))
   .build();


  /*
  private boolean isValidAlarmName(String alarmName) {
    try {
      AlarmId alarmId = AlarmId.valueOf(alarmName);
      if (alarmId == null) {

      }
      return (alarmId != null);
    } catch (Exception e) {
      return false;
    }
  }
  */


  public static boolean isValidAlarmTypeName(String alarmTypeName) {
    return alarmNameMap.containsKey(alarmTypeName.toLowerCase());
  }

  private AlarmType alarmNameToType(String alarmName) throws CLIProcessingException {
    if (alarmName.startsWith("CLUSTER")) {
      return AlarmType.CLUSTER_ALARM;
    } else if (alarmName.startsWith("NODE")) {
      return AlarmType.NODE_ALARM;
    } else if (alarmName.startsWith("VOLUME")) {
      return AlarmType.VOLUME_ALARM;
    } else if (alarmName.startsWith("AE")) {
      return AlarmType.AE_ALARM;
    } else {
      throw new CLIProcessingException ("Invalid Alarm name: " + alarmName);
    }
  }

  public boolean validateInputParams(CommandOutput output)
      throws CLIProcessingException {
    boolean result = true;
    OutputHierarchy out = output.getOutput();

    /*
    if (isParamPresent(ALARM_NAME_PARAM_NAME)) {
      String alarm = getParamTextValue(ALARM_NAME_PARAM_NAME, 0).toUpperCase();
      if (!isValidAlarmName(alarm)) {
        /**
         * <MAPR_ERROR>
         * Message:Invalid alarm name
         * Function:AlarmCommands.validateInput()
         * Meaning:The specified alarm is invalid.
         * Resolution:Check the alarm name and try again.
         * </MAPR_ERROR>
         */
    /*
        String msg = "Invalid alarm name " + alarm;
        LOG.error(msg);
        out.addError(new OutputError(Errno.EOPFAILED, msg));
        result = false;
      }
    }
    */


    if (isParamPresent(ALARM_TYPE_PARAM_NAME)) {
      String type = getParamTextValue(ALARM_TYPE_PARAM_NAME, 0).toUpperCase();
      if (!isValidAlarmTypeName(type)) {
        /**
         * <MAPR_ERROR> 
         * Message:Invalid alarm type
         * Function:AlarmCommands.validateInput()
         * Meaning:The specified alarm type is invalid. 
         * Resolution:Check the alarm type and try again.
         * </MAPR_ERROR>
         */
        String msg = "Invalid alarm type " + type;
        LOG.error(msg);
        out.addError(new OutputError(Errno.EOPFAILED, msg));
        result = false;
      }
    }

    if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
      String clusterName = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0);
      if (!CLDBRpcCommonUtils.getInstance().isValidClusterName(clusterName)) {
        /**
         * <MAPR_ERROR>
         * Message:Invalid cluster name
         * Function:AlarmCommands.validateInput()
         * Meaning:The specified cluster could not be found.
         * Resolution:Check the cluster name and try again.
         * </MAPR_ERROR>
         */
        String msg = "Invalid cluster name " + clusterName;
        LOG.error(msg);
        output.getOutput().addError(new OutputError(Errno.EUCLUSTER, msg));
        result = false;
      }
    }

    return result;
  }

  /* This is a function that does the real work. */
  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    if (!validateInputParams(output)) {
      return output;
    }

    String cmd = cliCommand.getCommandName();
    if (cmd.equalsIgnoreCase("list")) {
      list(out);
    } else if (cmd.equalsIgnoreCase("add")) {
      if (verifyUserPermissions(out)) {
        addAlarm(out);
      }
    } else if (cmd.equalsIgnoreCase("view")) {
      lookupAlarm(out);
    } else if (cmd.equalsIgnoreCase("edit")) {
      if (verifyUserPermissions(out)) {
        editAlarm(out);
      }
    } else if (cmd.equalsIgnoreCase("delete")) {
      if (verifyUserPermissions(out)) {
        deleteAlarm(out);
      }
    } else if (cmd.equalsIgnoreCase("raise") ||
               cmd.equalsIgnoreCase("clear") ||
               cmd.equalsIgnoreCase("clearall")) {
      try {
        AlarmUpdateRequest req = null;
        if (cmd.equalsIgnoreCase("clearall")) {
           req = AlarmUpdateRequest.newBuilder()
                 .setCreds(getUserCredentials())
                 .setBulkState(false)
                 .build();
        } else {
          AlarmMsg.Builder alarmBuilder = AlarmMsg.newBuilder();
          alarmBuilder.setAlarmState(cmd.equalsIgnoreCase("raise"));
          if (isParamPresent(ALARM_NAME_PARAM_NAME)) {
            String alarm = getParamTextValue(ALARM_NAME_PARAM_NAME, 0).toUpperCase();
            AlarmId enumAlarmId = PluggableAlarmUtil.getAlarmId(alarm);
            if (enumAlarmId != null) {
              alarmBuilder.setAlarmId(enumAlarmId);
            }
            AlarmType type = alarmNameToType(alarm);
            alarmBuilder.setAlarmType(type);
            alarmBuilder.setAlarmName(alarm);

            if (isParamPresent(ENTITY_PARAM_NAME)) {
              alarmBuilder.setAlarmEntity(getParamTextValue(ENTITY_PARAM_NAME, 0));
            }
            if (isParamPresent(DESC_PARAM_NAME)) {
              String desc = getParamTextValue(DESC_PARAM_NAME, 0);
              if (desc != null && !desc.isEmpty()) {
                alarmBuilder.setAlarmDesc(desc);
              }
            }
          }
          req = AlarmUpdateRequest.newBuilder()
                  .setCreds(getUserCredentials())
                  .addAlarms(alarmBuilder)
                  .build();
        }

        byte[] replyData;
        if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
          replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
                    getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0),
                    Common.MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProg.AlarmUpdateProc.getNumber(),
                    req, AlarmUpdateResponse.class);
        } else {
        replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
                                    Common.MapRProgramId.CldbProgramId.getNumber(),
                                    CLDBProg.AlarmUpdateProc.getNumber(),
                                    req, AlarmUpdateResponse.class);
        }

        if (replyData == null) {
          /**
          * <MAPR_ERROR>
          * Message:Failed to connect to CLDB
          * Function:AlarmCommands.executeRealCommand()
          * Meaning:An error occurred.
          * Resolution:Contact technical support.
          * </MAPR_ERROR>
          */
          out.addError(new OutputError(Errno.EOPFAILED, "Failed to connect to CLDB"));
          return output;
        }

        AlarmUpdateResponse resp = AlarmUpdateResponse.parseFrom(replyData);
        if (resp.getStatus() != 0) {
          String errMsg;
          if (resp.getStatus() == Errno.ENOENT)
            errMsg = "No such entity";
          else if (resp.getStatus() == Errno.EINVAL) {
            errMsg = "Alarm not found";
          } else {
            errMsg =  Errno.toString(resp.getStatus());
          }
          out.addError(new OutputError(resp.getStatus(),
              "Operation failed. Error: " + errMsg));
          return output;
        }

      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);
      } catch (Exception e) {
        /**
        * <MAPR_ERROR>
        * Message:Failed due to exception
        * Function:AlarmCommands.executeRealCommand()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        out.addError(new OutputError(Errno.EOPFAILED, "Failed due to exception"));
        LOG.error("Exception: " + e.getLocalizedMessage());
      }
    } else if (cmd.equalsIgnoreCase("save")) {
      try {
        String values = getParamTextValue(VALUES_PARAM_NAME, 0);
        if ((values == null) || (values.length() == 0)) {
          /**
          * <MAPR_ERROR>
          * Message:Invalid configuration
          * Function:AlarmCommands.executeRealCommand()
          * Meaning:An error occurred.
          * Resolution:Contact technical support.
          * </MAPR_ERROR>
          */
          out.addError(new OutputError(Errno.EINVAL, "Invalid configuration"));
          return output;
        }

        AlarmUpdateRequest.Builder req = AlarmUpdateRequest
                                            .newBuilder()
                                            .setCreds(getUserCredentials())
                                            .setConfigUpdate(true);

        String [] obj = values.split(",");
        for (int i = 0; i < obj.length; i += 3) {
          if (obj[i] == null || obj[i].isEmpty())
            continue;

          AlarmMsg.Builder alarmBuilder = AlarmMsg.newBuilder();
          AlarmId alarmId = PluggableAlarmUtil.getAlarmId(obj[i]);
          if (alarmId != null) {
            alarmBuilder.setAlarmId(alarmId);
          }
          alarmBuilder.setAlarmName(obj[i]);

          // Build the AlarmConfigMsg
          AlarmConfigMsg.Builder cBuilder = AlarmConfigMsg.newBuilder();

          if ((obj.length > i + 1) && obj[i + 1] != null && !obj[i + 1].isEmpty()) {
            int individual = Integer.parseInt(obj[i + 1]);
            if (individual >= 0)
              cBuilder.setNeedIndividualEmail((individual > 0) ? true : false);
          }

          if ((obj.length > i + 2) && obj[i + 2] != null) {
            String email = obj[i + 2].trim();
            if (!email.isEmpty() && !MapRCliUtil.validateEmail(email)) {
                /**
                * <MAPR_ERROR>
                * Message:Invalid Email format in configuration
                * Function:AlarmCommands.executeRealCommand()
                * Meaning:An error occurred.
                * Resolution:Contact technical support.
                * </MAPR_ERROR>
                */
                out.addError(new OutputError(Errno.EOPFAILED,
                    "Invalid Email format in configuration"));
            }
            cBuilder.setEmail(email);
          }

          alarmBuilder.setAlarmConfigMsg(cBuilder);
          req.addAlarms(alarmBuilder);
        }

        /*for (int i = 0; i < obj.length; i += 5) {
          if (obj[i] == null || obj[i].isEmpty())
            continue;

          AlarmMsg.Builder alarmBuilder = AlarmMsg.newBuilder();
          AlarmId alarmId = Common.AlarmId.valueOf(obj[i]);
          alarmBuilder.setAlarmId(alarmId);

          // Build the AlarmConfigMsg
          AlarmConfigMsg.Builder cBuilder = AlarmConfigMsg.newBuilder();

          if ((obj.length > i + 1) && obj[i + 1] != null && !obj[i + 1].isEmpty()) {
            int individual = Integer.parseInt(obj[i + 1]);
            if (individual >= 0)
              cBuilder.setNeedIndividualEmail((individual > 0) ? true : false);
          }

          if ((obj.length > i + 2) && obj[i + 2] != null && !obj[i + 2].isEmpty()) {
            int aggregate = Integer.parseInt(obj[i + 2]);
            if (aggregate >= 0)
              cBuilder.setNeedAggregateEmail((aggregate > 0) ? true : false);
          }

          if ((obj.length > i + 3) && obj[i + 3] != null && !obj[i + 3].isEmpty()) {
            String email = obj[i + 3];
            if (!MapRCliUtil.validateEmail(email)) {
              out.addError(new OutputError(Errno.EOPFAILED,
                    "Invalid Email format in configuration"));
            }
            cBuilder.setEmail(email);
          }

          if ((obj.length > i + 4) && obj[i + 4] != null && !obj[i + 4].isEmpty()) {
            int threshold = Integer.parseInt(obj[i + 4]);
            if (threshold >= 0)
              cBuilder.setThreshold(threshold);
          }

          alarmBuilder.setAlarmConfigMsg(cBuilder);
          req.addAlarms(alarmBuilder);
        }*/

        byte[] replyData;

        if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
            replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
                    getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0),
                    Common.MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProg.AlarmUpdateProc.getNumber(),
                    req.build(), AlarmUpdateResponse.class);
        } else {
        replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
                                    Common.MapRProgramId.CldbProgramId.getNumber(),
                                    CLDBProg.AlarmUpdateProc.getNumber(),
                                    req.build(), AlarmUpdateResponse.class);
        }
        if (replyData == null) {
          /**
          * <MAPR_ERROR>
          * Message:Failed to connect to CLDB
          * Function:AlarmCommands.executeRealCommand()
          * Meaning:An error occurred.
          * Resolution:Contact technical support.
          * </MAPR_ERROR>
          */
          out.addError(new OutputError(Errno.EOPFAILED, "Failed to connect to CLDB"));
          return output;
        }
      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);
      } catch (Exception e) {
        /**
        * <MAPR_ERROR>
        * Message:Failed due to an exception
        * Function:AlarmCommands.executeRealCommand()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        out.addError(new OutputError(Errno.EOPFAILED, "Failed due to an exception"));
        LOG.error("Exception: " + e.getMessage());
      }
    } else if (cmd.equalsIgnoreCase("load")) {
      try {
        doConfLookup(out);
      } catch (Exception e) {
        /**
        * <MAPR_ERROR>
        * Message:Failed <error>
        * Function:AlarmCommands.executeRealCommand()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        out.addError(new OutputError(Errno.EOPFAILED, "Failed " + e.getLocalizedMessage()));
        LOG.error("Exception: " + e.getLocalizedMessage());
      }
    } else if (cmd.equalsIgnoreCase("names")) {
      for (AlarmId alarmId: AlarmId.values()) {
        System.out.println(alarmId.toString());
      }

      // TODO Will need to add all pluggable alarms, only getting first 50
      for (PluggableAlarm pa : PluggableAlarmUtil.getAlarms(getUserCredentials(), 0, 50)) {
        System.out.println(pa.getName());
      }


    }
    return output;
  }


  private void addAlarm(OutputHierarchy out) throws CLIProcessingException {
    // Run RPC to CLDB to add new alarm into pluggable Alarms KV store

    ServerCommands.fieldTable = PluggableAlarmUtil.appendNodeMap(getUserCredentials(), ServerCommands.fieldTableBuilder);

    PluggableAlarm.Builder builder = PluggableAlarm.newBuilder();
    AlarmAddResponse resp;

    builder.setType("NODE");
    String alarmName = getParamTextValue(ALARM_NAME_PARAM_NAME, 0);
    String alarmTerseName = getParamTextValue(ALARM_TERSE_NAME_PARAM_NAME, 0);
    String alarmUiDisplayName;
    String alarmDisplayName;
    if (isParamPresent(ALARM_DISPLAY_NAME_PARAM_NAME))
      alarmDisplayName = getParamTextValue(ALARM_DISPLAY_NAME_PARAM_NAME, 0);
    else
      alarmDisplayName = alarmName;

    if (isParamPresent(ALARM_UI_DISPLAY_NAME))
      alarmUiDisplayName = getParamTextValue(ALARM_UI_DISPLAY_NAME, 0);
    else
      alarmUiDisplayName = alarmName;

    if (verifyAlarmParams(out, ServerCommands.fieldTable, alarmName, alarmDisplayName, alarmTerseName)) {
      builder.setName(alarmName);
      builder.setTerse(alarmTerseName);
      builder.setDisplayName(alarmDisplayName);
      builder.setUiLabel(alarmUiDisplayName);
      if (isParamPresent(ALARM_SERVICE_KEY_NAME)) {
        String service = getParamTextValue(ALARM_SERVICE_KEY_NAME, 0);
        if (isParamPresent(ALARM_SERVICE_DISPLAY_NAME)) {
          builder.setServiceName(getParamTextValue(ALARM_SERVICE_DISPLAY_NAME, 0));
        } else {
          builder.setServiceName(service);
        }
        builder.setService(service);
      }
      if (isParamPresent(ALARM_TOOLTIP_NAME))
        builder.setTooltip(getParamTextValue(ALARM_TOOLTIP_NAME, 0));


      if (isParamPresent(ALARM_BASE_SERVICE_NAME)) {
        builder.setBaseService(getParamBooleanValue(ALARM_BASE_SERVICE_NAME, 0));
      }

      AlarmAddRequest req =
          AlarmAddRequest.newBuilder()
              .setCreds(getUserCredentials())
              .setPluggableAlarm(builder.build())
              .build();

      byte[] replyData;

      try {

        replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProg.AlarmAddProc.getNumber(),
            req, AlarmAddResponse.class);
        if (replyData == null) {
          /**
           * <MAPR_ERROR>
           * Message:Failed <error>
           * Function:AlarmCommands.executeRealCommand()
           * Meaning:An error occurred.
           * Resolution:Contact technical support.
           * </MAPR_ERROR>
           */
          out.addError(new OutputError(Errno.EOPFAILED, "Failed to connect to CLDB"));
        } else {
          resp = AlarmAddResponse.parseFrom(replyData);
          if (resp.getStatus() != 0) {
            String errMsg;
            if (resp.getStatus() == Errno.EEXIST) {
              errMsg = "Alarm name already exists. Cannot add duplicate alarm name.";
            } else {
              errMsg = "Did not successfully add alarm because of unknown errors";
            }
            out.addError(new OutputError(resp.getStatus(), errMsg));

          }

        }
      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);

      } catch (InvalidProtocolBufferException ipbe) {
        throw new CLIProcessingException("Exception while parsing the RPC "
            + "response data into AlarmViewResponse proto object.", ipbe);
      } catch (Exception e) {
        /**
         * <MAPR_ERROR>
         * Message:Failed due to exception
         * Function:AlarmCommands.executeRealCommand()
         * Meaning:An error occurred.
         * Resolution:Contact technical support.
         * </MAPR_ERROR>
         */
        out.addError(new OutputError(Errno.EOPFAILED, "Failed due to exception"));
        LOG.error("Exception: " + e.getLocalizedMessage());
      }
    }


  }

  private void editAlarm(OutputHierarchy out) throws  CLIProcessingException {
    String alarmName = getParamTextValue(ALARM_NAME_PARAM_NAME, 0);

    PluggableAlarm.Builder builder = PluggableAlarm.newBuilder();

    if (isParamPresent(ALARM_NEWNAME_PARAM_NAME))
      builder.setName(getParamTextValue(ALARM_NEWNAME_PARAM_NAME, 0));
    else
      builder.setName(alarmName);
    if (isParamPresent(ALARM_TERSE_NAME_PARAM_NAME))
      builder.setTerse(getParamTextValue(ALARM_TERSE_NAME_PARAM_NAME, 0));
    if (isParamPresent(ALARM_DISPLAY_NAME_PARAM_NAME))
      builder.setDisplayName(getParamTextValue(ALARM_DISPLAY_NAME_PARAM_NAME, 0));

    AlarmEditRequest req =
        AlarmEditRequest.newBuilder()
            .setAlarmname(alarmName)
            .setPluggableAlarm(builder.build())
            .setCreds(getUserCredentials())
            .build();


    AlarmEditResponse resp = null;
    byte[] replyData;

    try {

      replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
          Common.MapRProgramId.CldbProgramId.getNumber(),
          CLDBProg.AlarmEditProc.getNumber(),
          req, AlarmEditResponse.class);
      if (replyData == null) {
        /**
         * <MAPR_ERROR>
         * Message:Failed <error>
         * Function:AlarmCommands.executeRealCommand()
         * Meaning:An error occurred.
         * Resolution:Contact technical support.
         * </MAPR_ERROR>
         */
        out.addError(new OutputError(Errno.EOPFAILED, "Failed to connect to CLDB"));
      } else {
        resp = AlarmEditResponse.parseFrom(replyData);

        if (resp != null) {
          if (resp.getStatus() != 0) {
            String errMsg;
            if (resp.getStatus() == Errno.EINVAL) {
              errMsg = "Alarm not found";
            } else {
              errMsg = "Could not modify alarm, unknown error";
            }
            out.addError(new OutputError(resp.getStatus(), errMsg));
          }
        } else {
          out.addError(new OutputError(Errno.EINVAL, "Reply data did not get retrieved properly."));
        }
      }


    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);

    } catch (InvalidProtocolBufferException ipbe) {
      throw new CLIProcessingException("Exception while parsing the RPC "
          + "response data into AlarmEditResponse proto object.", ipbe);
    } catch (Exception e) {
      /**
       * <MAPR_ERROR>
       * Message:Failed due to exception
       * Function:AlarmCommands.executeRealCommand()
       * Meaning:An error occurred.
       * Resolution:Contact technical support.
       * </MAPR_ERROR>
       */
      out.addError(new OutputError(Errno.EOPFAILED, "Failed due to exception"));
      LOG.error("Exception: " + e.getLocalizedMessage());
    }



  }

  private void deleteAlarm(OutputHierarchy out) throws CLIProcessingException {
    String alarmName = getParamTextValue(ALARM_NAME_PARAM_NAME, 0);

    AlarmDeleteRequest req =
        AlarmDeleteRequest.newBuilder()
            .setAlarmname(alarmName)
            .setCreds(getUserCredentials())
            .build();

    AlarmDeleteResponse resp = null;
    byte[] replyData;

    try {

      replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
          Common.MapRProgramId.CldbProgramId.getNumber(),
          CLDBProg.AlarmRemoveProc.getNumber(),
          req, AlarmDeleteResponse.class);
      if (replyData == null) {
        /**
         * <MAPR_ERROR>
         * Message:Failed <error>
         * Function:AlarmCommands.executeRealCommand()
         * Meaning:An error occurred.
         * Resolution:Contact technical support.
         * </MAPR_ERROR>
         */
        out.addError(new OutputError(Errno.EOPFAILED, "Failed to connect to CLDB"));
      } else {
        resp = AlarmDeleteResponse.parseFrom(replyData);

        if (resp != null) {
          if (resp.getStatus() != 0) {
            // Most likely alarm was not found.
            out.addError(new OutputError(resp.getStatus(), "Could not delete alarm, alarm not found"));
          }
        } else {
          out.addError(new OutputError(Errno.EINVAL, "Reply data did not get retrieved properly."));
        }
      }


    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);

    } catch (InvalidProtocolBufferException ipbe) {
      throw new CLIProcessingException("Exception while parsing the RPC "
          + "response data into AlarmViewResponse proto object.", ipbe);
    } catch (Exception e) {
      /**
       * <MAPR_ERROR>
       * Message:Failed due to exception
       * Function:AlarmCommands.executeRealCommand()
       * Meaning:An error occurred.
       * Resolution:Contact technical support.
       * </MAPR_ERROR>
       */
      out.addError(new OutputError(Errno.EOPFAILED, "Failed due to exception"));
      LOG.error("Exception: " + e.getLocalizedMessage());
    }

  }

  private void lookupAlarm(OutputHierarchy out) throws CLIProcessingException {
    String alarmName = getParamTextValue(ALARM_NAME_PARAM_NAME, 0);
    // Reset cache in case of maprcli being run in a process (IE webserver). Bug #13362
    PluggableAlarmUtil.resetCache();
    List<PluggableAlarm> pluggableAlarm = PluggableAlarmUtil.getAlarms(getUserCredentials(),
        alarmName, getParamIntValue(START_PARAM_NAME, 0), getParamIntValue(LIMIT_PARAM_NAME, 0));

    String clusterName = null;
    String zkConnectString = null;
    if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
      clusterName = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0);
    }

    if (isParamPresent(NodesCommonUtils.ZK_CONNECTSTRING)) {
      zkConnectString = getParamTextValue(NodesCommonUtils.ZK_CONNECTSTRING, 0);

    }
    for (PluggableAlarm pa : pluggableAlarm) {
      OutputNode node = new OutputNode();
      node.addChild(new OutputNode(ALARM_NAME_PARAM_NAME, pa.getName()));
      node.addChild(new OutputNode(ALARM_TERSE_NAME_PARAM_NAME, pa.getTerse()));
      node.addChild(new OutputNode(ALARM_DISPLAY_NAME_PARAM_NAME, pa.getDisplayName()));
      if (pa.hasUiLabel()) {
        node.addChild(new OutputNode(ALARM_UI_DISPLAY_NAME, pa.getUiLabel()));
      }
      if (pa.hasTooltip()) {
        node.addChild(new OutputNode(ALARM_TOOLTIP_NAME, pa.getTooltip()));
      }
      node.addChild(new OutputNode(ALARM_TYPE_PARAM_NAME, pa.getType()));
      node.addChild(new OutputNode("Id", pa.getId()));
      node.addChild(new OutputNode(ALARM_SERVICE_DISPLAY_NAME, pa.getServiceName()));
      node.addChild(new OutputNode(ALARM_SERVICE_KEY_NAME, pa.getService()));
      node.addChild(new OutputNode(ALARM_BASE_SERVICE_NAME, pa.getBaseService() ? 1 : 0));

      String link = ServiceLinkCommand.getUrl(new OutputHierarchy(), pa.getService(), clusterName, zkConnectString, false);
      node.addChild(new OutputNode("url", link != null ? 1 : 0));

      out.addNode(node);
    }
  }

  void doConfLookup(OutputHierarchy out) throws Exception {

    AlarmLookupRequest req =
      AlarmLookupRequest.newBuilder()
                        .setCreds(getUserCredentials())
                        .setSummary(true)
                        .build();

    byte[] replyData;

    if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM))
    {
         replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
                 getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
                Common.MapRProgramId.CldbProgramId.getNumber(),
                CLDBProg.AlarmLookupProc.getNumber(),
                req, AlarmLookupResponse.class);
    } else {
      replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
                                Common.MapRProgramId.CldbProgramId.getNumber(),
                                CLDBProg.AlarmLookupProc.getNumber(),
                                req, AlarmLookupResponse.class);
    }
    if (replyData == null) {
      /**
      * <MAPR_ERROR>
      * Message:Failed <error>
      * Function:AlarmCommands.executeRealCommand()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      out.addError(new OutputError(Errno.EOPFAILED, "Failed to connect to CLDB"));
      return;
    }

    String alarm, individual, email;
    //String aggregate, threshold;
    boolean terse = getParamTextValue(OUTPUT_PARAM_NAME, 0).equalsIgnoreCase("terse");
    if (terse) {
      alarm = "an";
      individual = "ind";
      email = "em";
      //aggregate = "ag";
      //threshold = "th";
    } else {
      alarm = "alarm";
      individual = "individual";
      email = "email";
      //aggregate = "aggregate";
      //threshold = "threshold";
    }

    AlarmLookupResponse resp = AlarmLookupResponse.parseFrom(replyData);
    for (AlarmMsg respAlarmMsg: resp.getAlarmsList()) {
       OutputNode node = new OutputNode();

       // Add Alarm Name
       node.addChild(new OutputNode(alarm, respAlarmMsg.getAlarmName()));

       AlarmConfigMsg config = respAlarmMsg.getAlarmConfigMsg();
       /*node.addChild(new OutputNode(threshold, config.getThreshold()));
       node.addChild(new OutputNode(aggregate,
                       config.getNeedAggregateEmail() ? 1 : 0));*/
       node.addChild(new OutputNode(individual,
           config.getNeedIndividualEmail() ? 1 : 0));
       node.addChild(new OutputNode(email,
           config.hasEmail() ? config.getEmail() : ""));
       out.addNode(node);
    }
    return;
  }

  private AlarmLookupRequest.Builder getAlarmLookupRequestBuilder() throws CLIProcessingException {

    AlarmLookupRequest.Builder reqBuilder =
             AlarmLookupRequest.newBuilder()
                               .setCreds(getUserCredentials())
                               .setLimiter(getNextLimiter(getParamIntValue(START_PARAM_NAME, 0), 0,
                                   getParamIntValue(START_PARAM_NAME, 0), getParamIntValue(LIMIT_PARAM_NAME, 0),
                                   NUM_ALARMS_PER_RPC));

    // Check if summary is requested
    boolean summary = false;
    if (isParamPresent(SUMMARY_PARAM_NAME)) {
      summary = getParamBooleanValue(SUMMARY_PARAM_NAME, 0);
    }
    if (summary) {
      reqBuilder.setSummary(true);
    } else {
      AlarmMsg.Builder alarmBuilder = getAlarmMsgBuilder();

      if (alarmBuilder != null) {
        reqBuilder.addAlarms(alarmBuilder);
      }
    }
    return reqBuilder;
  }

  private AlarmMsg.Builder getAlarmMsgBuilder() throws CLIProcessingException {
    AlarmMsg.Builder alarmBuilder = AlarmMsg.newBuilder();
    AlarmType type = null;
    if (isParamPresent(ALARM_TYPE_PARAM_NAME)) {
      type = alarmNameToType(getParamTextValue(ALARM_TYPE_PARAM_NAME, 0).toUpperCase());
      alarmBuilder.setAlarmType(type);
    }

    if (isParamPresent(ALARM_NAME_PARAM_NAME)) {
      String alarm = getParamTextValue(ALARM_NAME_PARAM_NAME, 0).toUpperCase();
      AlarmId enumAlarmId = PluggableAlarmUtil.getAlarmId(alarm);
      if (enumAlarmId != null) {
        alarmBuilder.setAlarmId(enumAlarmId);
      }
      alarmBuilder.setAlarmName(alarm);
      if (type == null) {
        type = alarmNameToType(alarm);
        alarmBuilder.setAlarmType(type);
      }
    }

    if (type != null && isParamPresent(ENTITY_PARAM_NAME)) {
      alarmBuilder.setAlarmEntity(getParamTextValue(ENTITY_PARAM_NAME, 0));
    }

    if (type != null) {
      return alarmBuilder;
    } else {
      return null;
    }
  }

  private AlarmLookupResponse getAlarmLookupResponse(byte[] replyData)
      throws CLIProcessingException {
    try {
      return AlarmLookupResponse.parseFrom(replyData);
    } catch (InvalidProtocolBufferException ipbe) {
      throw new CLIProcessingException("Exception while parsing the RPC "
          + "response data into AlarmLookupResponse proto object.", ipbe);
    }
  }

  @Override
  public MessageLite buildNextRequest(MessageLite prevReq, MessageLite prevResp)
      throws CLIProcessingException {
    AlarmLookupRequest.Builder newReqBuilder = null;
    if (prevReq != null) {
      newReqBuilder = AlarmLookupRequest.newBuilder((AlarmLookupRequest) prevReq);
    } else {
      newReqBuilder = getAlarmLookupRequestBuilder();
    }

    if (prevResp != null) {
      int prevStart = newReqBuilder.getLimiter().getStart();
      int prevCount = ((AlarmLookupResponse) prevResp).getAlarmsCount();
      int origStart = getParamIntValue(START_PARAM_NAME, 0);
      int origLimit = getParamIntValue(LIMIT_PARAM_NAME, 0);
      newReqBuilder.setLimiter(getNextLimiter(prevStart, prevCount,
          origStart, origLimit, NUM_ALARMS_PER_RPC));
    }

    return newReqBuilder.build();
  }

  @Override
  public boolean hasMore(MessageLite prevReq, MessageLite prevResp) throws CLIProcessingException {
    return hasMore(getParamIntValue(START_PARAM_NAME, 0),
        getParamIntValue(LIMIT_PARAM_NAME, 0),
        ((AlarmLookupRequest) prevReq).getLimiter().getStart(),
        ((AlarmLookupResponse) prevResp).getAlarmsCount());
  }

  private boolean displayInstanceInfo(AlarmId alarmId) {
    return ((alarmId.compareTo(AlarmId.NODE_ALARM_DEBUG_LOGGING) == 0) ||
            (alarmId.compareTo(AlarmId.NODE_ALARM_HB_PROCESSING_SLOW) == 0) ||
            (alarmId.compareTo(AlarmId.NODE_ALARM_DISK_FAILURE) == 0) ||
            (alarmId.compareTo(AlarmId.NODE_ALARM_HIGH_MFS_MEMORY) == 0) ||
            (alarmId.compareTo(AlarmId.NODE_ALARM_NO_HEARTBEAT) == 0) ||
            (alarmId.compareTo(AlarmId.NODE_ALARM_TOO_MANY_CONTAINERS) == 0));
  }

  @Override
  public void processResponse(OutputHierarchy out, MessageLite response) throws CLIProcessingException {
    AlarmLookupResponse resp = (AlarmLookupResponse) response;
    boolean terse = getParamTextValue(OUTPUT_PARAM_NAME, 0).equalsIgnoreCase("terse");
    for (AlarmMsg respAlarmMsg: resp.getAlarmsList()) {
      // Ignore cleared alarms for now
      if (!respAlarmMsg.getAlarmState())
        continue;

      OutputNode node = new OutputNode();
      // Add Alarm Id
      String alarmName = respAlarmMsg.getAlarmName();
      AlarmId alarmId = respAlarmMsg.getAlarmId();
      // Add entity name, 0 = CLUSTER, 1 = hostname, 2 = Volume name
      String entityName = null;
      entityName = respAlarmMsg.getAlarmEntity();
      if (entityName != null && entityName.length() != 0) {
        node.addChild(new OutputNode(fieldTable.get(AlarmEntryField.en).getName(terse),
                                     entityName));
      }
      // Add Alarm Name
      node.addChild(new OutputNode(fieldTable.get(AlarmEntryField.an).getName(terse),
                                   alarmName));
      // Add Alarm State
      node.addChild(new OutputNode(fieldTable.get(AlarmEntryField.st).getName(terse),
                                   respAlarmMsg.getAlarmState() ? 1 : 0));
      // Add Alarm Timestamp
      node.addChild(new OutputNode(fieldTable.get(AlarmEntryField.sct).getName(terse),
                                   respAlarmMsg.getAlarmTimeStamp()));
      if (displayInstanceInfo(respAlarmMsg.getAlarmId())) {
        for(int port : respAlarmMsg.getInstanceIdsList()) {
          node.addChild(new OutputNode("instances", port));
        }
      }

      String alarmDesc = respAlarmMsg.getAlarmDesc();
      if (alarmName.equals("VOLUME_ALARM_TABLE_REPL_ERROR") ||
          alarmName.equals("VOLUME_ALARM_TABLE_REPL_LAG_HIGH") ||
          alarmName.equals("VOLUME_ALARM_TABLE_REPL_ASYNC")) {
        MapRFileSystem fs = MapRCliUtil.getMapRFileSystem();
        int index = alarmDesc.indexOf("Fids: ");
        if (index > 0) {
          String errPaths = "eg.";
          String desc = alarmDesc.substring(0, index);
          String fidsStr = alarmDesc.substring(index + 6);
          String fids[] = fidsStr.split(" ");
          for (String fid: fids) {
            String newPath = null;
            try {
              newPath = fs.getMountPathFid(fid);
            } catch (Exception e) {
              newPath = null;
            }

            if (newPath == null)
              errPaths += " " + fid;
            else
              errPaths += " " + newPath;
          }
          alarmDesc = desc + errPaths;
        }
      } else if (alarmName.equals("VOLUME_ALARM_LARGE_ROW_WARNING")) {
        MapRFileSystem fs = MapRCliUtil.getMapRFileSystem();
        int index = alarmDesc.indexOf("TFID:");
        if (index > 0) {
          String fidStr = alarmDesc.substring(index + 5);
          String desc = alarmDesc.substring(0, index);

          try {
            String tablePath = fs.getMountPathFid(fidStr);
            alarmDesc = desc + tablePath;
          } catch (Exception e) {
            alarmDesc = desc + fidStr;
          }
        }
      }

      // Add Alarm Description
      node.addChild(new OutputNode(fieldTable.get(AlarmEntryField.des).getName(terse),
                                   alarmDesc));

      out.addNode(node);
   }
  }

  @Override
  public MessageLite sendRequest(MessageLite request) throws CLIProcessingException {
    AlarmLookupRequest req = (AlarmLookupRequest) request;
    byte[] replyData;

    try {
      if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
          replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
                  getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
                  Common.MapRProgramId.CldbProgramId.getNumber(),
                  CLDBProg.AlarmLookupProc.getNumber(),
                  req, AlarmLookupResponse.class);
  
      } else {
          replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
                                Common.MapRProgramId.CldbProgramId.getNumber(),
                                CLDBProg.AlarmLookupProc.getNumber(),
                                req, AlarmLookupResponse.class);
      }
    } catch (Exception e) {
      throw new CLIProcessingException(e);
    }

    if (replyData != null) {
      return getAlarmLookupResponse(replyData);
    } else {
      /**
      * <MAPR_ERROR>
      * Message:RPC Request to list alarms failed. No data returned
      * Function:AlarmCommands.executeRealCommand()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      LOG.error("RPC Request to list alarms failed. No data returned");
      return null;
    }
  }

  /**
   * Check if the user has enough permissiont to run
   * @param out {link OutputHierarchy} class that adds any errors if any were encountered
   * @return true if the user has permissions to run the start/stop command. False otherwise
   * @throws CLIProcessingException
   */
  private boolean verifyUserPermissions(OutputHierarchy out) throws CLIProcessingException{
    CredentialsMsg creds = getUserCredentials();

    boolean isRoot = (creds.hasUid() && (creds.getUid() == 0));
    if (isRoot) {
	    return true;
	  }
	  
	  return authManager.canPerformClusterActions(
	      SecurityCommandHelper.CLUSTER_START_STOP_SERVICES_MASK, creds, out);
  }

  private boolean verifyAlarmParams(OutputHierarchy out, Map<NodeField, FieldInfo> fieldTable,
                                    String alarmName, String alarmDisplayName, String alarmTerseName) {
    return true;
  }

}
