package com.mapr.cli;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import java.net.InetAddress;

import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import org.apache.log4j.Logger;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.InvalidProtocolBufferException;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandLineOutput;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.TextCommandOutput;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.NoValueInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.fs.cldb.proto.CLDBProto.CLDBProg;
import com.mapr.fs.cldb.proto.CLDBProto.ClusterActions;
import com.mapr.fs.cldb.proto.CLDBProto.SecureObjectType;
import com.mapr.fs.cldb.proto.CLDBProto.SecurityGetAclRequest;
import com.mapr.fs.cldb.proto.CLDBProto.SecurityGetAclResponse;
import com.mapr.fs.cldb.proto.CLDBProto.SecurityModifyAclRequest;
import com.mapr.fs.cldb.proto.CLDBProto.SecurityModifyAclResponse;
import com.mapr.fs.proto.Common.VolumeActions;
import com.mapr.fs.cldb.security.ACL;
import com.mapr.baseutils.acls.SecurityCommandHelper;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Security.AccessControlList;
import com.mapr.fs.proto.Security.AclEntry;
import com.mapr.fs.proto.Security.CredentialsMsg;
import com.mapr.fs.proto.Security.SecurityPrincipal;
import com.mapr.security.UnixUserGroupHelper;
import com.mapr.security.MaprSecurityException;
// import com.mapr.security.UserInformation;

public class AclCommands extends CLIBaseClass implements CLIInterface {
  static final String OBJECT_PARAM_NAME = "name";
  static final String UID_PARAM_NAME = "user";
  static final String GID_PARAM_NAME = "group";
  static final String OUTPUT_PARAM_NAME = "output";
  static final String OBJECT_TYPE_PARAM_NAME = "type";
  static final String PERM_PARAM_NAME="perm";
  static final String SHOW_ADMIN_PARAM_NAME="showadmin";
  private static final Logger LOG = Logger.getLogger(AclCommands.class);
  
  static final Map<String, BaseInputParameter> params = new ImmutableMap.Builder<String, BaseInputParameter>()
  .put(AclCommands.OBJECT_TYPE_PARAM_NAME,
      new TextInputParameter(AclCommands.OBJECT_TYPE_PARAM_NAME,
          "object type [cluster|volume]", 
          CLIBaseClass.REQUIRED, 
          null))
  .put(AclCommands.OBJECT_PARAM_NAME,
      new TextInputParameter(AclCommands.OBJECT_PARAM_NAME,
          "name",
          CLIBaseClass.NOT_REQUIRED, 
          null))
  .put(MapRCliUtil.CLUSTER_NAME_PARAM,
      new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
          "cluster name",
          CLIBaseClass.NOT_REQUIRED,
          null))
  .build();
  
  static final CLICommand showAclCommand = new CLICommand(
      "show",
      "display ACL for an object",
      AclCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(AclCommands.params)
        .put(AclCommands.UID_PARAM_NAME,
            new TextInputParameter(AclCommands.UID_PARAM_NAME,
                "userName whose ACL is queried",
                CLIBaseClass.NOT_REQUIRED,
                null))
        .put(AclCommands.GID_PARAM_NAME,
            new TextInputParameter(AclCommands.GID_PARAM_NAME,
                "groupName whose ACL is queried",
                CLIBaseClass.NOT_REQUIRED,
                null))
        .put(AclCommands.OUTPUT_PARAM_NAME,
            new TextInputParameter(AclCommands.OUTPUT_PARAM_NAME,
                "output format short|long|terse (default short)",
                CLIBaseClass.NOT_REQUIRED,
                "short"))
        .put(AclCommands.PERM_PARAM_NAME,
            new NoValueInputParameter(AclCommands.PERM_PARAM_NAME, 
                "list of available permissions", 
                CLIBaseClass.NOT_REQUIRED,
                false))
        .put(AclCommands.SHOW_ADMIN_PARAM_NAME,
            new BooleanInputParameter(AclCommands.SHOW_ADMIN_PARAM_NAME,
                "show cluster admin", 
                CLIBaseClass.NOT_REQUIRED,
                false).setInvisible(true))
        .build(),
      null)
  .setShortUsage("show -type cluster|volume -perm -name <volume name> -user <userName> -group <groupName> -output <format>");
  
  static final CLICommand setAclCommand = new CLICommand(
      "set",
      "set ACL for an object",
      AclCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(AclCommands.params)
        .put(AclCommands.UID_PARAM_NAME,
            new TextInputParameter(AclCommands.UID_PARAM_NAME,
                "space separated list of user:permissions,perimssions,.. to be set",
                CLIBaseClass.NOT_REQUIRED,
                null))
        .put(AclCommands.GID_PARAM_NAME,
            new TextInputParameter(AclCommands.GID_PARAM_NAME,
                "space separated list of group:permissions,permissions,... to be set",
                CLIBaseClass.NOT_REQUIRED,
                null))
        .build(),
        null)
  .setShortUsage("set -type cluster|volume -name <volume name> -user {userName:permissions}+ -group {groupName:permissions}+");
  
  static final CLICommand editAclCommand = new CLICommand(
      "edit",
      "edit ACL for an object",
      AclCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(AclCommands.params)
        .put(AclCommands.UID_PARAM_NAME,
            new TextInputParameter(AclCommands.UID_PARAM_NAME,
                "space separated list of user:permissions,perimssions,.. to be changed",
                CLIBaseClass.NOT_REQUIRED,
                null))
        .put(AclCommands.GID_PARAM_NAME,
            new TextInputParameter(AclCommands.GID_PARAM_NAME,
                "space separated list of group:permissions,permissions,... to be changed",
                CLIBaseClass.NOT_REQUIRED,
                null))
        .build(),
        null)
  .setShortUsage("edit -type cluster|volume -name <volume name> -user {userName:permissions}+ -group {groupName:permissions}+");
  
  static final CLICommand[] aclSubCommands = {
    showAclCommand,
    setAclCommand,
    editAclCommand,
  };
  
  static final String usageStr = "acl [show|set|edit] -type [cluster|volume] -name <volume name>";
  public static final CLICommand aclCommands = 
    new CLICommand(
        "acl",
        "usage: " + usageStr,
        CLIUsageOnlyCommand.class,
        ExecutionTypeEnum.NATIVE,
        aclSubCommands)
    .setShortUsage(usageStr);

  UnixUserGroupHelper userInfo;
  static String MULTI_ARG_SEP = ",";
  static String ALLOW_MASK_SEP = ":";
  
  public AclCommands(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }
  
  public static OutputHierarchy formatAcl(AccessControlList acl, SecureObjectType objType, String outputFormat, UnixUserGroupHelper uInfo, List<Integer> unknownUids, List<Integer> unknownGids) {
    return formatAcl(acl, objType, outputFormat, uInfo, unknownUids, unknownGids, -1);
  }

  private static OutputHierarchy formatAcl(AccessControlList acl, SecureObjectType objType, String outputFormat, UnixUserGroupHelper uInfo, List<Integer> unknownUids, List<Integer> unknownGids, int clusterOwner) {
    List<OutputNode> result = new ArrayList<OutputNode>();
    OutputHierarchy oh = new OutputHierarchy();
    
    // do not mark immutable for non-cluster ACLs
    if (objType != SecureObjectType.OBJECT_TYPE_CLUSTER) clusterOwner = -1;

    /*
     * key, value pairs are the following:
     * key = "user <uid>" or "group <gid>"
     * value = 2 OutputNodes
     * 1st OutputNode:
     * key = "allow list"
     * value = list of OutputNodes - key = action #, value = description of action
     */
    for (AclEntry aclEntry : acl.getAclList()) {
      OutputNode aclDesc = new OutputNode();
      boolean addAcl = false;
      boolean isClusterOwner = false;

      int princId = aclEntry.getPrincipal().getPrincId();
      String ugName;
      
      if (ACL.allUsers(princId)) {
        ugName = "All users";
      } else if (ACL.isUid(princId)) {
        if (clusterOwner != -1) isClusterOwner = (princId == clusterOwner);
        // convert uid to username
        String userName = "";
        try {
          userName = uInfo.getUsername(princId);
          ugName = "User " + userName;
        } catch (SecurityException se) {
          ugName = "Uid " + princId;
          if (unknownUids != null) unknownUids.add(princId);
        }
      } else {
        String groupName = "";
        try {
          groupName = uInfo.getGroupname(ACL.getGid(princId));
          ugName = "Group " + groupName;
        } catch (SecurityException se) {
          int gid = ACL.getGid(princId);
          ugName = "Gid " + gid;
          if (unknownGids != null) unknownGids.add(gid);
        }
      }

      if (outputFormat.equalsIgnoreCase("terse")) {
				if (ACL.allUsers(princId)) {
					ugName = "User allusers";
				}
        // terse output format
        if (aclEntry.hasAllow()) {
          addAcl = true;
          aclDesc.addNode(new OutputNode(ugName, 
            SecurityCommandHelper.formatActionMask(aclEntry.getAllow(), 
            objType, true)));
        }
      } else {
        int allowMask = 0;
        int denyMask = 0;
        boolean shortFormat = outputFormat.equalsIgnoreCase("short");

        // verbose output format
        if (aclEntry.hasAllow()) {
          allowMask = aclEntry.getAllow();
        }

        if (aclEntry.hasDeny()) {
          denyMask = aclEntry.getDeny();
        }

        if ((allowMask != 0) || (denyMask != 0)) {
          addAcl = true;
          aclDesc.addNode(new OutputNode("Principal", ugName));
        }

        if (allowMask != 0) {
          List<String> privs = 
            SecurityCommandHelper.formatActionMask(allowMask, objType, shortFormat);
          addAcl = true;
          aclDesc.addNode(new OutputNode("Allowed actions", privs));
        }
        if (denyMask != 0) {
          List<String> privs = 
            SecurityCommandHelper.formatActionMask(denyMask, objType, shortFormat);
          addAcl = true;
          aclDesc.addNode(new OutputNode("Denied actions", privs));
        }
      }
          
      if (addAcl) {
        if (isClusterOwner) aclDesc.addNode(new OutputNode("ClusterAdmin", true));
        result.add(aclDesc);
      }
    }
    for(OutputNode a: result) {
      oh.addNode(a);
    }
    return oh;
  }
  
  String getHostName() {
    String hostname = MapRCliUtil.getHostname();
    if (hostname != null) return hostname;

    try {
      hostname = InetAddress.getLocalHost().getHostName();
    } catch (Exception e) {
    }

    if (hostname == null) {
      try {
        hostname = InetAddress.getLocalHost().getHostAddress();
      } catch (Exception e) {
      }
    }
    return hostname;
  }

  String formatList(List<Integer> list) {
    if (list.size() == 0) return null;
    StringBuffer result = new StringBuffer();
    boolean needsComma = false;
    for(int id : list) {
      if (needsComma) {
        result.append(", ");
        result.append(id);
      } else {
        result.append(id);
        needsComma = true;
      }
    }
    return result.toString();
  }

  CommandOutput showAcl() throws CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();

    // is this a cluster or a volume ACL command?
    String objectType = getParamTextValue(OBJECT_TYPE_PARAM_NAME, 0);
    SecureObjectType objType;
    
    if (objectType.equalsIgnoreCase("cluster")) {
      objType = SecureObjectType.OBJECT_TYPE_CLUSTER;
    } else if (objectType.equalsIgnoreCase("volume")) {
      objType = SecureObjectType.OBJECT_TYPE_VOLUME;
    } else {
      // return an error
      /**
      * <MAPR_ERROR>
      * Message:Acl show failed with invalid object type <type>. Valid object types are cluster|volume.
      * Function:AclCommands.showAcl()
      * Meaning:The {{acl show}} command requires a {{type}} of {{cluster}} or {{volume}}.
      * Resolution:Check the command syntax and try again.
      * </MAPR_ERROR>
      */

      out.addError(new OutputError(Errno.EOPFAILED, 
          "Acl show failed with invalid object type " + objectType
          + ". Valid object types are cluster|volume.")
        .setField(OBJECT_TYPE_PARAM_NAME));
      output.setOutput(out);
      return output;
    }
    
    if(isParamPresent(AclCommands.PERM_PARAM_NAME)) {
      String[] headers = {"Permissions","Description"};
      output.setNodeOrder(headers);
      List<OutputNode> results = new ArrayList<OutputNode>();
      if(objType == SecureObjectType.OBJECT_TYPE_CLUSTER) {
        for(int i=0;i<SecurityCommandHelper.clusterActionsDescription.length-1;i+=2) {
          OutputNode node = new OutputNode();
          if(SecurityCommandHelper.clusterActionsDescription[i] != null && 
            SecurityCommandHelper.clusterActionsDescription[i+1] != null) {
            node.addNode(new OutputNode("Permissions",
              SecurityCommandHelper.clusterActionsDescription[i])); 
            node.addNode(new OutputNode("Description",
              SecurityCommandHelper.clusterActionsDescription[i+1]));
            results.add(node);
          }
        }
      } else if(objType == SecureObjectType.OBJECT_TYPE_VOLUME) {
        for(int i=0;i<SecurityCommandHelper.volumeActionsDescription.length-1;i+=2) {
          OutputNode node = new OutputNode();
          if(SecurityCommandHelper.volumeActionsDescription[i] != null && 
            SecurityCommandHelper.volumeActionsDescription[i+1] != null) {
            node.addNode(new OutputNode("Permissions",
              SecurityCommandHelper.volumeActionsDescription[i]));
            node.addNode(new OutputNode("Description",
              SecurityCommandHelper.volumeActionsDescription[i+1]));
            results.add(node);
    			}
    		}
    	}
    	for(OutputNode a:results) {
    	  out.addNode(a);
    	}
    	output.setOutput(out);
    	return output;
    }
    String name = null;
    if (isParamPresent(OBJECT_PARAM_NAME)) {
      name = getParamTextValue(OBJECT_PARAM_NAME, 0);
    }
    
    if ((name == null) && (objType == SecureObjectType.OBJECT_TYPE_VOLUME)) {
      /**
      * <MAPR_ERROR>
      * Message:Acl show requires the name of the volume
      * Function:AclCommands.showAcl()
      * Meaning:The acl show command requires a volume name when the type is volume.
      * Resolution:Check the command syntax and try again.
      * </MAPR_ERROR>
      */

      out.addError(new OutputError(Errno.EOPFAILED,
          "Acl show requires the name of the volume")
        .setField(OBJECT_PARAM_NAME));
      output.setOutput(out);
      return output;
    }

    if ((name != null) && (objType == SecureObjectType.OBJECT_TYPE_CLUSTER)) {
      out.addError(new OutputError(Errno.EOPFAILED,
          "Acl show for a cluster does not require a name")
        .setField(OBJECT_PARAM_NAME));
      output.setOutput(out);
      return output;
    }
    
    boolean isUserPresent = false;
    int princId = 0;
    if (isParamPresent(UID_PARAM_NAME)) {
      isUserPresent = true;
      String userName = getParamTextValue(UID_PARAM_NAME, 0);
      if (userName.equalsIgnoreCase("allUsers")) {
        princId = -1;
      } else {
      	try {
          princId = userInfo.getUserId(userName);
      	} catch(SecurityException se) {
      		/**
      		* <MAPR_ERROR>
      		* Message:Cannot get UID for user <userName>
      		* Function:AclCommands.showAcl()
      		* Meaning:Username incorrect or does not exist
      		* Resolution:Check the user in the command syntax and try again.
      		* </MAPR_ERROR>
      		*/
      		out.addError(new OutputError(Errno.ENOUSER,
            Errno.toString(Errno.ENOUSER) + ":" + userName)
      		  .setField(OBJECT_PARAM_NAME)); 
      		  output.setOutput(out);
      		  return output;
      	}
      }
    }
    
    boolean isGroupPresent = false;
    int gid = 0;
    if (isParamPresent(GID_PARAM_NAME)) {
      isGroupPresent = true;
      String groupName = getParamTextValue(GID_PARAM_NAME, 0);
      try {
      	gid = userInfo.getGroupId(groupName);	
      } catch(SecurityException se) {
    		/**
    		* <MAPR_ERROR>
    		* Message:Acl show requires the group to be valid to display the group permissions
    		* Function:AclCommands.showAcl()
    		* Meaning:The acl show requires the group to be valid to display the group permissions
    		* Resolution:Check the group in the command syntax and try again.
    		* </MAPR_ERROR>
    		*/
    		out.addError(new OutputError(Errno.ENOUSER,
          Errno.toString(Errno.ENOUSER) + ":" + groupName)
          .setField(OBJECT_PARAM_NAME));
    		  output.setOutput(out);
    		  return output;
    	}
      princId = 0x80000000 | gid;
    }
    
    if (isUserPresent && isGroupPresent) {
      /**
      * <MAPR_ERROR>
      * Message:Acl show does not accept both userName and groupName
      * Function:AclCommands.showAcl()
      * Meaning:The acl show command requires either a username or a groupname, but not both.
      * Resolution:Check the command syntax and try again.
      * </MAPR_ERROR>
      */

      out.addError(new OutputError(Errno.EOPFAILED,
          "Acl show does not accept both userName and groupName simultaneously")
          .setField(UID_PARAM_NAME));
      output.setOutput(out);
      return output;      
    }
    
    String outputFormat = "short";
    if (isParamPresent(OUTPUT_PARAM_NAME)) {
      outputFormat = getParamTextValue(OUTPUT_PARAM_NAME, 0);
    }
    
    if (outputFormat.equalsIgnoreCase("short")) {
    } else if (outputFormat.equalsIgnoreCase("long")) {
    } else if (outputFormat.equalsIgnoreCase("terse")) {
    } else {
      /**
      * <MAPR_ERROR>
      * Message:Invalid output format <format> expecting short|long|terse
      * Function:AclCommands.showAcl()
      * Meaning:The acl show command takes the following values for the output parameter: short, long, or terse.
      * Resolution:Check the command syntax and try again.
      * </MAPR_ERROR>
      */

      out.addError(new OutputError(Errno.EOPFAILED,
          "Invalid output format " + outputFormat +
          " expecting short|long|terse")
          .setField(OUTPUT_PARAM_NAME));
      output.setOutput(out);
      return output;            
    }

    boolean showClusterAdmin = getParamBooleanValue(SHOW_ADMIN_PARAM_NAME, 0);
    
    // no validation checks for name
    // SendRpc using the binding
    byte[] data;
    AccessControlList acl = null;
    int clusterAdmin = -1;
    
    try {
      SecurityGetAclRequest.Builder getAclBuilder = SecurityGetAclRequest.newBuilder();
      CredentialsMsg creds = getUserCredentials();
      // build the protobuf
      // cluster/volume
      getAclBuilder.setObjectType(objType);
      // object name
      if (name != null) getAclBuilder.setName(name);
      if (isUserPresent || isGroupPresent) {
        SecurityPrincipal.Builder sb = SecurityPrincipal.newBuilder();
        sb.setPrincId(princId);
        getAclBuilder.setPrincipal(sb.build());
      }

      // Credentials
      getAclBuilder.setCreds(creds);
      if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        data =  CLDBRpcCommonUtils.getInstance().sendRequest(
            getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0), 
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProg.SecurityGetAclProc.getNumber(),
            getAclBuilder.build(),
            SecurityGetAclResponse.class);
      } else {
        data =  CLDBRpcCommonUtils.getInstance().sendRequest(
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProg.SecurityGetAclProc.getNumber(),
            getAclBuilder.build(),
            SecurityGetAclResponse.class);
      }
    
      if (data == null) {
    	  out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
    	  output.setOutput(out);
    	  return output;
      }
      // parse the response
      SecurityGetAclResponse aclResp = SecurityGetAclResponse.parseFrom(data);
      int status = aclResp.getStatus();
      if (status != 0) {
        /**
        * <MAPR_ERROR>
        * Message:ACL lookup failed for <object> failed with error message. <error>
        * Function:AclCommands.showAcl()
        * Meaning:The specified ACL could not be found.
        * Resolution:Check the command syntax and try again.
        * </MAPR_ERROR>
        */

        out.addError(new OutputError(Errno.EOPFAILED, 
          "ACL lookup Failure: " + aclResp.getErrorString())
          .setField(OBJECT_PARAM_NAME));
        output.setOutput(out);
        return output;          
      }
      
      acl = aclResp.getAcl();
      if (showClusterAdmin && aclResp.hasClusterAdmin()) clusterAdmin = aclResp.getClusterAdmin();
    } catch (InvalidProtocolBufferException e) {
        throw new CLIProcessingException(
            "InvalidProtocolBufferException " + "Exception", e);
    } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      /**
      * <MAPR_ERROR>
      * Message:Error while trying to get ACL
      * Function:AclCommands.showAcl()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */

      out.addError(new OutputError(Errno.EOPFAILED, 
          "Error while trying to get ACL"));
        output.setOutput(out);
        return output;
    }
    
    // successfully did a lookup
    List<Integer> unknownUids = new ArrayList<Integer>();
    List<Integer> unknownGids = new ArrayList<Integer>();
    OutputHierarchy outAcl = AclCommands.formatAcl(acl, objType, outputFormat, userInfo, unknownUids, unknownGids, clusterAdmin);
    for(OutputError e:outAcl.getOutputErrors()) {
      LOG.info("Error in output:" + e.toString());  
      out.addError(e);
    }
    for (OutputNode a : outAcl.getOutputNodes()) {
      LOG.info("Output Node:" + a.toJSONString());
      out.addNode(a);
    }
    if ((unknownGids.size() > 0) ||
        (unknownUids.size() > 0)) {
      // some uids/gids are unknown
      // print an info message warning the user

      String missingUids = "";
      if (unknownUids.size() > 0) {
        missingUids = "uid(s) (" + formatList(unknownUids) + ")";
      }
      String missingGids = "";
      if (unknownGids.size() > 0) {
        if (unknownUids.size() > 0) missingGids = " and ";
        missingGids = missingGids + "gid(s) (" + formatList(unknownGids) + ")";
      }
      String hostname = getHostName();
      if (hostname == null) hostname = "localhost";
      String warnMsg = "Warning: The "
                          + missingUids
                          + missingGids
                          + " used on the cluster do not have an appropriate account on "
                          + hostname
                          + ". This indicates a possible mis-configuration on "
                          + hostname
                          + ". Please check with your cluster administrator";
      out.addMessage(warnMsg);
    }
    output.setOutput(out);
    return output;
  }

  // not used
  static List<AclEntry> getAclElements(String perms, UnixUserGroupHelper uInfo, boolean isUser) {
    List<String> permList = new ArrayList<String>();
    if (perms.contains(AclCommands.MULTI_ARG_SEP)) {
      permList.addAll(Arrays.asList(perms.split(
                                  AclCommands.MULTI_ARG_SEP)));
    } else {
      permList.add(perms);
    }
    
    List<AclEntry> result = new ArrayList<AclEntry>();

    for (String s : permList) {
      if (!s.contains(AclCommands.ALLOW_MASK_SEP)) {
        return null;
      }
      List<String> userPerm = new ArrayList<String>();
      userPerm.addAll(Arrays.asList(s.split(AclCommands.ALLOW_MASK_SEP)));
      if (userPerm.size() > 2) {
        return null;
      }
      String principal = (String)userPerm.get(0);
      int principalId = 0;
      if (principal.equalsIgnoreCase("allUsers")) {
        principalId = -1;
      } else if (isUser) {
        principalId = uInfo.getUserId(principal);
      } else {
        principalId = uInfo.getGroupId(principal);
        principalId = 0x80000000 | principalId;
      }
      String allowMaskStr = (String)userPerm.get(1);
      int allowMask = Integer.decode(allowMaskStr);

      AclEntry.Builder entry = AclEntry.newBuilder().setDeny(0).setAllow(allowMask);
      entry.setPrincipal(SecurityPrincipal.newBuilder().setPrincId(principalId));
      result.add(entry.build());
    }

    return result;
  }

  private static int convertToId (String principal, UnixUserGroupHelper uInfo, boolean isUser)
  {
    int principalId = 0;
    if (principal.equalsIgnoreCase("allUsers")) {
      principalId = -1;
    } else if (isUser) {
      principalId = uInfo.getUserId(principal);
    } else {
      principalId = uInfo.getGroupId(principal);
      principalId = 0x80000000 | principalId;
    }

    return principalId;
  }

  

  public static List<AclEntry> actionsToAcls (List<String> perms,
                                UnixUserGroupHelper uInfo,
                                SecureObjectType objType,
                                boolean isUser, 
                                OutputHierarchy out)
  {
    List<AclEntry> result = new ArrayList<AclEntry>();
    List<String> permissions = new ArrayList<String>();
    for (String perm : perms) {
      // split the permission string based on space as
      // UI sends it as a single space separated string.
      // For CLI, this code path ends up copying "perms" list
      // into "permissions" list.
      Collections.addAll(permissions, perm.split(" "));
    }

    for (String perm: permissions) {
      List<String> userPerm = Arrays.asList (perm.split(AclCommands.ALLOW_MASK_SEP));

      String principal = userPerm.get(0);
      int principalId;
      try {
        principalId = convertToId(principal, uInfo, isUser);  
      } catch(SecurityException se) {
        try {
          principalId = Integer.parseInt(principal);
          if(!isUser) {
            principalId = 0x80000000 | principalId;
          }
        } catch (NumberFormatException ne) {
          /**
           * <MAPR_ERROR>
           * Message:<Exception thrown while mask conversion>.
           * Function:AclCommands.actionsToAcls()
           * Meaning:Invalid user name/id
           * Resolution:Check the user name or id passed and pass valid ones.
           * </MAPR_ERROR>
           */
          out.addError(new OutputError(Errno.ENOUSER,
            "Error for user " + principal + ":" + Errno.toString(Errno.ENOUSER))
            .setField(OBJECT_PARAM_NAME));
          return null;
        }
      }
      
      int allowMask;

      if (userPerm.size() == 1) {
        // format -user sathya
        //   means remove user sathya
        allowMask = 0;
      } else {
        try {
          String actions = userPerm.get(1);
          allowMask = SecurityCommandHelper.convertActionsToMask (actions, AclCommands.MULTI_ARG_SEP, objType);
        } catch (Exception e) {
          /**
           * <MAPR_ERROR>
           * Message:<Exception thrown while mask conversion>.
           * Function:AclCommands.actionsToAcls()
           * Meaning:Invalid user actions
           * Resolution:Check the user actions passed and fix them to pass valid ones.
           * </MAPR_ERROR>
           */
          LOG.error(e.getMessage());
          out.addError(new OutputError(Errno.EINVAL,
            "Error for user " + principal + ":" + e.getMessage())
            .setField(OBJECT_PARAM_NAME));
          return null;
        }
      }

      AclEntry.Builder entry = AclEntry.newBuilder().setAllow(allowMask);
      entry.setPrincipal(SecurityPrincipal.newBuilder().setPrincId(principalId));
      result.add(entry.build());
    }

    return result;
  }
  
  CommandOutput setAcl() throws CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    SecurityModifyAclRequest.Builder aclRequestBuilder = SecurityModifyAclRequest.newBuilder();
    
    /*
     * set -type cluster|volume -name <name> -perms set|remove -u <uid> -g <gid> -a <mask> -h a.b.c.d -p <port> 
     */
    
    // is this a cluster or a volume ACL command?
    String objectType = getParamTextValue(OBJECT_TYPE_PARAM_NAME, 0);
    SecureObjectType objType;
    
    if (objectType.equalsIgnoreCase("cluster")) {
      objType = SecureObjectType.OBJECT_TYPE_CLUSTER;
    } else if (objectType.equalsIgnoreCase("volume")) {
      objType = SecureObjectType.OBJECT_TYPE_VOLUME;
    } else {
      // return an error
      /**
      * <MAPR_ERROR>
      * Message:Acl show failed with invalid object type <type>. Valid object types are cluster|volume.
      * Function:AclCommands.setAcl()
      * Meaning:The {{acl show}} command requires a {{type}} of {{cluster}} or {{volume}}.
      * Resolution:Check the command syntax and try again.
      * </MAPR_ERROR>
      */

      out.addError(new OutputError(Errno.EOPFAILED, 
          "Acl set failed with invalid object type " + objectType
          + ". Valid object types are cluster|volume.")
        .setField(OBJECT_TYPE_PARAM_NAME));
      output.setOutput(out);
      return output;
    }
    
    AccessControlList.Builder aclBuilder = AccessControlList.newBuilder();
    boolean principalPresent = false;
    
    if (isParamPresent(UID_PARAM_NAME)) {
      // read the usernames and add the list to the protobuf
      principalPresent = true;
      //aclString = getParamTextValue(UID_PARAM_NAME, 0);

      ProcessedInput.Parameter param = input.getParameterByName (UID_PARAM_NAME);
      List<String> values = param.getParamValues();

      List<AclEntry> elements = AclCommands.actionsToAcls(values, userInfo, objType, true, 
        out);
      if (elements == null) {
        String validPerms = objType == SecureObjectType.OBJECT_TYPE_CLUSTER ?
          SecurityCommandHelper.ClusterPerms: SecurityCommandHelper.VolumePerms;
        out.addError (new OutputError(Errno.EOPFAILED,
                "Acl set failed - invalid list of user permissions, " +
                  "valid permissions are: " + validPerms)
            .setField(UID_PARAM_NAME));
        output.setOutput(out);
        return output;
      }
      aclBuilder.addAllAcl(elements);
    }
    
    if (isParamPresent(GID_PARAM_NAME)) {
      principalPresent = true;
      //aclString = getParamTextValue(GID_PARAM_NAME, 0);
      ProcessedInput.Parameter param = input.getParameterByName (GID_PARAM_NAME);
      List<String> values = param.getParamValues();
      
      //List<AclEntry> elements = AclCommands.getAclElements(aclString, userInfo, false);
      List<AclEntry> elements = AclCommands.actionsToAcls(values, userInfo, objType, false, out);
      if (elements == null) {
        String validPerms = objType == SecureObjectType.OBJECT_TYPE_CLUSTER ?
          SecurityCommandHelper.ClusterPerms: SecurityCommandHelper.VolumePerms;
        out.addError(new OutputError(Errno.EOPFAILED,
                "Acl set failed - invalid list of group permissions, " +
                    "valid permissions are: " + validPerms)
            .setField(GID_PARAM_NAME));
        output.setOutput(out);
        return output;
      }
      aclBuilder.addAllAcl(elements);
    }
    
    String name = null;
    if (isParamPresent(OBJECT_PARAM_NAME)) {
      name = getParamTextValue(OBJECT_PARAM_NAME, 0);
    }
    
    if ((name == null) && (objType == SecureObjectType.OBJECT_TYPE_VOLUME)) { 
      /**
      * <MAPR_ERROR>
      * Message:Acl set failed - volume name missing
      * Function:AclCommands.setAcl()
      * Meaning:The acl set command requires a volume name when the type is volume.
      * Resolution:Check the command syntax and try again.
      * </MAPR_ERROR>
      */

      out.addError(new OutputError(Errno.EOPFAILED,
            "Acl set failed - volume name missing")
      .setField(OBJECT_PARAM_NAME));
      output.setOutput(out);
      return output;      
    }

    if ((name != null) && (objType == SecureObjectType.OBJECT_TYPE_CLUSTER)) {
      out.addError(new OutputError(Errno.EOPFAILED,
          "Acl set for a cluster does not require a name")
        .setField(OBJECT_PARAM_NAME));
      output.setOutput(out);
      return output;
    }

    byte[] data;
    
    try {
      CredentialsMsg creds = getUserCredentials();
      // build the protobuf
      // cluster/volume flag
      aclRequestBuilder.setObjectType(objType);
      // object name
      if (name != null) aclRequestBuilder.setName(name);
      // acl
      aclRequestBuilder.setAcl(aclBuilder);
      aclRequestBuilder.setCreds(creds);
      
      if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM) ) {
      data = CLDBRpcCommonUtils.getInstance().sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0), 
          Common.MapRProgramId.CldbProgramId.getNumber(),
          CLDBProg.SecurityModifyAclProc.getNumber(),
          aclRequestBuilder.build(), SecurityModifyAclResponse.class);
      } else {
    	  data = CLDBRpcCommonUtils.getInstance().sendRequest( 
    	          Common.MapRProgramId.CldbProgramId.getNumber(),
    	          CLDBProg.SecurityModifyAclProc.getNumber(),
    	          aclRequestBuilder.build(), SecurityModifyAclResponse.class);
      }
      if (data == null) {
    	  out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
    	  output.setOutput(out);
    	  return output;
      }
      // parse the response
      SecurityModifyAclResponse resp = SecurityModifyAclResponse.parseFrom(data);
      int status = resp.getStatus();
      if (status != 0) {
        /**
        * <MAPR_ERROR>
        * Message:ACL modification failed for <object> failed with error message. <error>
        * Function:AclCommands.setAcl()
        * Meaning:An error occurred.
        * Resolution:Check the command syntax and try again.
        * </MAPR_ERROR>
        */

        out.addError(new OutputError(Errno.EOPFAILED, 
          "ACL modification failed for " + objectType + 
          " failed with error message. " + resp.getErrorString())
          .setField(OBJECT_PARAM_NAME));
        output.setOutput(out);
        return output;          
      }
      
      // command was successful
    } catch (InvalidProtocolBufferException e) {
        throw new CLIProcessingException(
            "InvalidProtocolBufferException " + "Exception", e);
    } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      /**
      * <MAPR_ERROR>
      * Message:Error while trying to modify ACL
      * Function:AclCommands.setAcl()
      * Meaning:An error occurred.
      * Resolution:Check the command syntax and try again.
      * </MAPR_ERROR>
      */

      out.addError(new OutputError(Errno.EOPFAILED, 
          "Error while trying to modify ACL"));
        output.setOutput(out);
        return output;
    }
    
    output.setOutput(out);
    return output;
  }
  
  CommandOutput editAcl() throws CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    
    SecurityModifyAclRequest.Builder aclRequestBuilder = SecurityModifyAclRequest.newBuilder();
    
    /*
     * edit -type cluster|volume -name <name> -user <user>:<permissions> 
     */
    
    // is this a cluster or a volume ACL command?
    String objectType = getParamTextValue(OBJECT_TYPE_PARAM_NAME, 0);
    SecureObjectType objType;
    
    if (objectType.equalsIgnoreCase("cluster")) {
      objType = SecureObjectType.OBJECT_TYPE_CLUSTER;
    } else if (objectType.equalsIgnoreCase("volume")) {
      objType = SecureObjectType.OBJECT_TYPE_VOLUME;
    } else {
      // return an error
      /**
      * <MAPR_ERROR>
      * Message:Acl show failed with invalid object type <type>. Valid object types are cluster|volume.
      * Function:AclCommands.editAcl()
      * Meaning:The {{acl edit}} command requires a {{type}} of {{cluster}} or {{volume}}.
      * Resolution:Check the command syntax and try again.
      * </MAPR_ERROR>
      */

      out.addError(new OutputError(Errno.EOPFAILED, 
        "Acl edit failed with invalid object type " + objectType + 
        ". Valid object types are cluster|volume.")
        .setField(OBJECT_TYPE_PARAM_NAME));
      output.setOutput(out);
      return output;
    }
    
    AccessControlList.Builder aclBuilder = AccessControlList.newBuilder();
    List<AclEntry> elements = null;
    boolean principalPresent = false;
    
    if (isParamPresent(UID_PARAM_NAME)) {
      // read the usernames and add the list to the protobuf
      principalPresent = true;
      //aclString = getParamTextValue(UID_PARAM_NAME, 0);

      ProcessedInput.Parameter param = input.getParameterByName (UID_PARAM_NAME);
      List<String> values = param.getParamValues();

      elements = AclCommands.actionsToAcls(values, userInfo, objType, true, out);
      if (elements == null) {
        String validPerms = objType == SecureObjectType.OBJECT_TYPE_CLUSTER ?
          SecurityCommandHelper.ClusterPerms: SecurityCommandHelper.VolumePerms;
        out.addError (new OutputError(Errno.EOPFAILED,
          "Acl edit failed - invalid list of user permissions, " +
          "valid permissions are: " + validPerms)
          .setField(UID_PARAM_NAME));
        output.setOutput(out);
        return output;
      }
      aclBuilder.addAllAcl(elements);
    }
    
    if (isParamPresent(GID_PARAM_NAME)) {
      principalPresent = true;
      //aclString = getParamTextValue(GID_PARAM_NAME, 0);
      ProcessedInput.Parameter param = input.getParameterByName (GID_PARAM_NAME);
      List<String> values = param.getParamValues();
      
      //List<AclEntry> elements = AclCommands.getAclElements(aclString, userInfo, false);
      elements = AclCommands.actionsToAcls(values, userInfo, objType, false, out);
      if (elements == null) {
        String validPerms = objType == SecureObjectType.OBJECT_TYPE_CLUSTER ?
          SecurityCommandHelper.ClusterPerms: SecurityCommandHelper.VolumePerms;
        out.addError(new OutputError(Errno.EOPFAILED,
          "Acl edit failed - invalid list of group permissions, " +
          "valid permissions are: " + validPerms)
          .setField(GID_PARAM_NAME));
        output.setOutput(out);
        return output;
      }
      aclBuilder.addAllAcl(elements);
    }
    
    if(!principalPresent) {
    	out.addError (new OutputError(Errno.EOPFAILED,
        "Acl edit failed - no user/group specified, " +
        "use -user/-group option to specify the acls to be changed "));
      output.setOutput(out);
      return output;
    }
    
    String name = null;
    if (isParamPresent(OBJECT_PARAM_NAME)) {
      name = getParamTextValue(OBJECT_PARAM_NAME, 0);
    }
    
    if ((name == null) && (objType == SecureObjectType.OBJECT_TYPE_VOLUME)) { 
      /**
      * <MAPR_ERROR>
      * Message:Acl edit failed - volume name missing
      * Function:AclCommands.editAcl()
      * Meaning:The acl edit command requires a volume name when the type is volume.
      * Resolution:Check the command syntax and try again.
      * </MAPR_ERROR>
      */

      out.addError(new OutputError(Errno.EOPFAILED,
            "Acl edit failed - volume name missing")
      .setField(OBJECT_PARAM_NAME));
      output.setOutput(out);
      return output;      
    }

    if ((name != null) && (objType == SecureObjectType.OBJECT_TYPE_CLUSTER)) {
      out.addError(new OutputError(Errno.EOPFAILED,
          "Acl edit for a cluster does not require a name")
        .setField(OBJECT_PARAM_NAME));
      output.setOutput(out);
      return output;
    }

    byte[] data;
    
    try {
      CredentialsMsg creds = getUserCredentials();
      // build the protobuf
      // cluster/volume flag
      aclRequestBuilder.setObjectType(objType);
      // object name
      if (name != null) aclRequestBuilder.setName(name);
      // acl
      aclRequestBuilder.setAcl(aclBuilder);
      aclRequestBuilder.setCreds(creds);
      aclRequestBuilder.setEditFlag(true);
      
      if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM) ) {
      data = CLDBRpcCommonUtils.getInstance().sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0), 
        Common.MapRProgramId.CldbProgramId.getNumber(),
        CLDBProg.SecurityModifyAclProc.getNumber(),
        aclRequestBuilder.build(), SecurityModifyAclResponse.class);
      } else {
    	  data = CLDBRpcCommonUtils.getInstance().sendRequest( 
    	    Common.MapRProgramId.CldbProgramId.getNumber(),
    	    CLDBProg.SecurityModifyAclProc.getNumber(),
    	    aclRequestBuilder.build(), SecurityModifyAclResponse.class);
      }
      if (data == null) {
    	  out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
    	  output.setOutput(out);
    	  return output;
      }
      // parse the response
      SecurityModifyAclResponse resp = SecurityModifyAclResponse.parseFrom(data);
      int status = resp.getStatus();
      if (status != 0) {
        /**
        * <MAPR_ERROR>
        * Message:ACL modification failed for <object> failed with error message. <error>
        * Function:AclCommands.editAcl()
        * Meaning:An error occurred.
        * Resolution:Check the command syntax and try again.
        * </MAPR_ERROR>
        */
        if (resp.hasErrorString()) {
          out.addError(new OutputError(status,
            resp.getErrorString()));
        } else {
          out.addError(new OutputError(Errno.EOPFAILED, 
            "Unknown error while ACL modification for " + objectType)
            .setField(OBJECT_PARAM_NAME));
        }
        output.setOutput(out);
        return output;          
      }
      
      // command was successful
    } catch (InvalidProtocolBufferException e) {
        throw new CLIProcessingException(
            "InvalidProtocolBufferException " + "Exception", e);
    } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      /**
      * <MAPR_ERROR>
      * Message:Error while trying to modify ACL
      * Function:AclCommands.editAcl()
      * Meaning:An error occurred.
      * Resolution:Check the command syntax and try again.
      * </MAPR_ERROR>
      */

      out.addError(new OutputError(Errno.EOPFAILED, 
        "Error while trying to modify ACL"));
        output.setOutput(out);
        return output;
    }
    
    output.setOutput(out);
    return output;
  }
  
  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    if (!super.validateInput()) {
      CommandOutput output = new CommandOutput();
      OutputHierarchy out = new OutputHierarchy();
      output.setOutput(out);
      return output;
    }
    userInfo = new UnixUserGroupHelper();

    String command = cliCommand.getCommandName();
    if (command.equalsIgnoreCase("show")) {
      return showAcl();
    }
    
    if (command.equalsIgnoreCase("set")) {
      return setAcl();
    }
    
    if (command.equalsIgnoreCase("edit")) {
      return editAcl();
    }
    
    return new TextCommandOutput(("Acl command failed: unknown command " + 
        command + " received.").getBytes());
  }
  
  @Override
  public String getCommandUsage() {
    return usageStr;
  }
}
