package com.mapr.baseutils.zookeeper;

import java.io.IOException;
import java.util.List;

import org.apache.log4j.Logger;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.data.Stat;

/**
 * Class to take care of zookeeper handling utilities with "connection loss" and "session expiration" handling to do:
 * getData
 * check for znode existence
 * getChildren
 * @author yufeldman
 *
 */
public class ZKUtils {

    public static final int ZK_RETRIALS = 3;
    public static final int TIMEOUT = 30 * 1000;
    public static final int TIMEOUT_SERVER = 30 * 1000; // lesser timeout to allow faster notification
   // public static final String SERVICE_MASTER_NODE = "$master$";
   // BUG 5227 reverting this one back to allow job client to be released first, so it would be backward compatible
   // and fix this bug (changing to $master$) at later release
    public static final String SERVICE_MASTER_NODE = "master";
    
    private static final Logger LOG = Logger.getLogger(ZKUtils.class);
    
    /**
     * Helper method to get data from ZK with retrial logic in case of connection loss
     * @param s_zk
     * @param nodeName
     * @param watcher
     * @param stat
     * @param retrials
     * @return
     * @throws KeeperException
     * @throws InterruptedException
     */
    public static byte[] getData(ZooKeeper s_zk, String nodeName, Watcher watcher, Stat stat, int retrials) 
        throws KeeperException, InterruptedException, ZKClosedException {
                try {
                        return s_zk.getData(nodeName, watcher, stat);
                } catch (KeeperException.ConnectionLossException e) {
      /**
       * <MAPR_ERROR>
       * Message:Lost connection to ZK while trying to get data. Reconnecting...
       * Function:ZKUtils.getData()
       * Meaning:The connection to ZooKeeper has been lost. 
       * Resolution:No action is necessary. The system will try to re-establish the connection.
       * </MAPR_ERROR>
       */
                        LOG.error("Lost connection to ZK while trying to get data. Reconnecting...");
                        if (retrials > 0 && s_zk.getState().isAlive()) {
                                return getData(s_zk, nodeName, watcher, stat, retrials - 1);
                        } else {
                          if ( !s_zk.getState().isAlive() ){
      /**
       * <MAPR_ERROR>
       * Message:ZK is not alive. Throwing KeeperException after retrials: <retrials>
       * Function:ZKUtils.getData()
       * Meaning:The system is unable to locate ZooKeeper after the reported number of retrials and cannot continue.
       * Resolution:Make sure there is a ZooKeeper quorum and that the network is healthy.
       * </MAPR_ERROR>
       */
                          LOG.error("ZK is not alive. Throwing KeeperException after retrials: " + retrials);
                        }
                        throw new ZKClosedException(e);
                      }
                } catch (KeeperException.SessionExpiredException see) {
      /**
       * <MAPR_ERROR>
       * Message:ZK Session expired. Need to reset ZK completely for node: <nodeName>
       * Function:ZKUtils.getData()
       * Meaning:The ZooKeeper session has expired.
       * Resolution:Stop and restart ZooKeeper on the reported node.
       * </MAPR_ERROR>
       */
                        LOG.error("ZK Session expired. Need to reset ZK completely for node: " + nodeName);
                        throw new ZKClosedException(see);
        } 
    }

    /**
     * Helper method to check for node existence in ZK with retrial logic in case of connection loss
     * @param s_zk
     * @param nodeName
     * @param watcher
     * @param retrials
     * @return
     * @throws KeeperException
     * @throws InterruptedException
     */
    public static Stat checkZKNodeForExistence(ZooKeeper s_zk, String nodeName, Watcher watcher, int retrials)
    throws KeeperException, InterruptedException, ZKClosedException {
            try {
                    return s_zk.exists(nodeName, watcher);
            } catch (KeeperException.ConnectionLossException e) {
      /**
       * <MAPR_ERROR>
       * Message:Lost connection to ZK while trying to get data. Reconnecting...
       * Function:ZKUtils.checkZKNodeForExistence()
       * Meaning:The connection to ZooKeeper has been lost. 
       * Resolution:No action is necessary. The system will try to re-establish the connection.
       * </MAPR_ERROR>
       */
                    LOG.error(Thread.currentThread().getId() + " Lost connection to ZK while trying to get data. Reconnecting...");
                    if (retrials > 0 && s_zk.getState().isAlive()) {
                            return checkZKNodeForExistence(s_zk, nodeName, watcher, retrials - 1);
                    } else {
                    if ( !s_zk.getState().isAlive() ){
      /**
       * <MAPR_ERROR>
       * Message:ZK is not alive. Throwing KeeperException after retrials: <retrials>
       * Function:ZKUtils.checkZKNodeForExistence()
       * Meaning:The system is unable to locate ZooKeeper after the reported number of retrials and cannot continue.
       * Resolution:Make sure there is a ZooKeeper quorum and that the network is healthy.
       * </MAPR_ERROR>
       */
                      LOG.error("ZK is not alive. Throwing KeeperException after retrials: " + retrials);
                    }
                    throw new ZKClosedException(e);
                  }
            } catch (KeeperException.SessionExpiredException see) {
      /**
       * <MAPR_ERROR>
       * Message:ZK Session expired. Need to reset ZK completely for node: <nodeName>
       * Function:ZKUtils.checkZKNodeForExistence()
       * Meaning:The ZooKeeper session has expired.
       * Resolution:Reset the ZooKeeper client.
       * </MAPR_ERROR>
       */
                    LOG.error("ZK Session expired. Need to reset ZK completely for node: " + nodeName);
                    throw new ZKClosedException(see);
            } 
    }

    /**
     * Helper method to get znode children from ZK with retrial logic in case of connection loss
     * @param s_zk
     * @param nodeName
     * @param watcher
     * @param retrials
     * @return
     * @throws KeeperException
     * @throws InterruptedException
     */
  public static List<String> getZkNodeChildren(ZooKeeper s_zk, String nodeName, Watcher watcher, int retrials)
  throws KeeperException, InterruptedException, ZKClosedException {
          try {
                  return s_zk.getChildren(nodeName, watcher);
          } catch (KeeperException.ConnectionLossException e) {
      /**
       * <MAPR_ERROR>
       * Message:Lost connection to ZK while trying to get data. Reconnecting...
       * Function:ZKUtils.getZkNodeChildren()
       * Meaning:The connection to ZooKeeper has been lost. 
       * Resolution:No action is necessary. The system will try to re-establish the connection.
       * </MAPR_ERROR>
       */
                  LOG.error(Thread.currentThread().getId() + " Lost connection to ZK while trying to get data. Reconnecting...");
                  if (retrials > 0 && s_zk.getState().isAlive() ) {
                          return getZkNodeChildren(s_zk, nodeName, watcher, retrials - 1);
                  } else {
                    
                    if ( !s_zk.getState().isAlive() ){
                      LOG.error("ZK is not alive. Throwing KeeperException after retrials: " + retrials);
                    }
                    throw new ZKClosedException(e);
                  }
          } catch (KeeperException.SessionExpiredException see) {
      /**
       * <MAPR_ERROR>
       * Message:ZK Session expired. Need to reset ZK completely for node: <nodeName>
       * Function:ZKUtils.getZkNodeChildren()
       * Meaning:The ZooKeeper session has expired.
       * Resolution:Stop and restart ZooKeeper on the reported node.
       * </MAPR_ERROR>
       */
                  LOG.error("ZK Session expired. Need to reset ZK completely for node: " + nodeName);
                  throw new ZKClosedException(see);
          } 
  }

  /**
   * Method to reset ZK - in case of KeeperException.SessionExpiredException
   * that exception should actually remove znode from ZK - at least for nodes, as they are going to be
   * EPHEMERAL_SEQUENTIAL, so we need to make sure that even new "master" is in place because of this
   * (worst case master got SessionExpiredException)
   * we do not restart service on any other node if it is running on "this one"
   * Using writelock to prevent "reading" threads to get data that is in inconsistent state - during reset
   * @param s_zk
   * @param watcher
   * @throws IOException
   */
  public static void resetZookeeper(String zkAddresses, ZooKeeper s_zk, Watcher watcher) throws IOException {
    try {
        if ( s_zk != null ) {
              try {
                    s_zk.close();
              } catch (InterruptedException e) {
                if ( LOG.isInfoEnabled()) {
                      LOG.info("Interrupted Exception during ZK closure", e);
                }
              }
        }
        s_zk = null;
        s_zk = new ZooKeeper(zkAddresses, TIMEOUT, watcher);
    } finally {
    }
  }
}

