package com.mapr.baseutils.metric;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Types;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;

import com.mapr.fs.proto.clustermetrics.ClusterMetricsProto.Counters;
import com.mapr.fs.proto.clustermetrics.ClusterMetricsProto.MetricAttributes;
import com.mapr.fs.proto.clustermetrics.ClusterMetricsProto.Pair;
import com.mapr.fs.proto.clustermetrics.ClusterMetricsProto.TaskMetric;

/**
 * Task related DB Operations
 * @author yufeldman
 *
 */
public class TaskMetricRepo extends MetricRepoBase implements MetricRepo {

  private static final Logger LOG = Logger.getLogger(TaskMetricRepo.class);
  private static final String taskInsertSQL = 
    "insert into metrics.TASK (TASK_ID, JOB_ID, TYPE, STATUS, TIME_STARTED) values(?, ?, ?, ?, ?)";
  private static final String taskUpdateSQL = 
    "update metrics.TASK set SUCCESS_ATTEMPT_ID = IFNULL(SUCCESS_ATTEMPT_ID, ?), " +
  "INPUT_SPLIT_LOCATIONS = IFNULL(INPUT_SPLIT_LOCATIONS, ?), " +
  "INPUT_SPLIT_INFO = IFNULL(INPUT_SPLIT_INFO, ?), " +
  "STATUS = IFNULL(STATUS, ?), " +
  "TIME_STARTED = IFNULL(TIME_STARTED, ?), " +
  "TIME_FINISHED = IFNULL(TIME_FINISHED, ?) " +
  "WHERE TASK_ID = ?";
  
  private TaskMetric taskMetric;
  public TaskMetricRepo(TaskMetric taskMetric ) {
    this.taskMetric = taskMetric;
  }
  @Override
  public List<Counters> prepareTransactions() {
    List<Counters> metricTransacions = new ArrayList<Counters>();

    return metricTransacions;
  }
  @Override
  public MetricAttributes getMetricAttributes() {
    if ( taskMetric.hasAttributes()) {
      return taskMetric.getAttributes();
    }
    return null;
  }
  @Override
  public MetricType getMetricType() {
    return MetricType.TASK;
  }
  @Override
  public void executeDBOperations(Connection dbConnection) throws SQLException {
    if ( taskMetric.hasFinalMetric() ) {
      TaskMetric.FinalMetric tFinalMetric = taskMetric.getFinalMetric();
      String taskId = tFinalMetric.getId();
      PreparedStatement taskInsertPrepStmt = dbConnection.prepareStatement(taskInsertSQL);
      try {
      taskInsertPrepStmt.setString(1, taskId);
      taskInsertPrepStmt.setString(2, tFinalMetric.getJobId());
      taskInsertPrepStmt.setString(3, tFinalMetric.getType());
      if ( tFinalMetric.hasStatus() ) {
         taskInsertPrepStmt.setString(4, tFinalMetric.getStatus());
      } else {
         taskInsertPrepStmt.setNull(4, Types.CHAR);
      }
      if ( tFinalMetric.hasStartTime() ) {
         taskInsertPrepStmt.setLong(5, tFinalMetric.getStartTime());
      } else {
        taskInsertPrepStmt.setNull(5, Types.BIGINT);
      }
      try {
        taskInsertPrepStmt.executeUpdate();
      } catch (SQLException e) {
        if (e.getErrorCode() == 1062) {
        /*
        Error: 1062, SQLSTATE[23000]: Integrity constraint violation
        */
        // We better update record as it exist already 
        // Don't insert start only attributes
        } else {
         LOG.error("Exception while trying to insert Task: ", e);
         throw e;
        }
      }  
      } finally {
        taskInsertPrepStmt.close();
      }
        PreparedStatement taskUpdatePrepStmt = dbConnection.prepareStatement(taskUpdateSQL);
        try {
        if ( tFinalMetric.hasSuccessTaskAttemptId() ) {
          taskUpdatePrepStmt.setString(1, tFinalMetric.getSuccessTaskAttemptId());
        } else {
          taskUpdatePrepStmt.setNull(1, Types.CHAR);
        }
        if ( tFinalMetric.hasInputSplitLocations()) {
          taskUpdatePrepStmt.setString(2, tFinalMetric.getInputSplitLocations());
        } else {
         taskUpdatePrepStmt.setNull(2, Types.CHAR);
        }
        if ( tFinalMetric.hasInputSplitInfo() ) {
          taskUpdatePrepStmt.setString(3, tFinalMetric.getInputSplitInfo());
        } else {
          taskUpdatePrepStmt.setNull(3, Types.CHAR);
        }
        if ( tFinalMetric.hasStatus() ) {
           taskUpdatePrepStmt.setString(4, tFinalMetric.getStatus());
        } else {
           taskUpdatePrepStmt.setNull(4, Types.CHAR);
        }
        if ( tFinalMetric.hasStartTime() ) {
           taskUpdatePrepStmt.setLong(5, tFinalMetric.getStartTime());
        } else {
           taskUpdatePrepStmt.setNull(5, Types.BIGINT);
        }
        if ( tFinalMetric.hasEndTime() ) {
           taskUpdatePrepStmt.setLong(6, tFinalMetric.getEndTime());
        } else {
           taskUpdatePrepStmt.setNull(6, Types.BIGINT);
        }

        taskUpdatePrepStmt.setString(7, taskId);
        try {
          taskUpdatePrepStmt.executeUpdate();
        } catch (SQLException e) {
          if (e.getErrorCode() == 1062) {
          /*
          Error: 1062, SQLSTATE[23000]: Integrity constraint violation
          */
          // We better update record as it exist already 
          // Don't insert start only attributes
          } else {
           LOG.error("Exception while trying to update Task: ", e);
           throw e;
          }
        }
        } finally {
          taskUpdatePrepStmt.close();
        }
    }
  }
  
  @Override
  public List<Pair> prepareEvents() {
    List<Pair> events = new ArrayList<Pair>();
    if ( taskMetric.hasEventMetric() && taskMetric.hasAttributes() ) {
      TaskMetric.EventMetric taEventMetric = taskMetric.getEventMetric();
      if ( taEventMetric.hasStatus() ) {
        events.add(Pair.newBuilder().setName("TSTATUS").setValue(taEventMetric.getStatus()).build());
      }
    }
    return events;
  }

}
