package com.mapr.baseutils.metric;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Types;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;

import com.mapr.fs.proto.clustermetrics.ClusterMetricsProto.Counters;
import com.mapr.fs.proto.clustermetrics.ClusterMetricsProto.MetricAttributes;
import com.mapr.fs.proto.clustermetrics.ClusterMetricsProto.Pair;
import com.mapr.fs.proto.clustermetrics.ClusterMetricsProto.TaskAttemptMetric;

/** TaskAttempt related DB operations
 * 
 * @author yufeldman
 *
 */
public class TaskAttemptMetricRepo extends MetricRepoBase implements MetricRepo {

  private static final Logger LOG = Logger.getLogger(TaskAttemptMetricRepo.class);
  private static final String insertTaskAtSQL = 
    "insert into metrics.TASK_ATTEMPT " +
    "(TASK_ATTEMPT_ID, TASK_ID, TYPE, TIME_STARTED, HOST, LOG_LOCATION, STATUS) " +
    "values(?, ?, ?, ?, ?, ?, ?)";
  
  private static final String updateTaskAtSQL = 
    "update metrics.TASK_ATTEMPT " +
    "set SHUFFLE_FINISHED = IFNULL(SHUFFLE_FINISHED, ?), " +
    "SORT_FINISHED = IFNULL(SORT_FINISHED,?), " +
    "TIME_FINISHED = IFNULL(TIME_FINISHED, ?), " +
    "STATUS = IFNULL(STATUS, ?) " +
    "WHERE TASK_ATTEMPT_ID = ?";
  
  private TaskAttemptMetric taskAMetric;
  public TaskAttemptMetricRepo(TaskAttemptMetric taskAMetric) {
    this.taskAMetric = taskAMetric;
  }
  
  @Override
  public List<Counters> prepareTransactions() {
    List<Counters> metricTransactions = new ArrayList<Counters>();
    if ( taskAMetric.hasAttributes() ) {
      if ( taskAMetric.hasProgressPercent() ) {
        metricTransactions.add(Counters.newBuilder().setCounterName("PROGRESS").setCounterValue((long) taskAMetric.getProgressPercent()).build());
      }
      metricTransactions.addAll(taskAMetric.getCountersList());
    }
    return metricTransactions;
  }

  @Override
  public MetricAttributes getMetricAttributes() {
    if ( taskAMetric.hasAttributes() ) {
      return taskAMetric.getAttributes();
    }
    return null;
  }

  @Override
  public MetricType getMetricType() {
    return MetricType.TASKATTEMPT;
  }

  @Override
  public void executeDBOperations(Connection dbConnection) throws SQLException {
    if ( taskAMetric.hasFinalMetric() ) {      
      TaskAttemptMetric.FinalMetric taFinalMetric = taskAMetric.getFinalMetric();
      String taskAttemptId = taFinalMetric.getId();
      String taskId = taFinalMetric.getTaskId();
      PreparedStatement taskAttInsertPrepStmt = dbConnection.prepareStatement(insertTaskAtSQL);
      try {
      taskAttInsertPrepStmt.setString(1, taskAttemptId);
      taskAttInsertPrepStmt.setString(2, taskId);
      if ( taFinalMetric.hasType()) {
       taskAttInsertPrepStmt.setString(3, taFinalMetric.getType());
      } else {
       taskAttInsertPrepStmt.setNull(3, Types.CHAR);
      }

      if ( taFinalMetric.hasStartTime() ) {
       taskAttInsertPrepStmt.setLong(4, taFinalMetric.getStartTime());
      } else {
       taskAttInsertPrepStmt.setNull(4, Types.BIGINT);
      }

      if ( taFinalMetric.hasHost() ) {
        taskAttInsertPrepStmt.setString(5, taFinalMetric.getHost());
      } else {
        taskAttInsertPrepStmt.setNull(5, Types.CHAR);
      }

      if ( taFinalMetric.hasLogLocation() ) {
        taskAttInsertPrepStmt.setString(6, taFinalMetric.getLogLocation());
      } else {
        taskAttInsertPrepStmt.setNull(6, Types.CHAR);
      }

      if ( taFinalMetric.hasState() ) {
         taskAttInsertPrepStmt.setString(7, taFinalMetric.getState());
      } else {
         taskAttInsertPrepStmt.setNull(7, Types.CHAR);
      }

      try {
        taskAttInsertPrepStmt.executeUpdate();
      } catch (SQLException e) {
        if (e.getErrorCode() == 1062) {
        /*
        Error: 1062, SQLSTATE[23000]: Integrity constraint violation
        */
        // We better update record as it exist already 
        // Don't insert start only attributes
        } else {
         LOG.error("Exception while trying to insert TaskAttempt: ", e);
         throw e;
        }
      }
      } finally {
        taskAttInsertPrepStmt.close();
      }
      
      PreparedStatement taskAttUpdatePrepStmt = dbConnection.prepareStatement(updateTaskAtSQL);
      try {
      boolean isUpdateNeeded = false;
      if ( taFinalMetric.hasShuffleFinished()) {
        isUpdateNeeded = true;
        taskAttUpdatePrepStmt.setLong(1, taFinalMetric.getShuffleFinished());
      } else {
        taskAttUpdatePrepStmt.setNull(1, Types.BIGINT);
      }
      if ( taFinalMetric.hasSortFinished()) {
        isUpdateNeeded = true;
        taskAttUpdatePrepStmt.setLong(2, taFinalMetric.getSortFinished());
      } else {
        taskAttUpdatePrepStmt.setNull(2, Types.BIGINT);
      }
      if ( taFinalMetric.hasFinishTime() ) {
        isUpdateNeeded = true;
        taskAttUpdatePrepStmt.setLong(3, taFinalMetric.getFinishTime());
      } else {
        taskAttUpdatePrepStmt.setNull(3, Types.BIGINT);
      }
      if ( taFinalMetric.hasState() ) {
        isUpdateNeeded = true;
        taskAttUpdatePrepStmt.setString(4, taFinalMetric.getState());
      } else {
        taskAttUpdatePrepStmt.setNull(4, Types.CHAR);
      }

      if ( isUpdateNeeded ) {
        taskAttUpdatePrepStmt.setString(5, taskAttemptId);
        try {
          taskAttUpdatePrepStmt.executeUpdate();
        } catch (SQLException e) {
          if (e.getErrorCode() == 1062) {
        /*
        Error: 1062, SQLSTATE[23000]: Integrity constraint violation
        */
        // We better update record as it exist already 
        // Don't insert start only attributes
          } else {
            LOG.error("Exception while trying to update TaskAttempt: ", e);
            throw e;
          }
        }
      }
      } finally {
        taskAttUpdatePrepStmt.close();
      }
    }
    
  }

  @Override
  public List<Pair> prepareEvents() {
    List<Pair> events = new ArrayList<Pair>();
    
    if ( taskAMetric.hasEventMetric() && taskAMetric.hasAttributes() ) {
      TaskAttemptMetric.EventMetric taEventMetric = taskAMetric.getEventMetric();
      
      int i = 0;
      for ( String diagnosticInfo : taEventMetric.getDiagnosticInfoList()) {
        events.add(Pair.newBuilder().setName("DIAGINFO"+(i++)).setValue(diagnosticInfo).build());
      }
      if ( taEventMetric.hasState() ) {
        events.add(Pair.newBuilder().setName("STATE").setValue(taEventMetric.getState()).build());
      }
      if ( taEventMetric.hasStateString() ) {
        events.add(Pair.newBuilder().setName("STATESTRING").setValue(taEventMetric.getStateString()).build());
      }
    }
    return events;
  }

}
