package com.mapr.baseutils.metric;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;

import com.mapr.fs.proto.clustermetrics.ClusterMetricsProto.Counters;
import com.mapr.fs.proto.clustermetrics.ClusterMetricsProto.MetricAttributes;
import com.mapr.fs.proto.clustermetrics.ClusterMetricsProto.NodeMetric;
import com.mapr.fs.proto.clustermetrics.ClusterMetricsProto.Pair;

/**
 * Node related DB operations
 * @author yufeldman
 *
 */
public class NodeMetricRepo extends MetricRepoBase implements MetricRepo {

  private static final Logger LOG = Logger.getLogger(NodeMetricRepo.class);
  
  private static final String NODE_INSERT_SQL = 
    "insert into metrics.NODE " +
    "(NODE_ID, HOSTNAME, HOSTID, CLUSTER_ID) " +
    "values(?, ?, ?, ?)";
  
  private NodeMetric nodeMetric;
  public NodeMetricRepo(NodeMetric nodeMetric) {
    this.nodeMetric = nodeMetric;
  }
  @Override
  public List<Counters> prepareTransactions() {
    List<Counters> metricTransactions = new ArrayList<Counters>();

    if ( nodeMetric.hasMetricAttrs()  ) {
      if ( nodeMetric.hasCpuUptime()) 
        metricTransactions.add(Counters.newBuilder().setCounterName("CPUUPTIME").setCounterValue(nodeMetric.getCpuUptime()).build());   
      if ( nodeMetric.hasMemoryUsed())
        metricTransactions.add(Counters.newBuilder().setCounterName( "MEMORYUSED").setCounterValue(nodeMetric.getMemoryUsed()).build());
      if (nodeMetric.hasCpuIdle())
        metricTransactions.add(Counters.newBuilder().setCounterName("CPUIDLE").setCounterValue(nodeMetric.getCpuIdle()).build());
      if ( nodeMetric.hasReadOps())
        metricTransactions.add(Counters.newBuilder().setCounterName("READOPS").setCounterValue(nodeMetric.getReadOps()).build());
      if ( nodeMetric.hasReadKBytes())
        metricTransactions.add(Counters.newBuilder().setCounterName("READKBYTES").setCounterValue(nodeMetric.getReadKBytes()).build());
      if ( nodeMetric.hasWriteOps())
        metricTransactions.add(Counters.newBuilder().setCounterName("WRITEOPS").setCounterValue(nodeMetric.getWriteOps()).build());
      if ( nodeMetric.hasWriteKBytes())
        metricTransactions.add(Counters.newBuilder().setCounterName("WRITEKBYTES").setCounterValue(nodeMetric.getWriteKBytes()).build());
      if ( nodeMetric.hasBytesIn())
        metricTransactions.add(Counters.newBuilder().setCounterName("BYTESIN").setCounterValue(nodeMetric.getBytesIn()).build());
      if ( nodeMetric.hasBytesOut()) 
        metricTransactions.add(Counters.newBuilder().setCounterName("BYTESOUT").setCounterValue(nodeMetric.getBytesOut()).build());
      if ( nodeMetric.hasServerUsedSizeMB())
        metricTransactions.add(Counters.newBuilder().setCounterName( "SERVUSEDSIZEMB").setCounterValue(nodeMetric.getServerUsedSizeMB()).build());
      if ( nodeMetric.hasServerAvailableSizeMB())
        metricTransactions.add(Counters.newBuilder().setCounterName( "SERVAVAILSIZEMB").setCounterValue(nodeMetric.getServerAvailableSizeMB()).build());
      if (nodeMetric.hasRpcCount())
        metricTransactions.add(Counters.newBuilder().setCounterName("RPCCOUNT").setCounterValue(nodeMetric.getRpcCount()).build());
      if (nodeMetric.hasRpcInBytes())
        metricTransactions.add(Counters.newBuilder().setCounterName("RPCINBYTES").setCounterValue(nodeMetric.getRpcInBytes()).build());
      if ( nodeMetric.hasRpcOutBytes()) 
        metricTransactions.add(Counters.newBuilder().setCounterName("RPCOUTBYTES").setCounterValue(nodeMetric.getRpcOutBytes()).build());
      if ( nodeMetric.hasCpuNice())
        metricTransactions.add(Counters.newBuilder().setCounterName("CPUNICE").setCounterValue(nodeMetric.getCpuNice()).build());
      if ( nodeMetric.hasCpuSystem())
        metricTransactions.add(Counters.newBuilder().setCounterName("CPUSYSTEM").setCounterValue(nodeMetric.getCpuSystem()).build());
      if ( nodeMetric.hasCpuUser())
        metricTransactions.add(Counters.newBuilder().setCounterName("CPUUSER").setCounterValue(nodeMetric.getCpuUser()).build());
      if ( nodeMetric.hasMemoryCached())
        metricTransactions.add(Counters.newBuilder().setCounterName("MEMORYCACHED").setCounterValue(nodeMetric.getMemoryCached()).build());
      if ( nodeMetric.hasMemoryShared())
        metricTransactions.add(Counters.newBuilder().setCounterName("MEMORYSHARED").setCounterValue(nodeMetric.getMemoryShared()).build());
      if ( nodeMetric.hasMemoryBuffers())
        metricTransactions.add(Counters.newBuilder().setCounterName("MEMORYBUFFERS").setCounterValue(nodeMetric.getMemoryBuffers()).build());
      if ( nodeMetric.hasSwapFree())
        metricTransactions.add(Counters.newBuilder().setCounterName( "SWAPFREE").setCounterValue(nodeMetric.getSwapFree()).build());
      if (nodeMetric.hasPktsIn())
        metricTransactions.add(Counters.newBuilder().setCounterName("PKTSIN").setCounterValue(nodeMetric.getPktsIn()).build());
      if ( nodeMetric.hasPktsOut())
        metricTransactions.add(Counters.newBuilder().setCounterName( "PKTSOUT").setCounterValue(nodeMetric.getPktsOut()).build());
      if (nodeMetric.hasLoadOnePerc())
        metricTransactions.add(Counters.newBuilder().setCounterName("LOAD1PERCENT").setCounterValue(nodeMetric.getLoadOnePerc()).build());
      if ( nodeMetric.hasLoadFivePerc())
        metricTransactions.add(Counters.newBuilder().setCounterName("LOAD5PERCENT").setCounterValue(nodeMetric.getLoadFivePerc()).build());
      if ( nodeMetric.hasLoadFifteenPerc())
        metricTransactions.add(Counters.newBuilder().setCounterName( "LOAD15PERCENT").setCounterValue(nodeMetric.getLoadFifteenPerc()).build());
      if ( nodeMetric.hasProcRun())
        metricTransactions.add(Counters.newBuilder().setCounterName("PROCRUN").setCounterValue(nodeMetric.getProcRun()).build());
      if ( nodeMetric.hasTtmapused())
        metricTransactions.add(Counters.newBuilder().setCounterName("TTMAPUSED").setCounterValue(nodeMetric.getTtmapused()).build());
      if ( nodeMetric.hasTtreduceused())
        metricTransactions.add(Counters.newBuilder().setCounterName("TTREDUCEUSED").setCounterValue(nodeMetric.getTtreduceused()).build());
    }
    return metricTransactions;
  }
  @Override
  public MetricAttributes getMetricAttributes() {
    if ( nodeMetric.hasMetricAttrs()) {
      return nodeMetric.getMetricAttrs();
    }
    return null;
  }
  @Override
  public MetricType getMetricType() {
    return MetricType.NODE;
  }
  @Override
  public void executeDBOperations(Connection dbConnection) throws SQLException {
    if ( !nodeMetric.hasMetricAttrs() ) {
      LOG.error("Missing Node Metric Attributes. Node data won't be inserted");
      return;
    }
    MetricAttributes metricAttrs = nodeMetric.getMetricAttrs();
    if ( !metricAttrs.hasNodeId() ) {
      LOG.error("Node ID is missing from request. Node data won't be inserted");
      return;
    }
    PreparedStatement nodeInsertPrepStmt = dbConnection.prepareStatement(NODE_INSERT_SQL);
    try {
      if ( nodeMetric.hasFinalMetric() ) {
          NodeMetric.FinalMetric nFinalMetric = nodeMetric.getFinalMetric();
          nodeInsertPrepStmt.setString(1, metricAttrs.getNodeId());
        if ( nFinalMetric.hasHostname()) {
          nodeInsertPrepStmt.setString(2, nFinalMetric.getHostname());
        }
        if ( nFinalMetric.hasHostid()) {
          nodeInsertPrepStmt.setString(3, nFinalMetric.getHostid());
        }
        if ( nFinalMetric.hasCluster() && !nFinalMetric.getCluster().isEmpty()) {
          nodeInsertPrepStmt.setString(4, nFinalMetric.getCluster());
        } else {
          nodeInsertPrepStmt.setString(4, getClusterId());
        }
        try {
          nodeInsertPrepStmt.executeUpdate();
        } catch (SQLException e) {
          if (e.getErrorCode() == 1062) {
            /*
            Error: 1062, SQLSTATE[ 23000]: Integrity constraint violation
            */
            // We better update record as it exist already 
            // Don't insert start only attributes
          } else {
            LOG.error("Exception while trying to insert Node data: ", e);
          }
        }
      } 
    } finally {
      nodeInsertPrepStmt.close();
    }
 }
  
  @Override
  public List<Pair> prepareEvents() {
    List<Pair> events = new ArrayList<Pair>();
    
    if ( nodeMetric.hasEventMetric() && nodeMetric.hasMetricAttrs() ) {
      NodeMetric.EventMetric nEventMetric = nodeMetric.getEventMetric();
      if ( nEventMetric.hasDiskCount() ) 
        events.add(Pair.newBuilder().setName("DISKCOUNT").setValue(Integer.toString(nEventMetric.getDiskCount())).build());
      if ( nEventMetric.hasCpuCount()) 
        events.add(Pair.newBuilder().setName("CPUCOUNT").setValue(Integer.toString(nEventMetric.getCpuCount())).build());
      if ( nEventMetric.hasTotalMB())
        events.add(Pair.newBuilder().setName("TOTALMEMORY").setValue(Long.toString(nEventMetric.getTotalMB())).build());
      if ( nEventMetric.hasRootFull() ) 
        events.add(Pair.newBuilder().setName("ROOTFULLSTATUS").setValue(Boolean.toString(nEventMetric.getRootFull())).build());
      if ( nEventMetric.hasOptMaprFull() ) 
        events.add(Pair.newBuilder().setName("MAPRFULLSTATUS").setValue(Boolean.toString(nEventMetric.getOptMaprFull())).build());
      if ( nEventMetric.hasCorePresent() ) 
        events.add(Pair.newBuilder().setName("COREPRESENTSTATUS").setValue(Boolean.toString(nEventMetric.getCorePresent())).build());
      if ( nEventMetric.hasNicCount() ) 
        events.add(Pair.newBuilder().setName("TOTALNICS").setValue(Integer.toString(nEventMetric.getNicCount())).build());
      if ( nEventMetric.hasFaileddisks())
        events.add(Pair.newBuilder().setName( "FAILEDDISKS").setValue(Integer.toString(nEventMetric.getFaileddisks())).build());       
      if ( nEventMetric.hasMaprdiskCount())
        events.add(Pair.newBuilder().setName( "MAPRDISKCOUNT").setValue(Integer.toString(nEventMetric.getMaprdiskCount())).build()); 
      if ( nEventMetric.hasServerCapacitySizeMB())  
        events.add(Pair.newBuilder().setName( "SERVCAPACITYSIZEMB").setValue(Long.toString(nEventMetric.getServerCapacitySizeMB())).build());  
      if ( nEventMetric.hasSwapTotal())
        events.add(Pair.newBuilder().setName( "SWAPTOTAL").setValue(Long.toString(nEventMetric.getSwapTotal())).build());   
      if ( nEventMetric.hasTtmapslots())
        events.add(Pair.newBuilder().setName( "TTMAPSLOTS").setValue(Long.toString(nEventMetric.getTtmapslots())).build()); 
      if ( nEventMetric.hasTtreduceslots())
        events.add(Pair.newBuilder().setName( "TTREDUCESLOTS").setValue(Long.toString(nEventMetric.getTtreduceslots())).build()); 

      events.addAll(nEventMetric.getConfServiceList());
      events.addAll(nEventMetric.getRunningServiceList());
      events.addAll(nEventMetric.getServiceFailedList());
      events.addAll(nEventMetric.getServiceStoppedList());
      events.addAll(nEventMetric.getAlarmsList());
    }
    return events;
  }

}
