package com.mapr.baseutils.metric;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Types;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;

import com.mapr.fs.proto.clustermetrics.ClusterMetricsProto.Counters;
import com.mapr.fs.proto.clustermetrics.ClusterMetricsProto.JobMetric;
import com.mapr.fs.proto.clustermetrics.ClusterMetricsProto.MetricAttributes;
import com.mapr.fs.proto.clustermetrics.ClusterMetricsProto.Pair;

/**
 * Job related DB operations
 * @author yufeldman
 *
 */
public class JobMetricRepo extends MetricRepoBase implements MetricRepo {

  private static final Logger LOG = Logger.getLogger(JobMetricRepo.class);
  
  private static final String JOB_INSERT_SQL = 
    "insert into metrics.JOB (JOB_ID, JOB_NAME, PARENT_JOB_ID, USER_SUBMITTED, TIME_SUBMITTED, CLUSTER_ID) " +
    "values(?, ?, ?, ?, ?, ?)";
    
  private static final String JOB_UPDATE_SQL = 
    "update metrics.JOB set TIME_STARTED = IFNULL(TIME_STARTED, ?), " +
    "TIME_FINISHED = IFNULL(TIME_FINISHED, ?) WHERE JOB_ID = ?";
  
  private static final String JOB_ATTR_INSERT_SQL = 
    "insert into metrics.JOB_ATTRIBUTES " +
  		"(ATTR_ID, JOB_ID, ATTR_NAME, ATTR_VALUE) " +
  		"values(default, ?, ?, ?)";
  
  private JobMetric jobMetric;
 
  public JobMetricRepo(JobMetric jobMetric) {
    this.jobMetric = jobMetric;
  }
  
  @Override
  public List<Counters> prepareTransactions() {
    List<Counters> metricTransactions = new ArrayList<Counters>();
    if ( jobMetric.hasAttributes() ) {
      return jobMetric.getTimeBasedCountersList();
    }
    return metricTransactions;
  }

  @Override
  public MetricAttributes getMetricAttributes() {
    if ( jobMetric.hasAttributes() ) {
      return jobMetric.getAttributes();
    }
    return null;
  }

  @Override
  public MetricType getMetricType() {
    return MetricType.JOB;
  }

  @Override
  public void executeDBOperations(Connection dbConnection) throws SQLException {
    if ( dbConnection == null || dbConnection.isClosed() ) {
      LOG.error("DBConnection is null or closed. No Job Metric related DB operation will be performed");
      return;
    }
    if ( jobMetric.hasFinalMetric() ) {
      JobMetric.FinalMetric jFinalMetric = jobMetric.getFinalMetric();
      String jobId = jFinalMetric.getId();
      PreparedStatement jobInsertPrepStmt = dbConnection.prepareStatement(JOB_INSERT_SQL);
      try {
      // real meat of DB insertion/update
      jobInsertPrepStmt.setString(1, jobId);
      if ( jFinalMetric.hasName() ) {
        jobInsertPrepStmt.setString(2, jFinalMetric.getName());
      } else {
        jobInsertPrepStmt.setNull(2, Types.VARCHAR);
      }
      if ( jFinalMetric.hasParentJobId() ) {
        jobInsertPrepStmt.setString(3, jFinalMetric.getParentJobId() );
      } else {
        jobInsertPrepStmt.setNull(3, Types.VARCHAR);
      }
      if ( jFinalMetric.hasUser() ) {
        jobInsertPrepStmt.setString(4, jFinalMetric.getUser());
      } else {
        jobInsertPrepStmt.setNull(4, Types.VARCHAR);
      }
      if ( jFinalMetric.hasSubmitTime() ) {
        jobInsertPrepStmt.setLong(5, jFinalMetric.getSubmitTime());
      } else {
        jobInsertPrepStmt.setNull(5, Types.BIGINT);
      }
      jobInsertPrepStmt.setString(6, getClusterId());
      try {
       jobInsertPrepStmt.executeUpdate();
      } catch (SQLException e) {
        if ( e.getErrorCode() == 1062 ) {
          // Error: 1062, SQLSTATE[23000]: Integrity constraint violation
        } else {
          throw e;
        }
      }
      } finally {
        jobInsertPrepStmt.close();
      }
      PreparedStatement jobUpdatePrepStmt = dbConnection.prepareStatement(JOB_UPDATE_SQL);
      try {
        if ( jFinalMetric.hasStartTime() ) {
          jobUpdatePrepStmt.setLong(1, jFinalMetric.getStartTime());
        } else {
          jobUpdatePrepStmt.setNull(1, Types.BIGINT);
        }
        if ( jFinalMetric.hasFinishTime() ) {
          jobUpdatePrepStmt.setLong(2, jFinalMetric.getFinishTime());
        } else {
          jobUpdatePrepStmt.setNull(2, Types.BIGINT);
        }
        jobUpdatePrepStmt.setString(3, jobId);
        try {
          jobUpdatePrepStmt.executeUpdate();
        } catch (SQLException e) {
          LOG.error("Exception while trying to update Job data", e);
          throw e;
        }
      } finally {
        jobUpdatePrepStmt.close();
      }
        insertJObAttributes(dbConnection, jFinalMetric);
     }    
  }

  private void insertJObAttributes(Connection dbConnection, JobMetric.FinalMetric jFinalMetric) throws SQLException{
    PreparedStatement jobAttrInsertPrepStmt = dbConnection.prepareStatement(JOB_ATTR_INSERT_SQL);
    try {
      jobAttrInsertPrepStmt.setString(1, jFinalMetric.getId());
      if ( jFinalMetric.hasMapTasksCount() ) {
        jobAttrInsertPrepStmt.setString(2, "MAP_TASK_COUNT");
        jobAttrInsertPrepStmt.setLong(3, jFinalMetric.getMapTasksCount());
        jobAttrInsertPrepStmt.addBatch();
      }
      if ( jFinalMetric.hasReduceTasksCount()) {
        jobAttrInsertPrepStmt.setString(2, "REDUCE_TASK_COUNT");
        jobAttrInsertPrepStmt.setLong(3, jFinalMetric.getReduceTasksCount());
        jobAttrInsertPrepStmt.addBatch();
      }
      if ( jFinalMetric.hasInputSize() ) {
        jobAttrInsertPrepStmt.setString(2, "INPUT_SIZE");
        jobAttrInsertPrepStmt.setLong(3, jFinalMetric.getInputSize());
        jobAttrInsertPrepStmt.addBatch();
      }
      if ( jFinalMetric.hasPriority() ) {
        jobAttrInsertPrepStmt.setString(2, "PRIORITY");
        jobAttrInsertPrepStmt.setString(3, jFinalMetric.getPriority());
        jobAttrInsertPrepStmt.addBatch();
      }
      for ( String group : jFinalMetric.getGroupList()) {
        jobAttrInsertPrepStmt.setString(2, "GROUP");
        jobAttrInsertPrepStmt.setString(3, group);
        jobAttrInsertPrepStmt.addBatch();
      }
      for (  String label : jFinalMetric.getLabelList() ) {
        jobAttrInsertPrepStmt.setString(2, "LABEL");
        jobAttrInsertPrepStmt.setString(3, label);
        jobAttrInsertPrepStmt.addBatch();
      }
      for ( Counters counter : jFinalMetric.getSummarizedCountersList()  ) {
        jobAttrInsertPrepStmt.setString(2, counter.getCounterName());
        jobAttrInsertPrepStmt.setLong(3, counter.getCounterValue());
        jobAttrInsertPrepStmt.addBatch();
      }
      jobAttrInsertPrepStmt.executeBatch();
    } finally {
      jobAttrInsertPrepStmt.close();
    }
  }

  @Override
  public List<Pair> prepareEvents() {
    List<Pair> events = new ArrayList<Pair>();
    
    if ( jobMetric.hasEventMetric() && jobMetric.hasAttributes() ) {
      JobMetric.EventMetric jEventMetric = jobMetric.getEventMetric();
      if ( jEventMetric.hasStatus() ) 
        events.add(Pair.newBuilder().setName("STATUS").setValue(jEventMetric.getStatus()).build());
      if ( jEventMetric.hasPriority())
        events.add(Pair.newBuilder().setName("PRIORITY").setValue(jEventMetric.getPriority()).build());
      if ( jEventMetric.hasFailureInfo() )
        events.add(Pair.newBuilder().setName("FAILINFO").setValue(jEventMetric.getFailureInfo()).build());
      if ( jEventMetric.hasSchedulingInfo() ) 
        events.add(Pair.newBuilder().setName("SCHEDINFO").setValue(jEventMetric.getSchedulingInfo()).build());
    }
    return events;
  }
}
