package com.mapr.baseutils.fsrpcutils;

import java.util.Hashtable;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.log4j.Logger;

import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.MessageLite;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.fs.Rpc;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.ContainerLookupRequest;
import com.mapr.fs.cldb.proto.CLDBProto.ContainerLookupResponse;
import com.mapr.fs.cldb.proto.CLDBProto.ContainerOnFileServerFailRequest;
import com.mapr.fs.cldb.proto.CLDBProto.ContainerOnFileServerFailResponse;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Common.Server;
import com.mapr.fs.proto.Security.ServerKeyType;
import com.mapr.fs.proto.Security.CredentialsMsg;
import com.mapr.security.MutableInt;
import com.mapr.fs.proto.Security.TicketAndKey;
import com.mapr.security.JNISecurity;
import com.mapr.security.MaprSecurityException;
import com.mapr.security.Security;

public class FSRpcUtils {
  private static final Logger LOG = Logger.getLogger(FSRpcUtils.class);
  
  
  private String clusterName;
  private CredentialsMsg creds;
  private int maxRetry;
  private int srcPort;
  private ServerKeyType keyToUse;
  Hashtable <Integer, CidInfo> cidHash = new Hashtable<Integer, CidInfo>();
  ReentrantLock lock = new ReentrantLock();
  
  public FSRpcUtils(String clusterName, CredentialsMsg creds, int maxRetry,
    ServerKeyType keyToUse, int srcPort) {
    this.clusterName = clusterName;
    this.creds = creds;
    this.maxRetry = maxRetry;
    this.keyToUse = keyToUse;
    this.srcPort = srcPort;
  }

  public boolean CHasMServer(int cid) {
    CidInfo cidInfo = null;
    cidInfo = cidHash.get(cid);
    if (cidInfo == null) {
      return false;
    } else {
      if (cidInfo.cInfo == null) {
        return false;
      } else {
        return cidInfo.cInfo.hasMServer();
      }
    }
  }

  public int SendRequestToCid(int cid, boolean toMaster,
      int programid,
      int procid,
      MessageLite req, /* in parameter: msg to be sent */
      GetMsgStatus getMsgStatus /* interface to return status of msg */) throws Exception {

    int nretry = 0;
    int status = 0;
    int sleepTimeInMillis = 5 * 1000;
    
    lock.lock();
    CidInfo cidInfo = null;
    try {
      cidInfo = cidHash.get(cid);
      if (cidInfo == null) {
        cidInfo = new CidInfo(cid);
        cidHash.put(cid, cidInfo);
      }
    } finally {
      lock.unlock();
    }

    do {
      cidInfo.rwLock.readLock().lock();
      if (cidInfo.needUpdate(toMaster)) {

        // cidInfo needs to be updated take the exclusive 
        // lock on cidinfo and then try to update it
        cidInfo.rwLock.readLock().unlock();
        cidInfo.rwLock.writeLock().lock();

        // check again if we need updated
        status = 0;
        if (cidInfo.needUpdate(toMaster)) {
          status = UpdateCidInfo(cid, cidInfo);
        }

        cidInfo.rwLock.writeLock().unlock();

        // If the update cid info successfully complete
        // but we still need update then it means that
        // CLDB didn't return the required master / replica 
        // node for this container. So return EAGAIN without
        // retrying the caller is supposed to call again later
        if ((status == 0) && cidInfo.needUpdate(toMaster)) {
          LOG.info("Could not get the required server for cid " + cid
            + " returning EAGAIN caller should retry after waiting for sometime. "
            + " current cid info" + cidInfo);
          return Errno.EAGAIN;
        }

        // Could not talk to CLDB -- retry few times before giving up
        if ((status == Errno.EAGAIN) && (nretry < maxRetry)) {
          nretry++;
          try {
            Thread.sleep(sleepTimeInMillis);
          } catch (InterruptedException e) {
          }
          continue;
        }
        
        if (status != 0) {
          LOG.error("Failed to update cid info for container "
              + cid + " failing with error " + status);
          return status;
        }
        // try again the needUpdate condition after taking the read lock 
        continue;  
      }

      /*
       * Here we have shared lock and verified that cidInfo has required info
       * to send the request to file server
       */
      status = SendRequestToServer(cid, toMaster, 
          programid, procid, req, getMsgStatus);

      cidInfo.rwLock.readLock().unlock();
      if (status != 0) {
        LOG.error("SendRequestToServer failed to send request cid " + cid
            + " programid " + programid + " procid " + procid + " status " + status);
      }
     
      // If error is EAGAIN then retry after sometime 
      if (status == Errno.EAGAIN) {
        if (nretry < maxRetry) {
          ++nretry;
          try {
            Thread.sleep(sleepTimeInMillis);
          } catch (InterruptedException e) {
          }
          continue;
        }
      }

      if (status != 0) {
        LOG.error("SendRequestToServer failing the send request cid " + cid
          + " programid " + programid + " procid " + procid 
          + " status " + status + " nretry " + nretry
		  + " cidinfo " + cidInfo + " toMaster " + toMaster);
      }
      return status;
      
    } while (nretry < maxRetry);
    LOG.error("SendRequestToServer failing the send request cid " + cid
        + " programid " + programid + " procid " + procid 
        + " status " + status + " nretry " + nretry
		+ " cidinfo " + cidInfo + " toMaster " + toMaster);

    return status;
  }

  private int SendRequestToServer(
      int cid, boolean toMaster,
      int programid, int procid,
      MessageLite req, 
      GetMsgStatus getMsgStatus) throws Exception {
    
    CidInfo cidInfo = cidHash.get(cid);
    Server s = null;
    if (toMaster) {
      s = cidInfo.getMasterServer();
    } else {
      s = cidInfo.getNextReplicaServer();
    }
    
    int ips[] = new int [s.getIpsCount()];
    for (int i = 0; i < s.getIpsCount(); ++i) {
      ips[i] = s.getIps(i).getHost();
    }
    long binding;

    if (JNISecurity.IsSecurityEnabled(clusterName)) {
      //Check whether the required ticket is present. If it is not
      //present, load the maprserverticket file. 
      MutableInt mutableerr = new MutableInt();
      TicketAndKey ticket = Security.GetTicketAndKeyForCluster(keyToUse,
        clusterName, mutableerr);
      if (ticket == null) {
        String ticketFile = CLDBRpcCommonUtils.getInstance().
          getPathToServerTicketFile();
        int err = Security.SetTicketAndKeyFile(ticketFile);
        if (err != 0) {
          LOG.error("Error " + err + " in loading " + ticketFile);
        }
        ticket = Security.GetTicketAndKeyForCluster(keyToUse,
          clusterName, mutableerr);
        if (ticket == null) {
          LOG.error("Error " + mutableerr.GetValue() + " in finding ticket for cluster: " + clusterName);
          return 0;
        }
      }
    }

    if (srcPort > 0) {
      binding = Rpc.createBindingForIpsWithSrcPort(ips, s.getIps(0).getPort(),
      srcPort, clusterName, keyToUse.getNumber());
    } else {
      binding = Rpc.createBindingForIps(ips, s.getIps(0).getPort(), clusterName, keyToUse.getNumber());
    }
    byte [] data = null;
    
    try {
      data = Rpc.sendRequest(binding, programid, procid, req);

      if (data == null) {
        LOG.info("Request send failed cid " + cid 
            + " program id " + programid
            + " procedureid " + procid
            + " server " + Utils.PrintServerIpAddress(s));
        // report failure of server and return EAGAIN
        ReportServerFailure(cid, toMaster, s);
        return Errno.EAGAIN;
      }
      getMsgStatus.init(data);
    } catch (Exception e) {
      LOG.error("Request send failed cid " + cid 
          + " program id " + programid
          + " procedureid " + procid
          + " server " + Utils.PrintServerIpAddress(s), e);
     
      // Parsing of response data failed fail the operation 

      // If mfs does not have the server key yet then also exception will be
      // received. So retry the op. If it is persistent error, it is handled
      // in the callers.
      if (e instanceof MaprSecurityException) {
        return Errno.EAGAIN;
      }
      return Errno.EBADF;
    }
    
    int status = getMsgStatus.GetStatus();
    if (LOG.isDebugEnabled()) {
      LOG.debug("Request send cid " + cid 
          + " program id " + programid
          + " procedureid " + procid
          + " server " + Utils.PrintServerIpAddress(s)
          + " returned status " + status);
    }
    if (status != 0) {

      if ((status == Errno.ENODEV) ||
          (status == Errno.ENAVAIL) ||
          (status == Errno.ESRCH)) {
        status = ReportServerFailure(cid, toMaster, s);

        // If successfully reported error then return
        // EAGAIN so that caller calls the req again
        if (status == 0) {
          status = Errno.EAGAIN;
        }
        return status;
      }

      // Always return success from here as the error is there in
      // the protobuf->status field
      return 0;
    }
    return 0;
  }

  /*
   * This function should be called with cidInfo rwlock help in readmode
   * For master request it will report the master failure to cldb and 
   * get the new container info.
   * For replica failure it will move to the next node in the servers for
   * the container
   */
  private int ReportServerFailure(int cid, boolean toMaster, Server s) throws Exception {
    int status = 0;
    CidInfo cidInfo = cidHash.get(cid);
    cidInfo.rwLock.readLock().unlock();
    cidInfo.rwLock.writeLock().lock();
    if (toMaster) {
      status = ReportMasterServerFailure(cid, toMaster, s);
    } else {
      cidInfo.incNextReplicaServer();
    }
    cidInfo.rwLock.writeLock().unlock();
    cidInfo.rwLock.readLock().lock();
    return status;
  }
  
  private int ReportMasterServerFailure(int cid, boolean toMaster, Server s) throws Exception {

    ContainerOnFileServerFailRequest.Builder req;
    req = ContainerOnFileServerFailRequest.newBuilder();
    ContainerOnFileServerFailResponse resp;
    byte [] data = null;
    
    req.setContainerId(cid);
    req.setCreds(creds);
    req.setServerId(s.getServerId());
    
    data = CLDBRpcCommonUtils.getInstance().sendRequest(clusterName,
              Common.MapRProgramId.CldbProgramId.getNumber(),
              CLDBProto.CLDBProg.ContainerOnFileServerFailProc.getNumber(),
              req.build(), ContainerOnFileServerFailResponse.class,
              keyToUse, srcPort);

    // Could not communicate with CLDB -- return ECOMM
    // Caller should retry few times and then fail
    if (data == null) {
      LOG.error("Can not talk to CLDB while reporting " 
          + "fileserver failure response "
          + " clustername " + clusterName + " cid " + cid);
      return Errno.EAGAIN;
    }
    
    try {
      resp = ContainerOnFileServerFailResponse.parseFrom(data);
    } catch (InvalidProtocolBufferException e) {
      LOG.error("Exception in reporting fileserver failure response "
          + " clustername " + clusterName + " cid " + cid, e);
      return Errno.EOPFAILED;
    }

    if (resp.getStatus() != 0) {
      LOG.error("Failed to report fileserver failure "
          + " clustername " + clusterName + " cid " + cid
          + " error " + resp.getStatus());
      return resp.getStatus();
    }
    
    CidInfo cidInfo = cidHash.get(cid);
    cidInfo.UpdateContainerInfo(resp.getContainer());
    if (LOG.isDebugEnabled()) {
      LOG.debug("Updated the container info " + cidInfo);
    }
    return 0;
  }

  /*
   * UpdateCidInfo - This function updates the container info
   * by querying cldb about the container info.
   * This must always be called with cidinfo.rwlock held in 
   * write mode
   */
  private int UpdateCidInfo(int cid, CidInfo cidInfo) throws Exception {

    ContainerLookupRequest.Builder req = ContainerLookupRequest.newBuilder();
    ContainerLookupResponse resp;
    byte [] data = null;
    
    req.addContainerId(cid);
    req.setCreds(creds);
    
    data = CLDBRpcCommonUtils.getInstance().sendRequest(clusterName,
              Common.MapRProgramId.CldbProgramId.getNumber(),
              CLDBProto.CLDBProg.ContainerLookupProc.getNumber(),
              req.build(), ContainerLookupResponse.class,
              keyToUse, srcPort);

    // Could not communicate with CLDB -- return EAGAIN
    // Caller should retry few times and then fail
    if (data == null) {
      return Errno.EAGAIN;
    }
    
    try {
      resp = ContainerLookupResponse.parseFrom(data);
    } catch (InvalidProtocolBufferException e) {
      LOG.error("Exception in process container lookup response "
          + " clustername " + clusterName + " cid " + cid, e);
      return Errno.EOPFAILED;
    }

    if (resp.getStatus() != 0) {
      LOG.error("Failed to get container lookup response "
          + " clustername " + clusterName + " cid " + cid
          + " error " + resp.getStatus());
      return resp.getStatus();
    }
    // Got the container info -- put in the cidInfo
    cidInfo.UpdateContainerInfo(resp.getContainersList().get(0));
    if (LOG.isDebugEnabled()) {
      LOG.debug("Updated the container info " + cidInfo);
    }
    return 0;
  }
}
