package com.mapr.baseutils.cldbutils;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.lang.reflect.Method;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Pattern;

import org.apache.log4j.Logger;

import com.google.protobuf.ByteString;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.MessageLite;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.BaseUtilsHelper;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtilsException;
import com.mapr.fs.Rpc;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.isCLDBMasterRequest;
import com.mapr.fs.cldb.proto.CLDBProto.isCLDBMasterResponse;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Common.IPAddress;
import com.mapr.fs.proto.Security.ServerKeyType;
import com.mapr.fs.proto.Security.TicketAndKey;
import com.mapr.security.JNISecurity;
import com.mapr.security.MaprSecurityException;
import com.mapr.login.client.MapRLoginHttpsClient;
import com.mapr.security.MutableInt;
import com.mapr.security.Security;

public class CLDBRpcCommonUtils {

	private static final Logger LOG = Logger.getLogger(CLDBRpcCommonUtils.class);

	private static final String MAPR_CLUSTER_FILE_NAME = "/conf/mapr-clusters.conf";
	private static final String MAPR_SERVER_TICKET_FILE_NAME = "/conf/maprserverticket";
	private static final int CLDB_PORT_DEFAULT = 7222;
	private static final String CLDB_LOCAL_HOST = "127.0.0.1";

	static Pattern patternIP = Pattern.compile("[0-9]{1,3}.[0-9]{1,3}.[0-9]{1,3}.[0-9]{1,3}");
	// if/when we will support ipv6 we will turn following pattern on
	// static Pattern patternIP6 = Pattern.compile("[0-9a-fA-F]{1,4}(:[0-9a-fA-F]{0,4}){0,7}(/[0-9]{0,3}){0,1}");
	  
	static 
  {
    com.mapr.fs.ShimLoader.load();
  }


	// PLEASE DO NOT make this static. Otherwise I have hard time synchronizing it. 
	// This class is singleton, so you should not have more then one instance of this map anyway
	// If you need to get hold of this map use: CLDBRpcCommonUtils.getInstance().getClusterMap();
	private Map<String, List<IpPort>> clustersMap = new ConcurrentHashMap<String,List<IpPort>>();
	private static CLDBRpcCommonUtils s_instance = new CLDBRpcCommonUtils();
	
	private volatile String defaultClusterName = "default";
  private String origClusterName;
	private Map<String, String> zkConnectStringMap = new HashMap<String, String>();

	/**
	 * Singleton to get file data inited
	 */
	private CLDBRpcCommonUtils() {
		init();
	}
	
	public static CLDBRpcCommonUtils getInstance() {
		return s_instance;
	}
	
	public synchronized String getZkConnect() {
		if ( defaultClusterName == null) {
			LOG.error("Default cluster name is null. Can not get ZK string");
			return null;
		}
		return getZkConnect(defaultClusterName);
	}
	
	public synchronized String getZkConnect(String clusterName) {
		if ( clusterName == null || !clustersMap.containsKey(clusterName) ) {
			LOG.error("Invalid cluster name specified: " + clusterName);
			return null;
		}
		String zkConnectString = zkConnectStringMap.get(clusterName);
		if ( zkConnectString != null ) {
			return zkConnectString;
		} else {
			try {
			byte [] responseData  = sendRequest(clusterName, 
					Common.MapRProgramId.CldbProgramId.getNumber(),
					CLDBProto.CLDBProg.IsCLDBMasterProc.getNumber(),
					isCLDBMasterRequest.newBuilder().build(),
					isCLDBMasterResponse.class);
			if ( responseData == null ) {
				LOG.error("No data is received from any cldb");
				return null;
			}
			try {
				isCLDBMasterResponse response = isCLDBMasterResponse.parseFrom(responseData);
				if ( response.getStatus() != 0 ) {
					LOG.error("Non-valid status received from isCLDBMasterResponse: " + Errno.toString(response.getStatus()));
				}
				zkConnectString = response.getZkconnect();
				zkConnectStringMap.put(clusterName, zkConnectString);
				LOG.info("ZKConnect: " + zkConnectString);
				return zkConnectString;
			} catch (InvalidProtocolBufferException e) {
				LOG.error("Exception while parsing response from isCLDBMasterResponse", e);
				return null;
			}
			} catch (Throwable t) {
				LOG.error("Exception while trying to send RPC to CLDB", t);
				return null;
			}
		}
	}

	/**
	 * Static init during class loading to get all mapr-clusters.conf info
	 * made it protected only for test purposes, should be private otherwise
	 */
	public synchronized void init() {
	  if ( LOG.isDebugEnabled()) {
	    LOG.debug("init");
	  }
	  String tempClusterName = null;

	  Map<String, List<IpPort>> tempMap = new HashMap<String,List<IpPort>>();

		// Get the mapr home directory
		String maprHome = BaseUtilsHelper.getPathToMaprHome();
		
		// at this point path is known, not necessarily it will exist though
		String clusterConfFile = maprHome + MAPR_CLUSTER_FILE_NAME;
	  try {
		  BufferedReader bfr = new BufferedReader(new FileReader(clusterConfFile));
      String strLine;
      
      while ((strLine = bfr.readLine()) != null) {
        // Ignore lines with comments
        if (strLine.matches("^\\s*#.*"))
          continue;        
        String[] tokens = strLine.split("[\\s]+");
        // Ignore badly formed lines
        if (tokens.length < 2) {
          continue;
         } 
        String clusterName = tokens[0];
        List<IpPort> ipList = new ArrayList<IpPort>();          
        for (int i = 1; i < tokens.length; i++) {
	  // Parse Cluster Options if any exist, like security=ON
	  if(tokens[i].contains("=")) {
	    String[] arr = tokens[i].split("=");
	    if((arr.length != 2) ||
	       (JNISecurity.SetClusterOption(clusterName, arr[0], arr[1]) != 0)) {
	       LOG.error("Invalid Conf options:" + tokens[i] + 
	                 " for cluster " + clusterName);
	    }
	    continue;
	  }

          String[] cldbIps = tokens[i].split(";");
          IpPort ipPort = new IpPort();
          for (int j = 0; j < cldbIps.length; j++) {
            // break it by ":"
      	    int port = CLDB_PORT_DEFAULT;
            String[] arr = cldbIps[j].split(":");
      	    String host = arr[0];
      	    // in case both hostname and IP provided. Try to use IP
      	    int index = host.indexOf(',');
      	    if ( index != -1 ) {
      		  if ( index < (host.length() -1 )) {
      			  host = host.substring(index+1);
      		  } else {
      			  host = host.substring(0, index);
      		  }
      	    }
      	    if ( arr.length >=2 ) {
      		  try {
      		   port = Integer.valueOf(arr[1]);
      		  } catch (NumberFormatException nfe) {
      			  LOG.error("Port is not Integer: " + arr[1] + ". Using default CLDB port: " + CLDB_PORT_DEFAULT);
      		  }
      		  // check if port is a valid number
      		  if (port < 0 || port > 65535) {
      			  LOG.error("Port is invalid number: " + port + ". Skipping " + cldbIps[j]);
      			  continue;
      		  }
      	    }
            ipPort.addIPOrHost(host, port);
          }
          if (ipPort.getNumIPs() > 0) {
            ipList.add(ipPort);
          }
        }
        if (ipList.isEmpty()) {
      	  LOG.error("No CLDBs defined for cluster: " + clusterName);
        } else {
         tempMap.put(clusterName, ipList);
         if (tempClusterName == null) {
           defaultClusterName = clusterName;
           tempClusterName = clusterName;
         }
        }
      }

      clustersMap.putAll(tempMap);
      for (Map.Entry<String, List<IpPort>> entry : clustersMap.entrySet()) {
        if (!tempMap.containsKey(entry.getKey())) {
          clustersMap.remove(entry.getKey());
        }
      }
      
	  } catch(FileNotFoundException fex) {
      //Do nothing
	  } catch (Throwable t) {
		  LOG.error("Exception during init", t);
	  }

	  // Mark parsing done for Cluster Conf options
	  // Required to check any error where we call Get before parsing
	  JNISecurity.SetParsingDone();
	  
	  // clusters.conf missing or had no valid entries
	  if (clustersMap.size() == 0) {
      defaultClusterName = "default";
      List<IpPort> ipList = new ArrayList<IpPort>();
      IpPort ipPort = new IpPort();
      ipPort.addIPOrHost(CLDB_LOCAL_HOST, CLDB_PORT_DEFAULT);
      ipList.add(ipPort);
      clustersMap.put(defaultClusterName, ipList);
    }

	}

  public static int ipToInt(String addr) {
    if (addr.equals("localhost"))
      addr = "127.0.0.1";
    String[] addrArray = addr.split("\\.");
    long num = 0;
    for (int i=0;i<addrArray.length;i++) {
        int power = 3-i;
        num += ((Integer.parseInt(addrArray[i])%256 * Math.pow(256,power)));
    }
    return (int)num;
  }

	/**
	 * Get binging based host,port
	 * @param cldbPort
	 * @param cldbHostString
	 * @return
	 * @throws CLDBRpcCommonUtilsException
	 */
  private long init(int cldbPort, List<String> cldbHostString,
      String clusterName, ServerKeyType keyType, 
      int srcPort) throws CLDBRpcCommonUtilsException {

    if ((cldbHostString == null) || (cldbHostString.size() == 0)) {
      return 0;
    }

    int[] cldbIps = new int[cldbHostString.size()];
    int idx = 0;
    for (String cldbIp : cldbHostString) {
      cldbIps[idx] = ipToInt(cldbIp);
      idx++;
    }

    try {
      int port = Rpc.initialize(0, 0, clusterName); // Client
      if (port < 0)
        throw new IOException("Error in RPC init");
    } catch (Exception e) {
      throw new CLDBRpcCommonUtilsException("Exception in Rpc.initialize " + e);
    }
    long binding;

    if (JNISecurity.IsSecurityEnabled(clusterName)) {
      //Check whether the required ticket is present. If it is not
      //present, load the maprserverticket file. 
      MutableInt mutableerr = new MutableInt();
      TicketAndKey ticket = Security.GetTicketAndKeyForCluster(keyType,
          clusterName, mutableerr);
      if (ticket == null) {
        int err = Security.SetTicketAndKeyFile(getPathToServerTicketFile());
        if (err != 0) {
          LOG.error("Error " + err + " in loading " + getPathToServerTicketFile());
        }
        ticket = Security.GetTicketAndKeyForCluster(keyType,
            clusterName, mutableerr);
      }
    }

    if (srcPort > 0) {
      binding = Rpc.createBindingForIpsWithSrcPort(cldbIps, cldbPort, 
          srcPort, clusterName, keyType.getNumber());
    } else {
      binding = Rpc.createBindingForIps(cldbIps, cldbPort, clusterName, keyType.getNumber());
    }
    return binding;
  }

	/**
	 * Main entrance point to send RPC Request to CLDB for default cluster
	 * @param binding
	 * @param programId
	 * @param procedureId
	 * @param request
	 * @param responseClass
	 * @return
	 */
	public byte [] sendRequest(int programId, int  procedureId,
	    MessageLite request, Class<? extends MessageLite> responseClass) 
	        throws Exception
	{
	  return sendRequest(defaultClusterName, programId, procedureId,
	      request, responseClass, ServerKeyType.ServerKey);
	}

	public byte [] sendRequest(int programId, int  procedureId,
	    MessageLite request, Class<? extends MessageLite> responseClass,
            int cldbPort) throws Exception
	{
	  return sendRequest(defaultClusterName, programId, procedureId,
	      request, responseClass, ServerKeyType.ServerKey, 0, cldbPort);
	}

	/**
	 * Main entrance point to send RPC Request to CLDB for a cluster
	 * @param clusterName
	 * @param programId
	 * @param procedureId
	 * @param request
	 * @param responseClass
	 * @return
	 */
	public byte[] sendRequest(String clusterName, int programId, int procedureId,
	    MessageLite request, Class<? extends MessageLite> responseClass)
	        throws Exception
	{
	  return sendRequest(clusterName, programId, procedureId,
	      request, responseClass, ServerKeyType.ServerKey);
	}

	/**
	 * Full method (so far) to take into consideration ServerKeyType
	 * @param clusterName
	 * @param programId
	 * @param procedureId
	 * @param request
	 * @param responseClass
	 * @param keyType
	 * @return
	 */
	 public byte[] sendRequest(String clusterName, int programId, int procedureId,
       MessageLite request, Class<? extends MessageLite> responseClass,
       ServerKeyType keyType) throws Exception
	 {
       return sendRequest(clusterName, programId, procedureId, request,
        responseClass, keyType, 0);
   }

	 public byte[] sendRequest(String clusterName, int programId, int procedureId,
	     MessageLite request, Class<? extends MessageLite> responseClass,
	     ServerKeyType keyType, int srcPort) throws Exception
	 {
           return sendRequest(clusterName, programId, procedureId, request, responseClass,
                    keyType, srcPort, 0);
         }

	 public byte[] sendRequest(String clusterName, int programId, int procedureId,
	     MessageLite request, Class<? extends MessageLite> responseClass,
	     ServerKeyType keyType, int srcPort, int cldbPort) throws Exception
	 {
	   // cluster is not found
	   if ( clustersMap.get(clusterName) == null ) {
	     // try to reinit to see if new cluster came up
	     init();
	     if ( clustersMap.get(clusterName) == null ) {
	       // fail for sure
	       try {
	         Method newBuilder = responseClass.getMethod("newBuilder", new Class[]{});
	         Object returnObject =  newBuilder.invoke(null);
	         Method statusMethod = returnObject.getClass().getMethod("setStatus",  Integer.TYPE);
	         returnObject = statusMethod.invoke(returnObject, Errno.EUCLUSTER);
	         Method build = returnObject.getClass().getMethod("build", new Class[]{});
	         MessageLite buildObject = (MessageLite) build.invoke(returnObject, new Object[]{});
	         byte [] bytes = buildObject.toByteArray();

	         // mapr-clusters.conf file doesn't have entry for this cluster.
	         LOG.error("Unable to reach cluster with name: " + clusterName 
	             +". No entry found in file " + MAPR_CLUSTER_FILE_NAME 
	             + " for cluster " + clusterName 
	             + ". Failing the CLDB RPC with status " 
	             + Errno.EUCLUSTER);

	         return bytes;
	       } catch (Exception e) {
	         LOG.error("Error while trying to construct erroneous response", e);
	       } 
	       LOG.error("Cluster with name: " + clusterName + " is not found. Can not proceed with CLDB RPC");
	       return null;
	     }
	   }
	   if (clustersMap.get(clusterName).isEmpty() ) {
	     // looks like we went through all of the possibilities
	     // try to fetch from file again - may be it had changed
	     init();
	   }

           if (cldbPort > 0) {
             IpPort cldbCredentials = new IpPort();
             cldbCredentials.addIPOrHost(CLDB_LOCAL_HOST, cldbPort);

	     byte [] retBytes = getDataForParticularCLDB(clusterName, cldbCredentials, programId,
	         procedureId,
	         request,
	         responseClass,
	         keyType,
	         srcPort);

             return retBytes;
           }

	   // using iterator as I can remove cldb from list
	   // in order not to hit ConcurrentModificatioEception 
	   // create local copy of the List. May need to use Vector instead of List
	   List<IpPort> clusterCredentials = new ArrayList<IpPort>(clustersMap.get(clusterName));
	   int origSize = clusterCredentials.size();
	   Iterator<IpPort> iter = clusterCredentials.iterator();
	   while ( iter.hasNext()) {
	     IpPort cldbCredentials = iter.next();
	     byte [] retBytes = getDataForParticularCLDB(clusterName, cldbCredentials, programId,
	         procedureId,
	         request,
	         responseClass,
	         keyType,
	         srcPort);
	     if ( retBytes != null ) {
	       if ( clusterCredentials.size() != origSize ) {
	         clustersMap.put(clusterName,clusterCredentials);
	       }
	       return retBytes;
	     } else {
	       iter.remove();
	       LOG.info("Bad CLDB credentials removed: " + cldbCredentials);
	     }
	   }
	   clustersMap.put(clusterName,clusterCredentials);
	   return null;
	   // if we are here - nothing worked, try current default host/port
	   // not sure if this one is a good decision as it may mask absense of correct mapr-clusters.conf on control node 
	 }

	/**
	 * Method to validate a given cluster name 
	 * @param clusterName
	 * @return
	 */
	public boolean isValidClusterName(String clusterName) {
		if (clustersMap.get(clusterName) != null ) {
      return true;
    }
    // Re-read the mapr-cluster.conf file to see if new entry got added
    init();
		if (clustersMap.get(clusterName) != null ) {
      return true;
    }
	  LOG.error("Cluster with name: " + clusterName + " not found");
    return false;
  }

  public String getPathToClustersConfFile() {
    String maprHome = BaseUtilsHelper.getPathToMaprHome();
    return maprHome + MAPR_CLUSTER_FILE_NAME;
  }

  public String getPathToServerTicketFile() {
    String maprHome = BaseUtilsHelper.getPathToMaprHome();
    return maprHome + MAPR_SERVER_TICKET_FILE_NAME;
  }
		
	/**
	 * Helper method to issue a real RPC
	 * It is not very efficient as it is using Reflection
	 * But nothing more useful can be done at this time w/o affecting clients of this code too much
	 * @param clusterName
	 * @param cldbCredentials
	 * @param programId
	 * @param procedureId
	 * @param request
	 * @param responseClass
	 * @return
	 */
	private byte [] getDataForParticularCLDB(String clusterName, IpPort cldbCredentials, int  programId,
		      int  procedureId,
		      MessageLite request,
		      Class<? extends MessageLite> responseClass,
		      ServerKeyType keyType,
          int srcPort) throws Exception {
		try {
			//attempt to authenticate (relevant if using kerberos)
    			MapRLoginHttpsClient loginClient = new MapRLoginHttpsClient();
  			loginClient.quietAuthenticateIfNeeded(clusterName);

			long binding = init(cldbCredentials.getPort(), cldbCredentials.getAddr(), clusterName, keyType, srcPort);
      if (binding == 0)
         return null;

			Method parseFromMethod = responseClass.getMethod("parseFrom", ByteString.class);
			final int MAX_ATTEMPTS = 6;
			int nAttempt = 0;
			byte[] retBytes;
			Integer status = 0;

			do {
				retBytes = Rpc.sendRequest(binding, programId, procedureId, request);
				if (retBytes != null) {
					Object returnObject =  parseFromMethod.invoke(null, ByteString.copyFrom(retBytes));
					Method statusMethod = returnObject.getClass().getMethod("getStatus", new Class[]{});					
					status = (Integer) statusMethod.invoke(returnObject, new Object[]{});
					if ( status == null ) {
						LOG.error("Return Status is not Integer: " + status);
					} else if ( status == Errno.EROFS ) {
						// this is not master cldb and we need master one
						LOG.error(cldbCredentials.toString() + " is READ_ONLY CLDB. Trying another one");
					} else if ( status == Errno.ESRCH ) {
						++nAttempt;
						if (nAttempt == 1) {
							LOG.error(cldbCredentials.toString() + " is attempting to become a master. Retrying !");
						}
						if (nAttempt < MAX_ATTEMPTS) {
							Thread.sleep((2 * nAttempt + 1) * 1000);
						}
					} else {
						return retBytes;
					}
				} else {
					// probably bad one
					LOG.error("No data returned in RPC: " + cldbCredentials.toString() + 
							  ". Continue searching for correct CLDB");
				}
			} while ((retBytes != null) && (status == Errno.ESRCH) && (nAttempt < MAX_ATTEMPTS));
		} catch (CLDBRpcCommonUtilsException e) {
			LOG.error("Exception while trying to bind on: " + cldbCredentials.toString());
		} catch (MaprSecurityException se) {
      LOG.error(se);
      throw se;
		} catch (Exception ex) {
			LOG.error("Exception while trying to send RPC to CLDB: " + cldbCredentials.toString() + ". Trying another host/port", ex);
		}
		return null;
	}
	
	
	/** Get current valid IpPOrt for a particular cluster
	 * 
	 * @param clusterName
	 * @return
	 */
	public synchronized IpPort getCurrentValidIpPort(String clusterName) {
		if ( clustersMap.get(clusterName) != null && clustersMap.containsKey(clusterName) ) {
			return clustersMap.get(clusterName).get(0);
		}
		return null;
	}
	
	/**
	 * Get current validIPPort for Default cluster
	 * @return
	 */
	public IpPort getCurrentValidIpPort() {
		return getCurrentValidIpPort(defaultClusterName);
	}
	
	public String getCurrentClusterName() {
	  return defaultClusterName;
	}

  // Sets the defaultClusterName to the specified value, after making sure it's
  // a known cluster.  This interface only exists to support a hack in
  // DiffTablesWithCrc to submit mapreduce jobs to remote clusters.  Eventually,
  // this needs to be handled by the infrastructure, rather than being hacked up
  // by applications, which should allow us to delete this interface.
  public synchronized void setCurrentClusterName(String clusterName) {
    if (!clustersMap.containsKey(clusterName)) {
      throw new IllegalArgumentException("Cluster " + clusterName + " is not known");
    }
    if (origClusterName == null) {
      origClusterName = defaultClusterName;
    }
    defaultClusterName = clusterName;
  }
	
  // Resets the defaultClusterName to its original value.  This interface only
  // exists to support a hack in DiffTablesWithCrc to submit mapreduce jobs to
  // remote clusters.  Eventually, this needs to be handled by the
  // infrastructure, rather than being hacked up by applications, which should
  // allow us to delete this interface.
  public synchronized void resetCurrentClusterName() {
    if (origClusterName != null) {
      defaultClusterName = origClusterName;
    }
  }

	public Map<String, List<IpPort>> getClusterMap() {
	  return clustersMap;
	}

        public List<IpPort> getDefaultClusterIpPort() {
          return clustersMap.get(defaultClusterName);
        }
        
	
	/**
	 * Helper class to keep address port info
	 *
	 */
  public static class IpPort {
      private List<String> ips;
      private List<String> originalAddr;
      private int port;
	    
      public IpPort() {
        port = CLDB_PORT_DEFAULT;
        ips = new ArrayList<String>();
        originalAddr = new ArrayList<String>();
      }

      public void addIPOrHost(String addr, int port) {
        String ip = addr;
        if (!patternIP.matcher(addr).matches()) {
          // not an ip address
          ip = convertHostToIp(addr);
          if (ip == null) {
            LOG.error("Can not find non-local IP based on provided hostname: " + addr);
          }
        }

        if (ip != null) {
          // add this ip address to the list
          ips.add(ip);
          originalAddr.add(addr);
          this.port = port;
        }
      }

      public int getNumIPs() {
        return ips.size();
      }
	    
      public List<String> getAddr() {
        return ips;
      }
	    
      public List<String> getOriginalAddr() {
        return originalAddr;
      }

      public int getPort() {
        return port;
      }
	    
      @Override
      public String toString() {
        String ipStr = "";
        for(String ip : ips) {
          ipStr += ip + "-";
        }

        return "CLDB Ips: " + ipStr + ", Port: " + port;
      }
  }
  
    /**
     * Return first non-local IP based on provided host
     * if no such IP found returns null
     * @param host
     * @return
     */
	public static String convertHostToIp(String host) {
		// convert if needed hostNames to IPs
		try {
			InetAddress ia = InetAddress.getByName(host);
			String ip = ia.getHostAddress();
			return ip;
		} catch (UnknownHostException e) {
			/**
			* <MAPR_ERROR>
			* Message:Can not find IP for host: <host>
			* Function:NodesCommonUtils.convertHostToIp()
			* Meaning:An error occurred.
			* Resolution:Contact technical support.
			* </MAPR_ERROR>
			*/
			LOG.error("Can not find IP for host: " + host, e);
			
		}
		return null;
	}

  /**
   * Gets bindings for all the ips present in the clusterMap
   *
   * @return Array of cldb bindings.
   */
  public synchronized long[] getCldbBindings(String clusterName) 
                             throws CLDBRpcCommonUtilsException {
    // Using sychronized method takes lock over the complete class
    // TODO Need to avoid sychronized method and use granular lock later
    
    List<IpPort> ipPorts = clustersMap.get(clusterName);

    // re-initialize
    if (ipPorts == null || ipPorts.isEmpty())
      init();

    ipPorts = clustersMap.get(clusterName);

    if (ipPorts == null || ipPorts.isEmpty()) {
      LOG.error("Unable to get CLDB bindings for cluster " + clusterName);
      return null;
    }

    long[] cldbBindings = new long[ipPorts.size()];
    Iterator<IpPort> itr = ipPorts.iterator();
    IpPort ipPort;
    int index = 0;

    while(itr.hasNext()) {
      ipPort = itr.next();
      cldbBindings[index++] = init (ipPort.getPort(),
                                    ipPort.getAddr(),
                                    clusterName,
                                    ServerKeyType.ServerKey, 0);
    }

    return cldbBindings;
  }

}
