package com.mapr.baseutils.audit;

import java.util.List;
import com.mapr.baseutils.audit.KeyValue;
import static com.mapr.baseutils.audit.AuditConstants.*;

public class AuditJsonFormatter implements AuditFormatter {

  private static AuditJsonFormatter formatter = new AuditJsonFormatter();

  private AuditJsonFormatter() {
  }

  public static AuditJsonFormatter getInstance() {
    return formatter;
  }

  public String formatAuditRecord(AuditRecord auditRecord) {
    if (auditRecord == null)
      return null;

    StringBuilder sb = new StringBuilder();

    if (auditRecord.getKeyValues() != null) {
      if (auditRecord.getKeyValuesSR()) {
        formatMulti(sb,
               auditRecord.getIsoDate(),
               auditRecord.getOp().toString(),
               auditRecord.getUsername(),
               Integer.toString(auditRecord.getUid()),
               Integer.toString(auditRecord.getGid()),
               auditRecord.populateTargetUser(),
               Integer.toString(auditRecord.getTargetUid()),
               Integer.toString(auditRecord.getTargetGid()),
               auditRecord.getClientIp(),
               auditRecord.getResource(),
               auditRecord.getKeyValues(),
               Integer.toString(auditRecord.getStatus()));
        sb.append(NEWLINE);
      }
      else {
        for (KeyValue keyValue : auditRecord.getKeyValues()) {
          format(sb,
                 auditRecord.getIsoDate(),
                 auditRecord.getOp().toString(),
                 auditRecord.getUsername(),
                 Integer.toString(auditRecord.getUid()),
                 Integer.toString(auditRecord.getGid()),
                 auditRecord.populateTargetUser(),
                 Integer.toString(auditRecord.getTargetUid()),
                 Integer.toString(auditRecord.getTargetGid()),
                 auditRecord.getClientIp(),
                 auditRecord.getResource(),
                 keyValue,
                 Integer.toString(auditRecord.getStatus()));
          sb.append(NEWLINE);
        }
      }
    } else if (auditRecord.getOp() != null) {
      format(sb,
             auditRecord.getIsoDate(),
             auditRecord.getOp().toString(),
             auditRecord.getUsername(),
             Integer.toString(auditRecord.getUid()),
             Integer.toString(auditRecord.getGid()),
             auditRecord.populateTargetUser(),
             Integer.toString(auditRecord.getTargetUid()),
             Integer.toString(auditRecord.getTargetGid()),
             auditRecord.getClientIp(),
             auditRecord.getResource(),
             auditRecord.getKeyValue(),
             Integer.toString(auditRecord.getStatus()));
      sb.append(NEWLINE);
    } else {
      return null;
    }
  
    return sb.toString();
  }

  private void formatBasic(StringBuilder sb, String isoDate, String op,
                      String username, String uid, String gid,
                      boolean populateTargetUser, String targetUid,
                      String targetGid, String clientIp,
                      String resource) {

    addTimeStamp(sb, isoDate);
    sb.append(COMMA);
    
    addAuditField(sb, RESOURCE, resource, true);
    sb.append(COMMA);

    addAuditField(sb, OP, op, true);
    sb.append(COMMA);

    // populated only for authentication requests.
    if (username != null) {
      addAuditField(sb, USERNAME, username, true);
      sb.append(COMMA);
    } else {
      addAuditField(sb, UID, uid, false);
      sb.append(COMMA);
    }

    if (populateTargetUser) {
      addAuditField(sb, TARGETUID, targetUid, false);
      sb.append(COMMA);
      
      addAuditField(sb, TARGETGID, targetGid, false);
      sb.append(COMMA);
    }

    addAuditField(sb, CLIENTIP, clientIp, true);
    sb.append(COMMA);

  }

  private void format(StringBuilder sb, String isoDate, String op,
                      String username, String uid, String gid,
                      boolean populateTargetUser, String targetUid,
                      String targetGid, String clientIp,
                      String resource, KeyValue keyValue, String status) {
    sb.append(OPENFLOWER);

    formatBasic(sb, isoDate, op, username, uid, gid, populateTargetUser,
                targetUid, targetGid, clientIp, resource);

    if (keyValue.hasValues()) {
      sb.append(keyValue.toJsonString());
      sb.append(COMMA);
    }

    addAuditField(sb, STATUS, status, false);
    sb.append(CLOSEDFLOWER);
  }

  private void formatMulti(StringBuilder sb, String isoDate, String op,
                      String username, String uid, String gid,
                      boolean populateTargetUser, String targetUid,
                      String targetGid, String clientIp,
                      String resource, List<KeyValue> keyValues,
                      String status) {
    sb.append(OPENFLOWER);

    formatBasic(sb, isoDate, op, username, uid, gid, populateTargetUser,
                targetUid, targetGid, clientIp, resource);

    if (keyValues.size() > 0) {
      sb.append(KeyValue.toJsonStringSR(keyValues));
      sb.append(COMMA);
    }

    addAuditField(sb, STATUS, status, false);
    sb.append(CLOSEDFLOWER);
  }

  private void addTimeStamp(StringBuilder sb, String isoDate) {
    sb.append(QUOTE);
    sb.append(TIMESTAMP);
    sb.append(QUOTE);
    sb.append(COLON);
    sb.append(OPENFLOWER);
    sb.append(QUOTE);
    sb.append(DOLLAR);
    sb.append(DATE);
    sb.append(QUOTE);
    sb.append(COLON);
    sb.append(QUOTE);
    sb.append(isoDate);
    sb.append(QUOTE);
    sb.append(CLOSEDFLOWER); 
  }

  private void addAuditField(StringBuilder sb, String key, String value,
                             boolean addQuotesToValue) {
    sb.append(QUOTE);
    sb.append(key);
    sb.append(QUOTE);
    sb.append(COLON);
    if (addQuotesToValue)
      sb.append(QUOTE);
    if (value != null)
      sb.append(value);
    if (addQuotesToValue)
      sb.append(QUOTE);
  }
}
