package com.mapr.baseutils.acls;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import com.mapr.fs.cldb.proto.CLDBProto.SecureObjectType;

import com.mapr.fs.cldb.proto.CLDBProto.ClusterActions;
import com.mapr.fs.proto.Common.VolumeActions;

/*
 * This is a helper class that contains information about which incoming
 * RPCs are allowed by which bits on the ACL.
 */
public final class SecurityCommandHelper {
  // Cluster level actions
  /*
   * Volume creation requires permissions to create/delete volumes.
   */
  public static int CLUSTER_FULL_CONTROL_MASK =
     (1 << ClusterActions.CLUSTER_FULL_CONTROL.getNumber());

  public static int CLUSTER_ADMIN_MASK =
    (1 << ClusterActions.CLUSTER_ADMIN.getNumber());

  public static int CLUSTER_VOLUME_CREATE_MASK = 
    ((1 << ClusterActions.CLUSTER_CREATE_VOLUMES.getNumber()) |
     (CLUSTER_FULL_CONTROL_MASK));
  
  public static int CLUSTER_READ_MASK =
    ((1 << ClusterActions.CLUSTER_READ_ONLY.getNumber()) |
     (1 << ClusterActions.CLUSTER_CREATE_VOLUMES.getNumber()) |
     (1 << ClusterActions.CLUSTER_START_STOP_SERVICES.getNumber()) |
     (CLUSTER_FULL_CONTROL_MASK));
  
  public static int CLUSTER_EDIT_CONFIGURATION_MASK =
    CLUSTER_FULL_CONTROL_MASK;
  
  public static int CLUSTER_EDIT_ACL_MASK =
    (1 << ClusterActions.CLUSTER_ADMIN.getNumber());
  
  public static int CLUSTER_VIEW_QUOTAS_MASK =
    CLUSTER_READ_MASK;
  
  public static int CLUSTER_SET_QUOTAS_MASK =
    CLUSTER_FULL_CONTROL_MASK;

  public static int CLUSTER_START_STOP_SERVICES_MASK =
    ((1 << ClusterActions.CLUSTER_START_STOP_SERVICES.getNumber()) |
     CLUSTER_FULL_CONTROL_MASK);
  
  public static int VOLUME_FULL_CONTROL_MASK =
    (1 << VolumeActions.VOLUME_FULL_CONTROL.getNumber());

  public static int VOLUME_ADMIN_MASK =
    (1 << VolumeActions.VOLUME_ADMIN.getNumber());

  /*
   * Containers can be created by anyone who has permissions
   * to edit configuration or mirror the volume.
   */
  public static int VOLUME_CONTAINER_CREATE_DELETE_MASK = 
    ((1 << VolumeActions.VOLUME_EDIT_CONFIGURATION.getNumber()) |
     (1 << VolumeActions.VOLUME_CAN_MIRROR.getNumber()) |
     (VOLUME_FULL_CONTROL_MASK));
  
  public static int VOLUME_DELETE_MASK =
    ((1 << VolumeActions.VOLUME_DELETE.getNumber()) |
     (VOLUME_FULL_CONTROL_MASK));
  
  /*
   * Dumpers require the ability to create and delete volume snapshots.
   */
  public static int VOLUME_SNAPSHOT_CREATE_DELETE_MASK =
    ((1 << VolumeActions.VOLUME_EDIT_CONFIGURATION.getNumber()) |
     (VOLUME_FULL_CONTROL_MASK));
  
  public static int VOLUME_VIEW_CONFIGURATION_MASK =
    ((CLUSTER_READ_MASK) |
     (VOLUME_FULL_CONTROL_MASK));
  
  public static int VOLUME_EDIT_CONFIGURATION_MASK =
    ((1 << VolumeActions.VOLUME_EDIT_CONFIGURATION.getNumber()) |
     (VOLUME_FULL_CONTROL_MASK));
  
  /*
   * Dumpers dont need privileges to update snapshot interval on a volume.
   */
  public static int VOLUME_EDIT_SNAPSHOT_INTERVAL_MASK =
    ((VOLUME_SNAPSHOT_CREATE_DELETE_MASK) |
     (VOLUME_FULL_CONTROL_MASK));
  
  public static int VOLUME_EDIT_ACL_MASK =
    (VOLUME_ADMIN_MASK);
  
  /*
   * Users who want to dump/mirror a volume should be able to
   * view all containers in the mask.
   */
  public static int VOLUME_VIEW_CONTAINERS_MAP_MASK =
    ((1 << VolumeActions.VOLUME_CAN_DUMP.getNumber()) |
     (1 << VolumeActions.VOLUME_CAN_MIRROR.getNumber()) |
     (VOLUME_FULL_CONTROL_MASK));

  public static int VOLUME_DUMP_MASK =
     ((1 << VolumeActions.VOLUME_CAN_DUMP.getNumber()) |
      (VOLUME_FULL_CONTROL_MASK));

  public static int VOLUME_RESTORE_MASK =
     ((1 << VolumeActions.VOLUME_CAN_MIRROR.getNumber()) |
      (VOLUME_FULL_CONTROL_MASK));
  
  public static final String ClusterPerms = "[login, ss, cv, a, fc]";
  public static final String VolumePerms = "[dump, restore, m, d, a, fc]";
  
  /**
   * Every privilege has 2 entires for the short description and the long description.
   * Unused privileges should use null
   */
  public static final String[] clusterActionsDescription = {
    /*0*/ "login", "Login access",
    /*1*/ null, null,
    /*2*/ null, null,
    /*3*/ null, null,
    /*4*/ "ss", "Start/stop services in the cluster",
    /*5*/ "cv", "Create volumes",
    /*6*/ "a", "Administrator",
    /*7*/ "fc", "Full control",
    /*8*/ null, null,
    /*9*/ null, null,
    /*10*/null, null,
    /*11*/null, null,
    /*12*/null, null,
    /*13*/null, null,
    /*14*/null, null,
    /*15*/null, null,
    /*16*/null, null,
    /*17*/null, null,
    /*18*/null, null,
    /*19*/null, null,
    /*20*/null, null,
    /*21*/null, null,
    /*22*/null, null,
    /*23*/null, null,
    /*24*/null, null,
    /*25*/null, null,
    /*26*/null, null,
    /*27*/null, null,
    /*28*/null, null,
    /*29*/null, null,
    /*30*/null, null,
    /*31*/null, null,
  };
    
  public static final String[] volumeActionsDescription = {
    /*0*/ null, null,
    /*1*/ null, null,
    /*2*/ null, null,
    /*3*/ null, null,
    /*4*/ null, null,
    /*5*/ null, null,
    /*6*/ "dump", "Dump/backup the volume",
    /*7*/ "restore", "Mirror/Restore the volume",
    /*8*/ "m", "Edit volume configuration",
    /*9*/ "a", "Administrator",
    /*10*/null, null,
    /*11*/null, null,
    /*12*/"d", "Delete the volume",
    /*13*/null, null,
    /*14*/null, null,
    /*15*/null, null,
    /*16*/"fc", "Full control",
    /*17*/null, null,
    /*18*/null, null,
    /*19*/null, null,
    /*20*/null, null,
    /*21*/null, null,
    /*22*/null, null,
    /*23*/null, null,
    /*24*/null, null,
    /*25*/null, null,
    /*26*/null, null,
    /*27*/null, null,
    /*28*/null, null,
    /*29*/null, null,
    /*30*/null, null,
    /*31*/null, null,
  };
  
  public static List<String> formatActionMask(int actionMask, SecureObjectType objType, boolean shortFormat) {
    int mask;
    List<String> result = new ArrayList<String>();
    for(int i = 0; i < 32; ++i) {
      mask = (1 << i);
      if ((actionMask & mask) == mask) {
        // bit set
        String actionDesc = getActionDescription(objType, i, shortFormat);
        if (actionDesc != null) {
          result.add(actionDesc);
        }
      }
    }
    return result;
  }
  
  static String getActionDescription(SecureObjectType objType, int index, boolean shortFormat) {   
    int i = index * 2;
    if (!shortFormat) i++;

    if (objType == SecureObjectType.OBJECT_TYPE_CLUSTER) {
      return SecurityCommandHelper.clusterActionsDescription[i];
    } else if (objType == SecureObjectType.OBJECT_TYPE_VOLUME) {
      return SecurityCommandHelper.volumeActionsDescription[i];
    }
    
    return null;
  }
  
  public static int convertActionsToMask (String actions, String separator,SecureObjectType oType)
  throws Exception {
    int mask = 0;
    List<String> aList = Arrays.asList (actions.split(separator));
    for (String action: aList) {
      if (oType.getNumber() == SecureObjectType.OBJECT_TYPE_CLUSTER.getNumber()) {
        if (action.equalsIgnoreCase("login")) {
          mask |= 1 << ClusterActions.CLUSTER_READ_ONLY.getNumber();
        } else if (action.equalsIgnoreCase("ss")) {
          mask |= 1 << ClusterActions.CLUSTER_START_STOP_SERVICES.getNumber();
          mask |= 1 << ClusterActions.CLUSTER_READ_ONLY.getNumber();
        } else if (action.equalsIgnoreCase("cv")) {
          mask |= 1 << ClusterActions.CLUSTER_CREATE_VOLUMES.getNumber();
          mask |= 1 << ClusterActions.CLUSTER_READ_ONLY.getNumber();
        } else if (action.equalsIgnoreCase("a")) {
          mask |= 1 << ClusterActions.CLUSTER_ADMIN.getNumber();
        } else if (action.equalsIgnoreCase("fc")) {
          mask |= 1 << ClusterActions.CLUSTER_READ_ONLY.getNumber();
          mask |= 1 << ClusterActions.CLUSTER_START_STOP_SERVICES.getNumber();
          mask |= 1 << ClusterActions.CLUSTER_CREATE_VOLUMES.getNumber();
          mask |= 1 << ClusterActions.CLUSTER_FULL_CONTROL.getNumber();
        } else {
          throw new Exception ("unknown action: '" + action +
              "': valid cluster actions are: " + ClusterPerms);
        }
      } else {
        // Volume
        if (action.equalsIgnoreCase("dump")) {
          mask |= 1 << VolumeActions.VOLUME_CAN_DUMP.getNumber();
        } else if (action.equalsIgnoreCase("restore")) {
          mask |= 1 << VolumeActions.VOLUME_CAN_MIRROR.getNumber();
        } else if (action.equalsIgnoreCase("m")) {
          mask |= 1 << VolumeActions.VOLUME_EDIT_CONFIGURATION.getNumber();
        } else if (action.equalsIgnoreCase("d")) {
          mask |= 1 << VolumeActions.VOLUME_DELETE.getNumber();
        } else if (action.equalsIgnoreCase("a")) {
          mask |= 1 <<  VolumeActions.VOLUME_ADMIN.getNumber();
        } else if (action.equalsIgnoreCase("fc")) {
          mask |= 1 << VolumeActions.VOLUME_CAN_DUMP.getNumber();
          mask |= 1 << VolumeActions.VOLUME_CAN_MIRROR.getNumber();
          mask |= 1 << VolumeActions.VOLUME_EDIT_CONFIGURATION.getNumber();
          mask |= 1 << VolumeActions.VOLUME_DELETE.getNumber();
          mask |= 1 << VolumeActions.VOLUME_FULL_CONTROL.getNumber();
        } else {
          throw new Exception ("unknown action: '" + action +
              "': valid volume actions are: " + VolumePerms);
        }
      }
    }
    
    return mask;
  }
}


