/* Copyright (c) 2013 & onwards. MapR Technologies, Inc., All rights reserved */

package com.mapr.baseutils;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.apache.log4j.BasicConfigurator;
import org.apache.log4j.ConsoleAppender;
import org.apache.log4j.Logger;
import org.apache.log4j.TTCCLayout;

public class TimeTrace {
  private static final String DATE_FOMRAT = "yyyy-MM-dd HH:mm:ss,SSS";

  private static final Logger LOG = Logger.getLogger(TimeTrace.class);

  private static final DateFormat df = new SimpleDateFormat(DATE_FOMRAT);
  
  private static boolean shouldLog = false;

  private static boolean shouldPrint = true;

  private static ThreadLocal<Long> threshold_ = new ThreadLocal<Long>() {
    @Override
    protected Long initialValue() {
      return 0L;
    }
  };

  private static ThreadLocal<Map<String, Long>> traceMap =
      new ThreadLocal<Map<String,Long>>() {
    @Override
    protected Map<String, Long> initialValue() {
      return new HashMap<String, Long>();
    }
  };

  public static void setShouldLog(boolean shouldLog) {
    TimeTrace.shouldLog = shouldLog;
  }

  public static void setShouldPrint(boolean shouldPrint) {
    TimeTrace.shouldPrint = shouldPrint;
  }

  public static void setThreshold(long threshold) {
    TimeTrace.threshold_.set(threshold);
  }

  public static void startTrace(String traceName) {
    startTrace(traceName, TimeTrace.threshold_.get());
  }

  public static void startTrace(String traceName, long threshold) {
    if (threshold > 0) {
      Map<String, Long> map = traceMap.get();
      map.put(traceName, System.nanoTime());
    }
  }

  public static void endTrace(String traceName) {
    endTrace(traceName, TimeTrace.threshold_.get(), null, (Object[])null);
  }

  public static void endTrace(String traceName, long threshold) {
    endTrace(traceName, threshold, null, (Object[])null);
  }

  public static void endTrace(String traceName,
      String msg, Object... args) {
    endTrace(traceName, TimeTrace.threshold_.get(), msg, args);
  }

  public static void endTrace(String traceName, long threshold,
      String msg, Object... args) {
    if (threshold > 0) {
      long endTime = System.nanoTime();
      Map<String, Long> map = traceMap.get();
      Long startTime = map.remove(traceName);
      if (startTime != null) {
        long timeInMilliSec = (endTime - startTime) / 1000000L;
        if (timeInMilliSec >= threshold) {
          String traceMessage = String.format("[Trace '%s' duration: %dms] %s",
              traceName, timeInMilliSec, (msg==null ? "" : String.format(msg, args)));
          if (shouldPrint) {
            System.out.println(String.format("%s %s", 
                df.format(System.currentTimeMillis()), traceMessage));
          }
          if (shouldLog) {
            LOG.info(traceMessage);
          }
        }
      } else {
        String traceMessage = String.format(
            "Trace '%s' was not started.", traceName);
        if (shouldPrint) {
          System.out.println(String.format("%s [Trace Warning] %s", 
              df.format(System.currentTimeMillis()), traceMessage));
        }
        if (shouldLog) {
          LOG.warn(traceMessage);
        }
      }
    }
  }

  public static void main(String[] args) {
    TTCCLayout layout = new TTCCLayout(DATE_FOMRAT);
    layout.setCategoryPrefixing(false);
    BasicConfigurator.configure(new ConsoleAppender(layout ));

    // enable log4j output
    TimeTrace.setShouldLog(true);

    // disable stdout output
    TimeTrace.setShouldPrint(false);

    ExecutorService executorService =
        new ThreadPoolExecutor(
          50, // core thread pool size
          50, // maximum thread pool size
          1, // time to wait before resizing pool
          TimeUnit.MINUTES, 
          new ArrayBlockingQueue<Runnable>(50, true),
          new ThreadPoolExecutor.CallerRunsPolicy());

    while (true) {
      executorService.submit(new Runnable() {
        @Override
        public void run() {
          TimeTrace.setThreshold(500);
          TimeTrace.startTrace("XYZ");
          TimeTrace.startTrace("ABC");
          try {
            Thread.sleep((int)(Math.random()*1000));
          } catch (InterruptedException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
          }
          TimeTrace.endTrace("ABC", ""+Thread.currentThread().getId());
          TimeTrace.endTrace("XYZ", "Thread='%s', Key='%s'",
              Thread.currentThread().getName(), "DADASD");
        }
      });
    }
  }
}
