package com.mapr.baseutils;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.util.Properties;

import org.apache.log4j.Logger;

public class LogDirectoryFindingUtils {

	private static final Logger LOG = Logger.getLogger(LogDirectoryFindingUtils.class);
  private static final String hadoopVersionFile = "/conf/hadoop_version";

  /**
   * Method to read the hadoop_version file and find the given version
   * @param maprHome Mapr Home location (usually /opt/mapr)
   * @param version The version to look up. If 0 will get default hadoop version
   * @return The hadoop version that is requested
   */
  private static String getHadoopVersion(String maprHome, String version) {
    InputStream inputStream;
    Properties hadoopVersionProp = new Properties();
    String key; // The key to look up in properties

    try {
      inputStream = new FileInputStream(maprHome + hadoopVersionFile);
      // Load properties.
      // Did not cache properties in case that properties have default hadoop version updated
      hadoopVersionProp.load(inputStream);

      inputStream.close();
    } catch (FileNotFoundException e) {
      // Log in case there was an error loading
      LOG.error("Error while tring to get hadoop version", e);
    } catch (IOException e) {
      // Log in case there was an error
      LOG.error("Error while tring to get hadoop version", e);
    }

    // Check for given hadoop version and return the proper property
    if (version == null) {
      key = hadoopVersionProp.getProperty("default_mode");
    } else {
      key = version;
    }
    // In case there was an error and properties did not loaded, will return blank
    return hadoopVersionProp.getProperty(key + "_version", "");
  }

  /**
   * Get the hadoop log directory
   * @param maprHome The maprhome variable
   * @param version 1 for hadoop1, 2 for hadoop2/yarn and 0 for default version
   * @return The hadoop log directory filled out
   */
	public static String getHadoopLogsDirectory(String maprHome, String version) {
		File hadoopDir = new File(maprHome + "/hadoop");
    // Get the version from the hadoop_version file
    String currentVersion = getHadoopVersion(maprHome, version);
    final String currentHadoopDir = "hadoop-" +  currentVersion;

		if ( hadoopDir.isDirectory()) {
			File [] hadoopVersions = hadoopDir.listFiles(new FileFilter() {

				@Override
				public boolean accept(File pathname) {
					if ( pathname.isDirectory() && pathname.getName().equals(currentHadoopDir)) {
						return true;
					}
					return false;
				}}
			);
			for ( File hadoopVersion : hadoopVersions ) {
				try {
					return hadoopVersion.getCanonicalPath() + "/logs";
				} catch (IOException e) {
					LOG.error("Error while tring to get path for hadoop logs dir", e);
				}
			}
		}
		return null;
	}
	
	public static String getHbaseLogsDirectory(String maprHome) {
		File hadoopDir = new File(maprHome + "/hbase");
		if ( hadoopDir.isDirectory()) {
			File [] hadoopVersions = hadoopDir.listFiles(new FileFilter() {

				@Override
				public boolean accept(File pathname) {
					if ( pathname.isDirectory() && pathname.getName().startsWith("hbase")) {
						return true;
					}
					return false;
				}}
			);
			for ( File hadoopVersion : hadoopVersions ) {
				try {
					return hadoopVersion.getCanonicalPath() + "/logs";
				} catch (IOException e) {
					LOG.error("Error while tring to get path for hbase logs dir", e);
				}
			}
		}
		return null;
	}
	
	
}
